---
name: find-replace
description: "Modern find-and-replace using sd (simpler than sed) and batch replacement patterns. Triggers on: sd, find replace, batch replace, sed replacement, string replacement, rename."
compatibility: "Requires sd CLI tool. Install: brew install sd (macOS) or cargo install sd (cross-platform)."
allowed-tools: "Bash"
---

# Find Replace

Modern find-and-replace using sd.

## sd Basics

```bash
# Replace in file (in-place)
sd 'oldText' 'newText' file.txt

# Replace in multiple files
sd 'oldText' 'newText' *.js

# Preview without changing (pipe)
cat file.txt | sd 'old' 'new'
```

## sd vs sed

| sed | sd |
|-----|-----|
| `sed 's/old/new/g'` | `sd 'old' 'new'` |
| `sed -i 's/old/new/g'` | `sd 'old' 'new' file` |
| `sed 's#path/to#new/path#g'` | `sd 'path/to' 'new/path'` |

**Key difference:** sd is global by default, no delimiter issues.

## Common Patterns

```bash
# Variable/function rename
sd 'oldName' 'newName' src/**/*.ts

# Word boundaries (avoid partial matches)
sd '\boldName\b' 'newName' src/**/*.ts

# Import path update
sd "from '../utils'" "from '@/utils'" src/**/*.ts

# Capture groups
sd 'console\.log\((.*)\)' 'logger.info($1)' src/**/*.js
```

## Safe Batch Workflow

```bash
# 1. List affected files
rg -l 'oldPattern' src/

# 2. Preview replacements
rg 'oldPattern' -r 'newPattern' src/

# 3. Apply
sd 'oldPattern' 'newPattern' $(rg -l 'oldPattern' src/)

# 4. Verify
rg 'oldPattern' src/  # Should return nothing
git diff              # Review changes
```

## Special Characters

| Character | Escape |
|-----------|--------|
| `.` | `\.` |
| `*` | `\*` |
| `[` `]` | `\[` `\]` |
| `$` | `\$` |
| `\` | `\\` |

## Tips

| Tip | Reason |
|-----|--------|
| Always preview with `rg -r` first | Avoid mistakes |
| Use git before bulk changes | Easy rollback |
| Use `\b` for word boundaries | Avoid partial matches |
| Quote patterns | Prevent shell interpretation |

## Additional Resources

For detailed patterns, load:
- `./references/advanced-patterns.md` - Regex, batch workflows, real-world examples
