===== PAGE: https://docs.tigerdata.com/getting-started/try-key-features-timescale-products/ =====

# Try the key features in Tiger Data products



Tiger Cloud offers managed database services that provide a stable and reliable environment for your
applications.

Each Tiger Cloud service is a single optimised Postgres instance extended with innovations such as TimescaleDB in the database
engine, in a cloud infrastructure that delivers speed without sacrifice. A radically faster Postgres for transactional,
analytical, and agentic workloads at scale.

Tiger Cloud scales Postgres to ingest and query vast amounts of live data. Tiger Cloud
provides a range of features and optimizations that supercharge your queries while keeping the
costs down. For example:
* The hypercore row-columnar engine in TimescaleDB makes queries up to 350x faster, ingests 44% faster, and reduces
  storage by 90%.
* Tiered storage in Tiger Cloud seamlessly moves your data from high performance storage for frequently accessed data to
  low cost bottomless storage for rarely accessed data.

The following figure shows how TimescaleDB optimizes your data for superfast real-time analytics:

![Main features and tiered data](https://assets.timescale.com/docs/images/mutation.png )

This page shows you how to rapidly implement the features in Tiger Cloud that enable you to
ingest and query data faster while keeping the costs low.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

## Optimize time-series data in hypertables with hypercore

Time-series data represents the way a system, process, or behavior changes over time. Hypertables are Postgres tables
that help you improve insert and query performance by automatically partitioning your data by time. Each hypertable
is made up of child tables called chunks. Each chunk is assigned a range of time, and only
contains data from that range. When you run a query, TimescaleDB identifies the correct chunk and runs the query on
it, instead of going through the entire table. You can also tune hypertables to increase performance even more.

![Hypertable structure](https://assets.timescale.com/docs/images/hypertable-structure.png)

[Hypercore][hypercore] is the hybrid row-columnar storage engine in TimescaleDB used by hypertables. Traditional
databases force a trade-off between fast inserts (row-based storage) and efficient analytics
(columnar storage). Hypercore eliminates this trade-off, allowing real-time analytics without sacrificing
transactional capabilities.

Hypercore dynamically stores data in the most efficient format for its lifecycle:

* **Row-based storage for recent data**: the most recent chunk (and possibly more) is always stored in the rowstore,
   ensuring fast inserts, updates, and low-latency single record queries. Additionally, row-based storage is used as a
   writethrough for inserts and updates to columnar storage.
* **Columnar storage for analytical performance**: chunks are automatically compressed into the columnstore, optimizing
   storage efficiency and accelerating analytical queries.

Unlike traditional columnar databases, hypercore allows data to be inserted or modified at any stage, making it a
flexible solution for both high-ingest transactional workloads and real-time analytics—within a single database.

Hypertables exist alongside regular Postgres tables.
You use regular Postgres tables for relational data, and interact with hypertables
and regular Postgres tables in the same way.

This section shows you how to create regular tables and hypertables, and import
relational and time-series data from external files.

1.  **Import some time-series data into hypertables**

    1. Unzip [crypto_sample.zip](https://assets.timescale.com/docs/downloads/candlestick/crypto_sample.zip) to a `<local folder>`.

       This test dataset contains:
         - Second-by-second data for the most-traded crypto-assets. This time-series data is best suited for
           optimization in a [hypertable][hypertables-section].
         - A list of asset symbols and company names. This is best suited for a regular relational table.

       To import up to 100 GB of data directly from your current Postgres-based database,
       [migrate with downtime][migrate-with-downtime] using native Postgres tooling. To seamlessly import 100GB-10TB+
       of data, use the [live migration][migrate-live] tooling supplied by Tiger Data. To add data from non-Postgres data
       sources, see [Import and ingest data][data-ingest].

    1. Upload data into a hypertable:

       To more fully understand how to create a hypertable, how hypertables work, and how to optimize them for
       performance by tuning chunk intervals and enabling chunk skipping, see
       [the hypertables documentation][hypertables-section].





          The Tiger Cloud Console data upload creates hypertables and relational tables from the data you are uploading:
          1. In [Tiger Cloud Console][portal-ops-mode], select the service to add data to, then click `Actions` > `Import data` > `Upload .CSV`.
          1. Click to browse, or drag and drop `<local folder>/tutorial_sample_tick.csv` to upload.
          1. Leave the default settings for the delimiter, skipping the header, and creating a new table.
          1. In `Table`, provide `crypto_ticks` as the new table name.
          1. Enable `hypertable partition` for the `time` column and click `Process CSV file`.

              The upload wizard creates a hypertable containing the data from the CSV file.
          1. When the data is uploaded, close `Upload .CSV`.

              If you want to  have a quick look at your data, press `Run` .
          1. Repeat the process with `<local folder>/tutorial_sample_assets.csv` and rename to `crypto_assets`.

              There is no time-series data in this table, so you don't see the  `hypertable partition` option.





       1. In Terminal, navigate to `<local folder>` and connect to your service.
          ```bash
          psql -d "postgres://<username>:<password>@<host>:<port>/<database-name>"
          ```
          You use your [connection details][connection-info] to fill in this Postgres connection string.

       2. Create tables for the data to import:

          - For the time-series data:

             1. In your sql client, create a hypertable:

                Create a [hypertable][hypertables-section] for your time-series data using [CREATE TABLE][hypertable-create-table].
                For [efficient queries][secondary-indexes], remember to `segmentby` the column you will
                use most often to filter your data. For example:

                ```sql
                CREATE TABLE crypto_ticks (
                  "time" TIMESTAMPTZ,
                  symbol TEXT,
                  price DOUBLE PRECISION,
                  day_volume NUMERIC
                ) WITH (
                   tsdb.hypertable,
                   tsdb.partition_column='time',
                   tsdb.segmentby = 'symbol'
                );
                ```

                If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

          - For the relational data:

             In your sql client, create a normal Postgres table:
             ```sql
             CREATE TABLE crypto_assets (
              symbol TEXT NOT NULL,
              name TEXT NOT NULL
             );
            ```
       1. Speed up data ingestion:

          When you set `timescaledb.enable_direct_compress_copy` your data gets compressed in memory during ingestion with `COPY` statements.
By writing the compressed batches immediately in the columnstore, the IO footprint is significantly lower.
Also, the [columnstore policy][add_columnstore_policy] you set is less important, `INSERT` already produces compressed chunks.



Please note that this feature is a **tech preview** and not production-ready.
Using this feature could lead to regressed query performance and/or storage ratio, if the ingested batches are not
correctly ordered or are of too high cardinality.



To enable in-memory data compression during ingestion:

```sql
SET timescaledb.enable_direct_compress_copy=on;
```

**Important facts**
- High cardinality use cases do not produce good batches and lead to degreaded query performance.
- The columnstore is optimized to store 1000 records per batch, which is the optimal format for ingestion per segment by.
- WAL records are written for the compressed batches rather than the individual tuples.
- Currently only `COPY` is support, `INSERT` will eventually follow.
- Best results are achieved for batch ingestion with 1000 records or more, upper boundary is 10.000 records.
- Continous Aggregates are **not** supported at the moment.

       3. Upload the dataset to your service:

          ```sql
          \COPY crypto_ticks from './tutorial_sample_tick.csv' DELIMITER ',' CSV HEADER;
          ```

          ```sql
          \COPY crypto_assets from './tutorial_sample_assets.csv' DELIMITER ',' CSV HEADER;
          ```





1.  **Have a quick look at your data**

    You query hypertables in exactly the same way as you would a relational Postgres table.
    Use one of the following SQL editors to run a query and see the data you uploaded:
    - **Data mode**:  write queries, visualize data, and share your results in [Tiger Cloud Console][portal-data-mode] for all your Tiger Cloud services. This feature is not available under the Free pricing plan.
    - **SQL editor**: write, fix, and organize SQL faster and more accurately in [Tiger Cloud Console][portal-ops-mode] for a Tiger Cloud service.
    - **psql**: easily run queries on your Tiger Cloud services or self-hosted TimescaleDB deployment from Terminal.

    <TryItOutCodeBlock queryId="getting-started-crypto-srt-orderby" />

## Enhance query performance for analytics

Hypercore is the TimescaleDB hybrid row-columnar storage engine, designed specifically for real-time
analytics and
powered by time-series data. The advantage of hypercore is its ability to seamlessly switch between row-oriented and
column-oriented storage. This flexibility enables TimescaleDB to deliver the best of both worlds, solving the key
challenges in real-time analytics.

![Move from rowstore to columstore in hypercore](https://assets.timescale.com/docs/images/hypercore.png )

When TimescaleDB converts chunks from the rowstore to the columnstore, multiple records are grouped into a single row.
The columns of this row hold an array-like structure that stores all the data. Because a single row takes up less disk
space, you can reduce your chunk size by up to 98%, and can also speed up your queries. This helps you save on storage costs,
and keeps your queries operating at lightning speed.

hypercore is enabled by default when you call [CREATE TABLE][hypertable-create-table]. Best practice is to compress
data that is no longer needed for highest performance queries, but is still accessed regularly in the columnstore.
For example, yesterday's market data.

1. **Add a policy to convert chunks to the columnstore at a specific time interval**

   For example, yesterday's data:
   ``` sql
   CALL add_columnstore_policy('crypto_ticks', after => INTERVAL '1d');
   ```
   If you have not configured a `segmentby` column, TimescaleDB chooses one for you based on the data in your
   hypertable. For more information on how to tune your hypertables for the best performance, see
   [efficient queries][secondary-indexes].

1. **View your data space saving**

   When you convert data to the columnstore, as well as being optimized for analytics, it is compressed by more than
   90%. This helps you save on storage costs and keeps your queries operating at lightning speed. To see the amount of space
   saved, click `Explorer` > `public` > `crypto_ticks`.

   ![Columnstore data savings](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-columstore-data-savings.png )

## Write fast and efficient analytical queries

Aggregation is a way of combing data to get insights from it. Average, sum, and count are all
examples of simple aggregates. However, with large amounts of data, aggregation slows things down, quickly.
Continuous aggregates are a kind of hypertable that is refreshed automatically in
the background as new data is added, or old data is modified. Changes to your dataset are tracked,
and the hypertable behind the continuous aggregate is automatically updated in the background.

![Reduced data calls with continuous aggregates](https://assets.timescale.com/docs/images/continuous-aggregate.png)

You create continuous aggregates on uncompressed data in high-performance storage. They continue to work
on [data in the columnstore][test-drive-enable-compression]
and [rarely accessed data in tiered storage][test-drive-tiered-storage]. You can even
create [continuous aggregates on top of your continuous aggregates][hierarchical-caggs].

You use time buckets to create a continuous aggregate. Time buckets aggregate data in hypertables by time
interval. For example, a 5-minute, 1-hour, or 3-day bucket. The data grouped in a time bucket uses a single
timestamp. Continuous aggregates minimize the number of records that you need to look up to perform your
query.

This section shows you how to run fast analytical queries using time buckets and continuous aggregate in
Tiger Cloud Console. You can also do this using psql.





This feature is not available under the Free pricing plan.

1.  **Connect to your service**

    In [Tiger Cloud Console][portal-data-mode], select your service in the connection drop-down in the top right.

1.  **Create a continuous aggregate**

    For a continuous aggregate, data grouped using a time bucket is stored in a
    Postgres `MATERIALIZED VIEW` in a hypertable. `timescaledb.continuous` ensures that this data
    is always up to date.
    In data mode, use the following code to create a continuous aggregate on the real-time data in
    the `crypto_ticks` table:

    ```sql
    CREATE MATERIALIZED VIEW assets_candlestick_daily
    WITH (timescaledb.continuous) AS
    SELECT
      time_bucket('1 day', "time") AS day,
      symbol,
      max(price) AS high,
      first(price, time) AS open,
      last(price, time) AS close,
      min(price) AS low
    FROM crypto_ticks srt
    GROUP BY day, symbol;
    ```

    This continuous aggregate creates the [candlestick chart][charts] data you use to visualize
    the price change of an asset.

1. **Create a policy to refresh the view every hour**

   ```sql
   SELECT add_continuous_aggregate_policy('assets_candlestick_daily',
   start_offset => INTERVAL '3 weeks',
   end_offset => INTERVAL '24 hours',
   schedule_interval => INTERVAL '3 hours');
   ```

1.  **Have a quick look at your data**

    You query continuous aggregates exactly the same way as your other tables. To query the `assets_candlestick_daily`
    continuous aggregate for all assets:

    <TryItOutCodeBlock queryId="getting-started-crypto-cagg" />





1. **In [Tiger Cloud Console][portal-ops-mode], select the service you uploaded data to**
1. **Click `Explorer` > `Continuous Aggregates` > `Create a Continuous Aggregate` next to the `crypto_ticks` hypertable**
1. **Create a view called `assets_candlestick_daily` on the `time` column with an interval of `1 day`, then click `Next step`**
   ![continuous aggregate wizard](https://assets.timescale.com/docs/images/tiger-cloud-console/continuous-aggregate-wizard-tiger-console.png )
1. **Update the view SQL with the following functions, then click `Run`**
   ```sql
   CREATE MATERIALIZED VIEW assets_candlestick_daily
   WITH (timescaledb.continuous) AS
   SELECT
     time_bucket('1 day', "time") AS bucket,
     symbol,
     max(price) AS high,
     first(price, time) AS open,
     last(price, time) AS close,
     min(price) AS low
   FROM "public"."crypto_ticks" srt
   GROUP BY bucket, symbol;
    ```
1. **When the view is created, click `Next step`**
1. **Define a refresh policy with the following values:**
   - `How far back do you want to materialize?`: `3 weeks`
   - `What recent data to exclude?`: `24 hours`
   - `How often do you want the job to run?`: `3 hours`
1. **Click `Next step`, then click `Run`**

Tiger Cloud creates the continuous aggregate and displays the aggregate ID in Tiger Cloud Console. Click `DONE` to close the wizard.





To see the change in terms of query time and data returned between a regular query and
a continuous aggregate, run the query part of the continuous aggregate
( `SELECT ...GROUP BY day, symbol;` ) and compare the results.

## Slash storage charges

<Availability products={['cloud']} price_plans={['enterprise', 'scale']} />

In the previous sections, you used continuous aggregates to make fast analytical queries, and
hypercore to reduce storage costs on frequently accessed data. To reduce storage costs even more,
you create tiering policies to move rarely accessed data to the object store. The object store is
low-cost bottomless data storage built on Amazon S3. However, no matter the tier, you can
[query your data when you need][querying-tiered-data]. Tiger Cloud seamlessly accesses the correct storage
tier and generates the response.

![Tiered storage](https://assets.timescale.com/docs/images/tiered-storage.png )

To set up data tiering:

1. **Enable data tiering**

   1. In [Tiger Cloud Console][portal-ops-mode], select the service to modify.

   1. In `Explorer`, click `Storage configuration` > `Tiering storage`, then click `Enable tiered storage`.

      ![Enable tiered storage](https://assets.timescale.com/docs/images/tiger-cloud-console/enable-tiered-storage-tiger-console.png)

      When tiered storage is enabled, you see the amount of data in the tiered object storage.

1. **Set the time interval when data is tiered**

    In Tiger Cloud Console, click `Data` to switch to the data mode, then enable data tiering on a hypertable with the following query:
     ```sql
     SELECT add_tiering_policy('assets_candlestick_daily', INTERVAL '3 weeks');
     ```

1. **Query tiered data**

    You enable reads from tiered data for each query, for a session or for all future
    sessions. To run a single query on tiered data:

    1. Enable reads on tiered data:
      ```sql
      set timescaledb.enable_tiered_reads = true
      ```
    1. Query the data:
      ```sql
      SELECT * FROM crypto_ticks srt LIMIT 10
      ```
    1. Disable reads on tiered data:
      ```sql
      set timescaledb.enable_tiered_reads = false;
      ```
    For more information, see [Querying tiered data][querying-tiered-data].

## Reduce the risk of downtime and data loss

<Availability products={['cloud']} price_plans={['enterprise', 'scale']} />

By default, all Tiger Cloud services have rapid recovery enabled. However, if your app has very low tolerance
for downtime, Tiger Cloud offers high-availability replicas. HA replicas are exact, up-to-date copies
of your database hosted in multiple AWS availability zones (AZ) within the same region as your primary node.
HA replicas automatically take over operations if the original primary data node becomes unavailable.
The primary node streams its write-ahead log (WAL) to the replicas to minimize the chances of
data loss during failover.

1.  In [Tiger Cloud Console][cloud-login], select the service to enable replication for.
1.  Click `Operations`, then select `High availability`.
1.  Choose your replication strategy, then click `Change configuration`.

    ![Tiger Cloud service replicas](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-ha-replicas.png)

1. In `Change high availability configuration`, click `Change config`.

For more information, see [High availability][high-availability].

What next? See the [use case tutorials][tutorials], interact with the data in your Tiger Cloud service using
[your favorite programming language][connect-with-code], integrate your Tiger Cloud service with a range of
[third-party tools][integrations], plain old [Use Tiger Data products][use-timescale], or dive into [the API][use-the-api].


===== PAGE: https://docs.tigerdata.com/getting-started/start-coding-with-timescale/ =====

# Start coding with Tiger Data



Easily integrate your app with Tiger Cloud or self-hosted TimescaleDB. Use your favorite programming language to connect to your
Tiger Cloud service, create and manage hypertables, then ingest and query data.





# "Quick Start: Ruby and TimescaleDB"


## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

*   Install [Rails][rails-guide].

## Connect a Rails app to your service

Every Tiger Cloud service is a 100% Postgres database hosted in Tiger Cloud with
Tiger Data extensions such as TimescaleDB. You connect to your Tiger Cloud service
from a standard Rails app configured for Postgres.

1.  **Create a new Rails app configured for Postgres**

    Rails creates and bundles your app, then installs the standard Postgres Gems.

    ```bash
    rails new my_app -d=postgresql
    cd my_app
    ```

1. **Install the TimescaleDB gem**

   1.  Open `Gemfile`, add the following line, then save your changes:

       ```ruby
       gem 'timescaledb'
       ```

   1. In Terminal, run the following command:

      ```bash
      bundle install
      ```

1. **Connect your app to your Tiger Cloud service**

   1.  In `<my_app_home>/config/database.yml` update the configuration to read securely connect to your Tiger Cloud service
       by adding `url: <%= ENV['DATABASE_URL'] %>` to the default configuration:

       ```yaml
       default: &default
         adapter: postgresql
         encoding: unicode
         pool: <%= ENV.fetch("RAILS_MAX_THREADS") { 5 } %>
         url: <%= ENV['DATABASE_URL'] %>
       ```

   1.  Set the environment variable for `DATABASE_URL` to the value of `Service URL` from
       your [connection details][connection-info]
       ```bash
       export DATABASE_URL="value of Service URL"
       ```

   1. Create the database:
      - **Tiger Cloud**: nothing to do. The database is part of your Tiger Cloud service.
      - **Self-hosted TimescaleDB**, create the database for the project:

          ```bash
          rails db:create
          ```

   1.  Run migrations:

       ```bash
       rails db:migrate
       ```

   1.  Verify the connection from your app to your Tiger Cloud service:

       ```bash
       echo "\dx" | rails dbconsole
       ```

       The result shows the list of extensions in your Tiger Cloud service

      |  Name  | Version | Schema | Description  |
      | --  | -- | -- | -- |
      | pg_buffercache      | 1.5     | public     | examine the shared buffer cache|
      | pg_stat_statements  | 1.11    | public     | track planning and execution statistics of all SQL statements executed|
      | plpgsql             | 1.0     | pg_catalog | PL/pgSQL procedural language|
      | postgres_fdw        | 1.1     | public     | foreign-data wrapper for remote Postgres servers|
      | timescaledb         | 2.18.1  | public     | Enables scalable inserts and complex queries for time-series data (Community Edition)|
      | timescaledb_toolkit | 1.19.0  | public     | Library of analytical hyperfunctions, time-series pipelining, and other SQL utilities|

## Optimize time-series data in hypertables

Hypertables are Postgres tables designed to simplify and accelerate data analysis. Anything
you can do with regular Postgres tables, you can do with hypertables - but much faster and more conveniently.

In this section, you use the helpers in the TimescaleDB gem to create and manage a [hypertable][about-hypertables].

1.  **Generate a migration to create the page loads table**

    ```bash
    rails generate migration create_page_loads
    ```

   This creates the `<my_app_home>/db/migrate/<migration-datetime>_create_page_loads.rb` migration file.

1. **Add hypertable options**

   Replace the contents of `<my_app_home>/db/migrate/<migration-datetime>_create_page_loads.rb`
   with the following:

    ```ruby
    class CreatePageLoads < ActiveRecord::Migration[8.0]
      def change
        hypertable_options = {
          time_column: 'created_at',
          chunk_time_interval: '1 day',
          compress_segmentby: 'path',
          compress_orderby: 'created_at',
          compress_after: '7 days',
          drop_after: '30 days'
        }

        create_table :page_loads, id: false, primary_key: [:created_at, :user_agent, :path], hypertable: hypertable_options do |t|
          t.timestamptz :created_at, null: false
          t.string :user_agent
          t.string :path
          t.float :performance
        end
      end
    end
    ```

    The `id` column is not included in the table. This is because TimescaleDB requires that any `UNIQUE` or `PRIMARY KEY`
    indexes on the table include all partitioning columns. In this case, this is the time column. A new
    Rails model includes a `PRIMARY KEY` index for id by default: either remove the column or make sure that the index
    includes time as part of a "composite key."

   For more information, check the Roby docs around [composite primary keys][rails-compostite-primary-keys].

1.  **Create a `PageLoad` model**

    Create a new file called `<my_app_home>/app/models/page_load.rb` and add the following code:

    ```ruby
    class PageLoad < ApplicationRecord
      extend Timescaledb::ActsAsHypertable
      include Timescaledb::ContinuousAggregatesHelper

      acts_as_hypertable time_column: "created_at",
        segment_by: "path",
        value_column: "performance"

      scope :chrome_users, -> { where("user_agent LIKE ?", "%Chrome%") }
      scope :firefox_users, -> { where("user_agent LIKE ?", "%Firefox%") }
      scope :safari_users, -> { where("user_agent LIKE ?", "%Safari%") }

      scope :performance_stats, -> {
        select("stats_agg(#{value_column}) as stats_agg")
      }

      scope :slow_requests, -> { where("performance > ?", 1.0) }
      scope :fast_requests, -> { where("performance < ?", 0.1) }

      continuous_aggregates scopes: [:performance_stats],
        timeframes: [:minute, :hour, :day],
        refresh_policy: {
          minute: {
            start_offset: '3 minute',
            end_offset: '1 minute',
            schedule_interval: '1 minute'
          },
          hour: {
            start_offset: '3 hours',
            end_offset: '1 hour',
            schedule_interval: '1 minute'
          },
          day: {
            start_offset: '3 day',
            end_offset: '1 day',
            schedule_interval: '1 minute'
          }
        }
    end
    ```

1.  **Run the migration**

    ```bash
    rails db:migrate
    ```

## Insert data your service

The TimescaleDB gem provides efficient ways to insert data into hypertables. This section
shows you how to ingest test data into your hypertable.

1.  **Create a controller to handle page loads**

    Create a new file called `<my_app_home>/app/controllers/application_controller.rb` and add the following code:

    ```ruby
    class ApplicationController < ActionController::Base
      around_action :track_page_load

      private

      def track_page_load
        start_time = Time.current
        yield
        end_time = Time.current

        PageLoad.create(
          path: request.path,
          user_agent: request.user_agent,
          performance: (end_time - start_time)
        )
      end
    end
    ```

1.  **Generate some test data**

    Use `bin/console` to join a Rails console session and run the following code
    to define some random page load access data:

    ```ruby
    def generate_sample_page_loads(total: 1000)
      time = 1.month.ago
      paths = %w[/ /about /contact /products /blog]
      browsers = [
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.114 Safari/537.36",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:89.0) Gecko/20100101 Firefox/89.0",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/14.1.1 Safari/605.1.15"
      ]

      total.times.map do
        time = time + rand(60).seconds
        {
          path: paths.sample,
          user_agent: browsers.sample,
          performance: rand(0.1..2.0),
          created_at: time,
          updated_at: time
        }
      end
    end
    ```

1. **Insert the generated data into your Tiger Cloud service**

    ```bash
    PageLoad.insert_all(generate_sample_page_loads, returning: false)
   ```

1.  **Validate the test data in your Tiger Cloud service**

   ```bash
   PageLoad.count
   PageLoad.first
   ```

## Reference

This section lists the most common tasks you might perform with the TimescaleDB gem.

### Query scopes

The TimescaleDB gem provides several convenient scopes for querying your time-series data.


- Built-in time-based scopes:

    ```ruby
    PageLoad.last_hour.count
    PageLoad.today.count
    PageLoad.this_week.count
    PageLoad.this_month.count
    ```

- Browser-specific scopes:

    ```ruby
    PageLoad.chrome_users.last_hour.count
    PageLoad.firefox_users.last_hour.count
    PageLoad.safari_users.last_hour.count

    PageLoad.slow_requests.last_hour.count
    PageLoad.fast_requests.last_hour.count
    ```

- Query continuous aggregates:

  This query fetches the average and standard deviation from the performance stats for the `/products` path over the last day.

    ```ruby
    PageLoad::PerformanceStatsPerMinute.last_hour
    PageLoad::PerformanceStatsPerHour.last_day
    PageLoad::PerformanceStatsPerDay.last_month

    stats = PageLoad::PerformanceStatsPerHour.last_day.where(path: '/products').select("average(stats_agg) as average, stddev(stats_agg) as stddev").first
    puts "Average: #{stats.average}"
    puts "Standard Deviation: #{stats.stddev}"
    ```

### TimescaleDB features

The TimescaleDB gem provides utility methods to access hypertable and chunk information. Every model that uses
the `acts_as_hypertable` method has access to these methods.


#### Access hypertable and chunk information

- View chunk or hypertable information:

    ```ruby
    PageLoad.chunks.count
    PageLoad.hypertable.detailed_size
    ```

- Compress/Decompress chunks:

    ```ruby
    PageLoad.chunks.uncompressed.first.compress!  # Compress the first uncompressed chunk
    PageLoad.chunks.compressed.first.decompress!  # Decompress the oldest chunk
    PageLoad.hypertable.compression_stats # View compression stats

    ```

#### Access hypertable stats

You collect hypertable stats using methods that provide insights into your hypertable's structure, size, and compression
status:

- Get basic hypertable information:

    ```ruby
    hypertable = PageLoad.hypertable
    hypertable.hypertable_name  # The name of your hypertable
    hypertable.schema_name      # The schema where the hypertable is located
    ```

- Get detailed size information:

    ```ruby
    hypertable.detailed_size # Get detailed size information for the hypertable
    hypertable.compression_stats # Get compression statistics
    hypertable.chunks_detailed_size # Get chunk information
    hypertable.approximate_row_count # Get approximate row count
    hypertable.dimensions.map(&:column_name) # Get dimension information
    hypertable.continuous_aggregates.map(&:view_name) # Get continuous aggregate view names
    ```

#### Continuous aggregates

The `continuous_aggregates` method generates a class for each continuous aggregate.

- Get all the continuous aggregate classes:

   ```ruby
   PageLoad.descendants # Get all continuous aggregate classes
   ```

- Manually refresh a continuous aggregate:

   ```ruby
   PageLoad.refresh_aggregates
   ```

- Create or drop a continuous aggregate:

  Create or drop all the continuous aggregates in the proper order to build them hierarchically. See more about how it
  works in this [blog post][ruby-blog-post].

   ```ruby
   PageLoad.create_continuous_aggregates
   PageLoad.drop_continuous_aggregates
   ```




## Next steps

Now that you have integrated the ruby gem into your app:

* Learn more about the [TimescaleDB gem](https://github.com/timescale/timescaledb-ruby).
* Check out the [official docs](https://timescale.github.io/timescaledb-ruby/).
* Follow the [LTTB][LTTB], [Open AI long-term storage][open-ai-tutorial], and [candlesticks][candlesticks] tutorials.




## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

*   Install the `psycopg2` library.

   For more information, see the [psycopg2 documentation][psycopg2-docs].
*   Create a [Python virtual environment][virtual-env]. [](#)

## Connect to TimescaleDB

In this section, you create a connection to TimescaleDB using the `psycopg2`
library. This library is one of the most popular Postgres libraries for
Python. It allows you to execute raw SQL queries efficiently and safely, and
prevents common attacks such as SQL injection.

1.  Import the psycogpg2 library:

    ```python
    import psycopg2
    ```

1.  Locate your TimescaleDB credentials and use them to compose a connection
   string for `psycopg2`.

    You'll need:

      *   password
      *   username
      *   host URL
      *   port
      *   database name

1.  Compose your connection string variable as a
    [libpq connection string][pg-libpq-string], using this format:

    ```python
    CONNECTION = "postgres://username:password@host:port/dbname"
    ```

    If you're using a hosted version of TimescaleDB, or generally require an SSL
    connection, use this version instead:

    ```python
    CONNECTION = "postgres://username:password@host:port/dbname?sslmode=require"
    ```

    Alternatively you can specify each parameter in the connection string as follows

    ```python
    CONNECTION = "dbname=tsdb user=tsdbadmin password=secret host=host.com port=5432 sslmode=require"
    ```



    This method of composing a connection string is for test or development
    purposes only. For production, use environment variables for sensitive
    details like your password, hostname, and port number.



1.  Use the `psycopg2` [connect function][psycopg2-connect] to create a new
    database session and create a new [cursor object][psycopg2-cursor] to
    interact with the database.

    In your `main` function, add these lines:

    ```python
    CONNECTION = "postgres://username:password@host:port/dbname"
    with psycopg2.connect(CONNECTION) as conn:
        cursor = conn.cursor()
        # use the cursor to interact with your database
        # cursor.execute("SELECT * FROM table")
    ```

    Alternatively, you can create a connection object and pass the object
    around as needed, like opening a cursor to perform database operations:

    ```python
    CONNECTION = "postgres://username:password@host:port/dbname"
    conn = psycopg2.connect(CONNECTION)
    cursor = conn.cursor()
    # use the cursor to interact with your database
    cursor.execute("SELECT 'hello world'")
    print(cursor.fetchone())
    ```

## Create a relational table

In this section, you create a table called `sensors` which holds the ID, type,
and location of your fictional sensors. Additionally, you create a hypertable
called `sensor_data` which holds the measurements of those sensors. The
measurements contain the time, sensor_id, temperature reading, and CPU
percentage of the sensors.

1.  Compose a string which contains the SQL statement to create a relational
    table. This example creates a table called `sensors`, with columns `id`,
    `type` and `location`:

    ```python
    query_create_sensors_table = """CREATE TABLE sensors (
                                        id SERIAL PRIMARY KEY,
                                        type VARCHAR(50),
                                        location VARCHAR(50)
                                    );
                                    """
    ```

1.  Open a cursor, execute the query you created in the previous step, and
    commit the query to make the changes persistent. Afterward, close the cursor
    to clean up:

    ```python
    cursor = conn.cursor()
    # see definition in Step 1
    cursor.execute(query_create_sensors_table)
    conn.commit()
    cursor.close()
    ```

## Create a hypertable

When you have created the relational table, you can create a hypertable.
Creating tables and indexes, altering tables, inserting data, selecting data,
and most other tasks are executed on the hypertable.

1.  Create a string variable that contains the `CREATE TABLE` SQL statement for
    your hypertable. Notice how the hypertable has the compulsory time column:

    ```python
    # create sensor data hypertable
    query_create_sensordata_table = """CREATE TABLE sensor_data (
                                            time TIMESTAMPTZ NOT NULL,
                                            sensor_id INTEGER,
                                            temperature DOUBLE PRECISION,
                                            cpu DOUBLE PRECISION,
                                            FOREIGN KEY (sensor_id) REFERENCES sensors (id)
                                        );
                                        """
    ```

2.  Formulate a `SELECT` statement that converts the `sensor_data` table to a
    hypertable. You must specify the table name to convert to a hypertable, and
    the name of the time column as the two arguments. For more information, see
    the [`create_hypertable` docs][create-hypertable-docs]:

    ```python
    query_create_sensordata_hypertable = "SELECT create_hypertable('sensor_data', by_range('time'));"
    ```



	The `by_range` dimension builder is an addition to TimescaleDB 2.13.



3.  Open a cursor with the connection, execute the statements from the previous
    steps, commit your changes, and close the cursor:

    ```python
    cursor = conn.cursor()
    cursor.execute(query_create_sensordata_table)
    cursor.execute(query_create_sensordata_hypertable)
    # commit changes to the database to make changes persistent
    conn.commit()
    cursor.close()
    ```

## Insert rows of data

You can insert data into your hypertables in several different ways. In this
section, you can use `psycopg2` with prepared statements, or you can use
`pgcopy` for a faster insert.

1.  This example inserts a list of tuples, or relational data, called `sensors`,
    into the relational table named `sensors`. Open a cursor with a connection
    to the database, use prepared statements to formulate the `INSERT` SQL
    statement, and then execute that statement:

    ```python
    sensors = [('a', 'floor'), ('a', 'ceiling'), ('b', 'floor'), ('b', 'ceiling')]
    cursor = conn.cursor()
    for sensor in sensors:
      try:
        cursor.execute("INSERT INTO sensors (type, location) VALUES (%s, %s);",
                    (sensor[0], sensor[1]))
      except (Exception, psycopg2.Error) as error:
        print(error.pgerror)
    conn.commit()
    ```

1.  [](#)Alternatively, you can pass variables to the `cursor.execute`
    function and separate the formulation of the SQL statement, `SQL`, from the
    data being passed with it into the prepared statement, `data`:

    ```python
    SQL = "INSERT INTO sensors (type, location) VALUES (%s, %s);"
    sensors = [('a', 'floor'), ('a', 'ceiling'), ('b', 'floor'), ('b', 'ceiling')]
    cursor = conn.cursor()
    for sensor in sensors:
      try:
        data = (sensor[0], sensor[1])
        cursor.execute(SQL, data)
      except (Exception, psycopg2.Error) as error:
        print(error.pgerror)
    conn.commit()
    ```

If you choose to use `pgcopy` instead, install the `pgcopy` package
[using pip][pgcopy-install], and then add this line to your list of
`import` statements:

```python
from pgcopy import CopyManager
```

1.  Generate some random sensor data using the `generate_series` function
    provided by Postgres. This example inserts a total of 480 rows of data (4
    readings, every 5 minutes, for 24 hours). In your application, this would be
    the query that saves your time-series data into the hypertable:

    ```python
    # for sensors with ids 1-4
    for id in range(1, 4, 1):
        data = (id,)
        # create random data
        simulate_query = """SELECT generate_series(now() - interval '24 hour', now(), interval '5 minute') AS time,
                                %s as sensor_id,
                                random()*100 AS temperature,
                                random() AS cpu;
                                """
        cursor.execute(simulate_query, data)
        values = cursor.fetchall()
    ```

1.  Define the column names of the table you want to insert data into. This
    example uses the `sensor_data` hypertable created earlier. This hypertable
    consists of columns named `time`, `sensor_id`, `temperature` and `cpu`. The
    column names are defined in a list of strings called `cols`:

    ```python
    cols = ['time', 'sensor_id', 'temperature', 'cpu']
    ```

1.  Create an instance of the `pgcopy` CopyManager, `mgr`, and pass the
    connection variable, hypertable name, and list of column names. Then use the
    `copy` function of the CopyManager to insert the data into the database
    quickly using `pgcopy`.

    ```python
    mgr = CopyManager(conn, 'sensor_data', cols)
    mgr.copy(values)
    ```

1.  Commit to persist changes:

    ```python
    conn.commit()
    ```

1.  [](#)The full sample code to insert data into TimescaleDB using
    `pgcopy`, using the example of sensor data from four sensors:

    ```python
    # insert using pgcopy
    def fast_insert(conn):
        cursor = conn.cursor()

        # for sensors with ids 1-4
        for id in range(1, 4, 1):
            data = (id,)
            # create random data
            simulate_query = """SELECT generate_series(now() - interval '24 hour', now(), interval '5 minute') AS time,
                                    %s as sensor_id,
                                    random()*100 AS temperature,
                                    random() AS cpu;
                                    """
            cursor.execute(simulate_query, data)
            values = cursor.fetchall()

            # column names of the table you're inserting into
            cols = ['time', 'sensor_id', 'temperature', 'cpu']

            # create copy manager with the target table and insert
            mgr = CopyManager(conn, 'sensor_data', cols)
            mgr.copy(values)

        # commit after all sensor data is inserted
        # could also commit after each sensor insert is done
        conn.commit()
    ```

1.  [](#)You can also check if the insertion worked:

    ```python
    cursor.execute("SELECT * FROM sensor_data LIMIT 5;")
    print(cursor.fetchall())
    ```

## Execute a query

This section covers how to execute queries against your database.

The first procedure shows a simple `SELECT *` query. For more complex queries,
you can use prepared statements to ensure queries are executed safely against
the database.

For more information about properly using placeholders in `psycopg2`, see the
[basic module usage document][psycopg2-docs-basics].
For more information about how to execute more complex queries in `psycopg2`,
see the [psycopg2 documentation][psycopg2-docs-basics].

### Execute a query

1.  Define the SQL query you'd like to run on the database. This example is a
    simple `SELECT` statement querying each row from the previously created
    `sensor_data` table.

    ```python
    query = "SELECT * FROM sensor_data;"
    ```

1.  Open a cursor from the existing database connection, `conn`, and then execute
   the query you defined:

    ```python
    cursor = conn.cursor()
    query = "SELECT * FROM sensor_data;"
    cursor.execute(query)
    ```

1.  To access all resulting rows returned by your query, use one of `pyscopg2`'s
    [results retrieval methods][results-retrieval-methods],
    such as `fetchall()` or `fetchmany()`. This example prints the results of
    the query, row by row. Note that the result of `fetchall()` is a list of
    tuples, so you can handle them accordingly:

    ```python
    cursor = conn.cursor()
    query = "SELECT * FROM sensor_data;"
    cursor.execute(query)
    for row in cursor.fetchall():
        print(row)
    cursor.close()
    ```

1.  [](#)If you want a list of dictionaries instead, you can define the
    cursor using [`DictCursor`][dictcursor-docs]:

    ```python
    cursor = conn.cursor(cursor_factory=psycopg2.extras.DictCursor)
    ```

    Using this cursor, `cursor.fetchall()` returns a list of dictionary-like objects.

For more complex queries, you can use prepared statements to ensure queries are
executed safely against the database.

### Execute queries using prepared statements

1.  Write the query using prepared statements:

    ```python
    # query with placeholders
    cursor = conn.cursor()
    query = """
               SELECT time_bucket('5 minutes', time) AS five_min, avg(cpu)
               FROM sensor_data
               JOIN sensors ON sensors.id = sensor_data.sensor_id
               WHERE sensors.location = %s AND sensors.type = %s
               GROUP BY five_min
               ORDER BY five_min DESC;
               """
    location = "floor"
    sensor_type = "a"
    data = (location, sensor_type)
    cursor.execute(query, data)
    results = cursor.fetchall()
    ```




## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

*   Install [Node.js][node-install].
*   Install the Node.js package manager [npm][npm-install].

## Connect to TimescaleDB

In this section, you create a connection to TimescaleDB  with a common Node.js
ORM (object relational mapper) called [Sequelize][sequelize-info].

1.  At the command prompt, initialize a new Node.js app:

    ```bash
    npm init -y
    ```

    This creates a `package.json` file in your directory, which contains all
    of the dependencies for your project. It looks something like this:

    ```json
    {
      "name": "node-sample",
      "version": "1.0.0",
      "description": "",
      "main": "index.js",
      "scripts": {
        "test": "echo \"Error: no test specified\" && exit 1"
      },
      "keywords": [],
      "author": "",
      "license": "ISC"
    }
    ```

1.  Install Express.js:

    ```bash
    npm install express
    ```

1.  Create a simple web page to check the connection. Create a new file called
    `index.js`, with this content:

    ```java
    const express = require('express')
    const app = express()
    const port = 3000;

    app.use(express.json());
    app.get('/', (req, res) => res.send('Hello World!'))
    app.listen(port, () => console.log(`Example app listening at http://localhost:${port}`))
    ```

1.  Test your connection by starting the application:

    ```bash
    node index.js
    ```

   In your web browser, navigate to `http://localhost:3000`. If the connection
   is successful, it shows "Hello World!"

1.  Add Sequelize to your project:

    ```bash
    npm install sequelize sequelize-cli pg pg-hstore
    ```

1.  Locate your TimescaleDB  credentials and use them to compose a connection
   string for Sequelize.

    You'll need:

      *   password
      *   username
      *   host URL
      *   port
      *   database name

1.  Compose your connection string variable, using this format:

    ```java
    'postgres://<user>:<password>@<host>:<port>/<dbname>'
    ```

1.  Open the `index.js` file you created. Require Sequelize in the application,
    and declare the connection string:

    ```java
    const Sequelize = require('sequelize')
    const sequelize = new Sequelize('postgres://<user>:<password>@<host>:<port>/<dbname>',
        {
            dialect: 'postgres',
            protocol: 'postgres',
            dialectOptions: {
                ssl: {
                    require: true,
                    rejectUnauthorized: false
                }
            }
        })
    ```

    Make sure you add the SSL settings in the `dialectOptions` sections. You
    can't connect to TimescaleDB  using SSL without them.

1.  You can test the connection by adding these lines to `index.js` after the
    `app.get` statement:

    ```java
    sequelize.authenticate().then(() => {
        console.log('Connection has been established successfully.');
    }).catch(err => {
        console.error('Unable to connect to the database:', err);
    });
    ```

    Start the application on the command line:

    ```bash
    node index.js
    ```

    If the connection is successful, you'll get output like this:

    ```bash
    Example app listening at http://localhost:3000
    Executing (default): SELECT 1+1 AS result
    Connection has been established successfully.
    ```

## Create a relational table

In this section, you create a relational table called `page_loads`.

1.  Use the Sequelize command line tool to create a table and model called `page_loads`:

    ```bash
    npx sequelize model:generate --name page_loads \
    --attributes userAgent:string,time:date
    ```

    The output looks similar to this:

    ```bash
    Sequelize CLI [Node: 12.16.2, CLI: 5.5.1, ORM: 5.21.11]

    New model was created at <PATH>.
    New migration was created at <PATH>.
    ```

1.  Edit the migration file so that it sets up a migration key:

    ```java
    'use strict';
    module.exports = {
      up: async (queryInterface, Sequelize) => {
        await queryInterface.createTable('page_loads', {
          userAgent: {
            primaryKey: true,
            type: Sequelize.STRING
          },
          time: {
            primaryKey: true,
            type: Sequelize.DATE
          }
        });
      },
      down: async (queryInterface, Sequelize) => {
        await queryInterface.dropTable('page_loads');
      }
    };
    ```

1.  Migrate the change and make sure that it is reflected in the database:

    ```bash
    npx sequelize db:migrate
    ```

    The output looks similar to this:

    ```bash
    Sequelize CLI [Node: 12.16.2, CLI: 5.5.1, ORM: 5.21.11]

    Loaded configuration file "config/config.json".
    Using environment "development".
    == 20200528195725-create-page-loads: migrating =======
    == 20200528195725-create-page-loads: migrated (0.443s)
    ```

1.  Create the `PageLoads` model in your code. In the `index.js` file, above the
    `app.use` statement, add these lines:

    ```java
    let PageLoads = sequelize.define('page_loads', {
        userAgent: {type: Sequelize.STRING, primaryKey: true },
        time: {type: Sequelize.DATE, primaryKey: true }
    }, { timestamps: false });
    ```

1.  Instantiate a `PageLoads` object and save it to the database.

## Create a hypertable

When you have created the relational table, you can create a hypertable.
Creating tables and indexes, altering tables, inserting data, selecting data,
and most other tasks are executed on the hypertable.

1.  Create a migration to modify the `page_loads` relational table, and change
    it to a hypertable by first running the following command:

    ```bash
    npx sequelize migration:generate --name add_hypertable
    ```

    The output looks similar to this:

    ```bash
    Sequelize CLI [Node: 12.16.2, CLI: 5.5.1, ORM: 5.21.11]

    migrations folder at <PATH> already exists.
    New migration was created at <PATH>/20200601202912-add_hypertable.js .
    ```

1.  In the `migrations` folder, there is now a new file. Open the
    file, and add this content:

    ```js
    'use strict';

    module.exports = {
      up: (queryInterface, Sequelize) => {
        return queryInterface.sequelize.query("SELECT create_hypertable('page_loads', by_range('time'));");
      },

      down: (queryInterface, Sequelize) => {
      }
    };
    ```



	The `by_range` dimension builder is an addition to TimescaleDB  2.13.



1.  At the command prompt, run the migration command:

    ```bash
    npx sequelize db:migrate
    ```

    The output looks similar to this:

    ```bash
    Sequelize CLI [Node: 12.16.2, CLI: 5.5.1, ORM: 5.21.11]

    Loaded configuration file "config/config.json".
    Using environment "development".
    == 20200601202912-add_hypertable: migrating =======
    == 20200601202912-add_hypertable: migrated (0.426s)
    ```

## Insert rows of data

This section covers how to insert data into your hypertables.

1.  In the `index.js` file, modify the `/` route to get the `user-agent` from
    the request object (`req`) and the current timestamp. Then, call the
    `create` method on `PageLoads` model, supplying the user agent and timestamp
    parameters. The `create` call executes an `INSERT` on the database:

    ```java
    app.get('/', async (req, res) => {
        // get the user agent and current time
        const userAgent = req.get('user-agent');
        const time = new Date().getTime();

        try {
            // insert the record
            await PageLoads.create({
                userAgent, time
            });

            // send response
            res.send('Inserted!');
        } catch (e) {
            console.log('Error inserting data', e)
        }
    })
    ```

## Execute a query

This section covers how to execute queries against your database. In this
example, every time the page is reloaded, all information currently in the table
is displayed.

1.  Modify the `/` route in the `index.js` file to call the Sequelize `findAll`
    function and retrieve all data from the `page_loads` table using the
    `PageLoads` model:

    ```java
    app.get('/', async (req, res) => {
        // get the user agent and current time
        const userAgent = req.get('user-agent');
        const time = new Date().getTime();

        try {
            // insert the record
            await PageLoads.create({
                userAgent, time
            });

            // now display everything in the table
            const messages = await PageLoads.findAll();
            res.send(messages);
        } catch (e) {
            console.log('Error inserting data', e)
        }
    })
    ```

Now, when you reload the page, you should see all of the rows currently in the
`page_loads` table.




## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

- Install [Go][golang-install].
- Install the [PGX driver for Go][pgx-driver-github].

## Connect to your Tiger Cloud service

In this section, you create a connection to Tiger Cloud using the PGX driver.
PGX is a toolkit designed to help Go developers work directly with Postgres.
You can use it to help your Go application interact directly with TimescaleDB.

1.  Locate your TimescaleDB credentials and use them to compose a connection
    string for PGX.

    You'll need:

    *   password
    *   username
    *   host URL
    *   port number
    *   database name

1.  Compose your connection string variable as a
    [libpq connection string][libpq-docs], using this format:

    ```go
    connStr := "postgres://username:password@host:port/dbname"
    ```

    If you're using a hosted version of TimescaleDB, or if you need an SSL
    connection, use this format instead:

    ```go
    connStr := "postgres://username:password@host:port/dbname?sslmode=require"
    ```

1.  [](#)You can check that you're connected to your database with this
    hello world program:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"

        "github.com/jackc/pgx/v5"
    )

    //connect to database using a single connection
    func main() {
        /***********************************************/
        /* Single Connection to TimescaleDB/ PostgreSQL */
        /***********************************************/
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        conn, err := pgx.Connect(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer conn.Close(ctx)

        //run a simple query to check our connection
        var greeting string
        err = conn.QueryRow(ctx, "select 'Hello, Timescale!'").Scan(&greeting)
        if err != nil {
            fmt.Fprintf(os.Stderr, "QueryRow failed: %v\n", err)
            os.Exit(1)
        }
        fmt.Println(greeting)
    }

    ```

    If you'd like to specify your connection string as an environment variable,
    you can use this syntax to access it in place of the `connStr` variable:

    ```go
    os.Getenv("DATABASE_CONNECTION_STRING")
    ```

Alternatively, you can connect to TimescaleDB using a connection pool.
Connection pooling is useful to conserve computing resources, and can also
result in faster database queries:

1.  To create a connection pool that can be used for concurrent connections to
   your database, use the `pgxpool.New()` function instead of
   `pgx.Connect()`. Also note that this script imports
   `github.com/jackc/pgx/v5/pgxpool`, instead of `pgx/v5` which was used to
   create a single connection:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {

        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        //run a simple query to check our connection
        var greeting string
        err = dbpool.QueryRow(ctx, "select 'Hello, Tiger Data (but concurrently)'").Scan(&greeting)
        if err != nil {
            fmt.Fprintf(os.Stderr, "QueryRow failed: %v\n", err)
            os.Exit(1)
        }
        fmt.Println(greeting)
    }
    ```

## Create a relational table

In this section, you create a table called `sensors` which holds the ID, type,
and location of your fictional sensors. Additionally, you create a hypertable
called `sensor_data` which holds the measurements of those sensors. The
measurements contain the time, sensor_id, temperature reading, and CPU
percentage of the sensors.

1.  Compose a string that contains the SQL statement to create a relational
    table. This example creates a table called `sensors`, with columns for ID,
    type, and location:

    ```go
    queryCreateTable := `CREATE TABLE sensors (id SERIAL PRIMARY KEY, type VARCHAR(50), location VARCHAR(50));`
    ```

1.  Execute the `CREATE TABLE` statement with the `Exec()` function on the
    `dbpool` object, using the arguments of the current context and the
    statement string you created:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        /********************************************/
        /* Create relational table                      */
        /********************************************/

        //Create relational table called sensors
        queryCreateTable := `CREATE TABLE sensors (id SERIAL PRIMARY KEY, type VARCHAR(50), location VARCHAR(50));`
        _, err = dbpool.Exec(ctx, queryCreateTable)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to create SENSORS table: %v\n", err)
            os.Exit(1)
        }
        fmt.Println("Successfully created relational table SENSORS")
    }
    ```

## Generate a hypertable

When you have created the relational table, you can create a hypertable.
Creating tables and indexes, altering tables, inserting data, selecting data,
and most other tasks are executed on the hypertable.

1.  Create a variable for the `CREATE TABLE SQL` statement for your hypertable.
    Notice how the hypertable has the compulsory time column:

    ```go
    queryCreateTable := `CREATE TABLE sensor_data (
            time TIMESTAMPTZ NOT NULL,
            sensor_id INTEGER,
            temperature DOUBLE PRECISION,
            cpu DOUBLE PRECISION,
            FOREIGN KEY (sensor_id) REFERENCES sensors (id));
            `
    ```

1.  Formulate the `SELECT` statement to convert the table into a hypertable. You
    must specify the table name to convert to a hypertable, and its time column
    name as the second argument. For more information, see the
    [`create_hypertable` docs][create-hypertable-docs]:

    ```go
    queryCreateHypertable := `SELECT create_hypertable('sensor_data', by_range('time'));`
    ```



	The `by_range` dimension builder is an addition to TimescaleDB 2.13.



1.  Execute the `CREATE TABLE` statement and `SELECT` statement which converts
    the table into a hypertable. You can do this by calling the `Exec()`
    function on the `dbpool` object, using the arguments of the current context,
    and the `queryCreateTable` and `queryCreateHypertable` statement strings:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        /********************************************/
        /* Create Hypertable                        */
        /********************************************/
        // Create hypertable of time-series data called sensor_data
        queryCreateTable := `CREATE TABLE sensor_data (
            time TIMESTAMPTZ NOT NULL,
            sensor_id INTEGER,
            temperature DOUBLE PRECISION,
            cpu DOUBLE PRECISION,
            FOREIGN KEY (sensor_id) REFERENCES sensors (id));
            `

        queryCreateHypertable := `SELECT create_hypertable('sensor_data', by_range('time'));`

        //execute statement
        _, err = dbpool.Exec(ctx, queryCreateTable+queryCreateHypertable)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to create the `sensor_data` hypertable: %v\n", err)
            os.Exit(1)
        }
        fmt.Println("Successfully created hypertable `sensor_data`")
    }
    ```

## Insert rows of data

You can insert rows into your database in a couple of different
ways. Each of these example inserts the data from the two arrays, `sensorTypes` and
`sensorLocations`, into the relational table named `sensors`.

The first example inserts a single row of data at a time. The second example
inserts multiple rows of data. The third example uses batch inserts to speed up
the process.

1.  Open a connection pool to the database, then use the prepared statements to
    formulate an `INSERT` SQL statement, and execute it:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        /********************************************/
        /* INSERT into  relational table            */
        /********************************************/
        //Insert data into relational table

        // Slices of sample data to insert
        // observation i has type sensorTypes[i] and location sensorLocations[i]
        sensorTypes := []string{"a", "a", "b", "b"}
        sensorLocations := []string{"floor", "ceiling", "floor", "ceiling"}

        for i := range sensorTypes {
            //INSERT statement in SQL
            queryInsertMetadata := `INSERT INTO sensors (type, location) VALUES ($1, $2);`

            //Execute INSERT command
            _, err := dbpool.Exec(ctx, queryInsertMetadata, sensorTypes[i], sensorLocations[i])
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to insert data into database: %v\n", err)
                os.Exit(1)
            }
            fmt.Printf("Inserted sensor (%s, %s) into database \n", sensorTypes[i], sensorLocations[i])
        }
        fmt.Println("Successfully inserted all sensors into database")
    }
    ```

Instead of inserting a single row of data at a time, you can use this procedure
to insert multiple rows of data, instead:

1.  This example uses Postgres to generate some sample time-series to insert
    into the `sensor_data` hypertable. Define the SQL statement to generate the
    data, called `queryDataGeneration`. Then use the `.Query()` function to
    execute the statement and return the sample data. The data returned by the
    query is stored in `results`, a slice of structs, which is then used as a
    source to insert data into the hypertable:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"
        "time"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        // Generate data to insert

        //SQL query to generate sample data
        queryDataGeneration := `
            SELECT generate_series(now() - interval '24 hour', now(), interval '5 minute') AS time,
            floor(random() * (3) + 1)::int as sensor_id,
            random()*100 AS temperature,
            random() AS cpu
            `
        //Execute query to generate samples for sensor_data hypertable
        rows, err := dbpool.Query(ctx, queryDataGeneration)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to generate sensor data: %v\n", err)
            os.Exit(1)
        }
        defer rows.Close()

        fmt.Println("Successfully generated sensor data")

        //Store data generated in slice results
        type result struct {
            Time        time.Time
            SensorId    int
            Temperature float64
            CPU         float64
        }

        var results []result
        for rows.Next() {
            var r result
            err = rows.Scan(&r.Time, &r.SensorId, &r.Temperature, &r.CPU)
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to scan %v\n", err)
                os.Exit(1)
            }
            results = append(results, r)
        }

        // Any errors encountered by rows.Next or rows.Scan are returned here
        if rows.Err() != nil {
            fmt.Fprintf(os.Stderr, "rows Error: %v\n", rows.Err())
            os.Exit(1)
        }

        // Check contents of results slice
        fmt.Println("Contents of RESULTS slice")
        for i := range results {
            var r result
            r = results[i]
            fmt.Printf("Time: %s | ID: %d | Temperature: %f | CPU: %f |\n", &r.Time, r.SensorId, r.Temperature, r.CPU)
        }
    }
    ```

1.  Formulate an SQL insert statement for the `sensor_data` hypertable:

    ```go
    //SQL query to generate sample data
    queryInsertTimeseriesData := `
        INSERT INTO sensor_data (time, sensor_id, temperature, cpu) VALUES ($1, $2, $3, $4);
        `
    ```

1.  Execute the SQL statement for each sample in the results slice:

    ```go
    //Insert contents of results slice into TimescaleDB
    for i := range results {
        var r result
        r = results[i]
        _, err := dbpool.Exec(ctx, queryInsertTimeseriesData, r.Time, r.SensorId, r.Temperature, r.CPU)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to insert sample into TimescaleDB %v\n", err)
            os.Exit(1)
        }
        defer rows.Close()
    }
    fmt.Println("Successfully inserted samples into sensor_data hypertable")
    ```

1.  [](#)This example `main.go` generates sample data and inserts it into
    the `sensor_data` hypertable:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"
        "time"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        /********************************************/
        /* Connect using Connection Pool            */
        /********************************************/
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        /********************************************/
        /* Insert data into hypertable              */
        /********************************************/
        // Generate data to insert

        //SQL query to generate sample data
        queryDataGeneration := `
            SELECT generate_series(now() - interval '24 hour', now(), interval '5 minute') AS time,
            floor(random() * (3) + 1)::int as sensor_id,
            random()*100 AS temperature,
            random() AS cpu
            `
        //Execute query to generate samples for sensor_data hypertable
        rows, err := dbpool.Query(ctx, queryDataGeneration)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to generate sensor data: %v\n", err)
            os.Exit(1)
        }
        defer rows.Close()

        fmt.Println("Successfully generated sensor data")

        //Store data generated in slice results
        type result struct {
            Time        time.Time
            SensorId    int
            Temperature float64
            CPU         float64
        }
        var results []result
        for rows.Next() {
            var r result
            err = rows.Scan(&r.Time, &r.SensorId, &r.Temperature, &r.CPU)
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to scan %v\n", err)
                os.Exit(1)
            }
            results = append(results, r)
        }
        // Any errors encountered by rows.Next or rows.Scan are returned here
        if rows.Err() != nil {
            fmt.Fprintf(os.Stderr, "rows Error: %v\n", rows.Err())
            os.Exit(1)
        }

        // Check contents of results slice
        fmt.Println("Contents of RESULTS slice")
        for i := range results {
            var r result
            r = results[i]
            fmt.Printf("Time: %s | ID: %d | Temperature: %f | CPU: %f |\n", &r.Time, r.SensorId, r.Temperature, r.CPU)
        }

        //Insert contents of results slice into TimescaleDB
        //SQL query to generate sample data
        queryInsertTimeseriesData := `
            INSERT INTO sensor_data (time, sensor_id, temperature, cpu) VALUES ($1, $2, $3, $4);
            `

        //Insert contents of results slice into TimescaleDB
        for i := range results {
            var r result
            r = results[i]
            _, err := dbpool.Exec(ctx, queryInsertTimeseriesData, r.Time, r.SensorId, r.Temperature, r.CPU)
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to insert sample into TimescaleDB %v\n", err)
                os.Exit(1)
            }
            defer rows.Close()
        }
        fmt.Println("Successfully inserted samples into sensor_data hypertable")
    }
    ```

Inserting multiple rows of data using this method executes as many `insert`
statements as there are samples to be inserted. This can make ingestion of data
slow. To speed up ingestion, you can batch insert data instead.

Here's a sample pattern for how to do so, using the sample data you generated in
the previous procedure. It uses the pgx `Batch` object:

1.  This example batch inserts data into the database:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"
        "time"

        "github.com/jackc/pgx/v5"
        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        /********************************************/
        /* Connect using Connection Pool            */
        /********************************************/
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        // Generate data to insert

        //SQL query to generate sample data
        queryDataGeneration := `
            SELECT generate_series(now() - interval '24 hour', now(), interval '5 minute') AS time,
            floor(random() * (3) + 1)::int as sensor_id,
            random()*100 AS temperature,
            random() AS cpu
            `

        //Execute query to generate samples for sensor_data hypertable
        rows, err := dbpool.Query(ctx, queryDataGeneration)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to generate sensor data: %v\n", err)
            os.Exit(1)
        }
        defer rows.Close()

        fmt.Println("Successfully generated sensor data")

        //Store data generated in slice results
        type result struct {
            Time        time.Time
            SensorId    int
            Temperature float64
            CPU         float64
        }
        var results []result
        for rows.Next() {
            var r result
            err = rows.Scan(&r.Time, &r.SensorId, &r.Temperature, &r.CPU)
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to scan %v\n", err)
                os.Exit(1)
            }
            results = append(results, r)
        }
        // Any errors encountered by rows.Next or rows.Scan are returned here
        if rows.Err() != nil {
            fmt.Fprintf(os.Stderr, "rows Error: %v\n", rows.Err())
            os.Exit(1)
        }

        // Check contents of results slice
        /*fmt.Println("Contents of RESULTS slice")
        for i := range results {
            var r result
            r = results[i]
            fmt.Printf("Time: %s | ID: %d | Temperature: %f | CPU: %f |\n", &r.Time, r.SensorId, r.Temperature, r.CPU)
        }*/

        //Insert contents of results slice into TimescaleDB
        //SQL query to generate sample data
        queryInsertTimeseriesData := `
            INSERT INTO sensor_data (time, sensor_id, temperature, cpu) VALUES ($1, $2, $3, $4);
            `

        /********************************************/
        /* Batch Insert into TimescaleDB            */
        /********************************************/
        //create batch
        batch := &pgx.Batch{}
        //load insert statements into batch queue
        for i := range results {
            var r result
            r = results[i]
            batch.Queue(queryInsertTimeseriesData, r.Time, r.SensorId, r.Temperature, r.CPU)
        }
        batch.Queue("select count(*) from sensor_data")

        //send batch to connection pool
        br := dbpool.SendBatch(ctx, batch)
        //execute statements in batch queue
        _, err = br.Exec()
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to execute statement in batch queue %v\n", err)
            os.Exit(1)
        }
        fmt.Println("Successfully batch inserted data")

        //Compare length of results slice to size of table
        fmt.Printf("size of results: %d\n", len(results))
        //check size of table for number of rows inserted
        // result of last SELECT statement
        var rowsInserted int
        err = br.QueryRow().Scan(&rowsInserted)
        fmt.Printf("size of table: %d\n", rowsInserted)

        err = br.Close()
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to closer batch %v\n", err)
            os.Exit(1)
        }
    }
    ```

## Execute a query

This section covers how to execute queries against your database.

1.  Define the SQL query you'd like to run on the database. This example uses a
    SQL query that combines time-series and relational data. It returns the
    average CPU values for every 5 minute interval, for sensors located on
    location `ceiling` and of type `a`:

    ```go
    // Formulate query in SQL
    // Note the use of prepared statement placeholders $1 and $2
    queryTimebucketFiveMin := `
        SELECT time_bucket('5 minutes', time) AS five_min, avg(cpu)
        FROM sensor_data
        JOIN sensors ON sensors.id = sensor_data.sensor_id
        WHERE sensors.location = $1 AND sensors.type = $2
        GROUP BY five_min
        ORDER BY five_min DESC;
        `
    ```

1.  Use the `.Query()` function to execute the query string. Make sure you
    specify the relevant placeholders:

    ```go
    //Execute query on TimescaleDB
    rows, err := dbpool.Query(ctx, queryTimebucketFiveMin, "ceiling", "a")
    if err != nil {
        fmt.Fprintf(os.Stderr, "Unable to execute query %v\n", err)
        os.Exit(1)
    }
    defer rows.Close()

    fmt.Println("Successfully executed query")
    ```

1.  Access the rows returned by `.Query()`. Create a struct with fields
    representing the columns that you expect to be returned, then use the
    `rows.Next()` function to iterate through the rows returned and fill
    `results` with the array of structs. This uses the `rows.Scan()` function,
    passing in pointers to the fields that you want to scan for results.

    This example prints out the results returned from the query, but you might
    want to use those results for some other purpose. Once you've scanned
    through all the rows returned you can then use the results array however you
    like.

    ```go
    //Do something with the results of query
    // Struct for results
    type result2 struct {
        Bucket time.Time
        Avg    float64
    }

    // Print rows returned and fill up results slice for later use
    var results []result2
    for rows.Next() {
        var r result2
        err = rows.Scan(&r.Bucket, &r.Avg)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to scan %v\n", err)
            os.Exit(1)
        }
        results = append(results, r)
        fmt.Printf("Time bucket: %s | Avg: %f\n", &r.Bucket, r.Avg)
    }

    // Any errors encountered by rows.Next or rows.Scan are returned here
    if rows.Err() != nil {
        fmt.Fprintf(os.Stderr, "rows Error: %v\n", rows.Err())
        os.Exit(1)
    }

    // use results here…
    ```

1.  [](#)This example program runs a query, and accesses the results of
    that query:

    ```go
    package main

    import (
        "context"
        "fmt"
        "os"
        "time"

        "github.com/jackc/pgx/v5/pgxpool"
    )

    func main() {
        ctx := context.Background()
        connStr := "yourConnectionStringHere"
        dbpool, err := pgxpool.New(ctx, connStr)
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to connect to database: %v\n", err)
            os.Exit(1)
        }
        defer dbpool.Close()

        /********************************************/
        /* Execute a query                          */
        /********************************************/

        // Formulate query in SQL
        // Note the use of prepared statement placeholders $1 and $2
        queryTimebucketFiveMin := `
            SELECT time_bucket('5 minutes', time) AS five_min, avg(cpu)
            FROM sensor_data
            JOIN sensors ON sensors.id = sensor_data.sensor_id
            WHERE sensors.location = $1 AND sensors.type = $2
            GROUP BY five_min
            ORDER BY five_min DESC;
            `

        //Execute query on TimescaleDB
        rows, err := dbpool.Query(ctx, queryTimebucketFiveMin, "ceiling", "a")
        if err != nil {
            fmt.Fprintf(os.Stderr, "Unable to execute query %v\n", err)
            os.Exit(1)
        }
        defer rows.Close()

        fmt.Println("Successfully executed query")

        //Do something with the results of query
        // Struct for results
        type result2 struct {
            Bucket time.Time
            Avg    float64
        }

        // Print rows returned and fill up results slice for later use
        var results []result2
        for rows.Next() {
            var r result2
            err = rows.Scan(&r.Bucket, &r.Avg)
            if err != nil {
                fmt.Fprintf(os.Stderr, "Unable to scan %v\n", err)
                os.Exit(1)
            }
            results = append(results, r)
            fmt.Printf("Time bucket: %s | Avg: %f\n", &r.Bucket, r.Avg)
        }
        // Any errors encountered by rows.Next or rows.Scan are returned here
        if rows.Err() != nil {
            fmt.Fprintf(os.Stderr, "rows Error: %v\n", rows.Err())
            os.Exit(1)
        }
    }
    ```

## Next steps

Now that you're able to connect, read, and write to a TimescaleDB instance from
your Go application, be sure to check out these advanced TimescaleDB tutorials:

*   Refer to the [pgx documentation][pgx-docs] for more information about pgx.
*   Get up and running with TimescaleDB with the [Getting Started][getting-started]
    tutorial.
*   Want fast inserts on CSV data? Check out
    [TimescaleDB parallel copy][parallel-copy-tool], a tool for fast inserts,
    written in Go.




## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

*   Install the [Java Development Kit (JDK)][jdk].
*   Install the [PostgreSQL JDBC driver][pg-jdbc-driver].

All code in this quick start is for Java 16 and later. If you are working
with older JDK versions, use legacy coding techniques.

## Connect to your Tiger Cloud service

In this section, you create a connection to your service using an application in
a single file. You can use any of your favorite build tools, including `gradle`
or `maven`.

1.  Create a directory containing a text file called `Main.java`, with this content:

    ```java
    package com.timescale.java;

    public class Main {

        public static void main(String... args) {
            System.out.println("Hello, World!");
        }
    }
    ```

1.  From the command line in the current directory, run the application:

    ```bash
    java Main.java
    ```

    If the command is successful, `Hello, World!` line output is printed
    to your console.

1.  Import the PostgreSQL JDBC driver. If you are using a dependency manager,
   include the [PostgreSQL JDBC Driver][pg-jdbc-driver-dependency] as a
   dependency.

1.  Download the [JAR artifact of the JDBC Driver][pg-jdbc-driver-artifact] and
   save it with the `Main.java` file.

1.  Import the `JDBC Driver` into the Java application and display a list of
   available drivers for the check:

    ```java
    package com.timescale.java;

    import java.sql.DriverManager;

    public class Main {

        public static void main(String... args) {
            DriverManager.drivers().forEach(System.out::println);
        }
    }
    ```

1.  Run all the examples:

    ```bash
    java -cp *.jar Main.java
    ```

   If the command is successful, a string similar to
   `org.postgresql.Driver@7f77e91b` is printed to your console. This means that you
   are ready to connect to TimescaleDB from Java.

1.  Locate your TimescaleDB credentials and use them to compose a connection
   string for JDBC.

    You'll need:

      *   password
      *   username
      *   host URL
      *   port
      *   database name

1.  Compose your connection string variable, using this format:

    ```java
    var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
    ```

    For more information about creating connection strings, see the [JDBC documentation][pg-jdbc-driver-conn-docs].



    This method of composing a connection string is for test or development
    purposes only. For production, use environment variables for sensitive
    details like your password, hostname, and port number.



    ```java
    package com.timescale.java;

    import java.sql.DriverManager;
    import java.sql.SQLException;

    public class Main {

        public static void main(String... args) throws SQLException {
            var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
            var conn = DriverManager.getConnection(connUrl);
            System.out.println(conn.getClientInfo());
        }
    }
    ```

1.  Run the code:

    ```bash
    java -cp *.jar Main.java
    ```

    If the command is successful, a string similar to
    `{ApplicationName=PostgreSQL JDBC Driver}` is printed to your console.

## Create a relational table

In this section, you create a table called `sensors` which holds the ID, type,
and location of your fictional sensors. Additionally, you create a hypertable
called `sensor_data` which holds the measurements of those sensors. The
measurements contain the time, sensor_id, temperature reading, and CPU
percentage of the sensors.

1.  Compose a string which contains the SQL statement to create a relational
    table. This example creates a table called `sensors`, with columns `id`,
    `type` and `location`:

    ```sql
    CREATE TABLE sensors (
        id SERIAL PRIMARY KEY,
        type TEXT NOT NULL,
        location TEXT NOT NULL
    );
    ```

1.  Create a statement, execute the query you created in the previous step, and
    check that the table was created successfully:

    ```java
    package com.timescale.java;

    import java.sql.DriverManager;
    import java.sql.SQLException;

    public class Main {

        public static void main(String... args) throws SQLException {
            var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
            var conn = DriverManager.getConnection(connUrl);

            var createSensorTableQuery = """
                    CREATE TABLE sensors (
                        id SERIAL PRIMARY KEY,
                        type TEXT NOT NULL,
                        location TEXT NOT NULL
                    )
                    """;
            try (var stmt = conn.createStatement()) {
                stmt.execute(createSensorTableQuery);
            }

            var showAllTablesQuery = "SELECT tablename FROM pg_catalog.pg_tables WHERE schemaname = 'public'";
            try (var stmt = conn.createStatement();
                 var rs = stmt.executeQuery(showAllTablesQuery)) {
                System.out.println("Tables in the current database: ");
                while (rs.next()) {
                    System.out.println(rs.getString("tablename"));
                }
            }
        }
    }
    ```

## Create a hypertable

When you have created the relational table, you can create a hypertable.
Creating tables and indexes, altering tables, inserting data, selecting data,
and most other tasks are executed on the hypertable.

1.  Create a `CREATE TABLE` SQL statement for
    your hypertable. Notice how the hypertable has the compulsory time column:

    ```sql
    CREATE TABLE sensor_data (
        time TIMESTAMPTZ NOT NULL,
        sensor_id INTEGER REFERENCES sensors (id),
        value DOUBLE PRECISION
    );
    ```

1.  Create a statement, execute the query you created in the previous step:

    ```sql
    SELECT create_hypertable('sensor_data', by_range('time'));
    ```



	The `by_range` and `by_hash` dimension builder is an addition to TimescaleDB 2.13.



1.  Execute the two statements you created, and commit your changes to the
    database:

    ```java
    package com.timescale.java;

    import java.sql.Connection;
    import java.sql.DriverManager;
    import java.sql.SQLException;
    import java.util.List;

    public class Main {

        public static void main(String... args) {
            final var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
            try (var conn = DriverManager.getConnection(connUrl)) {
                createSchema(conn);
                insertData(conn);
            } catch (SQLException ex) {
                System.err.println(ex.getMessage());
            }
        }

        private static void createSchema(final Connection conn) throws SQLException {
            try (var stmt = conn.createStatement()) {
                stmt.execute("""
                        CREATE TABLE sensors (
                            id SERIAL PRIMARY KEY,
                            type TEXT NOT NULL,
                            location TEXT NOT NULL
                        )
                        """);
            }

            try (var stmt = conn.createStatement()) {
                stmt.execute("""
                        CREATE TABLE sensor_data (
                            time TIMESTAMPTZ NOT NULL,
                            sensor_id INTEGER REFERENCES sensors (id),
                            value DOUBLE PRECISION
                        )
                        """);
            }

            try (var stmt = conn.createStatement()) {
                stmt.execute("SELECT create_hypertable('sensor_data', by_range('time'))");
            }
        }
    }
    ```

## Insert data

You can insert data into your hypertables in several different ways. In this
section, you can insert single rows, or insert by batches of rows.

1.  Open a connection to the database, use prepared statements to formulate the
    `INSERT` SQL statement, then execute the statement:

    ```java
    final List<Sensor> sensors = List.of(
            new Sensor("temperature", "bedroom"),
            new Sensor("temperature", "living room"),
            new Sensor("temperature", "outside"),
            new Sensor("humidity", "kitchen"),
            new Sensor("humidity", "outside"));
    for (final var sensor : sensors) {
        try (var stmt = conn.prepareStatement("INSERT INTO sensors (type, location) VALUES (?, ?)")) {
            stmt.setString(1, sensor.type());
            stmt.setString(2, sensor.location());
            stmt.executeUpdate();
        }
    }
    ```

If you want to insert a batch of rows by using a batching mechanism. In this
example, you generate some sample time-series data to insert into the
`sensor_data` hypertable:

1.  Insert batches of rows:

    ```java
    final var sensorDataCount = 100;
    final var insertBatchSize = 10;
    try (var stmt = conn.prepareStatement("""
            INSERT INTO sensor_data (time, sensor_id, value)
            VALUES (
                generate_series(now() - INTERVAL '24 hours', now(), INTERVAL '5 minutes'),
                floor(random() * 4 + 1)::INTEGER,
                random()
            )
            """)) {
        for (int i = 0; i < sensorDataCount; i++) {
            stmt.addBatch();

            if ((i > 0 && i % insertBatchSize == 0) || i == sensorDataCount - 1) {
                stmt.executeBatch();
            }
        }
    }
    ```

## Execute a query

This section covers how to execute queries against your database.

## Execute queries on TimescaleDB

1.  Define the SQL query you'd like to run on the database. This example
    combines time-series and relational data. It returns the average values for
    every 15 minute interval for sensors with specific type and location.

    ```sql
    SELECT time_bucket('15 minutes', time) AS bucket, avg(value)
    FROM sensor_data
    JOIN sensors ON sensors.id = sensor_data.sensor_id
    WHERE sensors.type = ? AND sensors.location = ?
    GROUP BY bucket
    ORDER BY bucket DESC;
    ```

1.  Execute the query with the prepared statement and read out the result set for
   all `a`-type sensors located on the `floor`:

    ```java
    try (var stmt = conn.prepareStatement("""
            SELECT time_bucket('15 minutes', time) AS bucket, avg(value)
            FROM sensor_data
            JOIN sensors ON sensors.id = sensor_data.sensor_id
            WHERE sensors.type = ? AND sensors.location = ?
            GROUP BY bucket
            ORDER BY bucket DESC
            """)) {
        stmt.setString(1, "temperature");
        stmt.setString(2, "living room");

        try (var rs = stmt.executeQuery()) {
            while (rs.next()) {
                System.out.printf("%s: %f%n", rs.getTimestamp(1), rs.getDouble(2));
            }
        }
    }
    ```

    If the command is successful, you'll see output like this:

    ```bash
    2021-05-12 23:30:00.0: 0,508649
    2021-05-12 23:15:00.0: 0,477852
    2021-05-12 23:00:00.0: 0,462298
    2021-05-12 22:45:00.0: 0,457006
    2021-05-12 22:30:00.0: 0,568744
    ...
    ```

## Next steps

Now that you're able to connect, read, and write to a TimescaleDB instance from
your Java application, and generate the scaffolding necessary to build a new
application from an existing TimescaleDB instance, be sure to check out these
advanced TimescaleDB tutorials:

*   [Continuous Aggregates][continuous-aggregates]
*   [Migrate Your own Data][migrate]

## Complete code samples

This section contains complete code samples.

### Complete code sample

```java
package com.timescale.java;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.List;

public class Main {

    public static void main(String... args) {
        final var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
        try (var conn = DriverManager.getConnection(connUrl)) {
            createSchema(conn);
            insertData(conn);
        } catch (SQLException ex) {
            System.err.println(ex.getMessage());
        }
    }

    private static void createSchema(final Connection conn) throws SQLException {
        try (var stmt = conn.createStatement()) {
            stmt.execute("""
                    CREATE TABLE sensors (
                        id SERIAL PRIMARY KEY,
                        type TEXT NOT NULL,
                        location TEXT NOT NULL
                    )
                    """);
        }

        try (var stmt = conn.createStatement()) {
            stmt.execute("""
                    CREATE TABLE sensor_data (
                        time TIMESTAMPTZ NOT NULL,
                        sensor_id INTEGER REFERENCES sensors (id),
                        value DOUBLE PRECISION
                    )
                    """);
        }

        try (var stmt = conn.createStatement()) {
            stmt.execute("SELECT create_hypertable('sensor_data', by_range('time'))");
        }
    }

    private static void insertData(final Connection conn) throws SQLException {
        final List<Sensor> sensors = List.of(
                new Sensor("temperature", "bedroom"),
                new Sensor("temperature", "living room"),
                new Sensor("temperature", "outside"),
                new Sensor("humidity", "kitchen"),
                new Sensor("humidity", "outside"));
        for (final var sensor : sensors) {
            try (var stmt = conn.prepareStatement("INSERT INTO sensors (type, location) VALUES (?, ?)")) {
                stmt.setString(1, sensor.type());
                stmt.setString(2, sensor.location());
                stmt.executeUpdate();
            }
        }

        final var sensorDataCount = 100;
        final var insertBatchSize = 10;
        try (var stmt = conn.prepareStatement("""
                INSERT INTO sensor_data (time, sensor_id, value)
                VALUES (
                    generate_series(now() - INTERVAL '24 hours', now(), INTERVAL '5 minutes'),
                    floor(random() * 4 + 1)::INTEGER,
                    random()
                )
                """)) {
            for (int i = 0; i < sensorDataCount; i++) {
                stmt.addBatch();

                if ((i > 0 && i % insertBatchSize == 0) || i == sensorDataCount - 1) {
                    stmt.executeBatch();
                }
            }
        }
    }

    private record Sensor(String type, String location) {
    }
}
```

### Execute more complex queries

```java
package com.timescale.java;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.List;

public class Main {

    public static void main(String... args) {
        final var connUrl = "jdbc:postgresql://<HOSTNAME>:<PORT>/<DATABASE_NAME>?user=<USERNAME>&password=<PASSWORD>";
        try (var conn = DriverManager.getConnection(connUrl)) {
            createSchema(conn);
            insertData(conn);
            executeQueries(conn);
        } catch (SQLException ex) {
            System.err.println(ex.getMessage());
        }
    }

    private static void createSchema(final Connection conn) throws SQLException {
        try (var stmt = conn.createStatement()) {
            stmt.execute("""
                    CREATE TABLE sensors (
                        id SERIAL PRIMARY KEY,
                        type TEXT NOT NULL,
                        location TEXT NOT NULL
                    )
                    """);
        }

        try (var stmt = conn.createStatement()) {
            stmt.execute("""
                    CREATE TABLE sensor_data (
                        time TIMESTAMPTZ NOT NULL,
                        sensor_id INTEGER REFERENCES sensors (id),
                        value DOUBLE PRECISION
                    )
                    """);
        }

        try (var stmt = conn.createStatement()) {
            stmt.execute("SELECT create_hypertable('sensor_data', by_range('time'))");
        }
    }

    private static void insertData(final Connection conn) throws SQLException {
        final List<Sensor> sensors = List.of(
                new Sensor("temperature", "bedroom"),
                new Sensor("temperature", "living room"),
                new Sensor("temperature", "outside"),
                new Sensor("humidity", "kitchen"),
                new Sensor("humidity", "outside"));
        for (final var sensor : sensors) {
            try (var stmt = conn.prepareStatement("INSERT INTO sensors (type, location) VALUES (?, ?)")) {
                stmt.setString(1, sensor.type());
                stmt.setString(2, sensor.location());
                stmt.executeUpdate();
            }
        }

        final var sensorDataCount = 100;
        final var insertBatchSize = 10;
        try (var stmt = conn.prepareStatement("""
                INSERT INTO sensor_data (time, sensor_id, value)
                VALUES (
                    generate_series(now() - INTERVAL '24 hours', now(), INTERVAL '5 minutes'),
                    floor(random() * 4 + 1)::INTEGER,
                    random()
                )
                """)) {
            for (int i = 0; i < sensorDataCount; i++) {
                stmt.addBatch();

                if ((i > 0 && i % insertBatchSize == 0) || i == sensorDataCount - 1) {
                    stmt.executeBatch();
                }
            }
        }
    }

    private static void executeQueries(final Connection conn) throws SQLException {
        try (var stmt = conn.prepareStatement("""
                SELECT time_bucket('15 minutes', time) AS bucket, avg(value)
                FROM sensor_data
                JOIN sensors ON sensors.id = sensor_data.sensor_id
                WHERE sensors.type = ? AND sensors.location = ?
                GROUP BY bucket
                ORDER BY bucket DESC
                """)) {
            stmt.setString(1, "temperature");
            stmt.setString(2, "living room");

            try (var rs = stmt.executeQuery()) {
                while (rs.next()) {
                    System.out.printf("%s: %f%n", rs.getTimestamp(1), rs.getDouble(2));
                }
            }
        }
    }

    private record Sensor(String type, String location) {
    }
}
```







You are not limited to these languages. Tiger Cloud is based on Postgres, you can interface
with TimescaleDB and Tiger Cloud using any [Postgres client driver][postgres-drivers].


===== PAGE: https://docs.tigerdata.com/getting-started/services/ =====

# Create your first Tiger Cloud service



Tiger Cloud is the modern Postgres data platform for all your applications. It enhances Postgres to handle time series, events,
real-time analytics, and vector search—all in a single database alongside transactional workloads.

You get one system that handles live data ingestion, late and out-of-order updates, and low latency queries, with the performance, reliability, and scalability your app needs. Ideal for IoT, crypto, finance, SaaS, and a myriad other domains, Tiger Cloud allows you to build data-heavy, mission-critical apps while retaining the familiarity and reliability of Postgres.

## What is a Tiger Cloud service?

A Tiger Cloud service is a single optimised Postgres instance extended with innovations in the database engine and cloud
infrastructure to deliver speed without sacrifice. A Tiger Cloud service is 10-1000x faster at scale! It
is ideal for applications requiring strong data consistency, complex relationships, and advanced querying capabilities.
Get ACID compliance, extensive SQL support, JSON handling, and extensibility through custom functions, data types, and
extensions.

Each service is associated with a project in Tiger Cloud. Each project can have multiple services. Each user is a [member of one or more projects][rbac].

You create free and standard services in Tiger Cloud Console, depending on your [pricing plan][pricing-plans]. A free service comes at zero cost and gives you limited resources to get to know Tiger Cloud. Once you are ready to try out more advanced features, you can switch to a paid plan and convert your free service to a standard one.

![Tiger Cloud pricing plans](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-pricing.svg)

The Free pricing plan and services are currently in beta.

To the Postgres you know and love, Tiger Cloud adds the following capabilities:

- **Standard services**:

    - _Real-time analytics_: store and query [time-series data][what-is-time-series] at scale for
      real-time analytics and other use cases. Get faster time-based queries with hypertables, continuous aggregates, and columnar storage. Save money by compressing data into the columnstore, moving cold data to low-cost bottomless storage in Amazon S3, and deleting old data with automated policies.
    - _AI-focused_: build AI applications from start to scale. Get fast and accurate similarity search
      with the pgvector and pgvectorscale extensions.
    - _Hybrid applications_: get a full set of tools to develop applications that combine time-based data and AI.

  All standard Tiger Cloud services include the tooling you expect for production and developer environments: [live migration][live-migration],
  [automatic backups and PITR][automatic-backups], [high availability][high-availability], [read replicas][readreplica], [data forking][operations-forking], [connection pooling][connection-pooling], [tiered storage][data-tiering],
  [usage-based storage][how-plans-work], secure in-Tiger Cloud Console [SQL editing][in-console-editors], service [metrics][metrics]
  and [insights][insights],&nbsp;[streamlined maintenance][maintain-upgrade],&nbsp;and much more. Tiger Cloud continuously monitors your services and prevents common Postgres out-of-memory crashes.

- **Free services**:

  _Postgres with TimescaleDB and vector extensions_

  Free services offer limited resources and a basic feature scope, perfect to get to know Tiger Cloud in a development environment.

You manage your Tiger Cloud services and interact with your data in Tiger Cloud Console using the following modes:

| **Ops mode**                                                                                                                                                                                                                                                                                                                 | **Data mode**                                                                                                                                                                                                                                                                                                                                                   |
|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| ![Tiger Cloud Console ops mode][ops-mode]                                                                                                                                                                                                                                                                                                   | ![Tiger Cloud Console data mode][data-mode]                                                                                                                                                                                                                                                                                                                                    |
| **You use the ops mode to:**  <ul> <li>Ensure data security with high availability and read replicas</li> <li>Save money with columnstore compression and tiered storage</li> <li>Enable Postgres extensions to add extra functionality</li> <li>Increase security using VPCs</li> <li>Perform day-to-day administration</li> </ul> | **Powered by PopSQL, you use the data mode to:**  <ul> <li>Write queries with autocomplete</li> <li>Visualize data with charts and dashboards</li> <li>Schedule queries and dashboards for alerts or recurring reports</li> <li>Share queries and dashboards</li> <li>Interact with your data on auto-pilot with SQL assistant</li></ul> This feature is not available under the Free pricing plan.  |

To start using Tiger Cloud for your data:

1. [Create a Tiger Data account][create-an-account]: register to get access to Tiger Cloud Console as a centralized point to administer and interact with your data.
1. [Create a Tiger Cloud service][create-a-service]: that is, a Postgres database instance, powered by [TimescaleDB][timescaledb], built for production, and extended with cloud features like transparent data tiering to object storage.
1. [Connect to your Tiger Cloud service][connect-to-your-service]: to run queries, add and migrate your data from other sources.

## Create a Tiger Data account

You create a Tiger Data account to manage your services and data in a centralized and efficient manner in Tiger Cloud Console. From there, you can create and delete services, run queries, manage access and billing, integrate other services, contact support, and more.





You create a standalone account to manage Tiger Cloud as a separate unit in your infrastructure, which includes separate billing and invoicing.

To set up Tiger Cloud:

1. **Sign up for a 30-day free trial**

   Open [Sign up for Tiger Cloud][timescale-signup] and add your details, then click `Start your free trial`. You receive a confirmation email in your inbox.

1. **Confirm your email address**

    In the confirmation email, click the link supplied.

1. **Select the [pricing plan][pricing-plans]**

   You are now logged into Tiger Cloud Console. You can change the pricing plan to better accommodate your growing needs on the [`Billing` page][console-billing].





To have Tiger Cloud as a part of your AWS infrastructure, you create a Tiger Data account through AWS Marketplace. In this
case, Tiger Cloud is a line item in your AWS invoice.

To set up Tiger Cloud via AWS:

1. **Open [AWS Marketplace][aws-marketplace] and search for `Tiger Cloud`**

   You see two pricing options, [pay-as-you-go][aws-paygo] and [annual commit][aws-annual-commit].

1. **Select the pricing option that suits you and click `View purchase options`**

1. **Review and configure the purchase details, then click `Subscribe`**

1. **Click `Set up your account` at the top of the page**

   You are redirected to Tiger Cloud Console.

1. **Sign up for a 30-day free trial**

   Add your details, then click `Start your free trial`. If you want to link an existing Tiger Data account to AWS, log in with your existing credentials.

1. **Select the [pricing plan][pricing-plans]**

   You are now logged into Tiger Cloud Console. You can change the pricing plan later to better accommodate your growing needs on the [`Billing` page][console-billing].

1. **In `Confirm AWS Marketplace connection`, click `Connect`**

    Your Tiger Cloud and AWS accounts are now connected.

## Create a Tiger Cloud service

Now that you have an active Tiger Data account, you create and manage your services in Tiger Cloud Console. When you create a service, you effectively create a blank Postgres database with additional Tiger Cloud features available under your pricing plan. You then add or migrate your data into this database.

To create a free or standard service:

1. In the [service creation page][create-service], click `+ New service`.

   Follow the wizard to configure your service depending on its type.

1. Click `Create service`.

   Your service is constructed and ready to use in a few seconds.

1. Click `Download the config` and store the configuration information you need to connect to this service in a secure location.

   This file contains the passwords and configuration information you need to connect to your service using the
   Tiger Cloud Console data mode, from the command line, or using third-party database administration tools.

If you choose to go directly to the service overview, [Connect to your service][connect-to-your-service]
shows you how to connect.

## Connect to your service

To run queries and perform other operations, connect to your service:

1. **Check your service is running correctly**

   In [Tiger Cloud Console][services-portal], check that your service is marked as `Running`.

   ![Check service is running](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-services-view.png)

1. **Connect to your service**

   Connect using data mode or SQL editor in Tiger Cloud Console, or psql in the command line:





   This feature is not available under the Free pricing plan.

   1. In Tiger Cloud Console, toggle `Data`.

   1. Select your service in the connection drop-down in the top right.

      ![Select a connection](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-data-mode-connection-dropdown.png)

   1. Run a test query:

      ```sql
      SELECT CURRENT_DATE;
      ```

      This query gives you the current date, you have successfully connected to your service.

   And that is it, you are up and running. Enjoy developing with Tiger Data.





   1. In Tiger Cloud Console, select your service.

   1. Click `SQL editor`.

      ![Check a service is running](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-ops-mode-sql-editor.png)

   1. Run a test query:

      ```sql
      SELECT CURRENT_DATE;
      ```

      This query gives you the current date, you have successfully connected to your service.

   And that is it, you are up and running. Enjoy developing with Tiger Data.





   1. Install [psql][psql].

   1. Run the following command in the terminal using the service URL from the config file you have saved during service creation:

      ```
      psql "<your-service-url>"
      ```

   1. Run a test query:

      ```sql
      SELECT CURRENT_DATE;
      ```

      This query returns the current date. You have successfully connected to your service.

   And that is it, you are up and running. Enjoy developing with Tiger Data.





Quick recap. You:
- Manage your services in the [ops mode][portal-ops-mode] in Tiger Cloud Console: add read replicas and enable
  high availability, compress data into the columnstore, change parameters, and so on.
- Analyze your data in the [data mode][portal-data-mode] in Tiger Cloud Console: write queries with
  autocomplete, save them in folders, share them, create charts/dashboards, and much more.
- Store configuration and security information in your config file.

What next? [Try the key features offered by Tiger Data][try-timescale-features], see the [tutorials][tutorials],
interact with the data in your Tiger Cloud service using [your favorite programming language][connect-with-code], integrate
your Tiger Cloud service with a range of [third-party tools][integrations], plain old [Use Tiger Data products][use-timescale], or dive
into the [API reference][use-the-api].


===== PAGE: https://docs.tigerdata.com/getting-started/get-started-devops-as-code/ =====

# "DevOps as code with Tiger"



Tiger Data supplies a clean, programmatic control layer for Tiger Cloud. This includes RESTful APIs and CLI commands
that enable humans, machines, and AI agents easily provision, configure, and manage Tiger Cloud services programmatically.





Tiger CLI is a command-line interface that you use to manage Tiger Cloud resources
including VPCs, services, read replicas, and related infrastructure. Tiger CLI calls Tiger REST API to communicate with
Tiger Cloud.

This page shows you how to install and set up secure authentication for Tiger CLI, then create your first
service.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Data account][create-account].


## Install and configure Tiger CLI

1. **Install Tiger CLI**

   Use the terminal to install the CLI:




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    brew install --cask timescale/tap/tiger-cli
    ```





    ```shell
    curl -fsSL https://cli.tigerdata.com | sh
    ```





1. **Set up API credentials**

   1. Log Tiger CLI into your Tiger Data account:

      ```shell
      tiger auth login
      ```
      Tiger CLI opens Console in your browser. Log in, then click `Authorize`.

      You can have a maximum of 10 active client credentials. If you get an error, open [credentials][rest-api-credentials]
      and delete an unused credential.

   1. Select a Tiger Cloud project:

      ```terminaloutput
      Auth URL is: https://console.cloud.timescale.com/oauth/authorize?client_id=lotsOfURLstuff
      Opening browser for authentication...
      Select a project:

      > 1. Tiger Project (tgrproject)
      2. YourCompany (Company wide project) (cpnproject)
      3. YourCompany Department (dptproject)

      Use ↑/↓ arrows or number keys to navigate, enter to select, q to quit
      ```
      If only one project is associated with your account, this step is not shown.

      Where possible, Tiger CLI stores your authentication information in the system keychain/credential manager.
      If that fails, the credentials are stored in `~/.config/tiger/credentials` with restricted file permissions (600).
      By default, Tiger CLI stores your configuration in `~/.config/tiger/config.yaml`.

1. **Test your authenticated connection to Tiger Cloud by listing services**

    ```bash
    tiger service list
    ```

   This call returns something like:
    - No services:
      ```terminaloutput
      🏜️  No services found! Your project is looking a bit empty.
      🚀 Ready to get started? Create your first service with: tiger service create
      ```
    - One or more services:

      ```terminaloutput
      ┌────────────┬─────────────────────┬────────┬─────────────┬──────────────┬──────────────────┐
      │ SERVICE ID │        NAME         │ STATUS │    TYPE     │    REGION    │     CREATED      │
      ├────────────┼─────────────────────┼────────┼─────────────┼──────────────┼──────────────────┤
      │ tgrservice │ tiger-agent-service │ READY  │ TIMESCALEDB │ eu-central-1 │ 2025-09-25 16:09 │
      └────────────┴─────────────────────┴────────┴─────────────┴──────────────┴──────────────────┘
      ```


## Create your first Tiger Cloud service

Create a new Tiger Cloud service using Tiger CLI:

1. **Submit a service creation request**

   By default, Tiger CLI creates a service for you that matches your [pricing plan][pricing-plans]:
   * **Free plan**: shared CPU/memory and the `time-series` and `ai` capabilities
   * **Paid plan**: 0.5 CPU and 2 GB memory with the `time-series` capability
   ```shell
   tiger service create
   ```
   Tiger Cloud creates a Development environment for you. That is, no delete protection, high-availability, spooling or
   read replication. You see something like:
   ```terminaloutput
    🚀 Creating service 'db-11111' (auto-generated name)...
    ✅ Service creation request accepted!
    📋 Service ID: tgrservice
    🔐 Password saved to system keyring for automatic authentication
    🎯 Set service 'tgrservice' as default service.
    ⏳ Waiting for service to be ready (wait timeout: 30m0s)...
    🎉 Service is ready and running!
   🔌 Run 'tiger db connect' to connect to your new service
   ┌───────────────────┬──────────────────────────────────────────────────────────────────────────────────────────────────┐
   │     PROPERTY      │                                              VALUE                                               │
   ├───────────────────┼──────────────────────────────────────────────────────────────────────────────────────────────────┤
   │ Service ID        │ tgrservice                                                                                       │
   │ Name              │ db-11111                                                                                         │
   │ Status            │ READY                                                                                            │
   │ Type              │ TIMESCALEDB                                                                                      │
   │ Region            │ us-east-1                                                                                        │
   │ CPU               │ 0.5 cores (500m)                                                                                 │
   │ Memory            │ 2 GB                                                                                             │
   │ Direct Endpoint   │ tgrservice.tgrproject.tsdb.cloud.timescale.com:39004                                             │
   │ Created           │ 2025-10-20 20:33:46 UTC                                                                          │
   │ Connection String │ postgresql://tsdbadmin@tgrservice.tgrproject.tsdb.cloud.timescale.com:0007/tsdb?sslmode=require │
   │ Console URL       │ https://console.cloud.timescale.com/dashboard/services/tgrservice                                │
   └───────────────────┴──────────────────────────────────────────────────────────────────────────────────────────────────┘
   ```
   This service is set as default by the CLI.

1. **Check the CLI configuration**
   ```shell
   tiger config show
   ```
   You see something like:
   ```terminaloutput
   api_url:     https://console.cloud.timescale.com/public/api/v1
   console_url: https://console.cloud.timescale.com
   gateway_url: https://console.cloud.timescale.com/api
   docs_mcp:       true
   docs_mcp_url:   https://mcp.tigerdata.com/docs
   project_id:  tgrproject
   service_id:  tgrservice
   output:      table
   analytics:   true
   password_storage: keyring
   debug:       false
   config_dir:  /Users/<username>/.config/tiger
   ```

And that is it, you are ready to use Tiger CLI to manage your services in Tiger Cloud.

## Commands

You can use the following commands with Tiger CLI. For more information on each command, use the `-h` flag. For example:
`tiger auth login -h`

| Command | Subcommand                                   | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
|---------|----------------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| auth    |                                              | Manage authentication and credentials for your Tiger Data account                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
|         | login                                        | Create an authenticated connection to your Tiger Data account                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
|         | logout                                       | Remove the credentials used to create authenticated connections to Tiger Cloud                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
|         | status                                       | Show your current authentication status and project ID                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
| version |                                              | Show information about the currently installed version of Tiger CLI                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| config  |                                              | Manage your Tiger CLI configuration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
|         | show                                         | Show the current configuration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
|         | set `<key>` `<value>`                        | Set a specific value in your configuration. For example, `tiger config set debug true`                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
|         | unset `<key>`                                | Clear the value of a configuration parameter. For example, `tiger config unset debug`                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
|         | reset                                        | Reset the configuration to the defaults. This also logs you out from the current Tiger Cloud project                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| service |                                              | Manage the Tiger Cloud services in this project                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
|         | create                                       | Create a new service in this project. Possible flags are: <ul><li>`--name`: service name (auto-generated if not provided)</li><li>`--addons`: addons to enable (time-series, ai, or none for PostgreSQL-only)</li><li>`--region`: region code where the service will be deployed</li><li>`--cpu-memory`: CPU/memory allocation combination</li><li>`--replicas`: number of high-availability replicas</li><li>`--no-wait`: don't wait for the operation to complete</li><li>`--wait-timeout`: wait timeout duration (for example, 30m, 1h30m, 90s)</li><li>`--no-set-default`: don't set this service as the default service</li><li>`--with-password`: include password in output</li><li>`--output, -o`: output format (`json`, `yaml`, table)</li></ul> <br/> Possible `cpu-memory` combinations are: <ul><li>shared/shared</li><li>0.5 CPU/2 GB</li><li>1 CPU/4 GB</li><li>2 CPU/8 GB</li><li>4 CPU/16 GB</li><li>8 CPU/32 GB</li><li>16 CPU/64 GB</li><li>32 CPU/128 GB</li></ul> |
|         | delete `<service-id>`                        | Delete a service from this project. This operation is irreversible and requires confirmation by typing the service ID                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
|         | fork `<service-id>`                          | Fork an existing service to create a new independent copy. Key features are: <ul><li><strong>Timing options</strong>: `--now`, `--last-snapshot`, `--to-timestamp`</li><li><strong>Resource configuration</strong>: `--cpu-memory`</li><li><strong>Naming</strong>: `--name <name>`. Defaults to `{source-service-name}-fork`</li><li><strong>Wait behavior</strong>: `--no-wait`, `--wait-timeout`</li><li><strong>Default service</strong>: `--no-set-default`</li></ul>                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
|         | get `<service-id>` (aliases: describe, show) | Show detailed information about a specific service in this project                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
|         | list                                         | List all the services in this project                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
|         | update-password `<service-id>`               | Update the master password for a service                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| db      |                                              | Database operations and management                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
|         | connect `<service-id>`                       | Connect to a service                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
|         | connection-string `<service-id>`             | Retrieve the connection string for a service                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
|         | save-password  `<service-id>`                 | Save the password for a service                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
|         | test-connection `<service-id>`               | Test the connectivity to a service                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| mcp     |                                              | Manage the Tiger Model Context Protocol Server for AI Assistant integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
|         | install `[client]`                           | Install and configure Tiger Model Context Protocol Server for a specific client (`claude-code`, `cursor`, `windsurf`, or other). If no client is specified, you'll be prompted to select one interactively                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
|         | start                                        | Start the Tiger Model Context Protocol Server. This is the same as `tiger mcp start stdio`                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
|         | start stdio                                  | Start the Tiger Model Context Protocol Server with stdio transport (default)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
|         | start http                                   | Start the Tiger Model Context Protocol Server with HTTP transport. Includes flags: `--port` (default: `8080`), `--host` (default: `localhost`)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |


## Global flags

You can use the following global flags with Tiger CLI:

| Flag                          | Default           | Description                                                                 |
|-------------------------------|-------------------|-----------------------------------------------------------------------------|
| `--analytics`                 | `true`            | Set to `false` to disable usage analytics                                   |
| `--color `                    | `true`            | Set to `false` to disable colored output                                    |
| `--config-dir` string         | `.config/tiger`  | Set the directory that holds `config.yaml`                                  |
| `--debug`                     | No debugging      | Enable debug logging                                                        |
| `--help`                      | -                 | Print help about the current command. For example, `tiger service --help`   |
| `--password-storage` string   | keyring           | Set the password storage method. Options are `keyring`, `pgpass`, or `none` |
| `--service-id` string         | -                 | Set the Tiger Cloud service to manage                                             |
| ` --skip-update-check `       | -                 | Do  not check if a new version of Tiger CLI is available|


## Configuration parameters

By default, Tiger CLI stores your configuration in `~/.config/tiger/config.yaml`. The name of these
variables matches the flags you use to update them. However, you can override them using the following
environmental variables:

- **Configuration parameters**
    - `TIGER_CONFIG_DIR`: path to configuration directory (default: `~/.config/tiger`)
    - `TIGER_API_URL`: Tiger REST API base endpoint (default: https://console.cloud.timescale.com/public/api/v1)
    - `TIGER_CONSOLE_URL`: URL to Tiger Cloud Console (default: https://console.cloud.timescale.com)
    - `TIGER_GATEWAY_URL`: URL to the Tiger Cloud Console gateway (default: https://console.cloud.timescale.com/api)
    - `TIGER_DOCS_MCP`: enable/disable docs MCP proxy (default: `true`)
    - `TIGER_DOCS_MCP_URL`: URL to the Tiger MCP Server for Tiger Data docs (default: https://mcp.tigerdata.com/docs)
    - `TIGER_SERVICE_ID`: ID for the service updated when you call CLI commands
    - `TIGER_ANALYTICS`: enable or disable analytics (default: `true`)
    - `TIGER_PASSWORD_STORAGE`: password storage method (keyring, pgpass, or none)
    - `TIGER_DEBUG`: enable/disable debug logging (default: `false`)
    - `TIGER_COLOR`: set to `false` to disable colored output (default: `true`)


- **Authentication parameters**

  To authenticate without using the interactive login, either:
  - Set the following parameters with your [client credentials][rest-api-credentials], then `login`:
    ```shell
    TIGER_PUBLIC_KEY=<public_key> TIGER_SECRET_KEY=<secret_key> TIGER_PROJECT_ID=<project_id>\
    tiger auth login
    ```
  - Add your [client credentials][rest-api-credentials] to the `login` command:
    ```shell
    tiger auth login --public-key=<public_key> --secret-key=<secret-key> --project-id=<project_id>
    ```





[Tiger REST API][rest-api-reference] is a comprehensive RESTful API you use to manage Tiger Cloud resources
including VPCs, services, and read replicas.

This page shows you how to set up secure authentication for the Tiger REST API and create your first service.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Data account][create-account].

* Install [curl][curl].


## Configure secure authentication

Tiger REST API uses HTTP Basic Authentication with access keys and secret keys. All API requests must include
proper authentication headers.

1. **Set up API credentials**

    1. In Tiger Cloud Console [copy your project ID][get-project-id] and store it securely using an environment variable:

      ```bash
      export TIGERDATA_PROJECT_ID="your-project-id"
      ```

    1. In Tiger Cloud Console [create your client credentials][create-client-credentials] and store them securely using environment variables:

       ```bash
       export TIGERDATA_ACCESS_KEY="Public key"
       export TIGERDATA_SECRET_KEY="Secret key"
       ```

1. **Configure the API endpoint**

   Set the base URL in your environment:

    ```bash
    export API_BASE_URL="https://console.cloud.timescale.com/public/api/v1"
    ```

1. **Test your authenticated connection to Tiger REST API by listing the services in the current Tiger Cloud project**

    ```bash
    curl -X GET "${API_BASE_URL}/projects/${TIGERDATA_PROJECT_ID}/services" \
      -u "${TIGERDATA_ACCESS_KEY}:${TIGERDATA_SECRET_KEY}" \
      -H "Content-Type: application/json"
    ```

   This call returns something like:
    - No services:
      ```terminaloutput
      []%
      ```
    - One or more services:

      ```terminaloutput
      [{"service_id":"tgrservice","project_id":"tgrproject","name":"tiger-eon",
      "region_code":"us-east-1","service_type":"TIMESCALEDB",
      "created":"2025-10-20T12:21:28.216172Z","paused":false,"status":"READY",
      "resources":[{"id":"104977","spec":{"cpu_millis":500,"memory_gbs":2,"volume_type":""}}],
      "metadata":{"environment":"DEV"},
      "endpoint":{"host":"tgrservice.tgrproject.tsdb.cloud.timescale.com","port":11111}}]
      ```


## Create your first Tiger Cloud service

Create a new service using the Tiger REST API:

1. **Create a service using the POST endpoint**
   ```bash
   curl -X POST "${API_BASE_URL}/projects/${TIGERDATA_PROJECT_ID}/services" \
     -u "${TIGERDATA_ACCESS_KEY}:${TIGERDATA_SECRET_KEY}" \
     -H "Content-Type: application/json" \
     -d '{
        "name": "my-first-service",
        "addons": ["time-series"],
        "region_code": "us-east-1",
        "replica_count": 1,
        "cpu_millis": "1000",
        "memory_gbs": "4"
        }'
   ```
   Tiger Cloud creates a Development environment for you. That is, no delete protection, high-availability, spooling or
   read replication. You see something like:
   ```terminaloutput
    {"service_id":"tgrservice","project_id":"tgrproject","name":"my-first-service",
    "region_code":"us-east-1","service_type":"TIMESCALEDB",
    "created":"2025-10-20T22:29:33.052075713Z","paused":false,"status":"QUEUED",
    "resources":[{"id":"105120","spec":{"cpu_millis":1000,"memory_gbs":4,"volume_type":""}}],
    "metadata":{"environment":"PROD"},
    "endpoint":{"host":"tgrservice.tgrproject.tsdb.cloud.timescale.com","port":00001},
    "initial_password":"notTellingYou",
    "ha_replicas":{"sync_replica_count":0,"replica_count":1}}
   ```

1. Save `service_id` from the response to a variable:

   ```bash
   # Extract service_id from the JSON response
   export SERVICE_ID="service_id-from-response"
   ```

1. **Check the configuration for the service**

  ```bash
    curl -X GET "${API_BASE_URL}/projects/${TIGERDATA_PROJECT_ID}/services/${SERVICE_ID}" \
      -u "${TIGERDATA_ACCESS_KEY}:${TIGERDATA_SECRET_KEY}" \
      -H "Content-Type: application/json"
  ```
You see something like:
  ```terminaloutput
    {"service_id":"tgrservice","project_id":"tgrproject","name":"my-first-service",
    "region_code":"us-east-1","service_type":"TIMESCALEDB",
    "created":"2025-10-20T22:29:33.052075Z","paused":false,"status":"READY",
    "resources":[{"id":"105120","spec":{"cpu_millis":1000,"memory_gbs":4,"volume_type":""}}],
    "metadata":{"environment":"DEV"},
    "endpoint":{"host":"tgrservice.tgrproject.tsdb.cloud.timescale.com","port":11111},
    "ha_replicas":{"sync_replica_count":0,"replica_count":1}}
  ```

And that is it, you are ready to use the [Tiger REST API][rest-api-reference] to manage your
services in Tiger Cloud.

## Security best practices

Follow these security guidelines when working with the Tiger REST API:

- **Credential management**
    - Store API credentials as environment variables, not in code
    - Use credential rotation policies for production environments
    - Never commit credentials to version control systems

- **Network security**
    - Use HTTPS endpoints exclusively for API communication
    - Implement proper certificate validation in your HTTP clients

- **Data protection**
    - Use secure storage for service connection strings and passwords
    - Implement proper backup and recovery procedures for created services
    - Follow data residency requirements for your region


===== PAGE: https://docs.tigerdata.com/getting-started/run-queries-from-console/ =====

# Run your queries from Tiger Cloud Console



As Tiger Cloud is based on Postgres, you can use lots of [different tools][integrations] to
connect to your service and interact with your data.

In Tiger Cloud Console you can use the following ways to run SQL queries against your service:

- [Data mode][run-popsql]: a rich experience powered by PopSQL. You can write queries with
  autocomplete, save them in folders, share them, create charts/dashboards, and much more.

- [SQL Assistant in the data mode][sql-assistant]: write, fix, and organize SQL faster and more accurately.

- [SQL editor in the ops mode][run-sqleditor]: a simple SQL editor in the ops mode that lets you run ad-hoc ephemeral
  queries. This is useful for quick one-off tasks like creating an index on a small table or inspecting `pg_stat_statements`.

If you prefer the command line to the ops mode SQL editor in Tiger Cloud Console, use [psql][install-psql].

## 	Data mode

You use the data mode in Tiger Cloud Console to write queries, visualize data, and share your results.

![Tiger Cloud Console data mode](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-data-mode.png)

This feature is not available under the Free pricing plan.

Available features are:

- **Real-time collaboration**: work with your team directly in the data mode query editor with live presence and multiple
   cursors.
- **[Schema browser][schema-browser]**: understand the structure of your service and see usage data on tables and columns.
- **[SQL Assistant][sql-assistant]**: write, fix, and organize SQL faster and more accurately using AI.
- **Autocomplete**: get suggestions as you type your queries.
- **[Version history][version-history]**: access previous versions of a query from the built-in revision history, or connect to a git repo.
- **[Charts][charts]**: visualize data from inside the UI rather than switch to Sheets or Excel.
- **[Schedules][schedules]**: automatically refresh queries and dashboards to create push alerts.
- **[Query variables][query-variables]**: use Liquid to parameterize your queries or use `if` statements.
- **Cross-platform support**: work from [Tiger Cloud Console][portal-data-mode] or download the [desktop app][popsql-desktop] for macOS, Windows, and Linux.
- **Easy connection**: connect to Tiger Cloud, Postgres, Redshift, Snowflake, BigQuery, MySQL, SQL Server, [and more][popsql-connections].

### Connect to your Tiger Cloud service in the data mode

To connect to a service:

1. **Check your service is running correctly**

   In [Tiger Cloud Console][services-portal], check that your service is marked as `Running`:

   ![Check Tiger Cloud service is running](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-services-view.png)

1. **Connect to your service**

   In the [data mode][portal-data-mode] in Tiger Cloud Console, select a service in the connection drop-down:

   ![Select a connection](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-data-mode-connection-dropdown.png)

1. **Run a test query**

   Type `SELECT CURRENT_DATE;` in `Scratchpad` and click `Run`:

   ![Run a simple query](https://assets.timescale.com/docs/images/tiger-cloud-console/run-query-in-scratchpad-tiger-console.png)

Quick recap. You:
- Manage your services in the [ops mode in Tiger Cloud Console][portal-ops-mode]
- Manage your data in the [data mode in Tiger Cloud Console][portal-data-mode]
- Store configuration and security information in your config file.

Now you have used the data mode in Tiger Cloud Console, see how to easily do the following:

- [Write a query][write-query]
- [Share a query with your teammates][share-query]
- [Create a chart from your data][create-chart]
- [Create a dashboard of multiple query results][create-dashboard]
- [Create schedules for your queries][create-schedule]

### Data mode FAQ

#### What if my service is within a vpc?

If your Tiger Cloud service runs inside a VPC, do one of the following to enable access for the PopSQL desktop app:

- Use PopSQL's [bridge connector][bridge-connector].
- Use an SSH tunnel: when you configure the connection in PopSQL, under `Advanced Options` enable `Connect over SSH`.
- Add PopSQL's static IPs (`23.20.131.72, 54.211.234.135`) to your allowlist.

#### What happens if another member of my Tiger Cloud project uses the data mode?

The number of data mode seats you are allocated depends on your [pricing plan][pricing-plan-features].

#### Will using the data mode affect the performance of my Tiger Cloud service?

There are a few factors to consider:

1. What instance size is your service?
1. How many users are running queries?
1. How computationally intensive are the queries?

If you have a small number of users running performant SQL queries against a
service with sufficient resources, then there should be no degradation to
performance. However, if you have a large number of users running queries, or if
the queries are computationally expensive, best practice is to create
a [read replica][read-replica] and send analytical queries there.

If you'd like to prevent write operations such as insert or update, instead
of using the `tsdbadmin` user, create a read-only user for your service and
use that in the data mode.

## SQL Assistant

SQL Assistant in [Tiger Cloud Console][portal-data-mode] is a chat-like interface that harnesses the power of AI to help you write, fix, and organize SQL faster and more accurately. Ask SQL Assistant to change existing queries, write new ones from scratch, debug error messages, optimize for query performance, add comments, improve readability—and really, get answers to any questions you can think of.

This feature is not available under the Free pricing plan.

<!--
<div class="relative w-fit mx-auto">

<iframe width="1120" height="630" style="max-width:100%"  src="https://www.youtube.com/embed/3Droej_E0cQ?si=C4RoL_PFpr8E5QtC" title="YouTube video player" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" referrerpolicy="strict-origin-when-cross-origin" allowfullscreen></iframe>

</div>
-->

### Key capabilities

SQL Assistant offers a range of features to improve your SQL workflow, including:

- **Real-time help**: SQL Assistant provides in-context help for writing and understanding SQL. Use it to:

  - **Understand functions**: need to know how functions like `LAG()` or `ROW_NUMBER()` work? SQL Assistant explains it with examples.
  - **Interpret complex queries**: SQL Assistant breaks down dense queries, giving you a clear view of each part.

- **Error resolution**: SQL Assistant diagnoses errors as they happen, you can resolve issues without leaving your editor. Features include:

  - **Error debugging**: if your query fails, SQL Assistant identifies the issue and suggests a fix.
  - **Performance tuning**: for slow queries, SQL Assistant provides optimization suggestions to improve performance immediately.

- **Query organization**: to keep your query library organized, and help your team understand the
  purpose of each query, SQL Assistant automatically adds titles and summaries to your queries.

- **Agent mode**: to get results with minimal involvement from you, SQL Assistant autopilots through complex tasks and troubleshoots its own problems. No need to go step by step, analyze errors, and try out solutions. Simply turn on the agent mode in the LLM picker and watch SQL Assistant do all the work for you. Recommended for use when your database connection is configured with read-only credentials.

### Supported LLMs

SQL Assistant supports a large number of LLMs, including:

- GPT-4o mini
- GPT-4o
- GPT-4.1 nano
- GPT-4.1 mini
- GPT-4.1
- o4-mini (low)
- o4-mini
- o4-mini (high)
- o3 (low)
- o3
- o3 (high)
- Claude 3.5 Haiku
- Claud 3.7 Sonnet
- Claud 3.7 Sonnet (extended thinking)
- Llama 3.3 70B Versatile
- Llama 3.3 70B Instruct
- Llama 3.1 405B Instruct
- Llama 4 Scout
- Llama 4 Maverick
- DeepSeek R1 Distill - Llama 3.3. 70B
- DeepSeek R1
- Gemini 2.0 Flash
- Sonnet 4
- Sonnet 4 (extended thinking)
- Opus 4
- Opus 4 (extended thinking)

Choose the LLM based on the particular task at hand. For simpler tasks, try the smaller and faster models like Gemini Flash, Haiku, or o4-mini. For more complex tasks, try the larger reasoning models like Claude Sonnet, Gemini Pro, or o3. We provide a description of each model to help you decide.

### Limitations to keep in mind

For best results with SQL Assistant:

* **Schema awareness**: SQL Assistant references schema data but may need extra context
  in complex environments. Specify tables, columns, or joins as needed.
* **Business logic**: SQL Assistant does not inherently know specific business terms
  such as active user. Define these terms clearly to improve results.

### Security, privacy, and data usage

Security and privacy is prioritized in Tiger Cloud Console. In [data mode][portal-data-mode], project members
manage SQL Assistant settings under [`User name` > `Settings` > `SQL Assistant`][sql-editor-settings].

![SQL assistant settings](https://assets.timescale.com/docs/images/tiger-console-sql-editor-preferences.png)

SQL Assistant settings are:

* **Opt-in features**: all AI features are off by default. Only [members][project-members] of your Tiger Cloud project
  can enable them.
* **Data protection**: your data remains private as SQL Assistant operates with strict security protocols. To provide AI support, Tiger Cloud Console may share your currently open SQL document, some basic metadata about your database, and portions of your database schema. By default, Tiger Cloud Console **does not include** any data from query results, but you can opt in to include this context to improve the results.
* **Sample data**: to give the LLM more context so you have better SQL suggestions, enable sample data sharing in the SQL Assistant preferences.
* **Telemetry**: to improve SQL Assistant, Tiger Data collects telemetry and usage data, including prompts, responses, and query metadata.

## Ops mode SQL editor

SQL editor is an integrated secure UI that you use to run queries and see the results
for a Tiger Cloud service.

![Tiger Cloud Console SQL editor](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-ops-mode-sql-editor.png)

To enable or disable SQL editor in your service, click `Operations` > `Service management`, then
update the setting for SQL editor.

To use SQL editor:

1.  **Open SQL editor from Tiger Cloud Console**

    In the [ops mode][portal-ops-mode] in Tiger Cloud Console, select a service, then click `SQL editor`.

    ![Check service is running](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-ops-mode-sql-editor-empty.png)

1. **Run a test query**

    Type `SELECT CURRENT_DATE;` in the UI and click `Run`. The results appear in the lower window:

    ![Run a simple query](https://assets.timescale.com/docs/images/tiger-cloud-console/run-a-query-in-tiger-ops-mode-sql-editor.png)

## Cloud SQL editor licenses

* **SQL editor in the ops mode**: free for anyone with a [Tiger Data account][create-cloud-account].
* **Data mode**: the number of seats you are allocated depends on your [pricing plan][pricing-plan-features].

  [SQL Assistant][sql-assistant] is currently free for all users. In the future, limits or paid options may be
  introduced as we work to build the best experience.
* **PopSQL standalone**: there is a free plan available to everyone, as well as paid plans. See  [PopSQL Pricing][popsql-pricing] for full details.

What next? [Try the key features offered by Tiger Data][try-timescale-features], see the [tutorials][tutorials],
interact with the data in your Tiger Cloud service using [your favorite programming language][connect-with-code], integrate
your Tiger Cloud service with a range of [third-party tools][integrations], plain old [Use Tiger Data products][use-timescale], or dive
into the [API reference][use-the-api].


===== PAGE: https://docs.tigerdata.com/use-timescale/hypertables/ =====

# Hypertables



Tiger Cloud supercharges your real-time analytics by letting you run complex queries continuously, with near-zero latency. Under the hood, this is achieved by using hypertables—Postgres tables that automatically partition your time-series data by time and optionally by other dimensions. When you run a query, Tiger Cloud identifies the correct partition, called chunk, and runs the query on it, instead of going through the entire table.

![Hypertable structure](https://assets.timescale.com/docs/images/hypertable.png)

Hypertables offer the following benefits:

- **Efficient data management with [automated partitioning by time][chunk-size]**: Tiger Cloud splits your data into chunks that hold data from a specific time range. For example, one day or one week. You can configure this range to better suit your needs.

- **Better performance with [strategic indexing][hypertable-indexes]**: an index on time in the descending order is automatically created when you create a hypertable. More indexes are created on the chunk level, to optimize performance. You can create additional indexes, including unique indexes, on the columns you need.

- **Faster queries with [chunk skipping][chunk-skipping]**: Tiger Cloud skips the chunks that are irrelevant in the context of your query, dramatically reducing the time and resources needed to fetch results. Even more—you can enable chunk skipping on non-partitioning columns.

- **Advanced data analysis with [hyperfunctions][hyperfunctions]**: Tiger Cloud enables you to efficiently process, aggregate, and analyze significant volumes of data while maintaining high performance.

To top it all, there is no added complexity—you interact with hypertables in the same way as you would with regular Postgres tables. All the optimization magic happens behind the scenes.



Inheritance is not supported for hypertables and may lead to unexpected behavior.

## Partition by time

Each hypertable is partitioned into child hypertables called chunks. Each chunk is assigned
a range of time, and only contains data from that range.


### Time partitioning

Typically, you partition hypertables on columns that hold time values.
[Best practice is to use `timestamptz`][timestamps-best-practice] column type. However, you can also partition on
`date`, `integer`, `timestamp` and [UUIDv7][uuidv7_functions] types.

By default, each hypertable chunk holds data for 7 days. You can change this to better suit your
needs. For example, if you set `chunk_interval` to 1 day, each chunk stores data for a single day.

TimescaleDB divides time into potential chunk ranges, based on the `chunk_interval`. Each hypertable chunk holds
data for a specific time range only. When you insert data from a time range that doesn't yet have a chunk, TimescaleDB
automatically creates a chunk to store it.

In practice, this means that the start time of your earliest chunk does not
necessarily equal the earliest timestamp in your hypertable. Instead, there
might be a time gap between the start time and the earliest timestamp. This
doesn't affect your usual interactions with your hypertable, but might affect
the number of chunks you see when inspecting it.

## Best practices for scaling and partitioning

Best practices for maintaining a high performance when scaling include:

- Limit the number of hypertables in your service; having tens of thousands of hypertables is not recommended.
- Choose a strategic chunk size.

Chunk size affects insert and query performance. You want a chunk small enough
to fit into memory so you can insert and query recent data without
reading from disk. However, having too many small and sparsely filled chunks can
affect query planning time and compression. The more chunks in the system, the slower that process becomes, even more so
when all those chunks are part of a single hypertable.

Postgres builds the index on the fly during ingestion. That means that to build a new entry on the index,
a significant portion of the index needs to be traversed during every row insertion. When the index does not fit
into memory, it is constantly flushed to disk and read back. This wastes IO resources which would otherwise
be used for writing the heap/WAL data to disk.

The default chunk interval is 7 days. However, best practice is to set `chunk_interval` so that prior to processing,
the indexes for chunks currently being ingested into fit within 25% of main memory. For example, on a system with 64
GB of memory, if index growth is approximately 2 GB per day, a 1-week chunk interval is appropriate. If index growth is
around 10 GB per day, use a 1-day interval.

You set `chunk_interval` when you [create a hypertable][hypertable-create-table], or by calling
[`set_chunk_time_interval`][chunk_interval] on an  existing hypertable.

For a detailed analysis of how to optimize your chunk sizes, see the
[blog post on chunk time intervals][blog-chunk-time]. To learn how
to view and set your chunk time intervals, see
[Optimize hypertable chunk intervals][change-chunk-intervals].

## Hypertable indexes

By default, indexes are automatically created when you create a hypertable. The default index is on time, descending.
You can prevent index creation by setting the `create_default_indexes` option to `false`.

Hypertables have some restrictions on unique constraints and indexes. If you
want a unique index on a hypertable, it must include all the partitioning
columns for the table. To learn more, see
[Enforce constraints with unique indexes on hypertables][hypertables-and-unique-indexes].

You can prevent index creation by setting the `create_default_indexes` option to `false`.

## Partition by dimension

Partitioning on time is the most common use case for hypertable, but it may not be enough for your needs. For example,
you may need to scan for the latest readings that match a certain condition without locking a critical hypertable.



The use case for a partitioning dimension is a multi-tenant setup. You isolate the tenants using the `tenant_id` space
partition. However, you must perform extensive testing to ensure this works as expected, and there is a strong risk of
partition explosion.



You add a partitioning dimension at the same time as you create the hypertable, when the table is empty. The good news
is that although you select the number of partitions at creation time, as your data grows you can change the number of
partitions later and improve query performance. Changing the number of partitions only affects chunks created after the
change, not existing chunks. To set the number of partitions for a partitioning dimension, call `set_number_partitions`.
For example:

1. **Create the hypertable with the 1-day interval chunk interval**

   ```sql
   CREATE TABLE conditions(
      "time"      timestamptz not null,
      device_id   integer,
      temperature float
   )
   WITH(
      timescaledb.hypertable,
      timescaledb.partition_column='time',
      timescaledb.chunk_interval='1 day'
   );
   ```

1. **Add a hash partition on a non-time column**

   ```sql
   select * from add_dimension('conditions', by_hash('device_id', 3));
   ```
   Now use your hypertable as usual, but you can also ingest and query efficiently by the `device_id` column.

1. **Change the number of partitions as you data grows**

   ```sql
   select set_number_partitions('conditions', 5, 'device_id');
   ```


===== PAGE: https://docs.tigerdata.com/use-timescale/hypercore/ =====

# Hypercore



Hypercore is a hybrid row-columnar storage engine in TimescaleDB. It is designed specifically for
real-time analytics and powered by time-series data. The advantage of hypercore is its ability
to seamlessly switch between row-oriented and column-oriented storage, delivering the best of both worlds:

![Hypercore workflow](https://assets.timescale.com/docs/images/hypertable-with-hypercore-enabled.png)

Hypercore solves the key challenges in real-time analytics:

- High ingest throughput
- Low-latency ingestion
- Fast query performance
- Efficient handling of data updates and late-arriving data
- Streamlined data management

Hypercore’s hybrid approach combines the benefits of row-oriented and column-oriented formats:

- **Fast ingest with rowstore**: new data is initially written to the rowstore, which is optimized for
  high-speed inserts and updates. This process ensures that real-time applications easily handle
  rapid streams of incoming data. Mutability—upserts, updates, and deletes happen seamlessly.

- **Efficient analytics with columnstore**: as the data **cools** and becomes more suited for
  analytics, it is automatically converted to the columnstore. This columnar format enables
  fast scanning and aggregation, optimizing performance for analytical workloads while also
  saving significant storage space.

- **Faster queries on compressed data in columnstore**: in the columnstore conversion, hypertable
  chunks are compressed by up to 98%, and organized for efficient, large-scale queries. Combined with [chunk skipping][chunk-skipping], this helps you save on storage costs and keeps your queries operating at lightning speed.

- **Fast modification of compressed data in columnstore**: just use SQL to add or modify data in the columnstore.
   TimescaleDB is optimized for superfast INSERT and UPSERT performance.

- **Full mutability with transactional semantics**: regardless of where data is stored,
  hypercore provides full ACID support. Like in a vanilla Postgres database, inserts and updates
  to the rowstore and columnstore are always consistent, and available to queries as soon as they are
  completed.

For an in-depth explanation of how hypertables and hypercore work, see the [Data model][data-model].

This section shows the following:

* [Optimize your data for real-time analytics][setup-hypercore]
* [Improve query and upsert performance using secondary indexes][secondary-indexes]
* [Compression methods in hypercore][compression-methods]
* [Troubleshooting][troubleshooting]


===== PAGE: https://docs.tigerdata.com/use-timescale/continuous-aggregates/ =====

# Continuous aggregates

From real-time dashboards to performance monitoring and historical trend analysis, data aggregation is a must-have for any sort of analytical application. To address this need, TimescaleDB uses continuous aggregates to precompute and store aggregate data for you. Using Postgres [materialized views][postgres-materialized-views], TimescaleDB incrementally refreshes the aggregation query in the background. When you do run the query, only the data that has changed needs to be computed, not the entire dataset. This means you always have the latest aggregate data at your fingertips—and spend as little resources on it, as possible.

In this section you:

*   [Learn about continuous aggregates][about-caggs] to understand how it works
    before you begin using it.
*   [Create a continuous aggregate][cagg-create] and query it.
*   [Create a continuous aggregate on top of another continuous aggregate][cagg-on-cagg].
*   [Add refresh policies][cagg-autorefresh] to an existing continuous aggregate.
*   [Manage time][cagg-time] in your continuous aggregates.
*   [Drop data][cagg-drop] from your continuous aggregates.
*   [Manage materialized hypertables][cagg-mat-hypertables].
*   [Use real-time aggregates][cagg-realtime].
*   [Convert continuous aggregates to the columnstore][cagg-compression].
*   [Migrate your continuous aggregates][cagg-migrate] from old to new format.
    Continuous aggregates created in TimescaleDB v2.7 and later are in the new
    format, unless explicitly created in the old format.
*   [Troubleshoot][cagg-tshoot] continuous aggregates.


===== PAGE: https://docs.tigerdata.com/use-timescale/services/ =====

# About Tiger Cloud services



Tiger Cloud is the modern Postgres data platform for all your applications. It enhances Postgres to handle time series, events,
real-time analytics, and vector search—all in a single database alongside transactional workloads.

You get one system that handles live data ingestion, late and out-of-order updates, and low latency queries, with the performance, reliability, and scalability your app needs. Ideal for IoT, crypto, finance, SaaS, and a myriad other domains, Tiger Cloud allows you to build data-heavy, mission-critical apps while retaining the familiarity and reliability of Postgres.

A Tiger Cloud service is a single optimised Postgres instance extended with innovations in the database engine and cloud
infrastructure to deliver speed without sacrifice. A Tiger Cloud service is 10-1000x faster at scale! It
is ideal for applications requiring strong data consistency, complex relationships, and advanced querying capabilities.
Get ACID compliance, extensive SQL support, JSON handling, and extensibility through custom functions, data types, and
extensions.

Each service is associated with a project in Tiger Cloud. Each project can have multiple services. Each user is a [member of one or more projects][rbac].

You create free and standard services in Tiger Cloud Console, depending on your [pricing plan][pricing-plans]. A free service comes at zero cost and gives you limited resources to get to know Tiger Cloud. Once you are ready to try out more advanced features, you can switch to a paid plan and convert your free service to a standard one.

![Tiger Cloud pricing plans](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-pricing.svg)

The Free pricing plan and services are currently in beta.

To the Postgres you know and love, Tiger Cloud adds the following capabilities:

- **Standard services**:

    - _Real-time analytics_: store and query [time-series data][what-is-time-series] at scale for
      real-time analytics and other use cases. Get faster time-based queries with hypertables, continuous aggregates, and columnar storage. Save money by compressing data into the columnstore, moving cold data to low-cost bottomless storage in Amazon S3, and deleting old data with automated policies.
    - _AI-focused_: build AI applications from start to scale. Get fast and accurate similarity search
      with the pgvector and pgvectorscale extensions.
    - _Hybrid applications_: get a full set of tools to develop applications that combine time-based data and AI.

  All standard Tiger Cloud services include the tooling you expect for production and developer environments: [live migration][live-migration],
  [automatic backups and PITR][automatic-backups], [high availability][high-availability], [read replicas][readreplica], [data forking][operations-forking], [connection pooling][connection-pooling], [tiered storage][data-tiering],
  [usage-based storage][how-plans-work], secure in-Tiger Cloud Console [SQL editing][in-console-editors], service [metrics][metrics]
  and [insights][insights],&nbsp;[streamlined maintenance][maintain-upgrade],&nbsp;and much more. Tiger Cloud continuously monitors your services and prevents common Postgres out-of-memory crashes.

- **Free services**:

  _Postgres with TimescaleDB and vector extensions_

  Free services offer limited resources and a basic feature scope, perfect to get to know Tiger Cloud in a development environment.

## Learn more about Tiger Cloud

Read about Tiger Cloud features in the documentation:

*   Create your first [hypertable][hypertable-info].
*   Run your first query using [time_bucket()][time-bucket-info].
*   Trying more advanced time-series functions, starting with
    [gap filling][gap-filling-info] or [real-time aggregates][aggregates-info].

## Keep testing during your free trial

You're now on your way to a great start with Tiger Cloud.

You have an unthrottled, 30-day free trial with Tiger Cloud to continue to
test your use case. Before the end of your trial, make sure you add your credit
card information. This ensures a smooth transition after your trial period
concludes.

If you have any questions, you can
[join our community Slack group][slack-info]
or [contact us][contact-timescale] directly.

## Advanced configuration

Tiger Cloud is a versatile hosting service that provides a growing list of
advanced features for your Postgres and time-series data workloads.

For more information about customizing your database configuration, see the
[Configuration section][configuration].



The [TimescaleDB Terraform provider](https://registry.terraform.io/providers/timescale/timescale/latest/)
provides configuration management resources for Tiger Cloud. You can use it to
create, rename, resize, delete, and import services. For more information about
the supported service configurations and operations, see the
[Terraform provider documentation](https://registry.terraform.io/providers/timescale/timescale/latest/docs).


===== PAGE: https://docs.tigerdata.com/use-timescale/write-data/ =====

# Write data

Writing data in TimescaleDB works the same way as writing data to regular
Postgres. You can add and modify data in both regular tables and hypertables
using `INSERT`, `UPDATE`, and `DELETE` statements.

*   [Learn about writing data in TimescaleDB][about-writing-data]
*   [Insert data][insert] into hypertables
*   [Update data][update] in hypertables
*   [Upsert data][upsert] into hypertables
*   [Delete data][delete] from hypertables

For more information about using third-party tools to write data
into TimescaleDB, see the [Ingest data from other sources][ingest-data] section.


===== PAGE: https://docs.tigerdata.com/use-timescale/query-data/ =====

# Query data

Hypertables in TimescaleDB are Postgres tables. That means you can query them
with standard SQL commands.

*   [About querying data][about-querying-data]
*   [Select data with `SELECT`][selecting-data]
*   [Get faster `DISTINCT` queries with SkipScan][skipscan]
*   [Perform advanced analytic queries][advanced-analytics]


===== PAGE: https://docs.tigerdata.com/use-timescale/time-buckets/ =====

# Time buckets

Time buckets enable you to aggregate data in [hypertables][create-hypertable] by time interval. For example, you can
group data into 5-minute, 1-hour, and 3-day buckets to calculate summary values.

*   [Learn how time buckets work][about-time-buckets]
*   [Use time buckets][use-time-buckets] to aggregate data


===== PAGE: https://docs.tigerdata.com/use-timescale/schema-management/ =====

# Schema management

A database schema defines how the tables and indexes in your database are
organized. Using a schema that is appropriate for your workload can result in
significant performance improvements.

*   [Learn about schema management][about-schema] to understand how it works
    before you begin using it.
*   [Learn about indexing][about-indexing] to understand how it works before you
    begin using it.
*   [Learn about tablespaces][about-tablespaces] to understand how they work before
    you begin using them.
*   [Learn about constraints][about-constraints] to understand how they work before
    you begin using them.
*   [Alter a hypertable][schema-alter] to modify your schema.
*   [Create an index][schema-indexing] to speed up your queries.
*   [Create triggers][schema-triggers] to propagate your schema changes to chunks.
*   [Use JSON and JSONB][schema-json] for semi-structured data.
*   [Query external databases][foreign-data-wrappers] with foreign data wrappers.
*   [Troubleshoot][troubleshoot-schemas] your schemas.


===== PAGE: https://docs.tigerdata.com/use-timescale/configuration/ =====

# Configuration

By default, Tiger Cloud uses the standard Postgres server configuration
settings. However, in some cases, these settings are not appropriate, especially
if you have larger servers that use more hardware resources such as CPU, memory,
and storage.

This section contains information about tuning your Tiger Cloud service.


===== PAGE: https://docs.tigerdata.com/use-timescale/alerting/ =====

# Alerting

Early issue detecting and prevention, ensuring high availability, and performance optimization are only a few of the reasons why alerting plays a major role for modern applications, databases, and services.

There are a variety of different alerting solutions you can use in conjunction
with Tiger Cloud that are part of the Postgres ecosystem. Regardless of
whether you are creating custom alerts embedded in your applications, or using
third-party alerting tools to monitor event data across your organization, there
are a wide selection of tools available.

## Grafana

Grafana is a great way to visualize your analytical queries, and it has a
first-class integration with Tiger Data products. Beyond data visualization, Grafana
also provides alerting functionality to keep you notified of anomalies.

Within Grafana, you can [define alert rules][define alert rules] which are
time-based thresholds for your dashboard data (for example, "Average CPU usage
greater than 80 percent for 5 minutes"). When those alert rules are triggered,
Grafana sends a message via the chosen notification channel. Grafana provides
integration with webhooks, email and more than a dozen external services
including Slack and PagerDuty.

To get started, first download and install [Grafana][Grafana-install]. Next, add
a new [Postgres data source][PostgreSQL datasource] that points to your
Tiger Cloud service. This data source was built by Tiger Data engineers, and
it is designed to take advantage of the database's time-series capabilities.
From there, proceed to your dashboard and set up alert rules as described above.



Alerting is only available in Grafana v4.0 and later.



## Other alerting tools

Tiger Cloud works with a variety of alerting tools within the Postgres
ecosystem. Users can use these tools to set up notifications about meaningful
events that signify notable changes to the system.

Some popular alerting tools that work with Tiger Cloud include:

*   [DataDog][datadog-install]
*   [Nagios][nagios-install]
*   [Zabbix][zabbix-install]

See the [integration guides][integration-docs] for details.


===== PAGE: https://docs.tigerdata.com/use-timescale/data-retention/ =====

# Data retention

Data retention helps you save on storage costs by deleting old data. You can
combine data retention with [continuous aggregates][caggs] to downsample your
data.

In this section:

*   [Learn about data retention][about-data-retention] before you start using it
*   [Learn about data retention with continuous aggregates][retention-with-caggs]
    for downsampling data
*   Create a [data retention policy][retention-policy]
*   [Manually drop chunks][manually-drop] of data
*   [Troubleshoot] data retention


===== PAGE: https://docs.tigerdata.com/use-timescale/data-tiering/ =====

# Storage in Tiger

Tiered storage is a [hierarchical storage management architecture][hierarchical-storage] for
[real-time analytics][create-service] services you create in [Tiger Cloud](https://console.cloud.timescale.com/).

Engineered for infinite low-cost scalability, tiered storage consists of the following:

* **High-performance storage tier**: stores the most recent and frequently queried data. This tier comes in two types,
standard and enhanced, and provides you with up to 64 TB of storage and 32,000 IOPS.

* **Object storage tier**: stores data that is rarely accessed and has lower performance requirements.
  For example, old data for auditing or reporting purposes over long periods of time, even forever.
  The object storage tier is low-cost and bottomless.

No matter the tier your data is stored in, you can [query it when you need it][querying-tiered-data].
Tiger Cloud seamlessly accesses the correct storage tier and generates the response.

<!-- vale Google.SmartQuotes = NO -->

You [define tiering policies][creating-data-tiering-policy] that automatically migrate
data from the high-performance storage tier to the object tier as it ages. You use
[retention policies][add-retention-policies] to remove very old data from the object storage tier.

With tiered storage you don't need an ETL process, infrastructure changes, or custom-built, bespoke
solutions to offload data to secondary storage and fetch it back in when needed. Kick back and relax,
we do the work for you.

<!-- vale Google.SmartQuotes = YES -->

In this section, you:
* [Learn more about storage tiers][about-data-tiering]: understand how the tiers are built and how they differ.
* [Manage storage and tiering][enabling-data-tiering]: configure high-performance storage, object storage, and data tiering.
* [Query tiered data][querying-tiered-data]: query the data in the object storage.
* [Learn about replicas and forks with tiered data][replicas-and-forks]: understand how tiered storage works
  with forks and replicas of your service.


===== PAGE: https://docs.tigerdata.com/use-timescale/metrics-logging/ =====

# Metrics and logging

Find metrics and logs for your services in Tiger Cloud Console, or integrate with third-party monitoring services:

*   [Monitor][monitor] your services in Tiger Cloud Console.
*   Export metrics to [Datadog][datadog].
*   Export metrics to [Amazon Cloudwatch][cloudwatch].
*   Export metrics to [Prometheus][prometheus].


===== PAGE: https://docs.tigerdata.com/use-timescale/ha-replicas/ =====

# High availability and read replication

In Tiger Cloud, replicas are copies of the primary data instance in a Tiger Cloud service.
If your primary becomes unavailable, Tiger Cloud automatically fails over to your HA replica.

The replication strategies offered by Tiger Cloud are:

- [High Availability(HA) replicas][ha-replica]: significantly reduce the risk of downtime and data
  loss due to system failure, and enable services to avoid downtime during routine maintenance.

- [Read replicas][read-replica]: safely scale a service to power your read-intensive
  apps and business intelligence tooling and remove the load from the primary data instance.
-
For MST, see [Failover in Managed Service for TimescaleDB][mst-failover].
For self-hosted TimescaleDB, see [Replication and high availability][self-hosted-ha].

## Rapid recovery

By default, all services have rapid recovery enabled.

Because compute and storage are handled separately in Tiger Cloud, services recover
quickly from compute failures, but usually need a full recovery from backup for storage failures.

- **Compute failure**: the most common cause of database failure. Compute failures
can be caused by hardware failing, or through things like unoptimized queries,
causing increased load that maxes out the CPU usage. In these cases, data on disk is unaffected
and only the compute and memory needs replacing. Tiger Cloud recovery immediately provisions
new compute infrastructure for the service and mounts the existing storage to the new node. Any WAL
that was in memory then replays. This process typically only takes thirty seconds. However,
depending on the amount of WAL that needs replaying this may take up to twenty minutes. Even in the
worst-case scenario, Tiger Cloud recovery is an order of magnitude faster than a standard recovery
from backup.

- **Storage failure**: in the rare occurrence of disk failure, Tiger Cloud automatically
[performs a full recovery from backup][backup-recovery].

If CPU usage for a service runs high for long periods of time, issues such as WAL archiving getting queued
behind other processes can occur. This can cause a failure and could result in a larger data loss.
To avoid data loss, services are monitored for this kind of scenario.


===== PAGE: https://docs.tigerdata.com/use-timescale/upgrades/ =====

# Maintenance and upgrades



Tiger Cloud offers managed database services that provide a stable and reliable environment for your
applications. Each service is based on a specific version of the Postgres database and the TimescaleDB extension.
To ensure that you benefit from the latest features, performance and security improvements, it is important that your
Tiger Cloud service is kept up to date with the latest versions of TimescaleDB and Postgres.

Tiger Cloud has the following upgrade policies:
* **Minor software upgrades**: handled automatically, you do not need to do anything.

  Upgrades are performed on your Tiger Cloud service during a maintenance window that you
  [define to suit your workload][define-maintenance-window]. You can also [manually upgrade TimescaleDB][minor-manual-upgrade].
* **Critical security upgrades**: installed outside normal maintenance windows when necessary, and sometimes require
  a short outage.

  Downtime is usually between 30 seconds and 5 minutes. Tiger Data aims to notify you by email
  if downtime is required, so that you can plan accordingly. However, in some cases this is not possible.
* **Major upgrades**: such as a new version of Postgres are performed [manually by you][manual-upgrade], or [automatically
  by Tiger Cloud][automatic-upgrade].



After a maintenance upgrade, the DNS name remains the same. However, the IP address often changes.



## Minor software upgrades

If you do not [manually upgrade TimescaleDB][minor-manual-upgrade] for non-critical upgrades,
Tiger Cloud performs upgrades automatically in the next available maintenance window. The upgrade is first applied to your services tagged `#dev`, and three weeks later to those tagged `#prod`. [Subscribe][subscribe] to get an email notification before your `#prod` services are upgraded. You can upgrade your `#prod` services manually sooner, if needed.

Most upgrades that occur during your maintenance windows do not require any downtime. This means that there is no
service outage during the upgrade. However, all connections and transactions in progress during the upgrade are
reset. Usually, the service connection is automatically restored after the reset.

Some minor upgrades do require some downtime. This is usually between 30 seconds and 5 minutes. If downtime is required
for an upgrade, Tiger Data endeavors to notify you by email ahead of the upgrade. However, in some cases, we might not be
able to do so. Best practice is to [schedule your maintenance window][define-maintenance-window] so that any downtime
disrupts your workloads as little as possible and [minimize downtime with replicas][minimize-downtime]. If there are no
pending upgrades available during a regular maintenance window, no changes are performed.

To track the status of maintenance events, see the Tiger Cloud [status page][status-page].

### Minimize downtime with replicas

Maintenance upgrades require up to two automatic failovers. Each failover takes less than a few seconds.
Tiger Cloud services with [high-availability replicas and read replicas][replicas-docs] require minimal write downtime during maintenance,
read-only queries keep working throughout.

During a maintenance event, services with replicas perform maintenance on each node independently. When maintenance is
complete on the primary node, it is restarted:
- If the restart takes more than a minute, a replica node is promoted to primary, given that the replica has no
  replication lag. Maintenance now proceeds on the newly promoted replica, following the same
  sequence. If the newly promoted replica takes more than a minute to restart, the former
  primary is promoted back. In total, the process may result in up to two minutes of write
  downtime and two failover events.
- If the maintenance on the primary node is completed within a minute and it comes back online, the replica remains
  the replica.


### Manually upgrade TimescaleDB for non-critical upgrades

Non-critical upgrades are available before the upgrade is performed automatically by Tiger Cloud. To upgrade
TimescaleDB manually:

1. **Connect to your service**

   In [Tiger Cloud Console][cloud-login], select the service you want to upgrade.

1. **Upgrade TimescaleDB**

   Either:
   - Click `SQL Editor`, then run `ALTEREXTENSION timescaledb UPDATE`.
   - Click `⋮`, then `Pause` and `Resume` the service.


Upgrading to a newer version of Postgres allows you to take advantage of new
features, enhancements, and security fixes. It also ensures that you are using a
version of Postgres that's compatible with the newest version of TimescaleDB,
allowing you to take advantage of everything it has to offer. For more
information about feature changes between versions, see the [Tiger Cloud release notes][timescale-changelog],
[supported systems][supported-systems], and the [Postgres release notes][postgres-relnotes].

## Deprecations

To ensure you benefit from the latest features, optimal performance, enhanced security, and full compatibility
with TimescaleDB, Tiger Cloud supports a defined set of Postgres major versions. To reduce the maintenance burden and
continue providing a high-quality managed experience, as Postgres and TimescaleDB evolve, Tiger Data periodically deprecates
older Postgres versions.

Tiger Data provides advance notification to allow you ample time to plan and perform your upgrade. The timeline
deprecation is as follows:
- **Deprecation notice period begins**: you receive email notification of the deprecation and the timeline for the
  upgrade.
- **Customer self-service upgrade window**: best practice is to [manually upgrade to a new Postgres version][manual-upgrade] in
  this time.
- **Automatic upgrade deadline**: Tiger Cloud performs an [automatic upgrade][automatic-upgrade] of your service.


## Manually upgrade Postgres for a service

Upgrading to a newer version of Postgres enables you to take advantage of new features, enhancements, and security fixes.
It also ensures that you are using a version of Postgres that's compatible with the newest version of TimescaleDB.

For a smooth upgrade experience, make sure you:

*  **Plan ahead**: upgrades cause downtime, so ideally perform an upgrade during a low traffic time.
*  **Run a test upgrade**: [fork your service][operations-forking], then try out the upgrade on the fork before
   running it on your production system. This gives you a good idea of what happens during the upgrade, and how long it
   might take.
*  **Keep a copy of your service**: if you're worried about losing your data,
   [fork your service][operations-forking] without upgrading, and keep this duplicate of your service.
   To reduce cost, you can immediately pause this fork and only pay for storage until you are comfortable deleting it
   after the upgrade is complete.



Tiger Cloud services with replicas cannot be upgraded. To upgrade a service
with a replica, you must first delete the replica and then upgrade the service.



The following table shows you the compatible versions of Postgres and TimescaleDB.

| TimescaleDB version |Postgres 17|Postgres 16|Postgres 15|Postgres 14|Postgres 13|Postgres 12|Postgres 11|Postgres 10|
|-----------------------|-|-|-|-|-|-|-|-|
| 2.22.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.21.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.20.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.17 - 2.19           |✅|✅|✅|✅|❌|❌|❌|❌|❌|
| 2.16.x                |❌|✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.13 - 2.15           |❌|✅|✅|✅|✅|❌|❌|❌|❌|
| 2.12.x                |❌|❌|✅|✅|✅|❌|❌|❌|❌|
| 2.10.x                |❌|❌|✅|✅|✅|✅|❌|❌|❌|
| 2.5 - 2.9             |❌|❌|❌|✅|✅|✅|❌|❌|❌|
| 2.4                   |❌|❌|❌|❌|✅|✅|❌|❌|❌|
| 2.1 - 2.3             |❌|❌|❌|❌|✅|✅|✅|❌|❌|
| 2.0                   |❌|❌|❌|❌|❌|✅|✅|❌|❌
| 1.7                   |❌|❌|❌|❌|❌|✅|✅|✅|✅|

We recommend not using TimescaleDB with Postgres 17.1, 16.5, 15.9, 14.14, 13.17, 12.21.
These minor versions [introduced a breaking binary interface change][postgres-breaking-change] that,
once identified, was reverted in subsequent minor Postgres versions 17.2, 16.6, 15.10, 14.15, 13.18, and 12.22.
When you build from source, best practice is to build with Postgres 17.2, 16.6, etc and higher.
Users of [Tiger Cloud](https://console.cloud.timescale.com/) and platform packages for Linux, Windows, MacOS,
Docker, and Kubernetes are unaffected.

For more information about feature changes between versions, see the
[Postgres release notes][postgres-relnotes] and
[TimescaleDB release notes][timescale-relnotes].



Your Tiger Cloud service is unavailable until the upgrade is complete. This can take up to 20 minutes. Best practice is to
test on a fork first, so you can estimate how long the upgrade will take.



To upgrade your service to a newer version of Postgres:

1. **Connect to your service**

   In [Tiger Cloud Console][cloud-login], select the service you want to upgrade.
1. **Disable high-availability replicas**

   1. Click `Operations` > `High Availability`, then click `Change configuaration`.
   1. Select `Non-production  (No replica)`, then click `Change configuration`.

1. **Disable read replicas**

   1. Click `Operations` > `Read scaling`, then click the trash icon next to all replica sets.

1. **Upgrade Postgres**
   1. Click `Operations` > `Service Upgrades`.
   1. Click `Upgrade service`, then confirm that you are ready to start the upgrade.

   Your Tiger Cloud service is unavailable until the upgrade is complete. This normally takes up to 20 minutes.
   However, it can take longer if you have a large or complex service.

   When the upgrade is finished, your service automatically resumes normal
   operations. If the upgrade is unsuccessful, the service returns to the state
   it was in before you started the upgrade.

1. **Enable high-availability replicas and replace your read replicas**

## Automatic Postgres upgrades for a service

If you do not manually upgrade your services within the [customer self-service upgrade window][deprecation-window],
Tiger Cloud performs an automatic upgrade. Automatic upgrades can result in downtime, best practice is to
[manually upgrade your services][manual-upgrade] during a low-traffic period for your application.

During an automatic upgrade:
1. Any configured [high-availability replicas][hareplica] or [read replicas][readreplica] are temporarily removed.
1. The primary service is upgraded.
1. High-availability replicas and read replicas are added back to the service.


## Define your maintenance window

When you are considering your maintenance window schedule, best practice is to choose a day and time that usually
has very low activity, such as during the early hours of the morning, or over the weekend. This helps minimize the
impact of a short service interruption. Alternatively, you might prefer to have your maintenance window occur during
office hours, so that you can monitor your system during the upgrade.

To change your maintenance window:

1. **Connect to your service**

   In [Tiger Cloud Console][cloud-login], select the service you want to manage.
1. **Set your maintenance window**
   1. Click `Operations` > `Environment`, then click  `Change maintenance window`.
       ![Maintenance and upgrades](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-maintenance-upgrades.png)
   1. Select the maintence window start time, then click `Apply`.

   Maintenance windows can run for up to four hours.


===== PAGE: https://docs.tigerdata.com/use-timescale/extensions/ =====

# Postgres extensions

The following Postgres extensions are installed with each Tiger Cloud service:

- [Tiger Data extensions][timescale-extensions]
- [Postgres built-in extensions][built-ins]
- [Third-party extensions][third-party]

## Tiger Data extensions

| Extension                                   | Description                                | Enabled by default                                                    |
|---------------------------------------------|--------------------------------------------|-----------------------------------------------------------------------|
| [pgai][pgai]                                | Helper functions for AI workflows          | For [AI-focused][services] services                            |
| [pg_textsearch][pg_textsearch]              | [BM25][bm25-wiki]-based full-text search   | Currently early access. For development and staging environments only |
| [pgvector][pgvector]                        | Vector similarity search for Postgres           | For [AI-focused][services] services                            |
| [pgvectorscale][pgvectorscale]              | Advanced indexing for vector data          | For [AI-focused][services] services                            |
| [timescaledb_toolkit][timescaledb-toolkit]  | TimescaleDB Toolkit                        | For [Real-time analytics][services] services                   |
| [timescaledb][timescaledb]                  | TimescaleDB                                | For all services                                               |

## Postgres built-in extensions

| Extension                                | Description                                                            | Enabled by default      |
|------------------------------------------|------------------------------------------------------------------------|-------------------------|
| [autoinc][autoinc]                       | Functions for autoincrementing fields                                  | -                       |
| [amcheck][amcheck]                       | Functions for verifying relation integrity                             | -                       |
| [bloom][bloom]                           | Bloom access method - signature file-based index                       | -                       |
| [bool_plperl][bool-plper]                | Transform between bool and plperl                                      | -                       |
| [btree_gin][btree-gin]                   | Support for indexing common datatypes in GIN                           | -                       |
| [btree_gist][btree-gist]                 | Support for indexing common datatypes in GiST                          | -                       |
| [citext][citext]                         | Data type for case-insensitive character strings                       | -                       |
| [cube][cube]                             | Data type for multidimensional cubes                                   | -                       |
| [dict_int][dict-int]                     | Text search dictionary template for integers                           | -                       |
| [dict_xsyn][dict-xsyn]                   | Text search dictionary template for extended synonym processing        | -                       |
| [earthdistance][earthdistance]           | Calculate great-circle distances on the surface of the Earth           | -                       |
| [fuzzystrmatch][fuzzystrmatch]           | Determine similarities and distance between strings                    | -                       |
| [hstore][hstore]                         | Data type for storing sets of (key, value) pairs                       | -                       |
| [hstore_plperl][hstore]                  | Transform between hstore and plperl                                    | -                       |
| [insert_username][insert-username]       | Functions for tracking who changed a table                             | -                       |
| [intagg][intagg]                         | Integer aggregator and enumerator (obsolete)                           | -                       |
| [intarray][intarray]                     | Functions, operators, and index support for 1-D arrays of integers     | -                       |
| [isn][isn]                               | Data types for international product numbering standards               | -                       |
| [jsonb_plperl][jsonb-plperl]             | Transform between jsonb and plperl                                     | -                       |
| [lo][lo]                                 | Large object maintenance                                               | -                       |
| [ltree][ltree]                           | Data type for hierarchical tree-like structures                        | -                       |
| [moddatetime][moddatetime]               | Functions for tracking last modification time                          | -                       |
| [old_snapshot][old-snapshot]             | Utilities in support of `old_snapshot_threshold`                       | -                       |
| [pgcrypto][pgcrypto]                     | Cryptographic functions                                                | -                       |
| [pgrowlocks][pgrowlocks]                 | Show row-level locking information                                     | -                       |
| [pgstattuple][pgstattuple]               | Obtain tuple-level statistics                                          | -                       |
| [pg_freespacemap][pg-freespacemap]       | Examine the free space map (FSM)                                       | -                       |
| [pg_prewarm][pg-prewarm]                 | Prewarm relation data                                                  | -                       |
| [pg_stat_statements][pg-stat-statements] | Track execution statistics of all SQL statements executed              | For all services |
| [pg_trgm][pg-trgm]                       | Text similarity measurement and index searching based on trigrams      | -                       |
| [pg_visibility][pg-visibility]           | Examine the visibility map (VM) and page-level visibility info         | -                       |
| [plperl][plperl]                         | PL/Perl procedural language                                            | -                       |
| [plpgsql][plpgsql]                       | SQL procedural language                                                | For all services |
| [postgres_fdw][postgres-fdw]             | Foreign data wrappers                                                  | For all services |
| [refint][refint]                         | Functions for implementing referential integrity (obsolete)            | -                       |
| [seg][seg]                               | Data type for representing line segments or floating-point intervals   | -                       |
| [sslinfo][sslinfo]                       | Information about SSL certificates                                     | -                       |
| [tablefunc][tablefunc]                   | Functions that manipulate whole tables, including crosstab             | -                       |
| [tcn][tcn]                               | Trigger change notifications                                           | -                       |
| [tsm_system_rows][tsm-system-rows]       | `TABLESAMPLE` method which accepts the number of rows as a limit       | -                       |
| [tsm_system_time][tsm-system-time]       | `TABLESAMPLE` method which accepts the time in milliseconds as a limit | -                       |
| [unaccent][unaccent]                     | Text search dictionary that removes accents                            | -                       |
| [uuid-ossp][uuid-ossp]                   | Generate universally unique identifiers (UUIDs)                        | -                       |

## Third-party extensions

| Extension                                        | Description                                                             | Enabled by default                                   |
|--------------------------------------------------|-------------------------------------------------------------------------|------------------------------------------------------|
| [h3][h3]                                         | H3 bindings for Postgres                                                     | -                                                    |
| [pgaudit][pgaudit]                               | Detailed session and/or object audit logging                            | -                                                    |
| [pgpcre][pgpcre]                                 | Perl-compatible RegEx                                                   | -                                                    |
| [pg_cron][pgcron]                                | SQL commands that you can schedule and run directly inside the database | [Contact us](mailto:support@tigerdata.com) to enable |
| [pg_repack][pgrepack]                            | Table reorganization in Postgres with minimal locks                          | -                                                    |
| [pgrouting][pgrouting]                           | Geospatial routing functionality                                        | -                                                    |
| [postgis][postgis]                               | PostGIS geometry and geography spatial types and functions              | -                                                    |
| [postgis_raster][postgis-raster]                 | PostGIS raster types and functions                                      | -                                                    |
| [postgis_sfcgal][postgis-sfcgal]                 | PostGIS SFCGAL functions                                                | -                                                    |
| [postgis_tiger_geocoder][postgis-tiger-geocoder] | PostGIS Tiger Cloud geocoder and reverse geocoder                       | -                                                    |
| [postgis_topology][postgis-topology]             | PostGIS topology spatial types and functions                            | -                                                    |
| [unit][unit]                                     | SI units for Postgres                                                        | -                                                    |


===== PAGE: https://docs.tigerdata.com/use-timescale/backup-restore/ =====

# Back up and recover your Tiger Cloud services



Tiger Cloud provides comprehensive backup and recovery solutions to protect your data, including automatic daily backups,
cross-region protection, and point-in-time recovery.

## Automatic backups

Tiger Cloud automatically handles backup for your Tiger Cloud services using the `pgBackRest` tool. You don't need to perform
backups manually. What's more, with [cross-region backup][cross-region], you are protected when an entire AWS region goes down.

Tiger Cloud automatically creates one full backup every week, and incremental backups every day in the same region as
your service. Additionally, all [Write-Ahead Log (WAL)][wal] files are retained back to the oldest full backup.
This means that you always have a full backup available for the current and previous week:

![Backup in Tiger](https://assets.timescale.com/docs/images/database-backup-recovery.png)

On [Scale and Performance][pricing-and-account-management] pricing plans, you can check the list of backups for the previous 14 days in Tiger Cloud Console. To do so, select your service, then click `Operations` > `Backup and restore` > `Backup history`.

In the event of a storage failure, a service automatically recovers from a backup
to the point of failure. If the whole availability zone goes down, your Tiger Cloud services are recovered in a different zone. In the event of a user error, you can [create a point-in-time recovery fork][create-fork].

## Enable cross-region backup

<Availability products={['cloud']} price_plans={['enterprise']} />

For added reliability, you can enable cross-region backup. This protects your data when an entire AWS region goes down. In this case, you have two identical backups of your service at any time, but one of them is in a different AWS region. Cross-region backups are updated daily and weekly in the same way as a regular backup. You can have one cross-region backup for a service.

You enable cross-region backup when you create a service, or configure it for an existing service in Tiger Cloud Console:

1. In [Console][console], select your service and click `Operations` > `Backup & restore`.

1. In `Cross-region backup`, select the region in the dropdown and click `Enable backup`.

   ![Create cross-region backup](https://assets.timescale.com/docs/images/tiger-cloud-console/create-cross-region-backup-in-tiger-console.png)

   You can now see the backup, its region, and creation date in a list.

You can have one cross-region backup per service. To change the region of your backup:

1. In [Console][console], select your service and click `Operations` > `Backup & restore`.

1. Click the trash icon next to the existing backup to disable it.

   ![Disable cross-region backup](https://assets.timescale.com/docs/images/tiger-cloud-console/cross-region-backup-list-in-tiger-console.png)

1. Create a new backup in a different region.

## Create a point-in-time recovery fork

<Availability products={['cloud']} />

To recover your service from a destructive or unwanted action, create a point-in-time recovery fork. You can
recover a service to any point within the period [defined by your pricing plan][pricing-and-account-management].
The provision time for the recovery fork is typically less than twenty minutes, but can take longer depending on the
amount of WAL to be replayed. The original service stays untouched to avoid losing data created since the time
of recovery.

All tiered data remains recoverable during the PITR period. When restoring to any point-in-time recovery fork, your
service contains all data that existed at that moment - whether it was stored in high-performance or low-cost
storage.

When you restore a recovery fork:
- Data restored from a PITR point is placed into high-performance storage
- The tiered data, as of that point in time, remains in tiered storage



To avoid paying for compute for the recovery fork and the original service, pause the original to only pay
storage costs.

You initiate a point-in-time recovery from a same-region or cross-region backup in Tiger Cloud Console:





1.  In [Tiger Cloud Console][console], from the `Services` list, ensure the service
    you want to recover has a status of `Running` or `Paused`.
1.  Navigate to `Operations` > `Service management` and click `Create recovery fork`.
1.  Select the recovery point, ensuring the correct time zone (UTC offset).
1.  Configure the fork.

    ![Create recovery fork](https://assets.timescale.com/docs/images/tiger-cloud-console/create-recovery-fork-tiger-console.png)

    You can configure the compute resources, add an HA replica, tag your fork, and
    add a connection pooler. Best practice is to match
    the same configuration you had at the point you want to recover to.
1.  Confirm by clicking `Create recovery fork`.

    A fork of the service is created. The recovered service shows in `Services` with a label specifying which service it has been forked from.

1.  Update the connection strings in your app

    Since the point-in-time recovery is done in a fork, to migrate your
    application to the point of recovery, change the connection
    strings in your application to use the fork.





[Contact us](mailto:support@tigerdata.com), and we will assist in recovering your service.






## Create a service fork

To manage development forks:

1. **Install Tiger CLI**

   Use the terminal to install the CLI:




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    brew install --cask timescale/tap/tiger-cli
    ```





    ```shell
    curl -fsSL https://cli.tigerdata.com | sh
    ```





1. **Set up API credentials**

   1. Log Tiger CLI into your Tiger Data account:

      ```shell
      tiger auth login
      ```
      Tiger CLI opens Console in your browser. Log in, then click `Authorize`.

      You can have a maximum of 10 active client credentials. If you get an error, open [credentials][rest-api-credentials]
      and delete an unused credential.

   1. Select a Tiger Cloud project:

      ```terminaloutput
      Auth URL is: https://console.cloud.timescale.com/oauth/authorize?client_id=lotsOfURLstuff
      Opening browser for authentication...
      Select a project:

      > 1. Tiger Project (tgrproject)
      2. YourCompany (Company wide project) (cpnproject)
      3. YourCompany Department (dptproject)

      Use ↑/↓ arrows or number keys to navigate, enter to select, q to quit
      ```
      If only one project is associated with your account, this step is not shown.

      Where possible, Tiger CLI stores your authentication information in the system keychain/credential manager.
      If that fails, the credentials are stored in `~/.config/tiger/credentials` with restricted file permissions (600).
      By default, Tiger CLI stores your configuration in `~/.config/tiger/config.yaml`.

1. **Test your authenticated connection to Tiger Cloud by listing services**

    ```bash
    tiger service list
    ```

   This call returns something like:
    - No services:
      ```terminaloutput
      🏜️  No services found! Your project is looking a bit empty.
      🚀 Ready to get started? Create your first service with: tiger service create
      ```
    - One or more services:

      ```terminaloutput
      ┌────────────┬─────────────────────┬────────┬─────────────┬──────────────┬──────────────────┐
      │ SERVICE ID │        NAME         │ STATUS │    TYPE     │    REGION    │     CREATED      │
      ├────────────┼─────────────────────┼────────┼─────────────┼──────────────┼──────────────────┤
      │ tgrservice │ tiger-agent-service │ READY  │ TIMESCALEDB │ eu-central-1 │ 2025-09-25 16:09 │
      └────────────┴─────────────────────┴────────┴─────────────┴──────────────┴──────────────────┘
      ```

1. **Fork the service**

   ```shell
    tiger service fork tgrservice --now --no-wait --name bob
   ```
   By default a fork matches the resource of the parent Tiger Cloud services. For paid plans specify `--cpu` and/or `--memory` for dedicated resources.

   You see something like:

    ```terminaloutput
    🍴 Forking service 'tgrservice' to create 'bob' at current state...
    ✅ Fork request accepted!
    📋 New Service ID: <service_id>
    🔐 Password saved to system keyring for automatic authentication
    🎯 Set service '<service_id>' as default service.
    ⏳ Service is being forked. Use 'tiger service list' to check status.
    ┌───────────────────┬──────────────────────────────────────────────────────────────────────────────────────────────────┐
    │     PROPERTY      │                                              VALUE                                               │
    ├───────────────────┼──────────────────────────────────────────────────────────────────────────────────────────────────┤
    │ Service ID        │ <service_id>                                                                                       │
    │ Name              │ bob                                                                                              │
    │ Status            │                                                                                                  │
    │ Type              │ TIMESCALEDB                                                                                      │
    │ Region            │ eu-central-1                                                                                     │
    │ CPU               │ 0.5 cores (500m)                                                                                 │
    │ Memory            │ 2 GB                                                                                             │
    │ Direct Endpoint   │ <service-id>.<project-id>.tsdb.cloud.timescale.com:<port>                                             │
    │ Created           │ 2025-10-08 13:58:07 UTC                                                                          │
    │ Connection String │ postgresql://tsdbadmin@<service-id>.<project-id>.tsdb.cloud.timescale.com:<port>/tsdb?sslmode=require │
    └───────────────────┴──────────────────────────────────────────────────────────────────────────────────────────────────┘
   ```

1. **When you are done, delete your forked service**

    1. Use the CLI to request service delete:

       ```shell
       tiger service delete <service_id>
       ```
    1. Validate the service delete:

        ```terminaloutput
        Are you sure you want to delete service '<service_id>'? This operation cannot be undone.
        Type the service ID '<service_id>' to confirm:
        <service_id>
        ```
       You see something like:
        ```terminaloutput
        🗑️  Delete request accepted for service '<service_id>'.
        ✅ Service '<service_id>' has been successfully deleted.
        ```


===== PAGE: https://docs.tigerdata.com/use-timescale/fork-services/ =====

# Fork services



Modern development is highly iterative. Developers and AI agents need safe spaces to test changes before deploying them
to production. Forkable services make this natural and easy. Spin up a branch, run your test, throw it away, or
merge it back.

A fork is an exact copy of a service at a specific point in time, with its own independent data and configuration,
including:
- The database data and schema
- Configuration
- An admin `tsdbadmin` user with a new password

Forks are fully independent. Changes to the fork don't affect the parent service. You can query
them, run migrations, add indexes, or test new features against the fork without affecting the original service.

Forks are a powerful way to share production-scale data safely. Testing, BI and data science teams often need access
to real datasets to build models or generate insights. With forkable services, you easily create fast, zero-copy
branches of a production service that are isolated from production, but contain all the data needed for
analysis. Rapid fork creation dramatically reduces friction getting insights from live data.

## Understand service forks

You can use service forks for disaster recovery, CI/CD automation, and testing and development. For example, you
can automatically test a major Postgres upgrade on a fork before applying it to your production service.

Tiger Cloud offers the following fork strategies:

- `now`: create a fresh fork of your database at the current time.
   Use when:
   - You need the absolute latest data
   - Recent changes must be included in the fork

- `last-snapshot`: fork from the most recent [automatic backup or snapshot][automatic-backups].
  Use when:
  - You want the fastest possible fork creation
  - Slightly behind current data is acceptable

- `timestamp`: fork from a specific point in time within your [retention period][pricing].
  Use when:
  - Disaster recovery from a known-good state
  - Investigating issues that occurred at a specific time
  - Testing "what-if" scenarios from historical data

The retention period for point-in-time recovery and forking depends on your [pricing plan][pricing-plan-features].

### Fork creation speed

Fork creation speed depends on your type of service you want to create:

- Free: ~30-90 seconds. Uses a Copy-on-Write storage architecture with zero-copy between a fork and the parent.
- Paid: varies with the size of your service, typically 5-20+ minutes. Uses tradional storage architecture
   with backup restore + WAL replay.

### Billing

You can fork a free service to a free or a paid service. However, you cannot fork a paid
service to a free service.

Billing on storage works in the following way:

- High-performance storage:
  - Copy-on-Write: you are only billed for storage for the chunks that diverge from the parent service.
  - Traditional: you are billed for storage for the whole service.
- Object storage tier:
  - [Tiered data][data-tiering] is shared across forks using copy-on-write and traditional storage:
  - Chunks in tiered storage are only billed once, regardless of the number of forks
  - Only new or modified chunks in a fork incur additional costs

For details, see [Replicas and forks with tiered data][tiered-forks].

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

## Manage forks using Tiger CLI

To manage development forks:

1. **Install Tiger CLI**

   Use the terminal to install the CLI:




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    brew install --cask timescale/tap/tiger-cli
    ```





    ```shell
    curl -fsSL https://cli.tigerdata.com | sh
    ```





1. **Set up API credentials**

   1. Log Tiger CLI into your Tiger Data account:

      ```shell
      tiger auth login
      ```
      Tiger CLI opens Console in your browser. Log in, then click `Authorize`.

      You can have a maximum of 10 active client credentials. If you get an error, open [credentials][rest-api-credentials]
      and delete an unused credential.

   1. Select a Tiger Cloud project:

      ```terminaloutput
      Auth URL is: https://console.cloud.timescale.com/oauth/authorize?client_id=lotsOfURLstuff
      Opening browser for authentication...
      Select a project:

      > 1. Tiger Project (tgrproject)
      2. YourCompany (Company wide project) (cpnproject)
      3. YourCompany Department (dptproject)

      Use ↑/↓ arrows or number keys to navigate, enter to select, q to quit
      ```
      If only one project is associated with your account, this step is not shown.

      Where possible, Tiger CLI stores your authentication information in the system keychain/credential manager.
      If that fails, the credentials are stored in `~/.config/tiger/credentials` with restricted file permissions (600).
      By default, Tiger CLI stores your configuration in `~/.config/tiger/config.yaml`.

1. **Test your authenticated connection to Tiger Cloud by listing services**

    ```bash
    tiger service list
    ```

   This call returns something like:
    - No services:
      ```terminaloutput
      🏜️  No services found! Your project is looking a bit empty.
      🚀 Ready to get started? Create your first service with: tiger service create
      ```
    - One or more services:

      ```terminaloutput
      ┌────────────┬─────────────────────┬────────┬─────────────┬──────────────┬──────────────────┐
      │ SERVICE ID │        NAME         │ STATUS │    TYPE     │    REGION    │     CREATED      │
      ├────────────┼─────────────────────┼────────┼─────────────┼──────────────┼──────────────────┤
      │ tgrservice │ tiger-agent-service │ READY  │ TIMESCALEDB │ eu-central-1 │ 2025-09-25 16:09 │
      └────────────┴─────────────────────┴────────┴─────────────┴──────────────┴──────────────────┘
      ```

1. **Fork the service**

   ```shell
    tiger service fork tgrservice --now --no-wait --name bob
   ```
   By default a fork matches the resource of the parent Tiger Cloud services. For paid plans specify `--cpu` and/or `--memory` for dedicated resources.

   You see something like:

    ```terminaloutput
    🍴 Forking service 'tgrservice' to create 'bob' at current state...
    ✅ Fork request accepted!
    📋 New Service ID: <service_id>
    🔐 Password saved to system keyring for automatic authentication
    🎯 Set service '<service_id>' as default service.
    ⏳ Service is being forked. Use 'tiger service list' to check status.
    ┌───────────────────┬──────────────────────────────────────────────────────────────────────────────────────────────────┐
    │     PROPERTY      │                                              VALUE                                               │
    ├───────────────────┼──────────────────────────────────────────────────────────────────────────────────────────────────┤
    │ Service ID        │ <service_id>                                                                                       │
    │ Name              │ bob                                                                                              │
    │ Status            │                                                                                                  │
    │ Type              │ TIMESCALEDB                                                                                      │
    │ Region            │ eu-central-1                                                                                     │
    │ CPU               │ 0.5 cores (500m)                                                                                 │
    │ Memory            │ 2 GB                                                                                             │
    │ Direct Endpoint   │ <service-id>.<project-id>.tsdb.cloud.timescale.com:<port>                                             │
    │ Created           │ 2025-10-08 13:58:07 UTC                                                                          │
    │ Connection String │ postgresql://tsdbadmin@<service-id>.<project-id>.tsdb.cloud.timescale.com:<port>/tsdb?sslmode=require │
    └───────────────────┴──────────────────────────────────────────────────────────────────────────────────────────────────┘
   ```

1. **When you are done, delete your forked service**

    1. Use the CLI to request service delete:

       ```shell
       tiger service delete <service_id>
       ```
    1. Validate the service delete:

        ```terminaloutput
        Are you sure you want to delete service '<service_id>'? This operation cannot be undone.
        Type the service ID '<service_id>' to confirm:
        <service_id>
        ```
       You see something like:
        ```terminaloutput
        🗑️  Delete request accepted for service '<service_id>'.
        ✅ Service '<service_id>' has been successfully deleted.
        ```

## Manage forks using Console

To manage development forks:

1.  In [Tiger Cloud Console][console], from the `Services` list, ensure the service
    you want to recover has a status of `Running` or `Paused`.
1.  Navigate to `Operations` > `Service Management` and click `Fork service`.
1.  Configure the fork, then click `Fork service`.

    A fork of the service is created. The forked service shows in `Services` with a label
    specifying which service it has been forked from.

    ![See the forked service](https://assets.timescale.com/docs/images/tsc-forked-service.webp)

1.  Update the connection strings in your app to use the fork.

## Integrate service forks in your CI/CD pipeline

To fork your Tiger Cloud service using GitHub actions:

1. **Store your Tiger Cloud API key as a GitHub Actions secret**

   1. In [Tiger Cloud Console][rest-api-credentials], click `Create credentials`.
   2. Save the `Public key` and `Secret key` locally, then click `Done`.
   1. In your GitHub repository, click `Settings`, open `Secrets and variables`, then click `Actions`.
   3. Click `New repository secret`, then set `Name` to `TIGERDATA_API_KEY`
   4. Set `Secret` to your Tiger Cloud API key in the following format `<Public key>:<Secret key>`, then click `Add secret`.

1. **Add the [GitHub Actions Marketplace][github-action] to your workflow YAML files**

   For example, the following workflow forks a service when a pull request is opened,
   running tests against the fork, then automatically cleans up.

    ```yaml
    name: Test on a service fork
    on: pull_request

    jobs:
      test:
        runs-on: ubuntu-latest
        steps:
          - uses: actions/checkout@v4

          - name: Fork Database
            id: fork
            uses: timescale/fork-service@v1
            with:
              project_id: ${{ secrets.TIGERDATA_PROJECT_ID }}
              service_id: ${{ secrets.TIGERDATA_SERVICE_ID }}
              api_key: ${{ secrets.TIGERDATA_API_KEY }}
              fork_strategy: last-snapshot
              cleanup: true
              name: pr-${{ github.event.pull_request.number }}

          - name: Run Integration Tests
            env:
              DATABASE_URL: postgresql://tsdbadmin:${{ steps.fork.outputs.initial_password }}@${{ steps.fork.outputs.host }}:${{ steps.fork.outputs.port }}/tsdb?sslmode=require
            run: |
              npm install
              npm test
          - name: Run Migrations
            env:
              DATABASE_URL: postgresql://tsdbadmin:${{ steps.fork.outputs.initial_password }}@${{ steps.fork.outputs.host }}:${{ steps.fork.outputs.port }}/tsdb?sslmode=require
            run: npm run migrate
    ```

    For the full list of inputs, outputs, and configuration options, see the [Tiger Data - Fork Service][github-action] in GitHub marketplace.


===== PAGE: https://docs.tigerdata.com/use-timescale/jobs/ =====

# Jobs in TimescaleDB

TimescaleDB natively includes some job-scheduling policies, such as:

*   [Continuous aggregate policies][caggs] to automatically refresh continuous aggregates
*   [Hypercore policies][setup-hypercore] to optimize and compress historical data
*   [Retention policies][retention] to drop historical data
*   [Reordering policies][reordering] to reorder data within chunks

If these don't cover your use case, you can create and schedule custom-defined jobs to run within
your database. They help you automate periodic tasks that aren't covered by the native policies.

In this section, you see how to:

*   [Create and manage jobs][create-jobs]
*   Set up a [generic data retention][generic-retention] policy that applies across all hypertables
*   Implement [automatic moving of chunks between tablespaces][manage-storage]
*   Automatically [downsample and compress][downsample-compress] older chunks


===== PAGE: https://docs.tigerdata.com/use-timescale/security/ =====

# Security

Learn how Tiger Cloud protects your data and privacy.

*   Learn about [security in Tiger Cloud][overview]
*   Restrict access to your [project][console-rbac]
*   Restrict access to the [data in your service][read-only]
*   Set up [multifactor][mfa] and [SAML][saml] authentication
*   Generate multiple [client credentials][client-credentials] instead of using your username and password
*   Connect with a [stricter SSL mode][ssl]
*   Secure your services with [VPC peering][vpc-peering]
*   Connect to your services from any cloud with [AWS Transit Gateway][transit-gateway]
*   Restrict access with an [IP address allow list][ip-allowlist]


===== PAGE: https://docs.tigerdata.com/use-timescale/limitations/ =====

# Limitations

While TimescaleDB generally offers capabilities that go beyond what
Postgres offers, there are some limitations to using hypertables.

## Hypertable limitations

*   Time dimensions (columns) used for partitioning cannot have NULL values.
*   Unique indexes must include all columns that are partitioning dimensions.
*   `UPDATE` statements that move values between partitions (chunks) are not
    supported. This includes upserts (`INSERT ... ON CONFLICT UPDATE`).
*   Foreign key constraints from a hypertable referencing another hypertable are not supported.


===== PAGE: https://docs.tigerdata.com/use-timescale/tigerlake/ =====

# Integrate data lakes with Tiger Cloud



Tiger Lake enables you to build real-time applications alongside efficient data pipeline management within a single
system. Tiger Lake unifies the Tiger Cloud operational architecture with data lake architectures.

![Tiger Lake architecture](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-lake-integration-tiger.svg)

Tiger Lake is a native integration enabling synchronization between hypertables and relational tables
running in Tiger Cloud services to Iceberg tables running in [Amazon S3 Tables][s3-tables] in your AWS account.



Tiger Lake is currently in private beta. Please contact us to request access.



## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need your [connection details][connection-info].

## Integrate a data lake with your Tiger Cloud service

To connect a Tiger Cloud service to your data lake:





<Procedure >

1. **Set the AWS region to host your table bucket**
   1. In [AWS CloudFormation][cmc], select the current AWS region at the top-right of the page.
   1. Set it to the Region you want to create your table bucket in.

   **This must match the region your Tiger Cloud service is running in**: if the regions do not match AWS charges you for
   cross-region data transfer.

1. **Create your CloudFormation stack**
   1. Click `Create stack`, then select `With new resources (standard)`.
   1. In `Amazon S3 URL`, paste the following URL, then click `Next`.

      ```http request
      https://tigerlake.s3.us-east-1.amazonaws.com/tigerlake-connect-cloudformation.yaml
      ```

   1. In `Specify stack details`, enter the following details, then click `Next`:
      * `Stack Name`: a name for this CloudFormation stack
      * `BucketName`: a name for this S3 table bucket
      * `ProjectID` and `ServiceID`: enter the [connection details][get-project-id] for your Tiger Lake service
   1. In `Configure stack options` check `I acknowledge that AWS CloudFormation might create IAM resources`, then
      click `Next`.
   1. In `Review and create`, click `Submit`, then wait for the deployment to complete.
      AWS deploys your stack and creates the S3 table bucket and IAM role.
   1. Click `Outputs`, then copy all four outputs.

1. **Connect your service to the data lake**

   1. In [Tiger Cloud Console][services-portal], select the service you want to integrate with AWS S3 Tables, then click
      `Connectors`.

   1. Select the Apache Iceberg connector and supply the:
      - ARN of the S3Table bucket
      - ARN of a role with permissions to write to the table bucket

   Provisioning takes a couple of minutes.





<Procedure >

1. **Create your CloudFormation stack**

   Replace the following values in the command, then run it from the terminal:

   * `Region`: region of the S3 table bucket
   * `StackName`: the name for this CloudFormation stack
   * `BucketName`: the name of the S3 table bucket to create
   * `ProjectID`: enter your Tiger Cloud service [connection details][get-project-id]
   * `ServiceID`: enter your Tiger Cloud service [connection details][get-project-id]

   ```shell
   aws cloudformation create-stack \
    --capabilities CAPABILITY_IAM \
    --template-url https://tigerlake.s3.us-east-1.amazonaws.com/tigerlake-connect-cloudformation.yaml \
    --region <Region> \
    --stack-name <StackName> \
    --parameters \
      ParameterKey=BucketName,ParameterValue="<BucketName>" \
      ParameterKey=ProjectID,ParameterValue="<ProjectID>" \
      ParameterKey=ServiceID,ParameterValue="<ServiceID>"
   ```

  Setting up the integration through Tiger Cloud Console in Tiger Cloud, provides a convenient copy-paste option with the
  placeholders populated.

1. **Connect your service to the data lake**

   1. In [Tiger Cloud Console][services-portal], select the service you want to integrate with AWS S3 Tables, then click
      `Connectors`.

   1. Select the Apache Iceberg connector and supply the:
      - ARN of the S3Table bucket
      - ARN of a role with permissions to write to the table bucket

   Provisioning takes a couple of minutes.





<Procedure >

1. **Create a S3 Bucket**

   1. Set the AWS region to host your table bucket
      1. In [Amazon S3 console][s3-console], select the current AWS region at the top-right of the page.
      2. Set it to the Region your you want to create your table bucket in.

      **This must match the region your Tiger Cloud service is running in**: if the regions do not match AWS charges you for
      cross-region data transfer.
   1. In the left navigation pane, click `Table buckets`, then click `Create table bucket`.
   1. Enter `Table bucket name`, then click `Create table bucket`.
   1. Copy the `Amazon Resource Name (ARN)` for your table bucket.

1. **Create an ARN role**
   1. In [IAM Dashboard][iam-dashboard], click `Roles` then click `Create role`
   1. In `Select trusted entity`, click `Custom trust policy`, replace the **Custom trust policy** code block with the
      following:

      ```json
      {
          "Version": "2012-10-17",
          "Statement": [
              {
                  "Effect": "Allow",
                  "Principal": {
                      "AWS": "arn:aws:iam::142548018081:root"
                  },
                  "Action": "sts:AssumeRole",
                  "Condition": {
                      "StringEquals": {
                          "sts:ExternalId": "<ProjectID>/<ServiceID>"
                      }
                  }
              }
          ]
      }
      ```

      `"Principal": { "AWS": "arn:aws:iam::123456789012:root" }` does not mean `root` access. This delegates
        permissions to the entire AWS account, not just the root user.

   1. Replace `<ProjectID>` and `<ServiceID>` with the the [connection details][get-project-id] for your Tiger Lake
         service, then click `Next`.

   1. In `Permissions policies`. click `Next`.
   1. In `Role details`, enter `Role name`, then click `Create role`.
   1. In `Roles`, select the role you just created, then click `Add Permissions` > `Create inline policy`.
   1. Select `JSON` then replace the `Policy editor` code block with the following:

         ```json
         {
           "Version": "2012-10-17",
           "Statement": [
             {
               "Sid": "BucketOps",
               "Effect": "Allow",
               "Action": [
                 "s3tables:*"
               ],
               "Resource": "<S3TABLE_BUCKET_ARN>"
             },
             {
               "Sid": "BucketTableOps",
               "Effect": "Allow",
               "Action": [
                 "s3tables:*"
               ],
               "Resource": "<S3TABLE_BUCKET_ARN>/table/*"
             }
           ]
         }
         ```
   1. Replace `<S3TABLE_BUCKET_ARN>` with the `Amazon Resource Name (ARN)` for the table bucket you just created.
   1. Click `Next`, then give the inline policy a name and click `Create policy`.

1. **Connect your service to the data lake**

   1. In [Tiger Cloud Console][services-portal], select the service you want to integrate with AWS S3 Tables, then click
      `Connectors`.

   1. Select the Apache Iceberg connector and supply the:
      - ARN of the S3Table bucket
      - ARN of a role with permissions to write to the table bucket

   Provisioning takes a couple of minutes.





## Stream data from your Tiger Cloud service to your data lake

When you start streaming, all data in the table is synchronized to Iceberg. Records are imported in time order, from
oldest to youngest. The write throughput is approximately 40.000 records / second. For larger tables, a full import can
take some time.

For Iceberg to perform update or delete statements, your hypertable or relational table must have a primary key.
This includes composite primary keys.

To stream data from a Postgres relational table, or a hypertable in your Tiger Cloud service to your data lake, run the following
statement:

```sql
ALTER TABLE  SET (
  tigerlake.iceberg_sync = true | false,
  tigerlake.iceberg_partitionby = '<partition_specification>',
  tigerlake.iceberg_namespace = '<namespace>',
  tigerlake.iceberg_table = ''
)
```

* `tigerlake.iceberg_sync`: `boolean`, set to `true` to start streaming, or `false` to stop the stream. A stream
  **cannot** resume after being stopped.
* `tigerlake.iceberg_partitionby`: optional property to define a partition specification in Iceberg. By default the
   Iceberg table is partitioned as `day(<time-column of hypertable>)`. This default behavior is only applicable
   to hypertables. For more information, see [partitioning][partitioning].
* `tigerlake.iceberg_namespace`: optional property to set a namespace, the default is `timescaledb`.
* `tigerlake.iceberg_table`: optional property to specify a different table name. If no name is specified the Postgres table name is used.

### Partitioning intervals

By default, the partition interval for an Iceberg table is one day(time-column) for a hypertable.
Postgres table sync does not enable any partitioning in Iceberg for non-hypertables. You can set it using
[tigerlake.iceberg_partitionby][samples]. The following partition intervals and specifications are supported:

| Interval      | Description                                                               | Source types |
| ------------- |---------------------------------------------------------------------------| --- |
| `hour`        | Extract a date or timestamp day, as days from epoch. Epoch is 1970-01-01. | `date`, `timestamp`, `timestamptz` |
| `day`         | Extract a date or timestamp day, as days from epoch.                      | `date`, `timestamp`, `timestamptz` |
| `month`       | Extract a date or timestamp day, as days from epoch.                      | `date`, `timestamp`, `timestamptz` |
| `year`        | Extract a date or timestamp day, as days from epoch.                      | `date`, `timestamp`, `timestamptz` |
| `truncate[W]` | Value truncated to width W, see [options][iceberg-truncate-options]       |

These partitions define the behavior using the [Iceberg partition specification][iceberg-partition-spec]:

### Sample code

The following samples show you how to tune data sync from a hypertable or a Postgres relational table to your
data lake:

- **Sync a hypertable with the default one-day partitioning interval on the `ts_column` column**

   To start syncing data from a hypertable to your data lake using the default one-day chunk interval as the
   partitioning scheme to the Iceberg table, run the following statement:

   ```sql
   ALTER TABLE my_hypertable SET (tigerlake.iceberg_sync = true);
   ```

   This is equivalent to `day(ts_column)`.

- **Specify a custom partitioning scheme for a hypertable**

   You use the `tigerlake.iceberg_partitionby` property to specify a different partitioning scheme for the Iceberg
   table at sync start.  For example, to enforce an hourly partition scheme from the chunks on `ts_column` on a
   hypertable, run the following statement:

   ```sql
   ALTER TABLE my_hypertable SET (
     tigerlake.iceberg_sync = true,
     tigerlake.iceberg_partitionby = 'hour(ts_column)'
   );
   ```

- **Set the partition to sync relational tables**

   Postgres relational tables do not forward a partitioning scheme to Iceberg, you must specify the partitioning scheme using
   `tigerlake.iceberg_partitionby` when you start the sync. For example, for a standard Postgres table to sync to the Iceberg
   table with daily partitioning , run the following statement:

   ```sql
   ALTER TABLE my_postgres_table SET (
     tigerlake.iceberg_sync = true,
     tigerlake.iceberg_partitionby = 'day(timestamp_col)'
   );
   ```

- **Stop sync to an Iceberg table for a hypertable or a Postgres relational table**

   ```sql
   ALTER TABLE my_hypertable SET (tigerlake.iceberg_sync = false);
   ```

- **Update or add the partitioning scheme of an Iceberg table**

   To change the partitioning scheme of an Iceberg table, you specify the desired partitioning scheme using the `tigerlake.iceberg_partitionby` property.
   For example. if the `samples` table has an hourly (`hour(ts)`) partition on the `ts` timestamp column,
   to change to daily partitioning, call the following statement:

   ```sql
   ALTER TABLE samples SET (tigerlake.iceberg_partitionby = 'day(ts)');
   ```

   This statement is also correct for Iceberg tables without a partitioning scheme.
   When you change the partition, you **do not** have to pause the sync to Iceberg.
   Apache Iceberg handles the partitioning operation in function of the internal implementation.

**Specify a different namespace**

   By default, tables are created in the the `timescaledb` namespace. To specify a different namespace when you start the sync, use the  `tigerlake.iceberg_namespace` property. For example:

   ```sql
   ALTER TABLE my_hypertable SET (
     tigerlake.iceberg_sync = true,
     tigerlake.iceberg_namespace = 'my_namespace'
   );
   ```

**Specify a different Iceberg table name**

   The table name in Iceberg is the same as the source table in Tiger Cloud.
   Some services do not allow mixed case, or have other constraints for table names.
   To define a different table name for the Iceberg table at sync start,  use the `tigerlake.iceberg_table` property. For example:

   ```sql
   ALTER TABLE Mixed_CASE_TableNAME SET (
     tigerlake.iceberg_sync = true,
     tigerlake.iceberg_table = 'my_table_name'
   );
   ```

## Limitations

* Service requires Postgres 17.6 and above is supported.
* Consistent ingestion rates of over 30000 records / second can lead to a lost replication slot. Burst can be feathered out over time.
* [Amazon S3 Tables Iceberg REST][aws-s3-tables] catalog only is supported.
* In order to collect deletes made to data in the columstore, certain columnstore optimizations are disabled for hypertables.
* [Direct Compress][direct-compress] is not supported.
* The `TRUNCATE` statement is not supported, and does not truncate data in the corresponding Iceberg table.
* Data in a hypertable that has been moved to the [low-cost object storage tier][data-tiering] is not synced.
* Writing to the same S3 table bucket from multiple services is not supported, bucket-to-service mapping is one-to-one.
* Iceberg snapshots are pruned automatically if the amount exceeds 2500.


===== PAGE: https://docs.tigerdata.com/use-timescale/troubleshoot-timescaledb/ =====

# Troubleshooting TimescaleDB



If you run into problems when using TimescaleDB, there are a few things that you
can do. There are some solutions to common errors in this section as well as ways to
output diagnostic information about your setup. If you need more guidance, you
can join the community [Slack group][slack] or post an issue on the TimescaleDB
[GitHub][github].

## Common errors

### Error updating TimescaleDB when using a third-party Postgres administration tool

The `ALTER EXTENSION timescaledb UPDATE` command must be the first
command executed upon connection to a database. Some administration tools
execute commands before this, which can disrupt the process. You might
need to manually update the database with `psql`.  See the
[update docs][update-db] for details.

### Log error: could not access file "timescaledb"

If your Postgres logs have this error preventing it from starting up, you
should double-check that the TimescaleDB files have been installed to the
correct location. The installation methods use `pg_config` to get Postgres's
location. However, if you have multiple versions of Postgres installed on the
same machine, the location `pg_config` points to may not be for the version you
expect. To check which version of TimescaleDB is used:

```bash
$ pg_config --version
PostgreSQL 12.3
```

If that is the correct version, double-check that the installation path is
the one you'd expect. For example, for Postgres 11.0 installed via
Homebrew on macOS it should be `/usr/local/Cellar/postgresql/11.0/bin`:

```bash
$ pg_config --bindir
/usr/local/Cellar/postgresql/11.0/bin
```

If either of those steps is not the version you are expecting, you need to
either uninstall the incorrect version of Postgres if you can, or update your
`PATH` environmental variable to have the correct path of `pg_config` listed
first, that is, by prepending the full path:

```bash
export PATH = /usr/local/Cellar/postgresql/11.0/bin:$PATH
```

Then, reinstall TimescaleDB and it should find the correct installation
path.

### ERROR: could not access file "timescaledb-\<version\>": No such file or directory

If the error occurs immediately after updating your version of TimescaleDB and
the file mentioned is from the previous version, it is probably due to an
incomplete update process. Within the greater Postgres server instance, each
database that has TimescaleDB installed needs to be updated with the SQL command
`ALTER EXTENSION timescaledb UPDATE;` while connected to that database.
Otherwise, the database looks for the previous version of the `timescaledb` files.

See [our update docs][update-db] for more info.

### Scheduled jobs stop running

Your scheduled jobs might stop running for various reasons. On self-hosted
TimescaleDB, you can fix this by restarting background workers:

```sql
SELECT _timescaledb_internal.restart_background_workers();
```

On Tiger Cloud and Managed Service for TimescaleDB, restart background workers by doing one of the following:

*   Run `SELECT timescaledb_pre_restore()`, followed by `SELECT
    timescaledb_post_restore()`.
*   Power the service off and on again. This might cause a downtime of a few
    minutes while the service restores from backup and replays the write-ahead
    log.

### Failed to start a background worker

You might see this error message in the logs if background workers aren't
properly configured:

```bash
"<TYPE_OF_BACKGROUND_JOB>": failed to start a background worker
```

To fix this error, make sure that `max_worker_processes`,
`max_parallel_workers`, and `timescaledb.max_background_workers` are properly
set. `timescaledb.max_background_workers` should equal the number of databases
plus the number of concurrent background workers. `max_worker_processes` should
equal the sum of `timescaledb.max_background_workers` and
`max_parallel_workers`.

For more information, see the [worker configuration docs][worker-config].

### Cannot compress chunk

You might see this error message when trying to compress a chunk if
the permissions for the compressed hypertable are corrupt.

```sql
tsdb=> SELECT compress_chunk('_timescaledb_internal._hyper_65_587239_chunk');
ERROR: role 149910 was concurrently dropped
```

This can be caused if you dropped a user for the hypertable before
TimescaleDB 2.5. For this case, the user would be removed from
`pg_authid` but not revoked from the compressed table.

As a result, the compressed table contains permission items that
refer to numerical values rather than existing users (see below for
how to find the compressed hypertable from a normal hypertable):

```sql
tsdb=> \dp _timescaledb_internal._compressed_hypertable_2
                                 Access privileges
 Schema |     Name     | Type  |  Access privileges  | Column privileges | Policies
--------+--------------+-------+---------------------+-------------------+----------
 public | transactions | table | mats=arwdDxt/mats  +|                   |
        |              |       | wizard=arwdDxt/mats+|                   |
        |              |       | 149910=r/mats       |                   |
(1 row)
```

This means that the `relacl` column of `pg_class` needs to be updated
and the offending user removed, but it is not possible to drop a user
by numerical value. Instead, you can use the internal function
`repair_relation_acls` in the `_timescaledb_function` schema:

```sql
tsdb=> CALL _timescaledb_functions.repair_relation_acls();
```


This requires superuser privileges (since you're modifying the
`pg_class` table) and that it removes any user not present in
`pg_authid` from *all* tables, so use with caution.


The permissions are usually corrupted for the hypertable as well, but
not always, so it is better to look at the compressed hypertable to
see if the problem is present. To find the compressed hypertable for
an associated hypertable (`readings` in this case):

```sql
tsdb=> select ht.table_name,
tsdb->        (select format('%I.%I', schema_name, table_name)::regclass
tsdb->           from _timescaledb_catalog.hypertable
tsdb->			where ht.compressed_hypertable_id = id) as compressed_table
tsdb->   from _timescaledb_catalog.hypertable ht
tsdb->  where table_name = 'readings';
  format  |                     format
----------+------------------------------------------------
 readings | _timescaledb_internal._compressed_hypertable_2
(1 row)
```

## Getting more information

### EXPLAINing query performance

Postgres's EXPLAIN feature allows users to understand the underlying query
plan that Postgres uses to execute a query. There are multiple ways that
Postgres can execute a query: for example, a query might be fulfilled using a
slow sequence scan or a much more efficient index scan. The choice of plan
depends on what indexes are created on the table, the statistics that Postgres
has about your data, and various planner settings. The EXPLAIN output let's you
know which plan Postgres is choosing for a particular query. Postgres has a
[in-depth explanation][using explain] of this feature.

To understand the query performance on a hypertable, we suggest first
making sure that the planner statistics and table maintenance is up-to-date on the hypertable
by running `VACUUM ANALYZE <your-hypertable>;`. Then, we suggest running the
following version of EXPLAIN:

```sql
EXPLAIN (ANALYZE on, BUFFERS on) <original query>;
```

If you suspect that your performance issues are due to slow IOs from disk, you
can get even more information by enabling the
[track\_io\_timing][track_io_timing] variable with `SET track_io_timing = 'on';`
before running the above EXPLAIN.

## Dump TimescaleDB meta data

To help when asking for support and reporting bugs,
TimescaleDB includes a SQL script that outputs metadata
from the internal TimescaleDB tables as well as version information.
The script is available in the source distribution in `scripts/`
but can also be [downloaded separately][].
To use it, run:

```bash
psql [your connect flags] -d your_timescale_db < dump_meta_data.sql > dumpfile.txt
```

and then inspect `dump_file.txt` before sending it together with a bug report or support question.

## Debugging background jobs

By default, background workers do not print a lot of information about
execution. The reason for this is to avoid writing a lot of debug
information to the Postgres log unless necessary.

To aid in debugging the background jobs, it is possible to increase
the log level of the background workers without having to restart the
server by setting the `timescaledb.bgw_log_level` GUC and reloading
the configuration.

```sql
ALTER SYSTEM SET timescaledb.bgw_log_level TO 'DEBUG1';
SELECT pg_reload_conf();
```

This variable is set to the value of
[`log_min_messages`][log_min_messages] by default, which typically is
`WARNING`. If the value of [`log_min_messages`][log_min_messages] is
changed in the configuration file, it is used for
`timescaledb.bgw_log_level` when starting the workers.


Both `ALTER SYSTEM` and `pg_reload_conf()` require superuser
privileges by default. Grant `EXECUTE` permissions
to `pg_reload_conf()` and `ALTER SYSTEM` privileges to
`timescaledb.bgw_log_level` if you want this to work for a
non-superuser.

Since `ALTER SYSTEM` privileges only exist on Postgres 15 and later,
the necessary grants for executing these statements only exist on Tiger Cloud for Postgres 15 or later.


### Debug level 1

The amount of information printed at each level varies between jobs,
but the information printed at `DEBUG1` is currently shown below.

| Source            | Event                                                |
|-------------------|------------------------------------------------------|
| All jobs          | Job exit with runtime information                    |
| All jobs          | Job scheduled for fast restart                       |
| Custom job        | Execution started                                    |
| Recompression job | Recompression job completed                          |
| Reorder job       | Chunk reorder completed                              |
| Reorder job       | Chunk reorder started                                |
| Scheduler         | New jobs discovered and added to scheduled jobs list |
| Scheduler         | Scheduling job for launch                            |

### Debug level 2

The amount of information printed at each level varies between jobs,
but the information printed at `DEBUG2` is currently shown below.

Note that all messages at level `DEBUG1` are also printed when you set
the log level to `DEBUG2`, which is [normal Postgres
behaviour][log_min_messages].

| Source    | Event                              |
|-----------|------------------------------------|
| All jobs  | Job found in jobs table            |
| All jobs  | Job starting execution             |
| Scheduler | Scheduled jobs list update started |
| Scheduler | Scheduler dispatching job |

### Debug level 5

| Source    | Event                                |
|-----------|--------------------------------------|
| Scheduler | Scheduled wake up                    |
| Scheduler | Scheduler delayed in dispatching job |


## hypertable chunks are not discoverable by the Postgres CDC service

hypertables require special handling for CDC support. Newly created chunks are not
not published, which means they are  not discoverable by the CDC service.
To fix this problem, use the following trigger to automatically publishe newly created chunks on the replication slot.
Please be aware that TimescaleDB does not provide full CDC support.

```sql
CREATE OR REPLACE FUNCTION ddl_end_trigger_func() RETURNS EVENT_TRIGGER AS
$$
DECLARE
    r RECORD;
    pub NAME;
BEGIN
    FOR r IN SELECT * FROM pg_event_trigger_ddl_commands()
    LOOP
        SELECT pubname INTO pub
        FROM pg_inherits
        JOIN _timescaledb_catalog.hypertable ht
            ON inhparent = format('%I.%I', ht.schema_name, ht.table_name)::regclass
        JOIN pg_publication_tables
            ON schemaname = ht.schema_name AND tablename = ht.table_name
        WHERE inhrelid = r.objid;

        IF NOT pub IS NULL THEN
            EXECUTE format('ALTER PUBLICATION %s ADD TABLE %s', pub, r.objid::regclass);
        END IF;
    END LOOP;
END;
$$ LANGUAGE plpgsql;

CREATE EVENT TRIGGER ddl_end_trigger
ON ddl_command_end WHEN TAG IN ('CREATE TABLE') EXECUTE FUNCTION ddl_end_trigger_func();
```


===== PAGE: https://docs.tigerdata.com/use-timescale/compression/ =====

# Compression



Old API since [TimescaleDB v2.18.0](https://github.com/timescale/timescaledb/releases/tag/2.18.0) Replaced by <a href="https://docs.tigerdata.com/use-timescale/latest/hypercore/">hypercore</a>.

Time-series data can be compressed to reduce the amount of storage required, and
increase the speed of some queries. This is a cornerstone feature of
TimescaleDB. When new data is added to your database, it is in the form of
uncompressed rows. TimescaleDB uses a built-in job scheduler to convert this
data to the form of compressed columns. This occurs across chunks of TimescaleDB
hypertables.


===== PAGE: https://docs.tigerdata.com/tutorials/real-time-analytics-transport/ =====

# Analytics on transport and geospatial data



Real-time analytics refers to the process of collecting, analyzing, and interpreting data instantly as it
is generated. This approach enables you track and monitor activity, and make decisions based on real-time
insights on data stored in a Tiger Cloud service.

![Real-time analytics geolocation](https://assets.timescale.com/docs/images/use-case-rta-grafana-heatmap.png)

This page shows you how to integrate [Grafana][grafana-docs] with a Tiger Cloud service and make insights based on visualization
of data optimized for size and speed in the columnstore.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

* Install and run [self-managed Grafana][grafana-self-managed], or sign up for [Grafana Cloud][grafana-cloud].

## Optimize time-series data in hypertables

Hypertables are Postgres tables in TimescaleDB that automatically partition your time-series data by time. Time-series data represents the way a system, process, or behavior changes over time. Hypertables enable TimescaleDB to work efficiently with time-series data.  Each hypertable is made up of child tables called chunks. Each chunk is assigned a range
of time, and only contains data from that range. When you run a query, TimescaleDB identifies the correct chunk and
runs the query on it, instead of going through the entire table.

[Hypercore][hypercore] is the hybrid row-columnar storage engine in TimescaleDB used by hypertables. Traditional
databases force a trade-off between fast inserts (row-based storage) and efficient analytics
(columnar storage). Hypercore eliminates this trade-off, allowing real-time analytics without sacrificing
transactional capabilities.

Hypercore dynamically stores data in the most efficient format for its lifecycle:

* **Row-based storage for recent data**: the most recent chunk (and possibly more) is always stored in the rowstore,
   ensuring fast inserts, updates, and low-latency single record queries. Additionally, row-based storage is used as a
   writethrough for inserts and updates to columnar storage.
* **Columnar storage for analytical performance**: chunks are automatically compressed into the columnstore, optimizing
   storage efficiency and accelerating analytical queries.

Unlike traditional columnar databases, hypercore allows data to be inserted or modified at any stage, making it a
flexible solution for both high-ingest transactional workloads and real-time analytics—within a single database.

Because TimescaleDB is 100% Postgres, you can use all the standard Postgres tables, indexes, stored
procedures, and other objects alongside your hypertables. This makes creating and working with hypertables similar
to standard Postgres.

1.  **Import time-series data into a hypertable**

    1. Unzip [nyc_data.tar.gz](https://assets.timescale.com/docs/downloads/nyc_data.tar.gz) to a `<local folder>`.

       This test dataset contains historical data from New York's yellow taxi network.

       To import up to 100GB of data directly from your current Postgres-based database,
       [migrate with downtime][migrate-with-downtime] using native Postgres tooling. To seamlessly import 100GB-10TB+
       of data, use the [live migration][migrate-live] tooling supplied by Tiger Data. To add data from non-Postgres
       data sources, see [Import and ingest data][data-ingest].

    1. In Terminal, navigate to `<local folder>` and update the following string with [your connection details][connection-info]
      to connect to your service.

       ```bash
       psql -d "postgres://<username>:<password>@<host>:<port>/<database-name>?sslmode=require"
       ```

    1. Create an optimized hypertable for your time-series data:

          1. Create a [hypertable][hypertables-section] with [hypercore][hypercore] enabled by default for your
             time-series data using [CREATE TABLE][hypertable-create-table]. For [efficient queries][secondary-indexes]
             on data in the columnstore, remember to `segmentby` the column you will use most often to filter your data.

             In your sql client, run the following command:

             ```sql
             CREATE TABLE "rides"(
               vendor_id TEXT,
               pickup_datetime TIMESTAMP WITHOUT TIME ZONE NOT NULL,
               dropoff_datetime TIMESTAMP WITHOUT TIME ZONE NOT NULL,
               passenger_count NUMERIC,
               trip_distance NUMERIC,
               pickup_longitude  NUMERIC,
               pickup_latitude   NUMERIC,
               rate_code         INTEGER,
               dropoff_longitude NUMERIC,
               dropoff_latitude  NUMERIC,
               payment_type INTEGER,
               fare_amount NUMERIC,
               extra NUMERIC,
               mta_tax NUMERIC,
               tip_amount NUMERIC,
               tolls_amount NUMERIC,
               improvement_surcharge NUMERIC,
               total_amount NUMERIC
             ) WITH (
               tsdb.hypertable,
               tsdb.partition_column='pickup_datetime',
               tsdb.create_default_indexes=false,
               tsdb.segmentby='vendor_id',
               tsdb.orderby='pickup_datetime DESC'
             );
             ```
             If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

         1.  Add another dimension to partition your hypertable more efficiently:
             ```sql
             SELECT add_dimension('rides', by_hash('payment_type', 2));
             ```

         1.  Create an index to support efficient queries by vendor, rate code, and passenger count:
             ```sql
             CREATE INDEX ON rides (vendor_id, pickup_datetime DESC);
             CREATE INDEX ON rides (rate_code, pickup_datetime DESC);
             CREATE INDEX ON rides (passenger_count, pickup_datetime DESC);
             ```

    1. Create Postgres tables for relational data:

         1.  Add a table to store the payment types data:

             ```sql
             CREATE TABLE IF NOT EXISTS "payment_types"(
               payment_type INTEGER,
               description TEXT
             );
             INSERT INTO payment_types(payment_type, description) VALUES
               (1, 'credit card'),
               (2, 'cash'),
               (3, 'no charge'),
               (4, 'dispute'),
               (5, 'unknown'),
               (6, 'voided trip');
             ```

         1.  Add a table to store the rates data:

             ```sql
             CREATE TABLE IF NOT EXISTS "rates"(
              rate_code   INTEGER,
              description TEXT
             );
             INSERT INTO rates(rate_code, description) VALUES
              (1, 'standard rate'),
              (2, 'JFK'),
              (3, 'Newark'),
              (4, 'Nassau or Westchester'),
              (5, 'negotiated fare'),
              (6, 'group ride');
             ```

      1. Upload the dataset to your service
         ```sql
         \COPY rides FROM nyc_data_rides.csv CSV;
         ```

1.  **Have a quick look at your data**

    You query hypertables in exactly the same way as you would a relational Postgres table.
    Use one of the following SQL editors to run a query and see the data you uploaded:
       - **Data mode**:  write queries, visualize data, and share your results in [Tiger Cloud Console][portal-data-mode] for all your Tiger Cloud services.
       - **SQL editor**: write, fix, and organize SQL faster and more accurately in [Tiger Cloud Console][portal-ops-mode] for a Tiger Cloud service.
       - **psql**: easily run queries on your Tiger Cloud services or self-hosted TimescaleDB deployment from Terminal.

    For example:
    - Display the number of rides for each fare type:
       ```sql
       SELECT rate_code, COUNT(vendor_id) AS num_trips
       FROM rides
       WHERE pickup_datetime < '2016-01-08'
       GROUP BY rate_code
       ORDER BY rate_code;
       ```
       This simple query runs in 3 seconds. You see something like:

       | rate_code | num_trips	|
       |-----------------|-----------|
       |1 |   2266401|
       |2 |     54832|
       |3 |      4126|
       |4 |       967|
       |5 |      7193|
       |6 |        17|
       |99 |        42|

    - To select all rides taken in the first week of January 2016, and return the total number of trips taken for each rate code:
       ```sql
       SELECT rates.description, COUNT(vendor_id) AS num_trips
       FROM rides
       JOIN rates ON rides.rate_code = rates.rate_code
       WHERE pickup_datetime < '2016-01-08'
       GROUP BY rates.description
       ORDER BY LOWER(rates.description);
       ```
       On this large amount of data, this analytical query on data in the rowstore takes about 59 seconds. You see something like:

       | description	| num_trips	|
       |-----------------|-----------|
       | group ride | 	17 |
       | JFK	 | 54832 |
       | Nassau or Westchester | 	967 |
       | negotiated fare | 	7193 |
       | Newark | 	4126 |
       | standard rate | 	2266401 |

## Optimize your data for real-time analytics


When TimescaleDB converts a chunk to the columnstore, it automatically creates a different schema for your
data. TimescaleDB creates and uses custom indexes to incorporate the `segmentby` and `orderby` parameters when
you write to and read from the columstore.

To increase the speed of your analytical queries by a factor of 10 and reduce storage costs by up to 90%, convert data
to the columnstore:

1. **Connect to your Tiger Cloud service**

   In [Tiger Cloud Console][services-portal] open an [SQL editor][in-console-editors]. The in-Console editors display the query speed.
   You can also connect to your serviceusing [psql][connect-using-psql].

1. **Add a policy to convert chunks to the columnstore at a specific time interval**

   For example, convert data older than 8 days old to the columstore:
   ``` sql
   CALL add_columnstore_policy('rides', INTERVAL '8 days');
   ```
   See [add_columnstore_policy][add_columnstore_policy].

   The data you imported for this tutorial is from 2016, it was already added to the columnstore by default. However,
   you get the idea. To see the space savings in action, follow [Try the key Tiger Data features][try-timescale-features].

Just to hit this one home, by converting cooling data to the columnstore, you have increased the speed of your analytical
queries by a factor of 10, and reduced storage by up to 90%.


## Connect Grafana to Tiger Cloud

To visualize the results of your queries, enable Grafana to read the data in your service:

1. **Log in to Grafana**

   In your browser, log in to either:
    - Self-hosted Grafana: at `http://localhost:3000/`. The default credentials are `admin`, `admin`.
    - Grafana Cloud: use the URL and credentials you set when you created your account.
1. **Add your service as a data source**
   1. Open `Connections` > `Data sources`, then click `Add new data source`.
   1. Select `PostgreSQL` from the list.
   1. Configure the connection:
      - `Host URL`, `Database name`, `Username`, and `Password`

          Configure using your [connection details][connection-info]. `Host URL` is in the format `<host>:<port>`.
      - `TLS/SSL Mode`: select `require`.
      - `PostgreSQL options`: enable `TimescaleDB`.
      - Leave the default setting for all other fields.

   1. Click `Save & test`.

     Grafana checks that your details are set correctly.

## Monitor performance over time

A Grafana dashboard represents a view into the performance of a system, and each dashboard consists of one or
more panels, which represent information about a specific metric related to that system.

To visually monitor the volume of taxi rides over time:

1. **Create the dashboard**

   1. On the `Dashboards` page, click `New` and select `New dashboard`.

   1. Click `Add visualization`.
   1. Select the data source that connects to your Tiger Cloud service.
       The `Time series` visualization is chosen by default.
      ![Grafana create dashboard](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-configure-dashboard.png)
   1. In the `Queries` section, select `Code`, then select `Time series` in `Format`.
   1. Select the data range for your visualization:
      the data set is from 2016. Click the date range above the panel and set:
      - From: ```2016-01-01 01:00:00```
      - To: ```2016-01-30 01:00:00```

1. **Combine TimescaleDB and Grafana functionality to analyze your data**

   Combine a TimescaleDB [time_bucket][use-time-buckets], with the Grafana `_timefilter()` function to set the
   `pickup_datetime` column as the filtering range for your visualizations.
   ```sql
   SELECT
     time_bucket('1 day', pickup_datetime) AS "time",
     COUNT(*)
   FROM rides
   WHERE _timeFilter(pickup_datetime)
   GROUP BY time
   ORDER BY time;
   ```
   This query groups the results by day and orders them by time.

   ![Grafana real-time analytics](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-final-dashboard.png)

1. **Click `Save dashboard`**

## Optimize revenue potential

Having all this data is great but how do you use it? Monitoring data is useful to check what
has happened, but how can you analyse this information to your advantage? This section explains
how to create a visualization that shows how you can maximize potential revenue.

### Set up your data for geospatial queries

To add geospatial analysis to your ride count visualization, you need geospatial data to work out which trips
originated where. As TimescaleDB is compatible with all Postgres extensions, use [PostGIS][postgis] to slice
data by time and location.

1.  Connect to your [Tiger Cloud service][in-console-editors] and add the PostGIS extension:

    ```sql
    CREATE EXTENSION postgis;
    ```

1. Add geometry columns for pick up and drop off locations:

    ```sql
    ALTER TABLE rides ADD COLUMN pickup_geom geometry(POINT,2163);
    ALTER TABLE rides ADD COLUMN dropoff_geom geometry(POINT,2163);
    ```

1.  Convert the latitude and longitude points into geometry coordinates that work with PostGIS:

    ```sql
    UPDATE rides SET pickup_geom = ST_Transform(ST_SetSRID(ST_MakePoint(pickup_longitude,pickup_latitude),4326),2163),
       dropoff_geom = ST_Transform(ST_SetSRID(ST_MakePoint(dropoff_longitude,dropoff_latitude),4326),2163);
    ```
    This updates 10,906,860 rows of data on both columns, it takes a while. Coffee is your friend.

### Visualize the area where you can make the most money

In this section you visualize a query that returns rides longer than 5 miles for
trips taken within 2 km of Times Square. The data includes the distance travelled and
is `GROUP BY` `trip_distance` and location so that Grafana can plot the data properly.

This enables you to see where a taxi driver is most likely to pick up a passenger who wants a longer ride,
and make more money.

1. **Create a geolocalization dashboard**

   1. In Grafana, create a new dashboard that is connected to your Tiger Cloud service data source with a Geomap
      visualization.

   1. In the `Queries` section, select `Code`, then select the Time series `Format`.

      ![Real-time analytics geolocation](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-configure-dashboard.png)

   1. To find rides longer than 5 miles in Manhattan, paste the following query:

       ```sql
       SELECT time_bucket('5m', rides.pickup_datetime) AS time,
              rides.trip_distance AS value,
              rides.pickup_latitude AS latitude,
              rides.pickup_longitude AS longitude
       FROM rides
       WHERE rides.pickup_datetime BETWEEN '2016-01-01T01:41:55.986Z' AND '2016-01-01T07:41:55.986Z' AND
         ST_Distance(pickup_geom,
                     ST_Transform(ST_SetSRID(ST_MakePoint(-73.9851,40.7589),4326),2163)
         ) < 2000
       GROUP BY time,
                rides.trip_distance,
                rides.pickup_latitude,
                rides.pickup_longitude
       ORDER BY time
       LIMIT 500;
       ```
       You see a world map with a dot on New York.
   1. Zoom into your map to see the visualization clearly.

1. **Customize the visualization**

   1. In the Geomap options, under `Map Layers`, click `+ Add layer` and select `Heatmap`.
     You now see the areas where a taxi driver is most likely to pick up a passenger who wants a
     longer ride, and make more money.

      ![Real-time analytics geolocation](https://assets.timescale.com/docs/images/use-case-rta-grafana-heatmap.png)

You have integrated Grafana with a Tiger Cloud service and made insights based on visualization of
your data.


===== PAGE: https://docs.tigerdata.com/tutorials/real-time-analytics-energy-consumption/ =====

# Real-time analytics with Tiger Cloud and Grafana



Energy providers understand that customers tend to lose patience when there is not enough power for them
to complete day-to-day activities. Task one is keeping the lights on. If you are transitioning to renewable energy,
it helps to know when you need to produce energy so you can choose a suitable energy source.

Real-time analytics refers to the process of collecting, analyzing, and interpreting data instantly as it is generated.
This approach enables you to track and monitor activity, make the decisions based on real-time insights on data stored in
a Tiger Cloud service and keep those lights on.


[Grafana][grafana-docs] is a popular data visualization tool that enables you to create customizable dashboards
and effectively monitor your systems and applications.

![Grafana real-time analytics](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-energy-cagg.png)

This page shows you how to integrate Grafana with a Tiger Cloud service and make insights based on visualization of
data optimized for size and speed in the columnstore.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

* Install and run [self-managed Grafana][grafana-self-managed], or sign up for [Grafana Cloud][grafana-cloud].

## Optimize time-series data in hypertables

Hypertables are Postgres tables in TimescaleDB that automatically partition your time-series data by time. Time-series data represents the way a system, process, or behavior changes over time. Hypertables enable TimescaleDB to work efficiently with time-series data.  Each hypertable is made up of child tables called chunks. Each chunk is assigned a range
of time, and only contains data from that range. When you run a query, TimescaleDB identifies the correct chunk and
runs the query on it, instead of going through the entire table.

[Hypercore][hypercore] is the hybrid row-columnar storage engine in TimescaleDB used by hypertables. Traditional
databases force a trade-off between fast inserts (row-based storage) and efficient analytics
(columnar storage). Hypercore eliminates this trade-off, allowing real-time analytics without sacrificing
transactional capabilities.

Hypercore dynamically stores data in the most efficient format for its lifecycle:

* **Row-based storage for recent data**: the most recent chunk (and possibly more) is always stored in the rowstore,
   ensuring fast inserts, updates, and low-latency single record queries. Additionally, row-based storage is used as a
   writethrough for inserts and updates to columnar storage.
* **Columnar storage for analytical performance**: chunks are automatically compressed into the columnstore, optimizing
   storage efficiency and accelerating analytical queries.

Unlike traditional columnar databases, hypercore allows data to be inserted or modified at any stage, making it a
flexible solution for both high-ingest transactional workloads and real-time analytics—within a single database.

Because TimescaleDB is 100% Postgres, you can use all the standard Postgres tables, indexes, stored
procedures, and other objects alongside your hypertables. This makes creating and working with hypertables similar
to standard Postgres.

1.  **Import time-series data into a hypertable**

    1. Unzip [metrics.csv.gz](https://assets.timescale.com/docs/downloads/metrics.csv.gz) to a `<local folder>`.

       This test dataset contains energy consumption data.

       To import up to 100GB of data directly from your current Postgres based database,
       [migrate with downtime][migrate-with-downtime] using native Postgres tooling. To seamlessly import 100GB-10TB+
       of data, use the [live migration][migrate-live] tooling supplied by Tiger Data. To add data from non-Postgres
       data sources, see [Import and ingest data][data-ingest].

    1. In Terminal, navigate to `<local folder>` and update the following string with [your connection details][connection-info]
      to connect to your service.

       ```bash
       psql -d "postgres://<username>:<password>@<host>:<port>/<database-name>?sslmode=require"
       ```

    1. Create an optimized hypertable for your time-series data:

       1. Create a [hypertable][hypertables-section] with [hypercore][hypercore] enabled by default for your
          time-series data using [CREATE TABLE][hypertable-create-table]. For [efficient queries][secondary-indexes]
          on data in the columnstore, remember to `segmentby` the column you will use most often to filter your data.

          In your sql client, run the following command:

             ```sql
             CREATE TABLE "metrics"(
               created timestamp with time zone default now() not null,
               type_id integer                                not null,
               value   double precision                       not null
             ) WITH (
               tsdb.hypertable,
               tsdb.partition_column='created',
               tsdb.segmentby = 'type_id',
               tsdb.orderby = 'created DESC'
             );
             ```
             If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

      1. Upload the dataset to your service
         ```sql
         \COPY metrics FROM metrics.csv CSV;
         ```

1.  **Have a quick look at your data**

    You query hypertables in exactly the same way as you would a relational Postgres table.
    Use one of the following SQL editors to run a query and see the data you uploaded:
       - **Data mode**:  write queries, visualize data, and share your results in [Tiger Cloud Console][portal-data-mode] for all your Tiger Cloud services.
       - **SQL editor**: write, fix, and organize SQL faster and more accurately in [Tiger Cloud Console][portal-ops-mode] for a Tiger Cloud service.
       - **psql**: easily run queries on your Tiger Cloud services or self-hosted TimescaleDB deployment from Terminal.

    ```sql
    SELECT time_bucket('1 day', created, 'Europe/Berlin') AS "time",
    round((last(value, created) - first(value, created)) * 100.) / 100. AS value
    FROM metrics
    WHERE type_id = 5
    GROUP BY 1;
    ```

    On this amount of data, this query on data in the rowstore takes about 3.6 seconds. You see something like:

    | Time	                        | value |
    |------------------------------|-------|
    | 2023-05-29 22:00:00+00 | 23.1  |
    | 2023-05-28 22:00:00+00 | 19.5  |
    | 2023-05-30 22:00:00+00 | 25    |
    | 2023-05-31 22:00:00+00 | 8.1   |

## Optimize your data for real-time analytics

When TimescaleDB converts a chunk to the columnstore, it automatically creates a different schema for your
data. TimescaleDB creates and uses custom indexes to incorporate the `segmentby` and `orderby` parameters when
you write to and read from the columstore.

To increase the speed of your analytical queries by a factor of 10 and reduce storage costs by up to 90%, convert data
to the columnstore:

1. **Connect to your Tiger Cloud service**

   In [Tiger Cloud Console][services-portal] open an [SQL editor][in-console-editors]. The in-Console editors display the query speed.
   You can also connect to your service using [psql][connect-using-psql].

1. **Add a policy to convert chunks to the columnstore at a specific time interval**

   For example, 60 days after the data was added to the table:
   ``` sql
   CALL add_columnstore_policy('metrics', INTERVAL '8 days');
   ```
   See [add_columnstore_policy][add_columnstore_policy].

1. **Faster analytical queries on data in the columnstore**

   Now run the analytical query again:
   ```sql
    SELECT time_bucket('1 day', created, 'Europe/Berlin') AS "time",
    round((last(value, created) - first(value, created)) * 100.) / 100. AS value
    FROM metrics
    WHERE type_id = 5
    GROUP BY 1;
   ```
   On this amount of data, this analytical query on data in the columnstore takes about 250ms.

Just to hit this one home, by converting cooling data to the columnstore, you have increased the speed of your analytical
queries by a factor of 10, and reduced storage by up to 90%.

## Write fast analytical queries

Aggregation is a way of combining data to get insights from it. Average, sum, and count are all examples of simple
aggregates. However, with large amounts of data aggregation slows things down, quickly. Continuous aggregates
are a kind of hypertable that is refreshed automatically in the background as new data is added, or old data is
modified. Changes to your dataset are tracked, and the hypertable behind the continuous aggregate is automatically
updated in the background.

By default, querying continuous aggregates provides you with real-time data. Pre-aggregated data from the materialized
view is combined with recent data that hasn't been aggregated yet. This gives you up-to-date results on every query.

You create continuous aggregates on uncompressed data in high-performance storage. They continue to work
on [data in the columnstore][test-drive-enable-compression]
and [rarely accessed data in tiered storage][test-drive-tiered-storage]. You can even
create [continuous aggregates on top of your continuous aggregates][hierarchical-caggs].

1.  **Monitor energy consumption on a day-to-day basis**

    1.  Create a continuous aggregate `kwh_day_by_day` for energy consumption:

        ```sql
        CREATE MATERIALIZED VIEW kwh_day_by_day(time, value)
           with (timescaledb.continuous) as
        SELECT time_bucket('1 day', created, 'Europe/Berlin') AS "time",
               round((last(value, created) - first(value, created)) * 100.) / 100. AS value
        FROM metrics
        WHERE type_id = 5
        GROUP BY 1;
        ```

    1.  Add a refresh policy to keep `kwh_day_by_day` up-to-date:

        ```sql
        SELECT add_continuous_aggregate_policy('kwh_day_by_day',
           start_offset => NULL,
           end_offset => INTERVAL '1 hour',
           schedule_interval => INTERVAL '1 hour');
        ```

1.  **Monitor energy consumption on an hourly basis**

    1. Create a continuous aggregate `kwh_hour_by_hour` for energy consumption:

       ```sql
       CREATE MATERIALIZED VIEW kwh_hour_by_hour(time, value)
         with (timescaledb.continuous) as
       SELECT time_bucket('01:00:00', metrics.created, 'Europe/Berlin') AS "time",
              round((last(value, created) - first(value, created)) * 100.) / 100. AS value
       FROM metrics
       WHERE type_id = 5
       GROUP BY 1;
       ```

    1.  Add a refresh policy to keep the continuous aggregate up-to-date:

       ```sql
       SELECT add_continuous_aggregate_policy('kwh_hour_by_hour',
        start_offset => NULL,
           end_offset => INTERVAL '1 hour',
           schedule_interval => INTERVAL '1 hour');
       ```

1.  **Analyze your data**

    Now you have made continuous aggregates, it could be a good idea to use them to perform analytics on your data.
    For example, to see how average energy consumption changes during weekdays over the last year, run the following query:
    ```sql
      WITH per_day AS (
       SELECT
         time,
         value
       FROM kwh_day_by_day
       WHERE "time" at time zone 'Europe/Berlin' > date_trunc('month', time) - interval '1 year'
       ORDER BY 1
      ), daily AS (
          SELECT
             to_char(time, 'Dy') as day,
             value
          FROM per_day
      ), percentile AS (
          SELECT
              day,
              approx_percentile(0.50, percentile_agg(value)) as value
          FROM daily
          GROUP BY 1
          ORDER BY 1
      )
      SELECT
          d.day,
          d.ordinal,
          pd.value
      FROM unnest(array['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat']) WITH ORDINALITY AS d(day, ordinal)
      LEFT JOIN percentile pd ON lower(pd.day) = lower(d.day);
    ```

    You see something like:

      | day | ordinal | value |
      | --- | ------- | ----- |
      | Mon | 2 | 23.08078714975423 |
      | Sun | 1 | 19.511430831944395 |
      | Tue | 3 | 25.003118897837307 |
      | Wed | 4 | 8.09300571759772 |

## Connect Grafana to Tiger Cloud

To visualize the results of your queries, enable Grafana to read the data in your service:

1. **Log in to Grafana**

   In your browser, log in to either:
    - Self-hosted Grafana: at `http://localhost:3000/`. The default credentials are `admin`, `admin`.
    - Grafana Cloud: use the URL and credentials you set when you created your account.
1. **Add your service as a data source**
   1. Open `Connections` > `Data sources`, then click `Add new data source`.
   1. Select `PostgreSQL` from the list.
   1. Configure the connection:
      - `Host URL`, `Database name`, `Username`, and `Password`

          Configure using your [connection details][connection-info]. `Host URL` is in the format `<host>:<port>`.
      - `TLS/SSL Mode`: select `require`.
      - `PostgreSQL options`: enable `TimescaleDB`.
      - Leave the default setting for all other fields.

   1. Click `Save & test`.

     Grafana checks that your details are set correctly.

## Visualize energy consumption

A Grafana dashboard represents a view into the performance of a system, and each dashboard consists of one or
more panels, which represent information about a specific metric related to that system.

To visually monitor the volume of energy consumption over time:

1. **Create the dashboard**

   1. On the `Dashboards` page, click `New` and select `New dashboard`.

   1. Click `Add visualization`, then select the data source that connects to your Tiger Cloud service and the `Bar chart`
      visualization.

      ![Grafana create dashboard](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-configure-dashboard.png)
   1. In the `Queries` section, select `Code`, then run the following query based on your continuous aggregate:

       ```sql
       WITH per_hour AS (
       SELECT
       time,
       value
       FROM kwh_hour_by_hour
       WHERE "time" at time zone 'Europe/Berlin' > date_trunc('month', time) - interval '1 year'
       ORDER BY 1
       ), hourly AS (
        SELECT
             extract(HOUR FROM time) * interval '1 hour' as hour,
             value
        FROM per_hour
       )
       SELECT
           hour,
           approx_percentile(0.50, percentile_agg(value)) as median,
           max(value) as maximum
       FROM hourly
       GROUP BY 1
       ORDER BY 1;
       ```

      This query averages the results for households in a specific time zone by hour and orders them by time.
      Because you use a continuous aggregate, this data is always correct in real time.

      ![Grafana real-time analytics](https://assets.timescale.com/docs/images/use-case-rta-grafana-timescale-energy-cagg.png)

      You see that energy consumption is highest in the evening and at breakfast time. You also know that the wind
      drops off in the evening. This data proves that you need to supply a supplementary power source for peak times,
      or plan to store energy during the day for peak times.

1. **Click `Save dashboard`**

You have integrated Grafana with a Tiger Cloud service and made insights based on visualization of your data.


===== PAGE: https://docs.tigerdata.com/tutorials/simulate-iot-sensor-data/ =====

# Simulate an IoT sensor dataset



The Internet of Things (IoT) describes a trend where computing capabilities are embedded into IoT devices. That is, physical objects, ranging from light bulbs to oil wells. Many IoT devices collect sensor data about their environment and generate time-series datasets with relational metadata.

It is often necessary to simulate IoT datasets. For example, when you are
testing a new system. This tutorial shows how to simulate a basic dataset in your Tiger Cloud service, and then run simple queries on it.

To simulate a more advanced dataset, see [Time-series Benchmarking Suite (TSBS)][tsbs].

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

## Simulate a dataset

To simulate a dataset, run the following queries:

1. **Create the `sensors` table**:

    ```sql
    CREATE TABLE sensors(
      id SERIAL PRIMARY KEY,
      type VARCHAR(50),
      location VARCHAR(50)
    );
    ```

1. **Create the `sensor_data` hypertable**

    ```sql
    CREATE TABLE sensor_data (
      time TIMESTAMPTZ NOT NULL,
      sensor_id INTEGER,
      temperature DOUBLE PRECISION,
      cpu DOUBLE PRECISION,
      FOREIGN KEY (sensor_id) REFERENCES sensors (id)
    ) WITH (
      tsdb.hypertable,
      tsdb.partition_column='time'
    );
    ```
    If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

1. **Populate the `sensors` table**:

    ```sql
    INSERT INTO sensors (type, location) VALUES
    ('a','floor'),
    ('a', 'ceiling'),
    ('b','floor'),
    ('b', 'ceiling');
    ```

1. **Verify that the sensors have been added correctly**:

    ```sql
    SELECT * FROM sensors;
    ```

    Sample output:

    ```
     id | type | location
    ----+------+----------
      1 | a    | floor
      2 | a    | ceiling
      3 | b    | floor
      4 | b    | ceiling
    (4 rows)
    ```

1. **Generate and insert a dataset for all sensors:**

    ```sql
    INSERT INTO sensor_data (time, sensor_id, cpu, temperature)
    SELECT
      time,
      sensor_id,
      random() AS cpu,
      random()*100 AS temperature
    FROM generate_series(now() - interval '24 hour', now(), interval '5 minute') AS g1(time), generate_series(1,4,1) AS g2(sensor_id);
    ```

1. **Verify the simulated dataset**:

    ```sql
    SELECT * FROM sensor_data ORDER BY time;
    ```

    Sample output:

    ```
                 time              | sensor_id |    temperature     |         cpu
    -------------------------------+-----------+--------------------+---------------------
     2020-03-31 15:56:25.843575+00 |         1 |   6.86688972637057 |   0.682070567272604
     2020-03-31 15:56:40.244287+00 |         2 |    26.589260622859 |   0.229583469685167
     2030-03-31 15:56:45.653115+00 |         3 |   79.9925176426768 |   0.457779890391976
     2020-03-31 15:56:53.560205+00 |         4 |   24.3201029952615 |   0.641885648947209
     2020-03-31 16:01:25.843575+00 |         1 |   33.3203678019345 |  0.0159163917414844
     2020-03-31 16:01:40.244287+00 |         2 |   31.2673618085682 |   0.701185956597328
     2020-03-31 16:01:45.653115+00 |         3 |   85.2960689924657 |   0.693413889966905
     2020-03-31 16:01:53.560205+00 |         4 |   79.4769988860935 |   0.360561791341752
    ...
    ```

## Run basic queries

After you simulate a dataset, you can run some basic queries on it. For example:

- Average temperature and CPU by 30-minute windows:

   ```sql
   SELECT
     time_bucket('30 minutes', time) AS period,
     AVG(temperature) AS avg_temp,
     AVG(cpu) AS avg_cpu
   FROM sensor_data
   GROUP BY period;
   ```

   Sample output:

   ```
            period         |     avg_temp     |      avg_cpu
   ------------------------+------------------+-------------------
    2020-03-31 19:00:00+00 | 49.6615830013373 | 0.477344429974134
    2020-03-31 22:00:00+00 | 58.8521540844037 | 0.503637770501276
    2020-03-31 16:00:00+00 | 50.4250325243144 | 0.511075591299838
    2020-03-31 17:30:00+00 | 49.0742547437549 | 0.527267253802468
    2020-04-01 14:30:00+00 | 49.3416377226822 | 0.438027751864865
    ...
   ```

- Average and last temperature, average CPU by 30-minute windows:

   ```sql
   SELECT
     time_bucket('30 minutes', time) AS period,
     AVG(temperature) AS avg_temp,
     last(temperature, time) AS last_temp,
     AVG(cpu) AS avg_cpu
   FROM sensor_data
   GROUP BY period;
   ```

   Sample output:

   ```
            period         |     avg_temp     |    last_temp     |      avg_cpu
   ------------------------+------------------+------------------+-------------------
    2020-03-31 19:00:00+00 | 49.6615830013373 | 84.3963081017137 | 0.477344429974134
    2020-03-31 22:00:00+00 | 58.8521540844037 | 76.5528806950897 | 0.503637770501276
    2020-03-31 16:00:00+00 | 50.4250325243144 | 43.5192013625056 | 0.511075591299838
    2020-03-31 17:30:00+00 | 49.0742547437549 |  22.740753274411 | 0.527267253802468
    2020-04-01 14:30:00+00 | 49.3416377226822 | 59.1331578791142 | 0.438027751864865
   ...
   ```

- Query the metadata:

   ```sql
   SELECT
     sensors.location,
     time_bucket('30 minutes', time) AS period,
     AVG(temperature) AS avg_temp,
     last(temperature, time) AS last_temp,
     AVG(cpu) AS avg_cpu
   FROM sensor_data JOIN sensors on sensor_data.sensor_id = sensors.id
   GROUP BY period, sensors.location;
   ```

   Sample output:

   ```
    location |         period         |     avg_temp     |     last_temp     |      avg_cpu
   ----------+------------------------+------------------+-------------------+-------------------
    ceiling  | 20120-03-31 15:30:00+00 | 25.4546818090603 |  24.3201029952615 | 0.435734559316188
    floor    | 2020-03-31 15:30:00+00 | 43.4297036845237 |  79.9925176426768 |  0.56992522883229
    ceiling  | 2020-03-31 16:00:00+00 | 53.8454438598516 |  43.5192013625056 | 0.490728285357666
    floor    | 2020-03-31 16:00:00+00 | 47.0046211887772 |  23.0230117216706 |  0.53142289724201
    ceiling  | 2020-03-31 16:30:00+00 | 58.7817596504465 |  63.6621567420661 | 0.488188337767497
    floor    | 2020-03-31 16:30:00+00 |  44.611586847653 |  2.21919436007738 | 0.434762630766879
    ceiling  | 2020-03-31 17:00:00+00 | 35.7026890735142 |  42.9420990403742 | 0.550129583687522
    floor    | 2020-03-31 17:00:00+00 | 62.2794370166957 |  52.6636955793947 | 0.454323202022351
   ...
   ```

You have now successfully simulated and run queries on an IoT dataset.


===== PAGE: https://docs.tigerdata.com/tutorials/cookbook/ =====

# Tiger Data cookbook





This page contains suggestions from the [Tiger Data Community](https://timescaledb.slack.com/) about how to resolve
common issues. Use these code examples as guidance to work with your own data.


## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with the Real-time analytics capability.

   You need [your connection details][connection-info]. This procedure also
   works for [self-hosted TimescaleDB][enable-timescaledb].

## Hypertable recipes

This section contains recipes about hypertables.

### Remove duplicates from an existing hypertable

Looking to remove duplicates from an existing hypertable? One method is to run a `PARTITION BY` query to get
`ROW_NUMBER()` and then the `ctid` of rows where `row_number>1`. You then delete these rows.  However,
you need to check `tableoid` and `ctid`. This is because `ctid` is not unique and might be duplicated in
different chunks. The following code example took 17 hours to process a table with 40 million rows:

```sql
CREATE OR REPLACE FUNCTION deduplicate_chunks(ht_name TEXT, partition_columns TEXT, bot_id INT DEFAULT NULL)
    RETURNS TABLE
            (
                chunk_schema  name,
                chunk_name    name,
                deleted_count INT
            )
AS
$$
DECLARE
    chunk         RECORD;
    where_clause  TEXT := '';
    deleted_count INT;
BEGIN
    IF bot_id IS NOT NULL THEN
        where_clause := FORMAT('WHERE bot_id = %s', bot_id);
    END IF;

    FOR chunk IN
        SELECT c.chunk_schema, c.chunk_name
        FROM timescaledb_information.chunks c
        WHERE c.hypertable_name = ht_name
        LOOP
            EXECUTE FORMAT('
            WITH cte AS (
                SELECT ctid,
                       ROW_NUMBER() OVER (PARTITION BY %s ORDER BY %s ASC) AS row_num,
                       *
                FROM %I.%I
                %s
            )
            DELETE FROM %I.%I
            WHERE ctid IN (
                SELECT ctid
                FROM cte
                WHERE row_num > 1
            )
            RETURNING 1;
        ', partition_columns, partition_columns, chunk.chunk_schema, chunk.chunk_name, where_clause, chunk.chunk_schema,
                           chunk.chunk_name)
                INTO deleted_count;

            RETURN QUERY SELECT chunk.chunk_schema, chunk.chunk_name, COALESCE(deleted_count, 0);
        END LOOP;
END
$$ LANGUAGE plpgsql;


SELECT *
FROM deduplicate_chunks('nudge_events', 'bot_id, session_id, nudge_id, time', 2540);
```

Shoutout to **Mathias Ose** and **Christopher Piggott** for this recipe.

### Get faster JOIN queries with Common Table Expressions

Imagine there is a query that joins a hypertable to another table on a shared key:

```sql
    SELECT timestamp,
      FROM hypertable as h
      JOIN related_table as rt
        ON rt.id = h.related_table_id
     WHERE h.timestamp BETWEEN '2024-10-10 00:00:00' AND '2024-10-17 00:00:00'
```

If you run `EXPLAIN` on this query, you see that the query planner performs a `NestedJoin` between these two tables, which means querying the hypertable multiple times.  Even if the hypertable is well indexed, if it is also large, the query will be slow. How do you force a once-only lookup? Use materialized Common Table Expressions (CTEs).

If you split the query into two parts using CTEs, you can `materialize` the hypertable lookup and force Postgres to perform it only once.

```sql
WITH cached_query AS materialized (
  SELECT *
    FROM hypertable
   WHERE BETWEEN '2024-10-10 00:00:00' AND '2024-10-17 00:00:00'
)
  SELECT *
    FROM cached_query as c
    JOIN related_table as rt
      ON rt.id = h.related_table_id
```

Now if you run `EXPLAIN` once again, you see that this query performs only one lookup. Depending on the size of your hypertable, this could result in a multi-hour query taking mere seconds.

Shoutout to **Rowan Molony** for this recipe.

## IoT recipes

This section contains recipes for IoT issues:

### Work with columnar IoT data

Narrow and medium width tables are a great way to store IoT data. A lot of reasons are outlined in
[Designing Your Database Schema: Wide vs. Narrow Postgres Tables][blog-wide-vs-narrow].

One of the key advantages of narrow tables is that the schema does not have to change when you add new
sensors. Another big advantage is that each sensor can sample at different rates and times. This helps
support things like hysteresis, where new values are written infrequently unless the value changes by a
certain amount.

#### Narrow table format example

Working with narrow table data structures presents a few challenges. In the IoT world one concern is that
many data analysis approaches - including machine learning as well as more traditional data analysis -
require that your data is resampled and synchronized to a common time basis. Fortunately, TimescaleDB provides
you with [hyperfunctions][hyperfunctions] and other tools to help you work with this data.

An example of a narrow table format is:

| ts                      | sensor_id | value |
|-------------------------|-----------|-------|
| 2024-10-31 11:17:30.000 | 1007      | 23.45 |

Typically you would couple this with a sensor table:

| sensor_id | sensor_name  | units                    |
|-----------|--------------|--------------------------|
| 1007      | temperature  | degreesC                 |
| 1012      | heat_mode    | on/off                   |
| 1013      | cooling_mode | on/off                   |
| 1041      | occupancy    | number of people in room |

A medium table retains the generic structure but adds columns of various types so that you can
use the same table to store float, int, bool, or even JSON (jsonb) data:

| ts                      | sensor_id | d     | i    | b    | t    | j    |
|-------------------------|-----------|-------|------|------|------|------|
| 2024-10-31 11:17:30.000 | 1007      | 23.45 | null | null | null | null |
| 2024-10-31 11:17:47.000 | 1012      | null  | null | TRUE | null | null |
| 2024-10-31 11:18:01.000 | 1041      | null  | 4    | null | null | null |

To remove all-null entries, use an optional constraint such as:

```sql
    CONSTRAINT at_least_one_not_null
        CHECK ((d IS NOT NULL) OR (i IS NOT NULL) OR (b IS NOT NULL) OR (j IS NOT NULL) OR (t IS NOT NULL))
```

#### Get the last value of every sensor

There are several ways to get the latest value of every sensor. The following examples use the
structure defined in [Narrow table format example][setup-a-narrow-table-format] as a reference:

- [SELECT DISTINCT ON][select-distinct-on]
- [JOIN LATERAL][join-lateral]

##### SELECT DISTINCT ON

If you have a list of sensors, the easy way to get the latest value of every sensor is to use
`SELECT DISTINCT ON`:

```sql
WITH latest_data AS (
    SELECT DISTINCT ON (sensor_id) ts, sensor_id, d
    FROM iot_data
    WHERE d is not null
      AND ts > CURRENT_TIMESTAMP - INTERVAL '1 week'  -- important
    ORDER BY sensor_id, ts DESC
)
SELECT
    sensor_id, sensors.name, ts, d
FROM latest_data
LEFT OUTER JOIN sensors ON latest_data.sensor_id = sensors.id
WHERE latest_data.d is not null
ORDER BY sensor_id, ts; -- Optional, for displaying results ordered by sensor_id
```

The common table expression (CTE) used above is not strictly necessary. However, it is an elegant way to join
to the sensor list to get a sensor name in the output.  If this is not something you care about,
you can leave it out:

```sql
SELECT DISTINCT ON (sensor_id) ts, sensor_id, d
    FROM iot_data
    WHERE d is not null
      AND ts > CURRENT_TIMESTAMP - INTERVAL '1 week'  -- important
    ORDER BY sensor_id, ts DESC
```

It is important to take care when down-selecting this data. In the previous examples,
the time that the query would scan back was limited. However, if there any sensors that have either
not reported in a long time or in the worst case, never reported, this query devolves to a full table scan.
In a database with 1000+ sensors and 41 million rows, an unconstrained query takes over an hour.

#### JOIN LATERAL

An alternative to [SELECT DISTINCT ON][select-distinct-on] is to use a `JOIN LATERAL`. By selecting your entire
sensor list from the sensors table rather than pulling the IDs out using `SELECT DISTINCT`, `JOIN LATERAL` can offer
some improvements in performance:

```sql
SELECT sensor_list.id, latest_data.ts, latest_data.d
FROM sensors sensor_list
    -- Add a WHERE clause here to downselect the sensor list, if you wish
LEFT JOIN LATERAL (
    SELECT ts, d
    FROM iot_data raw_data
    WHERE sensor_id = sensor_list.id
    ORDER BY ts DESC
    LIMIT 1
) latest_data ON true
WHERE latest_data.d is not null -- only pulling out float values ("d" column) in this example
  AND latest_data.ts > CURRENT_TIMESTAMP - interval '1 week' -- important
ORDER BY sensor_list.id, latest_data.ts;
```

Limiting the time range is important, especially if you have a lot of data. Best practice is to use these
kinds of queries for dashboards and quick status checks. To query over a much larger time range, encapsulate
the previous example into a materialized query that refreshes infrequently, perhaps once a day.

Shoutout to **Christopher Piggott** for this recipe.


===== PAGE: https://docs.tigerdata.com/tutorials/blockchain-query/ =====

# Query the Bitcoin blockchain



The financial industry is extremely data-heavy and relies on real-time and historical data for decision-making, risk assessment, fraud detection, and market analysis. Tiger Data simplifies management of these large volumes of data, while also providing you with meaningful analytical insights and optimizing storage costs.

In this tutorial, you use Tiger Cloud to ingest, store, and analyze transactions
on the Bitcoin blockchain.

[Blockchains][blockchain-def] are, at their essence, a distributed database. The
[transactions][transactions-def] in a blockchain are an example of time-series data. You can use
TimescaleDB to query transactions on a blockchain, in exactly the same way as you
might query time-series transactions in any other database.

## Steps in this tutorial

This tutorial covers:

1.  [Ingest data into a service][blockchain-dataset]: set up and connect to a Tiger Cloud service, create tables and hypertables, and ingest data.
1.  [Query your data][blockchain-query]: obtain information, including finding the most recent transactions on the blockchain, and
   gathering information about the transactions using aggregation functions.
1.  [Compress your data using hypercore][blockchain-compress]: compress data that is no longer needed for highest performance queries, but is still accessed regularly
    for real-time analytics.

When you've completed this tutorial, you can use the same dataset to  [Analyze the Bitcoin data][analyze-blockchain],
using TimescaleDB hyperfunctions.


===== PAGE: https://docs.tigerdata.com/tutorials/blockchain-analyze/ =====

# Analyze the Bitcoin blockchain



The financial industry is extremely data-heavy and relies on real-time and historical data for decision-making, risk assessment, fraud detection, and market analysis. Tiger Data simplifies management of these large volumes of data, while also providing you with meaningful analytical insights and optimizing storage costs.

In this tutorial, you use Tiger Cloud to ingest, store, and analyze transactions
on the Bitcoin blockchain.

[Blockchains][blockchain-def] are, at their essence, a distributed database. The
[transactions][transactions-def] in a blockchain are an example of time-series data. You can use
TimescaleDB to query transactions on a blockchain, in exactly the same way as you
might query time-series transactions in any other database.

## Prerequisites

Before you begin, make sure you have:

*   Signed up for a [free Tiger Data account][cloud-install].
*   [](#)Signed up for a [Grafana account][grafana-setup] to graph your queries.

## Steps in this tutorial

This tutorial covers:

1.  [Setting up your dataset][blockchain-dataset]
1.  [Querying your dataset][blockchain-analyze]

## About analyzing the Bitcoin blockchain with Tiger Cloud

This tutorial uses a sample Bitcoin dataset to show you how to aggregate
blockchain transaction data, and construct queries to analyze information from
the aggregations. The queries in this tutorial help you
determine if a cryptocurrency has a high transaction fee, shows any correlation
between transaction volumes and fees, or if it's expensive to mine.

It starts by setting up and connecting to a Tiger Cloud service, create tables,
and load data into the tables using `psql`. If you have already completed the
[beginner blockchain tutorial][blockchain-query], then you already have the
dataset loaded, and you can skip straight to the queries.

You then learn how to conduct analysis on your dataset using Timescale
hyperfunctions. It walks you through creating a series of continuous aggregates,
and querying the aggregates to analyze the data. You can also use those queries
to graph the output in Grafana.


===== PAGE: https://docs.tigerdata.com/tutorials/financial-tick-data/ =====

# Analyze financial tick data with TimescaleDB



The financial industry is extremely data-heavy and relies on real-time and historical data for decision-making, risk assessment, fraud detection, and market analysis. Tiger Data simplifies management of these large volumes of data, while also providing you with meaningful analytical insights and optimizing storage costs.

To analyze financial data, you can chart the open, high, low, close, and volume
(OHLCV) information for a financial asset. Using this data, you can create
candlestick charts that make it easier to analyze the price changes of financial
assets over time. You can use candlestick charts to examine trends in stock,
cryptocurrency, or NFT prices.

In this tutorial, you use real raw financial data provided by
[Twelve Data][twelve-data], create an aggregated candlestick view, query the
aggregated data, and visualize the data in Grafana.

## OHLCV data and candlestick charts

The financial sector regularly uses [candlestick charts][charts] to visualize
the price change of an asset. Each candlestick represents a time period, such as
one minute or one hour, and shows how the asset's price changed during that time.

Candlestick charts are generated from the open, high, low, close, and volume
data for each financial asset during the time period. This is often abbreviated
as OHLCV:

*   Open: opening price
*   High: highest price
*   Low: lowest price
*   Close: closing price
*   Volume: volume of transactions

![candlestick](https://assets.timescale.com/docs/images/tutorials/intraday-stock-analysis/timescale_cloud_candlestick.png)

TimescaleDB is well suited to storing and analyzing financial candlestick data,
and many Tiger Data community members use it for exactly this purpose. Check out
these stories from some Tiger Datacommunity members:

*   [How Trading Strategy built a data stack for crypto quant trading][trading-strategy]
*   [How Messari uses data to open the cryptoeconomy to everyone][messari]
*   [How I power a (successful) crypto trading bot with TimescaleDB][bot]

## Steps in this tutorial

This tutorial shows you how to ingest real-time time-series data into a Tiger Cloud service:

1.  [Ingest data into a service][financial-tick-dataset]: load data from
    [Twelve Data][twelve-data] into your TimescaleDB database.
1.  [Query your dataset][financial-tick-query]: create candlestick views, query
    the aggregated data, and visualize the data in Grafana.
1.  [Compress your data using hypercore][financial-tick-compress]: learn how to store and query
your financial tick data more efficiently using compression feature of TimescaleDB.


To create candlestick views, query the aggregated data, and visualize the data in Grafana, see the
[ingest real-time websocket data section][advanced-websocket].


===== PAGE: https://docs.tigerdata.com/tutorials/financial-ingest-real-time/ =====

# Ingest real-time financial data using WebSocket



The financial industry is extremely data-heavy and relies on real-time and historical data for decision-making, risk assessment, fraud detection, and market analysis. Tiger Data simplifies management of these large volumes of data, while also providing you with meaningful analytical insights and optimizing storage costs.

This tutorial shows you how to ingest real-time time-series data into
TimescaleDB using a websocket connection. The tutorial sets up a data pipeline
to ingest real-time data from our data partner, [Twelve Data][twelve-data].
Twelve Data provides a number of different financial APIs, including stock,
cryptocurrencies, foreign exchanges, and ETFs. It also supports websocket
connections in case you want to update your database frequently. With
websockets, you need to connect to the server, subscribe to symbols, and you can
start receiving data in real-time during market hours.

When you complete this tutorial, you'll have a data pipeline set
up that ingests real-time financial data into your Tiger Cloud.

This tutorial uses Python and the API
[wrapper library][twelve-wrapper] provided by Twelve Data.

## Prerequisites

Before you begin, make sure you have:

*   Signed up for a [free Tiger Data account][cloud-install].
*   Installed Python 3
*   Signed up for [Twelve Data][twelve-signup]. The free tier is perfect for
    this tutorial.
*   Made a note of your Twelve Data [API key](https://twelvedata.com/account/api-keys).

## Steps in this tutorial

This tutorial covers:

1.  [Setting up your dataset][financial-ingest-dataset]: Load data from
    [Twelve Data][twelve-data] into your TimescaleDB database.
1.  [Querying your dataset][financial-ingest-query]: Create candlestick views, query
    the aggregated data, and visualize the data in Grafana.

    This tutorial shows you how to ingest real-time time-series data into a Tiger Cloud service using a websocket connection. To create candlestick views, query the
    aggregated data, and visualize the data in Grafana.

## About OHLCV data and candlestick charts

The financial sector regularly uses [candlestick charts][charts] to visualize
the price change of an asset. Each candlestick represents a time period, such as
one minute or one hour, and shows how the asset's price changed during that time.

Candlestick charts are generated from the open, high, low, close, and volume
data for each financial asset during the time period. This is often abbreviated
as OHLCV:

*   Open: opening price
*   High: highest price
*   Low: lowest price
*   Close: closing price
*   Volume: volume of transactions

![candlestick](https://assets.timescale.com/docs/images/tutorials/intraday-stock-analysis/candlestick_fig.png)

TimescaleDB is well suited to storing and analyzing financial candlestick data,
and many Tiger Datacommunity members use it for exactly this purpose.


===== PAGE: https://docs.tigerdata.com/api/hypertable/ =====

# Hypertables and chunks



Tiger Cloud supercharges your real-time analytics by letting you run complex queries continuously, with near-zero latency. Under the hood, this is achieved by using hypertables—Postgres tables that automatically partition your time-series data by time and optionally by other dimensions. When you run a query, Tiger Cloud identifies the correct partition, called chunk, and runs the query on it, instead of going through the entire table.

![Hypertable structure](https://assets.timescale.com/docs/images/hypertable.png)

Hypertables offer the following benefits:

- **Efficient data management with [automated partitioning by time][chunk-size]**: Tiger Cloud splits your data into chunks that hold data from a specific time range. For example, one day or one week. You can configure this range to better suit your needs.

- **Better performance with [strategic indexing][hypertable-indexes]**: an index on time in the descending order is automatically created when you create a hypertable. More indexes are created on the chunk level, to optimize performance. You can create additional indexes, including unique indexes, on the columns you need.

- **Faster queries with [chunk skipping][chunk-skipping]**: Tiger Cloud skips the chunks that are irrelevant in the context of your query, dramatically reducing the time and resources needed to fetch results. Even more—you can enable chunk skipping on non-partitioning columns.

- **Advanced data analysis with [hyperfunctions][hyperfunctions]**: Tiger Cloud enables you to efficiently process, aggregate, and analyze significant volumes of data while maintaining high performance.

To top it all, there is no added complexity—you interact with hypertables in the same way as you would with regular Postgres tables. All the optimization magic happens behind the scenes.



Inheritance is not supported for hypertables and may lead to unexpected behavior.

For more information about using hypertables, including chunk size partitioning,
see the [hypertable section][hypertable-docs].

## The hypertable workflow

Best practice for using a hypertable is to:

1. **Create a hypertable**

   Create a [hypertable][hypertables-section] for your time-series data using [CREATE TABLE][hypertable-create-table].
   For [efficient queries][secondary-indexes] on data in the columnstore, remember to `segmentby` the column you will
   use most often to filter your data. For example:

   ```sql
   CREATE TABLE conditions (
      time        TIMESTAMPTZ       NOT NULL,
      location    TEXT              NOT NULL,
      device      TEXT              NOT NULL,
      temperature DOUBLE PRECISION  NULL,
      humidity    DOUBLE PRECISION  NULL
   ) WITH (
      tsdb.hypertable,
      tsdb.partition_column='time',
      tsdb.segmentby = 'device',
      tsdb.orderby = 'time DESC'
   );
   ```
   If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

1. **Set the columnstore policy**

   ```sql
   CALL add_columnstore_policy('conditions', after => INTERVAL '1d');
   ```


===== PAGE: https://docs.tigerdata.com/api/hypercore/ =====

# Hypercore



Hypercore is a hybrid row-columnar storage engine in TimescaleDB. It is designed specifically for
real-time analytics and powered by time-series data. The advantage of hypercore is its ability
to seamlessly switch between row-oriented and column-oriented storage, delivering the best of both worlds:

![Hypercore workflow](https://assets.timescale.com/docs/images/hypertable-with-hypercore-enabled.png)

Hypercore solves the key challenges in real-time analytics:

- High ingest throughput
- Low-latency ingestion
- Fast query performance
- Efficient handling of data updates and late-arriving data
- Streamlined data management

Hypercore’s hybrid approach combines the benefits of row-oriented and column-oriented formats:

- **Fast ingest with rowstore**: new data is initially written to the rowstore, which is optimized for
  high-speed inserts and updates. This process ensures that real-time applications easily handle
  rapid streams of incoming data. Mutability—upserts, updates, and deletes happen seamlessly.

- **Efficient analytics with columnstore**: as the data **cools** and becomes more suited for
  analytics, it is automatically converted to the columnstore. This columnar format enables
  fast scanning and aggregation, optimizing performance for analytical workloads while also
  saving significant storage space.

- **Faster queries on compressed data in columnstore**: in the columnstore conversion, hypertable
  chunks are compressed by up to 98%, and organized for efficient, large-scale queries. Combined with [chunk skipping][chunk-skipping], this helps you save on storage costs and keeps your queries operating at lightning speed.

- **Fast modification of compressed data in columnstore**: just use SQL to add or modify data in the columnstore.
   TimescaleDB is optimized for superfast INSERT and UPSERT performance.

- **Full mutability with transactional semantics**: regardless of where data is stored,
  hypercore provides full ACID support. Like in a vanilla Postgres database, inserts and updates
  to the rowstore and columnstore are always consistent, and available to queries as soon as they are
  completed.

For an in-depth explanation of how hypertables and hypercore work, see the [Data model][data-model].

Since [TimescaleDB v2.18.0](https://github.com/timescale/timescaledb/releases/tag/2.18.0)

## Hypercore workflow

Best practice for using hypercore is to:

1. **Enable columnstore**

   Create a [hypertable][hypertables-section] for your time-series data using [CREATE TABLE][hypertable-create-table].
   For [efficient queries][secondary-indexes] on data in the columnstore, remember to `segmentby` the column you will
   use most often to filter your data. For example:

   * [Use `CREATE TABLE` for a hypertable][hypertable-create-table]

     ```sql
     CREATE TABLE crypto_ticks (
        "time" TIMESTAMPTZ,
        symbol TEXT,
        price DOUBLE PRECISION,
        day_volume NUMERIC
     ) WITH (
       tsdb.hypertable,
       tsdb.partition_column='time',
       tsdb.segmentby='symbol',
       tsdb.orderby='time DESC'
     );
     ```
     If you are self-hosting TimescaleDB v2.19.3 and below, create a [Postgres relational table][pg-create-table],
then convert it using [create_hypertable][create_hypertable]. You then enable hypercore with a call
to [ALTER TABLE][alter_table_hypercore].

   * [Use `ALTER MATERIALIZED VIEW` for a continuous aggregate][compression_continuous-aggregate]
     ```sql
     ALTER MATERIALIZED VIEW assets_candlestick_daily set (
        timescaledb.enable_columnstore = true,
        timescaledb.segmentby = 'symbol' );
     ```

1. **Add a policy to move chunks to the columnstore at a specific time interval**

   For example, 7 days after the data was added to the table:
   ``` sql
   CALL add_columnstore_policy('crypto_ticks', after => INTERVAL '7d');
   ```
   See [add_columnstore_policy][add_columnstore_policy].

1. **View the policies that you set or the policies that already exist**

   ``` sql
   SELECT * FROM timescaledb_information.jobs
   WHERE proc_name='policy_compression';
   ```
   See [timescaledb_information.jobs][informational-views].

You can also [convert_to_columnstore][convert_to_columnstore] and [convert_to_rowstore][convert_to_rowstore] manually
for more fine-grained control over your data.

## Limitations

Chunks in the columnstore have the following limitations:

*   `ROW LEVEL SECURITY` is not supported on chunks in the columnstore.


===== PAGE: https://docs.tigerdata.com/api/continuous-aggregates/ =====

# Continuous aggregates



In modern applications, data usually grows very quickly. This means that aggregating
it into useful summaries can become very slow. If you are collecting data very frequently, you might want to aggregate your
data into minutes or hours instead. For example, if an IoT device takes
temperature readings every second, you might want to find the average temperature
for each hour. Every time you run this query, the database needs to scan the
entire table and recalculate the average. TimescaleDB makes aggregating data lightning fast, accurate, and easy with continuous aggregates.

![Reduced data calls with continuous aggregates](https://assets.timescale.com/docs/images/continuous-aggregate.png)

Continuous aggregates in TimescaleDB are a kind of hypertable that is refreshed automatically
in the background as new data is added, or old data is modified. Changes to your
dataset are tracked, and the hypertable behind the continuous aggregate is
automatically updated in the background.

Continuous aggregates have a much lower maintenance burden than regular Postgres materialized
views, because the whole view is not created from scratch on each refresh. This
means that you can get on with working your data instead of maintaining your
database.

Because continuous aggregates are based on hypertables, you can query them in exactly the same way as your other tables. This includes continuous aggregates in the rowstore, compressed into the [columnstore][hypercore],
or [tiered to object storage][data-tiering]. You can even create [continuous aggregates on top of your continuous aggregates][hierarchical-caggs], for an even more fine-tuned aggregation.

[Real-time aggregation][real-time-aggregation] enables you to combine pre-aggregated data from the materialized view with the most recent raw data. This gives you up-to-date results on every query. In TimescaleDB v2.13 and later, real-time aggregates are **DISABLED** by default. In earlier versions, real-time aggregates are **ENABLED** by default; when you create a continuous aggregate, queries to that view include the results from the most recent raw data.

For more information about using continuous aggregates, see the documentation in [Use Tiger Data products][cagg-docs].


===== PAGE: https://docs.tigerdata.com/api/data-retention/ =====

# Data retention

An intrinsic part of time-series data is that new data is accumulated and old
data is rarely, if ever, updated. This means that the relevance of the data
diminishes over time. It is therefore often desirable to delete old data to save
disk space.

With TimescaleDB, you can manually remove old chunks of data or implement
policies using these APIs.

For more information about creating a data retention policy, see the
[data retention section][data-retention-howto].


===== PAGE: https://docs.tigerdata.com/api/jobs-automation/ =====

# Jobs

Jobs allow you to run functions and procedures implemented in a
language of your choice on a schedule within Timescale. This allows
automatic periodic tasks that are not covered by existing policies and
even enhancing existing policies with additional functionality.

The following APIs and views allow you to manage the jobs that you create and
get details around automatic jobs used by other TimescaleDB functions like
continuous aggregation refresh policies and data retention policies. To view the
policies that you set or the policies that already exist, see
[informational views][informational-views].


===== PAGE: https://docs.tigerdata.com/api/uuid-functions/ =====

# UUIDv7 functions

UUIDv7 is a time-ordered UUID that includes a Unix timestamp (with millisecond precision) in its first 48 bits. Like
other UUIDs, it uses 6 bits for version and variant info, and the remaining 74 bits are random.

![UUIDv7 microseconds](https://assets.timescale.com/docs/images/uuidv7-structure-microseconds.svg)

UUIDv7 is ideal anywhere you create lots of records over time, not only observability. Advantages are:

- **No extra column required to partition by time with sortability**: you can sort UUIDv7 instances by their value. This
   is useful for ordering records by creation time without the need for a separate timestamp column.
- **Indexing performance**: UUIDv7s increase with time, so new rows append near the end of a B-tree instead of
   This results in fewer page splits, less fragmentation, faster inserts, and efficient time-range scans.
- **Easy keyset pagination**: `WHERE id > :cursor` and natural sharding.
- **UUID**: safe across services, replicas, and unique across distributed systems.

UUIDv7 also increases query speed by reducing the number of chunks scanned during queries. For example, in a database
with 25 million rows, the following query runs in 25 seconds:

```sql
WITH ref AS (SELECT now() AS t0)
SELECT count(*) AS cnt_ts_filter
FROM events e, ref
WHERE uuid_timestamp(e.event_id) >= ref.t0 - INTERVAL '2 days';
```

Using UUIDv7 excludes chunks at startup and reduces the query time to 550ms:

```sql
WITH ref AS (SELECT now() AS t0)
SELECT count(*) AS cnt_boundary_filter
FROM events e, ref
WHERE e.event_id >= to_uuidv7_boundary(ref.t0 - INTERVAL '2 days')
```



You use UUIDvs for events, orders, messages, uploads, runs, jobs, spans, and more.

## Examples

- **High-rate event logs for observability and metrics**:

   UUIDv7 gives you globally unique IDs (for traceability) and time windows (“last hour”), without the need for a
   separate `created_at` column. UUIDv7 create less churn because inserts land at the end of the index, and you can
   filter by time using UUIDv7 objects.

  - Last hour:
      ```sql
      SELECT count(*) FROM logs WHERE id >= to_uuidv7_boundary(now() - interval '1 hour');
      ```
  - Keyset pagination
      ```sql
      SELECT * FROM logs WHERE id > to_uuidv7($last_seen'::timestamptz, true) ORDER BY id LIMIT 1000;
      ```

- **Workflow / durable execution runs**:

   Each run needs a stable ID for joins and retries, and you often ask “what started since X?”. UUIDs help by serving
   both as the primary key and a time cursor across services. For example:

    ```sql
    SELECT run_id, status
    FROM runs
    WHERE run_id >= to_uuidv7_boundary(now() - interval '5 minutes')
    ```

- **Orders / activity feeds / messages (SaaS apps)**:

    Human-readable timestamps are not mandatory in a table. However, you still need time-ordered pages and day/week ranges.
    UUIDv7 enables clean date windows and cursor pagination with just the ID. For example:

    ```sql
    SELECT * FROM orders
    WHERE id >= to_uuidv7('2025-08-01'::timestamptz, true)
    AND id <  to_uuidv7('2025-08-02'::timestamptz, true)
    ORDER BY id;
    ```




## Functions

- [generate_uuidv7()][generate_uuidv7]: generate a version 7 UUID based on current time
- [to_uuidv7()][to_uuidv7]: create a version 7 UUID from a PostgreSQL timestamp
- [to_uuidv7_boundary()][to_uuidv7_boundary]: create a version 7 "boundary" UUID from a PostgreSQL timestamp
- [uuid_timestamp()][uuid_timestamp]: extract a PostgreSQL timestamp from a version 7 UUID
- [uuid_timestamp_micros()][uuid_timestamp_micros]: extract a PostgreSQL timestamp with microsecond precision from a version 7 UUID
- [uuid_version()][uuid_version]: extract the version of a UUID


===== PAGE: https://docs.tigerdata.com/api/approximate_row_count/ =====

# approximate_row_count()

Get approximate row count for hypertable, distributed hypertable, or regular Postgres table based on catalog estimates.
This function supports tables with nested inheritance and declarative partitioning.

The accuracy of `approximate_row_count` depends on the database having up-to-date statistics about the table or hypertable, which are updated by `VACUUM`, `ANALYZE`, and a few DDL commands. If you have auto-vacuum configured on your table or hypertable, or changes to the table are relatively infrequent, you might not need to explicitly `ANALYZE` your table as shown below. Otherwise, if your table statistics are too out-of-date, running this command updates your statistics and yields more accurate approximation results.

### Samples

Get the approximate row count for a single hypertable.

```sql
ANALYZE conditions;

SELECT * FROM approximate_row_count('conditions');
```

The expected output:

```
approximate_row_count
----------------------
               240000
```

### Required arguments

|Name|Type|Description|
|---|---|---|
| `relation` | REGCLASS | Hypertable or regular Postgres table to get row count for. |


===== PAGE: https://docs.tigerdata.com/api/first/ =====

# first()

The `first` aggregate allows you to get the value of one column
as ordered by another. For example, `first(temperature, time)` returns the
earliest temperature value based on time within an aggregate group.


The `last` and `first` commands do not use indexes, they perform a sequential
scan through the group. They are primarily used for ordered selection within a
`GROUP BY` aggregate, and not as an alternative to an
`ORDER BY time DESC LIMIT 1` clause to find the latest value, which uses
indexes.


### Samples

Get the earliest temperature by device_id:

```sql
SELECT device_id, first(temp, time)
FROM metrics
GROUP BY device_id;
```

This example uses first and last with an aggregate filter, and avoids null
values in the output:

```sql
SELECT
   TIME_BUCKET('5 MIN', time_column) AS interv,
   AVG(temperature) as avg_temp,
   first(temperature,time_column) FILTER(WHERE time_column IS NOT NULL) AS beg_temp,
   last(temperature,time_column) FILTER(WHERE time_column IS NOT NULL) AS end_temp
FROM sensors
GROUP BY interv
```

### Required arguments

|Name|Type|Description|
|---|---|---|
|`value`|TEXT|The value to return|
|`time`|TIMESTAMP or INTEGER|The timestamp to use for comparison|


===== PAGE: https://docs.tigerdata.com/api/last/ =====

# last()

The `last` aggregate allows you to get the value of one column
as ordered by another. For example, `last(temperature, time)` returns the
latest temperature value based on time within an aggregate group.


The `last` and `first` commands do not use indexes, they perform a sequential
scan through the group. They are primarily used for ordered selection within a
`GROUP BY` aggregate, and not as an alternative to an
`ORDER BY time DESC LIMIT 1` clause to find the latest value, which uses
indexes.


### Samples

Get the temperature every 5 minutes for each device over the past day:

```sql
SELECT device_id, time_bucket('5 minutes', time) AS interval,
  last(temp, time)
FROM metrics
WHERE time > now () - INTERVAL '1 day'
GROUP BY device_id, interval
ORDER BY interval DESC;
```

This example uses first and last with an aggregate filter, and avoids null
values in the output:

```sql
SELECT
   TIME_BUCKET('5 MIN', time_column) AS interv,
   AVG(temperature) as avg_temp,
   first(temperature,time_column) FILTER(WHERE time_column IS NOT NULL) AS beg_temp,
   last(temperature,time_column) FILTER(WHERE time_column IS NOT NULL) AS end_temp
FROM sensors
GROUP BY interv
```

### Required arguments

|Name|Type|Description|
|---|---|---|
|`value`|ANY ELEMENT|The value to return|
|`time`|TIMESTAMP or INTEGER|The timestamp to use for comparison|


===== PAGE: https://docs.tigerdata.com/api/histogram/ =====

# histogram()

The `histogram()` function represents the distribution of a set of
values as an array of equal-width buckets. It partitions the dataset
into a specified number of buckets (`nbuckets`) ranging from the
inputted `min` and `max` values.

The return value is an array containing `nbuckets`+2 buckets, with the
middle `nbuckets` bins for values in the stated range, the first
bucket at the head of the array for values under the lower `min` bound,
and the last bucket for values greater than or equal to the `max` bound.
Each bucket is inclusive on its lower bound, and exclusive on its upper
bound. Therefore, values equal to the `min` are included in the bucket
starting with `min`, but values equal to the `max` are in the last bucket.

### Samples

A simple bucketing of device's battery levels from the `readings` dataset:

```sql
SELECT device_id, histogram(battery_level, 20, 60, 5)
FROM readings
GROUP BY device_id
LIMIT 10;
```

The expected output:

```sql
 device_id  |          histogram
------------+------------------------------
 demo000000 | {0,0,0,7,215,206,572}
 demo000001 | {0,12,173,112,99,145,459}
 demo000002 | {0,0,187,167,68,229,349}
 demo000003 | {197,209,127,221,106,112,28}
 demo000004 | {0,0,0,0,0,39,961}
 demo000005 | {12,225,171,122,233,80,157}
 demo000006 | {0,78,176,170,8,40,528}
 demo000007 | {0,0,0,126,239,245,390}
 demo000008 | {0,0,311,345,116,228,0}
 demo000009 | {295,92,105,50,8,8,442}
```

### Required arguments

|Name|Type|Description|
|---|---|---|
| `value` | ANY VALUE | A set of values to partition into a histogram |
| `min` | NUMERIC | The histogram's lower bound used in bucketing (inclusive) |
| `max` | NUMERIC | The histogram's upper bound used in bucketing (exclusive) |
| `nbuckets` | INTEGER | The integer value for the number of histogram buckets (partitions) |


===== PAGE: https://docs.tigerdata.com/api/time_bucket/ =====

# time_bucket()

The `time_bucket` function is similar to the standard Postgres `date_bin`
function. Unlike `date_bin`, it allows for arbitrary time intervals of months or
longer. The return value is the bucket's start time.

Buckets are aligned to start at midnight in UTC+0. The time bucket size (`bucket_width`) can be set as INTERVAL or INTEGER. For INTERVAL-type `bucket_width`, you can change the time zone with the optional `timezone` parameter. In this case, the buckets are realigned to start at midnight in the time zone you specify.

Note that during shifts to and from daylight savings, the amount of data
aggregated into the corresponding buckets can be irregular. For example, if the
`bucket_width` is 2 hours, the number of bucketed hours is either three hours or one hour.

## Samples

Simple five-minute averaging:

```sql
SELECT time_bucket('5 minutes', time) AS five_min, avg(cpu)
FROM metrics
GROUP BY five_min
ORDER BY five_min DESC LIMIT 10;
```

To report the middle of the bucket, instead of the left edge:

```sql
SELECT time_bucket('5 minutes', time) + '2.5 minutes'
  AS five_min, avg(cpu)
FROM metrics
GROUP BY five_min
ORDER BY five_min DESC LIMIT 10;
```

For rounding, move the alignment so that the middle of the bucket is at the
five-minute mark, and report the middle of the bucket:

```sql
SELECT time_bucket('5 minutes', time, '-2.5 minutes'::INTERVAL) + '2.5 minutes'
  AS five_min, avg(cpu)
FROM metrics
GROUP BY five_min
ORDER BY five_min DESC LIMIT 10;
```

In this example, add the explicit cast to ensure that Postgres chooses the
correct function.

To shift the alignment of the buckets, you can use the origin parameter passed as
a timestamp, timestamptz, or date type. This example shifts the start of the
week to a Sunday, instead of the default of Monday:

```sql
SELECT time_bucket('1 week', timetz, TIMESTAMPTZ '2017-12-31')
  AS one_week, avg(cpu)
FROM metrics
GROUP BY one_week
WHERE time > TIMESTAMPTZ '2017-12-01'  AND time < TIMESTAMPTZ '2018-01-03'
ORDER BY one_week DESC LIMIT 10;
```

The value of the origin parameter in this example is `2017-12-31`, a Sunday
within the period being analyzed. However, the origin provided to the function
can be before, during, or after the data being analyzed. All buckets are
calculated relative to this origin. So, in this example, any Sunday could have
been used. Note that because `time < TIMESTAMPTZ '2018-01-03'` is used in this
example, the last bucket would have only 4 days of data. This cast to TIMESTAMP
converts the time to local time according to the server's time zone setting.

```sql
SELECT time_bucket(INTERVAL '2 hours', timetz::TIMESTAMP)
  AS five_min, avg(cpu)
FROM metrics
GROUP BY five_min
ORDER BY five_min DESC LIMIT 10;
```

Bucket temperature values to calculate the average monthly temperature. Set the
time zone to 'Europe/Berlin' so bucket start and end times are aligned to
midnight in Berlin.

```sql
SELECT time_bucket('1 month', ts, 'Europe/Berlin') AS month_bucket,
  avg(temperature) AS avg_temp
FROM weather
GROUP BY month_bucket
ORDER BY month_bucket DESC LIMIT 10;
```

## Required arguments for interval time inputs

|Name|Type|Description|
|-|-|-|
|`bucket_width`|INTERVAL|A Postgres time interval for how long each bucket is|
|`ts`|DATE, TIMESTAMP, or TIMESTAMPTZ|The timestamp to bucket|

If you use months as an interval for `bucket_width`, you cannot combine it with
a non-month component. For example, `1 month` and `3 months` are both valid
bucket widths, but `1 month 1 day` and `3 months 2 weeks` are not.

## Optional arguments for interval time inputs

|Name|Type| Description                                                                                                                                                                                                                                                                                    |
|-|-|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
|`timezone`|TEXT| The time zone for calculating bucket start and end times. Can only be used with `TIMESTAMPTZ`. Defaults to UTC+0.                                                                                                                                                                              |
|`origin`|DATE, TIMESTAMP, or TIMESTAMPTZ| Buckets are aligned relative to this timestamp. Defaults to midnight on January 3, 2000, for buckets that don't include a month or year interval, and to midnight on January 1, 2000, for month, year, and century buckets.                                                                    |
|`offset`|INTERVAL| The time interval to offset all time buckets by. A positive value shifts bucket start and end times later. A negative value shifts bucket start and end times earlier. `offset` must be surrounded with double quotes when used as a named argument, because it is a reserved key word in Postgres. |

## Required arguments for integer time inputs

|Name|Type|Description|
|-|-|-|
|`bucket_width`|INTEGER|The bucket width|
|`ts`|INTEGER|The timestamp to bucket|

## Optional arguments for integer time inputs

|Name|Type|Description|
|-|-|-|
|`offset`|INTEGER|The amount to offset all buckets by. A positive value shifts bucket start and end times later. A negative value shifts bucket start and end times earlier. `offset` must be surrounded with double quotes when used as a named argument, because it is a reserved key word in Postgres.|


===== PAGE: https://docs.tigerdata.com/api/time_bucket_ng/ =====

# timescaledb_experimental.time_bucket_ng()



The `time_bucket_ng()` function is an experimental version of the
[`time_bucket()`][time_bucket] function. It introduced some new capabilities,
such as monthly buckets and timezone support. Those features are now part of the
regular `time_bucket()` function.

This section describes a feature that is deprecated. We strongly
recommend that you do not use this feature in a production environment. If you
need more information, [contact us](https://www.tigerdata.com/contact/).


The `time_bucket()` and `time_bucket_ng()` functions are similar, but not
completely compatible. There are two main differences.

Firstly, `time_bucket_ng()` doesn't work with timestamps prior to `origin`,
while `time_bucket()` does.

Secondly, the default `origin` values differ. `time_bucket()` uses an origin
date of January 3, 2000, for buckets shorter than a month. `time_bucket_ng()`
uses an origin date of January 1, 2000, for all bucket sizes.


### Samples

In this example, `time_bucket_ng()` is used to create bucket data in three month
intervals:

```sql
SELECT timescaledb_experimental.time_bucket_ng('3 month', date '2021-08-01');
 time_bucket_ng
----------------
 2021-07-01
(1 row)
```

This example uses `time_bucket_ng()` to bucket data in one year intervals:

```sql
SELECT timescaledb_experimental.time_bucket_ng('1 year', date '2021-08-01');
 time_bucket_ng
----------------
 2021-01-01
(1 row)
```

To split time into buckets, `time_bucket_ng()` uses a starting point in time
called `origin`. The default origin is `2000-01-01`. `time_bucket_ng` cannot use
timestamps earlier than `origin`:

```sql
SELECT timescaledb_experimental.time_bucket_ng('100 years', timestamp '1988-05-08');
ERROR:  origin must be before the given date
```

Going back in time from `origin` isn't usually possible, especially when you
consider timezones and daylight savings time (DST). Note also that there is no
reasonable way to split time in variable-sized buckets (such as months) from an
arbitrary `origin`, so `origin` defaults to the first day of the month.

To bypass named limitations, you can override the default `origin`:

```sql
-- working with timestamps before 2000-01-01
SELECT timescaledb_experimental.time_bucket_ng('100 years', timestamp '1988-05-08', origin => '1900-01-01');
   time_bucket_ng
---------------------
 1900-01-01 00:00:00

-- unlike the default origin, which is Saturday, 2000-01-03 is Monday
SELECT timescaledb_experimental.time_bucket_ng('1 week', timestamp '2021-08-26', origin => '2000-01-03');
   time_bucket_ng
---------------------
 2021-08-23 00:00:00
```

This example shows how `time_bucket_ng()` is used to bucket data
by months in a specified timezone:

```sql
-- note that timestamptz is displayed differently depending on the session parameters
SET TIME ZONE 'Europe/Moscow';
SET

SELECT timescaledb_experimental.time_bucket_ng('1 month', timestamptz '2001-02-03 12:34:56 MSK', timezone => 'Europe/Moscow');
     time_bucket_ng
------------------------
 2001-02-01 00:00:00+03
```

You can use `time_bucket_ng()` with continuous aggregates. This example tracks
the temperature in Moscow over seven day intervals:

```sql
CREATE TABLE conditions(
  day DATE NOT NULL,
  city text NOT NULL,
  temperature INT NOT NULL);

SELECT create_hypertable(
  'conditions', by_range('day', INTERVAL '1 day')
);

INSERT INTO conditions (day, city, temperature) VALUES
  ('2021-06-14', 'Moscow', 26),
  ('2021-06-15', 'Moscow', 22),
  ('2021-06-16', 'Moscow', 24),
  ('2021-06-17', 'Moscow', 24),
  ('2021-06-18', 'Moscow', 27),
  ('2021-06-19', 'Moscow', 28),
  ('2021-06-20', 'Moscow', 30),
  ('2021-06-21', 'Moscow', 31),
  ('2021-06-22', 'Moscow', 34),
  ('2021-06-23', 'Moscow', 34),
  ('2021-06-24', 'Moscow', 34),
  ('2021-06-25', 'Moscow', 32),
  ('2021-06-26', 'Moscow', 32),
  ('2021-06-27', 'Moscow', 31);

CREATE MATERIALIZED VIEW conditions_summary_weekly
WITH (timescaledb.continuous) AS
SELECT city,
       timescaledb_experimental.time_bucket_ng('7 days', day) AS bucket,
       MIN(temperature),
       MAX(temperature)
FROM conditions
GROUP BY city, bucket;

SELECT to_char(bucket, 'YYYY-MM-DD'), city, min, max
FROM conditions_summary_weekly
ORDER BY bucket;

  to_char   |  city  | min | max
------------+--------+-----+-----
 2021-06-12 | Moscow |  22 |  27
 2021-06-19 | Moscow |  28 |  34
 2021-06-26 | Moscow |  31 |  32
(3 rows)
```


The `by_range` dimension builder is an addition to TimescaleDB
2.13. For simpler cases, like this one, you can also create the
hypertable using the old syntax:

```sql
SELECT create_hypertable('', '<time column name>');
```


For more information, see the [continuous aggregates documentation][caggs].


While `time_bucket_ng()` supports months and timezones,
continuous aggregates cannot always be used with monthly
buckets or buckets with timezones.


This table shows which `time_bucket_ng()` functions can be used in a continuous aggregate:

|Function|Available in continuous aggregate|TimescaleDB version|
|-|-|-|
|Buckets by seconds, minutes, hours, days, and weeks|✅|2.4.0 - 2.14.2|
|Buckets by months and years|✅|2.6.0 - 2.14.2|
|Timezones support|✅|2.6.0 - 2.14.2|
|Specify custom origin|✅|2.7.0 - 2.14.2|



### Required arguments

|Name|Type|Description|
|---|---|---|
| `bucket_width` | INTERVAL | A Postgres time interval for how long each bucket is |
| `ts` | DATE, TIMESTAMP or TIMESTAMPTZ | The timestamp to bucket |

### Optional arguments

|Name|Type|Description|
|---|---|---|
| `origin` | Should be the same as `ts` | Buckets are aligned relative to this timestamp |
| `timezone` | TEXT | The name of the timezone. The argument can be specified only if the type of `ts` is TIMESTAMPTZ |

For backward compatibility with `time_bucket()` the `timezone` argument is
optional. However, it is required for time buckets that are less than 24 hours.

If you call the TIMESTAMPTZ-version of the function without the `timezone`
argument, the timezone defaults to the session's timezone and so the function
can't be used with continuous aggregates. Best practice is to use
`time_bucket_ng(interval, timestamptz, text)` and specify the timezone.

### Returns

The function returns the bucket's start time. The return value type is the
same as `ts`.


===== PAGE: https://docs.tigerdata.com/api/days_in_month/ =====

# days_in_month()

Given a timestamptz, returns how many days are in that month.

### Samples

Calculate how many days in the month of January 1, 2022:

```sql
SELECT days_in_month('2021-01-01 00:00:00+03'::timestamptz)
```

The output looks like this:

```sql
days_in_month
----------------------
31
```

### Required arguments

|Name|Type|Description|
|-|-|-|
|`date`|`TIMESTAMPTZ`|Timestamp to use to calculate how many days in the month|


===== PAGE: https://docs.tigerdata.com/api/month_normalize/ =====

# month_normalize()

Translate a metric to a standard month. A standard month is calculated as the exact number of days in a year divided by the number of months in a year, so 365.25/12 = 30.4375. `month_normalize()` divides a metric by the number of days in the corresponding calendar month and multiplies it by 30.4375.

This enables you to compare metrics for different months and decide which one performed better, objectively. For example, in the following table that summarizes the number of sales for three months, January has the highest number of total sales:

| Month | Sales |
|-------|-------|
| Jan   | 3000  |
| Feb   | 2900  |
| Mar   | 2900  |

When you normalize the sales metrics, you get the following result, showing that February in fact performed better:

| Month | Normalized sales  |
|-------|-------------------|
| Jan   | 2945.56           |
| Feb   | 3152.46           |
| Mar   | 2847.38           |


### Samples

Get the normalized value for a metric of 1000, and a reference date of January
1, 2021:

```sql
SELECT month_normalize(1000,'2021-01-01 00:00:00+03'::timestamptz)
```

The output looks like this:

```sql
month_normalize
----------------------
981.8548387096774
```

### Required arguments

|Name|Type|Description|
|-|-|-|
|`metric`|`float8`||
|`reference_date`|`TIMESTAMPTZ`|Timestamp to normalize the metric with|
|`days`|`float8`|Optional, defaults to 365.25/12 if none provided|


===== PAGE: https://docs.tigerdata.com/api/gauge_agg/ =====

# gauge_agg()



Produces a `GaugeSummary` that can be used to accumulate gauge data for further
calculations.

```sql
gauge_agg (
    ts TIMESTAMPTZ,
    value DOUBLE PRECISION
) RETURNS GaugeSummary
```

Experimental features could have bugs. They might not be backwards compatible,
and could be removed in future releases. Use these features at your own risk, and
do not use any experimental features in production.

For more information about counter and gauge aggregation functions, see the
[hyperfunctions documentation][hyperfunctions-counter-agg].

## Required arguments

|Name|Type|Description
|-|-|-|
|`ts`|`TIMESTAMPTZ`|The time at each point|
|`value`|`DOUBLE PRECISION`|The value at that timestamp|

Only `DOUBLE PRECISION` values are accepted for the `value` parameter. For gauge
data stored as other numeric types, cast it to `DOUBLE PRECISION` when using the
function.


If there are `NULL` values in your data, the aggregate ignores them and
aggregates only non-`NULL` values. If you only have `NULL` values, the aggregate
returns `NULL`.


## Optional arguments

|Name|Type|Description|
|-|-|-|
|`bounds`|`TSTZRANGE`|The largest and smallest possible times that can be input to the aggregate. Calling with `NULL`, or leaving out the argument, results in an unbounded `GaugeSummary`|


Bounds are required for extrapolation, but not for other accessor functions.


## Returns

|Column|Type|Description|
|-|-|-|
|`gauge_agg`|`GaugeSummary`|A `GaugeSummary` object that can be passed to accessor functions or other objects in the gauge aggregate API|


The returned `GaugeSummary` can be used as an input the accessor functions
`delta`, `idelta_left`, and `idelta_right`. When this feature is mature, it will support
all the same accessor functions as `CounterSummary`, with the exception of
`num_resets`.


## Sample usage

Create a gauge summary from time-series data that has a timestamp, `ts`, and a
gauge value, `val`. Get the instantaneous rate of change from the last 2 time
intervals using the `irate_right` accessor:

```sql
WITH t as (
    SELECT
        time_bucket('1 day'::interval, ts) as dt,
        gauge_agg(ts, val) AS gs
    FROM foo
    WHERE id = 'bar'
    GROUP BY time_bucket('1 day'::interval, ts)
)
SELECT
    dt,
    irate_right(gs)
FROM t;
```


===== PAGE: https://docs.tigerdata.com/api/frequency-analysis/ =====

# Frequency analysis

This section includes frequency aggregate APIs, which find the most common elements out of a set of
vastly more varied values.

For these hyperfunctions, you need to install the [TimescaleDB Toolkit][install-toolkit] Postgres extension.

<HyperfunctionTable
    hyperfunctionFamily='frequency analysis'
    includeExperimental
    sortByType
/>


===== PAGE: https://docs.tigerdata.com/api/informational-views/ =====

# Information views

TimescaleDB makes complex database features like partitioning and data retention
easy to use with our comprehensive APIs. TimescaleDB works hard to provide
detailed information about the state of your data, hypertables, chunks, and any
jobs or policies you have in place.

These views provide the data and statistics you need to keep track of your
database.


===== PAGE: https://docs.tigerdata.com/api/configuration/ =====

# Service configuration

Tiger Cloud service use the default Postgres server configuration settings. You can optimize your service configuration
using the following TimescaleDB and Grand Unified Configuration (GUC) parameters.

* [TimescaleDB configuration and tuning][tigerpostgres-config]
* [Grand Unified Configuration (GUC) parameters][gucs]


===== PAGE: https://docs.tigerdata.com/api/administration/ =====

# Administrative functions

These administrative APIs help you prepare a database before and after a restore event. They also help you keep track of your TimescaleDB setup data.

## Dump TimescaleDB meta data

To help when asking for support and reporting bugs, TimescaleDB includes an SQL dump script. It outputs metadata from the internal TimescaleDB tables, along with version information.

This script is available in the source distribution in `scripts/`. To use it, run:

```bash
psql [your connect flags] -d your_timescale_db < dump_meta_data.sql > dumpfile.txt
```

Inspect `dumpfile.txt` before sending it together with a bug report or support question.

## get_telemetry_report()

Returns the background [telemetry][telemetry] string sent to Tiger Data.

If telemetry is turned off, it sends the string that would be sent if telemetry were enabled.

### Sample usage

View the telemetry report:

```sql
SELECT get_telemetry_report();
```

## timescaledb_post_restore()

Perform the required operations after you have finished restoring the database using `pg_restore`. Specifically, this resets the `timescaledb.restoring` GUC and restarts any background workers.

For more information, see [Migrate using pg_dump and pg_restore].

### Sample usage

Prepare the database for normal use after a restore:

```sql
SELECT timescaledb_post_restore();
```

## timescaledb_pre_restore()

Perform the required operations so that you can restore the database using `pg_restore`. Specifically, this sets the `timescaledb.restoring` GUC to `on` and stops any background workers which could have been performing tasks.

The background workers are stopped until the [timescaledb_post_restore()](#timescaledb_post_restore) function is run, after the restore operation is complete.

For more information, see [Migrate using pg_dump and pg_restore].


After using `timescaledb_pre_restore()`, you need to run [`timescaledb_post_restore()`](#timescaledb_post_restore) before you can use the database normally.


### Sample usage

Prepare to restore the database:

```sql
SELECT timescaledb_pre_restore();
```


===== PAGE: https://docs.tigerdata.com/api/api-tag-overview/ =====

# API reference tag overview



The TimescaleDB API Reference uses tags to categorize functions. The tags are
`Community`, `Experimental`, `Toolkit`, and `Experimental (Toolkit)`. This
section explains each tag.

## Community Community

This tag indicates that the function is available under TimescaleDB Community
Edition, and are not available under the Apache 2 Edition. For more information,
visit our [TimescaleDB License comparison sheet][tsl-comparison].

## Experimental (TimescaleDB Experimental Schema) Experimental

This tag indicates that the function is included in the TimescaleDB experimental
schema. Do not use experimental functions in production. Experimental features
could include bugs, and are likely to change in future versions. The
experimental schema is used by TimescaleDB to develop new features more quickly.
If experimental functions are successful, they can move out of the experimental
schema and go into production use.

When you upgrade the `timescaledb` extension, the experimental schema is removed
by default. To use experimental features after an upgrade, you need to add the
experimental schema again.

For more information about the experimental
schema, [read the Tiger Data blog post][experimental-blog].

## Toolkit Toolkit

This tag indicates that the function is included in the TimescaleDB Toolkit extension.
Toolkit functions are available under TimescaleDB Community Edition.
For installation instructions, [see the installation guide][toolkit-install].

## Experimental (TimescaleDB Toolkit) Experimental

This tag is used with the Toolkit tag. It indicates a Toolkit function that is
under active development. Do not use experimental toolkit functions in
production. Experimental toolkit functions could include bugs, and are likely to
change in future versions.

These functions might not correctly handle unusual use cases or errors, and they
could have poor performance. Updates to the TimescaleDB extension drop database
objects that depend on experimental features like this function. If you use
experimental toolkit functions on Timescale, this function is
automatically dropped when the Toolkit extension is updated. For more
information, [see the TimescaleDB Toolkit docs][toolkit-docs].


===== PAGE: https://docs.tigerdata.com/api/api-reference/ =====

# Tiger Cloud REST API reference

A comprehensive RESTful API for managing Tiger Cloud resources including VPCs, services, and read replicas.

## Overview

**API Version:** 1.0.0
**Base URL:** `https://console.cloud.timescale.com/public/api/v1`

## Authentication

The Tiger REST API uses HTTP Basic Authentication. Include your access key and secret key in the Authorization header.

### Basic Authentication
```http
Authorization: Basic <base64(access_key:secret_key)>
```

### Example
```bash
curl -X GET "https://console.cloud.timescale.com/public/api/v1/projects/{project_id}/services" \
  -H "Authorization: Basic $(echo -n 'your_access_key:your_secret_key' | base64)"
```

## Service Management

You use this endpoint to create a Tiger Cloud service with one of more of the following addons:

- `time-series`: a Tiger Cloud service optimized for real-time analytics. For time-stamped data like events,
  prices, metrics, sensor readings, or any information that changes over time.
- `ai`: a Tiger Cloud service instance with vector extensions.

To have multiple addons when you create a new service, set `"addons": ["time-series", "ai"]`. To create a
vanilla Postgres instance, set `addons` to an empty list `[]`.

### List All Services

```http
GET /projects/{project_id}/services
```

Retrieve all services within a project.

**Response:** `200 OK`
```json
[
  {
    "service_id": "p7zm9wqqii",
    "project_id": "jz22xtzemv",
    "name": "my-production-db",
    "region_code": "eu-central-1",
    "service_type": "TIMESCALEDB",
    "status": "READY",
    "created": "2024-01-15T10:30:00Z",
    "paused": false,
    "resources": [
      {
        "id": "resource-1",
        "spec": {
          "cpu_millis": 1000,
          "memory_gbs": 4,
          "volume_type": "gp2"
        }
      }
    ],
    "endpoint": {
      "host": "my-service.com",
      "port": 5432
    }
  }
]
```

### Create a Service

```http
POST /projects/{project_id}/services
```

Create a new Tiger Cloud service. This is an asynchronous operation.

**Request Body:**
```json
{
  "name": "test-2",
  "addons": ["time-series"],
  "region_code": "eu-central-1",
  "cpu_millis": 1000,
  "memory_gbs": 4
}
```

**Response:** `202 Accepted`
```json
{
  "service_id": "p7zm9wqqii",
  "project_id": "jz22xtzemv",
  "name": "test-2",
  "region_code": "eu-central-1",
  "service_type": "TIMESCALEDB",
  "created": "2025-09-04T20:46:46.265680278Z",
  "paused": false,
  "status": "READY",
  "resources": [
      {
          "id": "100927",
          "spec": {
              "cpu_millis": 1000,
              "memory_gbs": 4,
              "volume_type": ""
          }
      }
  ],
  "metadata": {
      "environment": "PROD"
  },
  "endpoint": {
      "host": "p7zm8wqqii.jz4qxtzemv.tsdb.cloud.timescale.com",
      "port": 35482
  },
  "initial_password": "oamv8ch9t4ar2j8g"
}
```

**Service Types:**
- `TIMESCALEDB`: a Tiger Cloud service instance optimized for real-time analytics service For time-stamped data like events,
   prices, metrics, sensor readings, or any information that changes over time
- `POSTGRES`: a vanilla Postgres instance
- `VECTOR`: a Tiger Cloud service instance with vector extensions

### Get a Service

```http
GET /projects/{project_id}/services/{service_id}
```

Retrieve details of a specific service.

**Response:** `200 OK`
```json
{
  "service_id": "p7zm9wqqii",
  "project_id": "jz22xtzemv",
  "name": "test-2",
  "region_code": "eu-central-1",
  "service_type": "TIMESCALEDB",
  "created": "2025-09-04T20:46:46.26568Z",
  "paused": false,
  "status": "READY",
  "resources": [
      {
          "id": "100927",
          "spec": {
              "cpu_millis": 1000,
              "memory_gbs": 4,
              "volume_type": ""
          }
      }
  ],
  "metadata": {
      "environment": "DEV"
  },
  "endpoint": {
      "host": "p7zm8wqqii.jz4qxtzemv.tsdb.cloud.timescale.com",
      "port": 35482
  }
}
```

**Service Status:**
- `QUEUED`: Service creation is queued
- `DELETING`: Service is being deleted
- `CONFIGURING`: Service is being configured
- `READY`: Service is ready for use
- `DELETED`: Service has been deleted
- `UNSTABLE`: Service is in an unstable state
- `PAUSING`: Service is being paused
- `PAUSED`: Service is paused
- `RESUMING`: Service is being resumed
- `UPGRADING`: Service is being upgraded
- `OPTIMIZING`: Service is being optimized

### Delete a Service

```http
DELETE /projects/{project_id}/services/{service_id}
```

Delete a specific service. This is an asynchronous operation.

**Response:** `202 Accepted`

### Resize a Service

```http
POST /projects/{project_id}/services/{service_id}/resize
```

Change CPU and memory allocation for a service.

**Request Body:**
```json
{
  "cpu_millis": 2000,
  "memory_gbs": 8
}
```

**Response:** `202 Accepted`

### Update Service Password

```http
POST /projects/{project_id}/services/{service_id}/updatePassword
```

Set a new master password for the service.

**Request Body:**
```json
{
  "password": "a-very-secure-new-password"
}
```

**Response:** `204 No Content`

### Set Service Environment

```http
POST /projects/{project_id}/services/{service_id}/setEnvironment
```

Set the environment type for the service.

**Request Body:**
```json
{
  "environment": "PROD"
}
```

**Environment Values:**
- `PROD`: Production environment
- `DEV`: Development environment

**Response:** `200 OK`
```json
{
    "message": "Environment set successfully"
}
```

### Configure High Availability

```http
POST /projects/{project_id}/services/{service_id}/setHA
```

Change the HA configuration for a service. This is an asynchronous operation.

**Request Body:**
```json
{
  "replica_count": 1
}
```

**Response:** `202 Accepted`

### Connection Pooler Management

#### Enable Connection Pooler

```http
POST /projects/{project_id}/services/{service_id}/enablePooler
```

Activate the connection pooler for a service.

**Response:** `200 OK`
```json
{
  "message": "Connection pooler enabled successfully"
}
```

#### Disable Connection Pooler

```http
POST /projects/{project_id}/services/{service_id}/disablePooler
```

Deactivate the connection pooler for a service.

**Response:** `200 OK`
```json
{
  "message": "Connection pooler disabled successfully"
}
```

### Fork a Service

```http
POST /projects/{project_id}/services/{service_id}/forkService
```

Create a new, independent service by taking a snapshot of an existing one.

**Request Body:**
```json
{
    "name": "fork-test2",
    "region_code": "eu-central-1",
    "cpu_millis": 1000,
    "memory_gbs": 4
}
```

**Response:** `202 Accepted`
```json
{
    "service_id": "otewd3pem2",
    "project_id": "jz22xtzemv",
    "name": "fork-test2",
    "region_code": "eu-central-1",
    "service_type": "TIMESCALEDB",
    "created": "2025-09-04T20:54:09.53380732Z",
    "paused": false,
    "status": "READY",
    "resources": [
        {
            "id": "100929",
            "spec": {
                "cpu_millis": 1000,
                "memory_gbs": 4,
                "volume_type": ""
            }
        }
    ],
    "forked_from": {
        "project_id": "jz22xtzemv",
        "service_id": "p7zm9wqqii",
        "is_standby": false
    },
    "initial_password": "ph33bl5juuri5gem"
}
```

## Read Replica Sets

Manage read replicas for improved read performance.

### List Read Replica Sets

```http
GET /projects/{project_id}/services/{service_id}/replicaSets
```

Retrieve all read replica sets associated with a primary service.

**Response:** `200 OK`
```json
[
  {
    "id": "l5alxb3s2g",
    "name": "replica-set-test2",
    "status": "active",
    "nodes": 1,
    "cpu_millis": 1000,
    "memory_gbs": 4,
    "endpoint": {
        "host": "l5alxb3s2g.jz4qxtzemv.tsdb.cloud.timescale.com",
        "port": 38448
    },
    "connection_pooler": {
        "endpoint": {
            "host": "l5alxb3s2g.jz4qxtzemv.tsdb.cloud.timescale.com",
            "port": 38543
        }
    },
    "metadata": {
        "environment": "DEV"
    }
  }
]
```

**Replica Set Status:**
- `creating`: Replica set is being created
- `active`: Replica set is active and ready
- `resizing`: Replica set is being resized
- `deleting`: Replica set is being deleted
- `error`: Replica set encountered an error

### Create a Read Replica Set

```http
POST /projects/{project_id}/services/{service_id}/replicaSets
```

Create a new read replica set. This is an asynchronous operation.

**Request Body:**
```json
{
  "name": "replica-set-test2",
  "cpu_millis": 1000,
  "memory_gbs": 4,
  "nodes": 1
}
```

**Response:** `202 Accepted`
```json
{
  "id": "dsldm715t2",
  "name": "replica-set-test2",
  "status": "active",
  "nodes": 1,
  "cpu_millis": 1000,
  "memory_gbs": 4
}
```

### Delete a Read Replica Set

```http
DELETE /projects/{project_id}/services/{service_id}/replicaSets/{replica_set_id}
```

Delete a specific read replica set. This is an asynchronous operation.

**Response:** `202 Accepted`

### Resize a Read Replica Set

```http
POST /projects/{project_id}/services/{service_id}/replicaSets/{replica_set_id}/resize
```

Change resource allocation for a read replica set. This operation is async.

**Request Body:**
```json
{
  "cpu_millis": 500,
  "memory_gbs": 2,
  "nodes": 2
}
```

**Response:** `202 Accepted`
```json
{
    "message": "Replica set resize request accepted"
}
```

### Read Replica Set Connection Pooler

#### Enable Replica Set Pooler

```http
POST /projects/{project_id}/services/{service_id}/replicaSets/{replica_set_id}/enablePooler
```

Activate the connection pooler for a read replica set.

**Response:** `200 OK`
```json
{
  "message": "Connection pooler enabled successfully"
}
```

#### Disable Replica Set Pooler

```http
POST /projects/{project_id}/services/{service_id}/replicaSets/{replica_set_id}/disablePooler
```

Deactivate the connection pooler for a read replica set.

**Response:** `200 OK`
```json
{
  "message": "Connection pooler disabled successfully"
}
```

### Set Replica Set Environment

```http
POST /projects/{project_id}/services/{service_id}/replicaSets/{replica_set_id}/setEnvironment
```

Set the environment type for a read replica set.

**Request Body:**
```json
{
  "environment": "PROD"
}
```

**Response:** `200 OK`
```json
{
  "message": "Environment set successfully"
}
```

## VPC Management

Virtual Private Clouds (VPCs) provide network isolation for your TigerData services.

### List All VPCs

```http
GET /projects/{project_id}/vpcs
```

List all Virtual Private Clouds in a project.

**Response:** `200 OK`
```json
[
  {
    "id": "1234567890",
    "name": "my-production-vpc",
    "cidr": "10.0.0.0/16",
    "region_code": "eu-central-1"
  }
]
```

### Create a VPC

```http
POST /projects/{project_id}/vpcs
```

Create a new VPC.

**Request Body:**
```json
{
  "name": "my-production-vpc",
  "cidr": "10.0.0.0/16",
  "region_code": "eu-central-1"
}
```

**Response:** `201 Created`
```json
{
  "id": "1234567890",
  "name": "my-production-vpc",
  "cidr": "10.0.0.0/16",
  "region_code": "eu-central-1"
}
```

### Get a VPC

```http
GET /projects/{project_id}/vpcs/{vpc_id}
```

Retrieve details of a specific VPC.

**Response:** `200 OK`
```json
{
  "id": "1234567890",
  "name": "my-production-vpc",
  "cidr": "10.0.0.0/16",
  "region_code": "eu-central-1"
}
```

### Rename a VPC

```http
POST /projects/{project_id}/vpcs/{vpc_id}/rename
```

Update the name of a specific VPC.

**Request Body:**
```json
{
  "name": "my-renamed-vpc"
}
```

**Response:** `200 OK`
```json
{
  "id": "1234567890",
  "name": "my-renamed-vpc",
  "cidr": "10.0.0.0/16",
  "region_code": "eu-central-1"
}
```

### Delete a VPC

```http
DELETE /projects/{project_id}/vpcs/{vpc_id}
```

Delete a specific VPC.

**Response:** `204 No Content`

## VPC Peering

Manage peering connections between VPCs across different accounts and regions.

### List VPC Peerings

```http
GET /projects/{project_id}/vpcs/{vpc_id}/peerings
```

Retrieve all VPC peering connections for a given VPC.

**Response:** `200 OK`
```json
[
  {
    "id": "1234567890",
    "peer_account_id": "acc-12345",
    "peer_region_code": "eu-central-1",
    "peer_vpc_id": "1234567890",
    "provisioned_id": "1234567890",
    "status": "active",
    "error_message": null
  }
]
```

### Create VPC Peering

```http
POST /projects/{project_id}/vpcs/{vpc_id}/peerings
```

Create a new VPC peering connection.

**Request Body:**
```json
{
  "peer_account_id": "acc-12345",
  "peer_region_code": "eu-central-1",
  "peer_vpc_id": "1234567890"
}
```

**Response:** `201 Created`
```json
{
  "id": "1234567890",
  "peer_account_id": "acc-12345",
  "peer_region_code": "eu-central-1",
  "peer_vpc_id": "1234567890",
  "provisioned_id": "1234567890",
  "status": "pending"
}
```

### Get VPC Peering

```http
GET /projects/{project_id}/vpcs/{vpc_id}/peerings/{peering_id}
```

Retrieve details of a specific VPC peering connection.

### Delete VPC Peering

```http
DELETE /projects/{project_id}/vpcs/{vpc_id}/peerings/{peering_id}
```

Delete a specific VPC peering connection.

**Response:** `204 No Content`

## Service VPC Operations

### Attach Service to VPC

```http
POST /projects/{project_id}/services/{service_id}/attachToVPC
```

Associate a service with a VPC.

**Request Body:**
```json
{
  "vpc_id": "1234567890"
}
```

**Response:** `202 Accepted`

### Detach Service from VPC

```http
POST /projects/{project_id}/services/{service_id}/detachFromVPC
```

Disassociate a service from its VPC.

**Request Body:**
```json
{
  "vpc_id": "1234567890"
}
```

**Response:** `202 Accepted`

## Data Models

### VPC Object
```json
{
  "id": "string",
  "name": "string",
  "cidr": "string",
  "region_code": "string"
}
```

### Service Object
```json
{
  "service_id": "string",
  "project_id": "string",
  "name": "string",
  "region_code": "string",
  "service_type": "TIMESCALEDB|POSTGRES|VECTOR",
  "created": "2024-01-15T10:30:00Z",
  "initial_password": "string",
  "paused": false,
  "status": "READY|CONFIGURING|QUEUED|...",
  "resources": [
    {
      "id": "string",
      "spec": {
        "cpu_millis": 1000,
        "memory_gbs": 4,
        "volume_type": "string"
      }
    }
  ],
  "metadata": {
    "environment": "PROD|DEV"
  },
  "endpoint": {
    "host": "string",
    "port": 5432
  },
  "connection_pooler": {
    "endpoint": {
      "host": "string",
      "port": 5432
    }
  }
}
```

### Peering Object
```json
{
  "id": "string",
  "peer_account_id": "string",
  "peer_region_code": "string",
  "peer_vpc_id": "string",
  "provisioned_id": "string",
  "status": "string",
  "error_message": "string"
}
```

### Read Replica Set Object

```json
{
  "id": "string",
  "name": "string",
  "status": "creating|active|resizing|deleting|error",
  "nodes": 2,
  "cpu_millis": 1000,
  "memory_gbs": 4,
  "metadata": {
    "environment": "PROD|DEV"
  },
  "endpoint": {
    "host": "string",
    "port": 5432
  },
  "connection_pooler": {
    "endpoint": {
      "host": "string",
      "port": 5432
    }
  }
}
```

## Error Handling

Tiger Cloud REST API uses standard HTTP status codes and returns error details in JSON format.

### Error Response Format
```json
{
  "code": "ERROR_CODE",
  "message": "Human-readable error description"
}
```

### Common Error Codes
- `400 Bad Request`: Invalid request parameters or malformed JSON
- `401 Unauthorized`: Missing or invalid authentication credentials
- `403 Forbidden`: Insufficient permissions for the requested operation
- `404 Not Found`: Requested resource does not exist
- `409 Conflict`: Request conflicts with current resource state
- `500 Internal Server Error`: Unexpected server error

### Example Error Response
```json
{
  "code": "INVALID_REQUEST",
  "message": "The service_type field is required"
}
```


===== PAGE: https://docs.tigerdata.com/api/glossary/ =====

# Tiger Data glossary of terms

This glossary defines technical terms, concepts, and terminology used in Tiger Data documentation, database industry, and real-time analytics.

## A

**ACL (Access Control List)**: a table that tells a computer operating system which access rights each user has to a particular system object, such as a file directory or individual file.

**ACID**: a set of properties (atomicity, consistency, isolation, durability) that guarantee database transactions are processed reliably.

**ACID compliance**: a set of database properties—Atomicity, Consistency, Isolation, Durability—ensuring reliable and consistent transactions. Inherited from [Postgres](#postgresql).

**Adaptive query optimization**: dynamic query plan adjustment based on actual execution statistics and data distribution patterns, improving performance over time.

**Aggregate (Continuous Aggregate)**: a materialized, precomputed summary of query results over time-series data, providing faster access to analytics.

**Alerting**: the process of automatically notifying administrators when predefined conditions or thresholds are met in system monitoring.

**Analytics database**: a system optimized for large-scale analytical queries, supporting complex aggregations, time-based queries, and data exploration.

**Anomaly detection**: the identification of abnormal patterns or outliers within time-series datasets, common in observability, IoT, and finance.

**Append-only storage**: a storage pattern where data is only added, never modified in place. Ideal for time-series workloads and audit trails.

**Archival**: the process of moving old or infrequently accessed data to long-term, cost-effective storage solutions.

**Auto-partitioning**: automatic division of a hypertable into chunks based on partitioning dimensions to optimize scalability and performance.

**Availability zone**: an isolated location within a cloud region that provides redundant power, networking, and connectivity.

## B

**B-tree**: a self-balancing tree data structure that maintains sorted data and allows searches, sequential access, insertions, and deletions in logarithmic time.

**Background job**: an automated task that runs in the background without user intervention, typically for maintenance operations like compression or data retention.

**Background worker**: a [Postgres](#postgresql) process that runs background tasks independently of client sessions.

**Batch processing**: handling data in grouped batches rather than as individual real-time events, often used for historical data processing.

**Backfill**: the process of filling in historical data that was missing or needs to be recalculated, often used during migrations or after schema changes.

**Backup**: a copy of data stored separately from the original data to protect against data loss, corruption, or system failure.

**Bloom filter**: a probabilistic data structure that tests set membership with possible false positives but no false negatives. [TimescaleDB](#timescaledb) uses blocked bloom filters to speed up point lookups by eliminating [chunks](#chunk) that don't contain queried values.

**Buffer pool**: memory area where frequently accessed data pages are cached to reduce disk I/O operations.

**BRIN (Block Range Index)**: a [Postgres](#postgresql) index type that stores summaries about ranges of table blocks, useful for large tables with naturally ordered data.

**Bytea**: a [Postgres](#postgresql) data type for storing binary data as a sequence of bytes.

## C

**Cache hit ratio**: the percentage of data requests served from memory cache rather than disk, indicating query performance efficiency.

**Cardinality**: the number of unique values in a dataset or database column.

**Check constraint**: a database constraint that limits the values that can be stored in a column by checking them against a specified condition.

<a id="chunk" href=""></a>

**Chunk**: a horizontal partition of a [hypertable](#hypertable) that contains data for a specific time interval and space partition. See [chunks][use-hypertables-chunks].

**Chunk interval**: the time period covered by each chunk in a hypertable, which affects query performance and storage efficiency.

**Chunk skipping**: a query optimization technique that skips chunks not relevant to the query's time range, dramatically improving performance.

**CIDR (Classless Inter-Domain Routing)**: a method for allocating IP addresses and routing IP packets.

**Client credentials**: authentication tokens used by applications to access services programmatically without user interaction.

**Close**: in financial data, the closing price of a security at the end of a trading period.

**Cloud**: computing services delivered over the internet, including servers, storage, databases, networking, software, analytics, and intelligence.

**Cloud deployment**: the use of public, private, or hybrid cloud infrastructure to host [TimescaleDB](#timescaledb), enabling elastic scalability and managed services.

**Cloud-native**: an approach to building applications that leverage cloud infrastructure, scalability, and services like Kubernetes.

**Cold storage**: a tier of data storage for infrequently accessed data that offers lower costs but higher access times.

**Columnar**: a data storage format that stores data column by column rather than row by row, optimizing for analytical queries.

**Columnstore**: [TimescaleDB](#timescaledb)'s columnar storage engine optimized for analytical workloads and [compression](#compression).

<a id="compression" href=""></a>

**Compression**: the process of reducing data size by encoding information using fewer bits, improving storage efficiency and query performance. See [compression][use-compression].

**Connection pooling**: a technique for managing multiple database connections efficiently, reducing overhead for high-concurrency environments.

**Consensus algorithm**: protocols ensuring distributed systems agree on data state, critical for multi-node database deployments.

**Compression policy**: an automated rule that compresses hypertable chunks after they reach a specified age or size threshold.

**Compression ratio**: the ratio between the original data size and the compressed data size, indicating compression effectiveness.

**Constraint**: a rule enforced by the database to maintain data integrity and consistency.

**Continuous aggregate**: a materialized view that incrementally updates with new data, providing fast access to pre-computed aggregations. See [continuous aggregates][use-continuous-aggregates].

**Counter aggregation**: aggregating monotonic counter data, handling counter resets and extrapolation.

**Cron**: a time-based job scheduler in Unix-like computer operating systems.

**Cross-region backup**: a backup stored in a different geographical region from the primary data for disaster recovery.

## D

**Data lake**: a centralized repository storing structured and unstructured data at scale, often integrated with time-series databases for analytics.

**Data lineage**: the tracking of data flow from source to destination, including transformations, essential for compliance and debugging.

**Data pipeline**: automated workflows for moving, transforming, and loading data between systems, often using tools like Apache Kafka or Apache Airflow.

**Data migration**: the process of moving data from one system, storage type, or format to another. See the [migration guides][migrate].

**Data retention**: the practice of storing data for a specified period before deletion, often governed by compliance requirements or storage optimization. See [data retention][use-data-retention].

**Data rollup**: the process of summarizing detailed historical data into higher-level aggregates, balancing storage needs with query efficiency.

**Data skew**: uneven distribution of data across partitions or nodes, potentially causing performance bottlenecks.

**Data tiering**: a storage management strategy that places data on different storage tiers based on access patterns and performance requirements.

**Data type**: a classification that specifies which type of value a variable can hold, such as integer, string, or boolean.

**Decompress**: the process of restoring compressed data to its original, uncompressed state.

**Delta**: the difference between two values, commonly used in counter aggregations to calculate the change over time.

**DHCP (Dynamic Host Configuration Protocol)**: a network management protocol used to automatically assign IP addresses and other network configuration parameters.

**Dimension**: a partitioning key in a hypertable that determines how data is distributed across chunks.

**Disaster recovery**: the process and procedures for recovering and protecting a business's IT infrastructure in the event of a disaster.

**Double precision**: a floating-point data type that provides more precision than the standard float type.

**Downsample**: the process of reducing the temporal resolution of time-series data by aggregating data points over longer time intervals.

**Downtime**: the period during which a system, service, or application is unavailable or not operational.

**Dual-write and backfill**: a migration approach where new data is written to both the source and target databases simultaneously, followed by backfilling historical data to ensure completeness.

**Dual-write**: a migration pattern where applications write data to both the source and target systems simultaneously.

## E

**Edge computing**: processing data at or near the data source such as IoT devices, rather than solely in centralized servers, reducing latency.

**Edge gateway**: a device that aggregates data from sensors and performs preprocessing before sending data to cloud or centralized databases.

**ELT (Extract, Load, Transform)**: a data pipeline pattern where raw data is loaded first, then transformed within the target system, leveraging database processing power.

**Embedding**: a vector representation of data such as text or images, that captures semantic meaning in a high-dimensional space.

**Error rate**: the percentage of requests or operations that result in errors over a given time period.

**Euclidean distance**: a measure of the straight-line distance between two points in multidimensional space.

**Exactly-once**: a message is delivered and processed precisely once. There is no loss and no duplicates.

**Explain**: a [Postgres](#postgresql) command that shows the execution plan for a query, useful for performance analysis.

**Event sourcing**: an architectural pattern storing all changes as a sequence of events, naturally fitting time-series database capabilities.

**Event-driven architecture**: a design pattern where components react to events such as sensor readings, requiring real-time data pipelines and storage.

**Extension**: a [Postgres](#postgresql) add-on that extends the database's functionality beyond the core features.

## F

**Fact table**: the central table in a star schema containing quantitative measures, often time-series data with foreign keys to dimension tables.

**Failover**: the automatic switching to a backup system, server, or network upon the failure or abnormal termination of the primary system.

**Financial time-series**: high-volume, timestamped datasets like stock market feeds or trade logs, requiring low-latency, scalable databases like [TimescaleDB](#timescaledb).

**Foreign key**: a database constraint that establishes a link between data in two tables by referencing the primary key of another table.

**Fork**: a copy of a database service that shares the same data but can diverge independently through separate writes.

<a id="free-tiger-service" href=""></a>

**Free service**: a free instance of Tiger Cloud with limited resources. You can create up to two free services under any pricing plan. When a free service reaches the resource limit, it converts to the read-only state. You can convert a free service to a [standard one](#standard-tiger-service) under paid pricing plans.

**FTP (File Transfer Protocol)**: a standard network protocol used for transferring files between a client and server on a computer network.

## G

**Gap filling**: a technique for handling missing data points in time-series by interpolation or other methods, often implemented with hyperfunctions.

**GIN (Generalized Inverted Index)**: a [Postgres](#postgresql) index type designed for indexing composite values and supporting fast searches.

**GiST (Generalized Search Tree)**: a [Postgres](#postgresql) index type that provides a framework for implementing custom index types.

**GP-LTTB**: an advanced downsampling algorithm that extends Largest-Triangle-Three-Buckets with Gaussian Process modeling.

**GUC (Grand Unified Configuration)**: [Postgres](#postgresql)'s configuration parameter system that controls various aspects of database behavior.

**GUID (Globally Unique Identifier)**: a unique identifier used in software applications, typically represented as a 128-bit value.

## H

**Hash**: an index type that provides constant-time lookups for equality comparisons but doesn't support range queries.

**High-cardinality**: refers to datasets with a large number of unique values, which can strain storage and indexing in time-series applications.

**Histogram bucket**: a predefined range of metrics organized for statistical analysis, commonly visualized in monitoring tools.

**Hot standby**: a replication configuration where the standby server can serve read-only queries while staying synchronized with the primary.

**High availability**: a system design that ensures an agreed level of operational performance, usually uptime, for a higher than normal period.

**High**: in financial data, the highest price of a security during a specific time period.

**Histogram**: a graphical representation of the distribution of numerical data, showing the frequency of data points in different ranges.

**Historical data**: previously recorded data that provides context and trends for analysis and decision-making.

**HNSW (Hierarchical Navigable Small World)**: a graph-based algorithm for approximate nearest neighbor search in high-dimensional spaces.

**Hot storage**: a tier of data storage for frequently accessed data that provides the fastest access times but at higher cost.

**Hypercore**: [TimescaleDB](#timescaledb)'s hybrid storage engine that seamlessly combines row and column storage for optimal performance. See [Hypercore][use-hypercore].

**Hyperfunction**: an SQL function in [TimescaleDB](#timescaledb) designed for time-series analysis, statistics, and specialized computations. See [Hyperfunctions][use-hyperfunctions].

**HyperLogLog**: a probabilistic data structure used for estimating the cardinality of large datasets with minimal memory usage.

**Hypershift**: a migration tool and strategy for moving data to [TimescaleDB](#timescaledb) with minimal downtime.

<a id="hypertable" href=""></a>

**Hypertable**: [TimescaleDB](#timescaledb)'s core abstraction that automatically partitions time-series data for scalability. See [Hypertables][use-hypertables].

## I

**Idempotency**: the property where repeated operations produce the same result, crucial for reliable data ingestion and processing.

**Ingest rate**: the speed at which new data is written to the system, measured in rows per second. Critical for IoT and observability.

**Inner product**: a mathematical operation that combines two vectors to produce a scalar, used in similarity calculations.

**Insert**: an SQL operation that adds new rows of data to a database table.

**Integer**: a data type that represents whole numbers without decimal points.

**Intercept**: a statistical measure representing the y-intercept in linear regression analysis.

**Internet gateway**: an AWS VPC component that enables communication between instances in a VPC and the internet.

**Interpolation**: a method of estimating unknown values that fall between known data points.

**IP allow list**: a security feature that restricts access to specified IP addresses or ranges.

**Isolation level**: a database transaction property that defines the degree to which operations in one transaction are isolated from those in other concurrent transactions.

## J

**Job**: an automated task scheduled to run at specific intervals or triggered by certain conditions.

**Job execution**: the process of running scheduled background tasks or automated procedures.

**JIT (Just-In-Time) compilation**: [Postgres](#postgresql) feature that compiles frequently executed query parts for improved performance, available in [TimescaleDB](#timescaledb).

**Job history**: a record of past job executions, including their status, duration, and any errors encountered.

**JSON (JavaScript Object Notation)**: a lightweight data interchange format that is easy for humans to read and write.

**JWT (JSON Web Token)**: a compact, URL-safe means of representing claims to be transferred between two parties.

## L

**Latency**: the time delay between a request being made and the response being received.

**Lifecycle policy**: a set of rules that automatically manage data throughout its lifecycle, including retention and deletion.

**Live migration**: a data migration technique that moves data with minimal or zero downtime.

**Load balancer**: a service distributing traffic across servers or database nodes to optimize resource use and avoid single points of failure.

**Log-Structured Merge (LSM) Tree**: a data structure optimized for write-heavy workloads, though [TimescaleDB](#timescaledb) primarily uses B-tree indexes for balanced read/write performance.

**LlamaIndex**: a framework for building applications with large language models, providing tools for data ingestion and querying.

**LOCF (Last Observation Carried Forward)**: a method for handling missing data by using the most recent known value.

**Logical backup**: a backup method that exports data in a human-readable format, allowing for selective restoration.

**Logical replication**: a [Postgres](#postgresql) feature that replicates data changes at the logical level rather than the physical level.

**Logging**: the process of recording events, errors, and system activities for monitoring and troubleshooting purposes.

**Low**: in financial data, the lowest price of a security during a specific time period.

**LTTB (Largest-Triangle-Three-Buckets)**: a downsampling algorithm that preserves the visual characteristics of time-series data.

## M

**Manhattan distance**: a distance metric calculated as the sum of the absolute differences of their coordinates.

**Manual compression**: the process of compressing chunks manually rather than through automated policies.

**Materialization**: the process of computing and storing the results of a query or view for faster access.

**Materialized view**: a database object that stores the result of a query and can be refreshed periodically.

**Memory-optimized query**: a query pattern designed to minimize disk I/O by leveraging available RAM and efficient data structures.

**Metric**: a quantitative measurement used to assess system performance, business outcomes, or operational efficiency.

**MFA (Multi-Factor Authentication)**: a security method that requires two or more verification factors to grant access.

**Migration**: the process of moving data, applications, or systems from one environment to another. See [migration guides][migrate].

**Monitoring**: the continuous observation and measurement of system performance and health.

**Multi-tenancy**: an architecture pattern supporting multiple customers or applications within a single database instance, with proper isolation.

**MQTT (Message Queuing Telemetry Transport)**: a lightweight messaging protocol designed for small sensors and mobile devices.

**MST (Managed Service for TimescaleDB)**: a fully managed [TimescaleDB](#timescaledb) service that handles infrastructure and maintenance tasks.

## N

**NAT Gateway**: a network address translation service that enables instances in a private subnet to connect to the internet.

**Node (database node)**: an individual server within a distributed system, contributing to storage, compute, or replication tasks.

**Normalization**: database design technique organizing data to reduce redundancy, though time-series data often benefits from denormalized structures.

**Not null**: a database constraint that ensures a column cannot contain empty values.

**Numeric**: a [Postgres](#postgresql) data type for storing exact numeric values with user-defined precision.

## O

**OAuth**: an open standard for access delegation commonly used for token-based authentication and authorization.

**Observability**: the ability to measure the internal states of a system by examining its outputs.

**OLAP (Online Analytical Processing)**: systems or workloads focused on large-scale, multidimensional, and complex analytical queries.

**OLTP (Online Transaction Processing)**: high-speed transactional systems optimized for data inserts, updates, and short queries.

**OHLC**: an acronym for Open, High, Low, Close prices, commonly used in financial data analysis.

**OHLCV**: an extension of OHLC that includes Volume data for complete candlestick analysis.

**Open**: in financial data, the opening price of a security at the beginning of a trading period.

**OpenTelemetry**: open standard for collecting, processing, and exporting telemetry data, often stored in time-series databases.

**Optimization**: the process of making systems, queries, or operations more efficient and performant.

## P

**Parallel copy**: a technique for copying large amounts of data using multiple concurrent processes to improve performance.

**Parallel Query Execution**: a [Postgres](#postgresql) feature that uses multiple CPU cores to execute single queries faster, inherited by [TimescaleDB](#timescaledb).

**Partitioning**: the practice of dividing large tables into smaller, more manageable pieces based on certain criteria.

**Percentile**: a statistical measure that indicates the value below which a certain percentage of observations fall.

**Performance**: a measure of how efficiently a system operates, often quantified by metrics like throughput, latency, and resource utilization.

**pg_basebackup**: a [Postgres](#postgresql) utility for taking base backups of a running [Postgres](#postgresql) cluster.

**pg_dump**: a [Postgres](#postgresql) utility for backing up database objects and data in various formats.

**pg_restore**: a [Postgres](#postgresql) utility for restoring databases from backup files created by `pg_dump`.

**pgVector**: a [Postgres](#postgresql) extension that adds vector similarity search capabilities for AI and machine learning applications. See [pgvector][ai-pgvector].

**pgai on Tiger Cloud**: a cloud solution for building search, RAG, and AI agents with [Postgres](#postgresql). Enables calling AI embedding and generation models directly from the database using SQL. See [pgai][ai-pgai].

**pgvectorscale**: a performance enhancement for pgvector featuring StreamingDiskANN indexing, binary quantization compression, and label-based filtering. See [pgvectorscale][ai-pgvectorscale].

**pgvectorizer**: a [TimescaleDB](#timescaledb) tool for automatically vectorizing and indexing data for similarity search.

**Physical backup**: a backup method that copies the actual database files at the storage level.

**PITR (Point-in-Time Recovery)**: the ability to restore a database to a specific moment in time.

**Policy**: an automated rule or procedure that performs maintenance tasks like compression, retention, or refresh operations.

**Predictive maintenance**: the use of time-series data to forecast equipment failure, common in IoT and industrial applications.

<a id="postgresql" href=""></a>

**Postgres**: an open-source object-relational database system known for its reliability, robustness, and performance.

**PostGIS**: a [Postgres](#postgresql) extension that adds support for geographic objects and spatial queries.

**Primary key**: a database constraint that uniquely identifies each row in a table.

**psql**: an interactive terminal-based front-end to [Postgres](#postgresql) that allows users to type queries interactively.

## Q

**QPS (Queries Per Second)**: a measure of database performance indicating how many queries a database can process per second.

**Query**: a request for data or information from a database, typically written in SQL.

**Query performance**: a measure of how efficiently database queries execute, including factors like execution time and resource usage.

**Query planner/optimizer**: a component determining the most efficient strategy for executing SQL queries based on database structure and indexes.

**Query planning**: the database process of determining the most efficient way to execute a query.

## R

**RBAC (Role-Based Access Control)**: a security model that assigns permissions to users based on their roles within an organization.

**Read committed**: an isolation level where transactions can read committed changes made by other transactions.

**Read scaling**: a technique for improving database performance by distributing read queries across multiple database replicas.

**Read uncommitted**: the lowest isolation level where transactions can read uncommitted changes from other transactions.

**Read-only role**: a database role with permissions limited to reading data without modification capabilities.

**Read replica**: a copy of the primary database that serves read-only queries, improving read scalability and geographic distribution.

**Real-time analytics**: the immediate analysis of incoming data streams, crucial for observability, trading platforms, and IoT monitoring.

**Real**: a [Postgres](#postgresql) data type for storing single-precision floating-point numbers.

**Real-time aggregate**: a continuous aggregate that includes both materialized historical data and real-time calculations on recent data.

**Refresh policy**: an automated rule that determines when and how continuous aggregates are updated with new data.

**Region**: a geographical area containing multiple data centers, used in cloud computing for data locality and compliance.

**Repeatable read**: an isolation level that ensures a transaction sees a consistent snapshot of data throughout its execution.

**Replica**: a copy of a database that can be used for read scaling, backup, or disaster recovery purposes.

**Replication**: the process of copying and maintaining data across multiple database instances to ensure availability and durability.

**Response time**: the time it takes for a system to respond to a request, measured from request initiation to response completion.

**REST API**: a web service architecture that uses HTTP methods to enable communication between applications.

**Restore**: the process of recovering data from backups to restore a database to a previous state.

**Restore point**: a snapshot of database state that can be used as a reference point for recovery operations.

**Retention policy**: an automated rule that determines how long data is kept before being deleted from the system.

**Route table**: a set of rules that determine where network traffic is directed within a cloud network.

**RTO (Recovery Time Objective)**: the maximum acceptable time that systems can be down after a failure or disaster.

**RPO (Recovery Point Objective)**: the maximum acceptable amount of data loss measured in time after a failure or disaster.

**Rowstore**: traditional row-oriented data storage where data is stored row by row, optimized for transactional workloads.

## S

**SAML (Security Assertion Markup Language)**: an XML-based standard for exchanging authentication and authorization data between security domains.

**Scheduled job**: an automated task that runs at predetermined times or intervals.

**Schema evolution**: the process of modifying database structure over time while maintaining compatibility with existing applications.

**Schema**: the structure of a database, including tables, columns, relationships, and constraints.

**Security group**: a virtual firewall that controls inbound and outbound traffic for cloud resources.

**Service discovery**: mechanisms allowing applications to dynamically locate services like database endpoints, often used in distributed environments.

**Segmentwise recompression**: a [TimescaleDB](#timescaledb) [compression](#compression) technique that recompresses data segments to improve [compression](#compression) ratios.

**Serializable**: the highest isolation level that ensures transactions appear to run serially even when executed concurrently.

**Service**: see [Tiger Cloud service](#tiger-service).

**Sharding**: horizontal partitioning of data across multiple database instances, distributing load and enabling linear scalability.

**SFTP (SSH File Transfer Protocol)**: a secure version of FTP that encrypts both commands and data during transmission.

**SkipScan**: query optimization for DISTINCT operations that incrementally jumps between ordered values without reading intermediate rows. Uses a Custom Scan node to efficiently traverse ordered indexes, dramatically improving performance over traditional DISTINCT queries.

**Similarity search**: a technique for finding items that are similar to a given query item, often used with vector embeddings.

**SLA (Service Level Agreement)**: a contract that defines the expected level of service between a provider and customer.

**SLI (Service Level Indicator)**: a quantitative measure of some aspect of service quality.

**SLO (Service Level Objective)**: a target value or range for service quality measured by an SLI.

**Slope**: a statistical measure representing the rate of change in linear regression analysis.

**SMTP (Simple Mail Transfer Protocol)**: an internet standard for email transmission across networks.

**Snapshot**: a point-in-time copy of data that can be used for backup and recovery purposes.

**SP-GiST (Space-Partitioned Generalized Search Tree)**: a [Postgres](#postgresql) index type for data structures that naturally partition search spaces.

**Storage optimization**: techniques for reducing storage costs and improving performance through compression, tiering, and efficient data organization.

**Streaming data**: continuous flows of data generated by devices, logs, or sensors, requiring high-ingest, real-time storage solutions.

**SQL (Structured Query Language)**: a programming language designed for managing and querying relational databases.

**SSH (Secure Shell)**: a cryptographic network protocol for secure communication over an unsecured network.

**SSL (Secure Sockets Layer)**: a security protocol that establishes encrypted links between networked computers.

<a id="standard-tiger-service" href=""></a>

**Standard service**: a regular [Tiger Cloud service](#tiger-service) that includes the resources and features according to the pricing plan. You can create standard services under any of the paid plans.

**Streaming replication**: a [Postgres](#postgresql) replication method that continuously sends write-ahead log records to standby servers.

**Synthetic monitoring**: simulated transactions or probes used to test system health, generating time-series metrics for performance analysis.

## T

**Table**: a database object that stores data in rows and columns, similar to a spreadsheet.

**Tablespace**: a [Postgres](#postgresql) storage structure that defines where database objects are physically stored on disk.

**TCP (Transmission Control Protocol)**: a connection-oriented protocol that ensures reliable data transmission between applications.

**TDigest**: a probabilistic data structure for accurate estimation of percentiles in distributed systems.

**Telemetry**: the collection of real-time data from systems or devices for monitoring and analysis.

**Text**: a [Postgres](#postgresql) data type for storing variable-length character strings.

**Throughput**: a measure of system performance indicating the amount of work performed or data processed per unit of time.

**Tiered storage**: a storage strategy that automatically moves data between different storage classes based on access patterns and age.

**Tiger Cloud**: Tiger Data's managed cloud platform that provides [TimescaleDB](#timescaledb) as a fully managed solution with additional features.

**Tiger Lake**: Tiger Data's service for integrating operational databases with data lake architectures.

<a id="tiger-service" href=""></a>

**Tiger Cloud service**: an instance of optimized [Postgres](#postgresql) extended with database engine innovations such as [TimescaleDB](#timescaledb), in a cloud infrastructure that delivers speed without sacrifice. You can create [free services](#free-tiger-service) and [standard services](#standard-tiger-service).

**Time series**: data points indexed and ordered by time, typically representing how values change over time.

**Time-weighted average**: a statistical calculation that gives more weight to values based on the duration they were held.

**Time bucketing**: grouping timestamps into uniform intervals for analysis, commonly used with hyperfunctions.

**Time-series forecasting**: the application of statistical models to time-series data to predict future trends or events.

<a id="timescaledb" href=""></a>

**TimescaleDB**: an open-source [Postgres](#postgresql) extension for real-time analytics that provides scalability and performance optimizations.

**Timestamp**: a data type that stores date and time information without timezone data.

**Timestamptz**: a [Postgres](#postgresql) data type that stores timestamp with timezone information.

**TLS (Transport Layer Security)**: a cryptographic protocol that provides security for communication over networks.

**Tombstone**: marker indicating deleted data in append-only systems, requiring periodic cleanup processes.

**Transaction isolation**: the database property controlling the visibility of uncommitted changes between concurrent transactions.

**TPS (Transactions Per Second)**: a measure of database performance indicating transaction processing capacity.

**Transaction**: a unit of work performed against a database that must be completed entirely or not at all.

**Trigger**: a database procedure that automatically executes in response to certain events on a table or view.

## U

**UDP (User Datagram Protocol)**: a connectionless communication protocol that provides fast but unreliable data transmission.

**Unique**: a database constraint that ensures all values in a column or combination of columns are distinct.

**Uptime**: the amount of time that a system has been operational and available for use.

**Usage-based storage**: a billing model where storage costs are based on actual data stored rather than provisioned capacity.

**UUID (Universally Unique Identifier)**: a 128-bit identifier used to uniquely identify information without central coordination.

## V

**Vacuum**: a [Postgres](#postgresql) maintenance operation that reclaims storage and updates database statistics.

**Varchar**: a variable-length character data type that can store strings up to a specified maximum length.

**Vector operations**: SIMD (Single Instruction, Multiple Data) optimizations for processing arrays of data, improving analytical query performance.

**Vertical scaling (scale up)**: increasing system capacity by adding more power (CPU, RAM) to existing machines, as opposed to horizontal scaling.

**Visualization tool**: a platform or dashboard used to display time-series data in charts, graphs, and alerts for easier monitoring and analysis.

**Vector**: a mathematical object with magnitude and direction, used in machine learning for representing data as numerical arrays.

**VPC (Virtual Private Cloud)**: a virtual network dedicated to your cloud account that provides network isolation.

**VWAP (Volume Weighted Average Price)**: a financial indicator that shows the average price weighted by volume over a specific time period.

## W

**WAL (Write-Ahead Log)**: [Postgres](#postgresql)'s method for ensuring data integrity by writing changes to a log before applying them to data files.

**Warm storage**: a storage tier that balances access speed and cost, suitable for data accessed occasionally.

**Watermark**: a timestamp that tracks the progress of continuous aggregate materialization.

**WebSocket**: a communication protocol that provides full-duplex communication channels over a single TCP connection.

**Window function**: an SQL function that performs calculations across related rows, particularly useful for time-series analytics and trend analysis.

**Workload management**: techniques for prioritizing and scheduling different types of database operations to optimize overall system performance.

## X

**XML (eXtensible Markup Language)**: a markup language that defines rules for encoding documents in a format that is both human-readable and machine-readable.

## Y

**YAML (YAML Ain't Markup Language)**: a human-readable data serialization standard commonly used for configuration files.

## Z

**Zero downtime**: a system design goal where services remain available during maintenance, upgrades, or migrations without interruption.

**Zero-downtime migration**: migration strategies that maintain service availability throughout the transition process, often using techniques like dual-write and gradual cutover.

<!-- Link references -->


===== PAGE: https://docs.tigerdata.com/api/compression/ =====

# Compression



Old API since [TimescaleDB v2.18.0](https://github.com/timescale/timescaledb/releases/tag/2.18.0) Replaced by <a href="https://docs.timescale.com/api/latest/hypercore/">Hypercore</a>.

Compression functionality is included in Hypercore.

Before you set up compression, you need to
[configure the hypertable for compression][configure-compression] and then
[set up a compression policy][add_compression_policy].


Before you set up compression for the first time, read
the compression
[blog post](https://www.tigerdata.com/blog/building-columnar-compression-in-a-row-oriented-database)
and
[documentation](https://docs.tigerdata.com/use-timescale/latest/compression/).


You can also [compress chunks manually][compress_chunk], instead of using an
automated compression policy to compress chunks as they age.

Compressed chunks have the following limitations:

*   `ROW LEVEL SECURITY` is not supported on compressed chunks.
*   Creation of unique constraints on compressed chunks is not supported. You
    can add them by disabling compression on the hypertable and re-enabling
    after constraint creation.

## Restrictions

In general, compressing a hypertable imposes some limitations on the types
of data modifications that you can perform on data inside a compressed chunk.

This table shows changes to the compression feature, added in different versions
of TimescaleDB:

|TimescaleDB version|Supported data modifications on compressed chunks|
|-|-|
|1.5 - 2.0|Data and schema modifications are not supported.|
|2.1 - 2.2|Schema may be modified on compressed hypertables. Data modification not supported.|
|2.3|Schema modifications and basic insert of new data is allowed. Deleting, updating and some advanced insert statements are not supported.|
|2.11|Deleting, updating and advanced insert statements are supported.|

In TimescaleDB 2.1&nbsp;and later, you can modify the schema of hypertables that
have compressed chunks. Specifically, you can add columns to and rename existing
columns of compressed hypertables.

In TimescaleDB v2.3 and later, you can insert data into compressed chunks
and to enable compression policies on distributed hypertables.

In TimescaleDB v2.11 and later, you can update and delete compressed data.
You can also use advanced insert statements like `ON CONFLICT` and `RETURNING`.


===== PAGE: https://docs.tigerdata.com/api/distributed-hypertables/ =====

# Distributed hypertables ( Sunsetted v2.14.x )


[Multi-node support is sunsetted][multi-node-deprecation].

TimescaleDB v2.13 is the last release that includes multi-node support for Postgres
versions 13, 14, and 15.


Distributed hypertables are an extension of regular hypertables, available when
using a [multi-node installation][getting-started-multi-node] of TimescaleDB.
Distributed hypertables provide the ability to store data chunks across multiple
data nodes for better scale-out performance.

Most management APIs used with regular hypertable chunks also work with distributed
hypertables as documented in this section. There are a number of APIs for
specifically dealing with data nodes and a special API for executing SQL commands
on data nodes.


===== PAGE: https://docs.tigerdata.com/self-hosted/install/ =====

# Install self-hosted TimescaleDB



TimescaleDB is an open-source Postgres extension that powers Tiger Cloud. Designed for running real-time analytics on time-series data, it supercharges ingest, query, storage, and analytics performance.

You can install self-hosted TimescaleDB from [source][self-hosted-source], with a [pre-built Docker container][self-hosted-container], or on one of the [supported platforms][platform-support]. This section provides instructions for installing the latest version of self-hosted TimescaleDB.

The following instructions are for development and testing installations. For a production environment, we strongly recommend
that you implement the following, many of which you can achieve using Postgres tooling:

- Incremental backup and database snapshots, with efficient point-in-time recovery.
- High availability replication, ideally with nodes across multiple availability zones.
- Automatic failure detection with fast restarts, for both non-replicated and replicated deployments.
- Asynchronous replicas for scaling reads when needed.
- Connection poolers for scaling client connections.
- Zero-down-time minor version and extension upgrades.
- Forking workflows for major version upgrades and other feature testing.
- Monitoring and observability.

Deploying for production?  With a Tiger Cloud service we tune your database for performance and handle scalability, high
availability, backups, and management, so you can relax.

## Installation

Refer to the installation documentation for detailed setup instructions.

For more details about the latest release, see the [release notes][release-notes] section.


===== PAGE: https://docs.tigerdata.com/self-hosted/configuration/ =====

# Configuration

By default, TimescaleDB uses the default Postgres server configuration
settings. However, in some cases, these settings are not appropriate, especially
if you have larger servers that use more hardware resources such as CPU, memory,
and storage.

*   [Learn about configuration][config] to understand how it works before you
    begin using it.
*   Use the [TimescaleDB tune tool][tstune-conf].
*   Manually edit the `postgresql.conf` [configuration file][postgresql-conf].
*   If you run TimescaleDB in a Docker container, configure
    [within Docker][docker-conf].
*   Find out more about the [data that we collect][telemetry].


===== PAGE: https://docs.tigerdata.com/self-hosted/backup-and-restore/ =====

# Backup and restore



TimescaleDB takes advantage of the reliable backup and restore functionality
provided by Postgres. There are a few different mechanisms you can use to
back up your self-hosted TimescaleDB database:

*   [Logical backup][logical-backups] with pg_dump and pg_restore.
*   [Physical backup][physical-backups] with `pg_basebackup` or another tool.
*   _DEPRECATED_ [Ongoing physical backups][ongoing-physical-backups] using write-ahead log
  (WAL) archiving.

Tiger Cloud is a fully managed service with automatic backup and restore, high
availability with replication, seamless scaling and resizing, and much more. You
can try Tiger Cloud free for thirty days.


===== PAGE: https://docs.tigerdata.com/self-hosted/migration/ =====

# Migrate your Postgres database to self-hosted TimescaleDB

You can migrate your existing Postgres database to self-hosted TimescaleDB.

There are several methods for migrating your data:

*   If the database you want to migrate is smaller than 100&nbsp;GB,
    [migrate your entire database at once][migrate-entire]:
    This method directly transfers all data and schemas, including
    Timescale-specific features. Your hypertables, continuous aggregates, and
    policies are automatically available in the new self-hosted TimescaleDB instance.
*   For databases larger than 100GB,
    [migrate your schema and data separately][migrate-separately]: With this
    method, you migrate your tables one by one for easier failure recovery. If
    migration fails mid-way, you can restart from the failure point rather than
    from the beginning. However, Timescale-specific features won't be
    automatically migrated. Follow the instructions to restore your hypertables,
    continuous aggregates, and policies.
*   If you need to move data from Postgres tables into hypertables within an
    existing self-hosted TimescaleDB instance,
    [migrate within the same database][migrate-same-db]: This method assumes that
    you have TimescaleDB set up in the same database instance as your existing table.
*   If you have data in an InfluxDB database,
    [migrate using Outflux][outflux]:
    Outflux pipes exported data directly to your self-hosted TimescaleDB instance, and manages schema
    discovery, validation, and creation. Outflux works with earlier versions of
    InfluxDB. It does not work with InfluxDB version 2 and later.

## Choose a migration method

Which method you choose depends on your database size, network upload and
download speeds, existing continuous aggregates, and tolerance for failure
recovery.


If you are migrating from an Amazon RDS service, Amazon charges for the amount
of data transferred out of the service. You could be charged by Amazon for all
data egressed, even if the migration fails.


If your database is smaller than 100&nbsp;GB, choose to migrate your entire
database at once. You can also migrate larger databases using this method, but
the copying process must keep running, potentially over days or weeks. If the
copy is interrupted, the process needs to be restarted. If you think an
interruption in the copy is possible, choose to migrate your schema and data
separately instead.


Migrating your schema and data separately does not retain continuous aggregates
calculated using already-deleted data. For example, if you delete raw data after
a month but retain downsampled data in a continuous aggregate for a year, the
continuous aggregate loses any data older than a month upon migration. If you
must keep continuous aggregates calculated using deleted data, migrate your
entire database at once regardless of database size.


If you aren't sure which method to use, try copying the entire database at once
to estimate the time required. If the time estimate is very long, stop the
migration and switch to the other method.

## Migrate an active database

If your database is actively ingesting data, take precautions to ensure that
your self-hosted TimescaleDB instance contains the data that is ingested while the migration
is happening. Begin by running ingest in parallel on the source and target
databases. This ensures that the newest data is written to both databases. Then
backfill your data with one of the two migration methods.


===== PAGE: https://docs.tigerdata.com/self-hosted/manage-storage/ =====

# Manage storage using tablespaces



If you are running TimescaleDB on your own hardware, you can save storage
by moving chunks between tablespaces. By moving older chunks to cheaper, slower
storage, you can save on storage costs while still using faster, more expensive
storage for frequently accessed data. Moving infrequently accessed chunks can
also improve performance, because it isolates historical data from the continual
read-and-write workload of more recent data.



Using tablespaces is one way to manage data storage costs with TimescaleDB. You
can also use [compression](https://docs.tigerdata.com/use-timescale/latest/compression) and
[data retention](https://docs.tigerdata.com/use-timescale/latest/data-retention) to reduce
your storage requirements.



Tiger Cloud is a fully managed service with automatic backup and restore, high
availability with replication, seamless scaling and resizing, and much more. You
can try Tiger Cloud free for thirty days.

## Move data

To move chunks to a new tablespace, you first need to create the new tablespace
and set the storage mount point. You can then use the
[`move_chunk`][api-move-chunk] API call to move individual chunks from the
default tablespace to the new tablespace. The `move_chunk` command also allows
you to move indexes belonging to those chunks to an appropriate tablespace.

Additionally, `move_chunk` allows you reorder the chunk during the migration.
This can be used to make your queries faster, and works in a similar way to the
[`reorder_chunk` command][api-reorder-chunk].



You must be logged in as a super user, such as the `postgres` user, to use the
`move_chunk()` API call.



### Moving data

1.  Create a new tablespace. In this example, the tablespace is called
    `history`, it is owned by the `postgres` super user, and the mount point is
    `/mnt/history`:

    ```sql
    CREATE TABLESPACE history
    OWNER postgres
    LOCATION '/mnt/history';
    ```

1.  List chunks that you want to move. In this example, chunks that contain data
    that is older than two days:

    ```sql
    SELECT show_chunks('conditions', older_than => INTERVAL '2 days');
    ```

1.  Move a chunk and its index to the new tablespace. You can also reorder the
    data in this step. In this example, the chunk called
    `_timescaledb_internal._hyper_1_4_chunk` is moved to the `history`
    tablespace, and is reordered based on its time index:

    ```sql
    SELECT move_chunk(
      chunk => '_timescaledb_internal._hyper_1_4_chunk',
      destination_tablespace => 'history',
      index_destination_tablespace => 'history',
      reorder_index => '_timescaledb_internal._hyper_1_4_chunk_netdata_time_idx',
      verbose => TRUE
    );
    ```

1.  You can verify that the chunk now resides in the correct tablespace by
    querying `pg_tables` to list all of the chunks on the tablespace:

    ```sql
    SELECT tablename from pg_tables
      WHERE tablespace = 'history' and tablename like '_hyper_%_%_chunk';
    ```

    You can also verify that the index is in the correct location:

    ```sql
    SELECT indexname FROM pg_indexes WHERE tablespace = 'history';
    ```

## Move data in bulk

To move several chunks at once, select the chunks you want to move by using
`FROM show_chunks(...)`. For example, to move chunks containing data between 1
and 3 weeks old, in a hypertable named `example`:

```sql
SELECT move_chunk(
  chunk => i,
  destination_tablespace => '')
FROM show_chunks('example', now() - INTERVAL '1 week', now() - INTERVAL '3 weeks') i;
```

## Examples

After moving a chunk to a slower tablespace, you can move it back to the
default, faster tablespace:

```sql
SELECT move_chunk(
  chunk => '_timescaledb_internal._hyper_1_4_chunk',
  destination_tablespace => 'pg_default',
  index_destination_tablespace => 'pg_default',
  reorder_index => '_timescaledb_internal._hyper_1_4_chunk_netdata_time_idx'
);
```

You can move a data chunk to the slower tablespace, but keep the chunk's indexes
on the default, faster tablespace:

```sql
SELECT move_chunk(
  chunk => '_timescaledb_internal._hyper_1_4_chunk',
  destination_tablespace => 'history',
  index_destination_tablespace => 'pg_default',
  reorder_index => '_timescaledb_internal._hyper_1_4_chunk_netdata_time_idx'
);
```

You can also keep the data in `pg_default` but move the index to `history`.
Alternatively, you can set up a third tablespace called `history_indexes`,
and move the data to `history` and the indexes to `history_indexes`.

In TimescaleDB v2.0 and later, you can use `move_chunk` with the job scheduler
framework. For more information, see the [jobs section][jobs].


===== PAGE: https://docs.tigerdata.com/self-hosted/replication-and-ha/ =====

# High availability



Postgres relies on replication for high availability, failover, and balancing
read loads across multiple nodes. Replication ensures that data written to the
primary Postgres database is mirrored on one or more nodes. By virtue of
having multiple nodes with an exact copy of the primary database available, the
primary database can be replaced with a replica node in the event of a failure
or outage on the primary server. Replica nodes can also be used as read only
databases, also called read replicas, allowing reads to be horizontally
scaled by spreading the read query volume across multiple nodes.

*   [Learn about high availability][about-ha] to understand how it works
    before you begin using it.
*   [Configure replication][replication-enable].

Tiger Cloud is a fully managed service with automatic backup and restore, high
availability with replication, seamless scaling and resizing, and much more. You
can try Tiger Cloud free for thirty days.


===== PAGE: https://docs.tigerdata.com/self-hosted/tooling/ =====

# Additional tooling

Get the most from TimescaleDB with open source tools that help you perform
common tasks.

*   Automatically configure your TimescaleDB instance with
    [`timescaledb-tune`][tstune]
*   Install [TimescaleDB Toolkit][tstoolkit] to access more hyperfunctions and
    function pipelines


===== PAGE: https://docs.tigerdata.com/self-hosted/upgrades/ =====

# Upgrade TimescaleDB



A major upgrade is when you update from TimescaleDB `X.<minor version>` to `Y.<minor version>`.
A minor upgrade is when you update from TimescaleDB `<major version>.x`, to TimescaleDB `<major version>.y`.
You upgrade your self-hosted TimescaleDB installation in-place.

Tiger Cloud is a fully managed service with automatic backup and restore, high
availability with replication, seamless scaling and resizing, and much more. You
can try Tiger Cloud free for thirty days.

This section shows you how to:

* Upgrade self-hosted TimescaleDB to a new [minor version][upgrade-minor].
* Upgrade self-hosted TimescaleDB to a new [major version][upgrade-major].
* Upgrade self-hosted TimescaleDB running in a [Docker container][upgrade-docker] to a new minor version.
* Upgrade [Postgres][upgrade-pg] to a new version.
* Downgrade self-hosted TimescaleDB to the [previous minor version][downgrade].


===== PAGE: https://docs.tigerdata.com/self-hosted/uninstall/ =====

# Uninstall TimescaleDB

If you want to uninstall TimescaleDB because it does not meet your requirements,
you can uninstall it without having to uninstall Postgres.

*   [Learn how to uninstall][uninstall-timescaledb] TimescaleDB in macOS


===== PAGE: https://docs.tigerdata.com/self-hosted/multinode-timescaledb/ =====

# Multi-node


[Multi-node support is sunsetted][multi-node-deprecation].

TimescaleDB v2.13 is the last release that includes multi-node support for Postgres
versions 13, 14, and 15.


If you have a larger workload, you might need more than one TimescaleDB
instance. TimescaleDB multi-node allows you to run and manage multiple instances,
giving you faster data ingest, and more responsive and efficient queries.

*   [Learn about multi-node][about-multi-node] to understand how it works
    before you begin using it.
*   Set up [multi-node][setup-selfhosted] in a self-hosted environment.
*   Set up [authentication][multi-node-auth] for your cluster
*   [Configure][multi-node-config] your cluster
*   [Administer][multi-node-administration] your cluster
*   [Grow or shrink][multi-node-grow-shrink] your cluster
*   Set up [high availability][multi-node-ha] (HA) for your cluster
*   [Maintain][multi-node-maintenance] your multi-node environment


===== PAGE: https://docs.tigerdata.com/self-hosted/distributed-hypertables/ =====

# Distributed hypertables


[Multi-node support is sunsetted][multi-node-deprecation].

TimescaleDB v2.13 is the last release that includes multi-node support for Postgres
versions 13, 14, and 15.


Distributed hypertables are hypertables that span multiple nodes. With
distributed hypertables, you can scale your data storage across multiple
machines and benefit from parallelized processing for some queries.

Many features of distributed hypertables work the same way as standard
hypertables. To learn how hypertables work in general, see the
[hypertables][hypertables] section.

In this section:

*   [Learn about distributed hypertables][about-distributed-hypertables] for
    multi-node databases
*   [Create a distributed hypertable][create]
*   [Insert data][insert] into distributed hypertables
*   [Query data][query] in distributed hypertables
*   [Alter and drop][alter-drop] distributed hypertables
*   [Create foreign keys][foreign-keys] on distributed hypertables
*   [Set triggers][triggers] on distributed hypertables


===== PAGE: https://docs.tigerdata.com/mst/about-mst/ =====

# About Managed Service for TimescaleDB



Managed Service for TimescaleDB (MST) is [TimescaleDB ](https://github.com/timescale/timescaledb) hosted on Azure and GCP.
MST is offered in partnership with Aiven.

Tiger Cloud is a high-performance developer focused cloud that provides Postgres services enhanced
with our blazing fast vector search. You can securely integrate Tiger Cloud with your AWS, GCS or Azure
infrastructure. [Create a Tiger Cloud service][timescale-service] and try for free.

If you need to run TimescaleDB on GCP or Azure, you're in the right place — keep reading.

Your Managed Service for TimescaleDB account has three main components:
projects, services, and databases.

## Projects

When you [sign up for Managed Service for TimescaleDB][mst-signup], an empty project is
created for you automatically. Projects are the highest organization level, and
they contain all your services and databases. You can use projects to organize
groups of services. Each project can also have its own billing settings.

To create a new project: In [MST Console][mst-login], click `Projects` > `Create project`.

<img class="main-content__illustration"
src="https://assets.timescale.com/docs/images/mst/create-project.png"
alt="MST projects"/>

## services

Each project contains one or more services. You can have multiple services under
each project, and each service corresponds to a cloud service provider tier. You
can access all your services from the `Services` tab within your projects.

<img class="main-content__illustration"
src="https://assets.timescale.com/docs/images/mst/services.png"
alt="MST services list"/>

For more information about getting your first service up and running, see the
[Managed Service for TimescaleDB installation section][mst-install].



When you have created, and named, a new Managed Service for TimescaleDB service,
you cannot rename it. If you need to have your service running under a different
name, you need to create a new service, and manually migrate the data. For more
information about migrating data, see
[migrating your data](https://docs.tigerdata.com/mst/latest/migrate-to-mst/).


For information about billing on Managed Service for TimescaleDB, see the
[billing section][mst-billing].

## Databases

Each service can contain one or more databases. To view existing databases, or
to create a new database, select a service in the services list,
click `Databases`, then click `Create database`.

<img class="main-content__illustration"
src="https://assets.timescale.com/docs/images/mst/create-database.png"
alt="MST databases list"/>

## Service level agreement

Managed Service for TimescaleDB is provided through a partnership with Aiven.
This provides you with a service commitment to deliver 99.99% availability. For
more information, see the
[Aiven Service Level Agreement policy][aiven-sla].

## Service configuration plans

When you create a new service, you need to select a configuration plan. The plan
determines the number of VMs the service runs in, the high availability
configuration, the number of CPU cores, and size of RAM and storage volumes.

The plans are:

*   Basic Plans: include 2 days of backups and automatic backup and restore if
    your instance fails.
*   Dev Plans: include 1 day of backups and automatic backup and restore if your
    instance fails.
*   Pro Plans: include 3 days of backups and automatic failover to a hot standby
    if your instance fails.

The Basic and Dev plans are serviced by a single virtual machine (VM) node. This
means that if the node fails, the service is unavailable until a new VM is
built. This can result in data loss, if some of the latest changes to the data
weren't backed up before the failure. Sometimes, it can also take a long time to
return the service back to normal operation, because a new VM needs to be
created and restored from backups before the service can resume. The time to
recover depends on the amount of data you have to restore.

The Pro plans are much more resilient to failures. A single node failure causes
no data loss, and the possible downtime is minimal. If an acting TimescaleDB
master node fails, an up-to-date replica node is automatically promoted to
become the new master. This means there is only a small outage while
applications reconnect to the database and access the new master.

You can upgrade your plan while the service is running. The service is
reconfigured to run on larger VMs in the background and when the reconfiguration
is complete, the DNS names are pointed to the new hosts. This can cause a short
disruption to your service while DNS changes are propagated.

Within each configuration plan option, there are several plan types available:

*   `IO-Optimized` and `Compute-Optimized` These configurations are optimized
    for input/output (I/O) performance, using SSD storage media.
*   `Storage-Optimized`: These configurations usually have larger amounts of
    overall storage, using HDD storage media.
*   `Dev-Only`: These configurations are typically smaller footprints, and lower
    cost, designed for development and testing scenarios.

<img class="main-content__illustration"
width={1375} height={944}
src="https://assets.timescale.com/docs/images/mst/service-plans.png"
alt="MST selecting a service configuration plan"/>

## High availability

Most minor failures are handled automatically without making any changes to your
service deployment. This includes failures such as service process crashes, or a
temporary loss of network access. The service automatically restores normal
operation when the crashed process restarts automatically or when the network
access is restored.

However, more severe failure modes, such as losing a single node entirely,
require more drastic recovery measures. Losing an entire node or a virtual
machine could happen for example due to hardware failure or a severe software
failure.

A failing node is automatically detected by the MST monitoring infrastructure.
Either the node starts reporting that its own self-diagnostics is reporting
problems or the node stops communicating entirely. The monitoring infrastructure
automatically schedules a new replacement node to be created when this happens.


In case of database failover, the service URL of your service remains the same.
Only the IP address changes to point at the new master node.


Managed Service for TimescaleDB availability features differ based on the service
plan:

*   Basic and Dev plans: These are single-node plans. Basic plans include a
    two-day backup history, and Dev plans include a one-day backup history.
*   Pro plans: These are two-node plans with a master and a standby for higher
    availability, and three-day backup histories.

### Single node

In the Basic and Dev plans, if you lose the only node from the service, it
immediately starts the automatic process of creating a new replacement node. The
new node starts up, restores its state from the latest available backup, and
resumes the service. Because there was just a single node providing the service,
the service is unavailable for the duration of the restore operation. Also, any
writes made since the backup of the latest write-ahead log (WAL) file is lost.
Typically this time window is limited to either five minutes, or one WAL file.

### Highly available nodes

In Pro plans, if a Postgres standby fails, the master node keeps running
normally and provides normal service level to the client applications. When the
new replacement standby node is ready and synchronized with the master, it
starts replicating the master in real time and normal operation resumes.

If the Postgres master fails, the combined information from the MST monitoring
infrastructure and the standby node is used to make a failover decision. On the
nodes, the open source monitoring daemon `PGLookout`, in combination with the
information from the MST system infrastructure, reports the failover. If the
master node is down completely, the standby node promotes itself as the new
master node and immediately starts serving clients. A new replacement node is
automatically scheduled and becomes the new standby node.

If both master and standby nodes fail at the same time, two new nodes are
automatically scheduled for creation and become the new master and standby
nodes respectively. The master node restores itself from the latest available
backup, which means that there can be some degree of data loss involved. For example,
any writes made since the backup of the latest write-ahead log (WAL) file can be
lost.

The amount of time it takes to replace a failed node depends mainly on the cloud
region and the amount of data that needs to be restored. However, in the case of
services with two-node Pro plans, the surviving node keeps serving clients even
during the recreation of the other node. This process is entirely automatic and requires
no manual intervention.

For backups and restoration, Managed Service for TimescaleDB uses the
open source backup daemon `PGHoard` that MST maintains. It makes real-time
copies of write-ahead log (WAL) files to an object store in a compressed and
encrypted format.

## Connection limits

Managed Service for TimescaleDB limits the maximum number of connections to each
service. The maximum number of allowed connections depends on your service plan.
To see the current connection limit for your service, navigate to the service
`Overview` tab and locate the `Connection Limit` section.

If you have a lot of clients or client threads connecting to your database, use
connection pooling to limit the number of connections. For more information
about connection pooling, see the
[connection pooling section][connection-pooling].


If you have a high number of connections to your database, your service might
run more slowly, and could run out of memory. Remain aware of how many open
connections your have to your database at any given time.


## Service termination protection

You can protect your services from accidentally being terminated, by enabling
service termination protection. When termination protection is enabled, you
cannot power down the service from the web console, the REST API, or with a
command-line client. To power down a protected service, you need to turn off
termination protection first. Termination protection does not interrupt service
migrations or upgrades.

To enable service termination protection, navigate to the service `Overview`
tab. Locate the `Termination protection` section, and toggle to enable
protection.


If you run out of free sign-up credit, and have not entered a valid credit card
for payment, your service is powered down, even if you have enabled termination
protection.


## Idle connections

Managed Service for TimescaleDB uses the default keep alive settings for TCP
connections. The default settings are:

*   `tcp_keepalives_idle`: 7200
*   `tcp_keepalive_count`: 9
*   `tcp_keepalives_interval`: 75

If you have long idle database connection sessions, you might need to adjust
these settings to ensure that your TCP connection remains stable. If you
experience a broken TCP connection, when you reconnect make sure that your
client resolves the DNS address correctly, as the underlying address changes
during automatic failover.

For more information about adjusting keep alive settings, see the
[Postgres documentation][pg-keepalive].

## Long running queries

Managed Service for TimescaleDB does not cancel database queries. If you
have created a query that is taking a very long time, or that has hung, it could
lock resources on your service, and could prevent database administration tasks
from being performed.

You can find out if you have any long-running queries by navigating to the
service `Current Queries` tab. You can also cancel long running queries from
this tab.

Alternatively, you can use your connection client to view running queries with
this command:

```sql
SELECT * FROM pg_stat_activity
    WHERE state <> 'idle';
```

Cancel long-running queries using this command, with the PID of the query you
want to cancel:

```sql
SELECT pg_terminate_backend(<PID>);
```

If you want to automatically cancel any query that runs over a specified length
of time, you can use this command:

```sql
SET statement_timeout = <milliseconds>
```


===== PAGE: https://docs.tigerdata.com/mst/installation-mst/ =====

# Get started with Managed Service for TimescaleDB



Managed Service for TimescaleDB (MST) is [TimescaleDB ](https://github.com/timescale/timescaledb) hosted on Azure and GCP.
MST is offered in partnership with Aiven.

Tiger Cloud is a high-performance developer focused cloud that provides Postgres services enhanced
with our blazing fast vector search. You can securely integrate Tiger Cloud with your AWS, GCS or Azure
infrastructure. [Create a Tiger Cloud service][timescale-service] and try for free.

If you need to run TimescaleDB on GCP or Azure, you're in the right place — keep reading.

## Create your first service

A service in Managed Service for TimescaleDB is a cloud instance on your chosen
cloud provider, which you can install your database on.

### Creating your first service

1.  [Sign in][mst-login] to your MST Console.
1.  Click `Create service` and choose `TimescaleDB`, and update your preferences:

    <img class="main-content__illustration"
    src="https://assets.timescale.com/docs/images/mst/new-service.png"
    alt="Create a new service in the Managed Service for TimescaleDB portal"/>

    *   In the `Select Your Cloud Service Provider` field, click your
        preferred provider.
    *   In the `Select Your Cloud Service Region` field, click your preferred
        server location. This is often the server that's physically closest
        to you.
    *   In the `Select Your Service Plan` field, click your preferred plan,
        based on the hardware configuration you require. If you are in your
        trial period, and just want to try the service out, or develop a proof
        of concept, we recommend the `Dev` plan, because it is the most
        cost-effective during your trial period.
1.  In the information bar on the right of the screen, review the settings you
    have selected for your service, and click `Create Service`. The service
    takes a few minutes to provision.

## Connect to your service from the command prompt

When you have a service up and running, you can connect to it from your local
system using the `psql` command-line utility. This is the same tool you might
have used to connect to Postgres before, but if you haven't installed it yet,
check out the [installing psql][install-psql] section.

### Connecting to your service from the command prompt

1.  [Sign in][mst-login] to your MST Console.
1.  In the `Services` tab, find the service you want to connect to, and check
    it is marked as `Running`.
1.  Click the name of the service you want to connect to see the connection
    information. Take a note of the `host`, `port`, and `password`.
1.  On your local system, at the command prompt, connect to the service, using
    your own service details:

    ```bash
    psql -x "postgres://tsdbadmin:<PASSWORD>@<HOSTNAME>:<PORT>/defaultdb?sslmode=require"
    ```

    If your connection is successful, you'll see a message like this, followed
    by the `psql` prompt:

    ```bash
    psql (13.3, server 13.4)
    SSL connection (protocol: TLSv1.3, cipher: TLS_AES_256_GCM_SHA384, bits: 256, compression: off)
    Type "help" for help.
    defaultdb=>
    ```

## Check that you have the TimescaleDB extension

TimescaleDB is provided as an extension to your Postgres database, and it is
enabled by default when you create a new service on Managed Service for TimescaleDB You can check that the TimescaleDB extension is installed by using
the `\dx` command at the `psql` prompt. It looks like this:

```sql
defaultdb=> \dx

List of installed extensions
-[ RECORD 1 ]------------------------------------------------------------------
Name        | plpgsql
Version     | 1.0
Schema      | pg_catalog
Description | PL/pgSQL procedural language
-[ RECORD 2 ]------------------------------------------------------------------
Name        | timescaledb
Version     | 2.5.1
Schema      | public
Description | Enables scalable inserts and complex queries for time-series data

defaultdb=>
```

## Install and update TimescaleDB Toolkit

Run this command on each database you want to use the Toolkit with:

```sql
CREATE EXTENSION timescaledb_toolkit;
```

Update an installed version of the Toolkit using this command:

```sql
ALTER EXTENSION timescaledb_toolkit UPDATE;
```

## Where to next

Now that you have your first service up and running, you can check out the
[Managed Service for TimescaleDB][mst-docs] section in the documentation, and
find out what you can do with it.

If you want to work through some tutorials to help you get up and running with
TimescaleDB and time-series data, check out the [tutorials][tutorials] section.

You can always [contact us][contact] if you need help working something out, or
if you want to have a chat.


===== PAGE: https://docs.tigerdata.com/mst/ingest-data/ =====

# Ingest data

There are several different ways of ingesting your data into Managed Service for TimescaleDB. This section contains instructions to:

*   Bulk upload [from a `.csv` file](#bulk-upload-from-csv-files)
*   Insert data
    [directly using a client driver](#insert-data-directly-using-a-client-driver),
    such as JDBC, ODBC, or Node.js
*   Insert data
    [directly using a message queue](#insert-data-directly-using-a-message-queue),
    such as Kafka

Before you begin, make sure you have
[created your service][create-managed-service],
and can connect to it using `psql`.

## Preparing your new database

1.  Use `psql` to connect to your service.

    ```sql
    psql -h <HOSTNAME> -p <PORT> -U <USERNAME> -W -d <DATABASE_NAME>
    ```

    You retrieve the service URL,
    port, and login credentials from the service overview in the [MST dashboard][mst-login].

1.  Create a new database for your data. In this example, the new database is
    called `new_db`:

    ```sql
    CREATE DATABASE new_db;
    \c new_db;
    ```

1.  Create a new SQL table in your database. The columns you create for the
    table must match the columns in your source data. In this example, the table
    is storing weather condition data, and has columns for the timestamp,
    location, and temperature:

    ```sql
    CREATE TABLE conditions (
      time        TIMESTAMPTZ         NOT NULL,
      location    text                NOT NULL,
      temperature DOUBLE PRECISION    NULL
    );
    ```

1.  Load the `timescaledb` Postgres extension:

    ```sql
    CREATE EXTENSION timescaledb;
    \dx
    ```

1.  Convert the SQL table into a hypertable:

    ```sql
    SELECT create_hypertable('conditions', by_range('time'));
    ```


	The `by_range` dimension builder is an addition to TimescaleDB 2.13.


When you have successfully set up your new database, you can ingest data using
one of these methods.

## Bulk upload from CSV files

If you have a dataset stored in a `.csv` file, you can import it into an empty
hypertable. You need to begin by creating the new table, before you
import the data.


Before you begin, make sure you have
[prepared your new database](#procedure-preparing-your-new-database).


### Bulk uploading from a CSV file

1.  Insert data into the new hypertable using the `timescaledb-parallel-copy`
    tool. You should already have the tool installed, but you can install it
    manually from [our GitHub repository][github-parallel-copy] if you need to.
    In this example, we are inserting the data using four workers:

    ```sql
    timescaledb-parallel-copy \
    --connection '<service_url>' \
    --table conditions \
    --file ~/Downloads/example.csv \
    --workers 4 \
    --copy-options "CSV" \
    --skip-header
    ```

    We recommend that you set the number of workers lower than the number of
    available CPU cores on your client machine or server, to prevent the workers
    having to compete for resources. This helps your ingest go faster.
1.  *OPTIONAL:* If you don't want to use the `timescaledb-parallel-copy` tool,
    or if you have a very small dataset, you can use the Postgres `COPY`
    command instead:

    ```sql
    psql '<service_url>/new_db?sslmode=require' -c "\copy conditions FROM <example.csv> WITH (FORMAT CSV, HEADER)"
    ```

## Insert data directly using a client driver

You can use a client driver such as JDBC, Python, or Node.js, to insert data
directly into your new database.

See the [Postgres instructions][postgres-odbc] for using the ODBC driver.

See the [Code Quick Starts][code-qs] for using various languages, including Python and node.js.

## Insert data directly using a message queue

If you have data stored in a message queue, you can import it into your
service. This section provides instructions on using the Kafka
Connect Postgres connector.

This connector deploys Postgres change events from Kafka Connect to a runtime
service. It monitors one or more schemas in a service, and writes all
change events to Kafka topics, which can then be independently consumed by one
or more clients. Kafka Connect can be distributed to provide fault tolerance,
which ensures the connectors are running and continually keeping up with changes
in the database.

You can also use the Postgres connector as a library without Kafka or Kafka
Connect. This allows applications and services to directly connect to
MST and obtain the ordered change events. In this environment, the
application must record the progress of the connector so that when it is
restarted, the connect can continue where it left off. This approach can be
useful for less critical use cases. However, for production use cases, we
recommend that you use the connector with Kafka and Kafka Connect.

See [these instructions][gh-kafkaconnector] for using the Kafka connector.


===== PAGE: https://docs.tigerdata.com/mst/user-management/ =====

# User management

You can add new users, and manage existing users, in MST Console. New users can be added to an entire project, or a single
service.

## Project members

You can invite new users to join your project as project members. There are
several roles available for project members:

|Role|Invite more users|Modify billing information|Manage existing services|Start and stop services|View service information|
|-|-|-|-|-|-|
|Admin|✅|✅|✅|✅|✅|
|Operator|❌|❌|✅|✅|✅|
|Developer|✅|❌|✅|❌|✅|
|Read-only|❌|❌|❌|❌|✅|

Users who can manage existing services can create databases and connect to them,
on a service that already exists. To create a new service, users need the start
and stop services permission.

### Adding project members

1.  [Sign in][mst-login] to your MST Console.
1.  Check that you are in the project that you want to change the members for,
    and click `Members`.
1.  In the `Project members` page, type the email address of the member you want
    to add, and select a role for the member.
1.  Click `Send invitation`.
1.  The new user is sent an email inviting them to the project, and the invite
    shows in the `Pending invitations` list. You can click `Withdraw invitation`
    to remove an invitation before it has been accepted.
1.  When they accept the invitation, the user details show in the `Members`
    list. You can edit a member role by selecting a new role in the list. You
    can delete a member by clicking the delete icon in the list.

## Service users

By default, when you create a new service, a new `tsdbadmin` user is created.
This is the user that you use to connect to your new service.

The `tsdbadmin` user is the owner of the database, but is not a superuser. To
access features requiring a superuser, log in as the `postgres` user instead.

The `tsdbadmin` user for Managed Service for TimescaleDBs can:

*   Create a database
*   Create a role
*   Perform replication
*   Bypass row level security (RLS)

This allows you to use the `tsdbadmin` user to create another user with any
other roles. For a complete list of roles available, see the
[Postgres role attributes documentation][pg-roles-doc].



Your service must be running before you can manage users.



### Adding service users

1.  [Sign in][mst-login] to MST Console. By
    default, you start in the `Services` view, showing any services you
    currently have in your project.
1.  Click the name of the service that you want to add users to.
1.  Select `Users`, then click `Add service user`:

    <img class="main-content__illustration"
    src="https://assets.timescale.com/docs/images/mst/create-service-user.png"
    alt="Add a new MST service user"/>

1.  In the `Username` field, type a name for your user. If you want to allow
    the user to be replicated, toggle `Allow replication`. Click
    `Add service user` to save the user.
1.  The new user shows in the `Username` list.

    To view the password, click the eye icon. Use the options in the list to change
    the replication setting and password, or delete the user.

## Multi-factor user authentication

You can use multi-factor authentication (MFA) to log in to MST Console. This requires an authentication code, provided by the
Google Authenticator app on your mobile device.

You can see which authentication method is in use by each member of your Managed Service for TimescaleDB project. From the dashboard, navigate to the `Members`
section. Each member is listed in the table with an authentication method of
either `Password` or `Two-Factor`.

Before you begin, install the Google Authenticator app on your mobile device.
For more information, and installation instructions, see
[the Google Authenticator documentation][install-google-authenticator].

### Configuring multi-factor authentication

1.  [Sign in][mst-login] to MST Console.
1.  Click the `User information` icon in the top-right of the dashboard to go to
    the `User profile` section.
1.  In the `Authentication` tab, toggle `Two-factor authentication` to
    `Enabled`, and enter your password.
1.  On your mobile device, open the Google Authenticator app, tap `+` and select
    `Scan a QR code`.
1.  On your mobile device, scan the QR code provided by Managed Service for TimescaleDB.
1.  In your MST dashboard, enter the confirmation
    code provided by the Google Authenticator app, and click
    `Enable Two-Factor Auth`.


If you lose access to the mobile device you use for multi-factor
authentication, you cannot sign in to your Managed Service for TimescaleDB
account. To regain access to your account, on the login screen, click
`Forgot password?` and follow the step to reset your password. When you have
regained access to your account, reconfigure multi-factor authentication.


## User authentication tokens

Every time a registered user logs in, Managed Service for TimescaleDB creates a
new authentication token. This occurs for login events using the portal, and
using the API. By default, authentication tokens expire after 30 days, but the
expiry date is adjusted every time the token is used. This means that tokens can
be used indefinitely, if the user logs in at least every 30 days.

You can see the list of all current authentication tokens in the Managed Service for TimescaleDB dashboard. Sign in to your account, and click the
`User information` icon in the top-right of the dashboard to go to the
`User profile` section. In the `Authentication` tab, the table lists all current
authentication tokens.

When you make authentication changes, such as enabling two factor authentication
or resetting a password, all existing tokens are revoked. In some cases, a new
token is immediately created so that the web console session remains valid. You
can also manually revoke authentication tokens from the `User profile` page
individually, or click `Revoke all tokens` to revoke all current tokens.

Additionally, you can click `Generate token` to create a new token. When you
generate a token on this page, you can provide a description, maximum age, and
an extension policy. Generating authentication tokens in this way allows you to
use them with monitoring applications that make automatic API calls to Managed Service for TimescaleDB.


There is a limit to how many valid authentication tokens are allowed per user.
This limit is different for tokens that are created as a result of a sign in
operation, and for tokens created explicitly. For automatically created tokens,
the system automatically deletes the oldest tokens as new ones are created. For
explicitly created tokens, older tokens are not deleted unless they expire or
are manually revoked. This can result in explicitly created tokens that stop
working, even though they haven't expired or been revoked. To avoid this, make
sure you sign out at the end of every user session, instead of just discarding
your authentication token. This is especially important for automation tools
that automatically sign in.


===== PAGE: https://docs.tigerdata.com/mst/billing/ =====

# Billing on Managed Service for TimescaleDB

By default, all new services require a credit
card, which is charged at the end of the month for all charges accrued over that
month. Each project is charged separately. Your credit card statement records
the transaction as coming from Aiven, as Aiven provides billing services for
Managed Service for TimescaleDB.

Managed Service for TimescaleDB uses hourly billing. This charge is
automatically calculated, based on the services you are running in your
project. The price charged for your project includes:

*   Virtual machine
*   Networking
*   Backups
*   Setting up


Managed Service for TimescaleDB does not charge you for network traffic used by
your service. However, your application cloud service provider might charge you
for the network traffic going to or from your service.


Terminating or powering a service down stops the accumulation of new charges
immediately. However, the minimum hourly charge unit is one hour. For example,
if you launch a service and shut it down after
40 minutes, you are charged for one full hour.

Migrating to different service plan levels does not incur extra charges for the
migration itself. Note, though, that some service plan levels are more costly
per hour, and your new service is charged at the new rate.

Migrating a service to another cloud region or different cloud provider does not
incur extra charges.


All prices listed for Managed Service for TimescaleDB are inclusive of
credit card and processing fees. However, in some cases, your credit card
provider might charge additional fees, such as an international transaction
fee. These fees are not charged by Tiger Data or Aiven.


## Billing groups

Create billing groups to set up common billing profiles for projects within an
organization. Billing groups make it easier to manage your costs since you
receive a consolidated invoice for all projects assigned to a billing group
and can pay with one saved payment method.

Billing groups can only be used in one organization. Credits are assigned
per billing group and are automatically used to cover charges of any project
assigned to that group.

You can track spending by exporting cost information to business
intelligence tools using the [invoice API][invoice-api].

To access billing groups in [MST Console][mst-console], you must be a
super admin or account owner.

### Create a billing group

To create a billing group, take the following steps:

1. In [MST Console][mst-console], click **Billing** > **Billing
   groups** > **Create billing group**.
1. Enter a name for the billing group and click **Continue**.
1. Enter the billing details.

   You can copy these details from another billing group by selecting it from
   the list. Click **Continue**.
1. Select the projects to add to this billing group and click **Continue**

   You can skip this step and add projects later.
1. Check the information in the **Summary** step. To make changes to any
   section, click **Edit**.
1. When you have confirmed everything is correct, click **Create & Assign**.

### Manage billing groups

To view and update your billing groups, take the following steps:

- Rename billing groups:

    1. In [MST Console][mst-console], go to **Billing** > **Billing
       groups** and find the billing group to rename.
    1. Click **Actions > Rename**.
    1. Enter the new name and click **Rename**.

- Update your billing information:

    1. In [MST Console][mst-console], go to **Billing** > **Billing
       groups** and click on the name of the group to update.
    1. Open the **Billing information** tab and click **Edit** to update the
       details for each section.

- Delete billing groups

    1. In [MST Console][mst-console], open **Billing** > **Billing groups**
       and select the group to delete.
    1. On the **Projects** tab, confirm that the billing group has no
       projects. If there are projects listed, move them to a different billing group.
    1. Go back to the list of billing groups and click **Actions** >
       **Delete** next to the group to be deleted.

### Assign and unassign projects

To manage projects in billing groups, take the following steps.

- Assign projects to a billing group:

  1. In [MST Console][mst-console], go to **Billing > Billing groups**.
  1. Select the billing group to assign the project to.
  1. On the **Projects** tab, click **Assign projects**.
  1. Select the projects and click **Assign projects**.
  1. Click **Cancel** to close the dialog box.


  Assigning a project that is already assigned to another billing group
  will unassign it from that billing group.


- Move a project to another billing group

  1. In [MST Console][mst-console], go to **Billing > Billing groups**.
  1. Click on the name of the billing group that the project is currently
     assigned to.
  1. On the **Projects** tab, find the project to move.
  1. Click the three dots for that project and select the billing group to
     move it to.

## Taxation

Aiven provides billing services for Managed Service for TimescaleDB. These
services are provided by Aiven Ltd, a private limited company incorporated in
Finland.

If you are within the European Union, Finnish law requires that you are charged
a value-added tax (VAT). The VAT percentage depends on where you are domiciled.
For business customers in EU countries other than Finland, you can use the
reverse charge mechanism of 2006/112/EC article 196, by entering a valid VAT ID
into the billing information of your project.

If you are within the United States, no tax is withheld from your payments. In
most cases, you do not require a W-8 form to confirm this, however, if you
require a `W-8BEN-E` form describing this status, you can
[request one][timescale-support].

If you are elsewhere in the world, no taxes are applied to your account,
according to the Value-Added Tax Act of Finland, section 69&nbsp;h.

## Corporate billing

If you prefer to pay by invoice, or if you are unable to provide a credit card
for billing, you can switch your project to corporate billing instead. Under
this model, invoices are generated at the end of the month based on actual
usage, and are sent in `.pdf` format by email to the billing email addresses you
configured in your dashboard.

Payment terms for corporate invoices are 14 days net, by bank transfer, to the
bank details provided on the invoice. By default, services are charged in US
Dollars (USD), but you can request your invoices be sent in either Euros (EUR)
or Pounds Sterling (GBP) at the invoice date's currency exchange rates.

To switch from credit card to corporate billing, make sure your billing profile
and email address is correct in your project's billing settings, and send a message
to the [Tiger Data support team][timescale-support] asking to be changed to corporate
billing.


===== PAGE: https://docs.tigerdata.com/mst/connection-pools/ =====

# Connection pools

When you connect to your database, you consume server resources. If you have a
lot of connections to your database, you can consume a lot of server resources.
One way to mitigate this is to use connection pooling, which allows you to have
high numbers of connections, but keep your server resource use low. The more
client connections you have to your database, the more useful connection pooling
becomes.

By default, Postgres creates a separate backend process for each connection to
the server. Connection pooling uses a tool called PGBouncer to pool multiple
connections to a single backend process. PGBouncer automatically interleaves the
client queries to use a limited number of backend connections more efficiently,
leading to lower resource use on the server and better total performance.

Without connection pooling, the database connections are handled directly by
Postgres backend processes, one process per connection:
<img class="main-content__illustration"
width={1375} height={944}
src="https://assets.timescale.com/docs/images/pgbouncer-pooling-none.webp"
alt="Connection pooling - pooling disabled"/>

When you add connection pooling, fewer backend connections are required. This
frees up server resources for other tasks, such as disk caching:
<img class="main-content__illustration"
width={1375} height={944}
src="https://assets.timescale.com/docs/images/pgbouncer-pooling-enabled.webp"
alt="Connection pooling - pooling enabled"/>

Connection pooling allows you to handle up to 5000 database client connections
simultaneously. You can calculate how many connections you can handle by the
number of CPU cores you have available. You should have at least one connection
per core, but make sure you are not overloading each core. A good number of
connections to aim for is three to five times the available CPU cores, depending
on your workload.

## Connection pooling modes

There are several different pool modes:

*   Transaction (default)
*   Session
*   Statement

### Transaction pooling mode

This is the default pooling mode. It allows each client connection to take turns
using a backend connection during a single transaction. When the transaction is
committed, the backend connection is returned back into the pool and the next
waiting client connection reuses the same connection immediately. This provides
quick response times for queries as long as the most transactions are performed
quickly. This is the most commonly used mode.

### Session pooling mode

This mode holds a client connection until the client disconnects. When the
client disconnects, the server connection is returned back into the connection
pool free connection list, to wait for the next client connection. Client
connections are accepted at TCP level, but their queries only proceed when
another client disconnects and frees up the backend connection back into the
pool. This mode is useful when you require a wait queue for incoming
connections, while keeping the server memory usage low. However, it is not
useful in most common scenarios because the backend connections are recycled
very slowly.

### Statement pooling mode

This mode is similar to the transaction pool mode, except that instead of
allowing a full transaction to be run, it cycles the server side connections
after each and every database statement (SELECT, INSERT, UPDATE, DELETE, for
example). Transactions containing multiple SQL statements are not allowed in
this mode. This mode is best suited to specialized workloads that use sharding
front-end proxies.

## Set up a connection pool

You can set up a connection pool from the MST Console. Make sure you have already created a service that you want to add
connection pooling to.

### Setting up a connection pool


1.  In [MST Console][mst-login], navigate to the `Services` list, and click the name of
    the service you want to add connection pooling to.
1.  In the `Service overview` page, navigate to the `Pools` tab. When you have
    created some pools, they are shown here.
1.  Click `Add Pool` to create a new pool.
1.  In the `Create New Connection Pool` dialog, use these settings:
    *   In the `Pool name` field, type a name for your new pool. This name
        becomes the database `dbname` connection parameter for your pooled
        client connectons.
    *   In the `Database` field, select a database to connect to. Each pool can
        only connect to one database.
    *   In the `Pool Mode` field, select which
        [pool mode](#connection-pooling-modes) to use.
    *   In the `Pool Size` field, select the maximum number of server
        connections this pool can use at any one time.
    *   In the `Username` field, select which database username to connect to
        the database with.
1.  Click `Create` to create the pool, and see the details of the new pool in
    the list. You can click `Info` next to the pool details to see more
    information, including the URI and port details.



Pooled servers use a different port number than regular servers. This allows you
to use both pooled and un-pooled connections at the same time.


===== PAGE: https://docs.tigerdata.com/mst/viewing-service-logs/ =====

# Viewing service logs

Occasionally there is a need to inspect logs from Managed Service for TimescaleDB. For example, to debug query performance or inspecting errors caused
by a specific workload.

There are different built-in ways to inspect service logs at Managed Service for TimescaleDB:

*   When you select a specific service, navigate to the `Logs` tab to see recent
    events. Logs can be browsed back in time.
*   Download logs using the [command-line client][command-line-client] by
    running:

    ```bash
    avn service logs -S desc -f --project <PROJECT_NAME> <SERVICE_NAME>
    ```

*   [REST API][] endpoint is available for fetching the same information two
    above methods output, in case programmatic access is needed.

Service logs included on the normal service price are stored only for a few
days. Unless you are using logs integration to another service, older logs are
not accessible.


===== PAGE: https://docs.tigerdata.com/mst/vpc-peering/ =====

# VPC peering

Virtual Private Cloud (VPC) peering is a method of connecting separate Cloud
private networks to each other. It makes it possible for the virtual machines in
the different VPCs to talk to each other directly without going through the
public internet. VPC peering is limited to VPCs that share the same Cloud
provider.

VPC peering setup is a per project and per region setting. This means that all
services created and running utilize the same VPC peering connection. If needed,
you can have multiple projects that peer with different connections.



services are only accessible using your VPC's internal network. They are not
accessible from the public internet. TLS certificates for VPC peered services are
signed by the MST project CA and cannot be validated against a public CA
(Let's Encrypt). You can choose whether you want to run on a VPC
peered network or on the public internet for every service.



You can set up VPC peering on:

*   [Amazon Web Services (AWS)] [vpc-aws]
*   [Google Cloud Platform (GCP)] [vpc-gcp]
*   [Microsoft Azure] [vpc-azure]


===== PAGE: https://docs.tigerdata.com/mst/integrations/ =====

# Integrations for Managed Service for TimescaleDB

Managed Service for TimescaleDB integrates with the other tools you are already
using. You can combine your services with third-party tools and build a complete cloud data platform.

You can integrate Managed Service for TimescaleDB with:

*   [Grafana]
*   [Loggly]
*   [Datadog]
*   [Prometheus]
*   Syslog
*   External Elasticsearch
*   External OpenSearch


===== PAGE: https://docs.tigerdata.com/mst/extensions/ =====

# Supported Postgres extensions in Managed Service for TimescaleDB

Managed Service for TimescaleDB supports many Postgres extensions. See
[available extensions](#available-extensions) for a full list.

## Add an extension

You can add a supported extension to your database from the command line.


Some extensions have dependencies. When adding these, make sure to create them
in the proper order.



Some extensions require disconnecting and reconnecting the client connection
before they are fully available.


### Adding an extension

1.  Connect to your database as the `tsdbadmin` user.
1.  Run `CREATE EXTENSION IF NOT EXISTS <extension_name>`.

## Available extensions

These extensions are available on Managed Service for TimescaleDB:

<!-- vale Vale.Spelling = NO -->

- address_standardizer
- address_standardizer_data_us
- aiven_extras
- amcheck
- anon
- autoinc
- bloom
- bool_plperl
- btree_gin
- btree_gist
- citext
- cube
- dblink
- dict_int
- dict_xsyn
- earthdistance
- file_fdw
- fuzzystrmatch
- h3
- h3_postgis
- hll
- hstore
- hstore_plperl
- insert_username
- intagg
- intarray
- isn
- jsonb_plperl
- lo
- ltree
- moddatetime
- pageinspect
- pg_buffercache
- pg_cron
- pg_freespacemap
- pg_prewarm
- pg_repack
- pg_similarity
- pg_stat_monitor
- pg_stat_statements
- pg_surgery
- pg_trgm
- pg_visibility
- pg_walinspect
- pgaudit
- pgcrypto
- pgrouting
- pgrowlocks
- pgstattuple
- plperl
- plpgsql
- postgis
- postgis_raster
- postgis_sfcgal
- postgis_tiger_geocoder
- postgis_topology
- postgres_fdw
- refint
- rum
- seg
- sslinfo
- tablefunc
- tcn
- timescaledb
- tsm_system_rows
- tsm_system_time
- unaccent
- unit
- uuid-ossp
- vector
- vectorscale
- xml2
- timescaledb_toolkit

<!-- vale Vale.Spelling = YES -->


The `postgis_legacy` extension is not packaged or supported as an extension by
the PostGIS project. Tiger Data provides the extension package for Managed Service for TimescaleDB.


## Request an extension

You can request an extension not on the list by contacting Support. In your
request, specify the database service and user database where you want to use
the extension.

Untrusted language extensions are not supported. This restriction preserves our
ability to offer the highest possible service level. An example of an untrusted
language extension is `plpythonu`.


You can contact Support directly from Managed Service for TimescaleDB. Click the
life-preserver icon in the upper-right corner of your dashboard.


===== PAGE: https://docs.tigerdata.com/mst/dblink-extension/ =====

# Using the `dblink` extension in Managed Service for TimescaleDB

The `dblink` [Postgres extension][dblink-extension] allows you to connect to
other Postgres databases and to run arbitrary queries.

You can use [foreign data wrappers][pg-fdw] (FDWs) to define a remote
`foreign server` to access its data. The database connection details such as
hostnames are kept in a single place, and you only need to create a
`user mapping` to store remote connections credentials.

## Prerequisites

Before you begin, sign in to your service,
navigate to the `Overview` tab, and take a note of these parameters for the
Postgres remote server. Alternatively, you can use the `avn service get`
command in the Aiven client:

*   `HOSTNAME`: The remote database hostname
*   `PORT`: The remote database port
*   `USER`: The remote database user to connect. The default user is `tsdbadmin`.
*   `PASSWORD`: The remote database password for the `USER`
*   `DATABASE_NAME`: The remote database name. The default database name is `defaultdb`.

### Enable the dblink extension

To enable the `dblink` extension on an MST Postgres service:

1.  Connect to the database as the `tsdbadmin` user:

    ```bash
    psql -x "postgres://tsdbadmin:<PASSWORD>@<HOSTNAME>:<PORT>/defaultdb?sslmode=require"
    ```

1.  Create the `dblink` extension

    ```sql
    CREATE EXTENSION dblink;
    ```

1.  Create a table named `inventory`:

   ```sql
    CREATE TABLE inventory (id int);
   ```

1.  Insert data into the `inventory` table:

   ```sql
    INSERT INTO inventory (id) VALUES (100), (200), (300);
   ```

### Create a foreign data wrapper using dblink_fdw

1.  Create a user `user1` who can access the `dblink`

   ```sql
    CREATE USER user1 PASSWORD 'secret1'
   ```

1.  Create a remote server definition named `mst_remote`, using `dblink_fdw` and
    the connection details of the service.

    ```sql

    CREATE SERVER mst_remote
        FOREIGN DATA WRAPPER dblink_fdw
        OPTIONS (
                 host 'HOST',
                 dbname 'DATABASE_NAME',
                 port 'PORT'
                 );
    ```

1.  Create a user mapping for the `user1` to automatically authenticate as the
    `tsdbadmin` when using the   `dblink`:

    ```sql

        CREATE USER MAPPING FOR user1
           SERVER mst_remote
           OPTIONS (
            user 'tsdbadmin',
            password 'PASSWORD'
            );
    ```

1.  Enable `user1` to use the remote Postgres connection `mst_remote`:

   ```sql
    GRANT USAGE ON FOREIGN SERVER mst_remote TO user1;
   ```

## Query data using a foreign data wrapper

In this example in the `user1` user queries the remote table `inventory` defined
in the target Postgres database from the `mst_remote` server definition:

### Quering data using a foreign data wrapper

To query a foreign data wrapper, you must be a database user with the necessary
permissions on the remote server.

1.  Connect to the service as `user1` with necessary grants to the remote server.

1.  Establish the `dblink` connection to the remote target server:

   ```sql
    SELECT dblink_connect('my_new_conn', 'mst_remote');
   ```

1.  Query using the foreign server definition as parameter:

   ```sql
    SELECT * FROM dblink('my_new_conn','SELECT * FROM inventory') AS t(a int);
   ```

Output is similar to:

   ```sql
       a
     -----
      100
      200
      300
    (3 rows)
   ```


===== PAGE: https://docs.tigerdata.com/mst/security/ =====

# Security overview

This section covers how Managed Service for TimescaleDB handles security of your data while it is
stored.

## Cloud provider accounts

services are hosted by cloud provider
accounts controlled by Tiger Data. These accounts are managed only by Tiger Data
and Aiven operations personnel. Members of the public cannot directly access the
cloud provider account resources.

## Virtual machines

Your services are located on one or more virtual
machines. Each virtual machine is dedicated to a single customer, and is never
multi-tenanted. Customer data never leaves the virtual machine, except when
uploaded to an offsite backup location.

When you create a new service, you need to select a cloud region. When the
virtual machine is launched, it does so in the cloud region you have chosen.
Your data never leaves the chosen cloud region.

If a cloud region has multiple Availability Zones, or a similar
high-availability mechanism, the virtual machines are distributed evenly across
the zones. This provides the best possible service if an Availability Zone
becomes unavailable.

Access to the virtual machine providing your service is restricted. Software
that is accessing your database needs to run on a different virtual machine. To
reduce latency, it is best for it to be using a virtual machine provided by the
same cloud provider, and in the same region, if possible.

Virtual machines are not reused. They are terminated and wiped when you upgrade
or delete your service.

## Project security

Every Managed Service for TimescaleDB project has its own certificate authority.
This certificate authority is used to sign certificates used internally by your
services to communicate between different cluster nodes and to management
systems.

You can download your project certificate authority in MST Console. In the `Services` tab, click the service you want to find
the certificate for. In the service `Overview` tab, under `Connection
information`, locate the `CA Certificate` section, and click `Show` to see the
certificate. It is recommended that you set up your browser or client to trust
that certificate.

All server certificates are signed by the project certificate authority OF MST Console.

## Data encryption

Managed Service for TimescaleDB at-rest data encryption covers both active
service instances as well as service backups in cloud object storage.

Service instances and the underlying virtual machines use full volume
encryption. The encryption method uses LUKS, with a randomly generated ephemeral
key per each instance, and per volume. The keys are never re-used, and are
disposed of when the instance is destroyed. This means that a natural key
rotation occurs with roll-forward upgrades. By default, the LUKS mode is
`aes-xts-plain64:sha256`, with a 512-bit key.

Backups are encrypted with a randomly generated key per file. These keys are in
turn encrypted with an RSA key-encryption key-pair, and stored in the header
section of each backup segment. The file encryption is performed with AES-256 in
CTR mode, with HMAC-SHA256 for integrity protection. The RSA key-pair is
randomly generated for each service. The key lengths are 256-bit for block
encryption, 512-bit for the integrity protection, and 3072-bits for the RSA key.

Encrypted backup files are stored in the object storage in the same region that
the virtual machines are located for the service.

## Networking security

Access to provided services is only provided over TLS encrypted connections. TLS
ensures that third-parties can't eavesdrop or modify the data while it's in
transit between your service and the clients accessing your service. You cannot
use unencrypted plain text connections.

Communication between virtual machines within Managed Service for TimescaleDB is
secured with either TLS or IPsec. You cannot use unencrypted plaintext
connections.

Virtual machines network interfaces are protected by a dynamically configured
firewall based on iptables, which only allows connections from specific
addresses. This is used for network traffic from the internal network to other
VMs in the same service, and for external public network, to client connections.

By default, new services accept incoming traffic from all sources, which is
used to simplify initial set up of your service. It is highly recommended that
you restrict the IP addresses that are allowed to establish connections to your
services.

### Configure allowed incoming IP addresses for your service

1.  In [MST Console][mst-login], select the service to update.
1.  In `Overview` check the `Port` number.

    This is the port that you are managing inbound access for.
1.  In `Network`, check `IP filters`. The default value is `Open for all.

1. Click the ellipsis (...) to the right of Network, then select `Set public IP filters`.

1. Set the `Allowed inbound IP addresses`:

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/mst/set-allowed-ip-addresses.png"
   alt="Add a new allowed incoming IP address for Managed Service for TimescaleDB services"/>

## Networking with VPC peering

When you set up VPC peering, you cannot access your services using public
internet-based access. Service addresses are published in the public DNS record,
but they can only be connected to from your peered VPC network using private
network addresses.

The virtual machines providing your service are hosted by cloud provider
accounts controlled by Tiger Data.

## Customer data privacy

Customer data privacy is of utmost importance at Tiger Data. Tiger Data works with
Aiven to provide Managed Service for TimescaleDB.

In most cases, all the resources required for providing your services are
automatically created, maintained, and terminated by the Managed Service for TimescaleDB infrastructure, with no manual operator intervention required.

The Tiger Data Operations Team are able to securely log in to your service
Virtual Machines, for the purposes of troubleshooting, as required. Tiger Data
operators never access customer data unless you explicitly request them to do
so, to troubleshoot a technical issue. This access is logged and audited.

There is no ability for any customer or member of the public to access any
virtual machines used in Managed Service for TimescaleDB.

Managed Service for TimescaleDB services are periodically assessed and penetration
tested for any security issues by an independent professional cyber-security vendor.

<!---
The most
recent evaluation report
[is available for download][cloud-security-eval].
This link is currently timing out.
-->

Aiven is fully GDPR-compliant, and has executed data processing agreements
(DPAs) with relevant cloud infrastructure providers. If you require a DPA, or if
you want more information about information security policies,
[contact Tiger Data][timescale-support].

<!---
-->


===== PAGE: https://docs.tigerdata.com/mst/postgresql-read-replica/ =====

# Create a read-only replica of Postgres

Postgres read-only replicas allow you to perform read-only queries against
the replica and reduce the load on the primary server. You can optimize query
response times across different geographical locations because the replica can
be created in different regions or on different cloud providers.
For information about creating a read-only replica using the Aiven client,
see the documentation on [creating a read replica using the CLI][read-replica-cli].



If you are running a Managed Service for TimescaleDB [Pro plan](https://docs.tigerdata.com/mst/latest/about-mst/#service-configuration-plans),
you have standby nodes available in a high availability setup. The standby nodes
support read-only queries to reduce the effect of slow queries on the primary
node.



## Creating a replica of Postgres

1.  In [MST Console][mst-login], click the
    service you want to create a remote replica for.

1.  In `Overview`, click `Create a read replica`.

1.  In `Create a PostgreSQL read replica`, type a name for the remote replica,
    select the cloud provider, location, plan that you want to use, and click
    `Create`.

When the read-only replica is created it is listed as a service in your
project. The `Overview` tab of the replica also lists the name of the primary
service for the replica. To promote a read-only replica as a master database,
click the `Promote to master` button.

## Using read-only replica for the service on MST

1.  In the `Overview` page of the read-only replica for the service on MST, copy
    the `Service URI`.

1.  At the psql prompt, connect to the read-only service:

    ```sql
    psql <SERVICE_URI>
    ```

1.  To check whether you are connected to a primary or replica node:

    ```sql
    SELECT * FROM pg_is_in_recovery();
    ```

    If the output is `TRUE` you are connected to the replica, and if the output is
    `FALSE` you are connected to the primary server.


Managed Service for TimescaleDB uses asynchronous replication, so some lag is
expected. When you run an `INSERT` operation on the primary node, a small
delay of less than a second is expected for the change to propagate to the
replica.


===== PAGE: https://docs.tigerdata.com/mst/maintenance/ =====

# Maintenance

On Managed Service for TimescaleDB, software updates are handled automatically,
and you do not need to perform any actions to keep up to date.

Non-critical software updates are applied during a maintenance window that you
can define to suit your workload. If a security vulnerability is found that
affects you, maintenance might be performed outside of your scheduled
maintenance window.

After maintenance updates have been applied, if a new version of the TimescaleDB
binary has been installed, you need to update the extension to use the new
version. To do this, use this command:

```sql
ALTER EXTENSION timescaledb UPDATE;
```


After a maintenance update, the DNS name remains the same, but the IP address
it points to changes.


## Non-critical maintenance updates

Non-critical upgrades are made available before the upgrade is performed
automatically. During this time you can click `Apply upgrades` to start the
upgrade at any time. However, after the time expires, usually around a week,
the upgrade is triggered automatically in the next available maintenance window
for your service. You can configure the maintenance window so that these
upgrades are started only at a particular time, on a set day of the week. If
there are no pending upgrades available during a regular maintenance window, no
changes are performed.

When you are considering your maintenance window schedule, you might prefer to
choose a day and time that usually has very low activity, such as during the
early hours of the morning, or over the weekend. This can help minimize the
impact of a short service interruption. Alternatively, you might prefer to have
your maintenance window occur during office hours, so that you can monitor your
system during the upgrade.

### Adjusting your maintenance window

1.  In [MST Console][mst-login], click the service that you want to manage the maintenance window for.
1.  Click the ellipses (...) to the right of `Maintenance`, then click `Change maintenence window`.
1.  In the `Service Maintenance Window` dialog, select the day of the week and
    the time (in Universal Coordinated Time) you want the maintenance window to
    start. Maintenance windows can run for up to four hours.
    <img class="main-content__illustration"
    src="https://assets.timescale.com/docs/images/mst/change-service-mainenence-window.png"
    alt="Adjust maintenance window"/>
1.  Click `Save Changes`.

## Critical updates

Critical upgrades and security fixes are installed outside normal maintenance
windows when necessary, and sometimes require a short outage.

Upgrades are performed as rolling upgrades where completely new server instances
are built alongside the old ones. When the new instances are up and running they
are synchronized with the old servers, and a controlled automatic failover is
performed to switch the service to the new upgraded servers. The old servers are
retired automatically after the new servers have taken over. The controlled
failover is a very quick and safe operation and it takes less than a minute to
get clients connected again. In most cases, there is five to ten second outage
during this process.


===== PAGE: https://docs.tigerdata.com/mst/failover/ =====

# Failover

One standby read-only replica server is configured, for each service on a Pro plan. You can query a read-only replica server, but cannot
write to a read-only replica server. When a master server fails, the standby replica
server is automatically promoted as master. If you manually created a read-only
replica service, then if a master server fails, the read-only replica services
are not promoted as master servers.

The two distinct cases during which failovers occur are:

*   When the master or replica fails unexpectedly, for example because the hardware
    hosting the virtual machine fails.
*   When controlled failover happens because of upgrades.

## Uncontrolled master or replica fail

When a replica server fails unexpectedly, there is no way to know
whether the server really failed, or whether there is a temporary network
glitch with the cloud provider's network.

There is a 300 second timeout before Managed Service for TimescaleDB
automatically decides the server is gone and spins up a new replica server.
During these 300 seconds, `replica.servicename.timescaledb.io` points to a
server that may not serve queries anymore. The DNS record pointing to the master
server `servicename.timescaledb.io` continues to serve the queries. If the replica
server does not come back up within 300 seconds,
`replica.servicename.timescaledb.io` points to the master server, until a new
replica server is built.

When the master server fails, a replica server waits for 60 seconds before
promoting itself as master. During this 60-second timeout, the master server
`servicename.timescaledb.io` remains unavailable and does not respond. However,
`replica.servicename.timescaledb.io` works in read-only mode. After the replica
server promotes itself as master, `servicename.timescaledb.io` points to the new
master server, and `replica.servicename.timescaledb.io` continues to point to
the new master server. A new replica server is built automatically, and after it
is in sync, `replica.servicename.timescaledb.io` points to the new replica
server.

## Controlled failover during upgrades

When applying upgrades or plan changes on business or premium plans, the standby
server is replaced:

A new server is started, the backup is restored, and the new server starts
following the old master server. After the new server is up and running,
`replica.servicename.timescaledb.io` is updated, and the old replica server is
deleted.

For premium plans, this step is executed for both replica servers before the master
server is replaced. Two new servers are started, a backup is restored, and one new
server is synced up to the old master server. When it is time to switch the master
to a new server, the old master is terminated and one of the new replica servers
is immediately promoted as a master. At this point, `servicename.timescaledb.io`
is updated to point at the new master server. Similarly, the new master is
removed from the `replica.servicename.timescaledb.io` record.


===== PAGE: https://docs.tigerdata.com/mst/manage-backups/ =====

# Back up and restore your Managed Service for TimescaleDB

services are automatically backed up, with full
backups daily, and write-ahead log (WAL) continuously recorded. All backups are
[encrypted][aiven-encrypt].

Managed Service for TimescaleDB uses [`pghoard`][pghoard], a Postgres backup
daemon and restore tool, to store backup data in cloud object stores. The number
of backups stored and the retention time of the backup depend on the service
plan.



The size of logical backups can be different from the size of the Managed Service for TimescaleDB backup that appears on the web console. In some cases,
the difference is significant. Backup sizes that appear in the MST Console are for daily backups, before encryption and
compression. To view the size of each database, including space consumed by
indexes, you can use the `\l+` command at the psql prompt.



## Logical and binary backups

The two types of backups are binary backups and logical backups. Full backups
are version-specific binary backups which, when combined with WAL, allow
consistent recovery to a point in time (PITR). You can create a logical backup
with the `pg_dump` command.

This table lists the differences between binary and logical backups when backing
up indexes, transactions, and data:

|Type|Binary|Logical|
|-|-|-|
|index|contains all data from indexes|does not contain index data, it contains only queries used to recreate indexes from other data|
|transactions|contains uncommitted transactions|does not contain uncommitted transactions|
|data|contains deleted and updated rows which have not been cleaned up by Postgres VACUUM process, and all databases, including templates|does not contain any data already deleted, and depending on the options given, the output might be compressed|

## Restore a service

Managed Service for TimescaleDB provides a point-in-time recovery (PITR). To
restore your service from a backup, click the `Restore` button in the `Backups`
tab for your service. The backups are taken automatically by Managed Service for TimescaleDB and retained for a few days depending on your plan type.

|Plan type|Backup retention period|
|-|-|
|Dev|1 day|
|Basic|2 days|
|Pro|3 days|

## Manually creating a backup

You can use `pg_dump` to create a backup manually. The `pg_dump` command allows
you to create backups that can be directly restored elsewhere if required.

Typical parameters for the command `pg_dump` include:

```bash
pg_dump '<SERVICE_URL_FROM_PORTAL>' -f '<TARGET_FILE/DIR>' -j '<NUMBER_OF_JOBS>' -F '<BACKUP_FORMAT>'
```

The `pg_dump` command can also be run against one of the standby nodes. For
example, use this command to create a backup in directory format using two
concurrent jobs. The results are stored to a directory named `backup`:

```bash
pg_dump 'postgres://tsdbadmin:password@mypg-myproject.a.timescaledb.io:26882/defaultdb?sslmode=require' -f backup -j 2 -F directory
```

You can put all backup files to single tar file and upload to Amazon S3. For example:

```bash
export BACKUP_NAME=backup-date -I.tartar -cf $BACKUP_NAME backup/s3cmd put $BACKUP_NAME s3://pg-backups/$BACKUP_NAME
```


===== PAGE: https://docs.tigerdata.com/mst/aiven-client/ =====

# Aiven Client for Managed Service for TimescaleDB

You can use Aiven Client to manage your services in Managed Service for TimescaleDB.

You can use the Aiven Client tool to:

*   Connect to Managed Service for TimescaleDB
*   Create a service
*   Create a fork
*   Add authentication plugins to your attached Grafana service

Instructions:

- [Install and configure the Aiven client]
- [Fork services with Aiven client]
- [Configure Grafana authentication plugins]
- [Send Grafana emails]
- [Create a read-only replica with the Aiven client]

## Install and configure the Aiven client

Aiven Client is a command line tool for fully managed services. To use Aiven Client, you first need to create an authentication token. Then, you configure the client to connect to your Managed Service for TimescaleDB using the command line.

### Create an authentication token in Managed Service for TimescaleDB

To connect to Managed Service for TimescaleDB using Aiven Client, create an authentication token.

1.  In [Managed Service for TimescaleDB][mst-login], click `User Information` in the top right corner.
1.  In the `User Profile` page, navigate to the `Authentication`tab.
1.  Click `Generate Token`.
2.  In the `Generate access token` dialog, type a descriptive name for the token. Leave the rest of the fields blank.
3.  Copy the generated authentication token and save it.

### Install the Aiven Client

The [Aiven Client][aiven-github] is provided as a Python package. If you've already installed Python, you can install the client on Linux, MacOS, or Windows systems using `pip`:

```bash
pip install aiven-client
```

For more information about installing the Aiven Client, see the [Aiven][aiven-github] documentation.

### Configure Aiven Client to connect to Managed Service for TimescaleDB

To access Managed Service for TimescaleDB with the Aiven Client, you need an authentication token. Aiven Client uses this to access your services on Managed Service for TimescaleDB.

#### Configuring Aiven Client to connect to Managed Service for TimescaleDB

1.  Change to the install directory that contains the configuration files:

    ```bash
    cd ~/.config/aiven/
    ```

1.  Open the `aiven-credentials.json` using any editor and update these lines with your Managed Service for TimescaleDB `User email`, and the
    `authentication token` that you generated:

    ```bash
    {
      "auth_token": "ABC1+123...TOKEN==",
      "user_email": "your.email@timescale.com"
    }
    ```

1.  Save the `aiven-credentials.json` file.

1.  To verify that you can access your services on Managed Service for TimescaleDB, type:

    ```bash
    avn project list
    ```

    This command shows a list of all your projects:

    ```bash
     PROJECT_NAME       DEFAULT_CLOUD            CREDIT_CARD
     =============     =======================   ===================
     project-xxxx      timescale-aws-us-east-1   xxxx-xxxx-xxxx-xxxx
     project-yyyy      timescale-aws-us-east-1   xxxx-xxxx-xxxx-xxxx
     project-zzzz      timescale-aws-us-east-1   xxxx-xxxx-xxxx-xxxx
    ```

## Fork services with Aiven client

When you a fork a service, you create an exact copy of the service, including
the underlying database. You can use a fork of your service to:

*   Create a development copy of your production environment.
*   Set up a snapshot to analyze an issue or test an upgrade.
*   Create an instance in a different cloud, geographical location, or under
    a different plan.

For more information about projects, plans, and other details about
services, see [About Managed Service for TimescaleDB][about-mst].

### Creating a fork of your service

1.  In the Aiven client, connect to your [service][Install and configure the Aiven client].

2.  Switch to the project that contains the service you want to fork:

    ```bash
     avn project switch <PROJECT>
    ```

3.  List the services in the project, and make a note of the service that you want to fork, listed under `SERVICE_NAME` column in the output.

    ```bash
     avn service list
    ```

4.  Get the details of the service that you want to fork:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Create the fork:

    ```bash
    avn service create <NAME_OF_FORK> --project <PROJECT_ID>\
    -t <SERVICE_TYPE> --plan <PLAN> --cloud <CLOUD_NAME>\
    -c service_to_fork_from=<NAME_OF_SERVICE_TO_FORK>
    ```

### Example

To create a fork named `grafana-fork` for a service named `grafana` with these parameters:

*   PROJECT_ID: `project-fork`
*   CLOUD_NAME: `timescale-aws-us-east-1`
*   PLAN_TYPE: `dashboard-1`

```bash
   avn service create grafana-fork --project project-fork -t grafana --plan dashboard-1 --cloud timescale-aws-us-east-1  -c service_to_fork_from=grafana
```

You can switch to `project-fork` and view the newly created `grafana-fork` using:

```bash
   avn service list
```

## Configure Grafana authentication plugins

Grafana supports multiple authentication plugins, in addition to built-in username and password authentication.

On Managed Service for TimescaleDB, Grafana supports Google, GitHub, and GitLab authentication. You can configure authentication integration using the Aiven command-line client.

### Integrating the Google authentication plugin

To integrate Google authentication with Grafana service on Managed Service for TimescaleDB, you need to create your
[Google OAuth keys][google-oauth-keys]. Copy your client ID and client secret to a secure location.

#### How to integrate the Google authentication plugin

1.  In the Aiven Client, connect to your
    [service][Install and configure the Aiven client].

2.  Switch to the project that contains the Grafana service you want to integrate:

    ```bash
     avn switch <PROJECT>
    ```

3.  List the services in the project. Make a note of the Grafana service that you want to integrate, listed under `SERVICE_NAME` column in the
    output.

    ```bash
     avn service list
    ```

4.  Get the details of the service that you want to integrate:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Integrate the plugin with your services using the `<CLIENT_ID>` and `<CLIENT_SECRET>` from your Google developer console:

    ```bash
    avn service update -c auth_google.allowed_domains=<G-SUITE_DOMAIN>\
    -c auth_google.client_id=<CLIENT_ID>\
    -c auth_google.client_secret=<CLIENT_SECRET><SERVICE_NAME>
    ```

6.  Log in to Grafana with your service credentials.

7.  Navigate to `Configuration` → `Plugins` and verify that the Google OAuth application is listed as a plugin.



When you allow sign-ups using the `-c auth_google.allow_sign_up=true` option, by default each new user is created with `viewer` permissions and added to their own newly created organizations. To specify different permissions, use `-c user_auto_assign_org_role=ROLE_NAME`. To add all new users to the main organization, use the  `-c user_auto_assign_org=true` option.



### Integrating the GitHub authentication plugin

To integrate GitHub authentication with Grafana service on Managed Service for TimescaleDB, you need to create your [GitHub OAuth application][github-oauth-keys]. Store your client ID and client secret in a secure location.

#### How to integrate the GitHub authentication plugin

1.  In the Aiven Client, connect to your [service][Install and configure the Aiven client].

2.  Switch to the project that contains the Grafana service you want to integrate:

    ```bash
     avn switch <PROJECT>
    ```

3.  List the services in the project, and make a note of the Grafana service
    that you want to integrate, listed under `SERVICE_NAME` column in the
    output.

    ```bash
     avn service list
    ```

4.  Get the details of the service that you want to integrate:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Integrate the plugin with your service using the `<CLIENT_ID>`, and
    `<CLIENT_SECRET>` from your GitHub OAuth application:

    ```bash
    avn service update -c auth_github.client_id=<CLIENT_ID>\
    -c auth_github.client_secret=<CLIENT_SECRET> <SERVICE_NAME>

    ```

6.  Log in to Grafana with your service credentials.
7.  Navigate to `Configuration` → `Plugins`. The Plugins page lists
    GitHub OAuth application for the Grafana instance.



When you allow sign-ups using the `-c auth_github.allow_sign_up=true` option, by default each new user is created with `viewer`permission and added to their own newly created organizations. To specify different permissions, use `-c user_auto_assign_org_role=ROLE_NAME`. To add all new users to the main organization, use the `-c user_auto_assign_org=true` option.



### Integrating the GitLab authentication plugin

To integrate the GitLab authentication with Grafana service on Managed Service for TimescaleDB, you need to create your [GitLab OAuth
application][gitlab-oauth-keys]. Copy your client ID, client secret, and GitLab groups name to a secure location.

If you use your own instance of GitLab instead of gitlab.com, then you need to set the following:

*   auth_gitlab.api_url
*   auth_github.auth_url
*   auth_github.token_url

#### How to integrate the GitLab authentication plugin

1.  In the Aiven Client, connect to your [MST_SERVICE_LONG][Install and configure the Aiven client].

2.  Switch to the project that contains the Grafana service you want to integrate:

    ```bash
     avn project switch <PROJECT>
    ```

3.  List the services in the project. Note the Grafana service that you want to integrate, listed under `SERVICE_NAME` column in the output.

    ```bash
     avn service list
    ```

4.  Get the details of the service that you want to integrate:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Integrate the plugin with your service using the `<CLIENT_ID>`, `<CLIENT_SECRET>`, and `<GITLAB_GROUPS>` from your GitLab OAuth application:

    ```bash
    avn service update -c auth_gitlab.client_id=<CLIENT_ID>\
    -c auth_gitlab.client_secret=<CLIENT_SECRET>\
    -c auth_gitlab.allowed_groups=<GITLAB_GROUPS> <SERVICE_NAME>

    ```

6.  Log in to Grafana with your service credentials.

7.  Navigate to `Configuration` → `Plugins`. The Plugins page lists GitLab OAuth application for the Grafana instance.



When you allow sign-ups using the `-c auth_gitlab.allow_sign_up=true` option, by default each new user is created with `viewer`permission and added to their own newly created organizations. To specify different permissions, use `-c user_auto_assign_org_role=ROLE_NAME`. To add all new users to the main organization, use the `-c user_auto_assign_org=true` option.



## Send Grafana emails

Use the Aiven client to configure the Simple Mail Transfer Protocol (SMTP) server settings and send emails from Managed Service for TimescaleDB for Grafana. This includes invite emails, reset password emails, and alert messages.

### Prerequisites

Before you begin, make sure you have:

*   (Optional): Made a note of these values in the SMTP server:
    `IP or hostname`, `SMTP server port`, `Username`, `Password`,
    `Sender email address`, and `Sender name`.

### Configuring the SMTP server for Grafana service

1.  In the Aiven client, connect to your [service][Install and configure the Aiven client].

2.  Switch to the project that contains the Grafana service you want to integrate:

    ```bash
     avn project switch <PROJECT>
    ```

3.  List the services in the project. Note the Grafana service that you want to configure, listed under `SERVICE_NAME` column in the
    output.

    ```bash
     avn service list
    ```

4.  Get the details of the service that you want to integrate:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Configure the Grafana service using the SMTP values:

    ```bash
       avn service update --project <PROJECT> <SERVICE_NAME>\
       -c smtp_server.host=smtp.example.com \
       -c smtp_server.port=465 \
       -c smtp_server.username=emailsenderuser \
       -c smtp_server.password=emailsenderpass \
       -c smtp_server.from_address="grafana@yourcompany.com"
    ```

6.  [](#) Review all available custom options, and configure:

    ```bash
       avn service types -v
    ```

You can now send emails for your Grafana service on MST.

## Create a read-only replica with Aiven client

Read-only replicas enable you to perform read-only queries against the replica and reduce the load on the primary server. They are also a
good way to optimize query response times across different geographical locations. You can achieve this by placing the replicas in different regions or even different cloud providers.

### Creating a read-only replica of your service

1.  In the Aiven client, connect to your [service][Install and configure the Aiven client].

2.  Switch to the project that contains the service you want to create a read-only replica for:

    ```bash
    avn project switch <PROJECT>
    ```

3.  List the services in the project. Note the service for which you will create a read-only replica. You can find it listed under the `SERVICE_NAME` column in the output:

    ```bash
    avn service list
    ```

4.  Get the details of the service that you want to fork:

    ```bash
    avn service get <SERVICE_NAME>
    ```

5.  Create a read-only replica:

    ```bash
    avn service create <NAME_OF_REPLICA> --project <PROJECT_ID>\
    -t pg --plan <PLAN_TYPE> --cloud timescale-aws-us-east-1\
    -c pg_read_replica=true\
    -c service_to_fork_from=<NAME_OF_SERVICE_TO_FORK>\
    -c pg_version=11 -c variant=timescale
    ```

### Example

To create a fork named `replica-fork` for a service named `timescaledb` with
these parameters:

*   PROJECT_ID: `fork-project`
*   CLOUD_NAME: `timescale-aws-us-east-1`
*   PLAN_TYPE: `timescale-basic-100-compute-optimized`

```bash
avn service create replica-fork --project fork-project\
-t pg --plan timescale-basic-100-compute-optimized\
--cloud timescale-aws-us-east-1 -c pg_read_replica=true\
-c service_to_fork_from=timescaledb -c\
pg_version=11 -c variant=timescale
```

You can switch to `project-fork` and view the newly created `replica-fork` using:

```bash
avn service list
```


===== PAGE: https://docs.tigerdata.com/mst/migrate-to-mst/ =====

# Migrate from self-hosted TimescaleDB to Managed Service for TimescaleDB

You can migrate your data from self-hosted TimescaleDB to Managed Service for TimescaleDB and automate most of the common operational tasks.

Each service has a database named `defaultdb`, and a default user account named `tsdbadmin`. You use
MST Console to create additional users and databases using the `Users` and `Databases` tabs.

You can switch between different plan sizes in Managed Service for TimescaleDB.
However, during the migration process, choose a plan size that has the same
storage size or slightly larger than the currently allocated plan. This allows
you to limit the downtime during the migration process and have sufficient compute and storage resources.



Depending on your database size and network speed, migration can take a very
long time. During this time, any new writes that happen during the migration
process are not included. To prevent data loss, turn off all the
writes to the source self-hosted TimescaleDB database before you start migration.

Before migrating for production, do a cold run without turning off writes to the source self-hosted TimescaleDB database.
This gives you an estimate of the time the migration process takes, and helps you to practice migrating without causing
downtime to your customers.



If you prefer the features of Tiger Cloud, you can easily [migrate your data][migrate-live] from an service
to a Tiger Cloud service.

## Prerequisites

Before you migrate your data, do the following:

* Set up the migration machine:

   You run the migration commands on the migration machine. It must have enough disk space to hold the dump file.
   * Install the Postgres [`pg_dump`][pg_dump] and [`pg_restore`][pg_restore] utilities on a migration machine.

   * Install a client to connect to self-hosted TimescaleDB and Managed Service for TimescaleDB.

      These instructions use [`psql`][psql], but any client works.

*  Create a target service:

    For more information, see the [Install Managed Service for TimescaleDB][install-mst]. Provision your target service with enough
    space for all your data.

*  On the source self-hosted TimescaleDB and the target service, ensure that you are running:
   *  The same major version of Postgres.

      For information, see [upgrade Postgres][upgrading-postgresql-self-hosted].

   *  The same major version of TimescaleDB

      For more information, see [Upgrade TimescaleDB to a major version][upgrading-timescaledb].

## Migrate your data to a service

To move your data from self-hosted TimescaleDB instance to a service, run the following commands from your migration
machine:

1. **Take offline the applications that connect to the source self-hosted TimescaleDB instance**

   The duration of migration is proportional to the amount of data stored in your database. By
   disconnecting your app from your database, you avoid possible data loss.

1. **Set your connection strings**

   These variables hold the connection information for the source self-hosted TimescaleDB instance and the target service:

   ```bash
   export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
   export TARGET="postgres://tsdbadmin:<password>@<host>:<port>/defaultdb?sslmode=require"
   ```

1. **Dump the data from your source Tiger Cloud service**

    ```bash
    pg_dump -d "source" --no-owner -Fc -v -f dump.bak
    ```

1. **Put your target service in the right state for restoring**

   ```bash
   psql -d "target" -c "SELECT timescaledb_pre_restore();"
   ```

1. **Upload your data to the target service**

    ```bash
    pg_restore -d "target" --jobs 4 -Fc dump.bak
    ```
   The `--jobs`  option specifies the number of CPUs to use to dump and restore the database concurrently.

1. **Return your target service to normal operations**

   ```bash
   psql -d "target" -c "SELECT timescaledb_post_restore();"
   ```

1.  Connect to your new database and update your table statistics by running
    [`ANALYZE`]   [analyze] on your entire dataset:

    ```sql
    psql -d "target" defaultdb=> ANALYZE;
    ```

To migrate from multiple databases, you repeat this migration procedure one database after another.

## Troubleshooting

If you see the following errors during migration, you can safely ignore them. The migration still runs
successfully.

-  For `pg_dump`:

    ```bash
    pg_dump: warning: there are circular foreign-key constraints on this table:
    pg_dump: hypertable
    pg_dump: You might not be able to restore the dump without using --disable-triggers or temporarily dropping the constraints.
    pg_dump: Consider using a full dump instead of a --data-only dump to avoid this problem.
    pg_dump: NOTICE:  hypertable data are in the chunks, no data will be copied
    DETAIL:  Data for hypertables are stored in the chunks of a hypertable so COPY TO of a hypertable will not copy any data.
    HINT:  Use "COPY (SELECT * FROM <hypertable>) TO ..." to copy all data in hypertable, or copy each chunk individually.
    ```

- For `pg_restore`:

   ```bash
   pg_restore: while PROCESSING TOC:
   pg_restore: from TOC entry 4142; 0 0 COMMENT EXTENSION timescaledb
   pg_restore: error: could not execute query: ERROR:  must be owner of extension timescaledb
   Command was: COMMENT ON EXTENSION timescaledb IS 'Enables scalable inserts and complex queries for time-series data';

 ```


===== PAGE: https://docs.tigerdata.com/mst/restapi/ =====

# Using REST API in Managed Service for TimescaleDB

Managed Service for TimescaleDB has an API for integration and automation tasks.
For information about using the endpoints, see the [API Documentation][aiven-api].
MST offers an HTTP API with token authentication and JSON-formatted data. You
can use the API for all the tasks that can be performed using the MST Console.
To get started you need to first create an authentication token, and then use
the token in the header to use the API endpoints.

1.  In [Managed Service for TimescaleDB][mst-login], click `User Information` in the top right corner.
1.  In the `User Profile` page, navigate to the `Authentication`tab.
1.  Click `Generate Token`.
1.  In the `Generate access token` dialog, type a descriptive name for the
    token and leave the rest of the fields blank.
1.  Copy the generated authentication token and save it.

### Using cURL to get your details

1.  Set the environment variable `MST_API_TOKEN` with the access token that you generate:

    ```bash
    export MST_API_TOKEN="access token"
    ```

1.  To get the details about the current user session using the `/me` endpoint:

    ```bash
    curl -s -H "Authorization: aivenv1 $MST_API_TOKEN" https://api.aiven.io/v1/me|json_pp
    ```

    The output looks similar to this:

    ```bash
    {
        "user": {
            "auth": [],
            "create_time": "string",
            "features": { },
            "intercom": {},
            "invitations": [],
            "project_membership": {},
            "project_memberships": {},
            "projects": [],
            "real_name": "string",
            "state": "string",
            "token_validity_begin": "string",
            "user": "string",
            "user_id": "string"
        }
    }
    ```


===== PAGE: https://docs.tigerdata.com/mst/identify-index-issues/ =====

# Identify and resolve issues with indexes in Managed Service for TimescaleDB

Postgres indexes can be corrupted for a variety of reasons, including
software bugs, hardware failures, or unexpected duplicated data. `REINDEX` allows
you to rebuild the index in such situations.

## Rebuild non-unique indexes

You can rebuild corrupted indexes that do not have `UNIQUE` in their definition.
You can run the `REINDEX` command for all indexes of a table (`REINDEX TABLE`),
and for all indexes in the entire database (`REINDEX DATABASE`).
For more information on the `REINDEX` command, see the [Postgres documentation][postgres-docs].

This command creates a new index that replaces the old one:

```sql
REINDEX INDEX <index-name>;
```



When you use `REINDEX`, the tables are locked and you may not be able to use the
database, until the operation is complete.



In some cases, you might need to manually build a second index concurrently
with the old index, and then remove the old index:

```sql
CREATE INDEX CONCURRENTLY test_index_new ON table_a (...);
DROP INDEX CONCURRENTLY test_index_old;
ALTER INDEX test_index_new RENAME TO test_index;
```

## Rebuild unique indexes

A `UNIQUE` index works on one or more columns where the combination is unique
in the table. When the index is corrupted or disabled, duplicated
physical rows appear in the table, breaking the uniqueness constraint of the
index. When you try to rebuild an index that is not unique, the `REINDEX` command fails.
To resolve this issue, first remove the duplicate rows from the table and then
rebuild the index.

### Identify conflicting duplicated rows

To identify conflicting duplicate rows, you need to run a query that counts the
number of rows for each combination of columns included in the index definition.

For example, this `route` table has a `unique_route_index` index defining
unique rows based on the combination of the `source` and `destination` columns:

```sql
CREATE TABLE route(
    source TEXT,
    destination TEXT,
    description TEXT
    );

CREATE UNIQUE INDEX unique_route_index
    ON route (source, destination);
```

If the `unique_route_index` is corrupt, you can find duplicated rows in the
`route` table using this query:

```sql
SELECT
    source,
    destination,
    count
FROM
    (SELECT
        source,
        destination,
        COUNT(*) AS count
    FROM route
    GROUP BY
        source,
        destination) AS foo
WHERE count > 1;
```

The query groups the data by the same `source` and `destination` fields defined
in the index, and filters any entries with more than one occurrence.

Resolve the problematic entries in the rows by manually deleting or merging the
entries until no duplicates exist. After all duplicate entries are removed, you
can use the `REINDEX` command to rebuild the index.


===== PAGE: https://docs.tigerdata.com/about/whitepaper/ =====

# Tiger Data architecture for real-time analytics

Tiger Data has created a powerful application database for real-time analytics on time-series data. It integrates seamlessly
with the Postgres ecosystem and enhances it with automatic time-based partitioning, hybrid row-columnar storage, and vectorized execution—enabling high-ingest performance, sub-second queries, and full SQL support at scale.

Tiger Cloud offers managed database services that provide a stable and reliable environment for your
applications. Each service is based on a Postgres database instance and the TimescaleDB extension.

By making use of incrementally updated materialized views and advanced analytical functions, TimescaleDB reduces compute overhead and improves query efficiency. Developers can continue using familiar SQL workflows and tools, while benefiting from a database purpose-built for fast, scalable analytics.

This document outlines the architectural choices and optimizations that power TimescaleDB and Tiger Cloud’s performance and
scalability while preserving Postgres’s reliability and transactional guarantees.

Want to read this whitepaper from the comfort of your own computer?

<center>
   [Tiger Data architecture for real-time analytics (PDF)](https://assets.timescale.com/docs/downloads/tigerdata-whitepaper.pdf)
</center>


## Introduction

### What is real-time analytics?

Real-time analytics enables applications to process and query data as it is generated and as it accumulates, delivering immediate and ongoing insights for decision-making. Unlike traditional analytics, which relies on batch processing and delayed reporting, real-time analytics supports *both* instant queries on fresh data and fast exploration of historical trends—powering applications with sub-second query performance across vast, continuously growing datasets.

Many modern applications depend on real-time analytics to drive critical functionality:



* **IoT monitoring systems** track sensor data over time, identifying long-term performance patterns while still surfacing anomalies as they arise. This allows businesses to optimize maintenance schedules, reduce costs, and improve reliability.
* **Financial and business intelligence platforms** analyze both current and historical data to detect trends, assess risk, and uncover opportunities—from tracking stock performance over a day, week, or year to identifying spending patterns across millions of transactions.
* **Interactive customer dashboards** empower users to explore live and historical data in a seamless experience—whether it's a SaaS product providing real-time analytics on business operations, a media platform analyzing content engagement, or an e-commerce site surfacing personalized recommendations based on recent and past behavior.

Real-time analytics isn't just about reacting to the latest data, although that is critically important. It's also about delivering fast, interactive, and scalable insights across all your data, enabling better decision-making and richer user experiences.  Unlike traditional ad-hoc analytics used by analysts, real-time analytics powers applications—driving dynamic dashboards, automated decisions, and user-facing insights at scale.

To achieve this, real-time analytics systems must meet several key requirements:

* **Low-latency queries** ensure sub-second response times even under high load, enabling fast insights for dashboards, monitoring, and alerting.
* **Low-latency ingest** minimizes the lag between when data is created and when it becomes available for analysis, ensuring fresh and accurate insights.
* **Data mutability** allows for efficient updates, corrections, and backfills, ensuring analytics reflect the most accurate state of the data.
* **Concurrency and scalability** enable systems to handle high query volumes and growing workloads without degradation in performance.
* **Seamless access to both recent and historical data** ensures fast queries across time, whether analyzing live, streaming data, or running deep historical queries on days or months of information.
* **Query flexibility** provides full SQL support, allowing for complex queries with joins, filters, aggregations, and analytical functions.


### Tiger Cloud: real-time analytics from Postgres

Tiger Cloud is a high-performance database that brings real-time analytics to applications. It combines fast queries,
high ingest performance, and full SQL support—all while ensuring scalability and reliability. Tiger Cloud extends Postgres with the TimescaleDB extension. It enables sub-second queries on vast amounts of incoming data while providing optimizations designed for continuously updating datasets.

Tiger Cloud achieves this through the following optimizations:

* **Efficient data partitioning:** automatically and transparently partitioning data into chunks, ensuring fast queries, minimal indexing overhead, and seamless scalability
* **Row-columnar storage:** providing the flexibility of a row store for transactions and the performance of a column store for analytics
* **Optimized query execution: **using techniques like chunk and batch exclusion, columnar storage, and vectorized execution to minimize latency
* **Continuous aggregates:** precomputing analytical results for fast insights without expensive reprocessing
* **Cloud-native operation: **compute/compute separation, elastic usage-based storage, horizontal scale out, data tiering to object storage
* **Operational simplicity: **offering high availability, connection pooling, and automated backups for reliable and scalable real-time applications

With Tiger Cloud, developers can build low-latency, high-concurrency applications that seamlessly handle streaming data, historical queries, and real-time analytics while leveraging the familiarity and power of Postgres.


## Data model

Today's applications demand a database that can handle real-time analytics and transactional queries without sacrificing speed, flexibility, or SQL compatibility (including joins between tables). TimescaleDB achieves this with **hypertables**, which provide an automatic partitioning engine, and **hypercore**, a hybrid row-columnar storage engine designed to deliver high-performance queries and efficient compression (up to 95%) within Postgres.


### Efficient data partitioning

TimescaleDB provides hypertables, a table abstraction that automatically partitions data into chunks in real time (using time stamps or incrementing IDs) to ensure fast queries and predictable performance as datasets grow. Unlike traditional relational databases that require manual partitioning, hypertables automate all aspects of partition management, keeping locking minimal even under high ingest load.

At ingest time, hypertables ensure that Postgres can deal with a constant stream of data without suffering from table bloat and index degradation by automatically partitioning data across time. Because each chunk is ordered by time and has its own indexes and storage, writes are usually isolated to small, recent chunks—keeping index sizes small, improving cache locality, and reducing the overhead of vacuum and background maintenance operations. This localized write pattern minimizes write amplification and ensures consistently high ingest performance, even as total data volume grows.

At query time, hypertables efficiently exclude irrelevant chunks from the execution plan when the partitioning column is used in a `WHERE` clause. This architecture ensures fast query execution, avoiding the gradual slowdowns that affect non-partitioned tables as they accumulate millions of rows. Chunk-local indexes keep indexing overhead minimal, ensuring index operations scans remain efficient regardless of dataset size.

<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/hypertable.png"
  alt=""
/>
</center>

Hypertables are the foundation for all of TimescaleDB’s real-time analytics capabilities. They enable seamless data ingestion, high-throughput writes, optimized query execution, and chunk-based lifecycle management—including automated data retention (drop a chunk) and data tiering (move a chunk to object storage).



### Row-columnar storage

Traditional databases force a trade-off between fast inserts (row-based storage) and efficient analytics (columnar storage). Hypercore eliminates this trade-off, allowing real-time analytics without sacrificing transactional capabilities.

Hypercore dynamically stores data in the most efficient format for its lifecycle:



* **Row-based storage for recent data**: the most recent chunk (and possibly more) is always stored in the rowstore, ensuring fast inserts, updates, and low-latency single record queries. Additionally, row-based storage is used as a writethrough for inserts and updates to columnar storage.
* **Columnar storage for analytical performance**: chunks are automatically compressed into the columnstore, optimizing storage efficiency and accelerating analytical queries.

Unlike traditional columnar databases, hypercore allows data to be inserted or modified at any stage, making it a flexible solution for both high-ingest transactional workloads and real-time analytics—within a single database.


### Columnar storage layout

TimescaleDB’s columnar storage layout optimizes analytical query performance by structuring data efficiently on disk, reducing scan times, and maximizing compression rates. Unlike traditional row-based storage, where data is stored sequentially by row, columnar storage organizes and compresses data by column, allowing queries to retrieve only the necessary fields in batches rather than scanning entire rows. But unlike many column store implementations, TimescaleDB’s columnstore supports full mutability—inserts, upserts, updates, and deletes, even at the individual record level—with transactional guarantees. Data is also immediately visible to queries as soon as it is written.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/hypertable-with-hypercore-enabled.png"
  alt=""
/>
</center>


#### Columnar batches

TimescaleDB uses columnar collocation and columnar compression within row-based storage to optimize analytical query performance while maintaining full Postgres compatibility. This approach ensures efficient storage, high compression ratios, and rapid query execution.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/columnstore.png"
  alt=""
/>
</center>

A rowstore chunk is converted to a columnstore chunk by successfully grouping together sets of rows (typically up to 1000) into a single batch, then converting the batch into columnar form.

Each compressed batch does the following:

* Encapsulates columnar data in compressed arrays of up to 1,000 values per column, stored as a single entry in the underlying compressed table
* Uses a column-major format within the batch, enabling efficient scans by co-locating values of the same column and allowing the selection of individual columns without reading the entire batch
* Applies advanced compression techniques at the column level, including run-length encoding, delta encoding, and Gorilla compression, to significantly reduce storage footprint (by up to 95%) and improve I/O performance.

While the chunk interval of rowstore and columnstore batches usually remains the same, TimescaleDB can also combine columnstore batches so they use a different chunk interval.

This architecture provides the benefits of columnar storage—optimized scans, reduced disk I/O, and improved analytical performance—while seamlessly integrating with Postgres’s row-based execution model.


#### Segmenting and ordering data

To optimize query performance, TimescaleDB allows explicit control over how data is physically organized within columnar storage. By structuring data effectively, queries can minimize disk reads and execute more efficiently, using vectorized execution for parallel batch processing where possible.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/columnstore-segmentby.png"
  alt=""
/>
</center>



* **Group related data together to improve scan efficiency**: organizing rows into logical segments ensures that queries filtering by a specific value only scan relevant data sections. For example, in the above, querying for a specific ID is particularly fast. *(Implemented with <code>SEGMENTBY</code>.)*
* **Sort data within segments to accelerate range queries**: defining a consistent order reduces the need for post-query sorting, making time-based queries and range scans more efficient. *(Implemented with <code>ORDERBY</code>.)*
* **Reduce disk reads and maximize vectorized execution**: a well-structured storage layout enables efficient batch processing (Single Instruction, Multiple Data, or SIMD vectorization) and parallel execution, optimizing query performance.

By combining segmentation and ordering, TimescaleDB ensures that columnar queries are not only fast but also resource-efficient, enabling high-performance real-time analytics.


### Data mutability

Traditional databases force a trade-off between fast updates and efficient analytics. Fully immutable storage is impractical in real-world applications, where data needs to change. Asynchronous mutability—where updates only become visible after batch processing—introduces delays that break real-time workflows. In-place mutability, while theoretically ideal, is prohibitively slow in columnar storage, requiring costly decompression, segmentation, ordering, and recompression cycles.

Hypercore navigates these trade-offs with a hybrid approach that enables immediate updates without modifying compressed columnstore data in place. By staging changes in an interim rowstore chunk, hypercore allows updates and deletes to happen efficiently while preserving the analytical performance of columnar storage.


<center>
<img
  class="main-content__illustration"
  src="https://assets.timescale.com/docs/images/mutation.png"
  alt=""
/>
</center>


#### Real-time writes without delays

All new data which is destined for a columnstore chunk is first written to an interim rowstore chunk, ensuring high-speed ingestion and immediate queryability. Unlike fully columnar systems that require ingestion to go through compression pipelines, hypercore allows fresh data to remain in a fast row-based structure before being later compressed into columnar format in ordered batches as normal.

Queries transparently access both the rowstore and columnstore chunks, meaning applications always see the latest data instantly, regardless of its storage format.


#### Efficient updates and deletes without performance penalties

When modifying or deleting existing data, hypercore avoids the inefficiencies of both asynchronous updates and in-place modifications. Instead of modifying compressed storage directly, affected batches are decompressed and staged in the interim rowstore chunk, where changes are applied immediately.

These modified batches remain in row storage until they are recompressed and reintegrated into the columnstore (which happens automatically via a background process). This approach ensures updates are immediately visible, but without the expensive overhead of decompressing and rewriting entire chunks. This approach avoids:



* The rigidity of immutable storage, which requires workarounds like versioning or copy-on-write strategies
* The delays of asynchronous updates, where modified data is only visible after batch processing
* The performance hit of in-place mutability, which makes compressed storage prohibitively slow for frequent updates
* The restrictions some databases have on not altering the segmentation or ordering keys


## Query optimizations

Real-time analytics isn’t just about raw speed—it’s about executing queries efficiently, reducing unnecessary work, and maximizing performance. TimescaleDB optimizes every step of the query lifecycle to ensure that queries scan only what’s necessary, make use of data locality, and execute in parallel for sub-second response times over large datasets.


### Skip unnecessary data

TimescaleDB minimizes the amount of data a query touches, reducing I/O and improving execution speed:


#### Primary partition exclusion (row and columnar)

Queries automatically skip irrelevant partitions (chunks) based on the primary partitioning key (usually a timestamp), ensuring they only scan relevant data.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/constraint-exclusion-partitioning-column.png"
  alt=""
/>
</center>


#### Secondary partition exclusion (columnar)

Min/max metadata allows queries filtering on correlated dimensions (e.g., `order_id` or secondary timestamps) to exclude chunks that don’t contain relevant data.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/chunk-skipping-on-secondary-columns.png"
  alt=""
/>
</center>


#### Postgres indexes (row and columnar)

Unlike many databases, TimescaleDB supports sparse indexes on columnstore data, allowing queries to efficiently locate specific values within both row-based and compressed columnar storage. These indexes enable fast lookups, range queries, and filtering operations that further reduce unnecessary data scans.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/rowstore-indexes.png"
  alt=""
/>
</center>


#### Batch-level filtering (columnar)

Within each chunk, compressed columnar batches are organized using `SEGMENTBY` keys and ordered by `ORDERBY` columns. Indexes and min/max metadata can be used to quickly exclude batches that don’t match the query criteria.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/batch-skipping-indexes.png"
  alt=""
/>
</center>


### Maximize locality

Organizing data for efficient access ensures queries are read in the most optimal order, reducing unnecessary random reads and reducing scans of unneeded data.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/segment-and-order.png"
  alt=""
/>
</center>



* **Segmentation**: Columnar batches are grouped using `SEGMENTBY` to keep related data together, improving scan efficiency.
* **Ordering**: Data within each batch is physically sorted using `ORDERBY`, increasing scan efficiency (and reducing I/O operations), enabling efficient range queries, and minimizing post-query sorting.
* **Column selection**: Queries read only the necessary columns, reducing disk I/O, decompression overhead, and memory usage.


### Parallelize execution

Once a query is scanning only the required columnar data in the optimal order, TimescaleDB is able to maximize performance through parallel execution. As well as using multiple workers, TimescaleDB accelerates columnstore query execution by using Single Instruction, Multiple Data (SIMD) vectorization, allowing modern CPUs to process multiple data points in parallel.



<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/simd.png"
  alt=""
/>
</center>

The TimescaleDB implementation of SIMD vectorization currently allows:



* **Vectorized decompression**, which efficiently restores compressed data into a usable form for analysis.
* **Vectorized filtering**, which rapidly applies filter conditions across data sets.
* **Vectorized aggregation**, which performs aggregate calculations, such as sum or average, across multiple data points concurrently.


## Accelerating queries with continuous aggregates

Aggregating large datasets in real time can be expensive, requiring repeated scans and calculations that strain CPU and I/O. While some databases attempt to brute-force these queries at runtime, compute and I/O are always finite resources—leading to high latency, unpredictable performance, and growing infrastructure costs as data volume increases.

**Continuous aggregates**, the TimescaleDB implementation of incrementally updated materialized views, solve this
by shifting computation from every query run to a single, asynchronous step after data is ingested. Only the time buckets that receive new or modified data are updated, and queries read precomputed results instead of scanning raw data—dramatically improving performance and efficiency.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/caggs.png"
  alt=""
/>
</center>

When you know the types of queries you'll need ahead of time, continuous aggregates allow you to pre-aggregate data along meaningful time intervals—such as per-minute, hourly, or daily summaries—delivering instant results without on-the-fly computation.

Continuous aggregates also avoid the time-consuming and error-prone process of maintaining manual rollups, while continuing to offer data mutability to support efficient updates, corrections, and backfills.  Whenever new data is inserted or modified in chunks which have been materialized, TimescaleDB stores invalidation records reflecting that these results are stale and need to be recomputed.  Then, an asynchronous process re-computes regions that include invalidated data, and updates the materialized results.  TimescaleDB tracks the lineage and dependencies between continuous aggregates and their underlying data, to ensure the continuous aggregates are regularly kept up-to-date. This happens in a resource-efficient manner, and where multiple invalidations can be coalesced into a single refresh (as opposed to refreshing any dependencies at write time, such as via a trigger-based approach).

Continuous aggregates themselves are stored in hypertables, and they can be converted to columnar storage for compression, and raw data can be dropped, reducing storage footprint and processing cost. Continuous aggregates also support hierarchical rollups (e.g., hourly to daily to monthly) and real-time mode, which merges precomputed results with the latest ingested data to ensure accurate, up-to-date analytics.

This architecture enables scalable, low-latency analytics while keeping resource usage predictable—ideal for dashboards, monitoring systems, and any workload with known query patterns.


### Hyperfunctions for real-time analytics

Real-time analytics requires more than basic SQL functions—efficient computation is essential as datasets grow in size and complexity. Hyperfunctions, available through the `timescaledb_toolkit` extension, provide high-performance, SQL-native functions tailored for time-series analysis. These include advanced tools for gap-filling, percentile estimation, time-weighted averages, counter correction, and state tracking, among others.

A key innovation of hyperfunctions is their support for partial aggregation, which allows TimescaleDB to store intermediate computational states rather than just final results. These partials can later be merged to compute rollups efficiently, avoiding expensive reprocessing of raw data and reducing compute overhead. This is especially effective when combined with continuous aggregates.

Consider a real-world example: monitoring request latencies across thousands of application instances. You might want to compute p95 latency per minute, then roll that up into hourly and daily percentiles for dashboards or alerts. With traditional SQL, calculating percentiles requires a full scan and sort of all underlying data—making multi-level rollups computationally expensive.

With TimescaleDB, you can use the `percentile_agg` hyperfunction in a continuous aggregate to compute and store a partial aggregation state for each minute. This state efficiently summarizes the distribution of latencies for that time bucket, without storing or sorting all individual values. Later, to produce an hourly or daily percentile, you simply combine the stored partials—no need to reprocess the raw latency values.

This approach provides a scalable, efficient solution for percentile-based analytics. By combining hyperfunctions with continuous aggregates, TimescaleDB enables real-time systems to deliver fast, resource-efficient insights across high-ingest, high-resolution datasets—without sacrificing accuracy or flexibility.


## Cloud-native architecture

Real-time analytics requires a scalable, high-performance, and cost-efficient database that can handle high-ingest rates and low-latency queries without overprovisioning. Tiger Cloud is designed for elasticity, enabling independent scaling of storage and compute, workload isolation, and intelligent data tiering.


### Independent storage and compute scaling

Real-time applications generate continuous data streams while requiring instant querying of both fresh and historical data. Traditional databases force users to pre-provision fixed storage, leading to unnecessary costs or unexpected limits. Tiger Cloud eliminates this constraint by dynamically scaling storage based on actual usage:



* Storage expands and contracts automatically as data is added or deleted, avoiding manual intervention.
* Usage-based billing ensures costs align with actual storage consumption, eliminating large upfront allocations.
* Compute can be scaled independently to optimize query execution, ensuring fast analytics across both recent and historical data.

With this architecture, databases grow alongside data streams, enabling seamless access to real-time and historical insights while efficiently managing storage costs.


### Workload isolation for real-time performance

Balancing high-ingest rates and low-latency analytical queries on the same system can create contention, slowing down performance. Tiger Cloud mitigates this by allowing read and write workloads to scale independently:



* The primary database efficiently handles both ingestion and real-time rollups without disruption.
* Read replicas scale query performance separately, ensuring fast analytics even under heavy workloads.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/compute-compute-seperation.png"
  alt=""
/>
</center>

This separation ensures that frequent queries on fresh data don’t interfere with ingestion, making it easier to support live monitoring, anomaly detection, interactive dashboards, and alerting systems.


### Intelligent data tiering for cost-efficient real-time analytics

Not all real-time data is equally valuable—recent data is queried constantly, while older data is accessed less frequently. Tiger Cloud can be configured to automatically tier data to cheaper bottomless object storage, ensuring that hot data remains instantly accessible, while historical data is still available.


<center>
<img
  class="main-content__illustration"
  width="80%"
  src="https://assets.timescale.com/docs/images/tiering.png"
  alt=""
/>
</center>



* **Recent, high-velocity data** stays in high-performance storage for ultra-fast queries.
* **Older, less frequently accessed data** is automatically moved to cost-efficient object storage but remains queryable and available for building continuous aggregates.

While many systems support this concept of data cooling, TimescaleDB ensures that the data can still be queried from the same hypertable regardless of its current location. For real-time analytics, this means applications can analyze live data streams without worrying about storage constraints, while still maintaining access to long-term trends when needed.


### Cloud-native database observability

Real-time analytics doesn’t just require fast queries—it requires the ability to understand why queries are fast or slow, where resources are being used, and how performance changes over time. That’s why Tiger Cloud is built with deep observability features, giving developers and operators full visibility into their database workloads.

At the core of this observability is Insights, Tiger Cloud’s built-in query monitoring tool. Insights captures
per-query
statistics from our whole fleet in real time, showing you exactly how your database is behaving under load. It tracks key metrics like execution time, planning time, number of rows read and returned, I/O usage, and buffer cache hit rates—not just for the database as a whole, but for each individual query.

Insights lets you do the following:



* Identify slow or resource-intensive queries instantly
* Spot long-term performance regressions or trends
* Understand query patterns and how they evolve over time
* See the impact of schema changes, indexes, or continuous aggregates on workload performance
* Monitor and compare different versions of the same query to optimize execution

All this is surfaced through an intuitive interface, available directly in Tiger Cloud, with no instrumentation or external monitoring infrastructure required.

Beyond query-level visibility, Tiger Cloud also exposes metrics around service resource consumption, compression, continuous aggregates, and data tiering, allowing you to track how data moves through the system—and how those background processes impact storage and query performance.

Together, these observability features give you the insight and control needed to operate a real-time analytics database at scale, with confidence, clarity, and performance you can trust**.**


## Ensuring reliability and scalability

Maintaining high availability, efficient resource utilization, and data durability is essential for real-time applications. Tiger Cloud provides robust operational features to ensure seamless performance under varying workloads.

* **High-availability (HA) replicas**: deploy multi-AZ HA replicas to provide fault tolerance and ensure minimal downtime. In the event of a primary node failure, replicas are automatically promoted to maintain service continuity.
* **Connection pooling**: optimize database connections by efficiently managing and reusing them, reducing overhead and improving performance for high-concurrency applications.
* **Backup and recovery**: leverage continuous backups, Point-in-Time Recovery (PITR), and automated snapshotting to protect against data loss. Restore data efficiently to minimize downtime in case of failures or accidental deletions.

These operational capabilities ensure Tiger Cloud remains reliable, scalable, and resilient, even under demanding real-time workloads.


## Conclusion

Real-time analytics is critical for modern applications, but traditional databases struggle to balance high-ingest performance, low-latency queries, and flexible data mutability. Tiger Cloud extends Postgres to solve this challenge, combining automatic partitioning, hybrid row-columnar storage, and intelligent compression to optimize both transactional and analytical workloads.

With continuous aggregates, hyperfunctions, and advanced query optimizations, Tiger Cloud ensures sub-second queries
even on massive datasets that combine current and historic data. Its cloud-native architecture further enhances scalability with independent compute and storage scaling, workload isolation, and cost-efficient data tiering—allowing applications to handle real-time and historical queries seamlessly.

For developers, this means building high-performance, real-time analytics applications without sacrificing SQL compatibility, transactional guarantees, or operational simplicity.

Tiger Cloud delivers the best of Postgres, optimized for real-time analytics.


===== PAGE: https://docs.tigerdata.com/about/pricing-and-account-management/ =====

# Billing and account management



As we enhance our offerings and align them with your evolving needs,
pricing plans provide more value, flexibility, and efficiency for your business.
Whether you're a growing startup or a well-established enterprise, our plans
are structured to support your journey towards greater success.

![Tiger Cloud pricing plans](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-pricing.svg)

This page explains pricing plans for Tiger Cloud, and how to easily manage your Tiger Data account.

Pricing plans give you:

* **Enhanced performance**: with increased CPU and storage capacities, your apps run smoother and more
  efficiently, even under heavy loads.
* **Improved scalability**: as your business grows, so do your demands. Pricing plans scale with
  you, they provide the resources and support you need at each stage of your growth. Scale up or down
  based on your current needs, ensuring that you only pay for what you use.
* **Better support**: access to enhanced support options, including production support and dedicated
  account management, ensures you have the help you need when you need it.
* **Greater flexibility**: we know that one size doesn't fit all. Pricing plans give you the
  flexibility to choose the features and support levels that best match your business
  and engineering requirements. The ability to add features like I/O boost and customize your pricing plan means you can tailor Tiger Cloud services to fit your specific needs.
* **Cost efficiency**: by aligning our pricing with the value delivered, we ensure that you get the most
  out of every dollar spent. Our goal is to help you achieve more with less.

It’s that simple! You don't pay for automated backups or networking costs, such as data ingest or egress.
There are no per-query fees, nor additional costs to read or write data. It's all completely transparent, easily understood, and up to you.

Using self-hosted TimescaleDB and our open-source products is still free.

If you create a Tiger Data account from AWS Marketplace, the pricing options are pay-as-you-go and annual commit. See [AWS pricing][aws-pricing] for details.

## Disaggregated, consumption-based compute and storage

With Tiger Cloud, you are not limited to pre-set compute and storage. Get as much as you need when
provisioning your services or later, as your needs grow.

* **Compute**: pay only for the compute resources you run. Compute is metered on an hourly
   basis, and you can [scale it up to 64,000 IOPS][change-compute] at any time. You can also [scale out using replicas][read-replication]
  as your application grows. We also provide services to help you lower your compute needs
  while improving query performance. Tiger Cloud is very efficient and generally needs less compute than other databases to deliver
  the same performance. The best way to size your needs is to sign up for a free trial and test
  with a realistic workload.

* **Storage**: pay only for the storage you consume. You have high-performance storage for more-accessed data, and
[low-cost bottomless storage in S3][data-tiering] for other data. The high-performance storage offers you up to 64 TB of compressed
(typically 80-100 TB uncompressed) data and is metered on your average GB consumption per hour. We can help you compress your data by up to 98% so you pay even less. For low-cost storage, Tiger Data charges only for the size of your data in S3 in the Apache Parquet format, regardless of whether it was compressed in Tiger Cloud before tiering. There are no additional expenses, such as data transfer or compute.
For easy upgrades, each service stores the TimescaleDB binaries. This contributes up to 900 MB to overall storage, which amounts to less than $.80/month in additional storage costs.

## Use Tiger Cloud for free

Are you just starting out with Tiger Cloud? On our Free pricing plan, you can create up to 2 zero-cost services with [limited resources][plan-features]. When a free service reaches the resource limit, it converts to a read-only state.

The Free pricing plan and services are currently in beta.

Ready to try a more feature-rich paid plan? Activate a 30-day free trial of our Performance (no credit card required) or Scale plan. After your trial ends, we may remove your data unless you’ve added a payment method.

After you have completed your 30-day trial period, choose the
[pricing plan][plan-features] that suits your business and engineering needs. And even when you upgrade from the Free pricing plan, you can still have up to 2 zero-cost services—or convert the ones you already have into standard ones, to have more resources.

If you want to try out features in a higher pricing plan before upgrading, contact us.

## Upgrade or downgrade your pricing plans at any time

You can upgrade or downgrade between the Free, Performance, and Scale plans
whenever you want using [Tiger Cloud Console][cloud-login]. To downgrade to the Free plan, you must only have free services running in your project.

If you switch your pricing plan mid-month,
your prices are prorated to when you switch. Your services are not interrupted when you switch, so
you can keep working without any hassle. To move to Enterprise, [get in touch with Tiger Data][contact-company].

## Monitor usage and costs

You keep track of your monthly usage in [Tiger Cloud Console][cloud-billing]. Console shows your
resource usage and dashboards with performance insights. This allows you to closely monitor your
services’ performance, and any need to scale your services or upgrade your pricing plan.

Console also shows your month-to-date accrued charges, as well as a forecast of your expected
month-end bill. Your previous invoices are also available as PDFs for download.

You are charged for all active services in your account, even if you are not actively using them. To reduce costs, pause or delete your unused services.

## Tiger Data support

Tiger Data runs a global support organization with Customer Satisfaction (CSAT) scores above 99%.
Support covers all timezones, and is fully staffed at weekend hours.

All paid pricing plans have free Developer Support through email with a target response time of 1 business
day; we are often faster. If you need 24x7 responsiveness, talk to us about
[Production Support][production-support].

## Charging for HA and read replicas

HA and read replicas are both charged at the same rate as your primary services, based on the
compute and primary storage consumed by your replicas. Data tiered to our bottomless storage
tier is shared by all database replicas; replicas accessing tiered storage do not add to your
bill.

## Charging over regions

Storage is priced the same across all regions. However, compute prices vary depending on the
region. This is because our cloud provider (AWS) prices infrastructure differently based on region.

## Features included in each pricing plan

The available pricing plans are:

* **Free**: for small non-production projects.
* **Performance**: for cost-focused, smaller projects. No credit card required to start.
* **Scale**: for developers handling critical and demanding apps.
* **Enterprise**: for enterprises with mission-critical apps.

The Free pricing plan and services are currently in beta.

The features included in each [pricing plan][pricing-plans] are:

| Feature                                                       | Free                              | Performance                           | Scale                                         | Enterprise                                      |
|---------------------------------------------------------------|-----------------------------------|----------------------------------------|------------------------------------------------|--------------------------------------------------|
| **Compute and storage**                                       |                                   |                                        |                                                |                                                  |
| Number of services	                                    | Up to 2 free services             | Up to 2 free and 4 standard services 	 | Up to 2 free and and unlimited standard services	  | Up to 2 free and and unlimited standard services |
| CPU limit per service                                  | Shared                            | 	Up to 8 CPU	                          | Up to 32 CPU	                                  | Up to 64 CPU                                     |
| Memory limit per service                               | Shared                            | 	Up to 32 GB                           | 	Up to 128 GB                                  | 	Up to 256 GB                                    |
| Storage limit per service	                             | 750 MB                            | Up to 16 TB	                           | Up to 16 TB	                                   | Up to 64 TB                                      |
| Bottomless storage on S3	                                     |                                   |                                        | 	Unlimited	                                    | Unlimited                                        |
| Independently scale compute and storage	                      |                                   | Standard services only                 | 	Standard services only	                           | Standard services only                           |
| **Data services and workloads**                               |                                   |                                        |                                                |
| Relational                                                    | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Time-series                                                   | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Vector search                                                 | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| AI workflows (coming soon)                                    | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Cloud SQL editor                                              | 3 seats                           | 3 seats                                | 10 seats                                       | 20 seats                                         |
| Charts                                                        | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Dashboards                                                    |                                   | 2                                      | Unlimited                                      | Unlimited                                        |
| **Storage and performance**                                   |                                   |                                        |                                                |                                                  |
| IOPS                                                          | Shared	                           | 	3,000 - 5,000	                        | 5,000 - 8,000                                  | 5,000 - 8,000                                    |
| Bandwidth (autoscales)	                                       | Shared                            | 125 - 250 Mbps                         | 	250 - 500 Mbps                                | 	Up to 500 mbps                                  |
| I/O boost	                                                    |                                   |                                        | 	Add-on: <br/>Up to 16K IOPS, 1000 Mbps BW	    | Add-on: <br/>Up to 32K IOPS, 4000 Mbps BW        |
| **Availability and monitoring**                               |                                   |                                        |                                                |                                                  |
| High-availability replicas <br/>(Automated multi-AZ failover) |                                   | ✓                                      | ✓                                              | ✓                                                |
| Read replicas		                                               |                                   |                                        | ✓                                              | ✓                                                |
| Cross-region backup                                           |                                   |                                        |                                                | ✓                                                |
| Backup reports                                                |                                   |                                        | 14 days                                        | 14 days                                          |
| Point-in-time recovery and forking                            | 	1 day                            | 	3 days                                | 14 days                                        | 14 days                                          |
| Performance insights                                          | Limited                           | ✓                                      | ✓                                              | ✓                                                |
| Metrics and log exporters	                                    |                                   |                                        | ✓                                              | ✓                                                |
| **Security and compliance**                                   |                                   |                                        |                                                |                                                  |
| Role-based access                                             | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| End-to-end encryption                                         | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Private Networking (VPC)                                      |                                   | 1 multi-attach VPC	                    | Unlimited multi-attach VPCs                    | 	Unlimited multi-attach VPCs                     |
| AWS Transit Gateway                                           |                                   |                                        | ✓                                              | ✓                                                |
| [HIPAA compliance][hipaa-compliance]                          |                                   |                                        |                                                | ✓                                                |
| IP address allow list                                         | 1 list with up to 10 IP addresses | 1 list with up to 10 IP addresses      | Up to 10 lists with up to 10 IP addresses each | Up to 10 lists with up to 100 IP addresses each  |
| Multi-factor authentication                                   | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Federated authentication (SAML)			                            |                                   |                                        |                                                | ✓                                                |
| SOC 2 Type 2 report		                                         |                                   |                                        | ✓                                              | ✓                                                |
| Penetration testing report                                    |                                   |                                        |                                                | ✓                                                |
| Security questionnaire and review                             |                                   |                                        |                                                | ✓                                                |
| Pay by invoice                                                |                                   | 	Available at minimum spend	           | Available at minimum spend                     | ✓                                                |
| [Uptime SLAs][commercial-sla]                                 | 	                                 | 	Standard                              | 	Standard                                      | 	Enterprise                                      |
| **Support and technical services**                            |                                   |                                        |                                                |                                                  |
| Community support                                             | ✓                                 | ✓                                      | ✓                                              | ✓                                                |
| Email support                                                 |                                   | ✓                                      | ✓                                              | ✓                                                |
| Production support                                            | 	                                 | 	Add-on                                | 	Add-on                                        | ✓                                                |
| Named account manager                                         |                                   |                                        |                                                | ✓                                                |
| JOIN services (Jumpstart Onboarding and INtegration)          |                                   |                                        | Available at minimum spend                     | ✓                                                |

For a personalized quote, [get in touch with Tiger Data][contact-company].

## Example billing calculation

You are billed at the end of each month in arrears, based on your actual usage that month. Your monthly invoice
includes an itemized cost accounting for each Tiger Cloud service and any additional charges.

Tiger Cloud charges are based on consumption:

- **Compute**: metered on an hourly basis. You can scale compute up and down at any time.
- **Storage**: metered based on your average GB consumption per hour. Storage grows and shrinks automatically
  with your data.

Your monthly price for compute and storage is computed similarly. For example, over the last month your
Tiger Cloud service has been running compute for 500 hours total:
- 375 hours with 2 CPU
- 125 hours 4 CPU

**Compute cost** = (`375` x `hourly price for 2 CPU`) + (`125` x `hourly price for 4 CPU`)

Some add-ons such as tiered storage, HA replicas, and connection pooling may incur
additional charges. These charges are clearly marked in your billing snapshot in Tiger Cloud Console.

## Manage your Tiger Cloud pricing plan

You handle all details about your Tiger Cloud project including updates to your pricing plan,
payment methods, and add-ons in the [billing section in Tiger Cloud Console][cloud-billing]:

<img class="main-content__illustration"
src="https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-billing.png"
alt="Adding a payment method in Tiger"/>

- **Details**: an overview of your pricing plan, usage, and payment details. You can add up
  to three credit cards to your `Wallet`. If you prefer to pay by invoice,
  [contact Tiger Data][contact-company] and ask to change to corporate billing.

- **History**: the list of your downloadable Tiger Cloud invoices.
- **Emails**: the addresses Tiger Data uses to communicate with you. Payment
  confirmations and alerts are sent to the email address you signed up with.
  Add another address to send details to other departments in your organization.

- **Pricing plan**: choose the pricing plan supplying the [features][plan-features] that suit your business and
  engineering needs.

- **Add-ons**: add `Production support` and improved database performance for mission-critical workloads.

## AWS Marketplace pricing

When you get Tiger Cloud at AWS Marketplace, the following pricing options are available:

- **Pay-as-you-go**: your consumption is calculated at the end of the month and included in your AWS invoice. No upfront costs, standard Tiger Cloud rates apply.
- **Annual commit**: your consumption is calculated at the end of the month ensuring predictable pricing and seamless billing through your AWS account. We confirm the contract terms with you before finalizing the commitment.


===== PAGE: https://docs.tigerdata.com/about/changelog/ =====

# Changelog

All the latest features and updates to Tiger Cloud.

## TimescaleDB 2.22.1 – configurable indexing, enhanced partitioning, and faster queries
<Label type="date">October 10, 2025</Label>

[TimescaleDB 2.22.1](https://github.com/timescale/timescaledb/releases) introduces major performance and flexibility improvements across indexing, compression, and query execution. TimescaleDB 2.22.1 was released on September 30th and is now available to all users of Tiger.

### Highlighted features

* **Configurable sparse indexes:** manually configure sparse indexes (min-max or bloom) on one or more columns of compressed hypertables, optimizing query performance for specific workloads and reducing I/O. In previous versions, these were automatically created based on heuristics and could not be modified.

* **UUIDv7 support:** native support for UUIDv7 for both compression and partitioning. UUIDv7 embeds a time component, improving insert locality and enabling efficient time-based range queries while maintaining global uniqueness.

    * **Vectorized UUID compression:** new vectorized compression for UUIDv7 columns doubles query performance and improves storage efficiency by up to 30%.

    * **UUIDv7 partitioning:** hypertables can now be partitioned on UUIDv7 columns, combining time-based chunking with globally unique IDs—ideal for large-scale event and log data.

* **Multi-column SkipScan:** expands SkipScan to support multiple distinct keys, delivering millisecond-fast deduplication and `DISTINCT ON` queries across billions of rows. Learn more in our [blog post](https://www.tigerdata.com/blog/skipscan-in-timescaledb-why-distinct-was-slow-how-we-built-it-and-how-you-can-use-it) and [documentation](https://docs.tigerdata.com/use-timescale/latest/query-data/skipscan/).
* **Compression improvements:** default `segmentby` and `orderby` settings are now applied at compression time for each chunk, automatically adapting to evolving data patterns for better performance. This was previously set at the hypertable level and fixed across all chunks.

### Deprecations

The experimental Hypercore Table Access Method (TAM) has been removed in this release following advancements in the columnstore architecture.

For a comprehensive list of changes, refer to the TimescaleDB [2.22](https://github.com/timescale/timescaledb/releases/tag/2.22.0) & [2.22.1](https://github.com/timescale/timescaledb/releases/tag/2.22.1) release notes.

## Kafka Source Connector (beta)
<Label type="date">September 19, 2025</Label>

The new [Kafka Source Connector](https://docs.tigerdata.com/migrate/latest/livesync-for-kafka/) enables you to connect your existing Kafka clusters directly to Tiger Cloud and ingest data from Kafka topics into hypertables. Developers often build proxies or run JDBC Sink Connectors to bridge Kafka and Tiger Cloud, which is error-prone and time-consuming. With the Kafka Source Connector, you can seamlessly start ingesting your Kafka data natively without additional middleware.

- Supported formats: AVRO
- Supported platforms: Confluent Cloud and Amazon Managed Streaming for Apache Kafka

![Kafka source connector in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/kafka-source-connector-tiger-data.png)

![Kafka source connector streaming in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/kafka-source-connector-streaming.png)

## Phased update rollouts, `pg_cron`, larger compute options, and backup reports
<Label type="date">September 12, 2025</Label>

### 🛡️ Phased rollouts for TimescaleDB minor releases

Starting with TimescaleDB 2.22.0, minor releases will now roll out in phases. Services tagged `#dev` will get upgraded first, followed by `#prod` after 21 days. This gives you time to validate upgrades in `#dev` before they reach `#prod` services. [Subscribe](https://status.timescale.com/?__hstc=231067136.cc62bfc44030d30e3b1c3d1bc78c9cab.1750169693582.1757669826871.1757685085606.116&__hssc=231067136.4.1757685085606&__hsfp=2801608430) to get an email notification before your `#prod` service is upgraded. See [Maintenance and upgrades](https://docs.tigerdata.com/use-timescale/latest/upgrades/) for details.

### ⏰ pg_cron extension

`pg_cron` is now available on Tiger Cloud! With `pg_cron`, you can:
- Schedule SQL commands to run automatically—like generating weekly sales reports or cleaning up old log entries every night at 2 AM.
- Automate routine maintenance tasks such as refreshing materialized views hourly to keep dashboards current.
- Eliminate external cron jobs and task schedulers, keeping all your automation logic within PostgreSQL.

To enable `pg_cron` on your service, contact our support team. We're working on making this self-service in future updates.

### ⚡️ Larger compute options: 48 and 64 CPU

For the most demanding workloads, you can now create services with 48 and 64 CPUs. These options are only available on our Enterprise plan, and they're dedicated instances that are not shared with other customers.

![CPU options in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-cloud-cpu-options.png)

### 📋 Backup report for compliance

Scale and Enterprise customers can now see a list of their backups in Tiger Cloud Console. For customers with SOC 2 or other compliance needs, this serves as auditable proof of backups.

![Backup reports in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/backup-history-tiger-cloud.png)

### 🗺️ New router for Tiger Cloud Console

The UI just got snappier and easier to navigate with improved interlinking. For example, click an object in the `Jobs` page to see what hypertable the job is associated with.

## New data import wizard
<Label type="date">September 5, 2025</Label>

To make navigation easier, we’ve introduced a cleaner, more intuitive UI for data import. It highlights the most common and recommended option, PostgreSQL Dump & Restore, while organizing all import options into clear categories, to make navigation easier.

The new categories include:
- **PostgreSQL Dump & Restore**
- **Upload Files**: CSV, Parquet, TXT
- **Real-time Data Replication**: source connectors
- **Migrations & Other Options**

  ![Data import in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/data-import-wizard-in-tiger-cloud.png)

A new data import component has been added to the overview dashboard, providing a clear view of your imports. This includes quick start, in-progress status, and completed imports:

  ![Overview dashboard in Tiger Cloud](https://assets.timescale.com/docs/images/tiger-cloud-console/service-dashboard-tiger-cloud.png)

## 🚁 Enhancements to the Postgres source connector
<Label type="date">August 28, 2025</Label>

- **Easy table selection**: You can now sync the complete source schema in one go. Select multiple tables from the
   drop-down menu and start the connector.
- **Sync metadata**: Connectors now display the following detailed metadata:
    - `Initial data copy`: The number of rows copied at any given point in time.
    - `Change data capture`: The replication lag represented in time and data size.
- **Improved UX design**: In-progress syncs with separate sections showing the tables and metadata for
   `initial data copy` and `change data capture`, plus a dedicated tab where you can add more tables to the connector.

   ![Connectors UX](https://assets.timescale.com/docs/images/tiger-cloud-console/connectors-new-ui.png)

## 🦋 Developer role GA and hypertable transformation in Console
<Label type="date">August 21, 2025</Label>

### Developer role (GA)

The [Developer role in Tiger Cloud](https://docs.tigerdata.com/use-timescale/latest/security/members/) is now
generally available. It’s a project‑scoped permission set that lets technical users build and
operate services, create or modify resources, run queries, and use observability—without admin or billing access.
This enforces least‑privilege by default, reducing risk and audit noise, while keeping governance with Admins/Owners and
billing with Finance. This means faster delivery (fewer access escalations), protected sensitive settings,
and clear boundaries, so the right users can ship changes safely, while compliance and cost control remain intact.

### Transform a table to a hypertable from the Explorer

In Console, you can now easily create hypertables from your regular Postgres tables directly from the Explorer.
Clicking on any Postgres table shows an option to open up the hypertable action. Follow the simple steps to set up your
partition key and transform the table to a hypertable.

![Transform a table to a hypertable](https://assets.timescale.com/docs/images/table_to_hypertable_1.png)

![Transform a table to a hypertable](https://assets.timescale.com/docs/images/table_to_hypertable_2.png)

## Cross-region backups, Postgres options, and onboarding
<Label type="date">August 14, 2025</Label>

### Cross-region backups

You can now store backups in a different region than your service, which improves resilience and helps meet enterprise compliance requirements. Cross‑region backups are available on our Enterprise plan for free at launch; usage‑based billing may be introduced later. For full details, please [see the docs](https://docs.tigerdata.com/use-timescale/latest/backup-restore/#enable-cross-region-backup).

### Standard Postgres instructions for onboarding
We have added basic instructions for INSERT, UPDATE, DELETE commands to the Tiger Cloud console.  It's now shown as an option in the Import Data page.

### Postgres-only service type
In Tiger Cloud, you now have an option to choose Postgres-only in the service creation flow. Just click `Looking for plan PostgreSQL?` on the `Service Type` screen.

## Viewer role GA, EXPLAIN plans, and chunk index sizes in Explorer
<Label type="date">July 31, 2025</Label>

### GA release of the viewer role in role-based access

The viewer role is now **generally available** for all projects and
organizations. It provides **read-only access** to services, metrics, and logs
without modify permissions. Viewers **cannot** create, update, or delete
resources, nor manage users or billing. It's ideal for auditors, analysts, and
cross-functional collaborators who need visibility but not control.

### EXPLAIN plans in Insights

You can now find automatically generated EXPLAIN plans on queries that take
longer than 10 seconds within Insights. EXPLAIN plans can be very useful to
determine how you may be able to increase the performance of your queries.

### Chunk index size in Explorer

Find the index size of hypertable chunks in the Explorer.
This information can be very valuable to determine if a hypertable's chunk size
is properly configured.

## TimescaleDB v2.21 and catalog objects in the Console Explorer
<Label type="date">July 25, 2025</Label>

### 🏎️ TimescaleDB v2.21—ingest millions of rows/second and faster columnstore UPSERTs and DELETEs

TimescaleDB v2.21 was released on July 8 and is now available to all developers on Tiger Cloud.

Highlighted features in TimescaleDB v2.21 include:
- **High-scale ingestion performance (tech preview)**: introducing a new approach that compresses data directly into the columnstore during ingestion, demonstrating over 1.2M rows/second in tests with bursts over 50M rows/second. We are actively seeking design partners for this feature.
- **Faster data updates (UPSERTs)**: columnstore UPSERTs are now 2.5x faster for heavily constrained tables, building on the 10x improvement from v2.20.
- **Faster data deletion**: DELETE operations on non-segmentby columns are 42x faster, reducing I/O and bloat.
- **Reduced bloat after recompression**: optimized recompression processes lead to less bloat and more efficient storage.
- **Enhanced continuous aggregates**:
  - Concurrent refresh policies enable multiple continuous aggregates to update concurrently.
  - Batched refreshes are now enabled by default for more efficient processing.
- **Complete chunk management**: full support for splitting columnstore chunks, complementing the existing merge capabilities.

For a comprehensive list of changes, refer to the [TimescaleDB v2.21 release notes](https://github.com/timescale/timescaledb/releases/tag/2.21.0).

### 🔬 Catalog objects available in the Console Explorer

You can now view catalog objects in the Console Explorer. Check out the internal schemas for PostgreSQL and TimescaleDB to better understand the inner workings of your database. To turn on/off visibility, select your service in Tiger Cloud Console, then click `Explorer` and toggle `Show catalog objects`.

![Explore catalog objects](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-cloud-explorer-catalog-objects.png)

## Iceberg Destination Connector (Tiger Lake)
<Label type="date">July 18, 2025</Label>

We have released a beta Iceberg destination connector that enables Scale and Enterprise users to integrate Tiger Cloud services with Amazon S3 tables. This enables you to connect Tiger Cloud to data lakes seamlessly. We are actively developing several improvements that will make the overall data lake integration process even smoother.

To use this feature, select your service in Tiger Cloud Console, then navigate to `Connectors` and select the `Amazon S3 Tables` destination connector. Integrate the connector to your S3 table bucket by providing the ARN roles, then simply select the tables that you want to sync into S3 tables. See the [documentation](https://docs.tigerdata.com/use-timescale/latest/tigerlake/) for details.

## 🔆Console just got better
<Label type="date">July 11, 2025</Label>

### ✏️ Editable jobs in Console

You can now edit jobs directly in Console! We've added the handy pencil icon in the top right corner of any
job view. Click a job, hit the edit button, then make your changes. This works for all jobs, even user-defined ones.
Tiger Cloud jobs come with custom wizards to guide you through the right inputs. This means you can spot and fix
issues without leaving the UI - a small change that makes a big difference!

![Edit jobs in console](https://assets.timescale.com/docs/images/console-jobs-edit.png)

### 📊 Connection history

Now you can see your historical connection counts right in the Connections tab! This helps spot those pesky connection
management bugs before they impact your app. We're logging max connections every hour (sampled every 5 mins) and might
adjust based on your feedback. Just another way we're making the Console more powerful for troubleshooting.

![View connection history in console](https://assets.timescale.com/docs/images/console-connection-history.png)


### 🔐 New in Public Beta: Read-Only Access through RBAC

We’ve just launched Read/Viewer-only access for Tiger Cloud projects into public beta!

You can now invite users with view-only permissions — perfect for folks who need to see dashboards, metrics,
and query results, without the ability to make changes.

This has been one of our most requested RBAC features, and it's a big step forward in making Tiger Cloud more secure and
collaborative.

No write access. No config changes. Just visibility.

In Console, Go to `Project Settings` > `Users & Roles` to try it out, and let us know what you think!

## 👀 Super useful doc updates
<Label type="date">July 4, 2025</Label>

### Updates to instructions for livesync

In the Console UI, we have clarified the step-by-step procedure for setting up your livesync from self-hosted installations by:
- Adding definitions for some flags when running your Docker container.
- Including more detailed examples of the output from the table synchronization list.

### New optional argument for add_continuous_aggregate_policy API

Added the new `refresh_newest_first` optional argument that controls the order of incremental refreshes.

## 🚀 Multi-command queries in SQL editor, improved job page experience, multiple AWS Transit Gateways, and a new service creation flow
<Label type="date">June 20, 2025</Label>

### Run multiple statements in SQL editor
Execute complex queries with multiple commands in a single run—perfect for data transformations, table setup, and batch operations.

### Branch conversations in SQL assistant
Start new discussion threads from any point in your SQL assistant chat to explore different approaches to your data questions more easily.

### Smarter results table
- Expand JSON data instantly: turn complex JSON objects into readable columns with one click—no more digging through nested data structures.
- Filter with precision: use a new smart filter to pick exactly what you want from a dropdown of all available values.

### Jobs page improvements
Individual job pages now display their corresponding configuration for TimescaleDB job types—for example, columnstore, retention, CAgg refreshes, tiering, and others.

### Multiple AWS Transit Gateways

You can now connect multiple AWS Transit Gateways, when those gateways use overlapping CIDRs. Ideal for teams with zero-trust policies, this lets you keep each network path isolated.

How it works: when you create a new peering connection, Tiger Cloud reuses the existing Transit Gateway if you supply the same ID—otherwise it automatically creates a new, isolated Transit Gateway.

### Updated service creation flow

The new service creation flow makes the choice of service type clearer. You can now create distinct types with Postgres extensions for real-time analytics (TimescaleDB), AI (pgvectorscale, pgai), and RTA/AI hybrid applications.

![Create a Tiger Cloud service](https://assets.timescale.com/docs/images/tiger-cloud-console/create-tiger-cloud-service.png)

## ⚙️ Improved Terraform support and TimescaleDB v2.20.3
<Label type="date">June 13, 2025</Label>

### Terraform support for Exporters and AWS Transit Gateway

The latest version of the Timescale Terraform provider (2.3.0) adds support for:
- Creating and attaching observability exporters to your services.
- Securing the connections to your Timescale Cloud services with AWS Transit Gateway.
- Configuring CIDRs for VPC and AWS Transit Gateway connections.

Check the [Timescale Terraform provider documentation](https://registry.terraform.io/providers/timescale/timescale/latest/docs) for more details.

### TimescaleDB v2.20.3

This patch release for TimescaleDB v2.20 includes several bug fixes and minor improvements.
Notable bug fixes include:
- Adjustments to SkipScan costing for queries that require a full scan of indexed data.
- A fix for issues encountered during dump and restore operations when chunk skipping is enabled.
- Resolution of a bug related to dropped "quals" (qualifications/conditions) in SkipScan.

For a comprehensive list of changes, refer to the [TimescaleDB 2.20.3 release notes](https://github.com/timescale/timescaledb/releases/tag/2.20.3).

## 🧘 Read replica sets, faster tables, new anthropic models, and VPC support in data mode
<Label type="date">June 6, 2025</Label>

### Horizontal read scaling with read replica sets

[Read replica sets](https://docs.timescale.com/use-timescale/latest/ha-replicas/read-scaling/) are an improved version of read replicas. They let you scale reads horizontally by creating up to 10 replica nodes behind a single read endpoint. Just point your read queries to the endpoint and configure the number of replicas you need without changing your application logic. You can increase or decrease the number of replicas in the set dynamically, with no impact on the endpoint.

Read replica sets are used to:

- Scale reads for read-heavy workloads and dashboards.
- Isolate internal analytics and reporting from customer-facing applications.
- Provide high availability and fault tolerance for read traffic.

All existing read replicas have been automatically upgraded to a replica set with one node—no action required. Billing remains the same.

Read replica sets are available for all Scale and Enterprise customers.

![Create a read replica set in Timescale Console](https://assets.timescale.com/docs/images/create-read-replica-set-timescale-console.png)

### Faster, smarter results tables in data mode

We've completely rebuilt how query results are displayed in the data mode to give you a faster, more powerful way to work with your data. The new results table can handle millions of rows with smooth scrolling and instant responses when you sort, filter, or format your data. You'll find it today in notebooks and presentation pages, with more areas coming soon.

What's new:

- **Your settings stick around**: when you customize how your table looks—applying filters, sorting columns, or formatting data—those settings are automatically saved. Switch to another tab and come back, and everything stays exactly how you left it.
- **Better ways to find what you need**: filter your results by any column value, with search terms highlighted so you can quickly spot what you're looking for. The search box is now available everywhere you work with data.
- **Export exactly what you want**: download your entire table or just select the specific rows and columns you need. Both CSV and Excel formats are supported.
- **See patterns in your data**: highlight cells based on their values to quickly spot trends, outliers, or important thresholds in your results.
- **Smoother navigation**: click any row number to see the full details in an expanded view. Columns automatically resize to show your data clearly, and web links in your results are now clickable.

As a result, working with large datasets is now faster and more intuitive. Whether you're exploring millions of rows or sharing results with your team, the new table keeps up with how you actually work with data.

### Latest anthropic models added to SQL assistant

Data mode's [SQL assistant](https://docs.timescale.com/getting-started/latest/run-queries-from-console/#sql-assistant) now supports Anthropic's latest models:

- Sonnet 4
- Sonnet 4 (extended thinking)
- Opus 4
- Opus 4 (extended thinking)

### VPC support for passwordless data mode connections

We previously made it much easier to connect newly created services to Timescale’s [data mode](https://docs.timescale.com/getting-started/latest/run-queries-from-console/#data-mode). We have now expanded this functionality to services using a VPC.

## 🕵🏻️ Enhanced service monitoring, TimescaleDB v2.20, and livesync for Postgres
<Label type="date">May 30, 2025</Label>

### Updated top-level navigation - Monitoring tab

In Timescale Console, we have consolidated multiple top-level service information tabs into the single Monitoring tab.
This tab houses information previously displayed in the Recommendations, Jobs, Connections, Metrics, Logs,
and `Insights` tabs.

![Insights](https://assets.timescale.com/docs/images/insights_overview_timescale.png)

### Monitor active connections

In the `Connections` section under `Monitoring`, you can now see information like the query being run, the application
name, and duration for all current connections to a service.

![Connections](https://assets.timescale.com/docs/images/console-monitoring-connections.png)

The information in `Connections` enables you to debug misconfigured applications, or
cancel problematic queries to free up other connections to your database.

### TimescaleDB v2.20 - query performance and faster data updates

All new services created on Timescale Cloud are created using
[TimescaleDB v2.20](https://github.com/timescale/timescaledb/releases/tag/2.20.0). Existing services will be
automatically upgraded during their maintenance window.

Highlighted features in TimescaleDB v2.20 include:
* Efficiently handle data updates and upserts (including backfills, that are now up to 10x faster).
* Up to 6x faster point queries on high-cardinality columns using new bloom filters.
* Up to 2500x faster DISTINCT operations with SkipScan, perfect for quickly getting a unique list or the latest reading
  from any device, event, or transaction.
* 8x more efficient Boolean column storage with vectorized processing, resulting in 30-45% faster queries.
* Enhanced developer flexibility with continuous aggregates now supporting window and mutable functions, plus
  customizable refresh orders.

### Postgres 13 and 14 deprecated on Tiger Cloud

[TimescaleDB version 2.20](https://github.com/timescale/timescaledb/releases/tag/2.20.0) is not compatible with Postgres versions v14 and below.
TimescaleDB 2.19.3 is the last bug-fix release for Postgres 14. Future fixes are for
Postgres 15+ only. To continue receiving critical fixes and security patches, and to take
advantage of the latest TimescaleDB features, you must upgrade to Postgres 15 or newer.
This deprecation affects all Tiger Cloud services currently running Postgres 13 or
Postgres 14.

The timeline for the Postgres 13 and 14 deprecation is as follows:

- **Deprecation notice period begins**: starting in early June 2025, you will receive email communication.
- **Customer self-service upgrade window**: June 2025 through September 14, 2025. We strongly encourage you to
  [manually upgrade Postgres](https://docs.tigerdata.com/use-timescale/latest/upgrades/#manually-upgrade-postgresql-for-a-service)
  during this period.
- **Automatic upgrade deadline**: your service will be
  [automatically upgraded](https://docs.timescale.com/use-timescale/latest/upgrades/#automatic-postgresql-upgrades-for-a-service)
  from September 15, 2025.

### Enhancements to livesync for Postgres

You now can:
* Edit a running livesync to add and drop tables from an existing configuration:
  - For existing tables, Timescale Console stops the livesync while keeping the target table intact.
  - Newly added tables sync their existing data and transition into the Change Data Capture (CDC) state.
* Create multiple livesync instances for Postgres per service. This is an upgrade from our initial launch which
  limited users to one LiveSync per service.

  This enables you to sync data from multiple Postgres source databases into a single Timescale Cloud service.
* No more hassle looking up schema and table names for livesync configuration from the source. Starting today, all
  schema and table names are available in a dropdown menu for seamless source table selection.

## ➕ More storage types and IOPS
<Label type="date">May 22, 2025</Label>

### 🚀 Enhanced storage: scale to 64 TB and 32,000 IOPS

We're excited to introduce enhanced storage, a new storage type in Timescale Cloud that significantly boosts both capacity and performance. Designed for customers with mission-critical workloads.

With enhanced storage, Timescale Cloud now supports:
- Up to 64 TB of storage per Timescale Cloud service (4x increase from the previous limit)
- Up to 32,000 IOPS, enabling high-throughput ingest and low-latency queries

Powered by AWS io2 volumes, enhanced storage gives your workloads the headroom they need—whether you're building financial data pipelines, developing IoT platforms, or processing billions of rows of telemetry. No more worrying about storage ceilings or IOPS bottlenecks.
Enable enhanced storage in Timescale Console under `Operations` → `Compute & Storage`. Enhanced storage is currently available on the Enterprise pricing plan only. [Learn more here](https://docs.timescale.com/use-timescale/latest/data-tiering/enabling-data-tiering/).

![I/O boost in Timescale Cloud](https://assets.timescale.com/docs/images/io-boost-timescale-cloud.png)

## ↔️ New export and import options
<Label type="date">May 15, 2025</Label>

### 🔥 Ship TimescaleDB metrics to Prometheus

We’re excited to release the Prometheus Exporter for Timescale Cloud, making it easy to ship TimescaleDB metrics to your Prometheus instance.
With the Prometheus Exporter, you can:

- Export TimescaleDB metrics like CPU, memory, and storage
- Visualize usage trends with your own Grafana dashboards
- Set alerts for high CPU load, low memory, or storage nearing capacity

To get started, create a Prometheus Exporter in the Timescale Console, attach it to your service, and configure Prometheus to scrape from the exposed URL. Metrics are secured with basic auth.
Available on Scale and Enterprise plans. [Learn more here](https://docs.timescale.com/use-timescale/latest/metrics-logging/metrics-to-prometheus/).

![Prometheus export user interface](https://assets.timescale.com/docs/images/timescale-create-prometheus-exporter.png)

### 📥 Import text files into Postgres tables
Our import options in Timescale Console have expanded to include local text files.  You can add the content of multiple text files (one file per row) into a Postgres table for use with Vectorizers while creating embeddings for evaluation and development. This new option is located in Service > Actions > Import Data.

## 🤖 Automatic document embeddings from S3 and a sample dataset for AI testing
<Label type="date">May 09, 2025</Label>

### Automatic document embeddings from S3

pgai vectorizer now supports automatic document vectorization. This makes it dramatically easier to build RAG and semantic search applications on top of unstructured data stored in Amazon S3. With just a SQL command, developers can create, update, and synchronize vector embeddings from a wide range of document formats—including PDFs, DOCX, XLSX, HTML, and more—without building or maintaining complex ETL pipelines.

Instead of juggling multiple systems and syncing metadata, vectorizer handles the entire process: downloading documents from S3, parsing them, chunking text, and generating vector embeddings stored right in Postgres using pgvector. As documents change, embeddings stay up-to-date automatically—keeping your Postgres database the single source of truth for both structured and semantic data.

![create a vectorizer](https://assets.timescale.com/docs/images/console-create-a-vectorizer.png )

### Sample dataset for AI testing

You can now import a dataset directly from Hugging Face using Timescale Console. This dataset is ideal for testing vectorizers, you find it in the Import Data page under the Service > Actions tab.

![hugging face sample data](https://assets.timescale.com/docs/images/console-import-huggingface-data.png)

## 🔁 Livesync for S3 and passwordless connections for data mode
<Label type="date">April 25, 2025</Label>

### Livesync for S3 (beta)

[Livesync for S3](https://docs.timescale.com/migrate/latest/livesync-for-s3/) is our second livesync offering in
Timescale Console, following livesync for Postgres. This feature helps users sync data in their S3 buckets to a
Timescale Cloud service, and simplifies data importing. Livesync handles both existing and new data in real time,
automatically syncing everything into a Timescale Cloud service. Users can integrate Timescale Cloud alongside S3, where
S3 stores data in raw form as the source for multiple destinations.

![Timescale Console new livesync](https://assets.timescale.com/docs/images/livesync-s3-start-new-livesync.png)

With livesync, users can connect Timescale Cloud with S3 in minutes, rather than spending days setting up and maintaining
an ingestion layer.

![Timescale Console livesync view status](https://assets.timescale.com/docs/images/livesync-s3-view-status.png)

### UX improvements to livesync for Postgres

In [livesync for Postgres](https://docs.timescale.com/migrate/latest/livesync-for-postgresql/), getting started
requires setting the `WAL_LEVEL` to `logical`, and granting specific permissions to start a publication
on the source database. To simplify this setup process, we have added a detailed two-step checklist with comprehensive
configuration instructions to Timescale Console.

![Timescale Console livesync Postgres instructions](https://assets.timescale.com/docs/images/livesync-postgres-console-config-instuctions.png)

### Passwordless data mode connections

We’ve made connecting to your Timescale Cloud services from [data mode](https://docs.timescale.com/getting-started/latest/run-queries-from-console/#connect-to-your-timescale-cloud-service-in-the-data-mode)
in Timescale Console even easier! All new services created in Timescale Cloud are now automatically accessible from
data mode without requiring you to enter your service credentials. Just open data mode, select your service, and
start querying.

![Timescale Console passwordless data mode](https://assets.timescale.com/docs/images/data-mode-connections.png)

We will be expanding this functionality to existing services in the coming weeks (including services using VPC peering),
so stay tuned.


## ☑️ Embeddings spot checks, TimescaleDB v2.19.3, and new models in SQL Assistant
<Label type="date">April 18, 2025</Label>

### Embeddings spot checks

In Timescale Cloud, you can now quickly check the quality of the embeddings from the vectorizers' outputs. Construct a similarity search query with additional filters on source metadata using a simple UI. Run the query right away, or copy it to the SQL editor or data mode and further customize it to your needs. Run the check in Timescale Console > `Services` > `AI`:

![Embedding Quality Inspection](https://assets.timescale.com/docs/images/ai-spot-checks.png)

### TimescaleDB v2.19.3

New services created in Timescale Cloud now use TimescaleDB v2.19.3. Existing services are in the process of being automatically upgraded to this version.

This release adds a number of bug fixes including:

- Fix segfault when running a query against columnstore chunks that group by multiple columns, including UUID segmentby columns.
- Fix hypercore table access method segfault on DELETE operations using a segmentby column.

### New OpenAI, Llama, and Gemini models in SQL Assistant

The data mode's SQL Assistant now includes support for the latest models from OpenAI and Llama: GPT-4.1 (including mini and nano) and Llama 4 (Scout and Maverick). Additionally, we've added support for Gemini models, in particular Gemini 2.0 Nano and 2.5 Pro (experimental and preview). With the new additions, SQL Assistant supports more than 20 language models so you can select the one best suited to your needs.

![SQL Assistant - New Models](https://assets.timescale.com/docs/images/sql-assistant-new-models.png)

## 🪵 TimescaleDB v2.19, new service overview page, and log improvements
<Label type="date">April 11, 2025</Label>

### TimescaleDB v2.19—query performance and concurrency improvements

Starting this week, all new services created on Timescale Cloud use [TimescaleDB v2.19](https://github.com/timescale/timescaledb/releases/tag/2.19.0). Existing services will be upgraded gradually during their maintenance window.

Highlighted features in TimescaleDB v2.19 include:

- Improved concurrency of `INSERT`, `UPDATE`, and `DELETE` operations on the columnstore by no longer blocking DML statements during the recompression of a chunk.
- Improved system performance during continuous aggregate refreshes by breaking them into smaller batches. This reduces systems pressure and minimizes the risk of spilling to disk.
- Faster and more up-to-date results for queries against continuous aggregates by materializing the most recent data first, as opposed to old data first in prior versions.
- Faster analytical queries with SIMD vectorization of aggregations over text columns and `GROUP BY` over multiple columns.
- Enable chunk size optimization for better query performance in the columnstore by merging them with `merge_chunk`.

### New service overview page

The service overview page in Timescale Console has been overhauled to make it simpler and easier to use. Navigate to the `Overview` tab for any of your services and you will find an architecture diagram and general information pertaining to it. You may also see recommendations at the top, for how to optimize your service.

![New Service Overview page](https://assets.timescale.com/docs/images/new-timescale-service-overview.png)

To leave the product team your feedback, open `Help & Support` on the left and select `Send feedback to the product team`.

### Find logs faster

Finding logs just got easier! We've added a date, time, and timezone picker, so you can jump straight to the exact moment you're interested in—no more endless scrolling.

![Find logs faster](https://assets.timescale.com/docs/images/find-logs-faster-timescale-console.png)

## 📒Faster vector search and improved job information
<Label type="date">April 4, 2025</Label>

### pgvectorscale 0.7.0: faster filtered filtered vector search with filtered indexes

This pgvectorscale release adds label-based filtered vector search to the StreamingDiskANN index.
This enables you to return more precise and efficient results by combining vector
similarity search with label filtering while still uitilizing the ANN index. This is a common need for large-scale RAG and Agentic applications
that rely on vector searches with metadata filters to return relevant results. Filtered indexes add
even more capabilities for filtered search at scale, complementing the high accuracy streaming filtering already
present in pgvectorscale. The implementation is inspired by Microsoft's Filtered DiskANN research.
For more information, see the [pgvectorscale release notes][log-28032025-pgvectorscale-rn] and a
[usage example][log-28032025-pgvectorscale-example].

### Job errors and individual job pages

Each job now has an individual page in Timescale Console, and displays additional details about job errors. You use
this information to debug failing jobs.

To see the job information page, in [Timescale Console][console], select the service to check, then click `Jobs` > job ID to investigate.

- Successful jobs:

  ![Log success in Timescale Console](https://assets.timescale.com/docs/images/changelog-job-success-page.png)

- Unsuccessful jobs with errors:

  ![Log errors in Timescale Console](https://assets.timescale.com/docs/images/changelog-job-error-page.png)

## 🤩 In-Console Livesync for Postgres
<Label type="date">March 21, 2025</Label>

You can now set up an active data ingestion pipeline with livesync for Postgres in Timescale Console. This tool enables you to replicate your source database tables into Timescale's hypertables indefinitely. Yes, you heard that right—keep livesync running for as long as you need, ensuring that your existing source Postgres tables stay in sync with Timescale Cloud. Read more about setting up and using [Livesync for Postgres](https://docs.timescale.com/migrate/latest/livesync-for-postgresql/).

![Livesync in Timescale Console](https://assets.timescale.com/docs/images/timescale-cloud-livesync-tile.png)

![Set up Timescale Livesync](https://assets.timescale.com/docs/images/set-up-timescale-cloud-livesync.png)

![Select tables for Livesync](https://assets.timescale.com/docs/images/select-tables-for-timescale-cloud-livesync.png)

![Timescale Livesync running](https://assets.timescale.com/docs/images/livesync-view-status.png)

## 💾 16K dimensions on pgvectorscale plus new pgai Vectorizer support
<Label type="date">March 14, 2025</Label>

### pgvectorscale 0.6 — store up to 16K dimension embeddings

pgvectorscale 0.6.0 now supports storing vectors with up to 16,000 dimensions, removing the previous limitation of 2,000 from pgvector. This lets you use larger embedding models like OpenAI's text-embedding-3-large (3072 dim) with Postgres as your vector database. This release also includes key performance and capability enhancements, including NEON support for SIMD distance calculations on aarch64 processors, improved inner product distance metric implementation, and improved index statistics. See the release details [here](https://github.com/timescale/pgvectorscale/releases/tag/0.6.0).

### pgai Vectorizer supports models from AWS Bedrock, Azure AI, Google Vertex via LiteLLM

Access embedding models from popular cloud model hubs like AWS Bedrock, Azure AI Foundry, Google Vertex, as well as HuggingFace and Cohere as part of the LiteLLM integration with pgai Vectorizer. To use these models with pgai Vectorizer on Timescale Cloud, select `Other` when adding the API key in the credentials section of Timescale Console.

## 🤖 Agent Mode for PopSQL and more
<Label type="date">March 7, 2025</Label>

### Agent Mode for PopSQL

Introducing Agent Mode, a new feature in Timescale Console SQL Assistant. SQL Assistant lets you query your database using natural language. However, if you ran into errors, you have to approve the implementation of the Assistant's suggestions.

With Agent Mode on, SQL Assistant automatically adjusts and executes your query without intervention. It runs, diagnoses, and fixes any errors that it runs into until you get your desired results.

Below you can see SQL Assistant run into an error, identify the resolution, execute the fixed query, display results, and even change the title of the query:

![Timescale SQL Assistant Agent Mode](https://assets.timescale.com/docs/images/timescale-sql-assistant-agent-mode.gif)

To use Agent Mode, make sure you have SQL Assistant enabled, then click on the model selector dropdown, and tick the `Agent Mode` checkbox.

### Improved AWS Marketplace integration for a smoother experience

We've enhanced the AWS Marketplace workflow to make your experience even better! Now, everything is fully automated,
ensuring a seamless process from setup to billing. If you're using the AWS Marketplace integration, you'll notice a
smoother transition and clearer billing visibility—your Timescale Cloud subscription will be reflected directly in AWS
Marketplace!

### Timescale Console recommendations

Sometimes it can be hard to know if you are getting the best use out of your service. To help with this, Timescale
Cloud now provides recommendations based on your service's context, assisting with onboarding or notifying if there is a configuration concern with your service, such as consistently failing jobs.

To start, recommendations are focused primarily on onboarding or service health, though we will regularly add new ones. You can see if you have any existing recommendations for your service by going to the `Actions` tab in Timescale Console.

![Timescale Console recommendations](https://assets.timescale.com/docs/images/timescale-console-recommendations.png)

## 🛣️ Configuration Options for Secure Connections and More
<Label type="date">February 28, 2025</Label>

### Edit VPC and AWS Transit Gateway CIDRs

You can now modify the CIDRs blocks for your VPC or Transit Gateway directly from Timescale Console, giving you greater control over network access and security. This update makes it easier to adjust your private networking setup without needing to recreate your VPC or contact support.

![VPC connection wizard](https://assets.timescale.com/docs/images/2025-02-27changelog_VPC_transit_gateway.png)

### Improved log filtering

We’ve enhanced the `Logs` screen with the new `Warning` and `Log` filters to help you quickly find the logs you need. These additions complement the existing `Fatal`, `Error`, and `Detail` filters, making it easier to pinpoint specific events and troubleshoot issues efficiently.

![Logs with filters](https://assets.timescale.com/docs/images/2025-02-27changelog_log_filtering.png)

### TimescaleDB v2.18.2 on Timescale Cloud

New services created in Timescale Cloud now use [TimescaleDB v2.18.2](https://github.com/timescale/timescaledb/releases/tag/2.18.2). Existing services are in the process of being automatically upgraded to this version.

This new release fixes a number of bugs including:

- Fix `ExplainHook` breaking the call chain.
- Respect `ExecutorStart` hooks of other extensions.
- Block dropping internal compressed chunks with `drop_chunk()`.

### SQL Assistant improvements

- Support for Claude 3.7 Sonnet and extended thinking including reasoning tokens.
- Ability to abort SQL Assistant requests while the response is streaming.

## 🤖 SQL Assistant Improvements and Pgai Docs Reorganization
<Label type="date">February 21, 2025</Label>

### New models and improved UX for SQL Assistant

We have added fireworks.ai and Groq as service providers, and several new LLM options for SQL Assistant:

- OpenAI o1
- DeepSeek R1
- Llama 3.3 70B
- Llama 3.1 405B
- DeepSeek R1 Distill - Llama 3.3

We've also improved the model picker by adding descriptions for each model:

![Timescale Cloud SQL Assistant AI model picker](https://assets.timescale.com/docs/images/sql-assistant-ai-models.png)

### Updated and reorganized docs for pgai

We have improved the GitHub docs for pgai. Now relevant sections have been grouped into their own folders and we've created a comprehensive summary doc.  Check it out [here](https://github.com/timescale/pgai/tree/main/docs).

## 💘 TimescaleDB v2.18.1 and AWS Transit Gateway Support Generally Available
<Label type="date">February 14, 2025</Label>

### TimescaleDB v2.18.1
New services created in Timescale Cloud now use [TimescaleDB v2.18.1](https://github.com/timescale/timescaledb/releases/tag/2.18.1). Existing services will be automatically upgraded in their next maintenance window starting next week.

This new release includes a number of bug fixes and small improvements including:

* Faster columnar scans when using the hypercore table access method
* Ensure all constraints are always applied when deleting data on the columnstore
* Pushdown all filters on scans for UPDATE/DELETE operations on the columnstore

###  AWS Transit Gateway support is now generally available!

Timescale Cloud now fully supports [AWS Transit Gateway](https://docs.timescale.com/use-timescale/latest/security/transit-gateway/), making it even easier to securely connect your database to multiple VPCs across different environments—including AWS, on-prem, and other cloud providers.

With this update, you can establish a peering connection between your Timescale Cloud services and an AWS Transit Gateway in your AWS account. This keeps your Timescale Cloud services safely behind a VPC while allowing seamless access across complex network setups.

## 🤖 TimescaleDB v2.18 and SQL Assistant Improvements in Data Mode and PopSQL

<Label type="date">February 6, 2025</Label>

### TimescaleDB v2.18 - dense indexes in the columnstore and query vectorization improvements
Starting this week, all new services created on Timescale Cloud use [TimescaleDB v2.18](https://github.com/timescale/timescaledb/releases/tag/2.18.0). Existing services will be upgraded gradually during their maintenance window.

Highlighted features in TimescaleDB v2.18.0 include:

* The ability to add dense indexes (btree and hash) to the columnstore through the new hypercore table access method.
* Significant performance improvements through vectorization (SIMD) for aggregations using a group by with one column and/or using a filter clause when querying the columnstore.
* Hypertables support triggers for transition tables, which is one of the most upvoted community feature requests.
* Updated methods to manage Timescale's hybrid row-columnar store (hypercore). These methods highlight columnstore usage. The columnstore includes an optimized columnar format as well as compression.

### SQL Assistant improvements

We made a few improvements to SQL Assistant:

**Dedicated SQL Assistant threads** 🧵

Each query, notebook, and dashboard now gets its own conversation thread, keeping your chats organized.

![Dedicated threads](https://assets.timescale.com/docs/images/timescale-cloud-sql-assistant-threads.gif)

**Delete messages** ❌

Made a typo? Asked the wrong question? You can now delete individual messages from your thread to keep the conversation clean and relevant.

![Delete messages in SQL Assistant threads](https://assets.timescale.com/docs/images/timescale-cloud-sql-assistant-delete-messages.png)

**Support for OpenAI `o3-mini` ⚡**

We’ve added support for OpenAI’s latest `o3-mini` model, bringing faster response times and improved reasoning for SQL queries.

![SQL Assistant o3 mini](https://assets.timescale.com/docs/images/timescale-cloud-sql-assistant-o3-mini.png)

## 🌐 IP Allowlists in Data Mode and PopSQL

<Label type="date">January 31, 2025</Label>

For enhanced network security, you can now also create IP allowlists in the Timescale Console data mode and PopSQL. Similarly to the [ops mode IP allowlists][ops-mode-allow-list], this feature grants access to your data only to certain IP addresses. For example, you might require your employees to use a VPN and add your VPN static egress IP to the allowlist.

This feature is available in:

- [Timescale Console][console] data mode, for all pricing tiers
- [PopSQL web][popsql-web]
- [PopSQL desktop][popsql-desktop]

Enable this feature in PopSQL/Timescale Console data mode > `Project` > `Settings` > `IP Allowlist`:

![Timescale Console data mode IP allowlist](https://assets.timescale.com/docs/images/timescale-data-mode-ip-allowlist.png)

## 🤖 pgai Extension and Python Library Updates
<Label type="date">January 24, 2025</Label>

### AI — pgai Postgres extension 0.7.0
This release enhances the Vectorizer functionality by adding configurable `base_url` support for OpenAI API. This enables pgai Vectorizer to use all OpenAI-compatible models and APIs via the OpenAI integration simply by changing the `base_url`. This release also includes public granting of vectorizers, superuser creation on any table, an upgrade to the Ollama client to 0.4.5, a new `docker-start` command, and various fixes for struct handling, schema qualification, and system package management. [See all changes on Github](https://github.com/timescale/pgai/releases/tag/extension-0.7.0).

### AI - pgai python library 0.5.0
This release adds comprehensive SQLAlchemy and Alembic support for vector embeddings, including operations for migrations and improved model inheritance patterns. You can now seamlessly integrate vector search capabilities with SQLAlchemy models while utilizing Alembic for database migrations. This release also adds key improvements to the Ollama integration and self-hosted Vectorizer configuration. [See all changes on Github](https://github.com/timescale/pgai/releases/tag/pgai-v0.5.0).

## AWS Transit Gateway Support
<Label type="date">January 17, 2025</Label>

### AWS Transit Gateway Support (Early Access)
Timescale Cloud now enables you to connect to your Timescale Cloud services through AWS Transit Gateway. This feature is available to Scale and Enterprise customers. It will be in Early Access for a short time and available in the Timescale Console very soon. If you are interested in implementing this Early Access Feature, reach out to your Rep.

## 🇮🇳 New region in India, Postgres 17 upgrades, and TimescaleDB on AWS Marketplace
<Label type="date">January 10, 2025</Label>

### Welcome India! (Support for a new region: Mumbai)
Timescale Cloud now supports the Mumbai region. Starting today, you can run Timescale Cloud services in Mumbai, bringing our database solutions closer to users in India.

### Postgres major version upgrades to PG 17
Timescale Cloud services can now be upgraded directly to Postgres 17 from versions 14, 15, or 16. Users running versions 12 or 13 must first upgrade to version 15 or 16, before upgrading to 17.

### Timescale Cloud available on AWS Marketplace
Timescale Cloud is now available in the [AWS Marketplace][aws-timescale]. This allows you to keep billing centralized on your AWS account, use your already committed AWS Enterprise Discount Program spend to pay your Timescale Cloud bill and simplify procurement and vendor management.

## 🎅 Postgres 17, feature requests, and Postgres Livesync
<Label type="date">December 20, 2024</Label>

### Postgres 17
All new Timescale Cloud services now come with Postgres 17.2, the latest version. Upgrades to Postgres 17 for services running on prior versions will be available in January.
Postgres 17 adds new capabilities and improvements to Timescale like:
* **System-wide Performance Improvements**. Significant performance boosts, particularly in high-concurrency workloads. Enhancements in the I/O layer, including improved Write-Ahead Log (WAL) processing, can result in up to a 2x increase in write throughput under heavy loads.
* **Enhanced JSON Support**. The new JSON_TABLE allows developers to convert JSON data directly into relational tables, simplifying the integration of JSON and SQL. The release also adds new SQL/JSON constructors and query functions, offering powerful tools to manipulate and query JSON data within a traditional relational schema.
* **More Flexible MERGE Operations**. The MERGE command now includes a RETURNING clause, making it easier to track and work with modified data. You can now also update views using MERGE, unlocking new use cases for complex queries and data manipulation.

### Submit feature requests from Timescale Console
You can now submit feature requests directly from Console and see the list of feature requests you have made. Just click on `Feature Requests` on the right sidebar.
All feature requests are automatically published to the [Timescale Forum](https://www.timescale.com/forum/c/cloud-feature-requests/39) and are reviewed by the product team, providing more visibility and transparency on their status as well as allowing other customers to vote for them.

![Submit a feature request in Timescale Console](https://assets.timescale.com/docs/images/submit-feature-request.png)

### Postgres Livesync (Alpha release)
We have built a new solution that helps you continuously replicate all or some of your Postgres tables directly into Timescale Cloud.

[Livesync](https://docs.timescale.com/migrate/latest/livesync-for-postgresql/) allows you to keep a current Postgres instance such as RDS as your primary database, and easily offload your real-time analytical queries to Timescale Cloud to boost their performance. If you have any questions or feedback, talk to us in [#livesync in Timescale Community](https://app.slack.com/client/T4GT3N2JK/C086NU9EZ88).

This is just the beginning—you'll see more from livesync in 2025!

## In-Console import from S3, I/O Boost, and Jobs Explorer
<Label type="date">December 13, 2024</Label>

### In-Console import from S3 (CSV and Parquet files)

Connect your S3 buckets to import data into Timescale Cloud. We support CSV (including `.zip` and `.gzip`) and Parquet files, with a 10 GB size limit in this initial release. This feature is accessible in the `Import your data` section right after service creation and through the `Actions` tab.

![Import data into Timescale with S3](https://assets.timescale.com/docs/images/import-your-data-s3.png)

![Import data into Timescale with S3 details](https://assets.timescale.com/docs/images/import-data-s3-details.png)

### Self-Serve I/O Boost 📈

I/O Boost is an add-on for customers on Scale or Enterprise tiers that maximizes the I/O capacity of EBS storage to 16,000 IOPS and 1,000 MBps throughput per service. To enable I/O Boost, navigate to `Services` > `Operations` in Timescale Console. A simple toggle allows you to enable the feature, with pricing clearly displayed at $0.41/hour per node.

![Timescale I/O Boost](https://assets.timescale.com/docs/images/timescale-i-o-boost.png)

### Jobs Explorer

See all the jobs associated with your service through a new `Jobs` tab. You can see the type of job, its status (`Running`, `Paused`, and others), and a detailed history of the last 100 runs, including success rates and runtime statistics.

![Timescale Console Jobs tab](https://assets.timescale.com/docs/images/timescale-console-jobs-tab.png)

![Timescale Console Jobs tab expanded](https://assets.timescale.com/docs/images/timescale-console-jobs-expanded.png)

## 🛝 New service creation flow
<Label type="date">December 6, 2024</Label>

- **AI and Vector:** the UI now lets you choose an option for creating AI and Vector-ready services right from the start. You no longer need to add the pgai, pgvector, and pgvectorscale extensions manually. You can combine this with time-series capabilities as well!

  ![Create Timescale Cloud service](https://assets.timescale.com/docs/images/create-timescale-service.png)

- **Compute size recommendations:** new (and old) users were sometimes unsure about what compute size to use for their workload.  We now offer compute size recommendations based on how much data you plan to have in your service.

  ![Service compute recommendation](https://assets.timescale.com/docs/images/timescale-service-compute-size.png)

- **More information about configuration options:** we've made it clearer what each configuration option does, so that you can make more informed choices about how you want your service to be set up.

## 🗝️ IP Allow Lists!
<Label type="date">November 21, 2024</Label>

IP Allow Lists let you specify a list of IP addresses that have access to your Timescale Cloud services and block any others. IP Allow Lists are a
lightweight but effective solution for customers concerned with security and compliance. They enable
you to prevent unauthorized connections without the need for a [Virtual Private Cloud (VPC)](https://docs.timescale.com/use-timescale/latest/security/vpc/).

To get started, in [Timescale Console](https://console.cloud.timescale.com/), select a service, then click
**Operations** > **Security** >  **IP Allow List**, then create an IP Allow List.

![IP Allow lists](https://assets.timescale.com/docs/images/IP-Allow-lists.png)

For more information, [see our docs](https://docs.timescale.com/use-timescale/latest/security/ip-allow-list/).

## 🤩 SQL Assistant, TimescaleDB v2.17, HIPAA compliance, and better logging
<Label type="date">November 14, 2024</Label>

### 🤖 New AI companion: SQL Assistant

SQL Assistant uses AI to help you write SQL faster and more accurately.

- **Real-time help:** chat with models like OpenAI 4o and Claude 3.5 Sonnet to get help writing SQL. Describe what you want in natural language and have AI write the SQL for you.

  <div class="relative w-fit mx-auto">

  <iframe width="1120" height="630" style="max-width:100%"  src="https://www.youtube.com/embed/3Droej_E0cQ?si=9IFB1Pk8Cl1bVKtD" title="YouTube video player" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" referrerpolicy="strict-origin-when-cross-origin" allowfullscreen></iframe>

  </div>

- **Error resolution**: when you run into an error, SQL Assistant proposes a recommended fix that you can choose to accept.

    ![AI error fix](https://assets.timescale.com/docs/images/ai-error-fix.png)

- **Generate titles and descriptions**: click a button and SQL Assistant generates a title and description for your query. No more untitled queries!

   ![AI generated query title](https://assets.timescale.com/docs/images/ai-generate-title.png)

See our [blog post](https://www.tigerdata.com/blog/postgres-gui-sql-assistant/) or [docs](https://docs.tigerdata.com/getting-started/latest/run-queries-from-console/#sql-assistant) for full details!

### 🏄 TimescaleDB v2.17 - performance improvements for analytical queries and continuous aggregate refreshes

Starting this week, all new services created on Timescale Cloud use [TimescaleDB v2.17](https://github.com/timescale/timescaledb/releases/tag/2.17.0). Existing services are upgraded gradually during their maintenance windows.

TimescaleDB v2.17 significantly improves the performance of [continuous aggregate refreshes](https://docs.timescale.com/use-timescale/latest/continuous-aggregates/refresh-policies/), and contains performance improvements for [analytical queries and delete operations](https://docs.timescale.com/use-timescale/latest/compression/modify-compressed-data/) over compressed hypertables.

Best practice is to upgrade at the next available opportunity.

Highlighted features in TimescaleDB v2.17 are:

* Significant performance improvements for continuous aggregate policies:

    * Continuous aggregate refresh now uses `merge` instead of deleting old materialized data and re-inserting.

    * Continuous aggregate policies are now more lightweight, use less system resources, and complete faster. This update:

      * Decreases dramatically the amount of data that must be written on the continuous aggregate in the presence of a small number of changes
      * Reduces the i/o cost of refreshing a continuous aggregate
      * Generates fewer Write-Ahead Logs (`WAL`)

* Increased performance for real-time analytical queries over compressed hypertables:

    * We are excited to introduce additional Single Instruction, Multiple Data (SIMD) vectorization optimization to TimescaleDB. This release supports vectorized execution for queries that _group by_ using the `segment_by` column(s), and _aggregate_ using the `sum`, `count`, `avg`, `min`, and `max` basic aggregate functions.

    * Stay tuned for more to come in follow-up releases! Support for grouping on additional columns, filtered aggregation, vectorized expressions, and `time_bucket` is coming soon.

    * Improved performance of deletes on compressed hypertables when a large amount of data is affected.

      This improvement speeds up operations that delete whole segments by skipping the decompression step. It is enabled for all deletes that filter by the `segment_by` column(s).

### HIPAA compliance

Timescale Cloud's [Enterprise plan](https://docs.timescale.com/about/latest/pricing-and-account-management/#features-included-in-each-pricing-plan) is now HIPAA (Health Insurance Portability and Accountability Act) compliant. This allows organizations to securely manage and analyze sensitive healthcare data, ensuring they meet regulatory requirements while building compliant applications.

### Expanded logging within Timescale Console

Customers can now access more than just the most recent 500 logs within the Timescale Console. We've updated the user experience, including scrollbar with infinite scrolling capabilities.

![Expanded console logs](https://assets.timescale.com/docs/images/console-expanded-logs.gif)

## ✨ Connect to Timescale from .NET Stack and check status of recent jobs
<Label type="date">November 07, 2024</Label>

### Connect to Timescale with your .NET stack
We've added instructions for connecting to Timescale using your .NET workflow. In Console after service creation, or in the **Actions** tab, you can now select .NET from the developer library list. The guide demonstrates how to use Npgsql to integrate Timescale with your existing software stack.

![.NET instructions](https://assets.timescale.com/docs/images/connect-via-net.png)

### ✅ Last 5 jobs status
In the **Jobs** section of the **Explorer**, users can now see the status (completed/failed) of the last 5 runs of each job.

![job status](https://assets.timescale.com/docs/images/explorer-job-list.png)

## 🎃 New AI, data integration, and performance enhancements
<Label type="date">October 31, 2024</Label>

### Pgai Vectorizer: vector embeddings as database indexes (early access)
This early access feature enables you to automatically create, update, and maintain embeddings as your data changes. Just like an index, Timescale handles all the complexity: syncing, versioning, and cleanup happen automatically.
This means no manual tracking, zero maintenance burden, and the freedom to rapidly experiment with different embedding models and chunking strategies without building new pipelines.
Navigate to the AI tab in your service overview and follow the instructions to add your OpenAI API key and set up your first vectorizer or read our [guide to automate embedding generation with pgai Vectorizer](https://github.com/timescale/pgai/blob/main/docs/vectorizer/overview.md) for more details.

![Vectorizer setup](https://s3.amazonaws.com/assets.timescale.com/docs/images/vectorizer-setup.png)

### Postgres-to-Postgres foreign data wrappers:
Fetch and query data from multiple Postgres databases, including time-series data in hypertables, directly within Timescale Cloud using [foreign data wrappers (FDW)](https://docs.timescale.com/use-timescale/latest/schema-management/foreign-data-wrappers/). No more complicated ETL processes or external tools—just seamless integration right within your SQL editor. This feature is ideal for developers who manage multiple Postgres and time-series instances and need quick, easy access to data across databases.

### Faster queries over tiered data
This release adds support for runtime chunk exclusion for queries that need to access [tiered storage](https://docs.timescale.com/use-timescale/latest/data-tiering/). Chunk exclusion now works with queries that use stable expressions in the `WHERE` clause. The most common form of this type of query is:

```
SELECT * FROM  hypertable WHERE timestamp_col > now() - '100 days'::interval
```

For more info on queries with immutable/stable/volatile filters, check our blog post on [Implementing constraint exclusion for faster query performance](https://www.timescale.com/blog/implementing-constraint-exclusion-for-faster-query-performance/).

If you no longer want to use tiered storage for a particular hypertable, you can now disable tiering and drop the associated tiering metadata on the hypertable with a call to [disable_tiering function](https://docs.timescale.com/use-timescale/latest/data-tiering/enabling-data-tiering/#disable-tiering).

### Chunk interval recommendations
Timescale Console now shows recommendations for services with too many small chunks in their hypertables.
Recommendations for new intervals that improve service performance are displayed for each underperforming service and hypertable. Users can then change their chunk interval and boost performance within Timescale Console.

![Chunk interval recommendation](https://s3.amazonaws.com/assets.timescale.com/docs/images/chunk-interval-recommendation.png)

## 💡 Help with hypertables and faster notebooks
<Label type="date">October 18, 2024</Label>

### 🧙Hypertable creation wizard
After creating a service, users can now create a hypertable directly in Timescale Console by first creating a table, then converting it into a hypertable. This is possible using the in-console SQL editor. All standard hypertable configuration options are supported, along with any customization of the underlying table schema.
![Hypertable creation wizard: image 1](https://assets.timescale.com/docs/images/hypertable-creation-wizard-1.png)

### 🍭 PopSQL Notebooks
The newest version of Data Mode Notebooks is now waaaay faster.  Why? We've incorporated the newly developed v3 of our query engine that currently powers Timescale Console's SQL Editor.  Check out the difference in query response times.

## ✨ Production-Ready Low-Downtime Migrations, MySQL Import, Actions Tab, and Current Lock Contention Visibility in SQL Editor
<Label type="date">October 10, 2024</Label>

### 🏗️ Live Migrations v1.0 Release

Last year, we began developing a solution for low-downtime migration from Postgres and TimescaleDB. Since then, this solution has evolved significantly, featuring enhanced functionality, improved reliability, and performance optimizations. We're now proud to announce that **live migration is production-ready** with the release of version 1.0.

Many of our customers have successfully migrated databases to Timescale using [live migration](https://docs.timescale.com/migrate/latest/live-migration/), with some databases as large as a few terabytes in size.

### 🔁 Actions Tab

As part of the service creation flow, we offer the following:

- Connect to services from different sources
- Import and migrate data from various sources
- Create hypertables

Previously, these actions were only visible during the service creation process and couldn't be accessed later. Now, these actions are **persisted within the service**, allowing users to leverage them on-demand whenever they're ready to perform these tasks.

![Timescale Console Actions tab](https://assets.timescale.com/docs/images/timescale-console-actions-tab.png)

### 🧭 Import Data from MySQL

We've noticed users struggling to convert their MySQL schema and data into their Timescale Cloud services. This was due to the semantic differences between MySQL and Postgres. To simplify this process, we now offer **easy-to-follow instructions** to import data from MySQL to Timescale Cloud.  This feature is available as part of the data import wizard, under the **Import from MySQL** option.

![MySQL import instructions](https://assets.timescale.com/docs/images/mysql-import-instructions.png)

### 🔐 Current Lock Contention

In Timescale Console, we offer the SQL editor so you can directly querying your service. As a new improvement,  **if a query is waiting on locks and can't complete execution**, Timescale Console now displays the current lock contention in the results section .

![View console services](https://assets.timescale.com/docs/images/current-lock-contention.png)

## CIDR & VPC Updates

<Label type="date">October 3, 2024</Label>

Timescale now supports multiple CIDRs on the customer VPC. Customers who want to take advantage of multiple CIDRs will need to recreate their peering.

## 🤝 New modes in Timescale Console: Ops and Data mode, and Console based Parquet File Import

<Label type="date">September 19, 2024</Label>

We've been listening to your feedback and noticed that Timescale Console users have diverse needs. Some of you are focused on operational tasks like adding replicas or changing parameters, while others are diving deep into data analysis to gather insights.

**To better serve you, we've introduced new modes to the Timescale Console UI—tailoring the experience based on what you're trying to accomplish.**

Ops mode is where you can manage your services, add replicas, configure compression, change parameters, and so on.

Data mode is the full PopSQL experience: write queries with autocomplete, visualize data with charts and dashboards, schedule queries and dashboards to create alerts or recurring reports, share queries and dashboards, and more.

Try it today and let us know what you think!

![Timescale Console Ops and Data mode](https://assets.timescale.com/docs/images/ops-data-mode.gif)

## Console based Parquet File Import

Now users can upload from Parquet to Timescale Cloud by uploading the file from their local file system. For files larger than 250 MB, or if you want to do it yourself, follow the three-step process to upload Parquet files to Timescale.

![Upload from Parquet to Timescale Cloud](https://assets.timescale.com/docs/images/upload_parquet.gif)


### SQL editor improvements

* In the Ops mode SQL editor, you can now highlight a statement to run a specific statement.

## High availability, usability, and migrations improvements
<Label type="date">September 12, 2024</Label>

### Multiple HA replicas

Scale and Enterprise customers can now configure two new multiple high availability (HA) replica options directly through Timescale Console:

* Two HA replicas (both asynchronous) - our highest availability configuration.
* Two HA replicas (one asynchronous, one synchronous) - our highest data integrity configuration.

Previously, Timescale offered only a single synchronous replica for customers seeking high availability. The single HA option is still available.

![Change Replica Configuration](https://s3.amazonaws.com/assets.timescale.com/docs/images/change-replica-configuration.png)

![High Availability](https://s3.amazonaws.com/assets.timescale.com/docs/images/high-availability.png)

For more details on multiple HA replicas, see [Manage high availability](https://docs.timescale.com/use-timescale/latest/ha-replicas/high-availability/).

### Other improvements

* In the Console SQL editor, we now indicate if your database session is healthy or has been disconnected. If it's been disconnected, the session will reconnect on your next query execution.

   ![Session Status Indicator](https://s3.amazonaws.com/assets.timescale.com/docs/images/session-status-indicator.gif)

* Released live-migration v0.0.26 and then v0.0.27 which includes multiple performance improvements and bugfixes as well as better support for Postgres 12.

## One-click SQL statement execution from Timescale Console, and session support in the SQL editor
<Label type="date">September 05, 2024</Label>

### One-click SQL statement execution from Timescale Console

Now you can simply click to run SQL statements in various places in the Console. This requires that the [SQL Editor][sql-editor] is enabled for the service.

* Enable Continuous Aggregates from the CAGGs wizard by clicking **Run** below the SQL statement.
![Enable Continuous Aggregates](https://s3.amazonaws.com/assets.timescale.com/docs/images/enable-continuous-aggregates.gif)

* Enable database extensions by clicking **Run** below the SQL statement.
![Enable extensions from Console](https://s3.amazonaws.com/assets.timescale.com/docs/images/enable-extensions-from-console.gif)

* Query data instantly with a single click in the Console after successfully uploading a CSV file.
![Query data after CSV import](https://s3.amazonaws.com/assets.timescale.com/docs/images/query-data-after-csv-import.gif)

### Session support in the SQL editor

Last week we announced the new in-console SQL editor. However, there was a limitation where a new database session was created for each query execution.

Today we removed that limitation and added support for keeping one database session for each user logged in, which means you can do things like start transactions:

```
begin;
insert into users (name, email) values ('john doe', 'john@example.com');
abort; -- nothing inserted
```

Or work with temporary tables:

```
create temporary table temp_users (email text);
insert into temp_sales (email) values ('john@example.com');
-- table will automatically disappear after your session ends
```

Or use the `set` command:

```
set search_path to 'myschema', 'public';
```


## 😎 Query your database directly from the Console and enhanced data import and migration options
<Label type="date">August 30, 2024</Label>

### SQL Editor in Timescale Console
We've added a new tab to the service screen that allows users to query their database directly, without having to leave the console interface.

* For existing services on Timescale, this is an opt-in feature. For all newly created services, the SQL Editor will be enabled by default.
* Users can disable the SQL Editor at any time by toggling the option under the Operations tab.
* The editor supports all DML and DDL operations (any single-statement SQL query), but doesn't support multiple SQL statements in a single query.

![SQL Editor](https://s3.amazonaws.com/assets.timescale.com/docs/images/sql-editor-query.png)

### Enhanced Data Import Options for Quick Evaluation
After service creation, we now offer a dedicated section for data import, including options to import from Postgres as a source or from CSV files.

The enhanced Postgres import instructions now offer several options: single table import, schema-only import, partial data import (allowing selection of a specific time range), and complete database import. Users can execute any of these data imports with just one or two simple commands provided in the data import section.

![Data import screen](https://s3.amazonaws.com/assets.timescale.com/docs/images/data-import-screen.png)

### Improvements to Live migration
We've released v0.0.25 of Live migration that includes the following improvements:
* Support migrating tsdb on non public schema to public schema
* Pre-migration compatibility checks
* Docker compose build fixes

## 🛠️ Improved tooling in Timescale Cloud and new AI and Vector extension releases
<Label type="date">August 22, 2024</Label>

### CSV import
We have added a CSV import tool to the Timescale Console.  For all TimescaleDB services, after service creation you can:
* Choose a local file
* Select the name of the data collection to be uploaded (default is file name)
* Choose data types for each column
* Upload the file as a new hypertable within your service
Look for the `Import data from .csv` tile in the `Import your data` step of service creation.

![CSV import](https://s3.amazonaws.com/assets.timescale.com/docs/images/csv-import.png)

### Replica lag
Customers now have more visibility into the state of replicas running on Timescale Cloud. We’ve released a new parameter called Replica Lag within the Service Overview for both Read and High Availability Replicas. Replica lag is measured in bytes against the current state of the primary database. For questions or concerns about the relative lag state of your replica, reach out to Customer Support.

![Replica lag indicator](https://s3.amazonaws.com/assets.timescale.com/docs/images/replica-lag-indicator.png)

### Adjust chunk interval
Customers can now adjust their chunk interval for their hypertables and continuous aggregates through the Timescale UI. In the Explorer, select the corresponding hypertable you would like to adjust the chunk interval for. Under *Chunk information*, you can change the chunk interval. Note that this only changes the chunk interval going forward, and does not retroactively change existing chunks.

![Edit chunk interval](https://s3.amazonaws.com/assets.timescale.com/docs/images/edit-chunk-interval.png)


### CloudWatch permissions via role assumption
We've released permission granting via role assumption to CloudWatch. Role assumption is both more secure and more convenient for customers who no longer need to rotate credentials and update their exporter config.

For more details take a look at [our documentation][integrations].

<img src="https://s3.amazonaws.com/assets.timescale.com/docs/images/cloudwatch-role-assumption.png" width="600px" alt="CloudWatch authentication via role assumption" />

### Two-factor authentication (2FA) indicator
We’ve added a 2FA status column to the Members page, allowing customers to easily see whether each project member has 2FA enabled or disabled.

![2FA status](https://s3.amazonaws.com/assets.timescale.com/docs/images/2FA-status-indicator.png)

### Anthropic and Cohere integrations in pgai
The pgai extension v0.3.0 now supports embedding creation and LLM reasoning using models from Anthropic and Cohere. For details and examples, see [this post for pgai and Cohere](https://www.timescale.com/blog/build-search-and-rag-systems-on-postgresql-using-cohere-and-pgai/), and [this post for pgai and Anthropic](https://www.timescale.com/blog/use-anthropic-claude-sonnet-3-5-in-postgresql-with-pgai/).

### pgvectorscale extension: ARM builds and improved recall for low dimensional vectors
pgvectorscale extension [v0.3.0](https://github.com/timescale/pgvectorscale/releases/tag/0.3.0) adds support for ARM processors and improves recall when using StreamingDiskANN indexes with low dimensionality vectors. We recommend updating to this version if you are self-hosting.


## 🏄 Optimizations for compressed data and extended join support in continuous aggregates
<Label type="date">August 15, 2024</Label>

TimescaleDB v2.16.0 contains significant performance improvements when working with compressed data, extended join
support in continuous aggregates, and the ability to define foreign keys from regular tables towards hypertables.
We recommend upgrading at the next available opportunity.

Any new service created on Timescale Cloud starting today uses TimescaleDB v2.16.0.

In TimescaleDB v2.16.0 we:

* Introduced multiple performance focused optimizations for data manipulation operations (DML) over compressed chunks.

  Improved upsert performance by more than 100x in some cases and more than 500x in some update/delete scenarios.

* Added the ability to define chunk skipping indexes on non-partitioning columns of compressed hypertables.

  TimescaleDB v2.16.0 extends chunk exclusion to use these skipping (sparse) indexes when queries filter on the relevant columns,
  and prune chunks that do not include any relevant data for calculating the query response.

* Offered new options for use cases that require foreign keys defined.

  You can now add foreign keys from regular tables towards hypertables. We have also removed
  some really annoying locks in the reverse direction that blocked access to referenced tables
  while compression was running.

* Extended Continuous Aggregates to support more types of analytical queries.

  More types of joins are supported, additional equality operators on join clauses, and
  support for joins between multiple regular tables.

**Highlighted features in this release**

* Improved query performance through chunk exclusion on compressed hypertables.

  You can now define chunk skipping indexes on compressed chunks for any column with one of the following
  integer data types: `smallint`, `int`, `bigint`, `serial`, `bigserial`, `date`, `timestamp`, `timestamptz`.

  After calling `enable_chunk_skipping` on a column, TimescaleDB tracks the min and max values for
  that column, using this information to exclude chunks for queries filtering on that
  column, where no data would be found.

* Improved upsert performance on compressed hypertables.

  By using index scans to verify constraints during inserts on compressed chunks, TimescaleDB speeds
  up some ON CONFLICT clauses by more than 100x.

* Improved performance of updates, deletes, and inserts on compressed hypertables.

  By filtering data while accessing the compressed data and before decompressing, TimescaleDB has
  improved performance for updates and deletes on all types of compressed chunks, as well as inserts
  into compressed chunks with unique constraints.

  By signaling constraint violations without decompressing, or decompressing only when matching
  records are found in the case of updates, deletes and upserts, TimescaleDB v2.16.0 speeds
  up those operations more than 1000x in some update/delete scenarios, and 10x for upserts.

* You can add foreign keys from regular tables to hypertables, with support for all types of cascading options.
  This is useful for hypertables that partition using sequential IDs, and need to reference these IDs from other tables.

* Lower locking requirements during compression for hypertables with foreign keys

  Advanced foreign key handling removes the need for locking referenced tables when new chunks are compressed.
  DML is no longer blocked on referenced tables while compression runs on a hypertable.

* Improved support for queries on Continuous Aggregates

  `INNER/LEFT` and `LATERAL` joins are now supported. Plus, you can now join with multiple regular tables,
  and have more than one equality operator on join clauses.

**Postgres 13 support removal announcement**

Following the deprecation announcement for Postgres 13 in TimescaleDB v2.13,
Postgres 13 is no longer supported in TimescaleDB v2.16.

The currently supported Postgres major versions are 14, 15, and 16.

## 📦 Performance, packaging and stability improvements for Timescale Cloud
<Label type="date">August 8, 2024</Label>

### New plans
To support evolving customer needs, Timescale Cloud now offers three plans to provide more value, flexibility, and efficiency.
- **Performance:** for cost-focused, smaller projects. No credit card required to start.
- **Scale:** for developers handling critical and demanding apps.
- **Enterprise:** for enterprises with mission-critical apps.

Each plan continues to bill based on hourly usage, primarily for compute you run and storage you consume.  You can upgrade or downgrade between Performance and Scale plans via the Console UI at any time.  More information about the specifics and differences between these pricing plans can be found [here in the docs](https://docs.timescale.com/about/latest/pricing-and-account-management/).
![Pricing plans in the console](https://assets.timescale.com/docs/images/pricing-plans-in-console.png)

### Improvements to the Timescale Console
The individual tiles on the services page have been enhanced with new information, including high-availability status.  This will let you better assess the state of your services at a glance.
![New service tile](https://assets.timescale.com/docs/images/new-service-tile-high-availability.png)

### Live migration release v0.0.24
Improvements:
- Automatic retries are now available for the initial data copy of the migration
- Now uses pgcopydb for initial data copy for PG to TSDB migrations also (already did for TS to TS) which has a significant performance boost.
- Fixes issues with TimescaleDB v2.13.x migrations
- Support for chunk mapping for hypertables with custom schema and table prefixes


## ⚡ Performance and stability improvements for Timescale Cloud and TimescaleDB
<Label type="date">July 12, 2024</Label>

The following improvements have been made to Timescale products:

- **Timescale Cloud**:
  - The connection pooler has been updated and now avoids multiple reloads
  - The tsdbadmin user can now grant the following roles to other users: `pg_checkpoint`,`pg_monitor`,`pg_signal_backend`,`pg_read_all_stats`,`pg_stat_scan_tables`
  - Timescale Console is far more reliable.

- **TimescaleDB**
  - The TimescaleDB v2.15.3 patch release improves handling of multiple unique indexes in a compressed INSERT,
    removes the recheck of ORDER when querying compressed data, improves memory management in DML functions, improves
    the tuple lock acquisition for tiered chunks on replicas, and fixes an issue with ORDER BY/GROUP BY in our
    HashAggregate optimization on PG16. For more information, see the [release note](https://github.com/timescale/timescaledb/releases/tag/2.15.3).
  - The TimescaleDB v2.15.2 patch release improves sort pushdown for partially compressed chunks, and compress_chunk with
    a primary space partition. The metadata function is removed from the update script, and hash partitioning on a
    primary column is disallowed. For more information, see the [release note](https://github.com/timescale/timescaledb/releases/tag/2.15.2).



## ⚡ Performance improvements for live migration to Timescale Cloud
<Label type="date">June 27, 2024</Label>

The following improvements have been made to the Timescale [live-migration docker image](https://hub.docker.com/r/timescale/live-migration/tags):

- Table-based filtering is now available during live migration.
- Improvements to pbcopydb increase performance and remove unhelpful warning messages.
- The user notification log enables you to always select the most recent release for a migration run.

For improved stability and new features, update to the latest [timescale/live-migration](https://hub.docker.com/r/timescale/live-migration/tags) docker image. To learn more, see the [live migration docs](https://docs.timescale.com/migrate/latest/live-migration/).

## 🦙Ollama integration in pgai

<Label type="date">June 21, 2024</Label>

Ollama is now integrated with [pgai](https://github.com/timescale/pgai).

Ollama is the easiest and most popular way to get up and running with open-source
language models. Think of Ollama as _Docker for LLMs_, enabling easy access and usage
of a variety of open-source models like Llama 3, Mistral, Phi 3, Gemma, and more.

With the pgai extension integrated in your database, embed Ollama AI into your app using
SQL. For example:

```sql
select ollama_generate
( 'llava:7b'
, 'Please describe this image.'
, _images=> array[pg_read_binary_file('/pgai/tests/postgresql-vs-pinecone.jpg')]
, _system=>'you are a helpful assistant'
, _options=> jsonb_build_object
  ( 'seed', 42
  , 'temperature', 0.9
  )
)->>'response'
;
```

To learn more, see the [pgai Ollama documentation](https://github.com/timescale/pgai/blob/main/docs/vectorizer/quick-start.md).

## 🧙 Compression Wizard

<Label type="date">June 13, 2024</Label>

The compression wizard is now available on Timescale Cloud. Select a hypertable and be guided through enabling compression through the UI!

To access the compression wizard, navigate to `Explorer`, and select the hypertable you would like to compress. In the top right corner, hover where it says `Compression off`, and open the wizard. You will then be guided through the process of configuring compression for your hypertable, and can compress it directly through the UI.

![Run the compression wizard in Timescale Console](https://assets.timescale.com/docs/images/compress-data-in-console.png)

## 🏎️💨 High Performance AI Apps With pgvectorscale

<Label type="date">June 11, 2024</Label>

The [vectorscale extension][pgvectorscale] is now available on [Timescale Cloud][signup].

pgvectorscale complements pgvector, the open-source vector data extension for Postgres, and introduces the
following key innovations for pgvector data:

- A new index type called StreamingDiskANN, inspired by the DiskANN algorithm, based on research from Microsoft.
- Statistical Binary Quantization: developed by Timescale researchers, This compression method improves on
  standard Binary Quantization.

On benchmark dataset of 50 million Cohere embeddings (768 dimensions each), Postgres with pgvector and
pgvectorscale achieves 28x lower p95 latency and 16x higher query throughput compared to Pinecone's storage
optimized (s1) index for approximate nearest neighbor queries at 99% recall, all at 75% less cost when
self-hosted on AWS EC2.

To learn more, see the [pgvectorscale documentation][pgvectorscale].

## 🧐Integrate AI Into Your Database Using pgai

<Label type="date">June 11, 2024</Label>

The [pgai extension][pgai] is now available on [Timescale Cloud][signup].

pgai brings embedding and generation AI models closer to the database. With pgai, you can now do the following directly
from within Postgres in a SQL query:

* Create embeddings for your data.
* Retrieve LLM chat completions from models like OpenAI GPT4o.
* Reason over your data and facilitate use cases like classification, summarization, and data enrichment on your existing relational data in Postgres.

To learn more, see the [pgai documentation][pgai].

## 🐅Continuous Aggregate and Hypertable Improvements for TimescaleDB
<Label type="date">June 7, 2024</Label>

The 2.15.x releases contains performance improvements and bug fixes. Highlights in these releases are:

- Continuous Aggregate now supports `time_bucket` with origin and/or offset.
- Hypertable compression has the following improvements:
  - Recommend optimized defaults for segment by and order by when configuring compression through analysis of table configuration and statistics.
  - Added planner support to check more kinds of WHERE conditions before decompression.
    This reduces the number of rows that have to be decompressed.
  - You can now use minmax sparse indexes when you compress columns with btree indexes.
  - Vectorize filters in the WHERE clause that contain text equality operators and LIKE expressions.

To learn more, see the [TimescaleDB release notes](https://github.com/timescale/timescaledb/releases).

## 🔍 Database Audit Logging with pgaudit
<Label type="date">May 31, 2024</Label>

The [Postgres Audit extension(pgaudit)](https://github.com/pgaudit/pgaudit/) is now available on [Timescale Cloud][signup].
pgaudit provides detailed database session and object audit logging in the Timescale
Cloud logs.

If you have strict security and compliance requirements and need to log all operations
on the database level, pgaudit can help. You can also export these audit logs to
[Amazon CloudWatch](https://aws.amazon.com/cloudwatch/).

To learn more, see the [pgaudit documentation](https://github.com/pgaudit/pgaudit/).

## 🌡 International System of Unit Support with postgresql-unit
<Label type="date">May 31, 2024</Label>

The [SI Units for Postgres extension(unit)](https://github.com/df7cb/postgresql-unit) provides support for the
[ISU](https://en.wikipedia.org/wiki/International_System_of_Units) in [Timescale Cloud][signup].

You can use Timescale Cloud to solve day-to-day questions. For example, to see what 50°C is in °F, run the following
query in your Timescale Cloud service:

```
SELECT '50°C'::unit @ '°F' as temp;
  temp
--------
 122 °F
(1 row)
```

To learn more, see the [postgresql-unit documentation](https://github.com/df7cb/postgresql-unit).


===== PAGE: https://docs.tigerdata.com/about/timescaledb-editions/ =====

# Compare TimescaleDB editions



The following versions of TimescaleDB are available:

*   TimescaleDB Apache 2 Edition
*   TimescaleDB Community Edition

## TimescaleDB Apache 2 Edition

TimescaleDB Apache 2 Edition is available under the [Apache 2.0 license][apache-license]. This is a classic open source license,
meaning that it is completely unrestricted - anyone can take this code and offer it as a service.

You can install TimescaleDB Apache 2 Edition on your own on-premises or cloud
infrastructure and run it for free.

You can sell TimescaleDB Apache 2 Edition as a service, even if you're not the
main contributor.

You can modify the TimescaleDB Apache 2 Edition source code and run it for
production use.

## 	TimescaleDB Community Edition

TimescaleDB Community Edition is the advanced, best, and most feature complete
version of TimescaleDB, available under the terms of the
[Tiger Data License (TSL)][timescale-license].

For more information about the Tiger Data license, see [this blog post][license-blog].

Many of the most recent features of TimescaleDB are only available in
TimescaleDB Community Edition.

You can install TimescaleDB Community Edition in your own on-premises or cloud
infrastructure and run it for free. TimescaleDB Community Edition is completely
free if you manage your own service.

You cannot sell TimescaleDB Community Edition as a service, even if you are the
main contributor.

You can modify the TimescaleDB Community Edition source code and run it for
production use. Developers using TimescaleDB Community Edition have the "right
to repair" and make modifications to the source code and run it in their own
on-premises or cloud infrastructure. However, you cannot make modifications to
the TimescaleDB Community Edition source code and offer it as a service.

You can access a hosted version of TimescaleDB Community Edition through
[Tiger Cloud][timescale-cloud], a cloud-native platform for time-series and real-time analytics.

## Feature comparison


  <tr>
    <th>Features</th>
    <th>TimescaleDB Apache 2 Edition</th>
    <th>TimescaleDB Community Edition</th>
  </tr>
  <tr>
    <td><strong>Hypertables and chunks</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/create_table/">CREATE TABLE</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/create_hypertable/">create_hypertable</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/show_chunks/">show_chunks</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/drop_chunks/">drop_chunks</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/split_chunk/">split_chunk</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/reorder_chunk/">reorder_chunk</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/move_chunk/">move_chunk</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/add_reorder_policy/">add_reorder_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/attach_tablespace/">attach_tablespace</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/detach_tablespace/">detach_tablespace()</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/detach_tablespaces/">detach_tablespaces()</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/show_tablespaces/">show_tablespaces</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/set_chunk_time_interval/">set_chunk_time_interval</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/set_integer_now_func/">set_integer_now_func</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/add_dimension/">add_dimension()</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/create_index/">create_index (Transaction Per Chunk)</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/hypertable_size/">hypertable_size</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/hypertable_detailed_size/">hypertable_detailed_size</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/hypertable_index_size/">hypertable_index_size</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypertable/chunks_detailed_size/">chunks_detailed_size</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.tigerdata.com/use-timescale/latest/query-data/skipscan/">SkipScan</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td colspan="3"><strong>Distributed hypertables</strong>: This feature is <a href="https://github.com/timescale/timescaledb/blob/2.14.0/docs/MultiNodeDeprecation.md">sunsetted in all editions</a> in TimescaleDB v2.14.x</td>
  </tr>

  <tr>
    <td><strong>Hypercore</strong>  Since [TimescaleDB v2.18.0](https://github.com/timescale/timescaledb/releases/tag/2.18.0)</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/alter_table/">ALTER TABLE (Hypercore)</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/add_columnstore_policy/">add_columnstore_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/remove_columnstore_policy/">remove_columnstore_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/convert_to_columnstore/">convert_to_columnstore</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/convert_to_rowstore/">convert_to_rowstore</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/hypertable_columnstore_settings/">hypertable_columnstore_settings</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/hypertable_columnstore_stats/">hypertable_columnstore_stats</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/chunk_columnstore_settings/">chunk_columnstore_settings</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hypercore/chunk_columnstore_stats/">chunk_columnstore_stats</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Continuous aggregates</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/create_materialized_view/">CREATE MATERIALIZED VIEW (Continuous Aggregate)</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/alter_materialized_view/">ALTER MATERIALIZED VIEW (Continuous Aggregate)</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/drop_materialized_view/">DROP MATERIALIZED VIEW (Continuous Aggregate)</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/add_continuous_aggregate_policy/">add_continuous_aggregate_policy()</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/refresh_continuous_aggregate/">refresh_continuous_aggregate</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/continuous-aggregates/remove_continuous_aggregate_policy/">remove_continuous_aggregate_policy()</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Data retention</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/data-retention/add_retention_policy/">add_retention_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/data-retention/remove_retention_policy/">remove_retention_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Jobs and automation</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/jobs-automation/add_job/">add_job</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/jobs-automation/alter_job/">alter_job</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/jobs-automation/delete_job/">delete_job</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/jobs-automation/run_job/">run_job</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Hyperfunctions</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/approximate_row_count/">approximate_row_count</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/first/">first</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/last/">last</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/histogram/">histogram</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/time_bucket/">time_bucket</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/time_bucket_ng/">time_bucket_ng (experimental feature)</a></td>
    <td>✅ </td>
    <td>✅ </td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/gapfilling/time_bucket_gapfill/">time_bucket_gapfill</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.tigerdata.com/api/latest/hyperfunctions/gapfilling/time_bucket_gapfill#locf">locf</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/gapfilling/time_bucket_gapfill#interpolate">interpolate</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#percentile-agg">percentile_agg</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#approx_percentile">approx_percentile</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#approx_percentile_rank">approx_percentile_rank</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#rollup">rollup</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/tdigest/#max_val">max_val</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#mean">mean</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#error">error</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/tdigest/#min_val">min_val</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#num_vals">num_vals</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/uddsketch/#uddsketch">uddsketch</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/percentile-approximation/tdigest/#tdigest">tdigest</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/time-weighted-calculations/time_weight/">time_weight</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
   <tr>
    <td><a href="https://docs.tigerdata.com/api/latest/hyperfunctions/time-weighted-calculations/time_weight#rollup">rollup</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/hyperfunctions/time-weighted-calculations/time_weight#average">average</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Informational Views</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/chunks/#available-columns">timescaledb_information.chunks</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/continuous_aggregates/#sample-usage">timescaledb_information.continuous_aggregates</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/compression_settings/#sample-usage">timescaledb_information.compression_settings</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/data_nodes/#sample-usage">timescaledb_information.data_nodes</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/dimensions/#timescaledb-information-dimensions">timescaledb_information.dimension</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/hypertables/">timescaledb_information.hypertables</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/jobs/#available-columns">timescaledb_information.jobs</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/informational-views/job_stats/#available-columns">timescaledb_information.job_stats</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Administration functions</strong></td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/administration/#timescaledb_pre_restore">timescaledb_pre_restore</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/administration/#timescaledb_post_restore">timescaledb_post_restore</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/administration/#get_telemetry_report">get_telemetry_report</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/administration/#dump-timescaledb-meta-data">dump_meta_data</a></td>
    <td>✅</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><strong>Compression</strong>  Old API since [TimescaleDB v2.18.0](https://github.com/timescale/timescaledb/releases/tag/2.18.0) replaced by Hypercore</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/alter_table_compression/">ALTER TABLE (Compression)</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/add_compression_policy/#sample-usage">add_compression_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/remove_compression_policy/">remove_compression_policy</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/compress_chunk/">compress_chunk</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/decompress_chunk/">decompress_chunk</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/hypertable_compression_stats/">hypertable_compression_stats</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
  <tr>
    <td><a href="https://docs.timescale.com/api/latest/compression/chunk_compression_stats/">chunk_compression_stats</a></td>
    <td>❌</td>
    <td>✅</td>
  </tr>
</table>

<!-- vale Google.Units = NO -->


===== PAGE: https://docs.tigerdata.com/about/supported-platforms/ =====

# Supported platforms



This page lists the platforms and systems that Tiger Data products have been tested on for the
following options:

* **Tiger Cloud**: all the latest features that just work. A reliable and worry-free Postgres cloud for all your workloads.
* **Self-hosted products**: create your best app from the comfort of your own developer environment.

## Tiger Cloud

Tiger Cloud always runs the latest version of all Tiger Data products. With Tiger Cloud you:

* Build everything on one service, and each service hosts one database
* Get faster queries using less compute
* Compress data without sacrificing performance
* View insights on performance, queries, and more
* Reduce storage with automated retention policies

See the available [service capabilities][service-types] and [regions][regions].

### Available service capabilities

<Availability products={['cloud']} price_plans={['enterprise', 'scale', 'performance']} />

Tiger Cloud services run optimized Tiger Data extensions on latest Postgres, in a highly secure cloud environment. Each service is a specialized database instance tuned for your workload. Available capabilities are:


    <thead>
        <tr>
            <th>Capability</th>
            <th>Extensions</th>
        </tr>
    </thead>
    <tbody>
        <tr>
            <td><strong>Real-time analytics</strong> <p>Lightning-fast ingest and querying of time-based and event data.</p></td>
            <td><ul><li>TimescaleDB</li><li>TimescaleDB Toolkit</li></ul>   </td>
        </tr>
        <tr>
            <td ><strong>AI and vector </strong><p>Seamlessly build RAG, search, and AI agents.</p></td>
            <td><ul><li>TimescaleDB</li><li>pgvector</li><li>pgvectorscale</li><li>pgai</li></ul></td>
        </tr>
        <tr>
            <td ><strong>Hybrid</strong><p>Everything for real-time analytics and AI workloads, combined.</p></td>
            <td><ul><li>TimescaleDB</li><li>TimescaleDB Toolkit</li><li>pgvector</li><li>pgvectorscale</li><li>pgai</li></ul></td>
        </tr>
        <tr>
            <td ><strong>Support</strong></td>
            <td><ul><li>24/7 support no matter where you are.</li><li> Continuous incremental backup/recovery. </li><li>Point-in-time forking/branching.</li><li>Zero-downtime upgrades. </li><li>Multi-AZ high availability. </li><li>An experienced global ops and support team that can build and manage Postgres at scale.</li></ul></td>
        </tr>
    </tbody>
</table>

### Available regions

<Availability products={['cloud']} price_plans={['enterprise', 'scale', 'performance']} />

Tiger Cloud services run in the following Amazon Web Services (AWS) regions:

| Region           | Zone          | Location       |
| ---------------- | ------------- | -------------- |
| `ap-south-1`     | Asia Pacific  | Mumbai         |
| `ap-southeast-1` | Asia Pacific  | Singapore      |
| `ap-southeast-2` | Asia Pacific  | Sydney         |
| `ap-northeast-1` | Asia Pacific  | Tokyo          |
| `ca-central-1`   | Canada        | Central        |
| `eu-central-1`   | Europe        | Frankfurt      |
| `eu-west-1`      | Europe        | Ireland        |
| `eu-west-2`      | Europe        | London         |
| `sa-east-1`      | South America | São Paulo      |
| `us-east-1`      | United States | North Virginia |
| `us-east-2`      | United States | Ohio           |
| `us-west-2`      | United States | Oregon         |

## Self-hosted products

You use Tiger Data's open-source products to create your best app from the comfort of your own developer environment.

See the [available services][available-services] and [supported systems][supported-systems].

### Available services

Tiger Data offers the following services for your self-hosted installations:


    <thead>
        <tr>
            <th>Service type</th>
            <th>Description</th>
        </tr>
    </thead>
    <tbody>
        <tr>
            <td><strong>Self-hosted support</strong></td>
            <td><ul><li>24/7 support no matter where you are.</li><li>An experienced global ops and support team that
            can build and manage Postgres at scale.</li></ul>
            Want to try it out? <a href="https://www.tigerdata.com/self-managed-support">See how we can help</a>.
            </td>
        </tr>
    </tbody>
</table>

### Postgres, TimescaleDB support matrix

TimescaleDB and TimescaleDB Toolkit run on Postgres v10, v11, v12, v13, v14, v15, v16, and v17. Currently Postgres 15 and higher are supported.

| TimescaleDB version |Postgres 17|Postgres 16|Postgres 15|Postgres 14|Postgres 13|Postgres 12|Postgres 11|Postgres 10|
|-----------------------|-|-|-|-|-|-|-|-|
| 2.22.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.21.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.20.x                |✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.17 - 2.19           |✅|✅|✅|✅|❌|❌|❌|❌|❌|
| 2.16.x                |❌|✅|✅|✅|❌|❌|❌|❌|❌|❌|
| 2.13 - 2.15           |❌|✅|✅|✅|✅|❌|❌|❌|❌|
| 2.12.x                |❌|❌|✅|✅|✅|❌|❌|❌|❌|
| 2.10.x                |❌|❌|✅|✅|✅|✅|❌|❌|❌|
| 2.5 - 2.9             |❌|❌|❌|✅|✅|✅|❌|❌|❌|
| 2.4                   |❌|❌|❌|❌|✅|✅|❌|❌|❌|
| 2.1 - 2.3             |❌|❌|❌|❌|✅|✅|✅|❌|❌|
| 2.0                   |❌|❌|❌|❌|❌|✅|✅|❌|❌
| 1.7                   |❌|❌|❌|❌|❌|✅|✅|✅|✅|

We recommend not using TimescaleDB with Postgres 17.1, 16.5, 15.9, 14.14, 13.17, 12.21.
These minor versions [introduced a breaking binary interface change][postgres-breaking-change] that,
once identified, was reverted in subsequent minor Postgres versions 17.2, 16.6, 15.10, 14.15, 13.18, and 12.22.
When you build from source, best practice is to build with Postgres 17.2, 16.6, etc and higher.
Users of [Tiger Cloud](https://console.cloud.timescale.com/) and platform packages for Linux, Windows, MacOS,
Docker, and Kubernetes are unaffected.

### Supported operating system

You can deploy TimescaleDB and TimescaleDB Toolkit on the following systems:





| Operation system                | Version                                                               |
|---------------------------------|-----------------------------------------------------------------------|
| Debian                          | 13 Trixe, 12 Bookworm, 11 Bullseye                                   |
| Ubuntu                          | 24.04 Noble Numbat, 22.04 LTS Jammy Jellyfish |
| Red Hat Enterprise              | Linux 9, Linux 8                                             |
| Fedora                          | Fedora 35, Fedora 34, Fedora 33                                       |
| Rocky Linux                     | Rocky Linux 9 (x86_64), Rocky Linux 8                                 |
| ArchLinux (community-supported) | Check the [available packages][archlinux-packages]                    |





| Operation system                            | Version    |
|---------------------------------------------|------------|
| Microsoft Windows                           | 10, 11     |
| Microsoft Windows Server                    | 2019, 2020 |





| Operation system              | Version                          |
|-------------------------------|----------------------------------|
| macOS                         | From 10.15 Catalina to 14 Sonoma |


===== PAGE: https://docs.tigerdata.com/about/contribute-to-timescale/ =====

# Contribute to Tiger Data

TimescaleDB, pgai, pgvectorscale, TimescaleDB Toolkit, and the Tiger Data documentation are all open source. They are available in GitHub for you to use, review, and update. This page shows you where you can add to Tiger Data products.

## Contribute to the code for Tiger Data products

Tiger Data appreciates any help the community can provide to make its products better! You can:

* Open an issue with a bug report, build issue, feature request or suggestion.
* Fork a corresponding repository and submit a pull request.

Head over to the Tiger Data source repositories to learn, review, and help improve our products!

* [TimescaleDB][timescaledb]: a Postgres extension for high-performance real-time analytics on time-series and event data.
* [pgai][pgai]: a suite of tools to develop RAG, semantic search, and other AI applications more easily with Postgres.
* [pgvectorscale][pgvectorscale]: a complement to pgvector for higher performance embedding search and cost-efficient storage for AI applications.
* [TimescaleDB Toolkit][toolkit]: all things analytics when using TimescaleDB, with a particular focus on developer ergonomics and performance.

## Contribute to Tiger Data documentation

Tiger Data documentation is hosted in the [docs GitHub repository][github-docs]
and open for contribution from all community members.

See the [README][readme] and [contribution guide][contribution-guide] for details.


===== PAGE: https://docs.tigerdata.com/about/release-notes/ =====

# Release notes

For information about new updates and improvement to Tiger Data products, see the [Changelog][changelog]. For release
notes about our downloadable products, see:

* [TimescaleDB](https://github.com/timescale/timescaledb/releases) -  an open-source database that makes SQL scalable
  for time-series data, packaged as a Postgres extension.
* [TimescaleDB Toolkit](https://github.com/timescale/timescaledb-toolkit/releases) - additional functions to ease all things analytics
  when using TimescaleDB.
* [pgai](https://github.com/timescale/pgai/releases) - brings AI workflows to your Postgres database.
* [pgvectorscale](https://github.com/timescale/pgvectorscale/releases/tag/0.2.0) -  higher performance embedding search and cost-efficient storage for AI applications on Postgres.
* [pgspot](https://github.com/timescale/pgspot/releases) - spot vulnerabilities in Postgres extension scripts.
* [live-migration](https://hub.docker.com/r/timescale/live-migration/tags) - a Docker image to migrate data to a Tiger Cloud service.



Want to stay up-to-date with new releases? On the main page for each repository
click `Watch`, select `Custom` and then check `Releases`.


===== PAGE: https://docs.tigerdata.com/migrate/livesync-for-postgresql/ =====

# Sync data from Postgres to your service



You use the source Postgres connector in Tiger Cloud to synchronize all data or specific tables from a Postgres database instance to your
service, in real time. You run the connector continuously, turning Postgres into a primary database with your
service as a logical replica. This enables you to leverage Tiger Cloud’s real-time analytics capabilities on
your replica data.

![Tiger Cloud connectors overview](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-connector-overview.png)

The source Postgres connector in Tiger Cloud leverages the well-established Postgres logical replication protocol. By relying on this protocol,
Tiger Cloud ensures compatibility, familiarity, and a broader knowledge base—making it easier for you to adopt the connector
and integrate your data.

You use the source Postgres connector for data synchronization, rather than migration. This includes:

* Copy existing data from a Postgres instance to a Tiger Cloud service:
  - Copy data at up to 150 GB/hr.

    You need at least a 4 CPU/16 GB source database, and a 4 CPU/16 GB target service.
  - Copy the publication tables in parallel.

    Large tables are still copied using a single connection. Parallel copying is in the backlog.
  - Forget foreign key relationships.

    The connector disables foreign key validation during the sync. For example, if a `metrics` table refers to
    the `id` column on the `tags` table, you can still sync only the `metrics` table without worrying about their
    foreign key relationships.
  - Track progress.

    Postgres exposes `COPY` progress under `pg_stat_progress_copy`.

* Synchronize real-time changes from a Postgres instance to a Tiger Cloud service.
* Add and remove tables on demand using the [Postgres PUBLICATION interface][postgres-publication-interface].
* Enable features such as [hypertables][about-hypertables], [columnstore][compression], and
   [continuous aggregates][caggs] on your logical replica.

Early access: this source Postgres connector is not yet supported for production use. If you have any questions or feedback, talk to us in <a href="https://app.slack.com/client/T4GT3N2JK/C086NU9EZ88">#livesync in the Tiger Community</a>.





## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with real-time analytics enabled.

  You need your [connection details][connection-info].

- Install the [Postgres client tools][install-psql] on your sync machine.

- Ensure that the source Postgres instance and the target Tiger Cloud service have the same extensions installed.

  The source Postgres connector does not create extensions on the target. If the table uses column types from an extension,
    first create the extension on the target Tiger Cloud service before syncing the table.

## Limitations

* The source Postgres instance must be accessible from the Internet.

  Services hosted behind a firewall or VPC are not supported. This functionality is on the roadmap.

* Indexes, including the primary key and unique constraints, are not migrated to the target Tiger Cloud service.

  We recommend that, depending on your query patterns, you create only the necessary indexes on the target Tiger Cloud service.

* This works for Postgres databases only as source. TimescaleDB is not yet supported.

* The source must be running Postgres 13 or later.

* Schema changes must be co-ordinated.

  Make compatible changes to the schema in your Tiger Cloud service first, then make
  the same changes to the source Postgres instance.

* Ensure that the source Postgres instance and the target Tiger Cloud service have the same extensions installed.

  The source Postgres connector does not create extensions on the target. If the table uses
  column types from an extension, first create the extension on the
  target Tiger Cloud service before syncing the table.

* There is WAL volume growth on the source Postgres instance during large table copy.

* Continuous aggregate invalidation

  The connector uses `session_replication_role=replica` during data replication,
  which prevents table triggers from firing. This includes the internal
  triggers that mark continuous aggregates as invalid when underlying data
  changes.

  If you have continuous aggregates on your target database, they do not
  automatically refresh for data inserted during the migration. This limitation
  only applies to data below the continuous aggregate's materialization
  watermark. For example, backfilled data. New rows synced above the continuous
  aggregate watermark are used correctly when refreshing.

  This can lead to:

  - Missing data in continuous aggregates for the migration period.
  - Stale aggregate data.
  - Queries returning incomplete results.

  If the continuous aggregate exists in the source database, best
  practice is to add it to the Postgres connector publication. If it only exists on the
  target database, manually refresh the continuous aggregate using the `force`
  option of [refresh_continuous_aggregate][refresh-caggs].

## Set your connection string

This variable holds the connection information for the source database. In the terminal on your migration machine,
set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
```



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool
requires a direct connection to the database to function properly.



## Tune your source database





Updating parameters on a Postgres instance will cause an outage. Choose a time that will cause the least issues to tune this database.

1. **Tune the Write Ahead Log (WAL) on the RDS/Aurora Postgres source database**

   1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
      select the RDS instance to migrate.

   1. Click `Configuration`, scroll down and note the `DB instance parameter group`, then click `Parameter Groups`

      <img class="main-content__illustration"
      src="https://assets.timescale.com/docs/images/migrate/awsrds-parameter-groups.png"
      alt="Create security rule to enable RDS EC2 connection"/>

   1. Click `Create parameter group`, fill in the form with the following values, then click `Create`.
      - **Parameter group name** - whatever suits your fancy.
      - **Description** - knock yourself out with this one.
      - **Engine type** - `PostgreSQL`
      - **Parameter group family** - the same as `DB instance parameter group` in your `Configuration`.
   1. In `Parameter groups`, select the parameter group you created, then click `Edit`.
   1. Update the following parameters, then click `Save changes`.
      - `rds.logical_replication` set to `1`: record the information needed for logical decoding.
      - `wal_sender_timeout` set to `0`: disable the timeout for the sender process.

   1. In RDS, navigate back to your [databases][databases], select the RDS instance to migrate, and click `Modify`.

   1. Scroll down to `Database options`, select your new parameter group, and click `Continue`.
   1. Click `Apply immediately` or choose a maintenance window, then click `Modify DB instance`.

      Changing parameters will cause an outage. Wait for the database instance to reboot before continuing.
   1. Verify that the settings are live in your database.

1. **Create a user for the source Postgres connector and assign permissions**

   1. Create `<pg connector username>`:

      ```sql
      psql source -c "CREATE USER <pg connector username> PASSWORD '<password>'"
      ```

      You can use an existing user. However, you must ensure that the user has the following permissions.

   1. Grant permissions to create a replication slot:

      ```sql
      psql source -c "GRANT rds_replication TO <pg connector username>"
      ```

   1. Grant permissions to create a publication:

      ```sql
      psql source -c "GRANT CREATE ON DATABASE <database name> TO <pg connector username>"
      ```

   1. Assign the user permissions on the source database:

      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA "public" TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA "public" TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

      If the tables you are syncing are not in the `public` schema, grant the user permissions for each schema you are syncing:
      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA <schema> TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA <schema> TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA <schema> GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

   1. On each table you want to sync, make `<pg connector username>` the owner:

      ```sql
      psql source -c 'ALTER TABLE  OWNER TO <pg connector username>;'
      ```
      You can skip this step if the replicating user is already the owner of the tables.

1. **Enable replication `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.




1. **Tune the Write Ahead Log (WAL) on the Postgres source database**

   ```sql
   psql source <<EOF
   ALTER SYSTEM SET wal_level='logical';
   ALTER SYSTEM SET max_wal_senders=10;
   ALTER SYSTEM SET wal_sender_timeout=0;
   EOF
   ```
   * [GUC “wal_level” as “logical”](https://www.postgresql.org/docs/current/runtime-config-wal.html#GUC-WAL-LEVEL)
   * [GUC “max_wal_senders” as 10](https://www.postgresql.org/docs/current/runtime-config-replication.html#GUC-MAX-WAL-SENDERS)
   * [GUC “wal_sender_timeout” as 0](https://www.postgresql.org/docs/current/runtime-config-replication.html#GUC-WAL-SENDER-TIMEOUT)

   This will require a restart of the Postgres source database.

1. **Create a user for the connector and assign permissions**

   1. Create `<pg connector username>`:

      ```sql
      psql source -c "CREATE USER <pg connector username> PASSWORD '<password>'"
      ```

      You can use an existing user. However, you must ensure that the user has the following permissions.

   1. Grant permissions to create a replication slot:

      ```sql
      psql source -c "ALTER ROLE <pg connector username> REPLICATION"
      ```

   1. Grant permissions to create a publication:

      ```sql
      psql source -c "GRANT CREATE ON DATABASE <database name> TO <pg connector username>"
      ```

   1. Assign the user permissions on the source database:

      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA "public" TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA "public" TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

      If the tables you are syncing are not in the `public` schema, grant the user permissions for each schema you are syncing:
      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA <schema> TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA <schema> TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA <schema> GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

   1. On each table you want to sync, make `<pg connector username>` the owner:

      ```sql
      psql source -c 'ALTER TABLE  OWNER TO <pg connector username>;'
      ```
      You can skip this step if the replicating user is already the owner of the tables.


1. **Enable replication `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.




## Synchronize data to your Tiger Cloud service

To sync data from your Postgres database to your Tiger Cloud service using Tiger Cloud Console:

1. **Connect to your Tiger Cloud service**

   In [Tiger Cloud Console][portal-ops-mode], select the service to sync live data to.

1. **Connect the source database and the target service**

   ![Postgres connector wizard](https://assets.timescale.com/docs/images/tiger-cloud-console/pg-connector-wizard-tiger-console.png)

   1. Click `Connectors` > `PostgreSQL`.
   1. Set the name for the new connector by clicking the pencil icon.
   1. Check the boxes for `Set wal_level to logical` and `Update your credentials`, then click `Continue`.
   1. Enter your database credentials or a Postgres connection string, then click `Connect to database`.
      This is the connection string for [`<pg connector username>`][livesync-tune-source-db]. Tiger Cloud Console connects to the source database and retrieves the schema information.

1. **Optimize the data to synchronize in hypertables**

   ![Postgres connector start](https://assets.timescale.com/docs/images/tiger-cloud-console/pg-connector-start-tiger-console.png)

   1. In the `Select table` dropdown, select the tables to sync.
   1. Click `Select tables +` .

      Tiger Cloud Console checks the table schema and, if possible, suggests the column to use as the time dimension in a hypertable.
   1. Click `Create Connector`.

      Tiger Cloud Console starts source Postgres connector between the source database and the target service and displays the progress.

1. **Monitor synchronization**

   ![Tiger Cloud connectors overview](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-connector-overview.png)

    1. To view the amount of data replicated, click `Connectors`. The diagram in `Connector data flow` gives you an overview of the connectors you have created, their status, and how much data has been replicated.

    1. To review the syncing progress for each table, click `Connectors` > `Source connectors`, then select the name of your connector in the table.

1. **Manage the connector**

   ![Edit a Postgres connector](https://assets.timescale.com/docs/images/tiger-cloud-console/edit-pg-connector-tiger-console.png)

   1. To edit the connector, click `Connectors` > `Source connectors`, then select the name of your connector in the table. You can rename the connector, delete or add new tables for syncing.

   1. To pause a connector, click `Connectors` > `Source connectors`, then open the three-dot menu on the right and select `Pause`.

   1. To delete a connector, click `Connectors` > `Source connectors`, then open the three-dot menu on the right and select `Delete`. You must pause the connector before deleting it.

And that is it, you are using the source Postgres connector to synchronize all the data, or specific tables, from a Postgres database
instance to your Tiger Cloud service, in real time.




## Prerequisites

Best practice is to use an [Ubuntu EC2 instance][create-ec2-instance] hosted in the same region as your
Tiger Cloud service to move data. That is, the machine you run the commands on to move your
data from your source database to your target Tiger Cloud service.

Before you move your data:

- Create a target [Tiger Cloud service][created-a-database-service-in-timescale].

  Each Tiger Cloud service has a single Postgres instance that supports the
  [most popular extensions][all-available-extensions]. Tiger Cloud services do not support tablespaces,
  and there is no superuser associated with a service.
  Best practice is to create a Tiger Cloud service with at least 8 CPUs for a smoother experience. A higher-spec instance
  can significantly reduce the overall migration window.

- To ensure that maintenance does not run while migration is in progress, best practice is to [adjust the maintenance window][adjust-maintenance-window].

- Ensure that the source Postgres instance and the target Tiger Cloud service have the same extensions installed.

  The source Postgres connector does not create extensions on the target. If the table uses column types from an extension,
  first create the extension on the target Tiger Cloud service before syncing the table.

- [Install Docker][install-docker] on your sync machine.

  For a better experience, use a 4 CPU/16GB EC2 instance or greater to run the source Postgres connector.

- Install the [Postgres client tools][install-psql] on your sync machine.

  This includes `psql`, `pg_dump`, `pg_dumpall`, and `vacuumdb` commands.

## Limitations

- The schema is not migrated by the source Postgres connector, you use `pg_dump`/`pg_restore` to migrate it.

* This works for Postgres databases only as source. TimescaleDB is not yet supported.

* The source must be running Postgres 13 or later.

* Schema changes must be co-ordinated.

  Make compatible changes to the schema in your Tiger Cloud service first, then make
  the same changes to the source Postgres instance.

* Ensure that the source Postgres instance and the target Tiger Cloud service have the same extensions installed.

  The source Postgres connector does not create extensions on the target. If the table uses
  column types from an extension, first create the extension on the
  target Tiger Cloud service before syncing the table.

* There is WAL volume growth on the source Postgres instance during large table copy.

* Continuous aggregate invalidation

  The connector uses `session_replication_role=replica` during data replication,
  which prevents table triggers from firing. This includes the internal
  triggers that mark continuous aggregates as invalid when underlying data
  changes.

  If you have continuous aggregates on your target database, they do not
  automatically refresh for data inserted during the migration. This limitation
  only applies to data below the continuous aggregate's materialization
  watermark. For example, backfilled data. New rows synced above the continuous
  aggregate watermark are used correctly when refreshing.

  This can lead to:

  - Missing data in continuous aggregates for the migration period.
  - Stale aggregate data.
  - Queries returning incomplete results.

  If the continuous aggregate exists in the source database, best
  practice is to add it to the Postgres connector publication. If it only exists on the
  target database, manually refresh the continuous aggregate using the `force`
  option of [refresh_continuous_aggregate][refresh-caggs].

## Set your connection strings

The `<user>` in the `SOURCE` connection must have the replication role granted in order to create a replication slot.

These variables hold the connection information for the source database and target Tiger Cloud service.
In Terminal on your migration machine, set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
```
You find the connection information for your Tiger Cloud service in the configuration file you
downloaded when you created the service.



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool requires a direct connection to the database to function properly.

## Tune your source database





Updating parameters on a Postgres instance will cause an outage. Choose a time that will cause the least issues to tune this database.

1. **Update the DB instance parameter group for your source database**

   1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
      select the RDS instance to migrate.

   1. Click `Configuration`, scroll down and note the `DB instance parameter group`, then click `Parameter groups`

      <img class="main-content__illustration"
      src="https://assets.timescale.com/docs/images/migrate/awsrds-parameter-groups.png"
      alt="Create security rule to enable RDS EC2 connection"/>

   1. Click `Create parameter group`, fill in the form with the following values, then click `Create`.
      - **Parameter group name** - whatever suits your fancy.
      - **Description** - knock yourself out with this one.
      - **Engine type** - `PostgreSQL`
      - **Parameter group family** - the same as `DB instance parameter group` in your `Configuration`.
   1. In `Parameter groups`, select the parameter group you created, then click `Edit`.
   1. Update the following parameters, then click `Save changes`.
      - `rds.logical_replication` set to `1`: record the information needed for logical decoding.
      - `wal_sender_timeout` set to `0`: disable the timeout for the sender process.

   1. In RDS, navigate back to your [databases][databases], select the RDS instance to migrate, and click `Modify`.

   1. Scroll down to `Database options`, select your new parameter group, and click `Continue`.
   1. Click `Apply immediately` or choose a maintenance window, then click `Modify DB instance`.

      Changing parameters will cause an outage. Wait for the database instance to reboot before continuing.
   1. Verify that the settings are live in your database.

1. **Enable replication `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.




1. **Tune the Write Ahead Log (WAL) on the Postgres source database**

   ```sql
   psql source <<EOF
   ALTER SYSTEM SET wal_level='logical';
   ALTER SYSTEM SET max_wal_senders=10;
   ALTER SYSTEM SET wal_sender_timeout=0;
   EOF
   ```
   * [GUC “wal_level” as “logical”](https://www.postgresql.org/docs/current/runtime-config-wal.html#GUC-WAL-LEVEL)
   * [GUC “max_wal_senders” as 10](https://www.postgresql.org/docs/current/runtime-config-replication.html#GUC-MAX-WAL-SENDERS)
   * [GUC “wal_sender_timeout” as 0](https://www.postgresql.org/docs/current/runtime-config-replication.html#GUC-WAL-SENDER-TIMEOUT)

   This will require a restart of the Postgres source database.

1. **Create a user for the connector and assign permissions**

   1. Create `<pg connector username>`:

      ```sql
      psql source -c "CREATE USER <pg connector username> PASSWORD '<password>'"
      ```

      You can use an existing user. However, you must ensure that the user has the following permissions.

   1. Grant permissions to create a replication slot:

      ```sql
      psql source -c "ALTER ROLE <pg connector username> REPLICATION"
      ```

   1. Grant permissions to create a publication:

      ```sql
      psql source -c "GRANT CREATE ON DATABASE <database name> TO <pg connector username>"
      ```

   1. Assign the user permissions on the source database:

      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA "public" TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA "public" TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

      If the tables you are syncing are not in the `public` schema, grant the user permissions for each schema you are syncing:
      ```sql
      psql source <<EOF
      GRANT USAGE ON SCHEMA <schema> TO <pg connector username>;
      GRANT SELECT ON ALL TABLES IN SCHEMA <schema> TO <pg connector username>;
      ALTER DEFAULT PRIVILEGES IN SCHEMA <schema> GRANT SELECT ON TABLES TO <pg connector username>;
      EOF
      ```

   1. On each table you want to sync, make `<pg connector username>` the owner:

      ```sql
      psql source -c 'ALTER TABLE  OWNER TO <pg connector username>;'
      ```
      You can skip this step if the replicating user is already the owner of the tables.


1. **Enable replication `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.




## Migrate the table schema to the Tiger Cloud service

Use `pg_dump` to:

1. **Download the schema from the source database**

  ```shell
  pg_dump source \
  --no-privileges \
  --no-owner \
  --no-publications \
  --no-subscriptions \
  --no-table-access-method \
  --no-tablespaces \
  --schema-only \
  --file=schema.sql
  ```

1. **Apply the schema on the target service**
  ```shell
  psql target -f schema.sql
  ```

## Convert partitions and tables with time-series data into hypertables

For efficient querying and analysis, you can convert tables which contain time-series or
events data, and tables that are already partitioned using Postgres declarative partition into
[hypertables][about-hypertables].

1. **Convert tables to hypertables**

   Run the following on each table in the target Tiger Cloud service to convert it to a hypertable:

   ```shell
   psql -X -d target -c "SELECT public.create_hypertable('', by_range('<partition column>', '<chunk interval>'::interval));"
   ```

   For example, to convert the *metrics* table into a hypertable with *time* as a partition column and
   *1 day* as a partition interval:

   ```shell
   psql -X -d target -c "SELECT public.create_hypertable('public.metrics', by_range('time', '1 day'::interval));"
   ```

1. **Convert Postgres partitions to hypertables**

   Rename the partition and create a new regular table with the same name as the partitioned table, then
   convert to a hypertable:

   ```shell
   psql target -f - <<'EOF'
      BEGIN;
      ALTER TABLE public.events RENAME TO events_part;
      CREATE TABLE public.events(LIKE public.events_part INCLUDING ALL);
      SELECT create_hypertable('public.events', by_range('time', '1 day'::interval));
      COMMIT;
EOF
   ```


## Specify the tables to synchronize

After the schema is migrated, you [`CREATE PUBLICATION`][create-publication] on the source database that
specifies the tables to synchronize.

1. **Create a publication that specifies the table to synchronize**

   A `PUBLICATION` enables you to synchronize some or all the tables in the schema or database.

   ```sql
   CREATE PUBLICATION <publication_name> FOR TABLE , ;
   ```

    To add tables after to an existing publication, use [ALTER PUBLICATION][alter-publication]**

   ```sql
   ALTER PUBLICATION <publication_name> ADD TABLE ;
   ```

1. **Publish the Postgres declarative partitioned table**

   ```sql
   ALTER PUBLICATION <publication_name> SET(publish_via_partition_root=true);
   ```

   To convert partitioned table to hypertable, follow [Convert partitions and tables with time-series data into hypertables](#convert-partitions-and-tables-with-time-series-data-into-hypertables).

1. **Stop syncing a table in the `PUBLICATION`, use `DROP TABLE`**

   ```sql
   ALTER PUBLICATION <publication_name> DROP TABLE ;
   ```


## Synchronize data to your Tiger Cloud service

You use the source Postgres connector docker image to synchronize changes in real time from a Postgres database
instance to a Tiger Cloud service:

1. **Start the source Postgres connector**

   As you run the source Postgres connector continuously, best practice is to run it as a Docker daemon.

   ```shell
   docker run -d --rm --name livesync timescale/live-sync:v0.1.25 run \
      --publication <publication_name> --subscription <subscription_name> \
      --source source --target target --table-map
   ```

   `--publication`: The name of the publication as you created in the previous step. To use multiple publications, repeat the `--publication` flag.

   `--subscription`: The name that identifies the subscription on the target Tiger Cloud service.

   `--source`: The connection string to the source Postgres database.

   `--target`: The connection string to the target Tiger Cloud service.

   `--table-map`: (Optional) A JSON string that maps source tables to target tables. If not provided, the source and target table names are assumed to be the same.
   For example, to map the source table `metrics` to the target table `metrics_data`:

   ```
   --table-map '{"source": {"schema": "public", "table": "metrics"}, "target": {"schema": "public", "table": "metrics_data"}}'
   ```
   To map only the schema, use:

   ```
   --table-map '{"source": {"schema": "public"}, "target": {"schema": "analytics"}}'
   ```
   This flag can be repeated for multiple table mappings.

1. **Capture logs**

   Once the source Postgres connector is running as a docker daemon, you can also capture the logs:
   ```shell
   docker logs -f livesync
   ```

1. **View the progress of tables being synchronized**

   List the tables being synchronized by the source Postgres connector using the `_ts_live_sync.subscription_rel` table in the target Tiger Cloud service:

   ```bash
   psql target -c "SELECT * FROM _ts_live_sync.subscription_rel"
   ```

   You see something like the following:

   | subname  | pubname | schemaname | tablename | rrelid | state |    lsn     |          updated_at           |                                  last_error                                   |          created_at           | rows_copied | approximate_rows | bytes_copied | approximate_size | target_schema | target_table |
   |----------|---------|-------------|-----------|--------|-------|------------|-------------------------------|-------------------------------------------------------------------------------|-------------------------------|-------------|------------------|--------------|------------------|---------------|-------------|
 |livesync | analytics | public     | metrics   |  20856 | r     | 6/1A8CBA48 | 2025-06-24 06:16:21.434898+00 |                                                                               | 2025-06-24 06:03:58.172946+00 |    18225440 |         18225440 |   1387359359 |       1387359359 | public        | metrics  |

   The `state` column indicates the current state of the table synchronization.
   Possible values for `state` are:

   | state | description |
   |-------|-------------|
   | d | initial table data sync |
   | f | initial table data sync completed |
   | s | catching up with the latest changes |
   | r | table is ready, syncing live changes |

   To see the replication lag, run the following against the SOURCE database:

   ```bash
   psql source -f - <<'EOF'
   SELECT
      slot_name,
      pg_size_pretty(pg_current_wal_flush_lsn() - confirmed_flush_lsn) AS lag
   FROM pg_replication_slots
   WHERE slot_name LIKE 'live_sync_%' AND slot_type = 'logical'
EOF
   ```

1. **Add or remove tables from the publication**

   To add tables, use [ALTER PUBLICATION .. ADD TABLE][alter-publication]**

   ```sql
   ALTER PUBLICATION <publication_name> ADD TABLE ;
   ```

   To remove tables, use [ALTER PUBLICATION .. DROP TABLE][alter-publication]**

   ```sql
   ALTER PUBLICATION <publication_name> DROP TABLE ;
   ```

1. **Update table statistics**

   If you have a large table, you can run `ANALYZE` on the target Tiger Cloud service
   to update the table statistics after the initial sync is complete.

   This helps the query planner make better decisions for query execution plans.

   ```bash
   vacuumdb --analyze --verbose --dbname=target
   ```

1. **Stop the source Postgres connector**

   ```shell
   docker stop live-sync
   ```

1. **(Optional) Reset sequence nextval on the target Tiger Cloud service**

   The source Postgres connector does not automatically reset the sequence nextval on the target
   Tiger Cloud service.

   Run the following script to reset the sequence for all tables that have a
   serial or identity column in the target Tiger Cloud service:

   ```bash
   psql target -f - <<'EOF'
      DO $$
   DECLARE
     rec RECORD;
   BEGIN
     FOR rec IN (
       SELECT
         sr.target_schema  AS table_schema,
         sr.target_table   AS table_name,
         col.column_name,
         pg_get_serial_sequence(
           sr.target_schema || '.' || sr.target_table,
           col.column_name
         ) AS seqname
       FROM _ts_live_sync.subscription_rel AS sr
       JOIN information_schema.columns AS col
         ON col.table_schema = sr.target_schema
        AND col.table_name   = sr.target_table
       WHERE col.column_default LIKE 'nextval(%'  -- only serial/identity columns
     ) LOOP
       EXECUTE format(
         'SELECT setval(%L,
            COALESCE((SELECT MAX(%I) FROM %I.%I), 0) + 1,
            false
          );',
         rec.seqname,       -- the sequence identifier
         rec.column_name,   -- the column to MAX()
         rec.table_schema,  -- schema for MAX()
         rec.table_name     -- table for MAX()
       );
     END LOOP;
   END;
   $$ LANGUAGE plpgsql;
EOF
   ```

1. **Clean up**

   Use the `--drop` flag to remove the replication slots created by the source Postgres connector on the source database.

   ```shell
   docker run -it --rm --name livesync timescale/live-sync:v0.1.25 run \
      --publication <publication_name> --subscription <subscription_name> \
      --source source --target target \
      --drop
   ```


===== PAGE: https://docs.tigerdata.com/migrate/livesync-for-s3/ =====

# Sync data from S3 to your service



You use the source S3 connector in Tiger Cloud to synchronize CSV and Parquet files from an S3 bucket to your Tiger Cloud service in real time. The connector runs continuously, enabling you to leverage Tiger Cloud as your analytics database with data constantly synced from S3. This lets you take full advantage of Tiger Cloud's real-time analytics capabilities without having to develop or manage custom ETL solutions between S3 and Tiger Cloud.

![Tiger Cloud overview](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-connector-overview.png)

You can use the source S3 connector to synchronize your existing and new data. Here's what the connector can do:

* Sync data from an S3 bucket instance to a Tiger Cloud service:
    - Use glob patterns to identify the objects to sync.
    - Watch an S3 bucket for new files and import them automatically. It runs on a configurable schedule and tracks processed files.
    - **Important**: The connector processes files in [lexicographical order][lex-order]. It uses the name of the last file processed as a marker and fetches only files later in the alphabet in subsequent queries. Files added with names earlier in the alphabet than the marker are skipped and never synced. For example, if you add the file Bob when the marker is at Elephant, Bob is never processed.
    - For large backlogs, check every minute until caught up.

* Sync data from multiple file formats:
    - CSV: check for compression in GZ and ZIP format, then process using [timescaledb-parallel-copy][parallel-copy].
    - Parquet: convert to CSV, then process using [timescaledb-parallel-copy][parallel-copy].

* The source S3 connector offers an option to enable a [hypertable][about-hypertables] during the file-to-table schema mapping setup. You can enable [columnstore][compression] and [continuous aggregates][caggs] through the SQL editor once the connector has started running.

* The connector offers a default 1-minute polling interval. This means that Tiger Cloud checks the S3 source every minute for new data. You can customize this interval by setting up a cron expression.

The source S3 connector continuously imports data from an Amazon S3 bucket into your database. It monitors your S3 bucket for new files matching a specified pattern and automatically imports them into your designated database table.

**Note**: the connector currently only syncs existing and new files—it does not support updating or deleting records based on updates and deletes from S3 to tables in a Tiger Cloud service.

Early access: this source S3 connector is not supported for production use. If you have any questions or feedback, talk to us in <a href="https://app.slack.com/client/T4GT3N2JK/C086NU9EZ88">#livesync in the Tiger Community</a>.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with real-time analytics enabled.

  You need your [connection details][connection-info].

- Ensure access to a standard Amazon S3 bucket containing your data files.

  Directory buckets are not supported.
- Configure access credentials for the S3 bucket.
  The following credentials are supported:
    - [IAM Role][credentials-iam].

      - Configure the trust policy. Set the:

        - `Principal`: `arn:aws:iam::142548018081:role/timescale-s3-connections`.
        - `ExternalID`: set to the [Tiger Cloud project and Tiger Cloud service ID][connection-project-service-id] of the
           service you are syncing to in the format `<projectId>/<serviceId>`.

           This is to avoid the [confused deputy problem][confused-deputy-problem].
      - Give the following access permissions:

        - `s3:GetObject`.
        - `s3:ListBucket`.

    - [Public anonymous user][credentials-public].

## Limitations

- **File naming**:
  Files must follow lexicographical ordering conventions. Files with names that sort earlier than already-processed files are permanently skipped. Example: if `file_2024_01_15.csv` has been processed, a file named `file_2024_01_10.csv` added later will never be synced.
  Recommended naming patterns: timestamps (for example, `YYYY-MM-DD-HHMMSS`), sequential numbers with fixed padding (for example, `file_00001`, `file_00002`).

- **CSV**:
   - Maximum file size: 1 GB

      To increase this limit, contact sales@tigerdata.com
   - Maximum row size: 2 MB
   - Supported compressed formats:
      - GZ
      - ZIP
   - Advanced settings:
      - Delimiter: the default character is `,`, you can choose a different delimiter
      - Skip header: skip the first row if your file has headers
- **Parquet**:
   - Maximum file size: 1 GB
   - Maximum row size: 2 MB
- **Sync iteration**:

   To prevent system overload, the connector tracks up to 100 files for each sync iteration. Additional checks only fill
   empty queue slots.

## Synchronize data to your Tiger Cloud service

To sync data from your S3 bucket to your Tiger Cloud service using Tiger Cloud Console:

1. **Connect to your Tiger Cloud service**

   In [Tiger Cloud Console][portal-ops-mode], select the service to sync live data to.

1. **Connect the source S3 bucket to the target service**

   ![Connect Tiger Cloud to S3 bucket](https://assets.timescale.com/docs/images/tiger-cloud-console/s3-connector-tiger-console.png)

   1. Click `Connectors` > `Amazon S3`.
   1. Click the pencil icon, then set the name for the new connector.
   1. Set the `Bucket name` and `Authentication method`, then click `Continue`.

      For instruction on creating the IAM role to connect your S3 bucket, click `Learn how`. Tiger Cloud Console connects to the source bucket.
   1. In `Define files to sync`, choose the `File type` and set the `Glob pattern`.

      Use the following patterns:
      - `<folder name>/*`: match all files in a folder. Also, any pattern ending with `/` is treated as  `/*`.
      - `<folder name>/**`: match all recursively.
      - `<folder name>/**/*.csv`: match a specific file type.

      The source S3 connector uses prefix filters where possible, place patterns carefully at the end of your glob expression.
      AWS S3 doesn't support complex filtering. If your expression filters too many files, the list operation may time out.

   1. Click the search icon. You see the files to sync. Click `Continue`.

1. **Optimize the data to synchronize in hypertables**

   ![S3 connector table selection](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-s3-connector-create-tables.png)

   Tiger Cloud Console checks the file schema and, if possible, suggests the column to use as the time dimension in a
   [hypertable][about-hypertables].

   1. Choose `Create a new table for your data` or `Ingest data to an existing table`.
   1. Choose the `Data type` for each column, then click `Continue`.
   1. Choose the interval. This can be a minute, an hour, or use a [cron expression][cron-expression].
   1. Click `Start Connector`.

      Tiger Cloud Console starts the connection between the source database and the target service and displays the progress.

1. **Monitor synchronization**

    1. To view the amount of data replicated, click `Connectors`. The diagram in `Connector data flow` gives you an overview of the connectors you have created, their status, and how much data has been replicated.

       ![Tiger Cloud connectors overview](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-connector-overview.png)

    1. To view file import statistics and logs, click `Connectors` > `Source connectors`, then select the name of your connector in the table.

       ![S3 connector stats](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-s3-connector-import-stats.png)


1. **Manage the connector**

    1. To pause the connector, click `Connectors` > `Source connectors`. Open the three-dot menu next to your connector in the table, then click `Pause`.

      ![Edit S3 connector](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-s3-connector-pause.png)

    1. To edit the connector, click `Connectors` > `Source connectors`. Open the three-dot menu next to your connector in the table, then click `Edit` and scroll down to `Modify your Connector`. You must pause the connector before editing it.

      ![S3 connector change config](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-s3-connector-edit.png)

    1. To pause or delete the connector, click `Connectors` > `Source connectors`, then open the three-dot menu on the right and select an option. You must pause the connector before deleting it.

And that is it, you are using the source S3 connector to synchronize all the data, or specific files, from an S3 bucket to your
Tiger Cloud service in real time.


===== PAGE: https://docs.tigerdata.com/migrate/livesync-for-kafka/ =====

# Stream data from Kafka into your service



You use the Kafka source connector in Tiger Cloud to stream events from Kafka into your service. Tiger Cloud connects to your Confluent Cloud Kafka cluster and Schema Registry using SASL/SCRAM authentication and service account–based API keys. Only the Avro format is currently supported [with some limitations][limitations].

This page explains how to connect Tiger Cloud to your Confluence Cloud Kafka cluster.

Early access: the Kafka source connector is not yet supported for production use.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with real-time analytics enabled.

  You need your [connection details][connection-info].

- [Sign up][confluence-signup] for Confluence Cloud.
- [Create][create-kafka-cluster] a Kafka cluster in Confluence Cloud.

## Access your Kafka cluster in Confluent Cloud

Take the following steps to prepare your Kafka cluster for connection to Tiger Cloud:

1. **Create a service account**

       If you already have a service account for Tiger Cloud, you can reuse it.  To create a new service account:

       1. Log in to [Confluent Cloud][confluent-cloud].
       1. Click the burger menu at the top-right of the pane, then press
          `Access control` > `Service accounts` >`Add service account`.
       1. Enter the following details:

          - Name: `tigerdata-access`
          - Description: `Service account for the Tiger Cloud source connector`

       1. Add the service account owner role, then click `Next`.

       1. Select a role assignment, then click `Add`

       1. Click `Next`, then click `Create service account`.

   1. **Create API keys**

       1. In Confluent Cloud, click `Home` > `Environments` > Select your environment > Select your cluster.
       1. Under `Cluster overview` in the left sidebar, select `API Keys`.
       1. Click `Add key`, choose `Service Account` and click `Next`.
       1. Select `tigerdata-access`, then click `Next`.
       1. For your cluster, choose the `Operation` and select the following `Permission`s, then click `Next`:
          - `Resource type`: `Cluster`
          - `Operation`: `DESCRIBE`
          - `Permission`: `ALLOW`
       1. Click `Download and continue`, then securely store the ACL.
       1. Use the same procedure to add the following keys:
          - ACL 2: Topic access
            - `Resource type`: `Topic`
            - `Topic name`: Select the topics that Tiger Cloud should read
            - `Pattern type`: `LITERAL`
            - `Operation`: `READ`
            - `Permission`: `ALLOW`
          - ACL 3: Consumer group access
            - `Resource type`: `Consumer group`
            - `Consumer group ID`: `tigerdata-kafka/<tiger_cloud_project_id>`. See [Find your connection details][connection-info] for where to find your project ID
            - `Pattern type`: `PREFIXED`
            - `Operation`: `READ`
            - `Permission`: `ALLOW`
          You need these to configure your Kafka source connector in Tiger Cloud.

## Configure Confluent Cloud Schema Registry

Tiger Cloud requires access to the Schema Registry to fetch schemas for Kafka topics. To configure the Schema Registry:

1. **Navigate to Schema Registry**

      In Confluent Cloud, click `Environments` and select your environment, then click `Stream Governance`.

   1. **Create a Schema Registry API key**

      1. Click `API Keys`, then click `Add API Key`.
      1. Choose `Service Account`, select `tigerdata-access`, then click `Next`.
      1. Under `Resource scope`, choose `Schema Registry`, select the `default` environment, then click `Next`.
      2. In `Create API Key`, add the following, then click `Create API Key` :

         - `Name`: `tigerdata-schema-registry-access`
         - `Description`: `API key for Tiger Cloud schema registry access`

      1. Click `Download API Key` and securely store the API key and secret, then click `Complete`.

   1. **Assign roles for Schema Registry**

      1. Click the burger menu at the top-right of the pane, then press
          `Access control` > `Accounts & access` > `Service accounts`.
      1. Select the `tigerdata-access` service account.
      1. In the `Access` tab, add the following role assignments for `All schema subjects`:

         - `ResourceOwner` on the service account.
         - `DeveloperRead` on schema subjects.

            Choose `All schema subjects` or restrict to specific subjects as required.
      1. Save the role assignments.

Your Confluent Cloud Schema Registry is now accessible to Tiger Cloud using the API key and secret.

## Add Kafka source connector in Tiger Cloud

Take the following steps to create a Kafka source connector in Tiger Cloud Console.

1. **In [Console][console], select your service**
1. **Go to `Connectors` > `Source connectors`. Click `New Connector`, then select `Kafka`**
1. **Click the pencil icon, then set the connector name**
1. **Set up Kafka authentication**

   Enter the name of your cluster in Confluent Cloud and the information from the first `api-key-*.txt` that you
      downloaded, then click `Authenticate`.
1. **Set up the Schema Registry**

   Enter the service account ID and the information from the second `api-key-*.txt` that you
   downloaded, then click `Authenticate`.
1. **Select topics to sync**

    Add the schema and table, map the columns in the table, and click `Create connector`.


Your Kafka connector is configured and ready to stream events.

## Known limitations and unsupported types

The following Avro schema types are not supported:

### Union types

Multi-type non-nullable unions are blocked.

Examples:

- Multiple type union:

    ```
    {
      "type": "record",
      "name": "Message",
      "fields": [
        {"name": "content", "type": ["string", "bytes", "null"]}
      ]
    }
    ```

- Union as root schema:

    ```
    ["null", "string"]
    ```

### Reference types (named type references)

Referencing a previously defined named type by name, instead of inline, is not supported.

Examples:

- Named type definition:

    ```
    {
      "type": "record",
      "name": "Address",
      "fields": [
        {"name": "street", "type": "string"},
        {"name": "city", "type": "string"}
      ]
    }
    ```

- Failing reference:

    ```
    {
      "type": "record",
      "name": "Person",
      "fields": [
        {"name": "name", "type": "string"},
        {"name": "address", "type": "Address"}
      ]
    }
    ```

### Unsupported logical types

Only the logical types in the hardcoded supported list are supported. This includes:

* decimal, date, time-millis, time-micros

* timestamp-millis, timestamp-micros, timestamp-nanos

* local-timestamp-millis, local-timestamp-micros, local-timestamp-nanos

* uuid, duration

Unsupported examples:

```
{
  "type": "int",
  "logicalType": "date-time"
}

{
  "type": "string",
  "logicalType": "json"
}

{
  "type": "bytes",
  "logicalType": "custom-type"
}
```


===== PAGE: https://docs.tigerdata.com/migrate/upload-file-using-console/ =====

# Upload a file into your service using Tiger Cloud Console



You can upload files into your service using Tiger Cloud Console. This page explains how to upload CSV, Parquet, and text files, from your local machine and from an S3 bucket.





Tiger Cloud Console enables you to drag and drop files to upload from your local machine.

Early access

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with real-time analytics enabled.





To upload a CSV file to your service:

1. **Select your service in [Console][console], then click `Actions` > `Import data` > `Upload your files` > `Upload CSV file`**

   ![Import from CSV into Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-import-csv-file.png)

1. **Click to browse, or drag the file to import**
1. **Configure the import**

   ![Configure the CSV import in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-configure-csv-file-import.png)

   - Set a delimiter.
   - Toggle to skip or keep the header.
   - Select to ingest the data into an existing table or create a new one.
   - Provide the new or existing table name.
   -  For a new table with a time column, toggle the time column to create a hypertable instead of a regular table.

1. **Click `Process CSV file`**

   When the processing is completed, to find the data your imported, click `Explorer`.





To upload a Parquet file to your service:

1. **Select your service in [Console][console], then click `Actions` > `Import data` > `Upload your files` > `Upload Parquet file`**

   ![Import from Parquet into Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-import-parquet-file.png)

1. **Click to browse, or drag the file to import**
1. **Configure the import**

   ![Configure the Parquet import in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-configure-parquet-file-import.png)

   - Select to ingest the data into an existing table or create a new one.
   - Provide the new or existing table name.
   -  For a new table with a time column, toggle the time column to create a hypertable instead of a regular table.

1. **Click `Process Parquet file`**

   When the processing is completed, to find the data your imported, click `Explorer`.





To upload a TXT or MD file to your service:

1. **Select your service in Console, then click `Actions` > `Import data` > `Upload your files` > `Upload Text file`**

   ![Import from a text file into Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-import-txt-file.png)

1. **Click to browse, or drag and drop the file to import**
1. **Configure the import**

   Provide a name to create a new table, or select an existing table to add data to.

   ![Configure the text file import in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-console-configure-txt-file-import.png)

1. **Click `Upload files`**

   When the upload is finished, find your data imported to a new or existing table in `Explorer`.









Tiger Cloud Console enables you to upload CSV and Parquet files, including archives compressed using GZIP and ZIP, by connecting to an S3 bucket.

This feature is not available under the Free pricing plan.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Cloud service][create-service] with real-time analytics enabled.

- Ensure access to a standard Amazon S3 bucket containing your data files.
- Configure access credentials for the S3 bucket. The following credentials are supported:
   - [IAM Role][credentials-iam].
   - [Public anonymous user][credentials-public].





To import a CSV file from an S3 bucket:

1. **Select your service in Console, then click `Actions` > `Import data` > `Explore import options` > `Import from S3`**

1. **Select your file in the S3 bucket**

   ![Import CSV from S3 in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/import-csv-file-from-s3.png)

   1. Provide your file path.
   1. Select `CSV` in the file type dropdown.
   1. Select the authentication method:
      - `IAM role` and provide the role.
      - `Public`.
   1. Click `Continue`.

1. **Configure the import**

   ![Configure CSV import from S3 in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/configure-csv-file-import-from-s3.png)

   - Set a delimiter.
   - Toggle to skip or keep the header.
   - Select to ingest the data into an existing table or create a new one.
   - Provide the new or existing table name.
   -  For a new table with a time column, toggle the time column to create a hypertable instead of a regular table.

1. **Click `Process CSV file`**

   When the processing is completed, to find the data your imported, click `Explorer`.





To import a Parquet file from an S3 bucket:

1. **Select your service in Console, then click `Actions` > `Import from S3`**

1. **Select your file in the S3 bucket**

   ![Import Parquet from S3 in Tiger](https://assets.timescale.com/docs/images/tiger-cloud-console/tiger-cloud-import-parquet-file-from-s3.png)

   1. Provide your file path.
   1. Select `Parquet` in the file type dropdown.
   1. Select the authentication method:
     - `IAM role` and provide the role.
     - `Public`.
   1. Click `Continue`.

1. **Configure the import**

   - Select `Create a new table for your data` or `Ingest data to an existing table`.
   - Provide the new or existing table name.
   -  For a new table with a time column, toggle the time column to create a hypertable instead of a regular table.

1. **Click `Process Parquet file`**

   When the processing is completed, to find the data your imported, click `Explorer`.










And that is it, you have imported your data to your Tiger Cloud service.


===== PAGE: https://docs.tigerdata.com/migrate/upload-file-using-terminal/ =====

# Upload a file into your service using the terminal



This page shows you how to upload CSV, MySQL, and Parquet files from a source machine into your service using the terminal.





The CSV file format is widely used for data migration. This page shows you how to import data into your Tiger Cloud service from a CSV file using the terminal.

## Prerequisites

To follow the procedure on this page you need to:

* Create a [target Tiger Cloud service][create-service].

   This procedure also works for [self-hosted TimescaleDB][enable-timescaledb].

- Install [Go](https://go.dev/doc/install) v1.13 or later

- Install [timescaledb-parallel-copy][install-parallel-copy]

  [timescaledb-parallel-copy][parallel importer] improves performance for large datasets by parallelizing the import
  process. It also preserves row order and uses a round-robin approach to optimize memory management and disk operations.

  To verify your installation, run `timescaledb-parallel-copy --version`.

- Ensure that the time column in the CSV file uses the `TIMESTAMPZ` data type.

For faster data transfer, best practice is that your target service and the system
running the data import are in the same region.

## Import data into your service

To import data from a CSV file:

1. **Set up your service connection string**

    This variable holds the connection information for the target Tiger Cloud service.

In the terminal on the source machine, set the following:

```bash
export TARGET=postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require
```
See where to [find your connection details][connection-info].

1. **Create a [hypertable][hypertable-docs] to hold your data**

   Create a hypertable with a schema that is compatible with the data in your parquet file. For example, if your parquet file contains the columns `ts`, `location`, and `temperature` with types`TIMESTAMP`, `STRING`, and `DOUBLE`:

   - TimescaleDB v2.20 and above:

     ```sql
     psql target -c "CREATE TABLE  ( \
     ts          TIMESTAMPTZ         NOT NULL, \
     location    TEXT                NOT NULL, \
     temperature DOUBLE PRECISION    NULL \
     ) WITH (timescaledb.hypertable, timescaledb.partition_column = 'ts');"

   - TimescaleDB v2.19.3 and below:

     1.  Create a new regular table:

         ```sql
         psql target -c  "CREATE TABLE  ( \
            ts          TIMESTAMPTZ         NOT NULL,  \
            location    TEXT                NOT NULL,  \
            temperature DOUBLE PRECISION    NULL  \
         );"
         ```

     1.  Convert the empty table to a hypertable:

         In the following command, replace `` with the name of the table you just created, and `<COLUMN_NAME>` with the partitioning column in ``.
         ```sql
         psql target -c  "SELECT create_hypertable('', by_range('<COLUMN_NAME>'))"
         ```

1. **Import your data**

   In the folder containing your CSV files, either:

    - Use [timescaledb-parallel-copy][install-parallel-copy]:

      ```bash
        timescaledb-parallel-copy \
        --connection target \
        --table  \
        --file <FILE_NAME>.csv \
        --workers <NUM_WORKERS> \
        --reporting-period 30s
      ```

      For the best performances while avoiding resource competition, set `<NUM_WORKERS>` to twice the
      number of CPUs in your service, but less than the available CPU cores.

      For self-hosted TimescaleDB, set `target` to `host=localhost user=postgres sslmode=disable`

    - Use `psql`:

       ```bash
       psql target
       \c <DATABASE_NAME>
       \COPY  FROM <FILENAME>.csv CSV"
       ```

      `psql` COPY is single-threaded, and may be slower for large datasets.

1. **Verify the data was imported correctly into your service**

And that is it, you have imported your data from a CSV file.





MySQL is an open-source relational database management system (RDBMS). This page shows you how to import data into your Tiger Cloud service from a database running on MySQL version 8 or earlier.

## Prerequisites

To follow the procedure on this page you need to:

* Create a [target Tiger Cloud service][create-service].

   This procedure also works for [self-hosted TimescaleDB][enable-timescaledb].

- [Install Docker][install-docker] on your migration machine.

  This machine needs sufficient space to store the buffered changes that occur while your data is
  being copied. This space is proportional to the amount of new uncompressed data being written to
  the Tiger Cloud service during migration. A general rule of thumb is between 100GB and 500GB.

For faster data transfer, best practice is for your source database, target service, and
the system running the data import are in the same region .

## Import data into your service

To import data from a MySQL database:

1. **Set up the connection string for your target service**

    This variable holds the connection information for the target Tiger Cloud service.

In the terminal on the source machine, set the following:

```bash
export TARGET=postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require
```
See where to [find your connection details][connection-info].

1. **Set up the connection string for your source database**

   ```bash
   SOURCE="mysql://<mysql_username>:<mysql_password>@<mysql_host>:<mysql_port>/<mysql_database>?sslmode=require"
   ```
   where:

    - `<mysql_username>`: your MySQL username
    - `<mysql_password>`: your MySQL password
    - `<mysql_host>`: the MySQL server hostname or IP address
    - `<mysql_port>`: the MySQL server port, the default is 3306
    - `<mysql_database>`: the name of your MySQL database

1. **Import your data**

   On your data import machine, run the following command:

    ```docker
    docker run -it ghcr.io/dimitri/pgloader:latest pgloader
    --no-ssl-cert-verification \
    "source" \
    "target"
    ```

1. **Verify the data was imported correctly into your service**

And that is it, you have imported your data from MySQL.





[Apache Parquet][apache-parquet] is a free and open-source column-oriented data storage format in the
Apache Hadoop ecosystem. It provides efficient data compression and encoding schemes with
enhanced performance to handle complex data in bulk. This page shows you how to import data into your Tiger Cloud service from a Parquet file.

## Prerequisites

To follow the procedure on this page you need to:

* Create a [target Tiger Cloud service][create-service].

   This procedure also works for [self-hosted TimescaleDB][enable-timescaledb].

- [Install DuckDB][install-duckdb] on the source machine where the Parquet file is located.
- Ensure that the time column in the Parquet file uses the `TIMESTAMP` data type.

For faster data transfer, best practice is that your target service and the system
running the data import are in the same region.

## Import data into your service

To import data from a Parquet file:

1. **Set up your service connection string**

    This variable holds the connection information for the target Tiger Cloud service.

In the terminal on the source machine, set the following:

```bash
export TARGET=postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require
```
See where to [find your connection details][connection-info].

1. **Create a [hypertable][hypertable-docs] to hold your data**

   Create a hypertable with a schema that is compatible with the data in your parquet file. For example, if your parquet file contains the columns `ts`, `location`, and `temperature` with types`TIMESTAMP`, `STRING`, and `DOUBLE`:

    - TimescaleDB v2.20 and above:

      ```sql
      psql target -c "CREATE TABLE  ( \
      ts          TIMESTAMPTZ         NOT NULL, \
      location    TEXT                NOT NULL, \
      temperature DOUBLE PRECISION    NULL \
      ) WITH (timescaledb.hypertable, timescaledb.partition_column = 'ts');"

    - TimescaleDB v2.19.3 and below:

        1.  Create a new regular table:

            ```sql
            psql target -c  "CREATE TABLE  ( \
               ts          TIMESTAMPTZ         NOT NULL,  \
               location    TEXT                NOT NULL,  \
               temperature DOUBLE PRECISION    NULL  \
            );"
            ```

        1.  Convert the empty table to a hypertable:

            In the following command, replace `` with the name of the table you just created, and `<COLUMN_NAME>` with the partitioning column in ``.
            ```sql
            psql target -c  "SELECT create_hypertable('', by_range('<COLUMN_NAME>'))"
            ```

1. **Set up a DuckDB connection to your service**

    1.  In a terminal on the source machine with your Parquet files, start a new DuckDB interactive session:

        ```bash
        duckdb
        ```
    1. Connect to your service in your DuckDB session:

       ```bash
       ATTACH '<Paste the value of target here' AS db (type postgres);
       ```
       `target` is the connection string you used to connect to your service using psql.

1. **Import data from Parquet to your service**

    1. In DuckDB, upload the table data to your service
       ```bash
       COPY db. FROM '<FILENAME>.parquet' (FORMAT parquet);
       ```
       Where:

        - ``: the hypertable you created to import data to
        - `<FILENAME>`: the Parquet file to import data from

    1. Exit the DuckDB session:

        ```bash
        EXIT;
        ```

1. **Verify the data was imported correctly into your service**

   In your `psql` session, view the data in ``:
   ```sql
   SELECT * FROM ;
   ```

And that is it, you have imported your data from a Parquet file to your Tiger Cloud service.


===== PAGE: https://docs.tigerdata.com/migrate/pg-dump-and-restore/ =====

# Migrate with downtime




You use downtime migration to move less than 100GB of data from a self-hosted database to a Tiger Cloud service.

Downtime migration uses the native Postgres [`pg_dump`][pg_dump] and [`pg_restore`][pg_restore] commands.
If you are migrating from self-hosted TimescaleDB, this method works for hypertables compressed into the columnstore without having
to convert the data back to the rowstore before you begin.

If you want to migrate more than 400GB of data, create a [Tiger Cloud Console support request](https://console.cloud.timescale.com/dashboard/support), or
send us an email at [support@tigerdata.com](mailto:support@tigerdata.com) saying how much data you want to migrate. We pre-provision
your Tiger Cloud service for you.

However, downtime migration for large amounts of data takes a large amount of time. For more than 100GB of data, best
practice is to follow [live migration].

This page shows you how to move your data from a self-hosted database to a Tiger Cloud service using
shell commands.

## Prerequisites

Best practice is to use an [Ubuntu EC2 instance][create-ec2-instance] hosted in the same region as your
Tiger Cloud service to move data. That is, the machine you run the commands on to move your
data from your source database to your target Tiger Cloud service.

Before you move your data:

- Create a target [Tiger Cloud service][created-a-database-service-in-timescale].

  Each Tiger Cloud service has a single Postgres instance that supports the
  [most popular extensions][all-available-extensions]. Tiger Cloud services do not support tablespaces,
  and there is no superuser associated with a service.
  Best practice is to create a Tiger Cloud service with at least 8 CPUs for a smoother experience. A higher-spec instance
  can significantly reduce the overall migration window.

- To ensure that maintenance does not run while migration is in progress, best practice is to [adjust the maintenance window][adjust-maintenance-window].


- Install the Postgres client tools on your migration machine.

  This includes `psql`, `pg_dump`, and `pg_dumpall`.

- Install the GNU implementation of `sed`.

  Run `sed --version` on your migration machine. GNU sed identifies itself
  as GNU software, BSD sed returns `sed: illegal option -- -`.


### Migrate to Tiger Cloud

To move your data from a self-hosted database to a Tiger Cloud service:





This section shows you how to move your data from self-hosted TimescaleDB to a Tiger Cloud service
using `pg_dump` and `psql` from Terminal.

## Prepare to migrate
1. **Take the applications that connect to the source database offline**

   The duration of the migration is proportional to the amount of data stored in your database. By
   disconnection your app from your database you avoid and possible data loss.

1. **Set your connection strings**

   These variables hold the connection information for the source database and target Tiger Cloud service:

   ```bash
   export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
   export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
   ```
   You find the connection information for your Tiger Cloud service in the configuration file you
   downloaded when you created the service.

## Align the version of TimescaleDB on the source and target
1. Ensure that the source and target databases are running the same version of TimescaleDB.

    1. Check the version of TimescaleDB running on your Tiger Cloud service:

       ```bash
       psql target -c "SELECT extversion FROM pg_extension WHERE extname = 'timescaledb';"
       ```

    1. Update the TimescaleDB extension in your source database to match the target service:

       If the TimescaleDB extension is the same version on the source database and target service,
       you do not need to do this.

       ```bash
       psql source -c "ALTER EXTENSION timescaledb UPDATE TO '<version here>';"
       ```

       For more information and guidance, see [Upgrade TimescaleDB](https://docs.tigerdata.com/self-hosted/latest/upgrades/).

1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Migrate the roles from TimescaleDB to your Tiger Cloud service

Roles manage database access permissions. To migrate your role-based security hierarchy to your Tiger Cloud service:
1. **Dump the roles from your source database**

   Export your role-based security hierarchy. `<db_name>` has the same value as `<db_name>` in `source`.
   I know, it confuses me as well.

   ```bash
   pg_dumpall -d "source" \
     -l <db_name>
     --quote-all-identifiers \
     --roles-only \
     --file=roles.sql
   ```

   If you only use the default `postgres` role, this step is not necessary.

1. **Remove roles with superuser access**

   Tiger Cloud service do not support roles with superuser access. Run the following script
   to remove statements, permissions and clauses that require superuser permissions from `roles.sql`:

   ```bash
   sed -i -E \
   -e '/CREATE ROLE "postgres";/d' \
   -e '/ALTER ROLE "postgres"/d' \
   -e '/CREATE ROLE "tsdbadmin";/d' \
   -e '/ALTER ROLE "tsdbadmin"/d' \
   -e 's/(NO)*SUPERUSER//g' \
   -e 's/(NO)*REPLICATION//g' \
   -e 's/(NO)*BYPASSRLS//g' \
   -e 's/GRANTED BY "[^"]*"//g' \
   roles.sql
   ```

1. **Dump the source database schema and data**

   The `pg_dump` flags remove superuser access and tablespaces from your data. When you run
   `pgdump`, check the run time, [a long-running `pg_dump` can cause issues][long-running-pgdump].

   ```bash
   pg_dump -d "source" \
   --format=plain \
   --quote-all-identifiers \
   --no-tablespaces \
   --no-owner \
   --no-privileges \
   --file=dump.sql
   ```
   To dramatically reduce the time taken to dump the source database, using multiple connections. For more information,
   see [dumping with concurrency][dumping-with-concurrency] and [restoring with concurrency][restoring-with-concurrency].

## Upload your data to the target Tiger Cloud service

This command uses the [timescaledb_pre_restore] and [timescaledb_post_restore] functions to put your database in the
correct state.

 ```bash
 psql target -v ON_ERROR_STOP=1 --echo-errors \
 -f roles.sql \
 -c "SELECT timescaledb_pre_restore();" \
 -f dump.sql \
 -c "SELECT timescaledb_post_restore();"
 ```

## Validate your Tiger Cloud service and restart your app
1. Update the table statistics.

    ```bash
    psql target -c "ANALYZE;"
    ```

1. Verify the data in the target Tiger Cloud service.

   Check that your data is correct, and returns the results that you expect,

1. Enable any Tiger Cloud features you want to use.

   Migration from Postgres moves the data only. Now manually enable Tiger Cloud features like
   [hypertables][about-hypertables], [hypercore][data-compression] or [data retention][data-retention]
   while your database is offline.

1. Reconfigure your app to use the target database, then restart it.

And that is it, you have migrated your data from a self-hosted instance running TimescaleDB to a Tiger Cloud service.




This section shows you how to move your data from self-hosted Postgres to a Tiger Cloud service
using `pg_dump` and `psql` from Terminal.

Migration from Postgres moves the data only. You must manually enable Tiger Cloud features like
[hypertables][about-hypertables], [hypercore][data-compression] or [data retention][data-retention] after the migration is complete. You enable Tiger Cloud features while your database is offline.


## Prepare to migrate
1. **Take the applications that connect to the source database offline**

   The duration of the migration is proportional to the amount of data stored in your database. By
   disconnection your app from your database you avoid and possible data loss.

1. **Set your connection strings**

   These variables hold the connection information for the source database and target Tiger Cloud service:

   ```bash
   export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
   export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
   ```
   You find the connection information for your Tiger Cloud service in the configuration file you
   downloaded when you created the service.

## Align the extensions on the source and target

1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Migrate the roles from TimescaleDB to your Tiger Cloud service

Roles manage database access permissions. To migrate your role-based security hierarchy to your Tiger Cloud service:

1. **Dump the roles from your source database**

   Export your role-based security hierarchy. `<db_name>` has the same value as `<db_name>` in `source`.
   I know, it confuses me as well.

   ```bash
   pg_dumpall -d "source" \
     -l <db_name>
     --quote-all-identifiers \
     --roles-only \
     --file=roles.sql
   ```

   If you only use the default `postgres` role, this step is not necessary.

1. **Remove roles with superuser access**

   Tiger Cloud service do not support roles with superuser access. Run the following script
   to remove statements, permissions and clauses that require superuser permissions from `roles.sql`:

   ```bash
   sed -i -E \
   -e '/CREATE ROLE "postgres";/d' \
   -e '/ALTER ROLE "postgres"/d' \
   -e '/CREATE ROLE "tsdbadmin";/d' \
   -e '/ALTER ROLE "tsdbadmin"/d' \
   -e 's/(NO)*SUPERUSER//g' \
   -e 's/(NO)*REPLICATION//g' \
   -e 's/(NO)*BYPASSRLS//g' \
   -e 's/GRANTED BY "[^"]*"//g' \
   roles.sql
   ```

1. **Dump the source database schema and data**

   The `pg_dump` flags remove superuser access and tablespaces from your data. When you run
   `pgdump`, check the run time, [a long-running `pg_dump` can cause issues][long-running-pgdump].

   ```bash
   pg_dump -d "source" \
   --format=plain \
   --quote-all-identifiers \
   --no-tablespaces \
   --no-owner \
   --no-privileges \
   --file=dump.sql
   ```
   To dramatically reduce the time taken to dump the source database, using multiple connections. For more information,
   see [dumping with concurrency][dumping-with-concurrency] and [restoring with concurrency][restoring-with-concurrency].

## Upload your data to the target Tiger Cloud service

```bash
psql target -v ON_ERROR_STOP=1 --echo-errors \
-f roles.sql \
-f dump.sql
```

## Validate your Tiger Cloud service and restart your app
1. Update the table statistics.

    ```bash
    psql target -c "ANALYZE;"
    ```

1. Verify the data in the target Tiger Cloud service.

   Check that your data is correct, and returns the results that you expect,

1. Enable any Tiger Cloud features you want to use.

   Migration from Postgres moves the data only. Now manually enable Tiger Cloud features like
   [hypertables][about-hypertables], [hypercore][data-compression] or [data retention][data-retention]
   while your database is offline.

1. Reconfigure your app to use the target database, then restart it.


And that is it, you have migrated your data from a self-hosted instance running Postgres to a Tiger Cloud service.





To migrate your data from an Amazon RDS/Aurora Postgres instance to a Tiger Cloud service, you extract the data to an intermediary
EC2 Ubuntu instance in the same AWS region as your RDS/Aurora Postgres instance. You then upload your data to a Tiger Cloud service.
To make this process as painless as possible, ensure that the intermediary machine has enough CPU and disk space to
rapidLy extract and store your data before uploading to Tiger Cloud.

Migration from RDS/Aurora Postgres moves the data only. You must manually enable Tiger Cloud features like
[hypertables][about-hypertables], [data compression][data-compression] or [data retention][data-retention] after the migration is complete. You enable Tiger Cloud
features while your database is offline.

This section shows you how to move your data from a Postgres database running in an Amazon RDS/Aurora Postgres instance to a
Tiger Cloud service using `pg_dump` and `psql` from Terminal.


## Create an intermediary EC2 Ubuntu instance
1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
   select the RDS/Aurora Postgres instance to migrate.
1. Click `Actions` > `Set up EC2 connection`.
   Press `Create EC2 instance` and use the following settings:
    - **AMI**: Ubuntu Server.
    - **Key pair**: use an existing pair or create a new one that you will use to access the intermediary machine.
    - **VPC**: by default, this is the same as the database instance.
    - **Configure Storage**: adjust the volume to at least the size of RDS/Aurora Postgres instance you are migrating from.
    You can reduce the space used by your data on Tiger Cloud using [Hypercore][hypercore].
1. Click `Lauch instance`. AWS creates your EC2 instance, then click `Connect to instance` > `SSH client`.
   Follow the instructions to create the connection to your intermediary EC2 instance.

## Install the psql client tools on the intermediary instance

1. Connect to your intermediary EC2 instance. For example:
   ```sh
   ssh -i "<key-pair>.pem" ubuntu@<EC2 instance's Public IPv4>
   ```
1. On your intermediary EC2 instance, install the Postgres client.
   ```sh
   sudo sh -c 'echo "deb http://apt.postgresql.org/pub/repos/apt $(lsb_release -cs)-pgdg main" > /etc/apt/sources.list.d/pgdg.list'
   wget -qO- https://www.postgresql.org/media/keys/ACCC4CF8.asc | sudo tee /etc/apt/trusted.gpg.d/pgdg.asc &>/dev/null
   sudo apt update
   sudo apt install postgresql-client-16 -y # "postgresql-client-16" if your source DB is using PG 16.
   psql --version && pg_dump --version
   ```

  Keep this terminal open, you need it to connect to the RDS/Aurora Postgres instance for migration.

## Set up secure connectivity between your RDS/Aurora Postgres and EC2 instances

1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
    select the RDS/Aurora Postgres instance to migrate.
1. Scroll down to `Security group rules (1)` and select the `EC2 Security Group - Inbound` group. The
   `Security Groups (1)` window opens. Click the `Security group ID`, then click `Edit inbound rules`

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/rds-add-security-rule-to-ec2-instance.svg"
   alt="Create security group rule to enable RDS/Aurora Postgres EC2 connection"/>

1. On your intermediary EC2 instance, get your local IP address:
   ```sh
   ec2metadata --local-ipv4
   ```
   Bear with me on this one, you need this IP address to enable access to your RDS/Aurora Postgres instance.
1. In `Edit inbound rules`, click `Add rule`, then create a `PostgreSQL`, `TCP` rule granting access
   to the local IP address for your EC2 instance (told you :-)). Then click `Save rules`.

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/rds-add-inbound-rule-for-ec2-instance.png"
   alt="Create security rule to enable RDS/Aurora Postgres EC2 connection"/>

## Test the connection between your RDS/Aurora Postgres and EC2 instances

1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
    select the RDS/Aurora Postgres instance to migrate.
1. On your intermediary EC2 instance, use the values of `Endpoint`, `Port`, `Master username`, and `DB name`
   to create the postgres connectivity string to the `SOURCE` variable.

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/migrate-source-rds-instance.svg"
   alt="Record endpoint, port, VPC details"/>

   ```sh
   export SOURCE="postgres://<Master username>:<Master password>@<Endpoint>:<Port>/<DB name>"
   ```
   The value of `Master password` was supplied when this RDS/Aurora Postgres instance was created.

1. Test your connection:
   ```sh
   psql -d source
   ```
   You are connected to your RDS/Aurora Postgres instance from your intermediary EC2 instance.

## Migrate your data to your Tiger Cloud service

To securely migrate data from your RDS instance:
## Prepare to migrate
1. **Take the applications that connect to the RDS instance offline**

   The duration of the migration is proportional to the amount of data stored in your database.
   By disconnection your app from your database you avoid and possible data loss. You should also ensure that your
   source RDS instance is not receiving any DML queries.

1. **Connect to your intermediary EC2 instance**

   For example:
   ```sh
   ssh -i "<key-pair>.pem" ubuntu@<EC2 instance's Public IPv4>
   ```

1. **Set your connection strings**

   These variables hold the connection information for the RDS instance and target Tiger Cloud service:

   ```bash
   export SOURCE="postgres://<Master username>:<Master password>@<Endpoint>:<Port>/<DB name>"
   export TARGET=postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require
   ```
   You find the connection information for `SOURCE` in your RDS configuration. For `TARGET` in the configuration file you
   downloaded when you created the Tiger Cloud service.

## Align the extensions on the source and target
1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Migrate roles from RDS to your Tiger Cloud service

Roles manage database access permissions. To migrate your role-based security hierarchy to your Tiger Cloud service:

1. **Dump the roles from your RDS instance**

   Export your role-based security hierarchy. If you only use the default `postgres` role, this
   step is not necessary.

   ```bash
   pg_dumpall -d "source" \
     --quote-all-identifiers \
     --roles-only \
     --no-role-passwords \
     --file=roles.sql
   ```

   AWS RDS does not allow you to export passwords with roles. You assign passwords to these roles
   when you have uploaded them to your Tiger Cloud service.

1. **Remove roles with superuser access**

   Tiger Cloud services do not support roles with superuser access. Run the following script
   to remove statements, permissions and clauses that require superuser permissions from `roles.sql`:

   ```bash
   sed -i -E \
   -e '/CREATE ROLE "postgres";/d' \
   -e '/ALTER ROLE "postgres"/d' \
   -e '/CREATE ROLE "rds/d' \
   -e '/ALTER ROLE "rds/d' \
   -e '/TO "rds/d' \
   -e '/GRANT "rds/d' \
   -e 's/(NO)*SUPERUSER//g' \
   -e 's/(NO)*REPLICATION//g' \
   -e 's/(NO)*BYPASSRLS//g' \
   -e 's/GRANTED BY "[^"]*"//g' \
   roles.sql
   ```
1. **Upload the roles to your Tiger Cloud service**

   ```bash
   psql -X -d "target" \
     -v ON_ERROR_STOP=1 \
     --echo-errors \
     -f roles.sql
   ```

1. **Manually assign passwords to the roles**

   AWS RDS did not allow you to export passwords with roles. For each role, use the following command to manually
   assign a password to a role:

   ```bash
    psql target -c "ALTER ROLE <role name> WITH PASSWORD '<highly secure password>';"
    ```

## Migrate data from your RDS instance to your Tiger Cloud service

1. **Dump the data from your RDS instance to your intermediary EC2 instance**

   The `pg_dump` flags remove superuser access and tablespaces from your data. When you run
   `pgdump`, check the run time, [a long-running `pg_dump` can cause issues][long-running-pgdump].

   ```bash
   pg_dump -d "source" \
   --format=plain \
   --quote-all-identifiers \
   --no-tablespaces \
   --no-owner \
   --no-privileges \
   --file=dump.sql
   ```
   To dramatically reduce the time taken to dump the RDS instance, using multiple connections. For more information,
   see [dumping with concurrency][dumping-with-concurrency] and [restoring with concurrency][restoring-with-concurrency].

1. **Upload your data to your Tiger Cloud service**

   ```bash
   psql -d target -v ON_ERROR_STOP=1 --echo-errors \
     -f dump.sql
   ```

## Validate your Tiger Cloud service and restart your app

1. Update the table statistics.

    ```bash
    psql target -c "ANALYZE;"
    ```

1. Verify the data in the target Tiger Cloud service.

   Check that your data is correct, and returns the results that you expect,

1. Enable any Tiger Cloud features you want to use.

   Migration from Postgres moves the data only. Now manually enable Tiger Cloud features like
   [hypertables][about-hypertables], [hypercore][data-compression] or [data retention][data-retention]
   while your database is offline.

1. Reconfigure your app to use the target database, then restart it.

And that is it, you have migrated your data from an RDS/Aurora Postgres instance to a Tiger Cloud service.






This section shows you how to move your data from a Managed Service for TimescaleDB instance to a
Tiger Cloud service using `pg_dump` and `psql` from Terminal.

## Prepare to migrate
1. **Take the applications that connect to the source database offline**

   The duration of the migration is proportional to the amount of data stored in your database. By
   disconnection your app from your database you avoid and possible data loss.

1. **Set your connection strings**

   These variables hold the connection information for the source database and target Tiger Cloud service:

   ```bash
   export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
   export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
   ```
   You find the connection information for your Tiger Cloud service in the configuration file you
   downloaded when you created the service.

## Align the version of TimescaleDB on the source and target
1. Ensure that the source and target databases are running the same version of TimescaleDB.

    1. Check the version of TimescaleDB running on your Tiger Cloud service:

       ```bash
       psql target -c "SELECT extversion FROM pg_extension WHERE extname = 'timescaledb';"
       ```

    1. Update the TimescaleDB extension in your source database to match the target service:

       If the TimescaleDB extension is the same version on the source database and target service,
       you do not need to do this.

       ```bash
       psql source -c "ALTER EXTENSION timescaledb UPDATE TO '<version here>';"
       ```

       For more information and guidance, see [Upgrade TimescaleDB](https://docs.tigerdata.com/self-hosted/latest/upgrades/).

1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Migrate the roles from TimescaleDB to your Tiger Cloud service

Roles manage database access permissions. To migrate your role-based security hierarchy to your Tiger Cloud service:
1. **Dump the roles from your source database**

   Export your role-based security hierarchy. `<db_name>` has the same value as `<db_name>` in `source`.
   I know, it confuses me as well.

   ```bash
   pg_dumpall -d "source" \
     -l <db_name>  \
     --quote-all-identifiers \
     --roles-only \
     --no-role-passwords \
     --file=roles.sql
   ```

   MST does not allow you to export passwords with roles. You assign passwords to these roles
   when you have uploaded them to your Tiger Cloud service.

1. **Remove roles with superuser access**

   Tiger Cloud services do not support roles with superuser access. Run the following script
   to remove statements, permissions and clauses that require superuser permissions from `roles.sql`:

   ```bash
   sed -i -E \
  -e '/DROP ROLE IF EXISTS "postgres";/d' \
  -e '/DROP ROLE IF EXISTS "tsdbadmin";/d' \
  -e '/CREATE ROLE "postgres";/d' \
  -e '/ALTER ROLE "postgres"/d' \
  -e '/CREATE ROLE "rds/d' \
  -e '/ALTER ROLE "rds/d' \
  -e '/TO "rds/d' \
  -e '/GRANT "rds/d' \
  -e '/GRANT "pg_read_all_stats" TO "tsdbadmin"/d' \
  -e 's/(NO)*SUPERUSER//g' \
  -e 's/(NO)*REPLICATION//g' \
  -e 's/(NO)*BYPASSRLS//g' \
  -e 's/GRANTED BY "[^"]*"//g' \
  -e '/CREATE ROLE "tsdbadmin";/d' \
  -e '/ALTER ROLE "tsdbadmin"/d' \
  -e 's/WITH ADMIN OPTION,/WITH /g' \
  -e 's/WITH ADMIN OPTION//g' \
  -e 's/GRANTED BY ".*"//g' \
  -e '/GRANT "pg_.*" TO/d' \
  -e '/CREATE ROLE "_aiven";/d' \
  -e '/ALTER ROLE "_aiven"/d' \
  -e '/GRANT SET ON PARAMETER "pgaudit\.[^"]+" TO "_tsdbadmin_auditing"/d' \
  -e '/GRANT SET ON PARAMETER "anon\.[^"]+" TO "tsdbadmin_group"/d' \
   roles.sql
   ```

1. **Dump the source database schema and data**

   The `pg_dump` flags remove superuser access and tablespaces from your data. When you run
   `pgdump`, check the run time, [a long-running `pg_dump` can cause issues][long-running-pgdump].

   ```bash
   pg_dump -d "source" \
   --format=plain \
   --quote-all-identifiers \
   --no-tablespaces \
   --no-owner \
   --no-privileges \
   --file=dump.sql
   ```

   To dramatically reduce the time taken to dump the source database, using multiple connections. For more information,
   see [dumping with concurrency][dumping-with-concurrency] and [restoring with concurrency][restoring-with-concurrency].

## Upload your data to the target Tiger Cloud service

This command uses the [timescaledb_pre_restore] and [timescaledb_post_restore] functions to put your database in the
correct state.

1. **Upload your data**
   ```bash
   psql target -v ON_ERROR_STOP=1 --echo-errors \
   -f roles.sql \
   -c "SELECT timescaledb_pre_restore();" \
   -f dump.sql \
   -c "SELECT timescaledb_post_restore();"
   ```
1. **Manually assign passwords to the roles**

   MST did not allow you to export passwords with roles. For each role, use the following command to manually
   assign a password to a role:

   ```bash
    psql target -c "ALTER ROLE <role name> WITH PASSWORD '<highly secure password>';"
    ```

## Validate your Tiger Cloud service and restart your app

1. Update the table statistics.

    ```bash
    psql target -c "ANALYZE;"
    ```

1. Verify the data in the target Tiger Cloud service.

   Check that your data is correct, and returns the results that you expect,

1. Enable any Tiger Cloud features you want to use.

   Migration from Postgres moves the data only. Now manually enable Tiger Cloud features like
   [hypertables][about-hypertables], [hypercore][data-compression] or [data retention][data-retention]
   while your database is offline.

1. Reconfigure your app to use the target database, then restart it.

And that is it, you have migrated your data from a Managed Service for TimescaleDB instance to a Tiger Cloud service.


===== PAGE: https://docs.tigerdata.com/migrate/live-migration/ =====

# Live migration



Live migration is an end-to-end solution that copies the database schema and data to
your target Tiger Cloud service, then replicates the database activity in your source database to the target service in real time. Live migration uses the Postgres logical decoding functionality and leverages [pgcopydb].

You use the live migration Docker image to move 100GB-10TB+ of data to a Tiger Cloud service seamlessly with only a few minutes downtime.

If you want to migrate more than 400GB of data, create a [Tiger Cloud Console support request](https://console.cloud.timescale.com/dashboard/support), or
send us an email at [support@tigerdata.com](mailto:support@tigerdata.com) saying how much data you want to migrate. We pre-provision
your Tiger Cloud service for you.

Best practice is to use live migration when:
- Modifying your application logic to perform dual writes is a significant effort.
- The insert workload does not exceed 20,000 rows per second, and inserts are batched.

  Use [Dual write and backfill][dual-write-and-backfill] for greater workloads.
- Your source database:
  - Uses `UPDATE` and `DELETE` statements on uncompressed time-series data.

    Live-migration does not support replicating `INSERT`/`UPDATE`/`DELETE` statements on compressed data.
  - Has large, busy tables with primary keys.
  - Does not have many `UPDATE` or `DELETE` statements.

This page shows you how to move your data from a self-hosted database to a Tiger Cloud service using
the live-migration Docker image.

## Prerequisites

Best practice is to use an [Ubuntu EC2 instance][create-ec2-instance] hosted in the same region as your
Tiger Cloud service to move data. That is, the machine you run the commands on to move your
data from your source database to your target Tiger Cloud service.

Before you move your data:

- Create a target [Tiger Cloud service][created-a-database-service-in-timescale].

  Each Tiger Cloud service has a single Postgres instance that supports the
  [most popular extensions][all-available-extensions]. Tiger Cloud services do not support tablespaces,
  and there is no superuser associated with a service.
  Best practice is to create a Tiger Cloud service with at least 8 CPUs for a smoother experience. A higher-spec instance
  can significantly reduce the overall migration window.

- To ensure that maintenance does not run while migration is in progress, best practice is to [adjust the maintenance window][adjust-maintenance-window].

- [Install Docker][install-docker] on your migration machine.

  This machine needs sufficient space to store the buffered changes that occur while your data is
  being copied. This space is proportional to the amount of new uncompressed data being written to
  the Tiger Cloud service during migration. A general rule of thumb is between 100GB and 500GB.
  The CPU specifications of this EC2 instance should match those of your Tiger Cloud service for optimal performance. For example, if your service has an 8-CPU configuration, then your EC2 instance should also have 8 CPUs.

- Before starting live migration, read the [Frequently Asked Questions][FAQ].

### Migrate to Tiger Cloud

To move your data from a self-hosted database to a Tiger Cloud service:





This section shows you how to move your data from self-hosted TimescaleDB to a Tiger Cloud service
using live migration from Terminal.

## Set your connection strings

These variables hold the connection information for the source database and target Tiger Cloud service.
In Terminal on your migration machine, set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
```
You find the connection information for your Tiger Cloud service in the configuration file you
downloaded when you created the service.



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool requires a direct connection to the database to function properly.

## Align the version of TimescaleDB on the source and target
1. Ensure that the source and target databases are running the same version of TimescaleDB.

    1. Check the version of TimescaleDB running on your Tiger Cloud service:

       ```bash
       psql target -c "SELECT extversion FROM pg_extension WHERE extname = 'timescaledb';"
       ```

    1. Update the TimescaleDB extension in your source database to match the target service:

       If the TimescaleDB extension is the same version on the source database and target service,
       you do not need to do this.

       ```bash
       psql source -c "ALTER EXTENSION timescaledb UPDATE TO '<version here>';"
       ```

       For more information and guidance, see [Upgrade TimescaleDB](https://docs.tigerdata.com/self-hosted/latest/upgrades/).

1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Tune your source database
You need admin rights to to update the configuration on your source database. If you are using
a managed service, follow the instructions in the `From MST` tab on this page.

1. **Install the `wal2json` extension on your source database**

   [Install wal2json][install-wal2json] on your source database.

1. **Prevent Postgres from treating the data in a snapshot as outdated**

   ```shell
   psql -X -d source -c 'alter system set old_snapshot_threshold=-1'
   ```
   This is not applicable if the source database is Postgres 17 or later.

1. **Set the write-Ahead Log (WAL) to record the information needed for logical decoding**
   ```shell
   psql -X -d source -c 'alter system set wal_level=logical'
   ```

1. **Restart the source database**

   Your configuration changes are now active. However, verify that the
   settings are live in your database.

1. **Enable live-migration to replicate `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.

## Migrate your data, then start downtime
2. **Pull the live-migration docker image to you migration machine**

   ```shell
   sudo docker pull timescale/live-migration:latest
   ```
   To list the available commands, run:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest --help
   ```
   To see the available flags for each command, run `--help` for that command. For example:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest migrate --help
   ```

1. **Create a snapshot image of your source database in your Tiger Cloud service**

   This process checks that you have tuned your source database and target service correctly for replication,
   then creates a snapshot of your data on the migration machine:

   ```shell
   docker run --rm -it --name live-migration-snapshot \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest snapshot
   ```

   Live-migration supplies information about updates you need to make to the source database and target service. For example:

   ```shell
   2024-03-25T12:40:40.884 WARNING: The following tables in the Source DB have neither a primary key nor a REPLICA IDENTITY (FULL/INDEX)
   2024-03-25T12:40:40.884 WARNING: UPDATE and DELETE statements on these tables will not be replicated to the Target DB
   2024-03-25T12:40:40.884 WARNING:        - public.metrics
   ```

   If you have warnings, stop live-migration, make the suggested changes and start again.

1. **Synchronize data between your source database and your Tiger Cloud service**

    This command migrates data from the snapshot to your Tiger Cloud service, then streams
    transactions from the source to the target.

   ```shell
   docker run --rm -it --name live-migration-migrate \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest migrate
   ```



   If the source Postgres version is 17 or later, you need to pass additional
   flag `-e PGVERSION=17` to the `migrate` command.



   During this process, you see the migration process:

   ```shell
   Live-replay will complete in 1 minute 38.631 seconds (source_wal_rate: 106.0B/s, target_replay_rate: 589.0KiB/s, replay_lag: 56MiB)
   ```

   If `migrate` stops add `--resume` to start from where it left off.

   Once the data in your target Tiger Cloud service has almost caught up with the source database,
   you see the following message:

   ```shell
   Target has caught up with source (source_wal_rate: 751.0B/s, target_replay_rate: 0B/s, replay_lag: 7KiB)
       To stop replication, hit 'c' and then ENTER
   ```

   Wait until `replay_lag` is down to a few kilobytes before you move to the next step. Otherwise, data
   replication may not have finished.

1. **Start app downtime**

   1. Stop your app writing to the source database, then let the the remaining transactions
      finish to fully sync with the target. You can use tools like the `pg_top` CLI or
      `pg_stat_activity` to view the current transaction on the source database.

   1. Stop Live-migration.

      ```shell
      hit 'c' and then ENTER
      ```

      Live-migration continues the remaining work. This includes copying
      TimescaleDB metadata, sequences, and run policies. When the migration completes,
      you see the following message:

      ```sh
      Migration successfully completed
      ```

## Validate your data, then restart your app
1. **Validate the migrated data**

   The contents of both databases should be the same. To check this you could compare
   the number of rows, or an aggregate of columns. However, the best validation method
   depends on your app.

1. **Stop app downtime**

   Once you are confident that your data is successfully replicated, configure your apps
   to use your Tiger Cloud service.

1. **Cleanup resources associated with live-migration from your migration machine**

   This command removes all resources and temporary files used in the migration process.
   When you run this command, you can no longer resume live-migration.

   ```shell
   docker run --rm -it --name live-migration-clean \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest clean --prune
   ```




This section shows you how to move your data from self-hosted Postgres to a Tiger Cloud service using
live migration from Terminal.

## Set your connection strings

These variables hold the connection information for the source database and target Tiger Cloud service.
In Terminal on your migration machine, set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
```
You find the connection information for your Tiger Cloud service in the configuration file you
downloaded when you created the service.



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool requires a direct connection to the database to function properly.


## Align the extensions on the source and target
1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Tune your source database
You need admin rights to to update the configuration on your source database. If you are using
a managed service, follow the instructions in the `From AWS RDS/Aurora` tab on this page.

1. **Install the `wal2json` extension on your source database**

   [Install wal2json][install-wal2json] on your source database.

1. **Prevent Postgres from treating the data in a snapshot as outdated**

   ```shell
   psql -X -d source -c 'alter system set old_snapshot_threshold=-1'
   ```
   This is not applicable if the source database is Postgres 17 or later.

1. **Set the write-Ahead Log (WAL) to record the information needed for logical decoding**
   ```shell
   psql -X -d source -c 'alter system set wal_level=logical'
   ```

1. **Restart the source database**

   Your configuration changes are now active. However, verify that the
   settings are live in your database.

1. **Enable live-migration to replicate `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.

## Migrate your data, then start downtime
1. **Pull the live-migration docker image to you migration machine**

   ```shell
   sudo docker pull timescale/live-migration:latest
   ```
   To list the available commands, run:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest --help
   ```
   To see the available flags for each command, run `--help` for that command. For example:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest migrate --help
   ```


1. **Create a snapshot image of your source database in your Tiger Cloud service**

   This process checks that you have tuned your source database and target service correctly for replication,
   then creates a snapshot of your data on the migration machine:

   ```shell
   docker run --rm -it --name live-migration-snapshot \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest snapshot
   ```

   Live-migration supplies information about updates you need to make to the source database and target service. For example:

   ```shell
   2024-03-25T12:40:40.884 WARNING: The following tables in the Source DB have neither a primary key nor a REPLICA IDENTITY (FULL/INDEX)
   2024-03-25T12:40:40.884 WARNING: UPDATE and DELETE statements on these tables will not be replicated to the Target DB
   2024-03-25T12:40:40.884 WARNING:        - public.metrics
   ```

   If you have warnings, stop live-migration, make the suggested changes and start again.

1. **Synchronize data between your source database and your Tiger Cloud service**

    This command migrates data from the snapshot to your Tiger Cloud service, then streams
    transactions from the source to the target.

   ```shell
   docker run --rm -it --name live-migration-migrate \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest migrate
   ```



   If the source Postgres version is 17 or later, you need to pass additional
   flag `-e PGVERSION=17` to the `migrate` command.



   After migrating the schema, live-migration prompts you to create hypertables for tables that
   contain time-series data in your Tiger Cloud service. Run `create_hypertable()` to convert these
   table. For more information, see the [Hypertable docs][Hypertable docs].

   During this process, you see the migration process:

   ```shell
   Live-replay will complete in 1 minute 38.631 seconds (source_wal_rate: 106.0B/s, target_replay_rate: 589.0KiB/s, replay_lag: 56MiB)
   ```

   If `migrate` stops add `--resume` to start from where it left off.

   Once the data in your target Tiger Cloud service has almost caught up with the source database,
   you see the following message:

   ```shell
   Target has caught up with source (source_wal_rate: 751.0B/s, target_replay_rate: 0B/s, replay_lag: 7KiB)
       To stop replication, hit 'c' and then ENTER
   ```

   Wait until `replay_lag` is down to a few kilobytes before you move to the next step. Otherwise, data
   replication may not have finished.

1. **Start app downtime**

   1. Stop your app writing to the source database, then let the the remaining transactions
      finish to fully sync with the target. You can use tools like the `pg_top` CLI or
      `pg_stat_activity` to view the current transaction on the source database.

   1. Stop Live-migration.

      ```shell
      hit 'c' and then ENTER
      ```

      Live-migration continues the remaining work. This includes copying
      TimescaleDB metadata, sequences, and run policies. When the migration completes,
      you see the following message:

      ```sh
      Migration successfully completed
      ```

## Validate your data, then restart your app
1. **Validate the migrated data**

   The contents of both databases should be the same. To check this you could compare
   the number of rows, or an aggregate of columns. However, the best validation method
   depends on your app.

1. **Stop app downtime**

   Once you are confident that your data is successfully replicated, configure your apps
   to use your Tiger Cloud service.

1. **Cleanup resources associated with live-migration from your migration machine**

   This command removes all resources and temporary files used in the migration process.
   When you run this command, you can no longer resume live-migration.

   ```shell
   docker run --rm -it --name live-migration-clean \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest clean --prune
   ```




To migrate your data from an Amazon RDS/Aurora Postgres instance to a Tiger Cloud service, you extract the data to an intermediary
EC2 Ubuntu instance in the same AWS region as your RDS/Aurora instance. You then upload your data to a Tiger Cloud service.
To make this process as painless as possible, ensure that the intermediary machine has enough CPU and disk space to
rapidly extract and store your data before uploading to Tiger Cloud.

Migration from RDS/Aurora gives you the opportunity to create [hypertables][about-hypertables] before copying the data. Once the migration is complete, you can manually enable Tiger Cloud features like [data compression][data-compression] or [data retention][data-retention].

This section shows you how to move your data from an Amazon RDS/Aurora instance to a Tiger Cloud service
using live migration.


## Create an intermediary EC2 Ubuntu instance
1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
   select the RDS/Aurora Postgres instance to migrate.
1. Click `Actions` > `Set up EC2 connection`.
   Press `Create EC2 instance` and use the following settings:
    - **AMI**: Ubuntu Server.
    - **Key pair**: use an existing pair or create a new one that you will use to access the intermediary machine.
    - **VPC**: by default, this is the same as the database instance.
    - **Configure Storage**: adjust the volume to at least the size of RDS/Aurora Postgres instance you are migrating from.
    You can reduce the space used by your data on Tiger Cloud using [Hypercore][hypercore].
1. Click `Lauch instance`. AWS creates your EC2 instance, then click `Connect to instance` > `SSH client`.
   Follow the instructions to create the connection to your intermediary EC2 instance.

## Install the psql client tools on the intermediary instance

1. Connect to your intermediary EC2 instance. For example:
   ```sh
   ssh -i "<key-pair>.pem" ubuntu@<EC2 instance's Public IPv4>
   ```
1. On your intermediary EC2 instance, install the Postgres client.
   ```sh
   sudo sh -c 'echo "deb http://apt.postgresql.org/pub/repos/apt $(lsb_release -cs)-pgdg main" > /etc/apt/sources.list.d/pgdg.list'
   wget -qO- https://www.postgresql.org/media/keys/ACCC4CF8.asc | sudo tee /etc/apt/trusted.gpg.d/pgdg.asc &>/dev/null
   sudo apt update
   sudo apt install postgresql-client-16 -y # "postgresql-client-16" if your source DB is using PG 16.
   psql --version && pg_dump --version
   ```

  Keep this terminal open, you need it to connect to the RDS/Aurora Postgres instance for migration.

## Set up secure connectivity between your RDS/Aurora Postgres and EC2 instances

1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
    select the RDS/Aurora Postgres instance to migrate.
1. Scroll down to `Security group rules (1)` and select the `EC2 Security Group - Inbound` group. The
   `Security Groups (1)` window opens. Click the `Security group ID`, then click `Edit inbound rules`

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/rds-add-security-rule-to-ec2-instance.svg"
   alt="Create security group rule to enable RDS/Aurora Postgres EC2 connection"/>

1. On your intermediary EC2 instance, get your local IP address:
   ```sh
   ec2metadata --local-ipv4
   ```
   Bear with me on this one, you need this IP address to enable access to your RDS/Aurora Postgres instance.
1. In `Edit inbound rules`, click `Add rule`, then create a `PostgreSQL`, `TCP` rule granting access
   to the local IP address for your EC2 instance (told you :-)). Then click `Save rules`.

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/rds-add-inbound-rule-for-ec2-instance.png"
   alt="Create security rule to enable RDS/Aurora Postgres EC2 connection"/>

## Test the connection between your RDS/Aurora Postgres and EC2 instances

1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
    select the RDS/Aurora Postgres instance to migrate.
1. On your intermediary EC2 instance, use the values of `Endpoint`, `Port`, `Master username`, and `DB name`
   to create the postgres connectivity string to the `SOURCE` variable.

   <img class="main-content__illustration"
   src="https://assets.timescale.com/docs/images/migrate/migrate-source-rds-instance.svg"
   alt="Record endpoint, port, VPC details"/>

   ```sh
   export SOURCE="postgres://<Master username>:<Master password>@<Endpoint>:<Port>/<DB name>"
   ```
   The value of `Master password` was supplied when this RDS/Aurora Postgres instance was created.

1. Test your connection:
   ```sh
   psql -d source
   ```
   You are connected to your RDS/Aurora Postgres instance from your intermediary EC2 instance.

## Set your connection strings

These variables hold the connection information for the source database and target Tiger Cloud service.
In Terminal on your migration machine, set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
```
You find the connection information for your Tiger Cloud service in the configuration file you
downloaded when you created the service.



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool requires a direct connection to the database to function properly.

## Align the extensions on the source and target
1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Tune your source database

Updating parameters on a Postgres instance will cause an outage. Choose a time that will cause the least issues to tune this database.

1. **Update the DB instance parameter group for your source database**

   1. In [https://console.aws.amazon.com/rds/home#databases:][databases],
      select the RDS instance to migrate.

   1. Click `Configuration`, scroll down and note the `DB instance parameter group`, then click `Parameter groups`

      <img class="main-content__illustration"
      src="https://assets.timescale.com/docs/images/migrate/awsrds-parameter-groups.png"
      alt="Create security rule to enable RDS EC2 connection"/>

   1. Click `Create parameter group`, fill in the form with the following values, then click `Create`.
      - **Parameter group name** - whatever suits your fancy.
      - **Description** - knock yourself out with this one.
      - **Engine type** - `PostgreSQL`
      - **Parameter group family** - the same as `DB instance parameter group` in your `Configuration`.
   1. In `Parameter groups`, select the parameter group you created, then click `Edit`.
   1. Update the following parameters, then click `Save changes`.
      - `rds.logical_replication` set to `1`: record the information needed for logical decoding.
      - `wal_sender_timeout` set to `0`: disable the timeout for the sender process.

   1. In RDS, navigate back to your [databases][databases], select the RDS instance to migrate, and click `Modify`.

   1. Scroll down to `Database options`, select your new parameter group, and click `Continue`.
   1. Click `Apply immediately` or choose a maintenance window, then click `Modify DB instance`.

      Changing parameters will cause an outage. Wait for the database instance to reboot before continuing.
   1. Verify that the settings are live in your database.

1. **Enable replication `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.

## Migrate your data, then start downtime
1. **Pull the live-migration docker image to you migration machine**

   ```shell
   sudo docker pull timescale/live-migration:latest
   ```
   To list the available commands, run:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest --help
   ```
   To see the available flags for each command, run `--help` for that command. For example:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest migrate --help
   ```


1. **Create a snapshot image of your source database in your Tiger Cloud service**

   This process checks that you have tuned your source database and target service correctly for replication,
   then creates a snapshot of your data on the migration machine:

   ```shell
   docker run --rm -it --name live-migration-snapshot \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest snapshot
   ```

   Live-migration supplies information about updates you need to make to the source database and target service. For example:

   ```shell
   2024-03-25T12:40:40.884 WARNING: The following tables in the Source DB have neither a primary key nor a REPLICA IDENTITY (FULL/INDEX)
   2024-03-25T12:40:40.884 WARNING: UPDATE and DELETE statements on these tables will not be replicated to the Target DB
   2024-03-25T12:40:40.884 WARNING:        - public.metrics
   ```

   If you have warnings, stop live-migration, make the suggested changes and start again.

1. **Synchronize data between your source database and your Tiger Cloud service**

    This command migrates data from the snapshot to your Tiger Cloud service, then streams
    transactions from the source to the target.

   ```shell
   docker run --rm -it --name live-migration-migrate \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest migrate
   ```



   If the source Postgres version is 17 or later, you need to pass additional
   flag `-e PGVERSION=17` to the `migrate` command.



   After migrating the schema, live-migration prompts you to create hypertables for tables that
   contain time-series data in your Tiger Cloud service. Run `create_hypertable()` to convert these
   table. For more information, see the [Hypertable docs][Hypertable docs].

   During this process, you see the migration process:

   ```shell
   Live-replay will complete in 1 minute 38.631 seconds (source_wal_rate: 106.0B/s, target_replay_rate: 589.0KiB/s, replay_lag: 56MiB)
   ```

   If `migrate` stops add `--resume` to start from where it left off.

   Once the data in your target Tiger Cloud service has almost caught up with the source database,
   you see the following message:

   ```shell
   Target has caught up with source (source_wal_rate: 751.0B/s, target_replay_rate: 0B/s, replay_lag: 7KiB)
       To stop replication, hit 'c' and then ENTER
   ```

   Wait until `replay_lag` is down to a few kilobytes before you move to the next step. Otherwise, data
   replication may not have finished.

1. **Start app downtime**

   1. Stop your app writing to the source database, then let the the remaining transactions
      finish to fully sync with the target. You can use tools like the `pg_top` CLI or
      `pg_stat_activity` to view the current transaction on the source database.

   1. Stop Live-migration.

      ```shell
      hit 'c' and then ENTER
      ```

      Live-migration continues the remaining work. This includes copying
      TimescaleDB metadata, sequences, and run policies. When the migration completes,
      you see the following message:

      ```sh
      Migration successfully completed
      ```

## Validate your data, then restart your app
1. **Validate the migrated data**

   The contents of both databases should be the same. To check this you could compare
   the number of rows, or an aggregate of columns. However, the best validation method
   depends on your app.

1. **Stop app downtime**

   Once you are confident that your data is successfully replicated, configure your apps
   to use your Tiger Cloud service.

1. **Cleanup resources associated with live-migration from your migration machine**

   This command removes all resources and temporary files used in the migration process.
   When you run this command, you can no longer resume live-migration.

   ```shell
   docker run --rm -it --name live-migration-clean \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest clean --prune
   ```





This section shows you how to move your data from a MST instance to a
Tiger Cloud service using live migration from Terminal.

## Set your connection strings

These variables hold the connection information for the source database and target Tiger Cloud service.
In Terminal on your migration machine, set the following:

```bash
export SOURCE="postgres://<user>:<password>@<source host>:<source port>/<db_name>"
export TARGET="postgres://tsdbadmin:<PASSWORD>@<HOST>:<PORT>/tsdb?sslmode=require"
```
You find the connection information for your Tiger Cloud service in the configuration file you
downloaded when you created the service.



Avoid using connection strings that route through connection poolers like PgBouncer or similar tools. This tool requires a direct connection to the database to function properly.

## Align the version of TimescaleDB on the source and target
1. Ensure that the source and target databases are running the same version of TimescaleDB.

    1. Check the version of TimescaleDB running on your Tiger Cloud service:

       ```bash
       psql target -c "SELECT extversion FROM pg_extension WHERE extname = 'timescaledb';"
       ```

    1. Update the TimescaleDB extension in your source database to match the target service:

       If the TimescaleDB extension is the same version on the source database and target service,
       you do not need to do this.

       ```bash
       psql source -c "ALTER EXTENSION timescaledb UPDATE TO '<version here>';"
       ```

       For more information and guidance, see [Upgrade TimescaleDB](https://docs.tigerdata.com/self-hosted/latest/upgrades/).

1. Ensure that the Tiger Cloud service is running the Postgres extensions used in your source database.

    1. Check the extensions on the source database:
       ```bash
       psql source  -c "SELECT * FROM pg_extension;"
       ```
    1. For each extension, enable it on your target Tiger Cloud service:
       ```bash
       psql target  -c "CREATE EXTENSION IF NOT EXISTS <extension name> CASCADE;"
       ```

## Tune your source database

1. **Enable live-migration to replicate `DELETE` and`UPDATE` operations**

   Replica identity assists data replication by identifying the rows being modified. Your options are that
   each table and hypertable in the source database should either have:
- **A primary key**: data replication defaults to the primary key of the table being replicated.
  Nothing to do.
- **A viable unique index**: each table has a unique, non-partial, non-deferrable index that includes only columns
  marked as `NOT NULL`. If a UNIQUE index does not exist, create one to assist the migration. You can delete if after
  migration.

  For each table, set `REPLICA IDENTITY` to the viable unique index:

   ```shell
   psql -X -d source -c 'ALTER TABLE  REPLICA IDENTITY USING INDEX <_index_name>'
   ```
- **No primary key or viable unique index**: use brute force.

  For each table, set `REPLICA IDENTITY` to `FULL`:
  ```shell
  psql -X -d source -c 'ALTER TABLE {table_name} REPLICA IDENTITY FULL'
   ```
  For each `UPDATE` or `DELETE` statement, Postgres reads the whole table to find all matching rows. This results
  in significantly slower replication. If you are expecting a large number of `UPDATE` or `DELETE` operations on the table,
  best practice is to not use `FULL`.

## Migrate your data, then start downtime
2. **Pull the live-migration docker image to you migration machine**

   ```shell
   sudo docker pull timescale/live-migration:latest
   ```
   To list the available commands, run:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest --help
   ```
   To see the available flags for each command, run `--help` for that command. For example:
   ```shell
   sudo docker run --rm -it -e PGCOPYDB_SOURCE_PGURI=source  timescale/live-migration:latest migrate --help
   ```

1. **Create a snapshot image of your source database in your Tiger Cloud service**

   This process checks that you have tuned your source database and target service correctly for replication,
   then creates a snapshot of your data on the migration machine:

   ```shell
   docker run --rm -it --name live-migration-snapshot \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest snapshot
   ```

   Live-migration supplies information about updates you need to make to the source database and target service. For example:

   ```shell
   2024-03-25T12:40:40.884 WARNING: The following tables in the Source DB have neither a primary key nor a REPLICA IDENTITY (FULL/INDEX)
   2024-03-25T12:40:40.884 WARNING: UPDATE and DELETE statements on these tables will not be replicated to the Target DB
   2024-03-25T12:40:40.884 WARNING:        - public.metrics
   ```

   If you have warnings, stop live-migration, make the suggested changes and start again.

1. **Synchronize data between your source database and your Tiger Cloud service**

    This command migrates data from the snapshot to your Tiger Cloud service, then streams
    transactions from the source to the target.

   ```shell
   docker run --rm -it --name live-migration-migrate \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest migrate
   ```



   If the source Postgres version is 17 or later, you need to pass additional
   flag `-e PGVERSION=17` to the `migrate` command.



   During this process, you see the migration process:

   ```shell
   Live-replay will complete in 1 minute 38.631 seconds (source_wal_rate: 106.0B/s, target_replay_rate: 589.0KiB/s, replay_lag: 56MiB)
   ```

   If `migrate` stops add `--resume` to start from where it left off.

   Once the data in your target Tiger Cloud service has almost caught up with the source database,
   you see the following message:

   ```shell
   Target has caught up with source (source_wal_rate: 751.0B/s, target_replay_rate: 0B/s, replay_lag: 7KiB)
       To stop replication, hit 'c' and then ENTER
   ```

   Wait until `replay_lag` is down to a few kilobytes before you move to the next step. Otherwise, data
   replication may not have finished.

1. **Start app downtime**

   1. Stop your app writing to the source database, then let the the remaining transactions
      finish to fully sync with the target. You can use tools like the `pg_top` CLI or
      `pg_stat_activity` to view the current transaction on the source database.

   1. Stop Live-migration.

      ```shell
      hit 'c' and then ENTER
      ```

      Live-migration continues the remaining work. This includes copying
      TimescaleDB metadata, sequences, and run policies. When the migration completes,
      you see the following message:

      ```sh
      Migration successfully completed
      ```

## Validate your data, then restart your app
1. **Validate the migrated data**

   The contents of both databases should be the same. To check this you could compare
   the number of rows, or an aggregate of columns. However, the best validation method
   depends on your app.

1. **Stop app downtime**

   Once you are confident that your data is successfully replicated, configure your apps
   to use your Tiger Cloud service.

1. **Cleanup resources associated with live-migration from your migration machine**

   This command removes all resources and temporary files used in the migration process.
   When you run this command, you can no longer resume live-migration.

   ```shell
   docker run --rm -it --name live-migration-clean \
       -e PGCOPYDB_SOURCE_PGURI=source \
       -e PGCOPYDB_TARGET_PGURI=target \
       --pid=host \
       -v ~/live-migration:/opt/timescale/ts_cdc \
       timescale/live-migration:latest clean --prune
   ```




And you are done, your data is now in your Tiger Cloud service.

## Troubleshooting

This section shows you how to work around frequently seen issues when using live migration.

### ERROR: relation "xxx.yy" does not exist

This may happen when a relation is removed after executing the `snapshot` command. A relation can be
a table, index, view, or materialized view. When you see you this error:

- Do not perform any explicit DDL operation on the source database during the course of migration.

- If you are migrating from self-hosted TimescaleDB or MST, disable the chunk retention policy on your source database
  until you have finished migration.

### FATAL: remaining connection slots are reserved for non-replication superuser connections

This may happen when the number of connections exhaust `max_connections` defined in your target Tiger Cloud service.
By default, live-migration needs around ~6 connections on the source and ~12 connections on the target.

### Migration seems to be stuck with “x GB copied to Target DB (Source DB is y GB)”

When you are migrating a lot of data involved in aggregation, or there are many materialized views taking time
to complete the materialization, this may be due to `REFRESH MATERIALIZED VIEWS` happening at the end of initial
data migration.

To resolve this issue:

1. See what is happening on the target Tiger Cloud service:
   ```shell
   psql target -c "select * from pg_stat_activity where application_name ilike '%pgcopydb%';"
   ```

1. When you run the `migrate`, add the following flags to exclude specific materialized views being materialized:
   ```shell
   --skip-table-data <matview1> <matview2>”
   ```

1. When `migrate` has finished, manually refresh the materialized views you excluded.


### Restart migration from scratch after a non-resumable failure

If the migration halts due to a failure, such as a misconfiguration of the source or target database, you may need to
restart the migration from scratch. In such cases, you can reuse the original target Tiger Cloud service created for the
migration by utilizing the `--drop-if-exists` flag with the migrate command.

This flag ensures that the existing target objects created by the previous migration are dropped, allowing the migration
to proceed without trouble.

Note: This flag also requires you to manually recreate the TimescaleDB extension on the target.

Here’s an example command sequence to restart the migration:

```shell
psql target -c "DROP EXTENSION timescaledb CASCADE"

psql target -c 'CREATE EXTENSION timescaledb VERSION "<desired version>"'

docker run --rm -it --name live-migration-migrate \
    -e PGCOPYDB_SOURCE_PGURI=source \
    -e PGCOPYDB_TARGET_PGURI=target \
    --pid=host \
    -v ~/live-migration:/opt/timescale/ts_cdc \
    timescale/live-migration:latest migrate --drop-if-exists
```

This approach provides a clean slate for the migration process while reusing the existing target instance.

### Inactive or lagging replication slots

If you encounter an “Inactive or lagging replication slots” warning on your cloud provider console after using live-migration, it might be due to lingering replication slots created by the live-migration tool on your source database.

To clean up resources associated with live migration, use the following command:

```sh
docker run --rm -it --name live-migration-clean \
    -e PGCOPYDB_SOURCE_PGURI=source \
    -e PGCOPYDB_TARGET_PGURI=target \
    --pid=host \
    -v ~/live-migration:/opt/timescale/ts_cdc \
    timescale/live-migration:latest clean --prune
```

The `--prune` flag is used to delete temporary files in the `~/live-migration` directory
that were needed for the migration process. It's important to note that executing the
`clean` command means you cannot resume the interrupted live migration.


### Role passwords

Because of issues dumping passwords from various managed service providers, Live-migration
migrates roles without passwords. You have to migrate passwords manually.


### Table privileges

Live-migration does not migrate table privileges. After completing Live-migration:

1. Grant all roles to `tsdbadmin`.
   ```shell
   psql -d source -t -A -c "SELECT FORMAT('GRANT %I TO tsdbadmin;', rolname) FROM
   pg_catalog.pg_roles WHERE rolname not like 'pg_%' AND rolname != 'tsdbadmin'
   AND NOT rolsuper" | psql -d target -f -
   ```

1. On your migration machine, edit `/tmp/grants.psql` to match table privileges on your source database.
   ```shell
   pg_dump --schema-only --quote-all-identifiers
   --exclude-schema=_timescaledb_catalog --format=plain --dbname "source" | grep
   "(ALTER.*OWNER.*|GRANT|REVOKE)"  > /tmp/grants.psql
   ```

1. Run `grants.psql` on your target Tiger Cloud service.
   ```shell
   psql -d target -f /tmp/grants.psql
   ```

### Postgres to Tiger Cloud: “live-replay not keeping up with source load”

1. Go to Tiger Cloud Console -> `Monitoring` -> `Insights` tab and find the query which takes significant time
2. If the query is either UPDATE/DELETE, make sure the columns used on the WHERE clause have necessary indexes.
3. If the query is either UPDATE/DELETE on the tables which are converted as hypertables, make sure the REPLIDA IDENTITY(defaults to primary key) on the source is compatible with the target primary key. If not, create an UNIQUE index source database by including the hypertable partition column and make it as a REPLICA IDENTITY. Also, create the same UNIQUE index on target.

### ERROR: out of memory (or) Failed on request of size xxx in memory context "yyy" on a Tiger Cloud service

This error occurs when the Out of Memory (OOM) guard is triggered due to memory allocations exceeding safe limits. It typically happens when multiple concurrent connections to the TimescaleDB instance are performing memory-intensive operations. For example, during live migrations, this error can occur when large indexes are being created simultaneously.

The live-migration tool includes a retry mechanism to handle such errors. However, frequent OOM crashes may significantly delay the migration process.

One of the following can be used to avoid the OOM errors:

1. Upgrade to Higher Memory Spec Instances: To mitigate memory constraints, consider using a TimescaleDB instance with higher specifications, such as an instance with 8 CPUs and 32 GB RAM (or more). Higher memory capacity can handle larger workloads and reduce the likelihood of OOM errors.

1. Reduce Concurrency: If upgrading your instance is not feasible, you can reduce the concurrency of the index migration process using the `--index-jobs=<value>` flag in the migration command. By default, the value of `--index-jobs` matches the GUC max_parallel_workers. Lowering this value reduces the memory usage during migration but may increase the total migration time.

By taking these steps, you can prevent OOM errors and ensure a smoother migration experience with TimescaleDB.


===== PAGE: https://docs.tigerdata.com/migrate/dual-write-and-backfill/ =====

# Low-downtime migrations with dual-write and backfill



Dual-write and backfill is a migration strategy to move a large amount of
time-series data (100&nbsp;GB-10&nbsp;TB+) with low downtime (on the order of
minutes of downtime). It is significantly more complicated to execute than a
migration with downtime using [pg_dump/restore][pg-dump-and-restore], and has
some prerequisites on the data ingest patterns of your application, so it may
not be universally applicable.

In the context of migrations, your existing production database is referred to
as the SOURCE database, the Tiger Cloud service that you are migrating your data to is the TARGET.

Roughly, it consists of three steps:

1. Clone schema and relational data from source to target
1. Dual-write to source and target
1. Backfill time-series data

Dual-write and backfill can be used for any source database type, as long as it
can provide data in csv format. It can be used to move data from a PostgresSQL
source, and from TimescaleDB to TimescaleDB.

Dual-write and backfill works well when:
1. The bulk of the (on-disk) data is in time-series tables.
1. Writes by the application do not reference historical time-series data.
1. Writes to time-series data are append-only.
1. No `UPDATE` or `DELETE` queries will be run on time-series data in the
   source database during the migration process (or if they are, it happens in
   a controlled manner, such that it's possible to either ignore, or
   re-backfill).
1. Either the relational (non-time-series) data is small enough to be copied
   from source to target in an acceptable amount of time for this to be done
   with downtime, or the relational data can be copied asynchronously while the
   application continues to run (that is, changes relatively infrequently).

For more information, consult the step-by-step guide for your source database:

- [Dual-write and backfill from TimescaleDB][from-timescaledb]
- [Dual-write and backfill from Postgres][from-postgres]
- [Dual-write and backfill from other][from-other]

If you get stuck, you can get help by either opening a support request, or take
your issue to the `#migration` channel in the [community slack](https://slack.timescale.com/),
where the developers of this migration method are there to help.

You can open a support request directly from [Tiger Cloud Console][support-link],
or by email to [support@tigerdata.com](mailto:support@tigerdata.com).


===== PAGE: https://docs.tigerdata.com/migrate/troubleshooting/ =====

# FAQ and troubleshooting



## Unsupported in live migration

Live migration tooling is currently experimental. You may run into the following shortcomings:

- Live migration does not yet support mutable columnstore compression (`INSERT`, `UPDATE`,
  `DELETE` on data in the columnstore).
- By default, numeric fields containing `NaN`/`+Inf`/`-Inf` values are not
  correctly replicated, and will be converted to `NULL`. A workaround is available, but is not enabled by default.

Should you run into any problems, please open a support request before losing
any time debugging issues.
You can open a support request directly from [Tiger Cloud Console][support-link],
or by email to [support@tigerdata.com](mailto:support@tigerdata.com).

## Where can I find logs for processes running during live migration?

Live migration involves several background processes to manage different stages of
the migration. The logs of these processes can be helpful for troubleshooting
unexpected behavior. You can find these logs in the `<volume_mount>/logs` directory.


## Source and target databases have different TimescaleDB versions

When you migrate a [self-hosted][self hosted] or [Managed Service for TimescaleDB (MST)][mst]
database to Tiger Cloud, the source database and the destination
[Tiger Cloud service][timescale-service] must run the same version of TimescaleDB.

Before you start [live migration][live migration]:


1. Check the version of TimescaleDB running on the source database and the
   target Tiger Cloud service:
    ```sql
    select extversion from pg_extension where extname = 'timescaledb';
    ```

1. If the version of TimescaleDB on the source database is lower than your Tiger Cloud service, either:
    - **Downgrade**: reinstall an older version of TimescaleDB on your Tiger Cloud service that matches the source database:

        1. Connect to your Tiger Cloud service and check the versions of TimescaleDB available:
           ```sql
           SELECT version FROM pg_available_extension_versions WHERE name = 'timescaledb' ORDER BY 1 DESC;
           ```

        2. If an available TimescaleDB release matches your source database:

            1. Uninstall TimescaleDB from your Tiger Cloud service:
               ```sql
               DROP EXTENSION timescaledb;
               ```

            1. Reinstall the correct version of TimescaleDB:
               ```sql
               CREATE EXTENSION timescaledb VERSION '<version>';
               ```



           You may need to reconnect to your Tiger Cloud service using `psql -X` when you're creating the TimescaleDB extension.



    - **Upgrade**: for self-hosted databases, [upgrade TimescaleDB][self hosted upgrade] to match your Tiger Cloud service.



## Why does live migration log "no tuple identifier" warning?

Live migration logs a warning `WARNING: no tuple identifier for UPDATE in table`
when it cannot determine which specific rows should be updated after receiving an
`UPDATE` statement from the source database during replication. This occurs when tables
in the source database that receive `UPDATE` statements lack either a `PRIMARY KEY` or
a `REPLICA IDENTITY` setting. For live migration to successfully replicate `UPDATE` and
`DELETE` statements, tables must have either a `PRIMARY KEY` or `REPLICA IDENTITY` set
as a prerequisite.


## Set REPLICA IDENTITY on Postgres partitioned tables

If your Postgres tables use native partitioning, setting `REPLICA IDENTITY` on the
root (parent) table will not automatically apply it to the partitioned child tables.
You must manually set `REPLICA IDENTITY` on each partitioned child table.


## Can I use read/failover replicas as source database for live migration?

Live migration does not support replication from read or failover replicas. You must
provide a connection string that points directly to your source database for
live migration.


## Can I use live migration with a Postgres connection pooler like PgBouncer?

Live migration does not support connection poolers. You must provide a
connection string that points directly to your source and target databases
for live migration to work smoothly.


## Can I use Tiger Cloud instance as source for live migration?

No, Tiger Cloud cannot be used as a source database for live migration.


## How can I exclude a schema/table from being replicated in live migration?

At present, live migration does not allow for excluding schemas or tables from
replication, but this feature is expected to be added in future releases.
However, a workaround is available for skipping table data using the `--skip-table-data` flag.
For more information, please refer to the help text under the `migrate` subcommand.

## Large migrations blocked

Tiger Cloud automatically manages the underlying disk volume. Due to
platform limitations, it is only possible to resize the disk once every six
hours. Depending on the rate at which you're able to copy data, you may be
affected by this restriction. Affected instances are unable to accept new data
and error with: `FATAL: terminating connection due to administrator command`.

If you intend on migrating more than 400&nbspGB of data to Tiger Cloud, open a
support request requesting the required storage to be pre-allocated in your
Tiger Cloud service.

You can open a support request directly from [Tiger Cloud Console][support-link],
or by email to [support@tigerdata.com](mailto:support@tigerdata.com).

## Dumping and locks

When `pg_dump` starts, it takes an `ACCESS SHARE` lock on all tables which it
dumps. This ensures that tables aren't dropped before `pg_dump` is able to drop
them. A side effect of this is that any query which tries to take an
`ACCESS EXCLUSIVE` lock on a table is be blocked by the `ACCESS SHARE` lock.

A number of Tiger Cloud-internal processes require taking `ACCESS EXCLUSIVE`
locks to ensure consistency of the data. The following is a non-exhaustive list
of potentially affected operations:

- converting a chunk into the columnstore/rowstore and back
- continuous aggregate refresh (before 2.12)
- create hypertable with foreign keys, truncate hypertable
- enable hypercore on a hypertable
- drop chunks

The most likely impact of the above is that background jobs for retention
policies, columnstore compression policies, and continuous aggregate refresh policies are
blocked for the duration of the `pg_dump` command. This may have unintended
consequences for your database performance.

## Dumping with concurrency

When using the `pg_dump` directory format, it is possible to use concurrency to
use multiple connections to the source database to dump data. This speeds up
the dump process. Due to the fact that there are multiple connections, it is
possible for `pg_dump` to end up in a deadlock situation. When it detects a
deadlock it aborts the dump.

In principle, any query which takes an `ACCESS EXCLUSIVE` lock on a table
causes such a deadlock. As mentioned above, some common operations which take
an `ACCESS EXCLUSIVE` lock are:
- retention policies
- columnstore compression policies
- continuous aggregate refresh policies

If you would like to use concurrency nonetheless, turn off all background jobs
in the source database before running `pg_dump`, and turn them on once the dump
is complete. If the dump procedure takes longer than the continuous aggregate
refresh policy's window, you must manually refresh the continuous aggregate in
the correct time range. For more information, consult the
[refresh policies documentation].

To turn off the jobs:
```sql
SELECT public.alter_job(id::integer, scheduled=>false)
FROM _timescaledb_config.bgw_job
WHERE id >= 1000;
```

To turn on the jobs:
```sql
SELECT public.alter_job(id::integer, scheduled=>true)
FROM _timescaledb_config.bgw_job
WHERE id >= 1000;
```


## Restoring with concurrency

If the directory format is used for `pg_dump` and `pg_restore`, concurrency can be
employed to speed up the process. Unfortunately, loading the tables in the
`timescaledb_catalog` schema concurrently causes errors. Furthermore, the
`tsdbadmin` user does not have sufficient privileges to turn off triggers in
this schema. To get around this limitation, load this schema serially, and then
load the rest of the database concurrently.

```bash
pg_restore -d "target" \
    --format=directory \
    --schema='_timescaledb_catalog' \
    --exit-on-error \
    --no-tablespaces \
    --no-owner \
    --no-privileges \
    dump

pg_restore -d "target" \
    --format=directory \
    --jobs=8 \
    --exclude-schema='_timescaledb_catalog' \
    --exit-on-error \
    --disable-triggers \
    --no-tablespaces \
    --no-owner \
    --no-privileges \
    dump
```

## Ownership of background jobs

The `_timescaledb_config.bgw_jobs` table is used to manage background jobs.
This includes custom jobs, columnstore compression policies, retention
policies, and continuous aggregate refresh policies. On Tiger Cloud, this table
has a trigger which ensures that no database user can create or modify jobs
owned by another database user. This trigger can provide an obstacle for migrations.

If the `--no-owner` flag is used with `pg_dump` and `pg_restore`, all
objects in the target database are owned by the user that ran
`pg_restore`, likely `tsdbadmin`.

If all the background jobs in the source database were owned by a user of the
same name as the user running the restore (again likely `tsdbadmin`), then
loading the `_timescaledb_config.bgw_jobs` table should work.

If the background jobs in the source were owned by the `postgres` user, they
are be automatically changed to be owned by the `tsdbadmin` user. In this case,
one just needs to verify that the jobs do not make use of privileges that the
`tsdbadmin` user does not possess.

If background jobs are owned by one or more users other than the user
employed in restoring, then there could be issues. To work around this
issue, do not dump this table with `pg_dump`. Provide either
`--exclude-table-data='_timescaledb_config.bgw_job'` or
`--exclude-table='_timescaledb_config.bgw_job'` to `pg_dump` to skip
this table.  Then, use `psql` and the `COPY` command to dump and
restore this table with modified values for the `owner` column.

```bash
psql -d "source" -X -v ON_ERROR_STOP=1 --echo-errors -f - <<'EOF'
begin;
select string_agg
( case attname
    when 'owner' then $$'tsdbadmin' as "owner"$$
    else format('%I', attname)
  end
, ', '
) as cols
from pg_namespace n
inner join pg_class c
on (n.nspname = '_timescaledb_config'
and n.oid = c.relnamespace
and c.relname = 'bgw_job')
inner join pg_attribute a
on (c.oid = a.attrelid and a.attnum > 0)
\gset
copy
(
    select :cols
    from _timescaledb_config.bgw_job
    where id >= 1000
) to stdout with (format csv, header true)
\g bgw_job.csv
rollback;
\q
EOF

psql -X -d "target" -v ON_ERROR_STOP=1 --echo-errors \
    -c "\copy _timescaledb_config.bgw_job from 'bgw_job.csv' with (format csv, header match)"
```

Once the table has been loaded and the restore completed, you may then use SQL
to adjust the ownership of the jobs and/or the associated stored procedures and
functions as you wish.

## Extension availability

There are a vast number of Postgres extensions available in the wild.
Tiger Cloud supports many of the most popular extensions, but not all extensions.
Before migrating, check that the extensions you are using are supported on
Tiger Cloud. Consult the [list of supported extensions].


## TimescaleDB extension in the public schema

When self-hosting, the TimescaleDB extension may be installed in an arbitrary
schema. Tiger Cloud only supports installing the TimescaleDB extension in the
`public` schema. How to go about resolving this depends heavily on the
particular details of the source schema and the migration approach chosen.

## Tablespaces

Tiger Cloud does not support using custom tablespaces. Providing the
`--no-tablespaces` flag to `pg_dump` and `pg_restore` when
dumping/restoring the schema results in all objects being in the
default tablespace as desired.

## Only one database per instance

While Postgres clusters can contain many databases, Tiger Cloud services are
limited to a single database. When migrating a cluster with multiple databases
to Tiger Cloud, one can either migrate each source database to a separate
Tiger Cloud service or "merge" source databases to target schemas.

## Superuser privileges

The `tsdbadmin` database user is the most powerful available on Tiger Cloud, but it
is not a true superuser. Review your application for use of superuser privileged
operations and mitigate before migrating.

## Migrate partial continuous aggregates

In order to improve the performance and compatibility of continuous aggregates, TimescaleDB
v2.7 replaces _partial_ continuous aggregates with _finalized_ continuous aggregates.

To test your database for partial continuous aggregates, run the following query:

```SQL
SELECT exists (SELECT 1 FROM timescaledb_information.continuous_aggregates WHERE NOT finalized);
```

If you have partial continuous aggregates in your database, [migrate them][migrate]
from partial to finalized before you migrate your database.

If you accidentally migrate partial continuous aggregates across Postgres
versions, you see the following error when you query any continuous aggregates:

```
ERROR:  insufficient data left in message.
```


===== PAGE: https://docs.tigerdata.com/ai/mcp-server/ =====

# Integrate Tiger Cloud with your AI Assistant




The Tiger Model Context Protocol Server provides access to your Tiger Cloud resources through Claude and other AI Assistants. Tiger MCP Server
mirrors the functionality of Tiger CLI and is integrated directly into the CLI binary. You manage your
Tiger Cloud resources using natural language from your AI Assistant. As Tiger MCP Server is integrated with the
Tiger Data documentation, ask any question and you will get the best answer.

This page shows you how to install Tiger CLI and set up secure authentication for Tiger MCP Server, then manage the
resources in your Tiger Data account through the Tiger Model Context Protocol Server using your AI Assistant.

## Prerequisites

To follow the steps on this page:

* Create a target [Tiger Data account][create-account].

* Install an AI Assistant on your developer device with an active API key.

  The following AI Assistants are automatically configured by the Tiger Model Context Protocol Server: `claude-code`, `cursor`, `windsurf`, `codex`, `gemini/gemini-cli`, `vscode/code/vs-code`.
  You can also [manually configure][manual-config] Tiger MCP Server.

## Install and configure Tiger MCP Server

The Tiger MCP Server is bundled with Tiger CLI:

1. **Install Tiger CLI**

   Use the terminal to install the CLI:




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.deb.sh | sudo os=any dist=any bash
    sudo apt-get install tiger-cli
    ```




    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    curl -s https://packagecloud.io/install/repositories/timescale/tiger-cli/script.rpm.sh | sudo os=rpm_any dist=rpm_any bash
    sudo yum install tiger-cli
    ```





    ```shell
    brew install --cask timescale/tap/tiger-cli
    ```





    ```shell
    curl -fsSL https://cli.tigerdata.com | sh
    ```





1. **Set up API credentials**

   1. Log Tiger CLI into your Tiger Data account:

      ```shell
      tiger auth login
      ```
      Tiger CLI opens Console in your browser. Log in, then click `Authorize`.

      You can have a maximum of 10 active client credentials. If you get an error, open [credentials][rest-api-credentials]
      and delete an unused credential.

   1. Select a Tiger Cloud project:

      ```terminaloutput
      Auth URL is: https://console.cloud.timescale.com/oauth/authorize?client_id=lotsOfURLstuff
      Opening browser for authentication...
      Select a project:

      > 1. Tiger Project (tgrproject)
      2. YourCompany (Company wide project) (cpnproject)
      3. YourCompany Department (dptproject)

      Use ↑/↓ arrows or number keys to navigate, enter to select, q to quit
      ```
      If only one project is associated with your account, this step is not shown.

      Where possible, Tiger CLI stores your authentication information in the system keychain/credential manager.
      If that fails, the credentials are stored in `~/.config/tiger/credentials` with restricted file permissions (600).
      By default, Tiger CLI stores your configuration in `~/.config/tiger/config.yaml`.

1. **Test your authenticated connection to Tiger Cloud by listing services**

    ```bash
    tiger service list
    ```

   This call returns something like:
    - No services:
      ```terminaloutput
      🏜️  No services found! Your project is looking a bit empty.
      🚀 Ready to get started? Create your first service with: tiger service create
      ```
    - One or more services:

      ```terminaloutput
      ┌────────────┬─────────────────────┬────────┬─────────────┬──────────────┬──────────────────┐
      │ SERVICE ID │        NAME         │ STATUS │    TYPE     │    REGION    │     CREATED      │
      ├────────────┼─────────────────────┼────────┼─────────────┼──────────────┼──────────────────┤
      │ tgrservice │ tiger-agent-service │ READY  │ TIMESCALEDB │ eu-central-1 │ 2025-09-25 16:09 │
      └────────────┴─────────────────────┴────────┴─────────────┴──────────────┴──────────────────┘
      ```

1. **Configure your AI Assistant to interact with the project and services in your Tiger Data account**

   For example:
   ```shell
   tiger mcp install
   ```

1. **Choose the client to integrate with, then press `Enter` **

   ```shell
   Select an MCP client to configure:

   > 1. Claude Code
   2. Codex
   3. Cursor
   4. Gemini CLI
   5. VS Code
   6. Windsurf

   Use ↑/↓ arrows or number keys to navigate, enter to select, q to quit
   ```

And that is it, you are ready to use the Tiger Model Context Protocol Server to manage your services in Tiger Cloud.

## Manage the resources in your Tiger Data account through your AI Assistant

Your AI Assistant is connected to your Tiger Data account and the Tiger Data documentation, you can now use it to
manage your services and learn more about how to implement Tiger Cloud features. For example:

1. **Run your AI Assistant**
   ```shell
   claude
   ```
   Claude automatically runs the Tiger MCP Server server that enables you to interact with Tiger Cloud from your
   AI Assistant.

1. **Check your Tiger Model Context Protocol Server configuration**
   ```shell
   > is the tigerdata mcp server active for you?
   ```
   You see something like:
   ```shell
   MCP server is active. I can see the following Tiger Data-related tools available:

   - mcp__tiger__get_guide - Retrieve TimescaleDB guides and best practices
   - mcp__tiger__semantic_search_postgres_docs - Search PostgreSQL documentation
   - mcp__tiger__semantic_search_tiger_docs - Search Tiger Cloud and TimescaleDB documentation
   - mcp__tiger__tiger_service_create - Create new database services
   - mcp__tiger__tiger_service_list - List all database services
   - mcp__tiger__tiger_service_show - Show detailed service information
   - mcp__tiger__tiger_service_update_password - Update service passwords

   Is there something specific you'd like to do with the Tiger Data MCP server?
   ```

1. **Ask a basic question about your services**
   ```shell
   > can you list my active services please
   ```
   You see something like:
   ```shell
   ⏺ You have 3 active services, all with the same name "my-first-service":

   1. tgrservice1 - READY - TimescaleDB - us-east-1 - 1 CPU/4GB - Created: 2025-09-30
   2. tgrservice1 - READY - TimescaleDB - us-east-1 - 1 CPU/4GB - Created: 2025-09-30
   3. tgrservice1 - READY - TimescaleDB - us-east-1 - 1 CPU/4GB - Created: 2025-09-30
   ```

1. **Manage your services without having to learn how to**

   For example:
   ```shell
   Can you create a new ai service called bob with a replica for me please
   ```
   You see something like:
   ```shell
    tiger - Create Database Service (MCP)(name: "bob", addons: ["ai"], replicas: 1, wait: false)
   ```

1. **Find best practice for things you need to do**

   For example:
   ```shell
   >  I need to migrate a database with 25 million rows of data to Tiger Cloud,
   please tell me what service optimizations i should do before migration.

   ```

   You see something like:

   ```shell
   ⏺ For migrating a database with 25 million rows to Tiger Cloud, here are the key service optimizations to do before migration:

   Pre-Migration Service Optimizations

   1. Right-size your compute resources:
     - For 25M rows, start with at least 2 CPU/8GB or 4 CPU/16GB
     - You can scale up during migration if needed, then scale down after

   2. Enable the appropriate addons:
     - Use time-series addon if you are working with time-series data
     - Use ai addon if you need vector/AI capabilities

   3. Choose optimal region:
     - Select the region closest to your current database to minimize network transfer time
     - This significantly reduces migration duration

   4. Consider enhanced storage (Enterprise only):
     - If you expect high write throughput during migration, enhanced storage provides better IOPS (up to 32,000)

   5. Skip replicas initially:
     - Start with 0 replicas during migration
     - Add replicas after migration completes to avoid duplicate data copying

   6. Plan for storage space:
     - Estimate your uncompressed data size
     - Allocate sufficient high-performance storage (scales automatically but good to plan)
     - You can enable tiered storage later for older data

   During Migration Best Practices

   Migration method selection:
     - 25M rows: Use schema-then-data migration (migrate schema first, then copy data in chunks)
     - Allows restart if a copy operation fails
     - Better for databases >10GB

   Database configuration to set AFTER migration:
     - Set up compression policies (once data is loaded)
     - Configure retention policies
     - Create continuous aggregates
     - Enable tiered storage for cost optimization

   Connection settings:
     - Use the pooled connection endpoint for better performance during bulk inserts
     - Disable unnecessary constraints/indexes during migration, re-enable after

   Would you like me to help you create an optimized service for this migration?
   ```

That beats working. Let the Tiger MCP Server do it all for you.

## Manually configure the Tiger MCP Server

If your MCP client is not supported by `tiger mcp install`, follow the client's instructions to install
MCP servers. For example, many clients use a JSON file like the following that use `tiger mcp start` to
start Tiger Model Context Protocol Server:

```json
{
   "mcpServers": {
      "tiger": {
         "command": "tiger",
         "args": [
            "mcp",
            "start"
         ]
      }
   }
}
```

## Tiger Model Context Protocol Server commands

Tiger Model Context Protocol Server exposes the following MCP tools to your AI Assistant:

| Command                  | Parameter           | Required | Description                                                                                                                                                                                                                                                                              |
|--------------------------|---------------------|----------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `service_list`           | -                   | -        | Returns a list of the services in the current project.                                                                                                                                                                                                                     |
| `service_get`            | -                   | -        | Returns detailed information about a service.                                                                                                                                                                                                                                     |
|                          | `service_id`        | ✓        | The unique identifier of the service (10-character alphanumeric string).                                                                                                                                                                                                          |
|                          | `with_password`     | -        | Set to `true` to include the password in the response and connection string. <br/> **WARNING**: never do this unless the user explicitly requests the password.                                                                                                                          |
| `service_create`         | -                   | -        | Create a new service in Tiger Cloud. <br/> **WARNING**: creates billable resources.                                                                                                                                                                                               |
|                          | `name`              | -        | Set the human-readable name of up to 128 characters for this service.                                                                                                                                                                                                              |
|                          | `addons`            | -        | Set the array of [addons][create-service] to enable for the service. Options: <ul><li>`time-series`: enables TimescaleDB</li><li>`ai`: enables the AI and vector extensions</li></ul> Set an empty array for Postgres-only.                                                          |
|                          | `region`            | -        | Set the [AWS region][cloud-regions] to deploy this service in.                                                                                                                                                                                                                    |
|                          | `cpu_memory`        | -        | CPU and memory allocation combination. <br /> Available configurations are: <ul><li>shared/shared</li><li>0.5 CPU/2 GB</li><li>1 CPU/4 GB</li><li>2 CPU/8 GB</li><li>4 CPU/16 GB</li><li>8 CPU/32 GB</li><li>16 CPU/64 GB</li><li>32 CPU/128 GB</li></ul>                                |
|                          | `replicas`          | -        | Set the number of [high-availability replicas][readreplica] for fault tolerance.                                                                                                                                                                                                         |
|                          | `wait`              | -        | Set to `true` to wait for service to be fully ready before returning.                                                                                                                                                                                                             |
|                          | `timeout_minutes`   | -        | Set the timeout in minutes to wait for service to be ready. Only used when `wait=true`. Default: 30 minutes                                                                                                                                                                       |
|                          | `set_default`       | -        | By default, the new service is the default for following commands in CLI. Set to `false` to keep the previous service as the default.                                                                                                                               |
|                          | `with_password`     | -        | Set to `true` to include the password for this service in response and connection string. <br/> **WARNING**: never set to `true` unless user explicitly requests the password.                                                                                                    |
| `service_update_password` | -                   | -        | Update the password for the `tsdbadmin` for this service. The password change takes effect immediately and may terminate existing connections.                                                                                                                                    |
|                          | `service_id` | ✓        | The unique identifier of the service you want to update the password for.                                                                                                                                                                                                         |
|                          | `password`          | ✓        | The new password for the `tsdbadmin` user.                                                                                                                                                                                                                                               |
| `db_execute_query`                       | -                   | -        | Execute a single SQL query against a service. This command returns column metadata, result rows, affected row count, and execution time. Multi-statement queries are not supported.  <br/> **WARNING**: can execute destructive SQL including INSERT, UPDATE, DELETE, and DDL commands. |
|                          | `service_id` | ✓        | The unique identifier of the service. Use `tiger_service_list` to find service IDs.                                                                                                                                                                                        |
|                          | `query`             | ✓        | The SQL query to execute. Single statement queries are supported.                                                                                                                                                                                                                        |
|                          | `parameters`        | -        | Query parameters for parameterized queries. Values are substituted for the `$n` placeholders in the query.                                                                                                                                                                               |
|                          | `timeout_seconds`   | -        | The query timeout in seconds. Default: `30`.                                                                                                                                                                                                                                               |
|                          | `role`              | -        | The service role/username to connect as. Default: `tsdbadmin`.                                                                                                                                                                                                                    |
|                          | `pooled`            | -        | Use [connection pooling][Connection pooling]. This is only available if you have already enabled it for the service. Default: `false`.                                                                                                                                             |

## Tiger CLI commands for Tiger MCP Server

You can use the following Tiger CLI commands to run Tiger MCP Server:

Usage: `tiger mcp [subcommand] --<flags>`

| Command | Subcommand         | Description                                                                                                                                                                                                                                                                                                                                                                          |
|---------|--------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| mcp     |                    | Manage the Tiger Model Context Protocol Server                                                                                                                                                                                                                                                                                                                                                                 |
|         | install `[client]` | Install and configure Tiger MCP Server for a specific client installed on your developer device. <br/>Supported clients are: `claude-code`, `cursor`, `windsurf`, `codex`, `gemini/gemini-cli`, `vscode/code/vs-code`. <br/> Flags: <ul><li>`--no-backup`: do not back up the existing configuration</li><li>`--config-path`: open the configuration file at a specific location</li></ul> |
|         | start              | Start the Tiger MCP Server. This is the same as `tiger mcp start stdio`                                                                                                                                                                                                                                                                                                                    |
|         | start stdio        | Start the Tiger MCP Server with stdio transport                                                                                                                                                                                                                                                                                                                                            |
|         | start http         | Start the Tiger MCP Server with HTTP transport. This option is for users who wish to access Tiger Model Context Protocol Server without using stdio. For example, your AI Assistant does not support stdio, or you do not want to run CLI on your device. <br/>  Flags are: <ul><li>`--port <port number>`: the default is `8000`</li><li>`--host <hostname>`: the default is `localhost`</li></ul>           |

## Global flags

You can use the following Tiger CLI global flags when you run the Tiger MCP Server:

| Flag                          | Default           | Description                                                                 |
|-------------------------------|-------------------|-----------------------------------------------------------------------------|
| `--analytics`                 | `true`            | Set to `false` to disable usage analytics                                   |
| `--color `                    | `true`            | Set to `false` to disable colored output                                    |
| `--config-dir` string         | `.config/tiger`  | Set the directory that holds `config.yaml`                                  |
| `--debug`                     | No debugging      | Enable debug logging                                                        |
| `--help`                      | -                 | Print help about the current command. For example, `tiger service --help`   |
| `--password-storage` string   | keyring           | Set the password storage method. Options are `keyring`, `pgpass`, or `none` |
| `--service-id` string         | -                 | Set the Tiger Cloud service to manage                                             |
| ` --skip-update-check `       | -                 | Do  not check if a new version of Tiger CLI is available|


===== PAGE: https://docs.tigerdata.com/ai/tiger-eon/ =====

# Aggregate organizational data with AI agents



Your business already has the answers in Slack threads, GitHub pull requests, Linear tasks, your own docs, Salesforce
service tickets, anywhere you store data. However, those answers are scattered, hard to find, and often forgotten.
Tiger Eon automatically integrates Tiger Agents for Work with your organizational data so you can let AI Assistants analyze your
company data and give you the answers you need. For example:
- What did we ship last week?
- What's blocking the release?
- Summarize the latest GitHub pull requests.

Eon responds instantly, pulling from the tools you already use. No new UI, no new workflow, just answers in Slack.

![Query Tiger Agent](https://assets.timescale.com/docs/images/tiger-eon-big-question.png)

Tiger Eon:

- **Unlocks hidden value**: your data in Slack, GitHub, and Linear already contains the insights you need. Eon makes them accessible.
- **Enables faster decisions**: no need to search or ask around, you get answers in seconds.
- **Is easy to use**: Eon runs a Tiger Agent and MCP servers statelessly in lightweight Docker containers.
- **Integrates seamlessly with Tiger Cloud**: Eon uses a Tiger Cloud service so you securely and reliably store
    your company data. Prefer to self-host? Use a [Postgres instance with TimescaleDB][install-self-hosted].

Tiger Eon's real-time ingestion system connects to Slack and captures everything: every message, reaction, edit, and
channel update. It can also process historical Slack exports. Eon had instant access to years
of institutional knowledge from the very beginning.

All of this data is stored in your Tiger Cloud service as time-series data: conversations are events unfolding over time,
and Tiger Cloud is purpose-built for precisely this. Your data is optimized by:

- Automatically partitioning the data into 7-day chunks for efficient queries
- Compressing the data after 45 days to save space
- Segmenting by channel for faster retrieval

When someone asks Eon a question, it uses simple SQL to instantly retrieve the full thread context, related
conversations, and historical decisions. No rate limits. No API quotas. Just direct access to your data.

This page shows you how to install and run Eon.

## Prerequisites

To follow the procedure on this page you need to:

* Create a [Tiger Data account][create-account].

   This procedure also works for [self-hosted TimescaleDB][enable-timescaledb].

- [Install Docker][install-docker] on your developer device
- Install [Tiger CLI][tiger-cli]
- Have rights to create an [Anthropic API key][claude-api-key]
- Optionally:
  - Have rights to create a [GitHub token][github-token]
  - Have rights to create a [Logfire token][logfire-token]
  - Have rights to create a [Linear token][linear-token]

## Interactive setup

Tiger Eon is a production-ready repository running [Tiger CLI][tiger-cli] and [Tiger Agents for Work][tiger-agents] that creates
and runs the following components for you:

- An ingest Slack app that consumes all messages and reactions from public channels in your Slack workspace
- A [Tiger Agent][tiger-agents] that analyzes your company data for you
- A Tiger Cloud service instance that stores data from the Slack apps
- MCP servers that connect data sources to Eon
- A listener Slack app that passes questions to the Tiger Agent when you @tag it in a public channel, and returns the
  AI analysis on your data

All local components are run in lightweight Docker containers via Docker Compose.

This section shows you how to run the Eon setup to configure Eon to connect to your Slack app, and give it  access to your
data and analytics stored in Tiger Cloud.

1. **Install Tiger Eon to manage and run your AI-powered Slack bots**

    In a local folder, run the following command from the terminal:
    ```shell
    git clone git@github.com:timescale/tiger-eon.git
    ```

1. **Start the Eon setup**

   ```shell
   cd tiger-eon
   ./setup-tiger-eon.sh
   ```
   You see a summary of the setup procedure. Type `y` and press `Enter`.

1. **Create the Tiger Cloud service to use with Eon**

    You see `Do you want to use a free tier Tiger Cloud Database? [y/N]:`. Press `Y` to create a free
    Tiger Cloud service.

    Eon opens the Tiger Cloud authentication page in your browser. Click `Authorize`. Eon creates a
    Tiger Cloud service called [tiger-eon][services-portal] and stores the credentials in your local keychain.

    If you press `N`, the Eon setup creates and runs TimescaleDB in a local Docker container.

1. **Create the ingest Slack app**

   1. In the terminal, name your ingest Slack app:

      1. Eon proposes to create an ingest app called `tiger-slack-ingest`, press `Enter`.
      1. Do the same for the App description.

      Eon opens `Your Apps` in https://api.slack.com/apps/.

   1. Start configuring your ingest app in Slack:

      In the Slack `Your Apps` page:
      1. Click `Create New App`, click `From an manifest`, then select a workspace.
      1. Click `Next`. Slack opens `Create app from manifest`.

   1. Add the Slack app manifest:
      1. In terminal press `Enter`. The setup prints the Slack app manifest to terminal and adds it to your clipboard.
      1. In the Slack `Create app from manifest` window, paste the manifest.
      1. Click `Next`, then click `Create`.

   1. Configure an app-level token:

       1. In your app settings, go to `Basic Information`.
       1. Scroll to `App-Level Tokens`.
       1. Click `Generate Token and Scopes`.
       1. Add a `Token Name`, then click `Add Scope` add `connections:write`, then click `Generate`.
       1. Copy the `xapp-*` token and click `Done`.
       1. In the terminal, paste the token, then press `Enter`.

   1. Configure a bot user OAuth token:

       1. In your app settings, under `Features`, click `App Home`.
       1. Scroll down, then enable `Allow users to send Slash commands and messages from the messages tab`.
       1. In your app settings, under `Settings`, click `Install App`.
       1. Click `Install to <workspace name>`, then click `Allow`.
       1. Copy the `xoxb-` Bot User OAuth Token locally.
       1. In the terminal, paste the token, then press `Enter`.

1. **Create the Eon Slack app**

    Follow the same procedure as you did for the ingest Slack app.

1. **Integrate Eon with Anthropic**

   The Eon setup opens https://console.anthropic.com/settings/keys. Create a Claude Code key, then
   paste it in the terminal.

1. **Integrate Eon with Logfire**

   If you would like to integrate logfire with Eon, paste your token and press `Enter`. If not, press `Enter`.

1. **Integrate Eon with GitHub**

    The Eon setup asks if you would like to `Enable github MCP server?". For Eon to answer questions
    about the activity in your Github organization`. Press `y` to integrate with GitHub.

1. **Integrate Eon with Linear**

   The Eon setup asks if you would like to `Enable linear MCP server? [y/N]:`. Press `y` to integrate with Linear.

1. **Give Eon access to private repositories**

   1. The setup asks if you would like to include access to private repositories. Press `y`.
   1. Follow the GitHub token creation process.
   1. In the Eon setup add your organization name, then paste the GitHub token.

   The setup sets up a new Tiger Cloud service for you called `tiger-eon`, then starts Eon in Docker.

   ![Eon running in Docker](https://assets.timescale.com/docs/images/tiger-eon-docker-services.png)

You have created:
* The Eon ingest and chat apps in Slack
* A private MCP server connecting Eon to your data in GitHub
* A Tiger Cloud service that securely stores the data used by Eon

## Integrate Eon in your Slack workspace

To enable your AI Assistant to analyze your data for you when you ask a question, open a public channel,
invite `@eon` to join, then ask a question:

![Eon running in Docker](https://assets.timescale.com/docs/images/tiger-eon-slack-channel-add.png)


===== PAGE: https://docs.tigerdata.com/ai/tiger-agents-for-work/ =====

# Integrate a slack-native AI agent


Tiger Agents for Work is a Slack-native AI agent that you use to unify the knowledge in your company. This includes your Slack
history, docs, GitHub repositories, Salesforce and so on. You use your Tiger Agent to get instant answers for real
business, technical, and operations questions in your Slack channels.

![Query Tiger Agent](https://assets.timescale.com/docs/images/tiger-agent/query-in-slack.png)

Tiger Agents for Work can handle concurrent conversations with enterprise-grade reliability. They have the following features:

- **Durable and atomic event handling**: Postgres-backed event claiming ensures exactly-once processing, even under high concurrency and failure conditions
- **Bounded concurrency**: fixed worker pools prevent resource exhaustion while maintaining predictable performance under load
- **Immediate event processing**: Tiger Agents for Work provide real-time responsiveness. Events are processed within milliseconds of arrival rather than waiting for polling cycles
- **Resilient retry logic**: automatic retry with visibility thresholds, plus stuck or expired event cleanup
- **Horizontal scalability**: run multiple Tiger Agent instances simultaneously with coordinated work distribution across all instances
- **AI-Powered Responses**: use the AI model of your choice, you can also integrate with MCP servers
- **Extensible architecture**: zero code integration for basic agents. For more specialized use cases, easily customize your agent using [Jinja templates][jinja-templates]
- **Complete observability**: detailed tracing of event flow, worker activity, and database operations with full [Logfire][logfire] instrumentation

This page shows you how to install the Tiger Agent CLI, connect to the Tiger Data MCP server, and customize prompts for
your specific needs.

## Prerequisites

To follow the procedure on this page you need to:

* Create a [Tiger Data account][create-account].

   This procedure also works for [self-hosted TimescaleDB][enable-timescaledb].

* Install the [uv package manager][uv-install]
* Get an [Anthropic API key][claude-api-key]
* Optional: get a [Logfire token][logfire]

## Create a Slack app

Before installing Tiger Agents for Work, you need to create a Slack app that the Tiger Agent will connect to. This app
provides the security tokens for Slack integration with your Tiger Agent:

1. **Create a manifest for your Slack App**

   1. In a temporary directory, download the Tiger Agent Slack manifest template:

      ```bash
      curl -O https://raw.githubusercontent.com/timescale/tiger-agents-for-work/main/slack-manifest.json
      ```

   1. Edit `slack-manifest.json` and customize your name and description of your Slack App. For example:

      ```json
      "display_information": {
        "name": "Tiger Agent",
        "description": "Tiger AI Agent helps you easily access your business information, and tune your Tiger services",
        "background_color": "#000000"
      },
      "features": {
        "bot_user": {
          "display_name": "Tiger Agent",
          "always_online": true
        }
      },
      ```

   1. Copy the contents of `slack-manifest.json` to the clipboard:

      ```shell
      cat slack-manifest.json| pbcopy
      ```

1. **Create the Slack app**

    1. Go to [api.slack.com/apps](https://api.slack.com/apps).
    1. Click `Create New App`.
    1. Select `From a manifest`.
    1. Choose your workspace, then click `Next`.
    1. Paste the contents of `slack-manifest.json` and click `Next`.
    1. Click `Create`.
1. **Generate an app-level token**

    1. In your app settings, go to `Basic Information`.
    1. Scroll to `App-Level Tokens`.
    1. Click `Generate Token and Scopes`.
    1. Add a `Token Name`, then click `Add Scope`, add `connections:write` then click `Generate`.
    1. Copy the `xapp-*` token locally and click `Done`.

1. **Install your app to a Slack workspace**

    1. In the sidebar, under `Settings`, click `Install App`.
    1. Click `Install to <workspace name>`, then click `Allow`.
    1. Copy the `xoxb-` Bot User OAuth Token locally.

You have created a Slack app and obtained the necessary tokens for Tiger Agent integration.


## Install and configure your Tiger Agent instance

Tiger Agents for Work are a production-ready library and CLI written in Python that you use to create Slack-native AI agents.
This section shows you how to configure a Tiger Agent to connect to your Slack app, and give it access to your
data and analytics stored in Tiger Cloud.

1. **Create a project directory**

   ```bash
   mkdir my-tiger-agent
   cd my-tiger-agent
   ```

1. **Create a Tiger Agent environment with your Slack, AI Assistant, and database configuration**

   1. Download `.env.sample` to a local `.env` file:
     ```shell
     curl -L -o .env https://raw.githubusercontent.com/timescale/tiger-agent/refs/heads/main/.env.sample
     ```
   1. In `.env`, add your Slack tokens and Anthropic API key:

     ```bash
     SLACK_APP_TOKEN=xapp-your-app-token
     SLACK_BOT_TOKEN=xoxb-your-bot-token

     ANTHROPIC_API_KEY=sk-ant-your-api-key

     LOGFIRE_TOKEN=your-logfire-token
     ```
   1. Add the [connection details][connection-info] for the Tiger Cloud service you are using for this Tiger Agent:
     ```bash
     PGHOST=<host>
     PGDATABASE=tsdb
     PGPORT=<port>
     PGUSER=tsdbadmin
     PGPASSWORD=<password>
     ```
   1. Save and close `.env`.

1. **Add the default Tiger Agent prompts to your project**
     ```bash
     mkdir prompts
     curl -L -o prompts/system_prompt.md https://raw.githubusercontent.com/timescale/tiger-agent/refs/heads/main/prompts/system_prompt.md
     curl -L -o prompts/user_prompt.md https://raw.githubusercontent.com/timescale/tiger-agent/refs/heads/main/prompts/user_prompt.md
     ```

1. **Install Tiger Agents for Work to manage and run your AI-powered Slack bots**

   1. Install the Tiger Agent CLI using uv.

      ```bash
      uv tool install --from git+https://github.com/timescale/tiger-agents-for-work.git tiger-agent
      ```
      `tiger-agent` is installed in `~/.local/bin/tiger-agent`. If necessary, add this folder to your `PATH`.

   1. Verify the installation.

      ```bash
      tiger-agent --help
      ```

      You see the Tiger Agent CLI help output with the available commands and options.


1. **Connect your Tiger Agent with Slack**

    1. Run your Tiger Agent:
       ```bash
       tiger-agent run --prompts prompts/  --env .env
       ```
       If you open the explorer in [Tiger Cloud Console][portal-ops-mode], you can see the tables used by your Tiger Agent.

    1. In Slack, open a public channel app and ask Tiger Agent a couple of questions. You see the response in your
       public channel and log messages in the terminal.

      ![Query Tiger Agent](https://assets.timescale.com/docs/images/tiger-agent/query-in-terminal.png)

## Add information from MCP servers to your Tiger Agent

To increase the amount of specialized information your AI Assistant can use, you can add MCP servers supplying data
your users need. For example, to add the Tiger Data MCP server to your Tiger Agent:

1. **Copy the example `mcp_config.json` to your project**

   In `my-tiger-agent`, run the following command:

       ```bash
        curl -L -o mcp_config.json https://raw.githubusercontent.com/timescale/tiger-agent/refs/heads/main/examples/mcp_config.json
        ```

1. **Configure your Tiger Agent to connect to the most useful MCP servers for your organization**

    For example, to add the Tiger Data documentation MCP server to your Tiger Agent, update the docs entry to the
    following:
    ```json
    "docs": {
      "tool_prefix": "docs",
      "url": "https://mcp.tigerdata.com/docs",
      "allow_sampling": false
    },
    ```
    To avoid errors, delete all entries in `mcp_config.json` with invalid URLs. For example the `github` entry with `http://github-mcp-server/mcp`.

1. **Restart your Tiger Agent**
   ```bash
   tiger-agent run --prompts prompts/ --mcp-config mcp_config.json
   ```

You have configured your Tiger Agent to connect to the Tiger MCP Server. For more information,
see [MCP Server Configuration][mcp-configuration-docs].

## Customize prompts for personalization

Tiger Agents for Work uses Jinja2 templates for dynamic, context-aware prompt generation. This system allows for sophisticated
prompts that adapt to conversation context, user preferences, and event metadata. Tiger Agents for Work uses the following
templates:

- `system_prompt.md`: defines the AI Assistant's role, capabilities, and behavior patterns. This template sets the
   foundation for the way your Tiger Agent will respond and interact.
- `user_prompt.md`: formats the user's request with relevant context, providing the AI Assistant with the
   information necessary to generate an appropriate response.

To change the way your Tiger Agents interact with users in your Slack app:

1. **Update the prompt**

   For example, in `prompts/system_prompt.md`, add another item in the `Response Protocol` section to fine tune
   the behavior of your Tiger Agents. For example:
   ```shell
   5. Be snarky but vaguely amusing
   ```

1. **Test your configuration**

   Run Tiger Agent with your custom prompt:

   ```bash
   tiger-agent run --mcp-config mcp_config.json --prompts prompts/
   ```

For more information, see [Prompt tempates][prompt-templates].

## Advanced configuration options

For additional customization, you can modify the following Tiger Agent parameters:

* `--model`: change AI model (default: `anthropic:claude-sonnet-4-20250514`)
* `--num-workers`: adjust concurrent workers (default: `5`)
* `--max-attempts`: set retry attempts per event (default: `3`)

Example with custom settings:

```bash
tiger-agent run \
  --model claude-3-5-sonnet-latest \
  --mcp-config mcp_config.json \
  --prompts prompts/ \
  --num-workers 10 \
  --max-attempts 5
```

Your Tiger Agents are now configured with Tiger Data MCP server access and personalized prompts.


===== PAGE: https://docs.tigerdata.com/ai/key-vector-database-concepts-for-understanding-pgvector/ =====

# Key vector database concepts for understanding pgvector

<!-- vale Google.Headings = NO -->
<!-- vale Google.Headings = YES -->

## `Vector` data type provided by pgvector

Vectors inside of the database are stored in regular Postgres tables using `vector` columns. The `vector` column type is provided by the [pgvector](https://github.com/pgvector/pgvector) extension. A common way to store vectors is alongside the data they have indexed. For example, to store embeddings for documents, a common table structure is:

```sql
CREATE TABLE IF NOT EXISTS document_embedding  (
    id BIGINT PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
    document_id BIGINT FOREIGN KEY(document.id)
    metadata JSONB,
    contents TEXT,
    embedding VECTOR(1536)
)
```

This table contains a primary key, a foreign key to the document table, some metadata, the text being embedded (in the `contents` column), and the embedded vector.

This may seem like a bit of a weird design: why aren't the embeddings simply a separate column in the document table? The answer has to do with context length limits of embedding models and of LLMs. When embedding data, there is a limit to the length of content you can embed (for example, OpenAI's ada-002 has a limit of [8191 tokens](https://platform.openai.com/docs/guides/embeddings/embedding-models) ), and so, if you are embedding a long piece of text, you have to break it up into smaller chunks and embed each chunk individually. Therefore, when thinking about this at the database layer, there is usually a one-to-many relationship between the thing being embedded and the embeddings which is represented by a foreign key from the embedding to the thing.

Of course, if you do not want to store the original data in the database and you are just storing only the embeddings, that's totally fine too. Just omit the foreign key from the table. Another popular alternative is to put the foreign key into the metadata JSONB.

## Querying vectors using pgvector

The canonical query for vectors is for the closest query vectors to an embedding of the user's query. This is also known as finding the [K nearest neighbors](https://en.wikipedia.org/wiki/K-nearest_neighbors_algorithm).

In the example query below, `$1` is a parameter taking a query embedding, and the `<=>` operator calculates the distance between the query embedding and embedding vectors stored in the database (and returns a float value).

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```

The query above returns the 10 rows with the smallest distance between the query's embedding and the row's embedding. Of course, this being Postgres, you can add additional `WHERE` clauses (such as filters on the metadata), joins, etc.


### Vector distance types

The query shown above uses something called cosine distance (using the <=> operator) as a measure of how similar two embeddings are. But, there are multiple ways to quantify how far apart two vectors are from each other.



In practice, the choice of distance measure doesn't matters much and it is recommended to just stick with cosine distance for most applications.



#### Description of cosine distance, negative inner product, and Euclidean distance

Here's a succinct description of three common vector distance measures

- **Cosine distance a.k.a. angular distance**: This measures the cosine of the angle between two vectors. It's not a true "distance" in the mathematical sense but a similarity measure, where a smaller angle corresponds to a higher similarity. The cosine distance is particularly useful in high-dimensional spaces where the magnitude of the vectors (their length) is less important, such as in text analysis or information retrieval. It ranges from -1 (meaning exactly opposite) to 1 (exactly the same), with 0 typically indicating orthogonality (no similarity). See here for more on [cosine similarity](https://en.wikipedia.org/wiki/Cosine_similarity).

- **Negative inner product**: This is simply the negative of the inner product (also known as the dot product) of two vectors. The inner product measures vector similarity based on the vectors' magnitudes and the cosine of the angle between them. A higher inner product indicates greater similarity. However, it's important to note that, unlike cosine similarity, the magnitude of the vectors influences the inner product.

- **Euclidean distance**: This is the "ordinary" straight-line distance between two points in Euclidean space. In terms of vectors, it's the square root of the sum of the squared differences between corresponding elements of the vectors. This measure is sensitive to the magnitude of the vectors and is widely used in various fields such as clustering and nearest neighbor search.

Many embedding systems (for example OpenAI's ada-002) use vectors with length 1 (unit vectors). For those systems, the rankings (ordering) of all three measures is the same. In particular,
- The cosine distance is `1−dot product`.
- The negative inner product is `−dot product`.
- The Euclidean distance is related to the dot product, where the squared Euclidean distance is `2(1−dot product)`.

<!-- vale Google.Headings = NO -->
#### Recommended vector distance for use in Postgres
<!-- vale Google.Headings = YES -->

Using cosine distance, especially on unit vectors, is recommended. These recommendations are based on OpenAI's [recommendation](https://platform.openai.com/docs/guides/embeddings/which-distance-function-should-i-use) as well as the fact that the ranking of different distances on unit vectors is preserved.

## Vector search indexing (approximate nearest neighbor search)

In Postgres and other relational databases, indexing is a way to speed up queries. For vector data, indexes speed up the similarity search query shown above where you find the most similar embedding to some given query embedding. This problem is often referred to as finding the [K nearest neighbors](https://en.wikipedia.org/wiki/K-nearest_neighbors_algorithm).



The term "index" in the context of vector databases has multiple meanings. It can refer to both the storage mechanism for your data and the tool that enhances query efficiency. These docs use the latter meaning.



Finding the K nearest neighbors is not a new problem in Postgres, but existing techniques only work with low-dimensional data. These approaches cease to be effective when dealing with data larger than approximately 10 dimensions due to the "curse of dimensionality." Given that embeddings often consist of more than a thousand dimensions(OpenAI's are 1,536) new techniques had to be developed.

There are no known exact algorithms for efficiently searching in such high-dimensional spaces. Nevertheless, there are excellent approximate algorithms that fall into the category of approximate nearest neighbor algorithms.

<!-- vale Google.Colons = NO -->

There are 3 different indexing algorithms available as part of pgai on Tiger Cloud: StreamingDiskANN, HNSW, and ivfflat. The table below illustrates the high-level differences between these algorithms:

<!-- vale Google.Colons = YES -->

| Algorithm       | Build Speed | Query Speed | Need to rebuild after updates |
|------------------|-------------|-------------|-------------------------------|
| StreamingDiskANN | Fast        | Fastest     | No                            |
| HNSW    | Fast     | Fast      | No                            |
| ivfflat | Fastest     | Slowest     | Yes                           |


See the [performance benchmarks](https://www.timescale.com/blog/how-we-made-postgresql-the-best-vector-database) for details on how the each index performs on a dataset of 1 million OpenAI embeddings.

## Recommended index types

For most applications, the StreamingDiskANN index is recommended.


===== PAGE: https://docs.tigerdata.com/ai/sql-interface-for-pgvector-and-timescale-vector/ =====

# SQL inteface for pgvector and pgvectorscale

## Installing the pgvector and pgvectorscale extensions

If not already installed, install the `vector` and `vectorscale` extensions on your Tiger Data database.

```sql
CREATE EXTENSION IF NOT EXISTS vector;
CREATE EXTENSION IF NOT EXISTS vectorscale;
```

## Creating the table for storing embeddings using pgvector

Vectors inside of the database are stored in regular Postgres tables using `vector` columns. The `vector` column type is provided by the pgvector extension. A common way to store vectors is alongside the data they are embedding. For example, to store embeddings for documents, a common table structure is:

```sql
CREATE TABLE IF NOT EXISTS document_embedding  (
    id BIGINT PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
    document_id BIGINT FOREIGN KEY(document.id)
    metadata JSONB,
    contents TEXT,
    embedding VECTOR(1536)
)
```

This table contains a primary key, a foreign key to the document table, some metadata, the text being embedded (in the `contents` column) and the embedded vector.

You may ask why not just add an embedding column to the document table? The answer is that there is a limit on the length of text an embedding can encode and so there needs to be a one-to-many relationship between the full document and its embeddings.

The above table is just an illustration, it's totally fine to have a table without a foreign key and/or without a metadata column. The important thing is to have a column with the data being embedded and the vector in the same row, enabling you to return the raw data for a given similarity search query

The vector type can specify an optional number of dimensions (1,538) in the example above). If specified, it enforces the constraint that all vectors in the column have that number of dimensions. A plain `VECTOR` (without specifying the number of dimensions) column is also possible and allows a variable number of dimensions.

## Query the vector embeddings

The canonical query is:

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```

Which returns the 10 rows whose distance is the smallest. The distance function used here is cosine distance (specified by using the `<=>` operator). Other distance functions are available, see the [discussion][distance-functions].

The available distance types and their operators are:

| Distance type          | Operator      |
|------------------------|---------------|
| Cosine/Angular         | `<=>`           |
| Euclidean              | `<->`           |
| Negative inner product | `<#>`           |



If you are using an index, you need to make sure that the distance function used in index creation is the same one used during query (see below). This is important because if you create your index with one distance function but query with another, your index cannot be used to speed up the query.




## Indexing the vector data using indexes provided by pgvector and pgvectorscale

Indexing helps speed up similarity queries of the basic form:

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```

The key part is that the `ORDER BY` contains a distance measure against a constant or a pseudo-constant.

Note that if performing a query without an index, you always get an exact result, but the query is slow (it has to read all of the data you store for every query). With an index, your queries are an order-of-magnitude faster, but the results are approximate (because there are no known indexing techniques that are exact see [here for more][vector-search-indexing]).

<!-- vale Google.Colons = NO -->
Nevertheless, there are excellent approximate algorithms. There are 3 different indexing algorithms available on TimescaleDB: StreamingDiskANN, HNSW, and ivfflat. Below is the trade-offs between these algorithms:
<!-- vale Google.Colons = Yes -->

| Algorithm       | Build Speed | Query Speed | Need to rebuild after updates |
|------------------|-------------|-------------|-------------------------------|
| StreamingDiskANN | Fast        | Fastest     | No                            |
| HNSW    | Fast     | Fast      | No                            |
| ivfflat | Fastest     | Slowest     | Yes                           |


You can see [benchmarks](https://www.timescale.com/blog/how-we-made-postgresql-the-best-vector-database/) in the blog.

For most use cases, the StreamingDiskANN index is recommended.

Each of these indexes has a set of build-time options for controlling the speed/accuracy trade-off when creating the index and an additional query-time option for controlling accuracy during a particular query.

You can see the details of each index below.

### StreamingDiskANN index


The StreamingDiskANN index is a graph-based algorithm that was inspired by the [DiskANN](https://github.com/microsoft/DiskANN) algorithm.
You can read more about it in
[How We Made Postgres as Fast as Pinecone for Vector Data](https://www.timescale.com/blog/how-we-made-postgresql-as-fast-as-pinecone-for-vector-data).

To create an index named `document_embedding_idx` on table `document_embedding` having a vector column named `embedding`, with cosine distance metric, run:
```sql
CREATE INDEX document_embedding_cos_idx ON document_embedding
USING diskann (embedding vector_cosine_ops);
```

Since this index uses cosine distance, you should use the `<=>` operator in your queries.  StreamingDiskANN also supports L2 distance:
```sql
CREATE INDEX document_embedding_l2_idx ON document_embedding
USING diskann (embedding vector_l2_ops);
```
For L2 distance, use the `<->` operator in queries.

These examples create the index with smart defaults for all parameters not listed. These should be the right values for most cases. But if you want to delve deeper, the available parameters are below.

#### StreamingDiskANN index build-time parameters

These parameters can be set when an index is created.

| Parameter name   | Description                                                                                                                                                    | Default value |
|------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|
| `storage_layout` | `memory_optimized` which uses SBQ to compress vector data or `plain` which stores data uncompressed | memory_optimized
| `num_neighbors`    | Sets the maximum number of neighbors per node. Higher values increase accuracy but make the graph traversal slower.                                           | 50            |
| `search_list_size` | This is the S parameter used in the greedy search algorithm used during construction. Higher values improve graph quality at the cost of slower index builds. | 100           |
| `max_alpha`        | Is the alpha parameter in the algorithm. Higher values improve graph quality at the cost of slower index builds.                                              | 1.2           |
| `num_dimensions` | The number of dimensions to index. By default, all dimensions are indexed. But you can also index less dimensions to make use of [Matryoshka embeddings](https://huggingface.co/blog/matryoshka) | 0 (all dimensions)
| `num_bits_per_dimension` | Number of bits used to encode each dimension when using SBQ | 2 for less than 900 dimensions, 1 otherwise

An example of how to set the `num_neighbors` parameter is:

```sql
CREATE INDEX document_embedding_idx ON document_embedding
USING diskann (embedding) WITH(num_neighbors=50);
```

<!---
TODO: Add PQ options
-->


#### StreamingDiskANN query-time parameters

You can also set two parameters to control the accuracy vs. query speed trade-off at query time. We suggest adjusting `diskann.query_rescore` to fine-tune accuracy.

| Parameter name   | Description                                                                                                                                                    | Default value |
|------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|
| `diskann.query_search_list_size` | The number of additional candidates considered during the graph search. | 100
| `diskann.query_rescore` | The number of elements rescored (0 to disable rescoring) | 50

You can set the value by using `SET` before executing a query. For example:

```sql
SET diskann.query_rescore = 400;
```

Note the [SET command](https://www.postgresql.org/docs/current/sql-set.html) applies to the entire session (database connection) from the point of execution. You can use a transaction-local variant using `LOCAL` which will
be reset after the end of the transaction:

```sql
BEGIN;
SET LOCAL diskann.query_search_list_size= 10;
SELECT * FROM document_embedding ORDER BY embedding <=> $1 LIMIT 10
COMMIT;
```

#### StreamingDiskANN index-supported queries

You need to use the cosine-distance embedding measure (`<=>`) in your `ORDER BY` clause. A canonical query would be:

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```

### pgvector HNSW

Pgvector provides a graph-based indexing algorithm based on the popular [HNSW algorithm](https://arxiv.org/abs/1603.09320).

To create an index named `document_embedding_idx` on table `document_embedding` having a vector column named `embedding`, run:
```sql
CREATE INDEX document_embedding_idx ON document_embedding
USING hnsw(embedding vector_cosine_ops);
```

This command creates an index for cosine-distance queries because of `vector_cosine_ops`. There are also "ops" classes for Euclidean distance and negative inner product:

| Distance type          | Query operator | Index ops class  |
|------------------------|----------------|-------------------|
| Cosine / Angular       | `<=>`            | `vector_cosine_ops` |
| Euclidean / L2         | `<->`            | `vector_ip_ops`     |
| Negative inner product | `<#>`            | `vector_l2_ops`     |

Pgvector HNSW also includes several index build-time and query-time parameters.

#### pgvector HNSW index build-time parameters

These parameters can be set at index build time:

| Parameter name   | Description                                                                                                                                                    | Default value |
|------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|
| `m`    | Represents the maximum number of connections per layer. Think of these connections as edges created for each node during graph construction. Increasing m increases accuracy but also increases index build time and size.                                       | 16            |
| `ef_construction` | Represents the size of the dynamic candidate list for constructing the graph. It influences the trade-off between index quality and construction speed. Increasing `ef_construction` enables more accurate search results at the expense of lengthier index build times. | 64 |

An example of how to set the m parameter is:

```sql
CREATE INDEX document_embedding_idx ON document_embedding
USING hnsw(embedding vector_cosine_ops) WITH (m = 20);
```

#### pgvector HNSW query-time parameters

You can also set a parameter to control the accuracy vs. query speed trade-off at query time. The parameter is called `hnsw.ef_search`. This parameter specifies the size of the dynamic candidate list used during search. Defaults to 40. Higher values improve query accuracy while making the query slower.

You can set the value by running:

```sql
SET hnsw.ef_search = 100;
```

Before executing the query, note the [SET command](https://www.postgresql.org/docs/current/sql-set.html) applies to the entire session (database connection) from the point of execution. You can use a transaction-local variant using `LOCAL`:

```sql
BEGIN;
SET LOCAL hnsw.ef_search = 100;
SELECT * FROM document_embedding ORDER BY embedding <=> $1 LIMIT 10
COMMIT;
```

#### pgvector HNSW index-supported queries

You need to use the distance operator (`<=>`, `<->`, or `<#>`) matching the ops class you used during index creation in your `ORDER BY` clause. A canonical query would be:

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```

### pgvector ivfflat

Pgvector provides a clustering-based indexing algorithm. The [blog post](https://www.timescale.com/blog/nearest-neighbor-indexes-what-are-ivfflat-indexes-in-pgvector-and-how-do-they-work) describes how it works in detail. It provides the fastest index-build speed but the slowest query speeds of any indexing algorithm.

To create an index named `document_embedding_idx` on table `document_embedding` having a vector column named `embedding`, run:
```sql
CREATE INDEX document_embedding_idx ON document_embedding
USING ivfflat(embedding vector_cosine_ops) WITH (lists = 100);
```

This command creates an index for cosine-distance queries because of `vector_cosine_ops`. There are also "ops" classes for Euclidean distance and negative inner product:

| Distance type          | Query operator | Index ops class  |
|------------------------|----------------|-------------------|
| Cosine / Angular       | `<=>`            | `vector_cosine_ops` |
| Euclidean / L2         | `<->`            | `vector_ip_ops`     |
| Negative inner product | `<#>`            | `vector_l2_ops`     |

Note: *ivfflat should never be created on empty tables* because it needs to cluster data, and that only happens when an index is first created, not when new rows are inserted or modified. Also, if your table undergoes a lot of modifications, you need to rebuild this index occasionally to maintain good accuracy. See the [blog post](https://www.timescale.com/blog/nearest-neighbor-indexes-what-are-ivfflat-indexes-in-pgvector-and-how-do-they-work) for details.

Pgvector ivfflat has a `lists` index parameter that should be set. See the next section.

#### pgvector ivfflat index build-time parameters

Pgvector has a `lists` parameter that should be set as follows:
For datasets with less than one million rows, use lists =  rows / 1000.
For datasets with more than one million rows, use lists = sqrt(rows).
It is generally advisable to have at least 10 clusters.


You can use the following code to simplify creating ivfflat indexes:
```python
def create_ivfflat_index(conn, table_name, column_name, query_operator="<=>"):
    index_method = "invalid"
    if query_operator == "<->":
        index_method = "vector_l2_ops"
    elif query_operator == "<#>":
        index_method = "vector_ip_ops"
    elif query_operator == "<=>":
        index_method = "vector_cosine_ops"
    else:
        raise ValueError(f"unrecognized operator {query_operator}")

    with conn.cursor() as cur:
        cur.execute(f"SELECT COUNT(*) as cnt FROM {table_name};")
        num_records = cur.fetchone()[0]

        num_lists = num_records / 1000
        if num_lists < 10:
            num_lists = 10
        if num_records > 1000000:
            num_lists = math.sqrt(num_records)

        cur.execute(f'CREATE INDEX ON {table_name} USING ivfflat ({column_name} {index_method}) WITH (lists = {num_lists});')
        conn.commit()
```


#### pgvector ivfflat query-time parameters

You can also set a parameter to control the accuracy vs. query speed tradeoff at query time. The parameter is called `ivfflat.probes`. This parameter specifies the number of clusters searched during a query. It is recommended to set this parameter to `sqrt(lists)` where lists is the parameter used above during index creation. Higher values improve query accuracy while making the query slower.

You can set the value by running:

```sql
SET ivfflat.probes = 100;
```

Before executing the query, note the [SET command](https://www.postgresql.org/docs/current/sql-set.html) applies to the entire session (database connection) from the point of execution. You can use a transaction-local variant using `LOCAL`:

```sql
BEGIN;
SET LOCAL ivfflat.probes = 100;
SELECT * FROM document_embedding ORDER BY embedding <=> $1 LIMIT 10
COMMIT;
```


#### pgvector ivfflat index-supported queries

You need to use the distance operator (`<=>`, `<->`, or `<#>`) matching the ops class you used during index creation in your `ORDER BY` clause. A canonical query would be:

```sql
SELECT *
FROM document_embedding
ORDER BY embedding <=> $1
LIMIT 10
```


===== PAGE: https://docs.tigerdata.com/ai/python-interface-for-pgvector-and-timescale-vector/ =====

# Python interface for pgvector and pgvectorscale

You use pgai to power production grade AI applications. `timescale_vector` is the
 Python interface you use to interact with a pgai on Tiger Cloud service programmatically.

Before you get started with `timescale_vector`:

- [Sign up for pgai on Tiger Cloud](https://console.cloud.timescale.com/signup?utm_campaign=vectorlaunch&utm_source=docs&utm_medium=direct): Get 90 days free to try pgai on Tiger Cloud.
- [Follow the Get Started Tutorial](https://timescale.github.io/python-vector/tsv_python_getting_started_tutorial.html):
Learn how to use pgai on Tiger Cloud for semantic search on a real-world dataset.

If you prefer to use an LLM development or data framework, see pgai's integrations with [LangChain](https://python.langchain.com/docs/integrations/vectorstores/timescalevector) and [LlamaIndex](https://gpt-index.readthedocs.io/en/stable/examples/vector_stores/Timescalevector.html).

## Prerequisites

`timescale_vector` depends on the source distribution of `psycopg2` and adheres
to [best practices for psycopg2](https://www.psycopg.org/docs/install.html#psycopg-vs-psycopg-binary).

Before you install `timescale_vector`:

* Follow the [psycopg2 build prerequisites](https://www.psycopg.org/docs/install.html#build-prerequisites).

## Install

To interact with pgai on Tiger Cloud using Python:

1. Install `timescale_vector`:

    ```bash
    pip install timescale_vector
    ```
1. Install `dotenv`:

    ```bash
    pip install python-dotenv
    ```

    In these examples, you use `dotenv` to pass secrets and keys.

That is it, you are ready to go.

## Basic usage of the timescale_vector library

First, import all the necessary libraries:

``` python
from dotenv import load_dotenv, find_dotenv
import os
from timescale_vector import client
import uuid
from datetime import datetime, timedelta
```

Load up your Postgres credentials, the safest way is with a `.env` file:

``` python
_ = load_dotenv(find_dotenv(), override=True)
service_url  = os.environ['TIMESCALE_SERVICE_URL']
```

Next, create the client. This tutorial, uses the sync client. But the library has an async client as well (with an identical interface that
uses async functions).

The client constructor takes three required arguments:

| name           | description                                                                               |
|----------------|-------------------------------------------------------------------------------------------|
| `service_url`    | Tiger Cloud service URL / connection string                                                     |
| `table_name`     | Name of the table to use for storing the embeddings. Think of this as the collection name |
| `num_dimensions` | Number of dimensions in the vector                                                        |

``` python
vec  = client.Sync(service_url, "my_data", 2)
```

Next, create the tables for the collection:

``` python
vec.create_tables()
```

Next, insert some data. The data record contains:

- A UUID to uniquely identify the embedding
- A JSON blob of metadata about the embedding
- The text the embedding represents
- The embedding itself

Because this data includes UUIDs which become primary keys, upserts should be used for ingest.

``` python
vec.upsert([\
    (uuid.uuid1(), {"animal": "fox"}, "the brown fox", [1.0,1.3]),\
    (uuid.uuid1(), {"animal": "fox", "action":"jump"}, "jumped over the", [1.0,10.8]),\
])
```

You can now create a vector index to speed up similarity search:

``` python
vec.create_embedding_index(client.TimescaleVectorIndex())
```

Then, you can query for similar items:

``` python
vec.search([1.0, 9.0])
```

    [[UUID('73d05df0-84c1-11ee-98da-6ee10b77fd08'),
      {'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456],
     [UUID('73d05d6e-84c1-11ee-98da-6ee10b77fd08'),
      {'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

There are many search options which are covered below in the
`Advanced search` section.

A simple search example that returns one item using a similarity search
constrained by a metadata filter is shown below:

``` python
vec.search([1.0, 9.0], limit=1, filter={"action": "jump"})
```

    [[UUID('73d05df0-84c1-11ee-98da-6ee10b77fd08'),
      {'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

The returned records contain 5 fields:

| name      | description                                             |
|-----------|---------------------------------------------------------|
| id        | The UUID of the record                                  |
| metadata  | The JSON metadata associated with the record            |
| contents  | the text content that was embedded                      |
| embedding | The vector embedding                                    |
| distance  | The distance between the query embedding and the vector |

You can access the fields by simply using the record as a dictionary
keyed on the field name:

``` python
records = vec.search([1.0, 9.0], limit=1, filter={"action": "jump"})
(records[0]["id"],records[0]["metadata"], records[0]["contents"], records[0]["embedding"], records[0]["distance"])
```

    (UUID('73d05df0-84c1-11ee-98da-6ee10b77fd08'),
     {'action': 'jump', 'animal': 'fox'},
     'jumped over the',
     array([ 1. , 10.8], dtype=float32),
     0.00016793422934946456)

You can delete by ID:

``` python
vec.delete_by_ids([records[0]["id"]])
```

Or you can delete by metadata filters:

``` python
vec.delete_by_metadata({"action": "jump"})
```

To delete all records use:

``` python
vec.delete_all()
```

## Advanced usage

This section goes into more detail about the Python interface. It covers:

1.  Search filter options - how to narrow your search by additional
    constraints
2.  Indexing - how to speed up your similarity queries
3.  Time-based partitioning - how to optimize similarity queries that
    filter on time
4.  Setting different distance types to use in distance calculations

### Search options

The `search` function is very versatile and allows you to search for the right vector in a wide variety of ways. This section describes the search option in 3 parts:

1.  Basic similarity search.
2.  How to filter your search based on the associated metadata.
3.  Filtering on time when time-partitioning is enabled.

The following examples are based on this data:

``` python
vec.upsert([\
    (uuid.uuid1(), {"animal":"fox", "action": "sit", "times":1}, "the brown fox", [1.0,1.3]),\
    (uuid.uuid1(),  {"animal":"fox", "action": "jump", "times":100}, "jumped over the", [1.0,10.8]),\
])
```

The basic query looks like this:

``` python
vec.search([1.0, 9.0])
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456],
     [UUID('7487af14-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

You could provide a limit for the number of items returned:

``` python
vec.search([1.0, 9.0], limit=1)
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

#### Narrowing your search by metadata

There are two main ways to filter results by metadata:
- `filters` for equality matches on metadata.
- `predicates` for complex conditions on metadata.

Filters are more limited in what they can express, but are also more performant. You should use filters if your use case allows it.

##### Using filters for equality matches

You could specify a match on the metadata as a dictionary where all keys
have to match the provided values (keys not in the filter are
unconstrained):

``` python
vec.search([1.0, 9.0], limit=1, filter={"action": "sit"})
```

    [[UUID('7487af14-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

You can also specify a list of filter dictionaries, where an item is
returned if it matches any dict:

``` python
vec.search([1.0, 9.0], limit=2, filter=[{"action": "jump"}, {"animal": "fox"}])
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456],
     [UUID('7487af14-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

##### Using predicates for more advanced filtering on metadata

Predicates allow for more complex search conditions. For example, you
could use greater than and less than conditions on numeric values.

``` python
vec.search([1.0, 9.0], limit=2, predicates=client.Predicates("times", ">", 1))
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

`Predicates`
objects are defined by the name of the metadata key, an operator, and a value.

The supported operators are: `==`, `!=`, `<`, `<=`, `>`, `>=`

The type of the values determines the type of comparison to perform. For
example, passing in `"Sam"` (a string) performs a string comparison while
a `10` (an int) performs an integer comparison, and a `10.0`
(float) performs a float comparison. It is important to note that using a
value of `"10"` performs a string comparison as well so it's important to
use the right type. Supported Python types are: `str`, `int`, and
`float`.

One more example with a string comparison:

``` python
vec.search([1.0, 9.0], limit=2, predicates=client.Predicates("action", "==", "jump"))
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

The real power of predicates is that they can also be combined using the
`&` operator (for combining predicates with `AND` semantics) and `|`(for
combining using OR semantic). So you can do:

``` python
vec.search([1.0, 9.0], limit=2, predicates=client.Predicates("action", "==", "jump") & client.Predicates("times", ">", 1))
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

Just for sanity, the next example shows a case where no results are returned because
of predicates:

``` python
vec.search([1.0, 9.0], limit=2, predicates=client.Predicates("action", "==", "jump") & client.Predicates("times", "==", 1))
```

    []

And one more example where the predicates are defined as a variable
and use grouping with parenthesis:

``` python
my_predicates = client.Predicates("action", "==", "jump") & (client.Predicates("times", "==", 1) | client.Predicates("times", ">", 1))
vec.search([1.0, 9.0], limit=2, predicates=my_predicates)
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

There is also semantic sugar for combining many predicates with `AND`
semantics. You can pass in multiple 3-tuples to
`Predicates`:

``` python
vec.search([1.0, 9.0], limit=2, predicates=client.Predicates(("action", "==", "jump"), ("times", ">", 10)))
```

    [[UUID('7487af96-84c1-11ee-98da-6ee10b77fd08'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

#### Filter your search by time

When using `time-partitioning` (see below) you can very efficiently
filter your search by time. Time-partitioning associates the timestamp embedded
in a UUID-based ID with an embedding. First,
create a collection with time partitioning and insert some data (one
item from January 2018 and another in January 2019):

``` python
tpvec = client.Sync(service_url, "time_partitioned_table", 2, time_partition_interval=timedelta(hours=6))
tpvec.create_tables()

specific_datetime = datetime(2018, 1, 1, 12, 0, 0)
tpvec.upsert([\
    (client.uuid_from_time(specific_datetime), {"animal":"fox", "action": "sit", "times":1}, "the brown fox", [1.0,1.3]),\
    (client.uuid_from_time(specific_datetime+timedelta(days=365)),  {"animal":"fox", "action": "jump", "times":100}, "jumped over the", [1.0,10.8]),\
])
```

Then, you can filter using the timestamps by specifying a
`uuid_time_filter`:

``` python
tpvec.search([1.0, 9.0], limit=4, uuid_time_filter=client.UUIDTimeRange(specific_datetime, specific_datetime+timedelta(days=1)))
```

    [[UUID('33c52800-ef15-11e7-be03-4f1f9a1bde5a'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

A
[`UUIDTimeRange`](https://timescale.github.io/python-vector/vector.html#uuidtimerange)
can specify a `start_date` or `end_date` or both(as in the example above).
Specifying only the `start_date` or `end_date` leaves the other end
unconstrained.

``` python
tpvec.search([1.0, 9.0], limit=4, uuid_time_filter=client.UUIDTimeRange(start_date=specific_datetime))
```

    [[UUID('ac8be800-0de6-11e9-889a-5eec84ba8a7b'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456],
     [UUID('33c52800-ef15-11e7-be03-4f1f9a1bde5a'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

You have the option to define whether the start and end dates
are inclusive with the `start_inclusive` and `end_inclusive` parameters. Setting
`start_inclusive` to true results in comparisons using the `>=`
operator, whereas setting it to false applies the `>` operator. By
default, the start date is inclusive, while the end date is exclusive.
One example:

``` python
tpvec.search([1.0, 9.0], limit=4, uuid_time_filter=client.UUIDTimeRange(start_date=specific_datetime, start_inclusive=False))
```

    [[UUID('ac8be800-0de6-11e9-889a-5eec84ba8a7b'),
      {'times': 100, 'action': 'jump', 'animal': 'fox'},
      'jumped over the',
      array([ 1. , 10.8], dtype=float32),
      0.00016793422934946456]]

Notice how the results are different when using the
`start_inclusive=False` option because the first row has the exact
timestamp specified by `start_date`.

It is also easy to integrate time filters using the `filter` and
`predicates` parameters described above using special reserved key names
to make it appear that the timestamps are part of your metadata. This
is useful when integrating with other systems that just want to
specify a set of filters (often these are "auto retriever" type
systems). The reserved key names are `__start_date` and `__end_date` for
filters and `__uuid_timestamp` for predicates. Some examples below:

``` python
tpvec.search([1.0, 9.0], limit=4, filter={ "__start_date": specific_datetime, "__end_date": specific_datetime+timedelta(days=1)})
```

    [[UUID('33c52800-ef15-11e7-be03-4f1f9a1bde5a'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

``` python
tpvec.search([1.0, 9.0], limit=4,
             predicates=client.Predicates("__uuid_timestamp", ">", specific_datetime) & client.Predicates("__uuid_timestamp", "<", specific_datetime+timedelta(days=1)))
```

    [[UUID('33c52800-ef15-11e7-be03-4f1f9a1bde5a'),
      {'times': 1, 'action': 'sit', 'animal': 'fox'},
      'the brown fox',
      array([1. , 1.3], dtype=float32),
      0.14489260377438218]]

### Indexing

Indexing speeds up queries over your data. By default, the system creates indexes
to query your data by the UUID and the metadata.

To speed up similarity search based on the embeddings, you have to
create additional indexes.

Note that if performing a query without an index, you always get an
exact result, but the query is slow (it has to read all of the data
you store for every query). With an index, your queries are
order-of-magnitude faster, but the results are approximate (because there
are no known indexing techniques that are exact).

Luckily, TimescaleDB provides 3 excellent approximate indexing algorithms,
StreamingDiskANN, HNSW, and ivfflat.

Below are the trade-offs between these algorithms:

| Algorithm        | Build speed | Query speed | Need to rebuild after updates |
|------------------|-------------|-------------|-------------------------------|
| StreamingDiskAnn | Fast        | Fastest     | No                            |
| HNSW    | Fast   | Faster      | No                            |
| ivfflat | Fastest     | Slowest     | Yes                           |

You can see
[benchmarks](https://www.timescale.com/blog/how-we-made-postgresql-the-best-vector-database/)
on the blog.

You should use the StreamingDiskANN index for most use cases. This
can be created with:

``` python
vec.create_embedding_index(client.TimescaleVectorIndex())
```

Indexes are created for a particular distance metric type. So it is
important that the same distance metric is set on the client during
index creation as it is during queries. See the `distance type` section
below.

Each of these indexes has a set of build-time options for controlling
the speed/accuracy trade-off when creating the index and an additional
query-time option for controlling accuracy during a particular query. The
library uses smart defaults for all of these options. The
details for how to adjust these options manually are below.

<!-- vale Google.Headings = NO -->
#### StreamingDiskANN index
<!-- vale Google.Headings = YES -->

The StreamingDiskANN index is a graph-based algorithm that uses the
[DiskANN](https://github.com/microsoft/DiskANN) algorithm. You can read
more about it in the
[blog](https://www.timescale.com/blog/how-we-made-postgresql-the-best-vector-database/)
announcing its release.

To create this index, run:

``` python
vec.create_embedding_index(client.TimescaleVectorIndex())
```

The above command creates the index using smart defaults. There are
a number of parameters you could tune to adjust the accuracy/speed
trade-off.

The parameters you can set at index build time are:

| Parameter name   | Description                                                                                                                                                   | Default value |
|------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|
| `num_neighbors`    | Sets the maximum number of neighbors per node. Higher values increase accuracy but make the graph traversal slower.                                           | 50            |
| `search_list_size` | This is the S parameter used in the greedy search algorithm used during construction. Higher values improve graph quality at the cost of slower index builds. | 100           |
| `max_alpha`        | Is the alpha parameter in the algorithm. Higher values improve graph quality at the cost of slower index builds.                                              | 1.0           |

To set these parameters, you could run:

``` python
vec.create_embedding_index(client.TimescaleVectorIndex(num_neighbors=50, search_list_size=100, max_alpha=1.0))
```

You can also set a parameter to control the accuracy vs. query speed
trade-off at query time. The parameter is set in the `search()` function
using the `query_params` argument. You can set the
`search_list_size`(default: 100). This is the number of additional
candidates considered during the graph search at query time. Higher
values improve query accuracy while making the query slower.

You can specify this value during search as follows:

``` python
vec.search([1.0, 9.0], limit=4, query_params=TimescaleVectorIndexParams(search_list_size=10))
```

To drop the index, run:

``` python
vec.drop_embedding_index()
```

#### pgvector HNSW index

Pgvector provides a graph-based indexing algorithm based on the popular
[HNSW algorithm](https://arxiv.org/abs/1603.09320).

To create this index, run:

``` python
vec.create_embedding_index(client.HNSWIndex())
```

The above command creates the index using smart defaults. There are
a number of parameters you could tune to adjust the accuracy/speed
trade-off.

The parameters you can set at index build time are:

| Parameter name  | Description                                                                                                                                                                                                                                                            | Default value |
|-----------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------|
| `m`               | Represents the maximum number of connections per layer. Think of these connections as edges created for each node during graph construction. Increasing m increases accuracy but also increases index build time and size.                                             | 16            |
| `ef_construction` | Represents the size of the dynamic candidate list for constructing the graph. It influences the trade-off between index quality and construction speed. Increasing `ef_construction` enables more accurate search results at the expense of lengthier index build times. | 64            |

To set these parameters, you could run:

``` python
vec.create_embedding_index(client.HNSWIndex(m=16, ef_construction=64))
```

You can also set a parameter to control the accuracy vs. query speed
trade-off at query time. The parameter is set in the `search()` function
using the `query_params` argument. You can set the `ef_search`(default:
40). This parameter specifies the size of the dynamic candidate list
used during search. Higher values improve query accuracy while making
the query slower.

You can specify this value during search as follows:

``` python
vec.search([1.0, 9.0], limit=4, query_params=HNSWIndexParams(ef_search=10))
```

To drop the index run:

``` python
vec.drop_embedding_index()
```

#### pgvector ivfflat index

Pgvector provides a clustering-based indexing algorithm. The [blog
post](https://www.timescale.com/blog/nearest-neighbor-indexes-what-are-ivfflat-indexes-in-pgvector-and-how-do-they-work/)
describes how it works in detail. It provides the fastest
index-build speed but the slowest query speeds of any indexing
algorithm.

To create this index, run:

``` python
vec.create_embedding_index(client.IvfflatIndex())
```

Note: *ivfflat should never be created on empty tables* because it needs
to cluster data, and that only happens when an index is first created,
not when new rows are inserted or modified. Also, if your table
undergoes a lot of modifications, you need to rebuild this index
occasionally to maintain good accuracy. See the [blog
post](https://www.timescale.com/blog/nearest-neighbor-indexes-what-are-ivfflat-indexes-in-pgvector-and-how-do-they-work/)
for details.

Pgvector ivfflat has a `lists` index parameter that is automatically set
with a smart default based on the number of rows in your table. If you
know that you'll have a different table size, you can specify the number
of records to use for calculating the `lists` parameter as follows:

``` python
vec.create_embedding_index(client.IvfflatIndex(num_records=1000000))
```

You can also set the `lists` parameter directly:

``` python
vec.create_embedding_index(client.IvfflatIndex(num_lists=100))
```

You can also set a parameter to control the accuracy vs. query speed
trade-off at query time. The parameter is set in the `search()` function
using the `query_params` argument. You can set the `probes`. This
parameter specifies the number of clusters searched during a query. It
is recommended to set this parameter to `sqrt(lists)` where lists is the
`num_list` parameter used above during index creation. Higher values
improve query accuracy while making the query slower.

You can specify this value during search as follows:

``` python
vec.search([1.0, 9.0], limit=4, query_params=IvfflatIndexParams(probes=10))
```

To drop the index, run:

``` python
vec.drop_embedding_index()
```

### Time partitioning

In many use cases where you have many embeddings, time is an important
component associated with the embeddings. For example, when embedding
news stories, you often search by time as well as similarity
(for example, stories related to Bitcoin in the past week or stories about
Clinton in November 2016).

Yet, traditionally, searching by two components "similarity" and "time"
is challenging for Approximate Nearest Neighbor (ANN) indexes and makes the
similarity-search index less effective.

One approach to solving this is partitioning the data by time and
creating ANN indexes on each partition individually. Then, during search,
you can:

- Step 1: filter partitions that don't match the time predicate.
- Step 2: perform the similarity search on all matching partitions.
- Step 3: combine all the results from each partition in step 2, re-rank,
  and filter out results by time.

Step 1 makes the search a lot more efficient by filtering out whole
swaths of data in one go.

Timescale-vector supports time partitioning using TimescaleDB's
hypertables. To use this feature, simply indicate the length of time for
each partition when creating the client:

``` python
from datetime import timedelta
from datetime import datetime
```

``` python
vec = client.Async(service_url, "my_data_with_time_partition", 2, time_partition_interval=timedelta(hours=6))
await vec.create_tables()
```

Then, insert data where the IDs use UUIDs v1 and the time component of
the UUIDspecifies the time of the embedding. For example, to create an
embedding for the current time, simply do:

``` python
id = uuid.uuid1()
await vec.upsert([(id, {"key": "val"}, "the brown fox", [1.0, 1.2])])
```

To insert data for a specific time in the past, create the UUID using the
`uuid_from_time` function

``` python
specific_datetime = datetime(2018, 8, 10, 15, 30, 0)
await vec.upsert([(client.uuid_from_time(specific_datetime), {"key": "val"}, "the brown fox", [1.0, 1.2])])
```

You can then query the data by specifying a `uuid_time_filter` in the
search call:

``` python
rec = await vec.search([1.0, 2.0], limit=4, uuid_time_filter=client.UUIDTimeRange(specific_datetime-timedelta(days=7), specific_datetime+timedelta(days=7)))
```

### Distance metrics

Cosine distance is used by default to measure how similarly an embedding
is to a given query. In addition to cosine distance, Euclidean/L2 distance is
also supported. The distance type is set when creating the client
using the `distance_type` parameter. For example, to use the Euclidean
distance metric, you can create the client with:

``` python
vec  = client.Sync(service_url, "my_data", 2, distance_type="euclidean")
```

Valid values for `distance_type` are `cosine` and `euclidean`.

It is important to note that you should use consistent distance types on
clients that create indexes and perform queries. That is because an
index is only valid for one particular type of distance measure.

Note that the StreamingDiskANN index only supports cosine distance at
this time.


===== PAGE: https://docs.tigerdata.com/ai/langchain-integration-for-pgvector-and-timescale-vector/ =====

# LangChain Integration for pgvector, pgvectorscale, and pgai

[LangChain](https://www.langchain.com/) is a popular framework for development applications powered by LLMs. pgai on Tiger Cloud has a native LangChain integration, enabling you to use it as a vector store and leverage all its capabilities in your applications built with LangChain.

Here are resources about using pgai on Tiger Cloud with LangChain:

- [Getting started with LangChain and pgvectorscale](https://python.langchain.com/docs/integrations/vectorstores/timescalevector): You'll learn how to use pgai on Tiger Data for (1) semantic search, (2) time-based vector search, (3) self-querying, and (4) how to create indexes to speed up queries.
- [Postgres Self Querying](https://python.langchain.com/docs/integrations/retrievers/self_query/timescalevector_self_query): Learn how to use pgai on Tiger Data with self-querying in LangChain.
- [Learn more about pgai on Tiger Data and LangChain](https://blog.langchain.dev/timescale-vector-x-langchain-making-postgresql-a-better-vector-database-for-ai-applications/):  A blog post about the unique capabilities that pgai on Tiger Cloud brings to the LangChain ecosystem.


===== PAGE: https://docs.tigerdata.com/ai/llamaindex-integration-for-pgvector-and-timescale-vector/ =====

# LlamaIndex Integration for pgvector and Tiger Data Vector

## LlamaIndex integration for pgvector and Tiger Data Vector

[LlamaIndex](https://www.llamaindex.ai/) is a popular data framework for connecting custom data sources to large language models (LLMs). Tiger Data Vector has a native LlamaIndex integration that supports all the features of pgvector and Tiger Data Vector. It enables you to use Tiger Data Vector as a vector store and leverage all its capabilities in your applications built with LlamaIndex.

Here are resources about using Tiger Data Vector with LlamaIndex:

- [Getting started with LlamaIndex and TigerData Vector](https://docs.llamaindex.ai/en/stable/examples/vector_stores/Timescalevector.html): You'll learn how to use Tiger Data Vector for (1) similarity search, (2) time-based vector search, (3) faster search with indexes, and (4) retrieval and query engine.
- [Time-based retrieval](https://youtu.be/EYMZVfKcRzM?si=I0H3uUPgzKbQw__W): Learn how to power RAG applications with time-based retrieval.
- [Llama Pack: Auto Retrieval with time-based search](https://github.com/run-llama/llama-hub/tree/main/llama_hub/llama_packs/timescale_vector_autoretrieval): This pack demonstrates performing auto-retrieval for hybrid search based on both similarity and time, using the timescale-vector (Postgres) vector store.
- [Learn more about TigerData Vector and LlamaIndex ](https://www.timescale.com/blog/timescale-vector-x-llamaindex-making-postgresql-a-better-vector-database-for-ai-applications/): How Tiger Data Vector is a better Postgres for AI applications.


===== PAGE: https://docs.tigerdata.com/ai/pgvectorizer/ =====

# Embed your Postgres data with PgVectorizer

## Embed Postgres data with PgVectorizer

PgVectorizer enables you to create vector embeddings from any data that
you already have stored in Postgres. You can get more background
information in the [blog
post](https://www.timescale.com/blog/a-complete-guide-to-creating-and-storing-embeddings-for-postgresql-data/)
announcing this feature, as well as the ["how we built
it"](https://www.timescale.com/blog/how-we-designed-a-resilient-vector-embedding-creation-system-for-postgresql-data/)
post going into the details of the design.

To create vector embeddings, simply attach PgVectorizer to any Postgres
table to automatically sync that table's data with a set of
embeddings stored in Postgres. For example, say you have a
blog table defined in the following way:

``` python
import psycopg2
from langchain.docstore.document import Document
from langchain.text_splitter import CharacterTextSplitter
from timescale_vector import client, pgvectorizer
from langchain.embeddings.openai import OpenAIEmbeddings
from langchain.vectorstores.timescalevector import TimescaleVector
from datetime import timedelta
```

``` python
with psycopg2.connect(service_url) as conn:
    with conn.cursor() as cursor:
        cursor.execute('''
        CREATE TABLE IF NOT EXISTS blog (
            id              INT PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
            title           TEXT NOT NULL,
            author          TEXT NOT NULL,
            contents        TEXT NOT NULL,
            category        TEXT NOT NULL,
            published_time  TIMESTAMPTZ NULL --NULL if not yet published
        );
        ''')
```

You can insert some data as follows:

``` python
with psycopg2.connect(service_url) as conn:
    with conn.cursor() as cursor:
        cursor.execute('''
            INSERT INTO blog (title, author, contents, category, published_time) VALUES ('First Post', 'Matvey Arye', 'some super interesting content about cats.', 'AI', '2021-01-01');
        ''')
```

Now, say you want to embed these blogs and store the embeddings in Postgres. First, you
need to define an `embed_and_write` function that takes a set of blog
posts, creates the embeddings, and writes them into TigerData Vector. For
example, if using LangChain, it could look something like the following.

``` python
def get_document(blog):
    text_splitter = CharacterTextSplitter(
        chunk_size=1000,
        chunk_overlap=200,
    )
    docs = []
    for chunk in text_splitter.split_text(blog['contents']):
        content = f"Author {blog['author']}, title: {blog['title']}, contents:{chunk}"
        metadata = {
            "id": str(client.uuid_from_time(blog['published_time'])),
            "blog_id": blog['id'],
            "author": blog['author'],
            "category": blog['category'],
            "published_time": blog['published_time'].isoformat(),
        }
        docs.append(Document(page_content=content, metadata=metadata))
    return docs

def embed_and_write(blog_instances, vectorizer):
    embedding = OpenAIEmbeddings()
    vector_store = TimescaleVector(
        collection_name="blog_embedding",
        service_url=service_url,
        embedding=embedding,
        time_partition_interval=timedelta(days=30),
    )

    metadata_for_delete = [{"blog_id": blog['locked_id']} for blog in blog_instances]
    vector_store.delete_by_metadata(metadata_for_delete)

    documents = []
    for blog in blog_instances:
        if blog['published_time'] != None:
            documents.extend(get_document(blog))

    if len(documents) == 0:
        return

    texts = [d.page_content for d in documents]
    metadatas = [d.metadata for d in documents]
    ids = [d.metadata["id"] for d in documents]
    vector_store.add_texts(texts, metadatas, ids)
```

Then, all you have to do is run the following code in a scheduled job
(cron job, Lambda job, etc):

``` python
vectorizer = pgvectorizer.Vectorize(service_url, 'blog')
while vectorizer.process(embed_and_write) > 0:
    pass
```

Every time that job runs, it syncs the table with your embeddings. It
syncs all inserts, updates, and deletes to an embeddings table called
`blog_embedding`.

Now, you can simply search the embeddings as follows (again, using
LangChain in the example):

``` python
embedding = OpenAIEmbeddings()
vector_store = TimescaleVector(
    collection_name="blog_embedding",
    service_url=service_url,
    embedding=embedding,
    time_partition_interval=timedelta(days=30),
)

res = vector_store.similarity_search_with_score("Blogs about cats")
res
```

    [(Document(page_content='Author Matvey Arye, title: First Post, contents:some super interesting content about cats.', metadata={'id': '4a784000-4bc4-11eb-855a-06302dbc8ce7', 'author': 'Matvey Arye', 'blog_id': 1, 'category': 'AI', 'published_time': '2021-01-01T00:00:00+00:00'}),
      0.12595687795193833)]


===== PAGE: https://docs.tigerdata.com/README/ =====

<div align=center>
<picture align=center>
    <source media="(prefers-color-scheme: dark)" srcset="https://assets.timescale.com/docs/images/tigerdata-gradient-white.svg">
    <source media="(prefers-color-scheme: light)" srcset="https://assets.timescale.com/docs/images/tigerdata-gradient-black.svg">
    <img alt="Tiger Data logo" >
</picture>
</div>

<div align=center>

<h3>Tiger Cloud is the modern Postgres data platform for all your applications. It enhances Postgres to handle time series, events, real-time analytics, and vector search—all in a single database alongside transactional workloads.
</h3>

[![Docs](https://img.shields.io/badge/Read_the_TigerData_docs-black?style=for-the-badge&logo=readthedocs&logoColor=white)](https://docs.tigerdata.com/)
[![SLACK](https://img.shields.io/badge/Ask_the_TigerData_community-black?style=for-the-badge&logo=slack&logoColor=white)](https://timescaledb.slack.com/archives/C4GT3N90X)
[![Try Tiger Cloud for free](https://img.shields.io/badge/Try_Tiger_for_free-black?style=for-the-badge&logo=timescale&logoColor=white)](https://console.cloud.timescale.com/signup)

</div>

This repository contains the current source for Tiger Data documentation available at https://docs.tigerdata.com/.

We welcome contributions! You can contribute to Tiger Data documentation in the following ways:

- [Create an issue][docs-issues] in this repository and describe the proposed change. Our doc team takes care of it.
- Update the docs yourself and have your change reviewed and published by our doc team.

## Contribute to the Tiger Data docs

To make the contribution yourself:

1. Get the documentation source:

    - No write access? [Fork this repository][github-fork].
    - Already have a write access? [Clone this repository][github-clone].

2. Create a branch from `latest`, make your changes, and raise a pull request back to `latest`.

3. Sign a Contributor License Agreement (CLA).

   You have to sign the CLA only the first time you raise a PR. This helps to ensure that the community is free to use your contributions.

4. Review your changes.

   The documentation site is generated in a separate private repository using [Gatsby][gatsby]. Once you raise a PR for any branch, GitHub **automatically** generates a preview for your changes and attaches the link in the comments. Any new commits are visible at the same URL. If you don't see the latest changes, try an incognito browser window. Automated builds are not available for PRs from forked repositories.

See the [Contributing guide](CONTRIBUTING.md) for style and language guidance.

## Learn about Tiger Data

Tiger Data is Postgres made powerful. To learn more about the company and its products, visit [tigerdata.com](https://www.tigerdata.com).


===== PAGE: https://docs.tigerdata.com/CONTRIBUTING/ =====

# Contribute to Tiger Data documentation

Tiger Data documentation is open for contribution from all community members. The current source is in this repository.

This page explains the structure and language guidelines for contributing to Tiger Data documentation. See the [README][readme] for how to contribute.

## Language

Write in a clear, concise, and actionable manner. Tiger Data documentation uses the [Google Developer Documentation Style Guide][google-style] with the following exceptions:

- Do not capitalize the first word after a colon.
- Use code font (back ticks) for UI elements instead of semi-bold.

## Edit individual pages

Each major doc section has a dedicated directory with `.md` files inside, representing its child pages. This includes an `index.md` file that serves as a landing page for that doc section by default, unless specifically changed in the navigation tree. To edit a page, modify the corresponding `.md` file following these recommendations:

- **Regular pages** should include:

  - A short intro describing the main subject of the page.
  - A visual illustrating the main concept, if relevant.
  - Paragraphs with descriptive headers, organizing the content into logical sections.
  - Procedures to describe the sequence of steps to reach a certain goal. For example, create a Tiger Cloud service.
  - Other visual aids, if necessary.
  - Links to other relevant resources.

- **API pages** should include:

  - The function name, with empty parentheses if it takes arguments.
  - A brief, specific description of the function, including any possible warnings.
  - One or two samples of the function being used to demonstrate argument syntax.
  - An argument table with `Name`, `Type`, `Default`, `Required`, `Description` columns.
  - A return table with `Column`, `Type`, and `Description` columns.

- **Troubleshooting pages** are not written as whole Markdown files, but are programmatically assembled from individual files in the`_troubleshooting` folder. Each entry describes a single troubleshooting case and its solution, and contains the following front matter:

    |Key| Type  |Required| Description                                                                                                                                                                           |
    |-|-------|-|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
    |`title`| string                                              |✅| The title of the troubleshooting entry, displayed as a heading above it                                                                                                               |
    |`section`| The literal string `troubleshooting`                |✅| Must be `troubleshooting`, used to identify troubleshooting entries during site build                                                                                                 |
    |`products` or `topics`| array of strings                                    |✅ (can have either or both, but must have at least one)| The products or topics related to the entry. The entry shows up on the troubleshooting pages for the listed products and topics.                                                      |
    |`errors`| object of form `{language: string, message: string}` |❌| The error, if any, related to the troubleshooting entry. Displayed as a code block right underneath the title. `language` is the programming language to use for syntax highlighting. |
    |`keywords`| array of strings                                    |❌| These are displayed at the bottom of every troubleshooting page. Each keyword links to a collection of all pages associated with that keyword.                                        |
    |`tags`| array of strings                                    |❌| Concepts, actions, or things associated with the troubleshooting entry. These are not displayed in the UI, but they affect the calculation of related pages.                          |

    Beneath the front matter, describe the error and its solution in regular Markdown. You can also use any other components allowed within the docs site.

    The entry shows up on the troubleshooting pages for its associated products and topics. If the page doesn't already exist, add an entry for it in the page
    index, setting `type` to `placeholder`. See [Navigation tree](#navigation-tree).

## Edit the navigation hierarchy

The navigation hierarchy of a doc section is governed by `page-index/page-index.js` within the corresponding directory. For example:

```js
     {
        title: "Tiger Cloud services",
        href: "services",
        excerpt: "About Tiger Cloud services",
        children: [
          {
            title: "Services overview",
            href: "service-overview",
            excerpt: "Tiger Cloud services overview",
          },
          {
            title: "Service explorer",
            href: "service-explorer",
            excerpt: "Tiger Cloud services explorer",
          },
          {
            title: "Troubleshooting Tiger Cloud services",
            href: "troubleshooting",
            type: "placeholder",
          },
        ],
      },
```

See [Use Tiger Cloud section navigation][use-navigation] for reference.

To change the structure, add or delete pages in a section, modify the corresponding `page-index.js`. An entry in a `page-index.js` includes the following fields:

| Key                | Type                                                      | Required | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
|--------------------|-----------------------------------------------------------|----------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `href`             | string                                                    | ✅      | The URL segment to use for the page. If there is a corresponding Markdown file, `href` must match the name of the Markdown file, minus the file extension.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| `title`            | string                                                    | ✅      | The title of the page, used as the page name within the TOC on the left. Must be the same as the first header in the corresponding Markdown file.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
| `excerpt`          | string                                                    | ✅       | The short description of the page, used for the page card if `pageComponents` is set to `featured-cards`. Should be up to 100 characters. See `pageComponents` for details.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| `type`             | One of `[directory, placeholder, redirect-to-child-page]` | ❌       | If no type is specified, the page is built as a regular webpage. The structure of its children, if present, is defined by `children` entries and the corresponding structure of subfolders.  If the type is `directory`, the corresponding file becomes a directory. The difference of the directory page is that its child pages sit at the same level as the `directory` page. They only become children during the site build. If the type is `placeholder`, the corresponding page is produced programmatically upon site build. If not produced, the link in the navigation tree returns a 404. In particular, this is used for troubleshooting pages. If the type is `redirect-to-child-page`, no page is built and the link in the navigation tree goes directly to the first child. |
| `children`         | Array of page entries                                     | ❌       | Child pages of the current page. For regular pages, the children should be located in a directory with the same name as the parent. The parent is the `index.md` file in that directory. For`directory` pages, the children should be located in the same directory as the parent.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| `pageComponents`   | One of `[['featured-cards'], ['content-list']]`           | ❌       | Any page that has child pages can list its children in either card or list style at the bottom of the page. Specify the desired style with this key.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 