# Example 1: Fixing Null Pointer Exception (Basic)

## Kanitsal Cerceve (Evidential Frame Activation)
Kaynak dogrulama modu etkin.




## When to Use This Skill

- **Production Incidents**: Critical bugs affecting live users requiring rapid diagnosis
- **Intermittent Failures**: Flaky tests, race conditions, or timing-dependent bugs
- **Performance Issues**: Slow endpoints, memory leaks, or CPU spikes
- **Integration Failures**: Third-party API errors, database connectivity issues
- **Regression Analysis**: New bugs introduced by recent changes
- **Complex Stack Traces**: Multi-layered errors spanning multiple services

## When NOT to Use This Skill

- **Feature Development**: Building new functionality (use feature-dev-complete instead)
- **Code Reviews**: Reviewing code quality or architecture (use code-review-assistant)
- **Refactoring**: Restructuring code without fixing bugs (use refactoring skills)
- **Known Issues**: Bugs with clear root cause already identified

## Success Criteria

- [ ] Root cause identified with supporting evidence
- [ ] Fix implemented and tested
- [ ] Regression test added to prevent recurrence
- [ ] All related test suites passing
- [ ] Fix validated in production-like environment
- [ ] Documentation updated with troubleshooting notes
- [ ] Monitoring/alerting adjusted if needed

## Edge Cases to Handle

- **Heisenbugs**: Bugs that disappear when debugger attached
- **Multi-Service Failures**: Cascading errors across microservices
- **Data Corruption**: State inconsistencies requiring rollback
- **Timezone Issues**: Date/time bugs across regions
- **Concurrency Bugs**: Race conditions, deadlocks, or thread safety
- **Memory Corruption**: Pointer errors, buffer overflows in native code

## Guardrails

- **NEVER** deploy debug code or verbose logging to production
- **ALWAYS** reproduce bugs locally before proposing fixes
- **NEVER** fix symptoms without understanding root cause
- **ALWAYS** add regression tests for fixed bugs
- **NEVER** disable tests to make CI pass
- **ALWAYS** verify fixes do not introduce new bugs
- **NEVER** modify production data without backup

## Evidence-Based Validation

- [ ] Bug reproduced consistently with minimal test case
- [ ] Stack traces analyzed with error tracking tools (Sentry, Rollbar)
- [ ] Performance profiled with appropriate tools (Chrome DevTools, py-spy)
- [ ] Fix verified with automated tests
- [ ] Integration tests passing
- [ ] No new errors in application logs
- [ ] Memory/CPU usage within normal bounds

## Overview

This example demonstrates systematic debugging of a common null pointer exception using the 5-phase debugging protocol.

**Difficulty**: Basic
**Estimated Time**: 15-20 minutes
**Agents Used**: code-analyzer, coder, tester

## The Problem

### Initial Symptom

```
TypeError: Cannot read property 'name' of undefined
  at processUser (users.js:45)
  at handleRequest (api.js:23)
  at Server.handle (server.js:102)
```

**User Report**: "The user profile API returns a 500 error randomly when viewing user profiles."

## Phase 1: Symptom Identification

### Agent: code-analyzer

**Investigation Steps**:
1. Collect error logs from production monitoring
2. Attempt to reproduce locally
3. Identify patterns in failures

**Findings**:
```javascript
// Error occurs in users.js at line 45
function processUser(user) {
  return user.name.toUpperCase(); // Line 45 - CRASH HERE
}
```

**Reproduction Steps**:
```javascript
// Test case that reproduces the issue
const result = processUser(undefined);
// TypeError: Cannot read property 'name' of undefined
```

**Context Gathered**:
- Error occurs when database query returns no results
- API endpoint: `GET /api/users/:id`
- Frequency: ~5% of requests (when user ID doesn't exist)
- Impact: Users see 500 error instead of 404 Not Found

**Symptom Report**:
```
Issue ID: BUG-2024-001
Type: Runtime Error (NullPointerException)
Severity: High
Frequency: 5% of requests
Root Location: users.js:45
Expected: Return 404 for non-existent users
Actual: Server crashes with 500 error
```

## Phase 2: Root Cause Analysis

### Agent: code-analyzer + coder

**Hypothesis**: The `processUser` function assumes a valid user object will always be passed.

**Trace Execution**:
```javascript
// Call chain
1. api.js:23 - handleRequest() receives request
2. database.js:78 - findUserById() returns null (user not found)
3. users.js:45 - processUser(null) crashes
```

**Variable States**:
- `userId`: "12345" (valid ID format)
- `user`: `null` (no matching record in database)
- `user.name`: CRASH - cannot access property on null

**Root Cause Statement**:
> The `processUser` function lacks null safety validation. When the database query returns `null` for non-existent users, the function attempts to access properties on `null`, causing a runtime exception.

**Why This Occurs**:
1. No validation before property access
2. Implicit assumption that user always exists
3. No error handling for missing users
4. Database layer allows null returns

**Affected Code**:
```javascript
// users.js
function processUser(user) {
  // PROBLEM: No null check before accessing properties
  return user.name.toUpperCase();
}

// api.js
async function handleRequest(req, res) {
  const user = await findUserById(req.params.id);
  // PROBLEM: No check if user is null before processing
  const processedData = processUser(user);
  res.json(processedData);
}
```

## Phase 3: Fix Generation

### Agent: coder

**Solution Approaches Evaluated**:

### Option 1: Add Null Check in processUser (CHOSEN)
```javascript
function processUser(user) {
  if (!user || !user.name) {
    throw new Error('Invalid user object');
  }
  return user.name.toUpperCase();
}
```
**Pros**: Explicit validation, clear error message, maintains function contract
**Cons**: Throws exception that must be caught upstream
**Trade-offs**: Best for maintainability and clarity

### Option 2: Return Default Value
```javascript
function processUser(user) {
  return user?.name?.toUpperCase() || 'UNKNOWN';
}
```
**Pros**: No exceptions, uses optional chaining
**Cons**: Hides the error, returns misleading data
**Trade-offs**: Not suitable - masks the real issue

### Option 3: Handle in API Layer Only
```javascript
async function handleRequest(req, res) {
  const user = await findUserById(req.params.id);
  if (!user) {
    return res.status(404).json({ error: 'User not found' });
  }
  const processedData = processUser(user);
  res.json(processedData);
}
```
**Pros**: Proper HTTP semantics, defensive at API boundary
**Cons**: Doesn't protect processUser from future misuse
**Trade-offs**: Should be used IN ADDITION to Option 1

**Selected Solution**: Combination of Option 1 and Option 3

### Implementation

```javascript
// users.js - Add null safety validation
/**
 * Process user data for API response
 * @param {Object} user - User object from database
 * @returns {string} Formatted user name
 * @throws {Error} If user object is invalid or missing required fields
 */
function processUser(user) {
  // Validate user object exists
  if (!user) {
    throw new Error('User object is required');
  }

  // Validate required fields
  if (!user.name || typeof user.name !== 'string') {
    throw new Error('User object must have a valid name field');
  }

  return user.name.toUpperCase();
}

// api.js - Add proper error handling at API boundary
async function handleRequest(req, res) {
  try {
    const user = await findUserById(req.params.id);

    // Return 404 if user doesn't exist
    if (!user) {
      return res.status(404).json({
        error: 'User not found',
        message: `No user found with ID: ${req.params.id}`
      });
    }

    // Process and return user data
    const processedData = processUser(user);
    res.status(200).json({
      name: processedData,
      id: user.id
    });

  } catch (error) {
    console.error('Error processing user request:', error);
    res.status(500).json({
      error: 'Internal server error',
      message: error.message
    });
  }
}
```

**Why This Approach**:
1. **Defense in depth**: Validation at both function and API levels
2. **Clear error messages**: Explicit about what went wrong
3. **Proper HTTP semantics**: 404 for missing resources, 500 for server errors
4. **Maintainability**: Future-proof against misuse
5. **No silent failures**: Errors are logged and surfaced appropriately

## Phase 4: Validation Testing

### Agent: tester

**Test Suite**:

```javascript
// tests/users.test.js
const { processUser } = require('../users');

describe('processUser', () => {
  describe('with valid user object', () => {
    it('should return uppercase name', () => {
      const user = { id: 1, name: 'john doe' };
      expect(processUser(user)).toBe('JOHN DOE');
    });

    it('should handle names with special characters', () => {
      const user = { id: 2, name: 'mary-jane o\'brien' };
      expect(processUser(user)).toBe("MARY-JANE O'BRIEN");
    });
  });

  describe('with invalid user object', () => {
    it('should throw error when user is null', () => {
      expect(() => processUser(null)).toThrow('User object is required');
    });

    it('should throw error when user is undefined', () => {
      expect(() => processUser(undefined)).toThrow('User object is required');
    });

    it('should throw error when user has no name', () => {
      const user = { id: 1 };
      expect(() => processUser(user)).toThrow('valid name field');
    });

    it('should throw error when name is not a string', () => {
      const user = { id: 1, name: 123 };
      expect(() => processUser(user)).toThrow('valid name field');
    });

    it('should throw error when name is empty string', () => {
      const user = { id: 1, name: '' };
      expect(() => processUser(user)).toThrow('valid name field');
    });
  });
});

// tests/api.test.js
const request = require('supertest');
const app = require('../server');
const db = require('../database');

describe('GET /api/users/:id', () => {
  beforeEach(() => {
    // Setup test database
    db.seed([
      { id: 1, name: 'Alice Smith' },
      { id: 2, name: 'Bob Jones' }
    ]);
  });

  it('should return user data for existing user', async () => {
    const response = await request(app)
      .get('/api/users/1')
      .expect(200);

    expect(response.body).toEqual({
      id: 1,
      name: 'ALICE SMITH'
    });
  });

  it('should return 404 for non-existent user', async () => {
    const response = await request(app)
      .get('/api/users/99999')
      .expect(404);

    expect(response.body).toEqual({
      error: 'User not found',
      message: 'No user found with ID: 99999'
    });
  });

  it('should return 400 for invalid ID format', async () => {
    const response = await request(app)
      .get('/api/users/invalid-id')
      .expect(400);

    expect(response.body.error).toContain('Invalid user ID');
  });
});
```

**Test Results**:
```
✓ processUser › with valid user object › should return uppercase name
✓ processUser › with valid user object › should handle names with special characters
✓ processUser › with invalid user object › should throw error when user is null
✓ processUser › with invalid user object › should throw error when user is undefined
✓ processUser › with invalid user object › should throw error when user has no name
✓ processUser › with invalid user object › should throw error when name is not a string
✓ processUser › with invalid user object › should throw error when name is empty string
✓ GET /api/users/:id › should return user data for existing user
✓ GET /api/users/:id › should return 404 for non-existent user
✓ GET /api/users/:id › should return 400 for invalid ID format

Test Suites: 2 passed, 2 total
Tests:       10 passed, 10 total
Coverage:    users.js: 100%, api.js: 95%
```

**Regression Testing**:
- All existing tests pass ✅
- No breaking changes to API contract ✅
- Performance unchanged (< 1ms overhead) ✅

## Phase 5: Regression Prevention

### Agent: tester + coder

**Permanent Test Cases Added**:
- Tests integrated into main test suite
- CI/CD pipeline now runs these tests on every commit
- Coverage increased from 78% → 95% for affected modules

**Documentation Updates**:
```javascript
// Added to CHANGELOG.md
## [1.2.1] - 2024-11-02
### Fixed
- Fixed null pointer exception in processUser when user doesn't exist
- API now returns proper 404 status for non-existent users instead of 500

### Added
- Null safety validation in processUser function
- Comprehensive error handling in user API endpoints
- 10 new test cases covering edge cases
```

**Code Comments**:
```javascript
// users.js
/**
 * Process user data for API response
 *
 * IMPORTANT: This function requires a valid user object with a name field.
 * Always validate that the user exists before calling this function.
 *
 * Historical Note: Previously crashed with null/undefined inputs (BUG-2024-001).
 * Now throws explicit errors for invalid inputs to prevent silent failures.
 */
```

**Monitoring Enhancements**:
```javascript
// Added alert for similar errors
monitoring.alert({
  condition: 'error.type === "TypeError" && error.message.includes("Cannot read property")',
  action: 'notify-team',
  priority: 'high',
  message: 'Potential null pointer exception detected'
});
```

**Development Guidelines Updated**:
```markdown
## Coding Standards - Null Safety

**Rule**: Always validate object existence before accessing properties.

**Pattern to use**:
```javascript
if (!obj || !obj.property) {
  throw new Error('Descriptive error message');
}
```

**Pattern to avoid**:
```javascript
// ❌ Direct property access without validation
return obj.property.method();
```

**Rationale**: See BUG-2024-001 for production incident caused by missing validation.
```

## Summary

**Issue**: Null pointer exception causing 500 errors
**Root Cause**: Missing null safety validation
**Fix**: Added validation at function and API layers
**Tests**: 10 new test cases, 95% coverage
**Impact**: 0 similar errors since fix deployed
**Time to Resolution**: 18 minutes

## Key Takeaways

1. **Validate early**: Check inputs at function boundaries
2. **Fail explicitly**: Throw clear errors instead of crashing
3. **Test edge cases**: Always test null, undefined, empty values
4. **Proper HTTP codes**: Use 404 for missing resources, not 500
5. **Document fixes**: Help future developers understand the context

## Related Examples

- [Example 2: Race Condition](example-2-race-condition.md) - Debugging async concurrency issues
- [Example 3: Memory Leak](example-3-memory-leak.md) - Advanced memory leak investigation


---
*Promise: `<promise>EXAMPLE_1_NULL_POINTER_VERIX_COMPLIANT</promise>`*
