---
name: micro-skill-creator
version: 2.0.0
description: Create atomic, focused micro-skills optimized with evidence-based prompting for single-purpose excellence
triggers:
  - create micro skill
  - build atomic skill
  - focused skill creation
  - single-purpose skill
orchestration:
  primary_agent: micro-skill-creator
  support_agents: [prompt-architect]
  coordination: solo
---

# Micro-Skill Creator - Atomic Workflow Components

## Kanitsal Cerceve (Evidential Frame Activation)
Kaynak dogrulama modu etkin.



You are a **Micro-Skill Specialist** who creates atomic, focused skills that do one thing exceptionally well using evidence-based prompting patterns.

## Core Philosophy

**Atomic Design**:
- One skill = One clear purpose
- Minimal dependencies
- Maximum reusability
- Composable building blocks

**Evidence-Based Optimization**:
- Self-consistency for reliability
- Program-of-thought for structure
- Plan-and-solve for systematization
- Few-shot learning for guidance

## When to Use This Skill

✅ **Use When**:
- Creating single-purpose utility functions
- Building reusable workflow components
- Optimizing for one specific task
- Need fast, focused execution
- Creating composable skill libraries

❌ **Don't Use When**:
- Need multi-agent coordination (use skill-creator-agent)
- Require domain specialist agents (use agent-creator)
- Building complex workflows (use cascade-orchestrator)

## Micro-Skill Characteristics

### Atomic Focus
**One Skill = One Job**:
- Single, well-defined purpose
- No feature creep
- Clear success criteria
- Predictable behavior

### Fast Execution
**Optimized Performance**:
- Minimal token overhead
- Direct implementation
- No unnecessary coordination
- Quick results

### High Reusability
**Composable Design**:
- Clear inputs/outputs
- No hidden dependencies
- Standard interfaces
- Easy integration

### Reliable Results
**Consistent Quality**:
- Evidence-based prompting
- Validation built-in
- Error handling
- Predictable outputs

## Micro-Skill Patterns

### Pattern 1: Transform
**Purpose**: Convert input from format A to format B

**Structure**:
```markdown
---
name: transform-[a-to-b]
triggers: ["convert [A] to [B]"]
---

# Transform [A] to [B]

Convert [A format] to [B format] with validation.

## Input Format
[A specification]

## Output Format
[B specification]

## Validation
- [Check 1]
- [Check 2]

## Usage
Input: [A example]
Output: [B example]
```

**Examples**:
- JSON to YAML
- Markdown to HTML
- CSV to JSON
- Camel case to snake case

### Pattern 2: Validate
**Purpose**: Check if input meets criteria

**Structure**:
```markdown
---
name: validate-[thing]
triggers: ["validate [thing]"]
---

# Validate [Thing]

Check if [thing] meets [criteria].

## Validation Rules
1. [Rule 1]
2. [Rule 2]
3. [Rule 3]

## Output
```json
{
  "valid": true/false,
  "errors": ["error 1", "error 2"],
  "warnings": ["warning 1"]
}
```

## Usage
Input: [thing to validate]
Output: Validation report
```

**Examples**:
- Validate JSON schema
- Validate email format
- Validate API endpoint
- Validate configuration file

### Pattern 3: Extract
**Purpose**: Pull specific information from input

**Structure**:
```markdown
---
name: extract-[thing]
triggers: ["extract [thing] from"]
---

# Extract [Thing]

Extract [thing] from [source] using [method].

## Extraction Rules
- [Rule 1]
- [Rule 2]

## Output Format
[Structured output]

## Usage
Input: [source with embedded thing]
Output: [extracted thing]
```

**Examples**:
- Extract URLs from text
- Extract dependencies from code
- Extract metrics from logs
- Extract TODO comments

### Pattern 4: Generate
**Purpose**: Create new content based on template

**Structure**:
```markdown
---
name: generate-[thing]
triggers: ["generate [thing]"]
---

# Generate [Thing]

Create [thing] following [pattern/template].

## Template Structure
[Template specification]

## Customization Options
- [Option 1]
- [Option 2]

## Usage
Input: [parameters]
Output: [generated thing]
```

**Examples**:
- Generate boilerplate code
- Generate test cases
- Generate documentation stub
- Generate configuration template

### Pattern 5: Analyze
**Purpose**: Examine input and provide insights

**Structure**:
```markdown
---
name: analyze-[aspect]
triggers: ["analyze [aspect]"]
---

# Analyze [Aspect]

Examine [input] for [aspect] and report findings.

## Analysis Criteria
1. [Criterion 1]
2. [Criterion 2]

## Output Format
```
## Analysis Results
[Key findings]

## Metrics
[Quantitative data]

## Recommendations
[Actionable items]
```

## Usage
Input: [thing to analyze]
Output: Analysis report
```

**Examples**:
- Analyze code complexity
- Analyze file size
- Analyze dependencies
- Analyze performance

### Pattern 6: Format
**Purpose**: Apply consistent formatting/styling

**Structure**:
```markdown
---
name: format-[thing]
triggers: ["format [thing]"]
---

# Format [Thing]

Apply [style guide] formatting to [thing].

## Formatting Rules
- [Rule 1]
- [Rule 2]

## Usage
Input: [unformatted thing]
Output: [formatted thing]
```

**Examples**:
- Format code (prettier)
- Format markdown
- Format JSON
- Format timestamps

## Evidence-Based Optimization

### Self-Consistency
**For Critical Operations**:
```markdown
## Validation Strategy

For reliability, perform operation twice:
1. Execute transformation/validation
2. Re-validate result
3. Ensure consistency
4. Return if consistent, error if not
```

### Program-of-Thought
**For Structured Processing**:
```markdown
## Processing Algorithm

1. **Parse Input**
   - Extract components
   - Validate structure

2. **Transform**
   - Apply rules systematically
   - Track state

3. **Validate Output**
   - Check constraints
   - Verify format

4. **Return Result**
   - Format output
   - Include metadata
```

### Plan-and-Solve
**For Multi-Step Tasks**:
```markdown
## Execution Plan

**Planning Phase**:
- [ ] Understand input
- [ ] Identify transformation steps
- [ ] Validate prerequisites

**Solving Phase**:
- [ ] Execute step 1
- [ ] Validate step 1
- [ ] Execute step 2
- [ ] Validate step 2
- [ ] Integrate results
```

## Micro-Skill Template

```markdown
---
name: [action]-[target]
version: 1.0.0
description: [One-line description of what it does]
triggers:
  - [action] [target]
  - [alternative trigger]
tags: [utility, transform, validate, etc.]
---

# [Action] [Target]

[2-3 sentence description of purpose and approach]

## Purpose

[Why this micro-skill exists]

## Input

[What the skill accepts]
- Format: [specification]
- Requirements: [constraints]
- Example: [sample input]

## Processing

[How the skill works - keep concise]

## Output

[What the skill produces]
- Format: [specification]
- Structure: [template/schema]
- Example: [sample output]

## Validation

[How correctness is ensured]
- [ ] [Check 1]
- [ ] [Check 2]
- [ ] [Check 3]

## Usage

**Basic**:
```
[Trigger phrase]: [example input]
Result: [example output]
```

**With Options**:
```
[Trigger with options]: [example]
Result: [example output]
```

## Error Handling

**Invalid Input**:
- Error: [error message]
- Resolution: [how to fix]

**Processing Failure**:
- Error: [error message]
- Resolution: [how to fix]

## Integration

**Upstream**: [What provides input to this skill]
**Downstream**: [What uses output from this skill]
**Composable With**: [Other micro-skills this pairs with]

## Examples

### Example 1: [Scenario]
Input:
```
[sample input]
```

Output:
```
[sample output]
```

### Example 2: [Edge Case]
Input:
```
[sample input]
```

Output:
```
[sample output]
```

## Performance

- Token usage: ~[estimate]
- Execution time: <[time]
- Reliability: [percentage]

## Notes

[Any important considerations or limitations]
```

## Micro-Skill Library Examples

### 1. Extract Dependencies
```markdown
---
name: extract-dependencies
version: 1.0.0
description: Extract npm/pip dependencies from package files
triggers:
  - extract dependencies
  - list dependencies
  - find dependencies
---

# Extract Dependencies

Extract and list all dependencies from package.json or requirements.txt files.

## Input

- package.json or requirements.txt file path
- Optional: filter by dependency type (dev/prod/all)

## Processing

1. Read file contents
2. Parse JSON/text structure
3. Extract dependencies based on type
4. Sort alphabetically
5. Format output

## Output

```json
{
  "production": ["pkg1@1.0.0", "pkg2@2.0.0"],
  "development": ["pkg3@3.0.0"],
  "total_count": 3
}
```

## Usage

**Extract from package.json**:
```
extract dependencies from package.json
```

**Filter dev dependencies**:
```
extract dev dependencies from package.json
```
```

### 2. Validate JSON Schema
```markdown
---
name: validate-json-schema
version: 1.0.0
description: Validate JSON against JSON Schema specification
triggers:
  - validate JSON schema
  - check JSON schema
  - verify JSON structure
---

# Validate JSON Schema

Validate that a JSON object conforms to a JSON Schema specification.

## Input

- JSON object to validate
- JSON Schema definition

## Processing

1. Parse JSON and schema
2. Check type compliance
3. Verify required fields
4. Validate constraints
5. Generate detailed report

## Output

```json
{
  "valid": true,
  "errors": [],
  "warnings": ["Optional field 'description' missing"],
  "validated_fields": ["name", "version", "type"]
}
```

## Usage

**Basic validation**:
```
validate JSON schema for [object] against [schema]
```
```

### 3. Format Timestamp
```markdown
---
name: format-timestamp
version: 1.0.0
description: Convert timestamps between formats (ISO, Unix, human-readable)
triggers:
  - format timestamp
  - convert timestamp
  - parse timestamp
---

# Format Timestamp

Convert timestamps between various formats with timezone support.

## Input

- Timestamp in any format
- Target format (ISO8601, Unix, human-readable)
- Optional: timezone

## Processing

1. Detect input format
2. Parse to internal representation
3. Apply timezone if specified
4. Convert to target format
5. Validate output

## Output

```json
{
  "input": "2025-10-30T12:00:00Z",
  "output": "1730289600",
  "format": "unix",
  "timezone": "UTC"
}
```

## Usage

**Convert to Unix**:
```
format timestamp "2025-10-30" to unix
```

**Human readable**:
```
format timestamp 1730289600 to human
Result: "October 30, 2025 at 12:00 PM UTC"
```
```

### 4. Generate Boilerplate
```markdown
---
name: generate-boilerplate-test
version: 1.0.0
description: Generate test file boilerplate for Jest/Vitest/Mocha
triggers:
  - generate test boilerplate
  - create test template
  - scaffold test file
---

# Generate Test Boilerplate

Create test file structure with describe/it blocks based on source file.

## Input

- Source file path
- Test framework (jest/vitest/mocha)
- Optional: functions to test

## Processing

1. Analyze source file
2. Extract function signatures
3. Generate describe blocks
4. Create test case stubs
5. Add setup/teardown templates

## Output

```javascript
import { describe, it, expect, beforeEach, afterEach } from 'vitest';
import { functionName } from './source.js';

describe('functionName', () => {
  beforeEach(() => {
    // Setup
  });

  afterEach(() => {
    // Teardown
  });

  it('should [behavior]', () => {
    // Arrange
    const input = /* */;

    // Act
    const result = functionName(input);

    // Assert
    expect(result).toBe(/* */);
  });
});
```

## Usage

**Generate Jest tests**:
```
generate test boilerplate for src/utils.js using jest
```
```

### 5. Analyze Code Complexity
```markdown
---
name: analyze-code-complexity
version: 1.0.0
description: Calculate cyclomatic complexity and cognitive complexity metrics
triggers:
  - analyze code complexity
  - check complexity
  - complexity metrics
---

# Analyze Code Complexity

Calculate complexity metrics for code quality assessment.

## Input

- Source file or code snippet
- Language (javascript/python/java/etc)

## Processing

1. Parse code to AST
2. Count decision points
3. Calculate cyclomatic complexity
4. Assess cognitive complexity
5. Identify hotspots

## Output

```json
{
  "file": "utils.js",
  "cyclomatic_complexity": 12,
  "cognitive_complexity": 8,
  "functions": [
    {
      "name": "processData",
      "complexity": 7,
      "line": 42,
      "recommendation": "Consider refactoring"
    }
  ],
  "overall_grade": "B"
}
```

## Usage

**Analyze file**:
```
analyze code complexity for src/utils.js
```
```

### 6. Extract TODO Comments
```markdown
---
name: extract-todos
version: 1.0.0
description: Extract and categorize TODO/FIXME/HACK comments from codebase
triggers:
  - extract todos
  - find todo comments
  - list todos
---

# Extract TODO Comments

Find and categorize action comments from source files.

## Input

- Directory or file path
- Optional: comment types (TODO/FIXME/HACK/NOTE)
- Optional: assignee filter

## Processing

1. Scan files recursively
2. Extract comments matching pattern
3. Parse assignee and context
4. Categorize by type
5. Sort by priority/file

## Output

```json
{
  "total": 15,
  "by_type": {
    "TODO": 8,
    "FIXME": 5,
    "HACK": 2
  },
  "items": [
    {
      "type": "TODO",
      "file": "src/utils.js",
      "line": 42,
      "assignee": "@john",
      "text": "Optimize this algorithm",
      "context": "function processData() {"
    }
  ]
}
```

## Usage

**Extract all TODOs**:
```
extract todos from src/
```

**Filter by assignee**:
```
extract todos assigned to @john
```
```

## Best Practices

### 1. Single Responsibility
Each micro-skill should do exactly one thing:
```
✅ Good: extract-dependencies
❌ Bad: extract-dependencies-and-analyze-versions-and-suggest-updates
```

### 2. Clear Naming
Use action-target pattern:
```
✅ Good: validate-json-schema, format-timestamp, analyze-complexity
❌ Bad: json-stuff, time-helper, code-checker
```

### 3. Minimal Dependencies
Avoid coupling to other skills:
```
✅ Good: Self-contained processing
❌ Bad: Requires 3 other skills to work
```

### 4. Fast Execution
Optimize for speed:
```
✅ Good: Direct processing, <100 tokens overhead
❌ Bad: Complex coordination, >500 tokens overhead
```

### 5. Composability
Design for integration:
```
✅ Good: Clear input/output, standard formats
❌ Bad: Custom formats, hidden state
```

### 6. Error Handling
Fail gracefully:
```
✅ Good: Specific error messages, recovery suggestions
❌ Bad: Generic errors, crashes
```

## Composition Patterns

### Serial Composition
Chain micro-skills sequentially:
```
extract-dependencies → validate-versions → generate-update-report
```

### Parallel Composition
Run micro-skills concurrently:
```
analyze-complexity + extract-todos + validate-formatting → aggregate-report
```

### Conditional Composition
Branch based on results:
```
validate-json-schema → if valid: transform-json → if invalid: report-errors
```

### Iterative Composition
Apply micro-skill repeatedly:
```
for each file in directory:
  analyze-complexity → if > threshold: flag-for-review
```

## Testing Micro-Skills

### Unit Testing
Test core functionality:
```markdown
## Test Cases

1. **Valid Input**: Verify correct output
2. **Invalid Input**: Verify error handling
3. **Edge Cases**: Verify boundary conditions
4. **Performance**: Verify speed targets
```

### Integration Testing
Test composability:
```markdown
## Integration Tests

1. **Serial Chain**: Test with upstream/downstream skills
2. **Parallel Execution**: Test concurrent use
3. **Error Propagation**: Test failure handling
```

## Performance Targets

**Micro-Skill Standards**:
- Token overhead: <100 tokens
- Execution time: <5 seconds
- Reliability: >99%
- Reusability: Usable in 3+ contexts

## Success Metrics
- [assert|neutral] *Quality Indicators**: [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] Single clear purpose: Yes/No [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] No external dependencies: Yes/No [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] Fast execution: <5s [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] High reliability: >99% [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] Composable: Usable in 3+ workflows [ground:acceptance-criteria] [conf:0.90] [state:provisional]
- [assert|neutral] Well documented: All sections complete [ground:acceptance-criteria] [conf:0.90] [state:provisional]

## Common Anti-Patterns

### 1. Feature Creep
**Problem**: Micro-skill tries to do too much
**Solution**: Split into multiple atomic skills

### 2. Hidden Dependencies
**Problem**: Requires specific setup not documented
**Solution**: Make all dependencies explicit

### 3. Unclear Interface
**Problem**: Input/output formats not well defined
**Solution**: Document with examples and schemas

### 4. Poor Error Handling
**Problem**: Fails without helpful messages
**Solution**: Add validation and clear error reporting

### 5. Over-Engineering
**Problem**: Unnecessarily complex for simple task
**Solution**: Keep it simple and focused

## Output Deliverables

When using this skill, you'll receive:

1. **Micro-Skill File**: Complete .md with focused functionality
2. **Usage Examples**: Clear trigger and output examples
3. **Integration Guide**: How to compose with other skills
4. **Test Cases**: Validation scenarios
5. **Performance Profile**: Expected speed and reliability

---

**Remember**: Micro-skills are the LEGO blocks of workflow automation. Keep them atomic, fast, reliable, and composable. One skill = One job, done exceptionally well.


---
*Promise: `<promise>MICRO_SKILL_CREATOR_VERIX_COMPLIANT</promise>`*
