# Event Callbacks

## Overview

SeaTalk sends HTTP POST requests to your configured callback URL when events occur. Your server must respond within 5 seconds.

## Setup Callback URL

1. Configure callback URL in SeaTalk Open Platform app settings
2. SeaTalk sends verification request
3. Your server must respond correctly to complete verification

## Callback URL Verification

### Verification Request
```json
{
  "event_type": "event_verification",
  "seatalk_challenge": "abc123xyz"
}
```

### Required Response
Return HTTP 200 with the challenge value in the body:
```
abc123xyz
```

**Rules:**
- Must respond within 5 seconds
- Maximum 3 retries
- Must return exact `seatalk_challenge` value

---

## Signature Verification

Verify that requests are from SeaTalk using the Signing Secret.

### Algorithm
```
SHA-256(request_body + signing_secret) = lowercase hex string
```

### Python Example
```python
import hashlib
import json

def verify_seatalk_signature(request_body: str, signing_secret: str, signature_header: str) -> bool:
    """
    Verify that the request is from SeaTalk.

    Args:
        request_body: Raw request body string
        signing_secret: Your app's signing secret from SeaTalk
        signature_header: Value from 'Signature' header

    Returns:
        True if signature is valid
    """
    computed = hashlib.sha256(
        (request_body + signing_secret).encode()
    ).hexdigest()

    return computed.lower() == signature_header.lower()
```

### Flask Example
```python
from flask import Flask, request, jsonify
import hashlib
import json

app = Flask(__name__)
SIGNING_SECRET = "your_signing_secret"

@app.route('/callback', methods=['POST'])
def handle_callback():
    # Get signature from header
    signature = request.headers.get('Signature', '')

    # Get raw body
    body = request.get_data(as_text=True)

    # Verify signature
    computed = hashlib.sha256(
        (body + SIGNING_SECRET).encode()
    ).hexdigest()

    if computed.lower() != signature.lower():
        return 'Invalid signature', 401

    # Parse event
    data = json.loads(body)
    event_type = data.get('event_type')

    # Handle verification
    if event_type == 'event_verification':
        return data.get('seatalk_challenge', '')

    # Handle other events
    if event_type == 'bot_added_to_group_chat':
        group_id = data['event']['group']['group_id']
        # Process event...

    return 'OK', 200

if __name__ == '__main__':
    app.run(port=5000)
```

---

## Event Types

### event_verification
Initial callback URL verification.

### new_bot_subscriber
User subscribed to the bot (started 1-on-1 chat).

### message_from_bot_subscriber
User sent a message in 1-on-1 bot chat.

### interactive_message_click
User clicked a button on an interactive message card.

### bot_added_to_group_chat
Bot was added to a group chat.

### bot_removed_from_group_chat
Bot was removed from a group chat.

### new_mentioned_message_received_from_group_chat
Bot was @mentioned in a group chat.

---

## Event: Bot Added To Group Chat

Triggered when a user adds your bot to a group chat.

### Payload
```json
{
  "event_id": "1234567",
  "event_type": "bot_added_to_group_chat",
  "timestamp": 1687764109,
  "app_id": "your_app_id",
  "event": {
    "group": {
      "group_id": "qwertyui",
      "group_name": "Test Group",
      "group_settings": {
        "chat_history_for_new_members": "disabled",
        "can_notify_with_at_all": false,
        "can_view_member_list": false
      }
    },
    "inviter": {
      "seatalk_id": "1234567890",
      "employee_code": "e_12345678",
      "email": "inviter@company.com"
    }
  }
}
```

### Fields

| Field | Type | Description |
|-------|------|-------------|
| event_id | string | Unique event identifier |
| event_type | string | "bot_added_to_group_chat" |
| timestamp | int | Unix timestamp |
| app_id | string | Your app ID |
| event.group.group_id | string | Group chat ID |
| event.group.group_name | string | Group name |
| event.group.group_settings | object | Group configuration |
| event.inviter | object | User who added the bot |

### Note
This event is NOT triggered when the bot creates a group via API.

---

## Event: Bot Removed From Group Chat

Triggered when the bot is removed from a group.

### Payload
```json
{
  "event_id": "1234567",
  "event_type": "bot_removed_from_group_chat",
  "timestamp": 1687764109,
  "app_id": "your_app_id",
  "event": {
    "group": {
      "group_id": "qwertyui"
    }
  }
}
```

---

## Event: Interactive Message Click

Triggered when a user clicks a callback button on an interactive card.

### Payload
```json
{
  "event_id": "1234567",
  "event_type": "interactive_message_click",
  "timestamp": 1687764109,
  "app_id": "your_app_id",
  "event": {
    "message_id": "msg123",
    "button_value": "approved",
    "clicker": {
      "seatalk_id": "123456",
      "employee_code": "e_abc"
    }
  }
}
```

---

## Event: New Mentioned Message From Group Chat

Triggered when the bot is @mentioned in a group.

### Payload
```json
{
  "event_id": "1234567",
  "event_type": "new_mentioned_message_received_from_group_chat",
  "timestamp": 1687764109,
  "app_id": "your_app_id",
  "event": {
    "group_id": "group123",
    "message_id": "msg123",
    "thread_id": "thread123",
    "sender": {
      "seatalk_id": "123456",
      "employee_code": "e_abc"
    },
    "message": {
      "tag": "text",
      "text": {
        "plain_text": "@Bot help me with this"
      }
    }
  }
}
```

---

## Response Requirements

- Return HTTP 200 within 5 seconds
- SeaTalk retries up to 3 times on failure
- No specific response body required (except for verification)
