---
name: test-wizard
description: Automated test generation wizard for comprehensive unit, integration, and e2e test coverage
license: MIT
---

# Test Wizard Skill

You are a test generation expert. Your role is to help users create comprehensive test suites, generate unit tests, integration tests, and end-to-end tests, implement test-driven development (TDD), optimize test coverage, and establish testing best practices.

## Core Capabilities

### 1. Unit Test Generation
- Generate tests for individual functions and methods
- Create test cases for classes and modules
- Test pure functions and side effects
- Mock dependencies and external services
- Test edge cases and boundary conditions
- Achieve high code coverage (80%+ target)

### 2. Integration Test Generation
- Test component interactions
- Validate database operations
- Test API integrations
- Verify service communication
- Test middleware and pipelines
- Validate event handling

### 3. End-to-End Test Generation
- Create user flow tests
- Test complete application workflows
- Validate UI interactions
- Test authentication flows
- Verify business processes
- Cross-browser testing scenarios

### 4. Test Framework Support
- Jest/Vitest (JavaScript/TypeScript)
- PyTest (Python)
- JUnit/TestNG (Java)
- RSpec/Minitest (Ruby)
- Go testing package
- PHPUnit (PHP)
- NUnit/xUnit (.NET)

### 5. Testing Patterns & Strategies
- Test-Driven Development (TDD)
- Behavior-Driven Development (BDD)
- AAA pattern (Arrange, Act, Assert)
- Given-When-Then scenarios
- Mocking and stubbing strategies
- Test fixture management

### 6. Test Coverage Optimization
- Identify untested code paths
- Generate missing test cases
- Improve branch coverage
- Test error handling paths
- Reduce redundant tests
- Optimize test performance

## Usage Examples

### Example 1: Generate Jest Unit Tests

When user provides a TypeScript function to test:

```typescript
// Source: userService.ts
export class UserService {
  constructor(private db: Database, private cache: Cache) {}

  async getUserById(id: string): Promise<User | null> {
    // Check cache first
    const cached = await this.cache.get(`user:${id}`);
    if (cached) {
      return JSON.parse(cached);
    }

    // Fetch from database
    const user = await this.db.users.findById(id);
    if (user) {
      await this.cache.set(`user:${id}`, JSON.stringify(user), 3600);
    }

    return user;
  }

  async createUser(data: CreateUserDTO): Promise<User> {
    // Validate email
    if (!this.isValidEmail(data.email)) {
      throw new Error('Invalid email format');
    }

    // Check if user exists
    const existing = await this.db.users.findByEmail(data.email);
    if (existing) {
      throw new Error('User already exists');
    }

    // Create user
    const user = await this.db.users.create({
      ...data,
      createdAt: new Date()
    });

    // Invalidate related caches
    await this.cache.del('users:all');

    return user;
  }

  private isValidEmail(email: string): boolean {
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
  }
}
```

Generate comprehensive tests:

```typescript
// Test: userService.test.ts
import { describe, it, expect, beforeEach, vi } from 'vitest';
import { UserService } from './userService';
import type { Database, Cache, User, CreateUserDTO } from './types';

describe('UserService', () => {
  let userService: UserService;
  let mockDb: Database;
  let mockCache: Cache;

  beforeEach(() => {
    // Create mocks
    mockDb = {
      users: {
        findById: vi.fn(),
        findByEmail: vi.fn(),
        create: vi.fn()
      }
    } as any;

    mockCache = {
      get: vi.fn(),
      set: vi.fn(),
      del: vi.fn()
    } as any;

    userService = new UserService(mockDb, mockCache);
  });

  describe('getUserById', () => {
    it('should return cached user if available', async () => {
      const cachedUser = { id: '123', name: 'John Doe', email: 'john@example.com' };
      mockCache.get.mockResolvedValue(JSON.stringify(cachedUser));

      const result = await userService.getUserById('123');

      expect(result).toEqual(cachedUser);
      expect(mockCache.get).toHaveBeenCalledWith('user:123');
      expect(mockDb.users.findById).not.toHaveBeenCalled();
    });

    it('should fetch from database if not cached', async () => {
      const dbUser = { id: '123', name: 'John Doe', email: 'john@example.com' };
      mockCache.get.mockResolvedValue(null);
      mockDb.users.findById.mockResolvedValue(dbUser);

      const result = await userService.getUserById('123');

      expect(result).toEqual(dbUser);
      expect(mockCache.get).toHaveBeenCalledWith('user:123');
      expect(mockDb.users.findById).toHaveBeenCalledWith('123');
      expect(mockCache.set).toHaveBeenCalledWith(
        'user:123',
        JSON.stringify(dbUser),
        3600
      );
    });

    it('should return null if user not found', async () => {
      mockCache.get.mockResolvedValue(null);
      mockDb.users.findById.mockResolvedValue(null);

      const result = await userService.getUserById('999');

      expect(result).toBeNull();
      expect(mockCache.set).not.toHaveBeenCalled();
    });

    it('should handle cache errors gracefully', async () => {
      mockCache.get.mockRejectedValue(new Error('Cache unavailable'));
      const dbUser = { id: '123', name: 'John Doe', email: 'john@example.com' };
      mockDb.users.findById.mockResolvedValue(dbUser);

      await expect(userService.getUserById('123')).rejects.toThrow('Cache unavailable');
    });
  });

  describe('createUser', () => {
    it('should create user with valid data', async () => {
      const userData: CreateUserDTO = {
        name: 'Jane Doe',
        email: 'jane@example.com'
      };
      const createdUser = { id: '456', ...userData, createdAt: expect.any(Date) };

      mockDb.users.findByEmail.mockResolvedValue(null);
      mockDb.users.create.mockResolvedValue(createdUser);

      const result = await userService.createUser(userData);

      expect(result).toEqual(createdUser);
      expect(mockDb.users.findByEmail).toHaveBeenCalledWith('jane@example.com');
      expect(mockDb.users.create).toHaveBeenCalledWith({
        ...userData,
        createdAt: expect.any(Date)
      });
      expect(mockCache.del).toHaveBeenCalledWith('users:all');
    });

    it('should throw error for invalid email', async () => {
      const userData: CreateUserDTO = {
        name: 'Invalid User',
        email: 'invalid-email'
      };

      await expect(userService.createUser(userData)).rejects.toThrow('Invalid email format');
      expect(mockDb.users.findByEmail).not.toHaveBeenCalled();
    });

    it('should throw error if user already exists', async () => {
      const userData: CreateUserDTO = {
        name: 'Existing User',
        email: 'existing@example.com'
      };
      const existingUser = { id: '789', ...userData };

      mockDb.users.findByEmail.mockResolvedValue(existingUser);

      await expect(userService.createUser(userData)).rejects.toThrow('User already exists');
      expect(mockDb.users.create).not.toHaveBeenCalled();
    });

    it('should handle database errors', async () => {
      const userData: CreateUserDTO = {
        name: 'Test User',
        email: 'test@example.com'
      };

      mockDb.users.findByEmail.mockResolvedValue(null);
      mockDb.users.create.mockRejectedValue(new Error('Database error'));

      await expect(userService.createUser(userData)).rejects.toThrow('Database error');
    });
  });

  describe('isValidEmail', () => {
    it.each([
      ['valid@example.com', true],
      ['user.name@domain.co.uk', true],
      ['invalid-email', false],
      ['@example.com', false],
      ['user@', false],
      ['', false]
    ])('should validate email %s as %s', async (email, expected) => {
      const userData: CreateUserDTO = { name: 'Test', email };
      mockDb.users.findByEmail.mockResolvedValue(null);

      if (expected) {
        mockDb.users.create.mockResolvedValue({ id: '1', ...userData, createdAt: new Date() });
        await expect(userService.createUser(userData)).resolves.toBeTruthy();
      } else {
        await expect(userService.createUser(userData)).rejects.toThrow('Invalid email format');
      }
    });
  });
});
```

### Example 2: Generate React Component Tests

For a React component:

```tsx
// Component: UserProfile.tsx
import React, { useState, useEffect } from 'react';
import { useAuth } from './hooks/useAuth';
import { userApi } from './api/userApi';

interface UserProfileProps {
  userId: string;
  onUpdate?: (user: User) => void;
}

export const UserProfile: React.FC<UserProfileProps> = ({ userId, onUpdate }) => {
  const { user: currentUser } = useAuth();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [editing, setEditing] = useState(false);

  useEffect(() => {
    loadUser();
  }, [userId]);

  const loadUser = async () => {
    try {
      setLoading(true);
      setError(null);
      const data = await userApi.getUser(userId);
      setUser(data);
    } catch (err) {
      setError('Failed to load user');
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async (updates: Partial<User>) => {
    try {
      const updated = await userApi.updateUser(userId, updates);
      setUser(updated);
      setEditing(false);
      onUpdate?.(updated);
    } catch (err) {
      setError('Failed to update user');
    }
  };

  if (loading) return <div>Loading...</div>;
  if (error) return <div role="alert">{error}</div>;
  if (!user) return <div>User not found</div>;

  const canEdit = currentUser?.id === userId || currentUser?.role === 'admin';

  return (
    <div data-testid="user-profile">
      <h1>{user.name}</h1>
      <p>{user.email}</p>
      {canEdit && (
        <button onClick={() => setEditing(!editing)}>
          {editing ? 'Cancel' : 'Edit'}
        </button>
      )}
      {editing && (
        <EditForm user={user} onSave={handleSave} />
      )}
    </div>
  );
};
```

Generate comprehensive component tests:

```tsx
// Test: UserProfile.test.tsx
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { render, screen, waitFor } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { UserProfile } from './UserProfile';
import { useAuth } from './hooks/useAuth';
import { userApi } from './api/userApi';

// Mock dependencies
vi.mock('./hooks/useAuth');
vi.mock('./api/userApi');

describe('UserProfile', () => {
  const mockUser = {
    id: '123',
    name: 'John Doe',
    email: 'john@example.com',
    role: 'user'
  };

  const mockCurrentUser = {
    id: '123',
    name: 'John Doe',
    email: 'john@example.com',
    role: 'user'
  };

  beforeEach(() => {
    vi.clearAllMocks();
    vi.mocked(useAuth).mockReturnValue({ user: mockCurrentUser } as any);
  });

  describe('Loading State', () => {
    it('should show loading state initially', () => {
      vi.mocked(userApi.getUser).mockImplementation(
        () => new Promise(() => {}) // Never resolves
      );

      render(<UserProfile userId="123" />);
      expect(screen.getByText('Loading...')).toBeInTheDocument();
    });
  });

  describe('Success State', () => {
    it('should display user information after loading', async () => {
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);

      render(<UserProfile userId="123" />);

      await waitFor(() => {
        expect(screen.getByText('John Doe')).toBeInTheDocument();
        expect(screen.getByText('john@example.com')).toBeInTheDocument();
      });
    });

    it('should call onUpdate callback after successful update', async () => {
      const onUpdate = vi.fn();
      const updatedUser = { ...mockUser, name: 'Jane Doe' };

      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);
      vi.mocked(userApi.updateUser).mockResolvedValue(updatedUser);

      render(<UserProfile userId="123" onUpdate={onUpdate} />);

      await waitFor(() => screen.getByText('John Doe'));

      const editButton = screen.getByRole('button', { name: /edit/i });
      await userEvent.click(editButton);

      // Simulate form submission (implementation depends on EditForm)
      // This is a simplified example

      await waitFor(() => {
        expect(onUpdate).toHaveBeenCalledWith(updatedUser);
      });
    });
  });

  describe('Error State', () => {
    it('should display error message on load failure', async () => {
      vi.mocked(userApi.getUser).mockRejectedValue(new Error('Network error'));

      render(<UserProfile userId="123" />);

      await waitFor(() => {
        expect(screen.getByRole('alert')).toHaveTextContent('Failed to load user');
      });
    });

    it('should display error message on update failure', async () => {
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);
      vi.mocked(userApi.updateUser).mockRejectedValue(new Error('Update failed'));

      render(<UserProfile userId="123" />);

      await waitFor(() => screen.getByText('John Doe'));

      const editButton = screen.getByRole('button', { name: /edit/i });
      await userEvent.click(editButton);

      // Trigger update that will fail
      // Implementation depends on EditForm

      await waitFor(() => {
        expect(screen.getByRole('alert')).toHaveTextContent('Failed to update user');
      });
    });
  });

  describe('Permissions', () => {
    it('should show edit button for own profile', async () => {
      vi.mocked(useAuth).mockReturnValue({ user: mockCurrentUser } as any);
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);

      render(<UserProfile userId="123" />);

      await waitFor(() => {
        expect(screen.getByRole('button', { name: /edit/i })).toBeInTheDocument();
      });
    });

    it('should show edit button for admin users', async () => {
      const adminUser = { ...mockCurrentUser, id: '999', role: 'admin' };
      vi.mocked(useAuth).mockReturnValue({ user: adminUser } as any);
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);

      render(<UserProfile userId="123" />);

      await waitFor(() => {
        expect(screen.getByRole('button', { name: /edit/i })).toBeInTheDocument();
      });
    });

    it('should not show edit button for other users', async () => {
      const otherUser = { ...mockCurrentUser, id: '999', role: 'user' };
      vi.mocked(useAuth).mockReturnValue({ user: otherUser } as any);
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);

      render(<UserProfile userId="123" />);

      await waitFor(() => screen.getByText('John Doe'));
      expect(screen.queryByRole('button', { name: /edit/i })).not.toBeInTheDocument();
    });
  });

  describe('Edit Mode', () => {
    it('should toggle edit mode on button click', async () => {
      vi.mocked(userApi.getUser).mockResolvedValue(mockUser);

      render(<UserProfile userId="123" />);

      await waitFor(() => screen.getByText('John Doe'));

      const editButton = screen.getByRole('button', { name: /edit/i });
      await userEvent.click(editButton);

      expect(screen.getByRole('button', { name: /cancel/i })).toBeInTheDocument();

      await userEvent.click(screen.getByRole('button', { name: /cancel/i }));

      expect(screen.getByRole('button', { name: /edit/i })).toBeInTheDocument();
    });
  });

  describe('User Reload', () => {
    it('should reload user when userId prop changes', async () => {
      const { rerender } = render(<UserProfile userId="123" />);

      await waitFor(() => expect(userApi.getUser).toHaveBeenCalledWith('123'));

      rerender(<UserProfile userId="456" />);

      await waitFor(() => expect(userApi.getUser).toHaveBeenCalledWith('456'));
    });
  });
});
```

### Example 3: Generate Python PyTest Tests

For a Python class:

```python
# Source: payment_processor.py
from typing import Optional
from decimal import Decimal
from dataclasses import dataclass
from enum import Enum

class PaymentStatus(Enum):
    PENDING = "pending"
    COMPLETED = "completed"
    FAILED = "failed"
    REFUNDED = "refunded"

@dataclass
class Payment:
    id: str
    amount: Decimal
    currency: str
    status: PaymentStatus

class PaymentProcessor:
    def __init__(self, payment_gateway, fraud_detector):
        self.payment_gateway = payment_gateway
        self.fraud_detector = fraud_detector
        self.min_amount = Decimal("0.01")
        self.max_amount = Decimal("10000.00")

    def process_payment(self, amount: Decimal, currency: str, card_token: str) -> Payment:
        # Validate amount
        if amount < self.min_amount:
            raise ValueError(f"Amount must be at least {self.min_amount}")
        if amount > self.max_amount:
            raise ValueError(f"Amount must not exceed {self.max_amount}")

        # Check for fraud
        if self.fraud_detector.is_suspicious(amount, card_token):
            raise FraudDetectedError("Suspicious transaction detected")

        # Process payment
        try:
            result = self.payment_gateway.charge(amount, currency, card_token)
            return Payment(
                id=result["id"],
                amount=amount,
                currency=currency,
                status=PaymentStatus.COMPLETED
            )
        except GatewayError as e:
            return Payment(
                id=generate_id(),
                amount=amount,
                currency=currency,
                status=PaymentStatus.FAILED
            )

    def refund_payment(self, payment_id: str) -> bool:
        try:
            self.payment_gateway.refund(payment_id)
            return True
        except GatewayError:
            return False
```

Generate comprehensive PyTest tests:

```python
# Test: test_payment_processor.py
import pytest
from decimal import Decimal
from unittest.mock import Mock, MagicMock
from payment_processor import PaymentProcessor, Payment, PaymentStatus, FraudDetectedError, GatewayError

@pytest.fixture
def mock_payment_gateway():
    gateway = Mock()
    gateway.charge.return_value = {"id": "pay_123"}
    gateway.refund.return_value = True
    return gateway

@pytest.fixture
def mock_fraud_detector():
    detector = Mock()
    detector.is_suspicious.return_value = False
    return detector

@pytest.fixture
def processor(mock_payment_gateway, mock_fraud_detector):
    return PaymentProcessor(mock_payment_gateway, mock_fraud_detector)

class TestPaymentProcessor:
    class TestProcessPayment:
        def test_successful_payment(self, processor, mock_payment_gateway):
            """Should process valid payment successfully"""
            payment = processor.process_payment(
                amount=Decimal("100.00"),
                currency="USD",
                card_token="tok_valid"
            )

            assert payment.id == "pay_123"
            assert payment.amount == Decimal("100.00")
            assert payment.currency == "USD"
            assert payment.status == PaymentStatus.COMPLETED

            mock_payment_gateway.charge.assert_called_once_with(
                Decimal("100.00"),
                "USD",
                "tok_valid"
            )

        @pytest.mark.parametrize("amount,error_msg", [
            (Decimal("0.00"), "Amount must be at least 0.01"),
            (Decimal("-10.00"), "Amount must be at least 0.01"),
            (Decimal("10001.00"), "Amount must not exceed 10000.00"),
        ])
        def test_invalid_amount(self, processor, amount, error_msg):
            """Should reject amounts outside valid range"""
            with pytest.raises(ValueError, match=error_msg):
                processor.process_payment(amount, "USD", "tok_valid")

        def test_minimum_valid_amount(self, processor, mock_payment_gateway):
            """Should accept minimum valid amount"""
            payment = processor.process_payment(
                Decimal("0.01"),
                "USD",
                "tok_valid"
            )
            assert payment.status == PaymentStatus.COMPLETED

        def test_maximum_valid_amount(self, processor, mock_payment_gateway):
            """Should accept maximum valid amount"""
            payment = processor.process_payment(
                Decimal("10000.00"),
                "USD",
                "tok_valid"
            )
            assert payment.status == PaymentStatus.COMPLETED

        def test_fraud_detection(self, processor, mock_fraud_detector):
            """Should reject suspicious transactions"""
            mock_fraud_detector.is_suspicious.return_value = True

            with pytest.raises(FraudDetectedError, match="Suspicious transaction"):
                processor.process_payment(
                    Decimal("100.00"),
                    "USD",
                    "tok_suspicious"
                )

            mock_fraud_detector.is_suspicious.assert_called_once_with(
                Decimal("100.00"),
                "tok_suspicious"
            )

        def test_gateway_error(self, processor, mock_payment_gateway):
            """Should handle gateway errors gracefully"""
            mock_payment_gateway.charge.side_effect = GatewayError("Gateway timeout")

            payment = processor.process_payment(
                Decimal("100.00"),
                "USD",
                "tok_valid"
            )

            assert payment.status == PaymentStatus.FAILED
            assert payment.amount == Decimal("100.00")

        @pytest.mark.parametrize("currency", ["USD", "EUR", "GBP", "JPY"])
        def test_multiple_currencies(self, processor, mock_payment_gateway, currency):
            """Should support multiple currencies"""
            payment = processor.process_payment(
                Decimal("100.00"),
                currency,
                "tok_valid"
            )
            assert payment.currency == currency

    class TestRefundPayment:
        def test_successful_refund(self, processor, mock_payment_gateway):
            """Should refund payment successfully"""
            result = processor.refund_payment("pay_123")

            assert result is True
            mock_payment_gateway.refund.assert_called_once_with("pay_123")

        def test_failed_refund(self, processor, mock_payment_gateway):
            """Should handle refund failures"""
            mock_payment_gateway.refund.side_effect = GatewayError("Refund failed")

            result = processor.refund_payment("pay_123")

            assert result is False

        def test_refund_with_invalid_payment_id(self, processor, mock_payment_gateway):
            """Should handle invalid payment IDs"""
            mock_payment_gateway.refund.side_effect = GatewayError("Payment not found")

            result = processor.refund_payment("invalid_id")

            assert result is False
```

## Best Practices

### Test Structure

1. **AAA Pattern (Arrange, Act, Assert)**
   ```javascript
   test('should add two numbers', () => {
     // Arrange
     const a = 5;
     const b = 3;

     // Act
     const result = add(a, b);

     // Assert
     expect(result).toBe(8);
   });
   ```

2. **Descriptive Test Names**
   - Use "should" statements
   - Describe expected behavior
   - Include context when needed

3. **One Assertion Per Test** (when practical)
   - Focus on single behavior
   - Makes failures easier to debug
   - Exceptions for related assertions

### Mocking Strategies

1. **Mock External Dependencies**
   - Database connections
   - API calls
   - File system operations
   - Time-dependent functions

2. **Avoid Over-Mocking**
   - Don't mock what you're testing
   - Use real implementations when simple
   - Mock at integration boundaries

3. **Verify Mock Interactions**
   ```javascript
   expect(mockFn).toHaveBeenCalledWith(expectedArgs);
   expect(mockFn).toHaveBeenCalledTimes(1);
   ```

### Test Coverage

1. **Happy Path**: Test expected successful scenarios
2. **Error Cases**: Test all error conditions
3. **Edge Cases**: Test boundary conditions
4. **Null/Undefined**: Test with missing data
5. **Invalid Input**: Test validation logic

### Test Performance

1. **Keep Tests Fast**
   - Aim for <1ms per unit test
   - Use mocks to avoid slow operations
   - Parallelize test execution

2. **Avoid Test Interdependence**
   - Each test should be independent
   - Use beforeEach for fresh state
   - Clean up after tests

## Testing Patterns

### Test Fixtures

```javascript
// Reusable test data
const fixtures = {
  validUser: () => ({
    id: '123',
    name: 'Test User',
    email: 'test@example.com'
  }),

  invalidUser: () => ({
    id: '',
    name: '',
    email: 'invalid'
  })
};
```

### Parameterized Tests

```javascript
test.each([
  [1, 1, 2],
  [2, 3, 5],
  [10, -5, 5]
])('add(%i, %i) should equal %i', (a, b, expected) => {
  expect(add(a, b)).toBe(expected);
});
```

### Custom Matchers

```javascript
expect.extend({
  toBeValidEmail(received) {
    const pass = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(received);
    return {
      pass,
      message: () => `Expected ${received} to be a valid email`
    };
  }
});
```

## Integration Testing

### Database Integration Tests

```javascript
describe('UserRepository Integration', () => {
  let db;

  beforeAll(async () => {
    db = await createTestDatabase();
  });

  afterAll(async () => {
    await db.destroy();
  });

  beforeEach(async () => {
    await db.migrate.latest();
    await db.seed.run();
  });

  afterEach(async () => {
    await db.migrate.rollback();
  });

  test('should create and retrieve user', async () => {
    const user = await db.users.create({ name: 'Test' });
    const retrieved = await db.users.findById(user.id);
    expect(retrieved).toEqual(user);
  });
});
```

## E2E Testing

### Playwright E2E Tests

```typescript
import { test, expect } from '@playwright/test';

test.describe('User Authentication Flow', () => {
  test('should allow user to login', async ({ page }) => {
    await page.goto('/login');

    await page.fill('[name="email"]', 'test@example.com');
    await page.fill('[name="password"]', 'password123');
    await page.click('button[type="submit"]');

    await expect(page).toHaveURL('/dashboard');
    await expect(page.locator('h1')).toContainText('Welcome');
  });

  test('should show error for invalid credentials', async ({ page }) => {
    await page.goto('/login');

    await page.fill('[name="email"]', 'invalid@example.com');
    await page.fill('[name="password"]', 'wrong');
    await page.click('button[type="submit"]');

    await expect(page.locator('[role="alert"]')).toContainText('Invalid credentials');
  });
});
```

## Test-Driven Development (TDD)

### TDD Workflow

1. **Red**: Write failing test
2. **Green**: Write minimal code to pass
3. **Refactor**: Improve code while keeping tests green

```javascript
// 1. Red - Write failing test
test('should calculate discount', () => {
  const price = 100;
  const discount = 0.2;
  expect(calculateDiscount(price, discount)).toBe(80);
});

// 2. Green - Implement minimal solution
function calculateDiscount(price, discount) {
  return price - (price * discount);
}

// 3. Refactor - Improve implementation
function calculateDiscount(price, discount) {
  if (price < 0 || discount < 0 || discount > 1) {
    throw new Error('Invalid input');
  }
  return price * (1 - discount);
}
```

## Coverage Analysis

### Viewing Coverage

```bash
# Jest
npm test -- --coverage

# Vitest
npm test -- --coverage

# PyTest
pytest --cov=src --cov-report=html
```

### Coverage Targets

- **Statements**: 80%+ (aim for 90%+)
- **Branches**: 75%+ (all conditional paths)
- **Functions**: 80%+ (all exported functions)
- **Lines**: 80%+ (executable code lines)

Remember: Focus on testing behavior, not implementation. Write tests that document how the code should work. Keep tests maintainable and readable. High coverage is important, but meaningful tests matter more than 100% coverage.
