---
title: Static Version Configuration
description: Configure and manage static versions directly in pyproject.toml. Covers PEP 440 format, version bumping via CLI, build process integration, and best practices for manual version control.
---

# Static Version Configuration

Static versioning is the simplest approach to version management in Hatchling, where the version is directly defined in the `pyproject.toml` file. Use static versioning for projects that require explicit manual version control or have specific versioning workflows that do not fit automated patterns.

## Basic Configuration

To use static versioning, simply define the `version` field in the `[project]` table:

```toml
[project]
name = "my-package"
version = "1.2.3"
```

## When to Use Static Versioning

Use static versioning when:

- Explicit manual control over version numbers is required
- Release processes involve human review of version changes
- Projects have simple structure with infrequent releases
- Migrating from legacy systems and time is needed for adaptation
- Organizations enforce specific versioning policies requiring manual approval

## Version Format Requirements

All static versions must follow [PEP 440](https://peps.python.org/pep-0440/) version formatting.

### Valid Version Examples

```toml
version = "1.0.0"          # Standard release
version = "2.1.3"          # Major.Minor.Patch
version = "1.0.0a1"        # Alpha release
version = "1.0.0b2"        # Beta release
version = "1.0.0rc1"       # Release candidate
version = "1.0.0.post1"    # Post release
version = "1.0.0.dev0"     # Development release
version = "2020.6.1"       # Calendar versioning
version = "1!2.0.0"        # Version with epoch
```

### Invalid Version Examples

```toml
# These will cause errors:
version = "v1.0.0"         # No 'v' prefix allowed
version = "1.0"            # Must have at least 3 components for standard scheme
version = "1.0.0-alpha"    # Use 'a', not '-alpha'
```

## Working with Static Versions

### Displaying the Version

Display the current version with the `hatch version` command:

```bash
$ hatch version
1.2.3
```

### Updating Static Versions

Update static versions through two approaches:

#### 1. Direct File Edit

Modify the `pyproject.toml` file directly:

```toml
[project]
name = "my-package"
version = "1.2.4"  # Updated from 1.2.3
```

#### 2. Using the CLI

Use the `hatch version` command with a specific version:

```bash
$ hatch version "1.2.4"
Old: 1.2.3
New: 1.2.4
```

Or use version bumping commands:

```bash
$ hatch version patch
Old: 1.2.3
New: 1.2.4

$ hatch version minor
Old: 1.2.4
New: 1.3.0

$ hatch version major
Old: 1.3.0
New: 2.0.0
```

## Integration with Build Process

Even with static versioning, the version is automatically included in built packages:

### In Wheel Metadata

The version appears in the `METADATA` file:

```text
Metadata-Version: 2.3
Name: my-package
Version: 1.2.3
```

### In Source Distributions

The version is included in `PKG-INFO`:

```text
Metadata-Version: 2.3
Name: my-package
Version: 1.2.3
```

## Combining with Version Build Hook

You can use static versioning with the version build hook to write the version to Python files:

```toml
[project]
name = "my-package"
version = "1.2.3"

[tool.hatch.build.hooks.version]
path = "src/my_package/_version.py"
template = '__version__ = "{version}"'
```

This creates `_version.py` during the build:

```python
__version__ = "1.2.3"
```

## Accessing Version in Python Code

### Method 1: Using importlib.metadata (Recommended Approach)

```python
from importlib import metadata

try:
    __version__ = metadata.version("my-package")
except metadata.PackageNotFoundError:
    __version__ = "unknown"
```

### Method 2: Using a Version File Generated by Build Hook

When using the version build hook to generate a version file:

```python
try:
    from ._version import __version__
except ImportError:
    __version__ = "unknown"
```

## Migration Scenarios

### From setup.py

If migrating from `setup.py`:

```python
# Old setup.py
setup(
    name="my-package",
    version="1.2.3",
    ...
)
```

Becomes:

```toml
# New pyproject.toml
[project]
name = "my-package"
version = "1.2.3"
```

### From setup.cfg

If migrating from `setup.cfg`:

```ini
# Old setup.cfg
[metadata]
name = my-package
version = 1.2.3
```

Becomes:

```toml
# New pyproject.toml
[project]
name = "my-package"
version = "1.2.3"
```

## Version Management Workflow

A typical workflow with static versioning:

1. **Development Phase**

   ```toml
   version = "1.2.4.dev0"
   ```

2. **Alpha Release**

   ```toml
   version = "1.2.4a1"
   ```

3. **Beta Release**

   ```toml
   version = "1.2.4b1"
   ```

4. **Release Candidate**

   ```toml
   version = "1.2.4rc1"
   ```

5. **Final Release**

   ```toml
   version = "1.2.4"
   ```

6. **Post Release** (if needed)
   ```toml
   version = "1.2.4.post1"
   ```

## Advantages and Disadvantages

### Advantages

- **Simplicity**: No additional configuration needed
- **Clarity**: Version is immediately visible in project configuration
- **Control**: Complete manual control over versioning
- **Compatibility**: Works with all Python packaging tools
- **No Dependencies**: No external tools or plugins required

### Disadvantages

- **Manual Process**: Requires remembering to update version
- **Human Error**: Prone to mistakes in version formatting
- **Synchronization**: Must manually keep version in sync across files
- **No Automation**: Can't automatically derive version from VCS tags

## Best Practices

1. **Use Semantic Versioning**: Follow MAJOR.MINOR.PATCH convention
2. **Document Changes**: Update CHANGELOG when changing versions
3. **Tag Releases**: Create git tags matching your versions
4. **Automate Checks**: Use CI to verify version format
5. **Consider Migration**: Evaluate dynamic versioning as project grows

## Common Issues

### Issue: Forgetting to Update Version

**Solution**: Add a pre-commit hook to remind about version updates:

```yaml
# .pre-commit-config.yaml
repos:
  - repo: local
    hooks:
      - id: check-version
        name: Check version update
        entry: Check if version was updated
        language: system
        files: pyproject.toml
```

### Issue: Version Mismatch with Git Tags

**Solution**: Use a CI check to verify version matches git tag:

```bash
#!/bin/bash
# ci/check-version.sh
PROJECT_VERSION=$(hatch version)
GIT_TAG=$(git describe --tags --exact-match 2>/dev/null)

if [ -n "$GIT_TAG" ] && [ "v$PROJECT_VERSION" != "$GIT_TAG" ]; then
    echo "Error: Version mismatch!"
    echo "Project version: $PROJECT_VERSION"
    echo "Git tag: $GIT_TAG"
    exit 1
fi
```

## See Also

- [Dynamic Version Sources Overview](./dynamic-version-overview.md)
- [Version Build Hook](./version-build-hook.md)
- [Version CLI Commands](./version-cli.md)
- [PEP 440 - Version Identification](https://peps.python.org/pep-0440/)
