#!/usr/bin/env python3
# /// script
# requires-python = ">=3.11"
# dependencies = []
# ///
"""Detect and run git hook tool (prek or pre-commit).

Reads .git/hooks/pre-commit to determine which tool generated it.
Falls back to 'prek' if file doesn't exist or is unrecognized.
"""

import subprocess
import sys
from pathlib import Path


def detect_tool(repo_path: Path) -> str:
    """Detect which git hook tool is installed.

    Args:
        repo_path: Path to git repository

    Returns:
        Tool name ('prek' or 'pre-commit')
    """
    hook_file = repo_path / ".git" / "hooks" / "pre-commit"

    if not hook_file.exists():
        return "prek"

    try:
        with hook_file.open() as f:
            f.readline()  # Skip first line
            second_line = f.readline()

        # Line format: "# File generated by <tool>: <url>"
        # Token 5 (0-indexed token 4) is the tool name with trailing colon
        tokens = second_line.split()
        if len(tokens) >= 5 and tokens[3] == "by":
            tool_name = tokens[4].rstrip(":")
            if tool_name in ("prek", "pre-commit"):
                return tool_name
    except (OSError, IndexError):
        pass

    return "prek"


def main() -> int:
    """Main entry point.

    Returns:
        Exit code (0 on success, tool exit code when executing)
    """
    # Parse --directory flag if present
    repo_path = Path.cwd()
    args = sys.argv[1:]

    if args and args[0] == "--directory":
        if len(args) < 2:
            print("Error: --directory requires a path argument", file=sys.stderr)
            return 1
        repo_path = Path(args[1])
        args = args[2:]  # Remove --directory and its value

    # Detect tool
    tool = detect_tool(repo_path)

    # If no additional args, just print tool name
    if not args:
        print(tool)
        return 0

    # Execute the tool with remaining args
    cmd = [tool, *args]
    try:
        result = subprocess.run(cmd, cwd=repo_path)
    except FileNotFoundError:
        print(f"Error: {tool} not found in PATH", file=sys.stderr)
        return 127
    else:
        return result.returncode


if __name__ == "__main__":
    sys.exit(main())
