#!/bin/bash

# Demonstration of Destructive Guard Feature
# This script shows examples of operations that would be guarded

echo "==================================================================="
echo "Claudisms 2.1 - Destructive Guard Examples"
echo "==================================================================="
echo ""
echo "The following operations are intercepted by the destructive guard:"
echo ""

echo "### CRITICAL SEVERITY 🚨"
echo ""
echo "1. Recursive file deletion:"
echo "   rm -rf /tmp/old-project"
echo "   rm -fr node_modules/"
echo ""

echo "2. Git push to protected branches:"
echo "   git push origin main"
echo "   git push origin master"
echo "   git push --force origin main"
echo ""

echo "3. Database table/schema deletion:"
echo "   psql -c 'DROP TABLE users;'"
echo "   psql -c 'DROP DATABASE old_db;'"
echo "   mysql -e 'DROP TABLE sessions;'"
echo ""

echo "4. Bulk database deletion (no WHERE clause):"
echo "   psql -c 'DELETE FROM users;'"
echo "   mysql -e 'DELETE FROM orders;'"
echo ""

echo "5. Database truncation:"
echo "   psql -c 'TRUNCATE TABLE logs;'"
echo "   mysql -e 'TRUNCATE TABLE sessions;'"
echo ""

echo "### MEDIUM SEVERITY ⚡"
echo ""
echo "1. Git push to feature branches:"
echo "   git push origin feature/new-feature"
echo "   git push origin develop"
echo ""

echo "2. Targeted database deletion (has WHERE clause):"
echo "   psql -c 'DELETE FROM users WHERE id=1;'"
echo "   mysql -e 'DELETE FROM orders WHERE status=\"cancelled\";'"
echo ""

echo "### ALLOWED OPERATIONS ✓"
echo ""
echo "These operations are NOT blocked (even with guard ON):"
echo ""
echo "1. Safe git operations:"
echo "   git status"
echo "   git diff"
echo "   git log"
echo "   git commit -m 'message'"
echo ""

echo "2. File operations:"
echo "   ls -la"
echo "   cat file.txt"
echo "   mkdir new-dir"
echo "   cp file.txt backup.txt"
echo ""

echo "3. Non-destructive database operations:"
echo "   psql -c 'SELECT * FROM users;'"
echo "   psql -c 'INSERT INTO users VALUES (...);'"
echo "   psql -c 'UPDATE users SET name=\"John\" WHERE id=1;'"
echo ""

echo "==================================================================="
echo "Guard Modes"
echo "==================================================================="
echo ""
echo "ON (default): Blocks all destructive operations"
echo "  - Maximum safety"
echo "  - Requires confirmation for ANY destructive operation"
echo ""
echo "EXCLUDE: Allow refactoring, block git push and DB operations"
echo "  - Ideal for large refactoring tasks"
echo "  - Freely edit/rename files"
echo "  - Still protects against accidental pushes and DB drops"
echo ""
echo "OFF: Allow all operations"
echo "  - For trusted environments"
echo "  - Automation scripts"
echo "  - When you know exactly what you're doing"
echo ""

echo "==================================================================="
echo "Usage Examples"
echo "==================================================================="
echo ""
echo "# View current settings"
echo "/claudisms-settings"
echo ""
echo "# Enable exclude mode for refactoring"
echo "/claudisms-settings set destructive_guard exclude"
echo ""
echo "# Disable guard for automation"
echo "/claudisms-settings set destructive_guard off"
echo ""
echo "# Re-enable full protection"
echo "/claudisms-settings set destructive_guard on"
echo ""
echo "==================================================================="
