# S3 Storage

## Contents

- [Bucket Management](#bucket-management)
- [Block Public Access](#block-public-access)
- [Bucket Policies](#bucket-policies)
- [Object Operations](#object-operations)
- [Presigned URLs](#presigned-urls)
- [Versioning](#versioning)
- [Encryption](#encryption)
- [Lifecycle Policies](#lifecycle-policies)
- [Transfer Acceleration](#transfer-acceleration)
- [Static Website Hosting](#static-website-hosting)
- [CORS Configuration](#cors-configuration)
- [Cross-Region Replication](#cross-region-replication)
- [S3 Select](#s3-select)
- [Object Lock (WORM)](#object-lock-worm)
- [Event Notifications](#event-notifications)
- [Useful Queries](#useful-queries)
- [Performance Optimization](#performance-optimization)
- [Best Practices](#best-practices)

---

## Bucket Management

### Create Bucket
```bash
# Create bucket (us-east-1 doesn't need LocationConstraint)
aws s3 mb s3://my-bucket

# Create in specific region
aws s3 mb s3://my-bucket --region eu-west-1

# Using s3api with explicit region constraint
aws s3api create-bucket \
    --bucket my-bucket \
    --region eu-west-1 \
    --create-bucket-configuration LocationConstraint=eu-west-1
```

### List and Describe Buckets
```bash
# List all buckets
aws s3 ls

# List bucket contents
aws s3 ls s3://my-bucket/
aws s3 ls s3://my-bucket/prefix/ --recursive

# Get bucket location
aws s3api get-bucket-location --bucket my-bucket

# Get bucket info
aws s3api head-bucket --bucket my-bucket
```

### Delete Bucket
```bash
# Delete empty bucket
aws s3 rb s3://my-bucket

# Force delete (remove all objects first)
aws s3 rb s3://my-bucket --force
```

## Block Public Access

### Configure Block Public Access
```bash
# Block all public access (recommended for most buckets)
aws s3api put-public-access-block \
    --bucket my-bucket \
    --public-access-block-configuration \
        BlockPublicAcls=true,\
IgnorePublicAcls=true,\
BlockPublicPolicy=true,\
RestrictPublicBuckets=true

# Get current settings
aws s3api get-public-access-block --bucket my-bucket
```

### Account-Level Block
```bash
# Block public access for entire account
aws s3control put-public-access-block \
    --account-id 123456789012 \
    --public-access-block-configuration \
        BlockPublicAcls=true,\
IgnorePublicAcls=true,\
BlockPublicPolicy=true,\
RestrictPublicBuckets=true
```

## Bucket Policies

### Set Bucket Policy
```bash
aws s3api put-bucket-policy \
    --bucket my-bucket \
    --policy file://bucket-policy.json
```

**bucket-policy.json (Deny non-HTTPS):**
```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "DenyNonHTTPS",
            "Effect": "Deny",
            "Principal": "*",
            "Action": "s3:*",
            "Resource": [
                "arn:aws:s3:::my-bucket",
                "arn:aws:s3:::my-bucket/*"
            ],
            "Condition": {
                "Bool": {"aws:SecureTransport": "false"}
            }
        }
    ]
}
```

**bucket-policy.json (Cross-account read access):**
```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "CrossAccountRead",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::OTHER_ACCOUNT_ID:root"
            },
            "Action": [
                "s3:GetObject",
                "s3:ListBucket"
            ],
            "Resource": [
                "arn:aws:s3:::my-bucket",
                "arn:aws:s3:::my-bucket/*"
            ]
        }
    ]
}
```

**bucket-policy.json (VPC endpoint only):**
```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "VPCEndpointOnly",
            "Effect": "Deny",
            "Principal": "*",
            "Action": "s3:*",
            "Resource": [
                "arn:aws:s3:::my-bucket",
                "arn:aws:s3:::my-bucket/*"
            ],
            "Condition": {
                "StringNotEquals": {
                    "aws:SourceVpce": "vpce-1234567890abcdef0"
                }
            }
        }
    ]
}
```

### Manage Bucket Policy
```bash
# Get current policy
aws s3api get-bucket-policy --bucket my-bucket

# Delete policy
aws s3api delete-bucket-policy --bucket my-bucket
```

## Object Operations

### Copy and Upload
```bash
# Upload single file
aws s3 cp document.pdf s3://my-bucket/docs/

# Upload with storage class
aws s3 cp archive.zip s3://my-bucket/ --storage-class STANDARD_IA

# Upload with metadata
aws s3 cp file.txt s3://my-bucket/ \
    --metadata '{"project":"alpha","owner":"team-a"}'

# Upload with content type
aws s3 cp index.html s3://my-bucket/ --content-type "text/html"

# Upload directory recursively
aws s3 cp ./local-dir s3://my-bucket/remote-dir/ --recursive
```

### Download
```bash
# Download single file
aws s3 cp s3://my-bucket/docs/document.pdf ./

# Download directory
aws s3 cp s3://my-bucket/logs/ ./logs/ --recursive

# Download with specific version
aws s3api get-object \
    --bucket my-bucket \
    --key myfile.txt \
    --version-id abc123 \
    output.txt
```

### Move and Rename
```bash
# Move (copy + delete)
aws s3 mv s3://my-bucket/old.txt s3://my-bucket/new.txt

# Move between buckets
aws s3 mv s3://source-bucket/file.txt s3://dest-bucket/file.txt
```

### Sync
```bash
# Sync local to S3 (upload changes only)
aws s3 sync ./build s3://my-bucket/site/

# Sync with delete (mirror)
aws s3 sync ./build s3://my-bucket/site/ --delete

# Sync with exclusions
aws s3 sync ./project s3://my-bucket/ \
    --exclude "*.git/*" \
    --exclude "node_modules/*" \
    --include "*.js"

# Sync between buckets
aws s3 sync s3://source-bucket s3://dest-bucket

# Dry run (show what would happen)
aws s3 sync ./build s3://my-bucket/ --dryrun
```

### Delete Objects
```bash
# Delete single object
aws s3 rm s3://my-bucket/file.txt

# Delete recursively
aws s3 rm s3://my-bucket/logs/ --recursive

# Delete with prefix
aws s3 rm s3://my-bucket/ --recursive --exclude "*" --include "temp/*"
```

## Presigned URLs

### Generate Download URLs
```bash
# Default 1 hour expiry
aws s3 presign s3://my-bucket/report.pdf

# Custom expiry (seconds)
aws s3 presign s3://my-bucket/report.pdf --expires-in 86400  # 24 hours
```

### Generate Upload URLs
```bash
# Using s3api for upload presigned URL
aws s3api put-object \
    --bucket my-bucket \
    --key uploads/file.txt \
    --generate-presigned-url \
    --expires-in 3600
```

## Versioning

### Enable Versioning
```bash
# Enable
aws s3api put-bucket-versioning \
    --bucket my-bucket \
    --versioning-configuration Status=Enabled

# Suspend (keeps existing versions)
aws s3api put-bucket-versioning \
    --bucket my-bucket \
    --versioning-configuration Status=Suspended

# Get status
aws s3api get-bucket-versioning --bucket my-bucket
```

### Work with Versions
```bash
# List all versions
aws s3api list-object-versions --bucket my-bucket

# List versions for specific key
aws s3api list-object-versions \
    --bucket my-bucket \
    --prefix myfile.txt

# Get specific version
aws s3api get-object \
    --bucket my-bucket \
    --key myfile.txt \
    --version-id "abc123xyz" \
    output.txt

# Delete specific version
aws s3api delete-object \
    --bucket my-bucket \
    --key myfile.txt \
    --version-id "abc123xyz"

# Restore deleted object (delete the delete marker)
aws s3api delete-object \
    --bucket my-bucket \
    --key myfile.txt \
    --version-id "delete-marker-version-id"
```

## Encryption

### Server-Side Encryption (Default)
```bash
# SSE-S3 (AES-256, AWS managed keys)
aws s3api put-bucket-encryption \
    --bucket my-bucket \
    --server-side-encryption-configuration '{
        "Rules": [{
            "ApplyServerSideEncryptionByDefault": {
                "SSEAlgorithm": "AES256"
            },
            "BucketKeyEnabled": true
        }]
    }'

# SSE-KMS (Customer managed key)
aws s3api put-bucket-encryption \
    --bucket my-bucket \
    --server-side-encryption-configuration '{
        "Rules": [{
            "ApplyServerSideEncryptionByDefault": {
                "SSEAlgorithm": "aws:kms",
                "KMSMasterKeyID": "arn:aws:kms:us-east-1:123456789012:key/abc123"
            },
            "BucketKeyEnabled": true
        }]
    }'
```

### Upload with Encryption
```bash
# SSE-S3
aws s3 cp file.txt s3://my-bucket/ --sse AES256

# SSE-KMS
aws s3 cp file.txt s3://my-bucket/ \
    --sse aws:kms \
    --sse-kms-key-id alias/my-key
```

## Lifecycle Policies

### Create Lifecycle Policy
```bash
aws s3api put-bucket-lifecycle-configuration \
    --bucket my-bucket \
    --lifecycle-configuration file://lifecycle.json
```

**lifecycle.json (Comprehensive):**
```json
{
    "Rules": [
        {
            "ID": "TransitionToIA",
            "Status": "Enabled",
            "Filter": {"Prefix": "data/"},
            "Transitions": [
                {"Days": 30, "StorageClass": "STANDARD_IA"},
                {"Days": 90, "StorageClass": "GLACIER"},
                {"Days": 365, "StorageClass": "DEEP_ARCHIVE"}
            ]
        },
        {
            "ID": "ExpireTempFiles",
            "Status": "Enabled",
            "Filter": {"Prefix": "temp/"},
            "Expiration": {"Days": 7}
        },
        {
            "ID": "CleanupIncompleteMultipart",
            "Status": "Enabled",
            "Filter": {"Prefix": ""},
            "AbortIncompleteMultipartUpload": {"DaysAfterInitiation": 7}
        },
        {
            "ID": "DeleteOldVersions",
            "Status": "Enabled",
            "Filter": {"Prefix": ""},
            "NoncurrentVersionExpiration": {"NoncurrentDays": 30}
        }
    ]
}
```

### Manage Lifecycle
```bash
# Get lifecycle configuration
aws s3api get-bucket-lifecycle-configuration --bucket my-bucket

# Delete lifecycle
aws s3api delete-bucket-lifecycle --bucket my-bucket
```

## Transfer Acceleration

### Enable Transfer Acceleration
```bash
# Enable
aws s3api put-bucket-accelerate-configuration \
    --bucket my-bucket \
    --accelerate-configuration Status=Enabled

# Get status
aws s3api get-bucket-accelerate-configuration --bucket my-bucket
```

### Use Accelerated Endpoint
```bash
# Upload using accelerated endpoint
aws s3 cp large-file.zip s3://my-bucket/ \
    --endpoint-url https://my-bucket.s3-accelerate.amazonaws.com

# Or set as default
aws configure set s3.use_accelerate_endpoint true
```

## Static Website Hosting

### Enable Website Hosting
```bash
aws s3api put-bucket-website \
    --bucket my-bucket \
    --website-configuration '{
        "IndexDocument": {"Suffix": "index.html"},
        "ErrorDocument": {"Key": "error.html"}
    }'
```

### Public Website Policy
```bash
# First, disable block public access
aws s3api put-public-access-block \
    --bucket my-bucket \
    --public-access-block-configuration \
        BlockPublicAcls=false,\
IgnorePublicAcls=false,\
BlockPublicPolicy=false,\
RestrictPublicBuckets=false

# Then apply public read policy
aws s3api put-bucket-policy \
    --bucket my-bucket \
    --policy '{
        "Version": "2012-10-17",
        "Statement": [{
            "Sid": "PublicRead",
            "Effect": "Allow",
            "Principal": "*",
            "Action": "s3:GetObject",
            "Resource": "arn:aws:s3:::my-bucket/*"
        }]
    }'
```

### Get Website URL
```bash
# Website endpoint format
# http://my-bucket.s3-website-us-east-1.amazonaws.com
aws s3api get-bucket-website --bucket my-bucket
```

## CORS Configuration

### Set CORS Rules
```bash
aws s3api put-bucket-cors \
    --bucket my-bucket \
    --cors-configuration file://cors.json
```

**cors.json:**
```json
{
    "CORSRules": [
        {
            "AllowedHeaders": ["*"],
            "AllowedMethods": ["GET", "PUT", "POST"],
            "AllowedOrigins": ["https://example.com", "https://app.example.com"],
            "ExposeHeaders": ["ETag"],
            "MaxAgeSeconds": 3600
        }
    ]
}
```

### Manage CORS
```bash
# Get CORS
aws s3api get-bucket-cors --bucket my-bucket

# Delete CORS
aws s3api delete-bucket-cors --bucket my-bucket
```

## Cross-Region Replication

### Enable Replication
```bash
# First enable versioning on both buckets
aws s3api put-bucket-versioning \
    --bucket source-bucket \
    --versioning-configuration Status=Enabled

aws s3api put-bucket-versioning \
    --bucket dest-bucket \
    --versioning-configuration Status=Enabled

# Set replication configuration
aws s3api put-bucket-replication \
    --bucket source-bucket \
    --replication-configuration file://replication.json
```

**replication.json:**
```json
{
    "Role": "arn:aws:iam::123456789012:role/S3ReplicationRole",
    "Rules": [
        {
            "ID": "ReplicateAll",
            "Status": "Enabled",
            "Priority": 1,
            "Filter": {},
            "Destination": {
                "Bucket": "arn:aws:s3:::dest-bucket",
                "StorageClass": "STANDARD"
            },
            "DeleteMarkerReplication": {"Status": "Enabled"}
        }
    ]
}
```

## S3 Select

Query data directly in S3 without downloading.

```bash
# Query CSV file
aws s3api select-object-content \
    --bucket my-bucket \
    --key data.csv \
    --expression "SELECT * FROM s3object s WHERE s.status = 'active'" \
    --expression-type SQL \
    --input-serialization '{"CSV": {"FileHeaderInfo": "USE"}}' \
    --output-serialization '{"JSON": {}}' \
    output.json

# Query JSON file
aws s3api select-object-content \
    --bucket my-bucket \
    --key data.json \
    --expression "SELECT s.name, s.email FROM s3object s WHERE s.age > 30" \
    --expression-type SQL \
    --input-serialization '{"JSON": {"Type": "LINES"}}' \
    --output-serialization '{"JSON": {}}' \
    output.json
```

## Object Lock (WORM)

### Enable Object Lock
```bash
# Must be enabled at bucket creation
aws s3api create-bucket \
    --bucket my-bucket \
    --object-lock-enabled-for-bucket

# Set default retention
aws s3api put-object-lock-configuration \
    --bucket my-bucket \
    --object-lock-configuration '{
        "ObjectLockEnabled": "Enabled",
        "Rule": {
            "DefaultRetention": {
                "Mode": "GOVERNANCE",
                "Days": 365
            }
        }
    }'
```

### Apply Retention to Object
```bash
# Governance mode (can be overridden with permission)
aws s3api put-object-retention \
    --bucket my-bucket \
    --key important-file.txt \
    --retention '{"Mode": "GOVERNANCE", "RetainUntilDate": "2025-12-31T00:00:00Z"}'

# Compliance mode (cannot be overridden)
aws s3api put-object-retention \
    --bucket my-bucket \
    --key legal-doc.pdf \
    --retention '{"Mode": "COMPLIANCE", "RetainUntilDate": "2030-12-31T00:00:00Z"}'

# Legal hold
aws s3api put-object-legal-hold \
    --bucket my-bucket \
    --key evidence.pdf \
    --legal-hold '{"Status": "ON"}'
```

## Event Notifications

### Configure Notifications
```bash
aws s3api put-bucket-notification-configuration \
    --bucket my-bucket \
    --notification-configuration file://notifications.json
```

**notifications.json (Lambda trigger):**
```json
{
    "LambdaFunctionConfigurations": [
        {
            "Id": "ProcessNewUploads",
            "LambdaFunctionArn": "arn:aws:lambda:us-east-1:123456789012:function:ProcessS3",
            "Events": ["s3:ObjectCreated:*"],
            "Filter": {
                "Key": {
                    "FilterRules": [
                        {"Name": "prefix", "Value": "uploads/"},
                        {"Name": "suffix", "Value": ".json"}
                    ]
                }
            }
        }
    ]
}
```

**notifications.json (SQS):**
```json
{
    "QueueConfigurations": [
        {
            "Id": "NewFileQueue",
            "QueueArn": "arn:aws:sqs:us-east-1:123456789012:s3-events",
            "Events": ["s3:ObjectCreated:*"]
        }
    ]
}
```

## Useful Queries

```bash
# Get bucket size
aws s3 ls s3://my-bucket --recursive --summarize | tail -2

# Find largest objects
aws s3api list-objects-v2 \
    --bucket my-bucket \
    --query 'sort_by(Contents, &Size)[-10:].{Key:Key,SizeMB:Size}' \
    --output table

# Count objects by prefix
aws s3api list-objects-v2 \
    --bucket my-bucket \
    --prefix "logs/" \
    --query 'length(Contents)'

# Get storage class distribution
aws s3api list-objects-v2 \
    --bucket my-bucket \
    --query 'Contents[].StorageClass' | sort | uniq -c

# Find objects modified in last 24 hours
aws s3api list-objects-v2 \
    --bucket my-bucket \
    --query "Contents[?LastModified>='$(date -u -d '1 day ago' '+%Y-%m-%dT%H:%M:%S')'].Key"
```

## Performance Optimization

| Technique | When to Use |
|:----------|:------------|
| **Transfer Acceleration** | Large files over long distances |
| **Multipart Upload** | Files > 100MB (automatic in `aws s3 cp`) |
| **Byte-range fetches** | Parallel downloads of large files |
| **S3 Select** | Query subset of data without downloading |
| **Prefix distribution** | Millions of objects, add random prefix |

```bash
# Parallel uploads with concurrent requests
aws configure set default.s3.max_concurrent_requests 20
aws configure set default.s3.multipart_chunksize 64MB

# Upload with higher concurrency
aws s3 sync ./large-dataset s3://my-bucket/ --only-show-errors
```

## Best Practices

| Practice | Description |
|:---------|:------------|
| **Block public access** | Enable account-wide public access block |
| **Encryption** | Use SSE-S3 or SSE-KMS for all buckets |
| **Versioning** | Enable for critical data buckets |
| **Lifecycle policies** | Transition to cheaper storage classes |
| **HTTPS only** | Use bucket policy to deny non-SSL requests |
| **VPC endpoints** | Access S3 without internet for private subnets |
| **Object Lock** | Use for compliance/regulatory requirements |
| **Bucket naming** | Use DNS-compliant names, avoid periods |
| **Logging** | Enable access logging for auditing |
| **Replication** | Use CRR for disaster recovery |
