#!/usr/bin/env python3
"""
Project Scaffold Generator - Creates project structure templates.
"""

import json
import os
import sys
from pathlib import Path
from typing import Optional


def generate_nextjs_structure() -> dict:
    """Generate Next.js 14+ App Router structure."""
    return {
        "name": "nextjs-app",
        "files": {
            "package.json": """{
  "name": "my-app",
  "version": "0.1.0",
  "private": true,
  "scripts": {
    "dev": "next dev",
    "build": "next build",
    "start": "next start",
    "lint": "next lint"
  },
  "dependencies": {
    "next": "^14.0.0",
    "react": "^18.2.0",
    "react-dom": "^18.2.0"
  },
  "devDependencies": {
    "@types/node": "^20",
    "@types/react": "^18",
    "@types/react-dom": "^18",
    "eslint": "^8",
    "eslint-config-next": "^14",
    "typescript": "^5"
  }
}""",
            "tsconfig.json": """{
  "compilerOptions": {
    "target": "ES2017",
    "lib": ["dom", "dom.iterable", "esnext"],
    "allowJs": true,
    "skipLibCheck": true,
    "strict": true,
    "noEmit": true,
    "esModuleInterop": true,
    "module": "esnext",
    "moduleResolution": "bundler",
    "resolveJsonModule": true,
    "isolatedModules": true,
    "jsx": "preserve",
    "incremental": true,
    "plugins": [{ "name": "next" }],
    "paths": { "@/*": ["./src/*"] }
  },
  "include": ["next-env.d.ts", "**/*.ts", "**/*.tsx", ".next/types/**/*.ts"],
  "exclude": ["node_modules"]
}""",
            "next.config.js": """/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
}

module.exports = nextConfig
""",
            ".eslintrc.json": """{
  "extends": "next/core-web-vitals"
}""",
            ".gitignore": """# dependencies
node_modules
.pnp
.pnp.js

# testing
coverage

# next.js
.next/
out/

# production
build

# misc
.DS_Store
*.pem

# debug
npm-debug.log*

# local env files
.env*.local

# vercel
.vercel

# typescript
*.tsbuildinfo
next-env.d.ts
"""
        },
        "directories": [
            "src/app",
            "src/components",
            "src/lib",
            "public"
        ],
        "app_files": {
            "src/app/layout.tsx": """import type { Metadata } from 'next'
import './globals.css'

export const metadata: Metadata = {
  title: 'My App',
  description: 'Generated by create next app',
}

export default function RootLayout({
  children,
}: {
  children: React.ReactNode
}) {
  return (
    <html lang="en">
      <body>{children}</body>
    </html>
  )
}
""",
            "src/app/page.tsx": """export default function Home() {
  return (
    <main className="flex min-h-screen flex-col items-center justify-center p-24">
      <h1 className="text-4xl font-bold">Welcome to Next.js</h1>
    </main>
  )
}
""",
            "src/app/globals.css": """@tailwind base;
@tailwind components;
@tailwind utilities;
"""
        }
    }


def generate_express_structure() -> dict:
    """Generate Express.js API structure."""
    return {
        "name": "express-api",
        "files": {
            "package.json": """{
  "name": "express-api",
  "version": "1.0.0",
  "main": "dist/index.js",
  "scripts": {
    "dev": "tsx watch src/index.ts",
    "build": "tsc",
    "start": "node dist/index.js",
    "test": "jest"
  },
  "dependencies": {
    "express": "^4.18.2",
    "cors": "^2.8.5",
    "helmet": "^7.1.0",
    "dotenv": "^16.3.1"
  },
  "devDependencies": {
    "@types/express": "^4.17.21",
    "@types/cors": "^2.8.17",
    "@types/node": "^20",
    "tsx": "^4.6.0",
    "typescript": "^5.3.0"
  }
}""",
            "tsconfig.json": """{
  "compilerOptions": {
    "target": "ES2022",
    "module": "NodeNext",
    "moduleResolution": "NodeNext",
    "outDir": "./dist",
    "rootDir": "./src",
    "strict": true,
    "esModuleInterop": true,
    "skipLibCheck": true,
    "forceConsistentCasingInFileNames": true
  },
  "include": ["src/**/*"],
  "exclude": ["node_modules", "dist"]
}""",
            ".env.example": """PORT=3000
NODE_ENV=development
""",
            ".gitignore": """node_modules
dist
.env
*.log
"""
        },
        "directories": [
            "src/routes",
            "src/middleware",
            "src/services",
            "src/types"
        ],
        "app_files": {
            "src/index.ts": """import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import { config } from 'dotenv';

config();

const app = express();
const port = process.env.PORT || 3000;

app.use(helmet());
app.use(cors());
app.use(express.json());

app.get('/health', (req, res) => {
  res.json({ status: 'ok' });
});

app.listen(port, () => {
  console.log(`Server running on port ${port}`);
});
""",
            "src/routes/index.ts": """import { Router } from 'express';

const router = Router();

router.get('/', (req, res) => {
  res.json({ message: 'API is running' });
});

export default router;
"""
        }
    }


def generate_python_fastapi_structure() -> dict:
    """Generate FastAPI structure."""
    return {
        "name": "fastapi-app",
        "files": {
            "pyproject.toml": """[project]
name = "fastapi-app"
version = "0.1.0"
requires-python = ">=3.11"
dependencies = [
    "fastapi>=0.109.0",
    "uvicorn[standard]>=0.27.0",
    "pydantic>=2.5.0",
    "python-dotenv>=1.0.0",
]

[project.optional-dependencies]
dev = [
    "pytest>=7.4.0",
    "httpx>=0.26.0",
    "ruff>=0.1.0",
]

[tool.ruff]
line-length = 100
target-version = "py311"
""",
            "requirements.txt": """fastapi>=0.109.0
uvicorn[standard]>=0.27.0
pydantic>=2.5.0
python-dotenv>=1.0.0
""",
            ".env.example": """DEBUG=true
""",
            ".gitignore": """__pycache__
*.pyc
.env
.venv
venv
.pytest_cache
.ruff_cache
"""
        },
        "directories": [
            "app/api",
            "app/models",
            "app/services",
            "tests"
        ],
        "app_files": {
            "app/__init__.py": "",
            "app/main.py": """from fastapi import FastAPI
from dotenv import load_dotenv

load_dotenv()

app = FastAPI(title="FastAPI App")


@app.get("/health")
def health():
    return {"status": "ok"}


@app.get("/")
def root():
    return {"message": "Welcome to FastAPI"}
""",
            "app/api/__init__.py": "",
            "app/models/__init__.py": "",
            "app/services/__init__.py": "",
            "tests/__init__.py": "",
            "tests/test_main.py": """from fastapi.testclient import TestClient
from app.main import app

client = TestClient(app)


def test_health():
    response = client.get("/health")
    assert response.status_code == 200
    assert response.json() == {"status": "ok"}
"""
        }
    }


def generate_structure_tree(structure: dict) -> str:
    """Generate a visual tree of the project structure."""
    lines = [f"{structure['name']}/"]

    # Add directories
    for directory in sorted(structure.get("directories", [])):
        lines.append(f"├── {directory}/")

    # Add root files
    for filename in sorted(structure.get("files", {}).keys()):
        lines.append(f"├── {filename}")

    # Add app files
    for filepath in sorted(structure.get("app_files", {}).keys()):
        lines.append(f"├── {filepath}")

    return "\n".join(lines)


def list_templates() -> str:
    """List available templates."""
    return """Available Templates:
====================

nextjs     - Next.js 14+ with App Router, TypeScript
express    - Express.js API with TypeScript
fastapi    - FastAPI with Python 3.11+

Usage: python scaffold_generator.py --template <name>
"""


if __name__ == "__main__":
    if len(sys.argv) > 1:
        cmd = sys.argv[1]

        if cmd == "--list":
            print(list_templates())

        elif cmd == "--template":
            template = sys.argv[2] if len(sys.argv) > 2 else "nextjs"

            if template == "nextjs":
                structure = generate_nextjs_structure()
            elif template == "express":
                structure = generate_express_structure()
            elif template == "fastapi":
                structure = generate_python_fastapi_structure()
            else:
                print(f"Unknown template: {template}")
                sys.exit(1)

            print(f"# {structure['name']} Structure\n")
            print(generate_structure_tree(structure))
            print("\n# Files to create:\n")
            for filepath, content in {**structure.get("files", {}), **structure.get("app_files", {})}.items():
                print(f"## {filepath}")
                print("```")
                print(content[:500] + "..." if len(content) > 500 else content)
                print("```\n")

        elif cmd == "--tree":
            template = sys.argv[2] if len(sys.argv) > 2 else "nextjs"
            structures = {
                "nextjs": generate_nextjs_structure,
                "express": generate_express_structure,
                "fastapi": generate_python_fastapi_structure
            }
            if template in structures:
                print(generate_structure_tree(structures[template]()))
    else:
        print("Project Scaffold Generator")
        print("Usage:")
        print("  --list              List available templates")
        print("  --template <name>   Show template structure")
        print("  --tree <name>       Show directory tree only")
