#!/bin/bash
#
# Git Hooks Setup for Micro-Commit Workflow
#
# Installs git hooks for:
#   - commit-msg: Validates conventional commit format
#   - prepare-commit-msg: Adds helpful template
#
# Usage:
#   ./setup-git-hooks.sh --install    Install hooks
#   ./setup-git-hooks.sh --uninstall  Remove hooks
#   ./setup-git-hooks.sh --check      Check installation status
#
# Exit codes:
#   0 - Success
#   1 - Error

set -e

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Paths
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
GIT_HOOKS_DIR=".git/hooks"
COMMIT_MSG_HOOK="$GIT_HOOKS_DIR/commit-msg"
PREPARE_COMMIT_MSG_HOOK="$GIT_HOOKS_DIR/prepare-commit-msg"

# Functions
print_error() {
    echo -e "${RED}❌ ERROR: $1${NC}" >&2
}

print_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠️  WARNING: $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

show_usage() {
    cat << EOF
Git Hooks Setup for Micro-Commit Workflow

Usage:
  $0 --install         Install git hooks for commit validation
  $0 --uninstall       Remove installed git hooks
  $0 --check           Check installation status
  $0 --help            Show this help

Hooks Installed:
  commit-msg           Validates commit message format
  prepare-commit-msg   Adds LINEAR ticket reference if on feature branch

What this does:
  - Validates conventional commit format before each commit
  - Ensures commit messages follow best practices
  - Blocks invalid commits automatically
  - Adds LINEAR ticket references from branch names

Examples:
  $0 --install         # Install hooks in current repo
  $0 --check           # Check if hooks are installed
  $0 --uninstall       # Remove hooks

EOF
}

check_git_repo() {
    if [ ! -d ".git" ]; then
        print_error "Not a git repository. Run this from the repository root."
        exit 1
    fi
}

check_hooks_installed() {
    local installed=true

    if [ ! -f "$COMMIT_MSG_HOOK" ]; then
        installed=false
    fi

    if [ ! -f "$PREPARE_COMMIT_MSG_HOOK" ]; then
        installed=false
    fi

    echo "$installed"
}

check_installation() {
    echo -e "${BLUE}=== Checking Git Hook Installation ===${NC}"
    echo ""

    if [ ! -d "$GIT_HOOKS_DIR" ]; then
        print_error "Git hooks directory not found"
        exit 1
    fi

    # Check commit-msg hook
    if [ -f "$COMMIT_MSG_HOOK" ]; then
        print_success "commit-msg hook is installed"
        echo "  Location: $COMMIT_MSG_HOOK"
        if [ -x "$COMMIT_MSG_HOOK" ]; then
            print_info "Hook is executable"
        else
            print_warning "Hook is not executable"
        fi
    else
        print_warning "commit-msg hook is NOT installed"
    fi

    echo ""

    # Check prepare-commit-msg hook
    if [ -f "$PREPARE_COMMIT_MSG_HOOK" ]; then
        print_success "prepare-commit-msg hook is installed"
        echo "  Location: $PREPARE_COMMIT_MSG_HOOK"
        if [ -x "$PREPARE_COMMIT_MSG_HOOK" ]; then
            print_info "Hook is executable"
        else
            print_warning "Hook is not executable"
        fi
    else
        print_warning "prepare-commit-msg hook is NOT installed"
    fi

    echo ""
}

install_hooks() {
    echo -e "${BLUE}=== Installing Git Hooks ===${NC}"
    echo ""

    # Create hooks directory if it doesn't exist
    mkdir -p "$GIT_HOOKS_DIR"

    # Backup existing hooks if they exist
    if [ -f "$COMMIT_MSG_HOOK" ]; then
        print_warning "Existing commit-msg hook found, backing up..."
        mv "$COMMIT_MSG_HOOK" "${COMMIT_MSG_HOOK}.backup.$(date +%Y%m%d_%H%M%S)"
    fi

    if [ -f "$PREPARE_COMMIT_MSG_HOOK" ]; then
        print_warning "Existing prepare-commit-msg hook found, backing up..."
        mv "$PREPARE_COMMIT_MSG_HOOK" "${PREPARE_COMMIT_MSG_HOOK}.backup.$(date +%Y%m%d_%H%M%S)"
    fi

    # Install commit-msg hook
    cat > "$COMMIT_MSG_HOOK" << 'EOF'
#!/bin/bash
#
# commit-msg hook - Validates conventional commit format
# Auto-generated by micro-commit-workflow skill

# Find the validator script
SKILL_DIR=".claude/skills/micro-commit-workflow"
VALIDATOR="$SKILL_DIR/scripts/validate-commit-msg.sh"

if [ ! -f "$VALIDATOR" ]; then
    echo "Warning: Commit message validator not found at $VALIDATOR"
    echo "Skipping validation..."
    exit 0
fi

# Run validation
"$VALIDATOR" "$1"
EOF

    chmod +x "$COMMIT_MSG_HOOK"
    print_success "Installed commit-msg hook"

    # Install prepare-commit-msg hook
    cat > "$PREPARE_COMMIT_MSG_HOOK" << 'EOF'
#!/bin/bash
#
# prepare-commit-msg hook - Adds LINEAR ticket reference from branch name
# Auto-generated by micro-commit-workflow skill

COMMIT_MSG_FILE="$1"
COMMIT_SOURCE="$2"

# Only add ticket reference for regular commits (not merges, amends, etc.)
if [ -z "$COMMIT_SOURCE" ]; then
    # Get current branch name
    BRANCH_NAME=$(git symbolic-ref --short HEAD 2>/dev/null || echo "")

    # Extract LINEAR ticket from branch name (e.g., feature/LINEAR-123-description)
    if [[ "$BRANCH_NAME" =~ LINEAR-([0-9]+) ]]; then
        LINEAR_TICKET="LINEAR-${BASH_REMATCH[1]}"

        # Check if commit message already contains the ticket
        if ! grep -q "$LINEAR_TICKET" "$COMMIT_MSG_FILE"; then
            # Add ticket reference to commit message
            echo "" >> "$COMMIT_MSG_FILE"
            echo "$LINEAR_TICKET" >> "$COMMIT_MSG_FILE"
        fi
    fi
fi
EOF

    chmod +x "$PREPARE_COMMIT_MSG_HOOK"
    print_success "Installed prepare-commit-msg hook"

    echo ""
    echo -e "${GREEN}=== Installation Complete! ===${NC}"
    echo ""
    echo "Git hooks have been installed successfully."
    echo ""
    echo "What this means:"
    echo "  • Every commit will be validated for conventional format"
    echo "  • Invalid commits will be rejected automatically"
    echo "  • LINEAR tickets will be added from branch names"
    echo ""
    echo "Usage:"
    echo "  git commit -m \"feat: add new feature\"  # Will be validated"
    echo "  git commit --no-verify                  # Skip validation (not recommended)"
    echo ""
    echo "To test the hook:"
    echo "  echo 'feat: test commit' | .git/hooks/commit-msg /dev/stdin"
    echo ""
}

uninstall_hooks() {
    echo -e "${BLUE}=== Uninstalling Git Hooks ===${NC}"
    echo ""

    local removed=0

    # Remove commit-msg hook
    if [ -f "$COMMIT_MSG_HOOK" ]; then
        rm "$COMMIT_MSG_HOOK"
        print_success "Removed commit-msg hook"
        ((removed++))
    fi

    # Remove prepare-commit-msg hook
    if [ -f "$PREPARE_COMMIT_MSG_HOOK" ]; then
        rm "$PREPARE_COMMIT_MSG_HOOK"
        print_success "Removed prepare-commit-msg hook"
        ((removed++))
    fi

    if [ $removed -eq 0 ]; then
        print_info "No hooks were installed"
    else
        echo ""
        print_success "Uninstallation complete ($removed hooks removed)"
    fi

    # Restore backups if they exist
    local backups=$(find "$GIT_HOOKS_DIR" -name "*.backup.*" 2>/dev/null || true)
    if [ -n "$backups" ]; then
        echo ""
        print_info "Backup files found:"
        echo "$backups"
        echo ""
        read -p "Restore backups? (y/N): " restore
        if [[ "$restore" =~ ^[Yy]$ ]]; then
            for backup in $backups; do
                original="${backup%.backup.*}"
                mv "$backup" "$original"
                print_success "Restored: $(basename "$original")"
            done
        fi
    fi

    echo ""
}

# Main script
main() {
    case "${1:-}" in
        --help|-h)
            show_usage
            exit 0
            ;;
        --install)
            check_git_repo
            install_hooks
            exit 0
            ;;
        --uninstall)
            check_git_repo
            uninstall_hooks
            exit 0
            ;;
        --check)
            check_git_repo
            check_installation
            exit 0
            ;;
        *)
            show_usage
            exit 1
            ;;
    esac
}

main "$@"
