#!/bin/bash

#######################################
# Git Hook Setup for Secure-Push
# Installs/removes pre-push security scanning hook
#######################################

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
SCAN_SCRIPT="$SCRIPT_DIR/pre-push-scan.sh"

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

log_info() { echo -e "${BLUE}[INFO]${NC} $1"; }
log_success() { echo -e "${GREEN}[SUCCESS]${NC} $1"; }
log_warning() { echo -e "${YELLOW}[WARNING]${NC} $1"; }
log_error() { echo -e "${RED}[ERROR]${NC} $1"; }

# Check if in a git repository
check_git_repo() {
    if ! git rev-parse --git-dir >/dev/null 2>&1; then
        log_error "Not in a git repository"
        exit 1
    fi
    GIT_DIR=$(git rev-parse --git-dir)
    HOOKS_DIR="$GIT_DIR/hooks"
    PRE_PUSH_HOOK="$HOOKS_DIR/pre-push"
}

# Backup existing hook
backup_existing_hook() {
    if [ -f "$PRE_PUSH_HOOK" ]; then
        local BACKUP="$PRE_PUSH_HOOK.backup.$(date +%s)"
        log_warning "Existing pre-push hook found"
        cp "$PRE_PUSH_HOOK" "$BACKUP"
        log_info "Backed up to: $BACKUP"
        return 0
    fi
    return 1
}

# Install the pre-push hook
install_hook() {
    log_info "Installing pre-push security scanning hook..."

    # Create hooks directory if it doesn't exist
    mkdir -p "$HOOKS_DIR"

    # Check for existing hook
    local EXISTING_BACKUP=false
    if [ -f "$PRE_PUSH_HOOK" ]; then
        # Check if it's already our hook
        if grep -q "secure-push/scripts/pre-push-scan.sh" "$PRE_PUSH_HOOK" 2>/dev/null; then
            log_warning "Secure-push hook already installed"
            log_info "To reinstall, first remove with: $0 --remove"
            exit 0
        fi

        # Backup existing hook
        backup_existing_hook
        EXISTING_BACKUP=true
    fi

    # Create the pre-push hook
    cat > "$PRE_PUSH_HOOK" << 'HOOK_EOF'
#!/bin/bash
#
# Pre-push hook for secure-push security scanning
# Auto-generated by secure-push skill
#
# This hook runs security scans (secrets, SAST, SCA) before allowing push
# Blocks on Critical/High severity findings
#
# To bypass: git push --no-verify (not recommended)
#

# Get the absolute path to the scan script
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
SCAN_SCRIPT="$SCRIPT_DIR/.claude/skills/secure-push/scripts/pre-push-scan.sh"

# Check if scan script exists
if [ ! -f "$SCAN_SCRIPT" ]; then
    echo "ERROR: Secure-push scan script not found at: $SCAN_SCRIPT"
    echo "Please reinstall the hook with: bash .claude/skills/secure-push/scripts/setup-git-hook.sh --install"
    exit 1
fi

# Run the security scan
bash "$SCAN_SCRIPT"
exit $?
HOOK_EOF

    # Make hook executable
    chmod +x "$PRE_PUSH_HOOK"

    log_success "Pre-push hook installed successfully!"
    echo ""
    echo "Location: $PRE_PUSH_HOOK"
    echo ""
    echo "The hook will now run automatically before every 'git push'"
    echo ""

    if [ "$EXISTING_BACKUP" = true ]; then
        log_warning "Your previous hook was backed up"
        log_info "You may need to merge it manually if it contained other checks"
    fi

    echo "To test the hook:"
    echo "  bash $SCAN_SCRIPT"
    echo ""
    echo "To bypass the hook (when needed):"
    echo "  git push --no-verify"
    echo ""
}

# Remove the pre-push hook
remove_hook() {
    log_info "Removing pre-push security scanning hook..."

    if [ ! -f "$PRE_PUSH_HOOK" ]; then
        log_warning "No pre-push hook found"
        exit 0
    fi

    # Check if it's our hook
    if ! grep -q "secure-push/scripts/pre-push-scan.sh" "$PRE_PUSH_HOOK" 2>/dev/null; then
        log_error "Pre-push hook exists but is not from secure-push"
        log_info "Review manually: $PRE_PUSH_HOOK"
        exit 1
    fi

    # Backup before removing
    backup_existing_hook

    # Remove the hook
    rm -f "$PRE_PUSH_HOOK"

    log_success "Pre-push hook removed successfully!"
    echo ""
    echo "Security scans will no longer run automatically"
    echo "You can still run manually with:"
    echo "  bash $SCAN_SCRIPT"
    echo ""
}

# Check hook status
check_status() {
    log_info "Checking pre-push hook status..."
    echo ""

    if [ ! -f "$PRE_PUSH_HOOK" ]; then
        log_warning "Status: NOT INSTALLED"
        echo ""
        echo "To install:"
        echo "  bash $0 --install"
        exit 0
    fi

    if grep -q "secure-push/scripts/pre-push-scan.sh" "$PRE_PUSH_HOOK" 2>/dev/null; then
        log_success "Status: INSTALLED"
        echo ""
        echo "Location: $PRE_PUSH_HOOK"

        # Check if executable
        if [ -x "$PRE_PUSH_HOOK" ]; then
            log_success "Executable: YES"
        else
            log_error "Executable: NO (fixing...)"
            chmod +x "$PRE_PUSH_HOOK"
            log_success "Fixed!"
        fi

        # Test if scan script exists
        if [ -f "$SCAN_SCRIPT" ]; then
            log_success "Scan script: FOUND"
        else
            log_error "Scan script: NOT FOUND"
            log_warning "Hook may not work properly"
        fi
    else
        log_warning "Status: DIFFERENT HOOK INSTALLED"
        echo ""
        echo "A different pre-push hook is installed"
        echo "Location: $PRE_PUSH_HOOK"
        echo ""
        log_info "To install secure-push (will backup existing):"
        echo "  bash $0 --install"
    fi
    echo ""
}

# Interactive installation
interactive_install() {
    echo "════════════════════════════════════════════════════════"
    echo "  Secure-Push Git Hook Setup"
    echo "════════════════════════════════════════════════════════"
    echo ""
    echo "This will install a pre-push hook that automatically scans"
    echo "your code for secrets, vulnerabilities, and CVEs before"
    echo "allowing a git push."
    echo ""
    echo "The hook will:"
    echo "  • Scan for secrets (API keys, tokens, passwords)"
    echo "  • Analyze code for security vulnerabilities"
    echo "  • Check dependencies for known CVEs"
    echo "  • Block pushes with Critical/High severity findings"
    echo ""

    # Check current status
    if [ -f "$PRE_PUSH_HOOK" ]; then
        if grep -q "secure-push/scripts/pre-push-scan.sh" "$PRE_PUSH_HOOK" 2>/dev/null; then
            log_warning "Secure-push hook is already installed"
            echo ""
            read -p "Reinstall? (y/N): " -n 1 -r
            echo ""
            if [[ ! $REPLY =~ ^[Yy]$ ]]; then
                log_info "Installation cancelled"
                exit 0
            fi
            remove_hook
            echo ""
        else
            log_warning "A different pre-push hook is already installed"
            echo ""
            read -p "Replace with secure-push hook? (y/N): " -n 1 -r
            echo ""
            if [[ ! $REPLY =~ ^[Yy]$ ]]; then
                log_info "Installation cancelled"
                exit 0
            fi
        fi
    fi

    read -p "Install pre-push hook? (Y/n): " -n 1 -r
    echo ""

    if [[ $REPLY =~ ^[Nn]$ ]]; then
        log_info "Installation cancelled"
        echo ""
        echo "You can install later with:"
        echo "  bash $0 --install"
        echo ""
        echo "Or run scans manually with:"
        echo "  bash $SCAN_SCRIPT"
        exit 0
    fi

    install_hook
}

# Show usage
show_usage() {
    cat << EOF
Usage: $0 [OPTIONS]

Install or manage the pre-push security scanning git hook.

OPTIONS:
    --install       Install the pre-push hook
    --remove        Remove the pre-push hook
    --status        Check hook installation status
    --interactive   Interactive installation (asks for confirmation)
    --help          Show this help message

EXAMPLES:
    $0 --install            # Install hook directly
    $0 --interactive        # Interactive installation
    $0 --status             # Check if hook is installed
    $0 --remove             # Remove the hook

HOOK BEHAVIOR:
    Once installed, the hook runs automatically before every 'git push'
    - Scans for secrets, code vulnerabilities, and dependency CVEs
    - Blocks push if Critical or High severity findings are detected
    - Shows warnings for Medium and Low severity findings

BYPASS HOOK (when needed):
    git push --no-verify

    Use this only when you've confirmed findings are false positives

EOF
}

# Main
main() {
    check_git_repo

    case "${1:-}" in
        --install)
            install_hook
            ;;
        --remove)
            remove_hook
            ;;
        --status)
            check_status
            ;;
        --interactive)
            interactive_install
            ;;
        --help|-h)
            show_usage
            ;;
        "")
            # No arguments - run interactive by default
            interactive_install
            ;;
        *)
            log_error "Unknown option: $1"
            show_usage
            exit 1
            ;;
    esac
}

main "$@"
