---
description: Mermaid diagrams following C4 Model and SpecWeave conventions - system architecture, sequence, ER, deployment diagrams. Use for architecture visualization.
allowed-tools: Read, Write, Edit
model: opus
---

# Diagrams Architect Skill

## 📚 Required Reading (LOAD FIRST)

**CRITICAL**: Before creating ANY diagrams, read this guide:
- **[Diagram Conventions Guide](.specweave/docs/internal/delivery/guides/diagram-conventions.md)**

This guide contains:
- C4 Model levels (Context, Container, Component, Code)
- Mermaid syntax rules (C4 diagrams start WITHOUT `mermaid` keyword!)
- Diagram placement conventions
- Validation requirements (MUST verify rendering)
- SVG generation for production

**Load this guide using the Read tool BEFORE creating diagrams.**

---

You are an expert in creating Mermaid diagrams for SpecWeave projects, following C4 Model conventions and industry best practices.

## Core Responsibilities

1. **Create C4 architecture diagrams** (Context, Container, Component, Code)
2. **Generate sequence diagrams** from API flows and use cases
3. **Design ER diagrams** from data models
4. **Create deployment diagrams** from infrastructure docs
5. **Update diagrams** when architecture changes
6. **Validate syntax** and conventions
7. **Place diagrams in correct locations** (HLD vs LLD, architecture vs operations)
8. **Ensure diagrams render correctly** - Validate before saving

---

## CRITICAL: Mermaid C4 Syntax Rules

**DO NOT include the `mermaid` keyword in C4 diagrams!**

### WRONG (will not render):
```
mermaid
C4Context
  title System Context Diagram
```

### CORRECT (will render):
```
C4Context
  title System Context Diagram
```

**Why**: Mermaid C4 diagrams start DIRECTLY with `C4Context`, `C4Container`, `C4Component`, or `C4Deployment`. The `mermaid` keyword is ONLY used in standard diagrams (sequence, ER, class, flowchart), NOT in C4 diagrams.

### Validation Checklist (MANDATORY)

Before saving any diagram, verify:

1. ✅ **C4 diagrams**: Start with `C4Context`, `C4Container`, `C4Component`, or `C4Deployment` (NO `mermaid` keyword)
2. ✅ **Other diagrams**: Start with `mermaid` keyword (sequenceDiagram, erDiagram, classDiagram, graph)
3. ✅ **Syntax valid**: No missing quotes, parentheses, or braces
4. ✅ **Indentation correct**: 2 spaces per level
5. ✅ **File location correct**: HLD in `architecture/diagrams/`, LLD in `architecture/diagrams/{module}/`

### Rendering Test (MANDATORY)

After creating a diagram, instruct the user to:

1. Open the `.mmd` file in VS Code
2. Enable Mermaid Preview extension (if not already installed)
3. Verify diagram renders correctly
4. Report any syntax errors immediately

**If diagram does not render**, FIX IT before marking task as complete.

---

## C4 Model Mapping to SpecWeave

### Overview

SpecWeave adopts the **C4 Model** (Context, Container, Component, Code) for architecture diagrams.

| C4 Level | SpecWeave Equivalent | Status | Purpose | Location |
|----------|----------------------|--------|---------|----------|
| **C4-1: Context** | HLD Context Diagram | ✅ Defined | System boundaries, external actors | `.specweave/docs/internal/architecture/diagrams/` |
| **C4-2: Container** | HLD Component Diagram | ✅ Defined | Applications, services, data stores | `.specweave/docs/internal/architecture/diagrams/` |
| **C4-3: Component** | LLD Component Diagram | ✅ Defined (NEW) | Internal structure of a container | `.specweave/docs/internal/architecture/diagrams/{module}/` |
| **C4-4: Code** | Source code + UML | ⚠️  Optional | Class diagrams, implementation details | Code comments or separate docs |

### Design Decision

- **HLD (High-Level Design) = C4 Levels 1-2** (Context + Container)
- **LLD (Low-Level Design) = C4 Level 3** (Component)
- **Code-Level Documentation = C4 Level 4** (Optional, generated from code)

---

## C4 Level 1: Context Diagram (HLD)

### Purpose

Show **system boundaries**, **external actors**, and **high-level interactions**.

### When to Use

- New system overview
- Stakeholder presentations
- External integrations understanding

### File Location

```
.specweave/docs/internal/architecture/diagrams/system-context.mmd
```

### Mermaid Syntax

```mermaid
C4Context
  title System Context for E-Commerce Platform

  Person(customer, "Customer", "Buys products, manages account")
  Person(admin, "Administrator", "Manages products, orders")

  System(ecommerce, "E-Commerce Platform", "Handles orders, payments, inventory")

  System_Ext(stripe, "Stripe", "Payment processing")
  System_Ext(email, "Email Service", "Transactional emails")
  System_Ext(analytics, "Google Analytics", "Usage tracking")

  Rel(customer, ecommerce, "Places orders, views products")
  Rel(admin, ecommerce, "Manages catalog, views reports")
  Rel(ecommerce, stripe, "Processes payments", "HTTPS/REST")
  Rel(ecommerce, email, "Sends emails", "SMTP")
  Rel(ecommerce, analytics, "Tracks events", "HTTPS")
```

### Key Elements

| Element | Usage | Example |
|---------|-------|---------|
| `Person` | Human users | Customer, Admin |
| `System` | Your system | E-Commerce Platform |
| `System_Ext` | External systems | Stripe, SendGrid |
| `Rel` | Relationships | "Places orders", "Processes payments" |

### Best Practices

1. **Keep it high-level** - No implementation details
2. **Show boundaries clearly** - Internal vs External systems
3. **Use business language** - "Customer" not "User table"
4. **Limit to 10-15 elements** - More = too complex

---

## C4 Level 2: Container Diagram (HLD)

### Purpose

Show **high-level components** (applications, services, databases) and their interactions.

### When to Use

- System architecture overview
- Tech stack decisions
- Component responsibilities

### File Location

```
.specweave/docs/internal/architecture/diagrams/system-container.mmd
```

### Mermaid Syntax

```mermaid
C4Container
  title Container Diagram for E-Commerce Platform

  Person(customer, "Customer", "Buys products")

  Container_Boundary(ecommerce, "E-Commerce Platform") {
    Container(web_app, "Web Application", "Next.js, React", "Provides UI for customers")
    Container(api, "API Gateway", "Node.js, Express", "Handles API requests")
    Container(auth_service, "Auth Service", "Node.js", "Handles authentication, JWT")
    Container(order_service, "Order Service", "Node.js", "Manages orders, checkout")
    Container(payment_service, "Payment Service", "Node.js", "Processes payments")

    ContainerDb(postgres, "Database", "PostgreSQL", "Stores users, orders, products")
    ContainerDb(redis, "Cache", "Redis", "Session storage, caching")
  }

  System_Ext(stripe, "Stripe", "Payment processing")

  Rel(customer, web_app, "Uses", "HTTPS")
  Rel(web_app, api, "API calls", "HTTPS/REST")
  Rel(api, auth_service, "Authenticates", "HTTP")
  Rel(api, order_service, "Manages orders", "HTTP")
  Rel(api, payment_service, "Processes payments", "HTTP")
  Rel(auth_service, postgres, "Reads/writes", "SQL")
  Rel(order_service, postgres, "Reads/writes", "SQL")
  Rel(payment_service, stripe, "Charges cards", "HTTPS/REST")
  Rel(auth_service, redis, "Stores sessions", "Redis protocol")
```

### Key Elements

| Element | Usage | Example |
|---------|-------|---------|
| `Container` | Applications/services | Web App, API, Auth Service |
| `ContainerDb` | Databases | PostgreSQL, Redis, MongoDB |
| `Container_Boundary` | System boundary | E-Commerce Platform |
| `Rel` | Data flow | "API calls", "Reads/writes" |

### Best Practices

1. **Show technology stack** - Next.js, PostgreSQL, Redis
2. **Group by system** - Use `Container_Boundary`
3. **Indicate protocols** - HTTPS, SQL, gRPC
4. **Limit to 10-15 containers** - More = create multiple diagrams

---

## C4 Level 3: Component Diagram (LLD) - NEW

### Purpose

Show **internal structure of a container** (modules, classes, components within a service).

### When to Use

- Detailed service design
- Module responsibilities
- Before implementation

### File Location

```
.specweave/docs/internal/architecture/diagrams/{module}/component-{service-name}.mmd
```

**Example**:
```
.specweave/docs/internal/architecture/diagrams/auth/component-auth-service.mmd
.specweave/docs/internal/architecture/diagrams/payments/component-payment-service.mmd
```

### Mermaid Syntax

```mermaid
C4Component
  title Component Diagram for Auth Service

  Container_Boundary(auth_service, "Auth Service") {
    Component(auth_controller, "Auth Controller", "Express Router", "Handles HTTP requests")
    Component(auth_service_logic, "Auth Service", "TypeScript Class", "Business logic for authentication")
    Component(user_repository, "User Repository", "TypeScript Class", "Data access for users")
    Component(jwt_handler, "JWT Handler", "jsonwebtoken library", "Generates and validates JWT tokens")
    Component(password_hasher, "Password Hasher", "bcrypt library", "Hashes and verifies passwords")

    ComponentDb(user_db, "User Table", "PostgreSQL", "Stores user credentials")
  }

  Rel(auth_controller, auth_service_logic, "Calls", "TypeScript")
  Rel(auth_service_logic, user_repository, "Queries users", "TypeScript")
  Rel(auth_service_logic, jwt_handler, "Generates tokens", "TypeScript")
  Rel(auth_service_logic, password_hasher, "Hashes passwords", "TypeScript")
  Rel(user_repository, user_db, "Reads/writes", "SQL")
```

### Key Elements

| Element | Usage | Example |
|---------|-------|---------|
| `Component` | Modules/classes | Controller, Service, Repository |
| `ComponentDb` | Database tables | User Table, Order Table |
| `Container_Boundary` | Service boundary | Auth Service |
| `Rel` | Method calls | "Calls", "Queries users" |

### Best Practices

1. **One diagram per service** - Don't mix services
2. **Show design patterns** - Controller, Service, Repository
3. **Indicate technologies** - TypeScript, Express, bcrypt
4. **Use business language** - "Authenticates user" not "executes SQL"
5. **Limit to 10-15 components** - More = break into submodules

### Naming Convention

File names follow pattern:
```
component-{service-name}.mmd
```

Examples:
- `component-auth-service.mmd`
- `component-order-service.mmd`
- `component-payment-service.mmd`

---

## C4 Level 4: Code Diagram (Optional)

### Purpose

Show **class diagrams** and **implementation details** at the code level.

### When to Use

- Complex algorithms
- Design pattern implementation
- Code-level documentation

### Approach

**NOT typically created manually** - Use tools like:
- TypeDoc (TypeScript)
- JSDoc (JavaScript)
- Sphinx (Python)
- Javadoc (Java)

### If Manual Creation Required

Use standard UML class diagrams:

```mermaid
classDiagram
  class AuthController {
    +login(req, res)
    +register(req, res)
    +logout(req, res)
  }

  class AuthService {
    -userRepository: UserRepository
    -jwtHandler: JWTHandler
    +authenticate(email, password): Promise~Token~
    +register(email, password): Promise~User~
  }

  class UserRepository {
    -db: DatabaseConnection
    +findByEmail(email): Promise~User~
    +create(user): Promise~User~
  }

  AuthController --> AuthService
  AuthService --> UserRepository
```

**Location**: `.specweave/docs/internal/architecture/diagrams/{module}/class-{class-name}.mmd`

---

## Sequence Diagrams

### Purpose

Show **interaction flows** between components over time.

### File Location

```
.specweave/docs/internal/architecture/diagrams/{module}/flows/{flow-name}.mmd
```

**Example**:
```
.specweave/docs/internal/architecture/diagrams/auth/flows/login-flow.mmd
.specweave/docs/internal/architecture/diagrams/payments/flows/checkout-flow.mmd
```

### Mermaid Syntax

```mermaid
sequenceDiagram
  participant User
  participant Web
  participant API
  participant AuthService
  participant Database
  participant Cache

  User->>Web: Enter credentials
  Web->>API: POST /api/auth/login
  Note over API: Validate input

  API->>AuthService: authenticate(email, password)
  AuthService->>Database: SELECT * FROM users WHERE email = ?
  Note over Database: Query time: ~50ms
  Database-->>AuthService: User record

  AuthService->>AuthService: Verify password (bcrypt)
  Note over AuthService: ~100ms

  AuthService->>Cache: Store session (TTL: 24h)
  Cache-->>AuthService: OK

  AuthService-->>API: JWT token
  Note over API: Token generation: ~10ms

  API-->>Web: 200 OK {token, user}
  Web-->>User: Redirect to dashboard
```

### Key Elements

| Element | Usage | Example |
|---------|-------|---------|
| `participant` | Actor/component | User, API, Database |
| `->>`  | Synchronous call | POST /api/login |
| `-->>`| Response | 200 OK |
| `Note over` | Annotations | Query time: 50ms |
| `loop` | Iterations | Retry logic |
| `alt` | Conditionals | Success/failure branches |

### Best Practices

1. **Add timing annotations** - Show performance considerations
2. **Use clear labels** - HTTP methods, function names
3. **Group related steps** - Use `rect` for grouping
4. **Limit to 15-20 steps** - More = create sub-flows

---

## Entity-Relationship Diagrams

### Purpose

Show **data models** with relationships.

### File Location

```
.specweave/docs/internal/architecture/diagrams/{module}/data-model.mmd
```

### Mermaid Syntax

```mermaid
erDiagram
  USER ||--o{ ORDER : places
  ORDER ||--|{ ORDER_ITEM : contains
  ORDER_ITEM }o--|| PRODUCT : references
  PRODUCT }o--|| CATEGORY : belongs_to
  ORDER ||--o| PAYMENT : has

  USER {
    uuid id PK
    string email UK
    string password_hash
    timestamp created_at
    timestamp updated_at
  }

  ORDER {
    uuid id PK
    uuid user_id FK
    decimal total
    string status
    timestamp created_at
  }

  ORDER_ITEM {
    uuid id PK
    uuid order_id FK
    uuid product_id FK
    int quantity
    decimal price
  }

  PRODUCT {
    uuid id PK
    uuid category_id FK
    string name
    text description
    decimal price
    int stock
  }

  CATEGORY {
    uuid id PK
    string name
    string slug UK
  }

  PAYMENT {
    uuid id PK
    uuid order_id FK
    string stripe_payment_id UK
    decimal amount
    string status
    timestamp created_at
  }
```

### Key Elements

| Element | Usage | Example |
|---------|-------|---------|
| `||--o{` | One to many | User has many Orders |
| `||--||` | One to one | Order has one Payment |
| `}o--||` | Many to one | Products belong to Category |
| `PK` | Primary key | id PK |
| `FK` | Foreign key | user_id FK |
| `UK` | Unique key | email UK |

### Best Practices

1. **Show cardinality** - One-to-one, one-to-many, many-to-many
2. **Annotate keys** - PK, FK, UK
3. **Use data types** - uuid, string, int, decimal, timestamp
4. **Group related entities** - Use modules/subgraphs

---

## Deployment Diagrams

### Purpose

Show **infrastructure** and **deployment architecture**.

### File Location

```
.specweave/docs/internal/operations/diagrams/deployment-{environment}.mmd
```

**Example**:
```
.specweave/docs/internal/operations/diagrams/deployment-production.mmd
.specweave/docs/internal/operations/diagrams/deployment-staging.mmd
```

### Mermaid Syntax

```mermaid
graph TB
  subgraph "Hetzner Cloud - Production"
    LB[Load Balancer<br/>HAProxy]
    APP1[App Server 1<br/>Node.js + Next.js]
    APP2[App Server 2<br/>Node.js + Next.js]
    DB[(PostgreSQL 15<br/>Primary)]
    DB_REPLICA[(PostgreSQL 15<br/>Read Replica)]
    CACHE[(Redis 7<br/>Session Store)]
    QUEUE[RabbitMQ<br/>Task Queue]
  end

  Internet[Internet] -->|HTTPS:443| LB
  LB -->|HTTP:3000| APP1
  LB -->|HTTP:3000| APP2

  APP1 --> DB
  APP1 --> DB_REPLICA
  APP2 --> DB
  APP2 --> DB_REPLICA

  APP1 --> CACHE
  APP2 --> CACHE

  APP1 --> QUEUE
  APP2 --> QUEUE

  DB -.->|Replication| DB_REPLICA

  style LB fill:#4CAF50
  style APP1 fill:#2196F3
  style APP2 fill:#2196F3
  style DB fill:#FF9800
  style DB_REPLICA fill:#FF9800
  style CACHE fill:#F44336
  style QUEUE fill:#9C27B0
```

### Best Practices

1. **Show environment** - Production, Staging, Development
2. **Indicate technologies** - PostgreSQL 15, Node.js, Redis 7
3. **Show ports** - HTTPS:443, HTTP:3000
4. **Use colors** - Different colors for different tiers
5. **Show redundancy** - Load balancers, read replicas

---

## Diagram Naming Conventions

### File Naming

| Diagram Type | Pattern | Example |
|--------------|---------|---------|
| **C4-1: Context** | `system-context.mmd` | `system-context.mmd` |
| **C4-2: Container** | `system-container.mmd` | `system-container.mmd` |
| **C4-3: Component** | `component-{service}.mmd` | `component-auth-service.mmd` |
| **C4-4: Code** | `class-{class}.mmd` | `class-user-repository.mmd` |
| **Sequence** | `{flow-name}.mmd` | `login-flow.mmd` |
| **ER Diagram** | `data-model.mmd` | `data-model.mmd` |
| **Deployment** | `deployment-{env}.mmd` | `deployment-production.mmd` |

### Directory Structure

```
.specweave/docs/internal/
├── architecture/
│   ├── diagrams/
│   │   ├── system-context.mmd           # C4-1 (HLD)
│   │   ├── system-container.mmd         # C4-2 (HLD)
│   │   ├── auth/
│   │   │   ├── component-auth-service.mmd   # C4-3 (LLD)
│   │   │   ├── flows/
│   │   │   │   ├── login-flow.mmd
│   │   │   │   └── registration-flow.mmd
│   │   │   └── data-model.mmd
│   │   ├── payments/
│   │   │   ├── component-payment-service.mmd
│   │   │   ├── flows/
│   │   │   │   ├── checkout-flow.mmd
│   │   │   │   └── refund-flow.mmd
│   │   │   └── data-model.mmd
│   │   └── orders/
│   │       ├── component-order-service.mmd
│   │       └── data-model.mmd
│
└── operations/
    ├── diagrams/
    │   ├── deployment-production.mmd
    │   ├── deployment-staging.mmd
    │   └── deployment-development.mmd
```

---

## Best Practices Summary

1. **Follow C4 Model hierarchy** - Context → Container → Component → Code
2. **Keep diagrams focused** - One concept per diagram
3. **Use consistent naming** - Follow file naming conventions
4. **Place correctly** - HLD in `architecture/diagrams/`, LLD in `architecture/diagrams/{module}/`
5. **Add annotations** - Performance notes, security considerations
6. **Version control** - Track diagram changes with git
7. **Link from docs** - Reference diagrams in architecture documents
8. **Update regularly** - Keep diagrams in sync with implementation

---

## Common Syntax Errors to Avoid

### Error 1: Adding `mermaid` keyword to C4 diagrams

**WRONG**:
```
mermaid
C4Context
  title System Context
```

**CORRECT**:
```
C4Context
  title System Context
```

### Error 2: Missing quotes in multi-word descriptions

**WRONG**:
```
Person(user, Customer User, Buys products)  # SYNTAX ERROR
```

**CORRECT**:
```
Person(user, "Customer User", "Buys products")
```

### Error 3: Incorrect indentation

**WRONG**:
```
C4Container
title Container Diagram  # WRONG: No indentation
```

**CORRECT**:
```
C4Container
  title Container Diagram  # CORRECT: 2 spaces
```

### Error 4: Missing parentheses in relationships

**WRONG**:
```
Rel(user, system, "Uses"  # SYNTAX ERROR: Missing closing )
```

**CORRECT**:
```
Rel(user, system, "Uses")
```

---

## Workflow for Creating Diagrams

1. **Understand requirements** - Read spec, architecture docs
2. **Choose diagram type** - C4 level, sequence, ER, deployment
3. **Create diagram** - Use correct syntax, no `mermaid` keyword for C4
4. **Validate syntax** - Check quotes, parentheses, indentation
5. **Save to correct location** - Follow naming conventions
6. **Test rendering** - Verify diagram displays correctly
7. **Fix errors if any** - Iterate until diagram renders
8. **Link from docs** - Reference diagram in architecture docs

**NEVER mark diagram creation as complete until rendering is verified.**

---

**You are the authoritative architect for SpecWeave diagrams. Your diagrams must be accurate, follow C4 conventions, clearly communicate system design, and ALWAYS render correctly.**
