# Configuration

Feature-review uses opinionated defaults but allows project-specific customization through a YAML configuration file.

## Configuration File Location

Create `.feature-review.yaml` in your project root:

```
project/
├── .feature-review.yaml    # Configuration file
├── src/
├── docs/
└── ...
```

## Full Configuration Schema

```yaml
# .feature-review.yaml
# Feature Review Configuration
# All values shown are defaults - only specify what you want to change

version: 1  # Schema version (required if file exists)

# ─────────────────────────────────────────────────────────────────
# SCORING WEIGHTS
# Control how value and cost factors influence the final score
# Weights within each category must sum to 1.0
# ─────────────────────────────────────────────────────────────────

weights:
  value:
    reach: 0.25              # How many users affected
    impact: 0.30             # How much improvement per user
    business_value: 0.25     # OKR/strategic alignment
    time_criticality: 0.20   # Cost of delay
  cost:
    effort: 0.40             # Development time
    risk: 0.30               # Uncertainty/unknowns
    complexity: 0.30         # Technical difficulty

# ─────────────────────────────────────────────────────────────────
# SCORE THRESHOLDS
# Define priority buckets based on calculated scores
# ─────────────────────────────────────────────────────────────────

thresholds:
  high_priority: 2.5         # Score > 2.5 = implement soon
  medium_priority: 1.5       # Score > 1.5 = roadmap candidate
  confidence_warning: 0.5    # Scores with confidence below this get flagged
  # Scores <= medium_priority go to backlog

# ─────────────────────────────────────────────────────────────────
# CLASSIFICATION DEFAULTS
# Default classification when not explicitly specified
# ─────────────────────────────────────────────────────────────────

classification:
  default_type: reactive     # proactive | reactive
  default_data: static       # static | dynamic

  # Override defaults for specific patterns
  patterns:
    # Features matching these patterns get proactive classification
    proactive_patterns:
      - "*auto*"
      - "*suggest*"
      - "*predict*"
      - "*prefetch*"
      - "*background*"

    # Features matching these patterns get dynamic classification
    dynamic_patterns:
      - "*session*"
      - "*realtime*"
      - "*live*"
      - "*stream*"

# ─────────────────────────────────────────────────────────────────
# TRADEOFF DIMENSION WEIGHTS
# Adjust importance of each quality dimension
# Set to 0.0 to disable a dimension entirely
# ─────────────────────────────────────────────────────────────────

tradeoffs:
  quality: 1.0               # Correctness of results
  latency: 1.0               # Response time
  token_usage: 1.0           # Context efficiency (LLM-specific)
  resource_usage: 0.8        # CPU/memory consumption
  redundancy: 0.5            # Fault tolerance
  readability: 1.0           # Code maintainability
  scalability: 0.8           # Growth handling
  integration: 1.0           # Ecosystem fit
  api_surface: 1.0           # Contract stability

# ─────────────────────────────────────────────────────────────────
# GITHUB INTEGRATION
# Configure issue creation behavior
# ─────────────────────────────────────────────────────────────────

github:
  enabled: true              # Enable GitHub integration
  auto_label: true           # Automatically apply labels
  label_prefix: "priority/"  # Prefix for priority labels

  default_labels:            # Labels applied to all created issues
    - enhancement
    - feature-review

  # Labels applied based on priority
  priority_labels:
    high: "priority/high"
    medium: "priority/medium"
    low: "priority/low"

  # Labels applied based on classification
  classification_labels:
    proactive: "type/proactive"
    reactive: "type/reactive"
    static: "data/static"
    dynamic: "data/dynamic"

  # Issue body template (supports {{ variable }} substitution)
  issue_template: |
    ## Feature Request

    **Generated by:** feature-review
    **Priority Score:** {{ score }}
    **Classification:** {{ type }}/{{ data }}
    **Kano Category:** {{ kano }}

    ---

    ### Description
    {{ description }}

    ### Value Proposition
    - **Reach:** {{ reach }} - {{ reach_rationale }}
    - **Impact:** {{ impact }} - {{ impact_rationale }}
    - **Business Value:** {{ business_value }} - {{ bv_rationale }}
    - **Time Criticality:** {{ time_criticality }} - {{ tc_rationale }}

    ### Cost Assessment
    - **Effort:** {{ effort }} - {{ effort_rationale }}
    - **Risk:** {{ risk }} - {{ risk_rationale }}
    - **Complexity:** {{ complexity }} - {{ complexity_rationale }}

    ### Tradeoff Considerations
    {{ tradeoffs_summary }}

    ### Acceptance Criteria
    {{ acceptance_criteria }}

    ---
    *Created via `/feature-review --create-issues`*

# ─────────────────────────────────────────────────────────────────
# INVENTORY SETTINGS
# Control feature discovery behavior
# ─────────────────────────────────────────────────────────────────

inventory:
  # Paths to scan for features
  scan_paths:
    - "commands/"
    - "skills/"
    - "agents/"
    - "src/"

  # Patterns to exclude from scanning
  exclude_patterns:
    - "**/test*"
    - "**/mock*"
    - "**/__pycache__/**"
    - "**/node_modules/**"

  # Feature detection patterns
  feature_markers:
    - "# @feature"           # Comment marker
    - "---\nname:"           # YAML frontmatter
    - "export function"      # JS/TS exports
    - "def "                 # Python functions
    - "class "               # Class definitions

# ─────────────────────────────────────────────────────────────────
# OUTPUT SETTINGS
# Control report generation
# ─────────────────────────────────────────────────────────────────

output:
  format: markdown           # markdown | json | yaml
  include_rationale: true    # Include scoring rationale
  include_tradeoffs: true    # Include tradeoff analysis
  max_suggestions: 10        # Maximum suggestions to generate

  # Report sections to include
  sections:
    - inventory
    - classification
    - scoring
    - tradeoffs
    - suggestions
    - github_actions

# ─────────────────────────────────────────────────────────────────
# BACKLOG SETTINGS
# Control backlog management
# ─────────────────────────────────────────────────────────────────

backlog:
  max_items: 25              # Maximum items in suggestion queue (guardrail)
  stale_days: 30             # Days until item is considered stale
  auto_archive: false        # Automatically archive stale items

  # Backlog file location
  file: "docs/backlog/feature-queue.md"
```

## Minimal Configuration Examples

### Startup (Move Fast)

```yaml
version: 1

thresholds:
  high_priority: 2.0         # Lower bar for "implement now"
  medium_priority: 1.0

tradeoffs:
  redundancy: 0.3            # Less focus on fault tolerance
  scalability: 0.5           # Scale later
  api_surface: 0.5           # API can change

backlog:
  max_items: 15              # Smaller backlog, more focus
```

### Enterprise (Stability First)

```yaml
version: 1

thresholds:
  high_priority: 3.0         # Higher bar for priority
  confidence_warning: 0.7    # Require more certainty

tradeoffs:
  api_surface: 1.5           # Extra weight on API stability
  redundancy: 1.2            # More focus on fault tolerance
  readability: 1.2           # Maintainability matters

github:
  default_labels:
    - enhancement
    - needs-review
    - feature-review
```

### LLM/AI Project

```yaml
version: 1

weights:
  value:
    reach: 0.20
    impact: 0.35             # User experience critical
    business_value: 0.20
    time_criticality: 0.25

tradeoffs:
  token_usage: 1.5           # Token efficiency very important
  quality: 1.3               # Result quality critical
  latency: 1.2               # Response time matters
```

## Guardrails (Always Enforced)

These rules apply regardless of configuration:

### 1. Minimum Dimensions

At least 5 tradeoff dimensions must have non-zero weight.

```yaml
# INVALID - only 4 dimensions
tradeoffs:
  quality: 1.0
  latency: 1.0
  token_usage: 1.0
  readability: 1.0
  # All others: 0.0
```

### 2. Weight Sum Validation

Weights within each category must sum to 1.0 (within 0.01 tolerance).

```yaml
# INVALID - sums to 1.1
weights:
  value:
    reach: 0.30
    impact: 0.30
    business_value: 0.30
    time_criticality: 0.20
```

### 3. Confidence Threshold

Features with confidence below `confidence_warning` are always flagged, regardless of score.

### 4. Breaking Change Warning

Features classified as having API surface changes require explicit acknowledgment:

```yaml
# Cannot be disabled
api_surface_changes:
  require_acknowledgment: true  # Always true
```

### 5. Backlog Limit

Maximum 25 items in backlog (forces prioritization decisions).

```yaml
# INVALID - exceeds guardrail
backlog:
  max_items: 50  # Will be capped to 25
```

## Environment Variable Overrides

Configuration can be overridden via environment variables:

```bash
# Override priority threshold
FEATURE_REVIEW_HIGH_PRIORITY=3.0

# Disable GitHub integration
FEATURE_REVIEW_GITHUB_ENABLED=false

# Change output format
FEATURE_REVIEW_OUTPUT_FORMAT=json
```

Pattern: `FEATURE_REVIEW_` + uppercase path with underscores.

## Configuration Validation

Run validation before using:

```bash
/feature-review --validate-config
```

Output:
```
Configuration: .feature-review.yaml
Status: Valid

Warnings:
- token_usage weight (1.5) exceeds typical range (0.5-1.2)

Applied Guardrails:
- backlog.max_items capped at 25 (was 50)
```

## Inheritance and Overrides

### Directory-Level Config

Configuration can be overridden per directory:

```
project/
├── .feature-review.yaml           # Project defaults
├── plugins/
│   └── .feature-review.yaml       # Plugin-specific overrides
└── experimental/
    └── .feature-review.yaml       # Experimental area config
```

Child configs inherit from parent and override specific values.

### Command-Line Overrides

Override any config value via command line:

```bash
# Override threshold
/feature-review --threshold.high_priority=3.0

# Override weight
/feature-review --weights.value.impact=0.4

# Disable GitHub
/feature-review --github.enabled=false
```

## Migration Guide

### From No Configuration

1. Run `/feature-review` with defaults
2. Review output for misaligned priorities
3. Identify dimensions needing adjustment
4. Create minimal `.feature-review.yaml` with only changed values

### From Other Frameworks

**From RICE:**
```yaml
# RICE uses only Reach, Impact, Confidence, Effort
# Feature-review extends this - adjust weights to match RICE behavior:
weights:
  value:
    reach: 0.35
    impact: 0.35
    business_value: 0.15
    time_criticality: 0.15
  cost:
    effort: 0.70
    risk: 0.15
    complexity: 0.15
```

**From MoSCoW:**
```yaml
# Map MoSCoW categories to thresholds:
thresholds:
  high_priority: 3.0      # Must Have
  medium_priority: 2.0    # Should Have
  # 1.0-2.0 = Could Have
  # < 1.0 = Won't Have
```

## Project-Type Templates

### LLM/AI Plugin Project

For projects building LLM tooling where token efficiency is critical:

```yaml
version: 1

weights:
  value:
    reach: 0.20              # Fewer users than consumer apps
    impact: 0.35             # Developer experience critical
    business_value: 0.20     # Community value focus
    time_criticality: 0.25   # Fast AI iteration

tradeoffs:
  token_usage: 1.4           # CRITICAL - users pay per token
  integration: 1.3           # Must compose with other tools
  api_surface: 1.3           # Breaking changes hurt ecosystem
  readability: 1.2           # OSS needs readable code
  scalability: 0.6           # Single-user CLI, not web scale
```

### SaaS Product

For B2B SaaS with paying customers:

```yaml
version: 1

weights:
  value:
    reach: 0.30              # Customer count matters
    impact: 0.25             # Per-customer value
    business_value: 0.30     # Revenue alignment
    time_criticality: 0.15   # Planned releases

thresholds:
  high_priority: 3.0         # High bar for resources
  confidence_warning: 0.7    # Require high certainty

tradeoffs:
  quality: 1.3               # Bugs hurt reputation
  redundancy: 1.2            # Uptime is critical
  scalability: 1.3           # Must handle growth
  api_surface: 1.2           # Customer integrations
```

### Internal Tool / Developer Platform

For internal developer productivity tools:

```yaml
version: 1

weights:
  value:
    reach: 0.35              # How many devs affected
    impact: 0.30             # Time savings per dev
    business_value: 0.15     # Hard to measure directly
    time_criticality: 0.20   # Developer frustration

tradeoffs:
  latency: 1.3               # Devs hate slow tools
  readability: 1.2           # Self-service maintenance
  integration: 1.3           # Must fit dev workflow
  redundancy: 0.5            # Internal = less critical
  api_surface: 0.7           # Can iterate faster
```

### Mobile App

For consumer mobile applications:

```yaml
version: 1

weights:
  value:
    reach: 0.30              # MAU/DAU focus
    impact: 0.35             # UX is everything
    business_value: 0.20     # Monetization alignment
    time_criticality: 0.15   # App store cycles

tradeoffs:
  quality: 1.3               # App store reviews
  latency: 1.4               # Mobile UX critical
  resource_usage: 1.3        # Battery/data constraints
  token_usage: 0.5           # Less relevant
  scalability: 0.8           # Backend concern
```

## Advanced Patterns

### Per-Plugin Configuration (Monorepo)

In a monorepo with multiple plugins/packages, use directory-specific configs:

```
monorepo/
├── .feature-review.yaml           # Global defaults
├── packages/
│   ├── core/
│   │   └── .feature-review.yaml   # Core: high stability
│   ├── experimental/
│   │   └── .feature-review.yaml   # Experimental: lower bar
│   └── deprecated/
│       └── .feature-review.yaml   # Deprecated: no new features
```

**packages/core/.feature-review.yaml:**
```yaml
version: 1
thresholds:
  high_priority: 3.5         # Very high bar for core
tradeoffs:
  api_surface: 1.5           # Stability paramount
```

**packages/experimental/.feature-review.yaml:**
```yaml
version: 1
thresholds:
  high_priority: 2.0         # Lower bar for experiments
  confidence_warning: 0.4    # OK to be less certain
tradeoffs:
  api_surface: 0.5           # Can break freely
```

### Custom Scoring Dimensions

Add project-specific scoring dimensions:

```yaml
version: 1

# Custom dimensions (extend tradeoffs)
custom_dimensions:
  regulatory_compliance:
    weight: 1.5
    description: "Meets GDPR/SOC2/HIPAA requirements"
    scoring:
      5: "Fully compliant, documented"
      3: "Compliant with minor gaps"
      1: "Compliance concerns"

  accessibility:
    weight: 1.2
    description: "WCAG 2.1 AA compliance"
    scoring:
      5: "Full WCAG 2.1 AA"
      3: "Partial compliance"
      1: "Accessibility issues"
```

### Conditional Configuration

Apply different weights based on feature classification:

```yaml
version: 1

# Base weights
weights:
  value:
    reach: 0.25
    impact: 0.30
    business_value: 0.25
    time_criticality: 0.20

# Override for proactive features
conditional:
  proactive:
    tradeoffs:
      latency: 0.6           # Less critical for background
      resource_usage: 1.2    # More important for background

  dynamic:
    tradeoffs:
      redundancy: 1.2        # Need fault tolerance
      scalability: 1.2       # More load concerns
```
