#!/usr/bin/env python3
"""
Security Report Generator - Generates security assessment reports in MD/HTML formats.
"""
import argparse
import json
from datetime import datetime
from pathlib import Path

SCRIPT_DIR = Path(__file__).parent.parent
TEMPLATE_DIR = SCRIPT_DIR / "assets" / "templates"


def calculate_risk_score(findings: dict) -> dict:
    """Calculate overall risk score based on findings."""
    scores = {"critical": 0, "high": 0, "medium": 0, "low": 0}
    for category, data in findings.items():
        if isinstance(data, dict):
            for key, value in data.items():
                if isinstance(value, list):
                    for item in value:
                        severity = item.get("severity", "").lower() if isinstance(item, dict) else ""
                        if severity in scores:
                            scores[severity] += 1
    total = scores["critical"] * 10 + scores["high"] * 5 + scores["medium"] * 2 + scores["low"]
    risk_score = min(total, 100)
    level = "Critical" if risk_score >= 80 else "High" if risk_score >= 50 else "Medium" if risk_score >= 20 else "Low"
    return {"score": risk_score, "level": level, "breakdown": scores}


def build_findings_section(findings: dict) -> str:
    """Build detailed findings section for markdown report."""
    sections = []
    if "dependencies" in findings:
        deps = findings["dependencies"]
        sections.append("### Dependency Vulnerabilities\n")
        if deps.get("npm"):
            vulns = deps["npm"].get("vulnerabilities", {})
            sections.append(f"**NPM Packages:** {len(vulns)} vulnerabilities found\n")
        if deps.get("pip"):
            sections.append(f"**Python Packages:** {len(deps['pip'])} vulnerabilities found\n")
    if "secrets" in findings:
        secrets = findings["secrets"]
        sections.append("### Secret Detection\n")
        if secrets.get("trufflehog"):
            sections.append(f"**Exposed Secrets Found:** {len(secrets['trufflehog'])}\n")
            sections.append("| Type | File |\n|------|------|\n")
            for s in secrets["trufflehog"][:10]:
                sections.append(f"| {s.get('DetectorName', 'Unknown')} | {s.get('SourceMetadata', {}).get('filename', 'N/A')} |\n")
        else:
            sections.append("No exposed secrets detected.\n")
    if "system" in findings:
        system = findings["system"]
        sections.append("### System Security\n")
        sections.append(f"- **Open Ports:** {system.get('open_ports', 'N/A')}\n")
        sections.append(f"- **SSH Config:** {'Present' if system.get('ssh_config', {}).get('exists') else 'Not found'}\n")
    if "remote" in findings:
        remote = findings["remote"]
        sections.append("### Remote Server Assessment\n")
        sections.append(f"- **Connection:** {'Success' if remote.get('connection') else 'Failed'}\n")
        if remote.get("checks"):
            sections.append(f"- **Pending Updates:** {remote['checks'].get('pending_updates', 'N/A')}\n")
    return "".join(sections)


def generate_markdown_report(scan_data: dict, output_path: Path) -> Path:
    """Generate Markdown format security report."""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    risk = calculate_risk_score(scan_data.get("findings", {}))
    findings_section = build_findings_section(scan_data.get("findings", {}))
    risk_msg = {
        "Critical": "**IMMEDIATE ACTION REQUIRED**",
        "High": "Significant vulnerabilities detected requiring prompt attention.",
        "Medium": "Some security improvements recommended.",
        "Low": "System security is acceptable with minor improvements."
    }
    report = f"""# Security Assessment Report

**Generated:** {timestamp} | **Target:** {scan_data.get('target', 'Unknown')} | **Scan Time:** {scan_data.get('timestamp', 'N/A')}

---

## Executive Summary

| Metric | Value |
|--------|-------|
| **Risk Score** | {risk['score']}/100 ({risk['level']}) |
| **Critical** | {risk['breakdown']['critical']} |
| **High** | {risk['breakdown']['high']} |
| **Medium** | {risk['breakdown']['medium']} |
| **Low** | {risk['breakdown']['low']} |

{risk_msg.get(risk['level'], '')}

---

## Detailed Findings

{findings_section}

---

## Remediation Recommendations

### Priority 1 - Critical (Immediate)
1. Patch all critical CVEs within 24 hours
2. Rotate any exposed credentials immediately

### Priority 2 - High (Within 1 Week)
1. Update all dependencies with known vulnerabilities
2. Enable audit logging where absent

### Priority 3 - Medium (Within 1 Month)
1. Review and harden system configurations
2. Set up continuous security monitoring

---

## Compliance Status

| Framework | Status |
|-----------|--------|
| OWASP Top 10 2025 | Review Required |
| CIS Benchmarks | Partial |
| NIST CSF | Partial |

---

*Report generated by DevSecOps Engineer Workflow*
"""
    report_file = output_path / f"security_report_{datetime.now().strftime('%Y%m%d_%H%M%S')}.md"
    report_file.write_text(report)
    return report_file


def generate_html_report(scan_data: dict, output_path: Path) -> Path:
    """Generate HTML format security report using template."""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    risk = calculate_risk_score(scan_data.get("findings", {}))
    template_path = TEMPLATE_DIR / "report_template.html"
    if template_path.exists():
        html = template_path.read_text()
    else:
        html = "<html><body><h1>Security Report</h1><p>Template not found.</p></body></html>"
    replacements = {
        "{{TIMESTAMP}}": timestamp,
        "{{TARGET}}": scan_data.get('target', 'Unknown'),
        "{{RISK_LEVEL}}": risk['level'],
        "{{RISK_LEVEL_LOWER}}": risk['level'].lower(),
        "{{RISK_SCORE}}": str(risk['score']),
        "{{CRITICAL}}": str(risk['breakdown']['critical']),
        "{{HIGH}}": str(risk['breakdown']['high']),
        "{{MEDIUM}}": str(risk['breakdown']['medium']),
        "{{LOW}}": str(risk['breakdown']['low']),
    }
    for key, value in replacements.items():
        html = html.replace(key, value)
    report_file = output_path / f"security_report_{datetime.now().strftime('%Y%m%d_%H%M%S')}.html"
    report_file.write_text(html)
    return report_file


def main():
    parser = argparse.ArgumentParser(description="Security Report Generator")
    parser.add_argument("--input", required=True, help="Directory with scan results")
    parser.add_argument("--format", default="markdown,html", help="Output formats (comma-separated)")
    parser.add_argument("--output", default="./final-reports", help="Output directory")
    args = parser.parse_args()

    input_dir, output_dir = Path(args.input), Path(args.output)
    output_dir.mkdir(parents=True, exist_ok=True)

    scan_files = list(input_dir.glob("scan_*.json"))
    if not scan_files:
        print("[!] No scan results found in input directory")
        return

    latest_scan = max(scan_files, key=lambda p: p.stat().st_mtime)
    print(f"[*] Processing scan: {latest_scan.name}")
    scan_data = json.loads(latest_scan.read_text())
    formats = [f.strip().lower() for f in args.format.split(",")]
    generated = []

    if "markdown" in formats:
        md_report = generate_markdown_report(scan_data, output_dir)
        print(f"[+] Markdown report: {md_report}")
        generated.append(md_report)
    if "html" in formats:
        html_report = generate_html_report(scan_data, output_dir)
        print(f"[+] HTML report: {html_report}")
        generated.append(html_report)

    print(f"\n[+] Generated {len(generated)} reports in {output_dir}")


if __name__ == "__main__":
    main()
