#!/usr/bin/env node

/**
 * Kali Linux Connector
 *
 * Connect to Kali Linux via Docker exec or SSH and execute pentesting commands.
 *
 * Usage:
 *   node kali_connector.js setup docker [container-name]
 *   node kali_connector.js setup ssh <user@host:port>
 *   node kali_connector.js test
 *   node kali_connector.js status
 *   node kali_connector.js start
 *   node kali_connector.js exec <command>
 *   node kali_connector.js scan <url> --level <level>
 *   node kali_connector.js collect-results <output-dir>
 *   node kali_connector.js report
 */

import { exec, spawn } from "node:child_process";
import { promisify } from "node:util";
import { readFile, writeFile, mkdir } from "node:fs/promises";
import { existsSync } from "node:fs";
import { homedir } from "node:os";
import { join } from "node:path";

const execPromise = promisify(exec);

const CONFIG_DIR = join(homedir(), ".kali-pentester");
const CONFIG_FILE = join(CONFIG_DIR, "config.json");
const RESULTS_DIR = join(CONFIG_DIR, "results");

// Default configuration
let config = {
  connectionType: "docker", // docker or ssh
  dockerContainer: "kali",
  sshHost: null,
  sshUser: null,
  sshPort: 22,
  lastTarget: null,
  testLevel: "basic"
};

/**
 * Initialize configuration directory
 */
async function initConfig() {
  if (!existsSync(CONFIG_DIR)) {
    await mkdir(CONFIG_DIR, { recursive: true });
  }
  if (!existsSync(RESULTS_DIR)) {
    await mkdir(RESULTS_DIR, { recursive: true });
  }
  if (existsSync(CONFIG_FILE)) {
    try {
      const data = await readFile(CONFIG_FILE, "utf-8");
      config = { ...config, ...JSON.parse(data) };
    } catch (e) {
      // Use defaults
    }
  }
}

/**
 * Save configuration
 */
async function saveConfig() {
  await writeFile(CONFIG_FILE, JSON.stringify(config, null, 2));
}

/**
 * Execute command in Kali container/host
 */
async function executeInKali(command, options = {}) {
  const timeout = options.timeout || 300000; // 5 min default

  let fullCommand;
  if (config.connectionType === "docker") {
    // Escape command for docker exec
    const escapedCmd = command.replace(/"/g, '\\"');
    fullCommand = `docker exec ${config.dockerContainer} bash -c "${escapedCmd}"`;
  } else {
    // SSH connection
    const sshTarget = `${config.sshUser}@${config.sshHost}`;
    const escapedCmd = command.replace(/"/g, '\\"');
    fullCommand = `ssh -p ${config.sshPort} ${sshTarget} "${escapedCmd}"`;
  }

  try {
    const { stdout, stderr } = await execPromise(fullCommand, { timeout });
    return { success: true, stdout: stdout.trim(), stderr: stderr.trim() };
  } catch (error) {
    return { success: false, error: error.message, stdout: error.stdout, stderr: error.stderr };
  }
}

/**
 * Setup connection
 */
async function setupConnection(type, target) {
  await initConfig();

  if (type === "docker") {
    config.connectionType = "docker";
    config.dockerContainer = target || "kali";
    console.log(`✅ Configured for Docker container: ${config.dockerContainer}`);
  } else if (type === "ssh") {
    // Parse user@host:port
    const match = target.match(/^([^@]+)@([^:]+)(?::(\d+))?$/);
    if (!match) {
      console.error("❌ Invalid SSH format. Use: user@host:port");
      process.exit(1);
    }
    config.connectionType = "ssh";
    config.sshUser = match[1];
    config.sshHost = match[2];
    config.sshPort = match[3] ? parseInt(match[3]) : 22;
    console.log(`✅ Configured for SSH: ${config.sshUser}@${config.sshHost}:${config.sshPort}`);
  }

  await saveConfig();
}

/**
 * Test connection
 */
async function testConnection() {
  await initConfig();
  console.log(`🔍 Testing ${config.connectionType} connection...`);

  const result = await executeInKali("uname -a && kali-tweaks --version 2>/dev/null || echo 'Kali tools available'");

  if (result.success) {
    console.log("✅ Connection successful!");
    console.log(`   System: ${result.stdout.split('\n')[0]}`);
    return true;
  } else {
    console.error("❌ Connection failed:", result.error);
    return false;
  }
}

/**
 * Check container/host status
 */
async function checkStatus() {
  await initConfig();

  if (config.connectionType === "docker") {
    try {
      const { stdout } = await execPromise(
        `docker ps --filter "name=${config.dockerContainer}" --format "{{.Status}}"`
      );
      if (stdout.trim()) {
        console.log(`🟢 Container '${config.dockerContainer}' is running`);
        console.log(`   Status: ${stdout.trim()}`);
        return true;
      } else {
        console.log(`🔴 Container '${config.dockerContainer}' is not running`);
        return false;
      }
    } catch (error) {
      console.error("❌ Failed to check status:", error.message);
      return false;
    }
  } else {
    // SSH - just test connection
    return await testConnection();
  }
}

/**
 * Start Docker container
 */
async function startContainer() {
  await initConfig();

  if (config.connectionType !== "docker") {
    console.log("⚠️ Start command only applies to Docker connections");
    return false;
  }

  try {
    // Check if container exists
    const { stdout: existing } = await execPromise(
      `docker ps -a --filter "name=${config.dockerContainer}" --format "{{.Names}}"`
    );

    if (existing.trim()) {
      // Start existing container
      await execPromise(`docker start ${config.dockerContainer}`);
      console.log(`✅ Container '${config.dockerContainer}' started`);
    } else {
      // Create and start new Kali container
      console.log("📦 Creating new Kali container...");
      await execPromise(
        `docker run -d --name ${config.dockerContainer} ` +
        `--cap-add=NET_ADMIN ` +
        `-v ${RESULTS_DIR}:/results ` +
        `kalilinux/kali-rolling tail -f /dev/null`
      );
      console.log(`✅ Container '${config.dockerContainer}' created and started`);

      // Install basic tools
      console.log("📥 Installing pentesting tools (this may take a while)...");
      await executeInKali("apt-get update && apt-get install -y kali-tools-web nmap sqlmap nikto", { timeout: 600000 });
      console.log("✅ Basic tools installed");
    }
    return true;
  } catch (error) {
    console.error("❌ Failed to start container:", error.message);
    return false;
  }
}

/**
 * Run automated scan
 */
async function runScan(url, level = "basic") {
  await initConfig();
  config.lastTarget = url;
  config.testLevel = level;
  await saveConfig();

  console.log(`\n🎯 Starting ${level} scan on: ${url}`);
  console.log("═".repeat(60));

  const timestamp = new Date().toISOString().replace(/[:.]/g, "-");
  const scanDir = `/results/scan_${timestamp}`;

  await executeInKali(`mkdir -p ${scanDir}`);

  // Level-based scanning
  const scans = [];

  // Recon level
  if (["recon", "basic", "standard", "advanced", "expert"].includes(level)) {
    console.log("\n📍 Phase 1: Reconnaissance");
    scans.push({ name: "Nmap scan", cmd: `nmap -sV -sC ${new URL(url).hostname} -oN ${scanDir}/nmap.txt` });
    scans.push({ name: "WhatWeb", cmd: `whatweb ${url} > ${scanDir}/whatweb.txt 2>&1` });
  }

  // Basic level
  if (["basic", "standard", "advanced", "expert"].includes(level)) {
    console.log("\n🔍 Phase 2: Vulnerability Scanning");
    scans.push({ name: "Nikto", cmd: `nikto -h ${url} -o ${scanDir}/nikto.txt` });
  }

  // Standard level
  if (["standard", "advanced", "expert"].includes(level)) {
    console.log("\n💉 Phase 3: Injection Testing");
    scans.push({ name: "SQLmap", cmd: `sqlmap -u "${url}" --batch --output-dir=${scanDir}/sqlmap` });
  }

  // Run scans
  for (const scan of scans) {
    console.log(`  ⏳ Running ${scan.name}...`);
    const result = await executeInKali(scan.cmd, { timeout: 600000 });
    if (result.success) {
      console.log(`  ✅ ${scan.name} completed`);
    } else {
      console.log(`  ⚠️ ${scan.name} had issues: ${result.error?.substring(0, 100)}`);
    }
  }

  console.log(`\n📁 Results saved to: ${scanDir}`);
  return scanDir;
}

/**
 * Collect results from container
 */
async function collectResults(outputDir) {
  await initConfig();

  const localDir = outputDir || join(RESULTS_DIR, `collected_${Date.now()}`);

  if (config.connectionType === "docker") {
    await execPromise(`docker cp ${config.dockerContainer}:/results/. "${localDir}"`);
  } else {
    await execPromise(`scp -P ${config.sshPort} -r ${config.sshUser}@${config.sshHost}:/tmp/pentest-results/. "${localDir}"`);
  }

  console.log(`✅ Results collected to: ${localDir}`);
}

/**
 * Generate summary report
 */
async function generateReport() {
  await initConfig();

  console.log("\n📊 Generating Penetration Test Report");
  console.log("═".repeat(60));

  const report = {
    timestamp: new Date().toISOString(),
    target: config.lastTarget,
    level: config.testLevel,
    connectionType: config.connectionType,
    findings: []
  };

  // Read and parse results
  const result = await executeInKali("find /results -name '*.txt' -exec cat {} \\; 2>/dev/null | head -500");

  if (result.success && result.stdout) {
    // Parse for vulnerabilities
    const vulnPatterns = [
      { pattern: /VULNERABLE/gi, severity: "HIGH" },
      { pattern: /SQL injection/gi, severity: "CRITICAL" },
      { pattern: /XSS/gi, severity: "HIGH" },
      { pattern: /Directory listing/gi, severity: "MEDIUM" },
      { pattern: /Server version/gi, severity: "INFO" }
    ];

    for (const { pattern, severity } of vulnPatterns) {
      const matches = result.stdout.match(pattern);
      if (matches) {
        report.findings.push({ type: pattern.source, severity, count: matches.length });
      }
    }
  }

  console.log(`\n📋 Report Summary`);
  console.log(`   Target: ${report.target}`);
  console.log(`   Level: ${report.level}`);
  console.log(`   Findings: ${report.findings.length} categories`);

  for (const finding of report.findings) {
    const icon = finding.severity === "CRITICAL" ? "🔴" :
                 finding.severity === "HIGH" ? "🟠" :
                 finding.severity === "MEDIUM" ? "🟡" : "🔵";
    console.log(`   ${icon} ${finding.type}: ${finding.count} occurrence(s)`);
  }

  // Save report
  const reportPath = join(RESULTS_DIR, `report_${Date.now()}.json`);
  await writeFile(reportPath, JSON.stringify(report, null, 2));
  console.log(`\n📄 Full report saved: ${reportPath}`);

  return report;
}

// CLI
async function main() {
  const args = process.argv.slice(2);
  const command = args[0];

  switch (command) {
    case "setup":
      await setupConnection(args[1], args[2]);
      break;

    case "test":
      await testConnection();
      break;

    case "status":
      await checkStatus();
      break;

    case "start":
      await startContainer();
      break;

    case "exec":
      await initConfig();
      const cmd = args.slice(1).join(" ");
      const result = await executeInKali(cmd);
      if (result.success) {
        console.log(result.stdout);
      } else {
        console.error("Error:", result.error);
        if (result.stderr) console.error(result.stderr);
      }
      break;

    case "scan":
      const url = args[1];
      const levelIdx = args.indexOf("--level");
      const level = levelIdx > -1 ? args[levelIdx + 1] : "basic";
      if (!url) {
        console.error("❌ URL required: node kali_connector.js scan <url> --level <level>");
        process.exit(1);
      }
      await runScan(url, level);
      break;

    case "collect-results":
      await collectResults(args[1]);
      break;

    case "report":
      await generateReport();
      break;

    default:
      console.log(`
Kali Linux Connector

Setup:
  setup docker [container]     Configure Docker connection (default: kali)
  setup ssh <user@host:port>   Configure SSH connection

Connection:
  test                         Test connection to Kali
  status                       Check container/host status
  start                        Start Docker container

Commands:
  exec <command>               Execute command in Kali
  scan <url> --level <level>   Run automated scan (recon/basic/standard/advanced/expert)
  collect-results [dir]        Collect results to local directory
  report                       Generate summary report

Examples:
  node kali_connector.js setup docker kali-linux
  node kali_connector.js test
  node kali_connector.js scan https://example.com --level standard
  node kali_connector.js exec "nmap -sV target.com"
      `);
  }
}

export {
  executeInKali,
  setupConnection,
  testConnection,
  checkStatus,
  startContainer,
  runScan,
  collectResults,
  generateReport
};

main().catch(console.error);
