#!/usr/bin/env node

/**
 * Professional Penetration Test Report Generator
 *
 * Generates executive-level security assessment reports in HTML/Markdown format.
 *
 * Usage:
 *   node report_generator.js generate <scan-result.json> [--format html|md] [--output <file>]
 *   node report_generator.js list
 *   node report_generator.js latest [--format html|md]
 */

import { readFile, writeFile, readdir } from "node:fs/promises";
import { existsSync } from "node:fs";
import { homedir } from "node:os";
import { join, basename } from "node:path";

const RESULTS_DIR = join(homedir(), ".kali-pentester", "results");

// CVSS Score mapping for severity
const CVSS_SCORES = {
  CRITICAL: { min: 9.0, max: 10.0, color: "#d32f2f" },
  HIGH: { min: 7.0, max: 8.9, color: "#f57c00" },
  MEDIUM: { min: 4.0, max: 6.9, color: "#fbc02d" },
  LOW: { min: 0.1, max: 3.9, color: "#1976d2" },
  INFO: { min: 0, max: 0, color: "#757575" }
};

// Remediation recommendations database
const REMEDIATION_DB = {
  "Missing X-Frame-Options": {
    description: "The X-Frame-Options header prevents clickjacking attacks by controlling whether a page can be embedded in frames.",
    impact: "Attackers could embed your site in a malicious page and trick users into clicking hidden elements.",
    remediation: "Add the header: `X-Frame-Options: DENY` or `X-Frame-Options: SAMEORIGIN`",
    references: [
      "https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options",
      "https://owasp.org/www-project-secure-headers/"
    ],
    cwe: "CWE-1021: Improper Restriction of Rendered UI Layers"
  },
  "Missing Content-Security-Policy": {
    description: "Content Security Policy (CSP) mitigates XSS and data injection attacks by defining approved content sources.",
    impact: "Without CSP, the application is more vulnerable to cross-site scripting (XSS) attacks.",
    remediation: "Implement a strict CSP: `Content-Security-Policy: default-src 'self'; script-src 'self'`",
    references: [
      "https://developer.mozilla.org/en-US/docs/Web/HTTP/CSP",
      "https://csp.withgoogle.com/docs/index.html"
    ],
    cwe: "CWE-79: Cross-site Scripting (XSS)"
  },
  "Missing HSTS": {
    description: "HTTP Strict Transport Security (HSTS) ensures browsers only connect via HTTPS.",
    impact: "Users could be vulnerable to man-in-the-middle attacks during initial HTTP connection.",
    remediation: "Add the header: `Strict-Transport-Security: max-age=31536000; includeSubDomains`",
    references: [
      "https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security",
      "https://hstspreload.org/"
    ],
    cwe: "CWE-319: Cleartext Transmission of Sensitive Information"
  },
  "Missing X-Content-Type-Options": {
    description: "This header prevents browsers from MIME-sniffing responses away from declared content-type.",
    impact: "Attackers could trick browsers into executing malicious content as scripts.",
    remediation: "Add the header: `X-Content-Type-Options: nosniff`",
    references: [
      "https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
    ],
    cwe: "CWE-16: Configuration"
  },
  "SQL Injection Vulnerability": {
    description: "SQL injection allows attackers to interfere with database queries.",
    impact: "Complete database compromise, data theft, authentication bypass, and potential server takeover.",
    remediation: "Use parameterized queries/prepared statements. Never concatenate user input into SQL.",
    references: [
      "https://owasp.org/www-community/attacks/SQL_Injection",
      "https://cheatsheetseries.owasp.org/cheatsheets/SQL_Injection_Prevention_Cheat_Sheet.html"
    ],
    cwe: "CWE-89: SQL Injection"
  },
  "Weak SSL/TLS Protocol": {
    description: "Server supports outdated SSL/TLS protocols vulnerable to known attacks.",
    impact: "Encrypted traffic could be decrypted by attackers (POODLE, BEAST, etc.).",
    remediation: "Disable SSLv2, SSLv3, TLS 1.0, TLS 1.1. Enable only TLS 1.2 and TLS 1.3.",
    references: [
      "https://wiki.mozilla.org/Security/Server_Side_TLS"
    ],
    cwe: "CWE-326: Inadequate Encryption Strength"
  },
  "Weak Cipher Suites": {
    description: "Server supports weak or deprecated cipher suites.",
    impact: "Encrypted communications could be compromised.",
    remediation: "Disable RC4, DES, 3DES, NULL, EXPORT ciphers. Use AEAD ciphers (AES-GCM, ChaCha20).",
    references: [
      "https://cipherlist.eu/"
    ],
    cwe: "CWE-327: Use of a Broken or Risky Cryptographic Algorithm"
  },
  "Server Version Disclosed": {
    description: "Server response headers reveal software version information.",
    impact: "Attackers can identify specific vulnerabilities for the disclosed versions.",
    remediation: "Configure server to hide version information in headers.",
    references: [
      "https://owasp.org/www-project-web-security-testing-guide/"
    ],
    cwe: "CWE-200: Information Exposure"
  },
  "Directory Listing Enabled": {
    description: "Web server displays directory contents when no index file exists.",
    impact: "Sensitive files and application structure could be exposed.",
    remediation: "Disable directory listing in web server configuration.",
    references: [
      "https://owasp.org/www-project-web-security-testing-guide/"
    ],
    cwe: "CWE-548: Exposure of Information Through Directory Listing"
  },
  "Potential SSRF": {
    description: "Server-Side Request Forgery allows attackers to make requests from the server.",
    impact: "Access to internal systems, cloud metadata, and internal network scanning.",
    remediation: "Validate and sanitize all URLs. Use allowlists for permitted destinations.",
    references: [
      "https://owasp.org/www-community/attacks/Server_Side_Request_Forgery"
    ],
    cwe: "CWE-918: Server-Side Request Forgery (SSRF)"
  }
};

// OWASP category descriptions
const OWASP_CATEGORIES = {
  A01: { name: "Broken Access Control", description: "Failures in access control enforcement" },
  A02: { name: "Cryptographic Failures", description: "Failures related to cryptography leading to data exposure" },
  A03: { name: "Injection", description: "User-supplied data not validated, filtered, or sanitized" },
  A04: { name: "Insecure Design", description: "Missing or ineffective security controls" },
  A05: { name: "Security Misconfiguration", description: "Missing security hardening or improper configurations" },
  A06: { name: "Vulnerable Components", description: "Using components with known vulnerabilities" },
  A07: { name: "Authentication Failures", description: "Confirmation of user identity and session management flaws" },
  A08: { name: "Data Integrity Failures", description: "Code and infrastructure not protected against integrity violations" },
  A09: { name: "Logging Failures", description: "Insufficient logging, detection, and response" },
  A10: { name: "SSRF", description: "Server-side request forgery vulnerabilities" }
};

/**
 * Get remediation info for a finding
 */
function getRemediation(title) {
  for (const [key, value] of Object.entries(REMEDIATION_DB)) {
    if (title.includes(key) || key.includes(title)) {
      return value;
    }
  }
  return {
    description: "Detailed analysis required.",
    impact: "Impact assessment pending manual review.",
    remediation: "Consult security team for specific remediation steps.",
    references: ["https://owasp.org/"],
    cwe: "N/A"
  };
}

/**
 * Calculate risk score
 */
function calculateRiskScore(findings) {
  let score = 100;
  for (const f of findings) {
    switch (f.severity) {
      case "CRITICAL": score -= 25; break;
      case "HIGH": score -= 15; break;
      case "MEDIUM": score -= 8; break;
      case "LOW": score -= 3; break;
    }
  }
  return Math.max(0, score);
}

/**
 * Get risk rating from score
 */
function getRiskRating(score) {
  if (score >= 90) return { rating: "Low Risk", color: "#4caf50" };
  if (score >= 70) return { rating: "Medium Risk", color: "#fbc02d" };
  if (score >= 50) return { rating: "High Risk", color: "#f57c00" };
  return { rating: "Critical Risk", color: "#d32f2f" };
}

/**
 * Generate executive summary
 */
function generateExecutiveSummary(data) {
  const riskScore = calculateRiskScore(data.findings);
  const riskRating = getRiskRating(riskScore);

  const criticalHigh = data.findings.filter(f =>
    f.severity === "CRITICAL" || f.severity === "HIGH"
  ).length;

  return {
    target: data.target,
    scanDate: new Date(data.timestamp).toLocaleDateString("en-US", {
      weekday: "long", year: "numeric", month: "long", day: "numeric"
    }),
    scanType: data.scanType,
    totalFindings: data.findings.length,
    criticalHighCount: criticalHigh,
    riskScore,
    riskRating: riskRating.rating,
    riskColor: riskRating.color,
    summary: data.summary
  };
}

/**
 * Generate HTML report
 */
function generateHTMLReport(data) {
  const exec = generateExecutiveSummary(data);
  const findings = data.findings.map(f => ({
    ...f,
    ...getRemediation(f.title),
    cvss: CVSS_SCORES[f.severity],
    owasp: OWASP_CATEGORIES[f.category] || { name: f.category, description: "" }
  }));

  return `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Penetration Test Report - ${exec.target}</title>
  <style>
    :root {
      --critical: #d32f2f;
      --high: #f57c00;
      --medium: #fbc02d;
      --low: #1976d2;
      --info: #757575;
    }
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; background: #f5f5f5; }
    .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
    .header { background: linear-gradient(135deg, #1a237e 0%, #283593 100%); color: white; padding: 40px; margin-bottom: 30px; border-radius: 8px; }
    .header h1 { font-size: 2.5em; margin-bottom: 10px; }
    .header .meta { opacity: 0.9; font-size: 1.1em; }
    .card { background: white; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-bottom: 20px; overflow: hidden; }
    .card-header { padding: 20px; border-bottom: 1px solid #eee; font-weight: 600; font-size: 1.2em; }
    .card-body { padding: 20px; }
    .executive-summary { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; }
    .stat-box { text-align: center; padding: 20px; border-radius: 8px; background: #f8f9fa; }
    .stat-box .value { font-size: 2.5em; font-weight: bold; }
    .stat-box .label { color: #666; margin-top: 5px; }
    .risk-meter { height: 20px; background: #eee; border-radius: 10px; overflow: hidden; margin: 20px 0; }
    .risk-meter .fill { height: 100%; transition: width 0.5s; }
    .severity-badge { display: inline-block; padding: 4px 12px; border-radius: 20px; color: white; font-weight: 500; font-size: 0.85em; }
    .severity-CRITICAL { background: var(--critical); }
    .severity-HIGH { background: var(--high); }
    .severity-MEDIUM { background: var(--medium); color: #333; }
    .severity-LOW { background: var(--low); }
    .severity-INFO { background: var(--info); }
    .finding { border: 1px solid #eee; border-radius: 8px; margin-bottom: 20px; overflow: hidden; }
    .finding-header { padding: 15px 20px; background: #f8f9fa; display: flex; justify-content: space-between; align-items: center; }
    .finding-title { font-weight: 600; font-size: 1.1em; }
    .finding-body { padding: 20px; }
    .finding-section { margin-bottom: 15px; }
    .finding-section h4 { color: #1a237e; margin-bottom: 8px; font-size: 0.95em; text-transform: uppercase; }
    .finding-section p { color: #555; }
    .code-block { background: #263238; color: #aed581; padding: 15px; border-radius: 4px; font-family: 'Consolas', monospace; overflow-x: auto; }
    .reference-list { list-style: none; }
    .reference-list li { margin-bottom: 5px; }
    .reference-list a { color: #1976d2; text-decoration: none; }
    .reference-list a:hover { text-decoration: underline; }
    .owasp-tag { display: inline-block; background: #e3f2fd; color: #1565c0; padding: 2px 8px; border-radius: 4px; font-size: 0.85em; margin-left: 10px; }
    .summary-table { width: 100%; border-collapse: collapse; }
    .summary-table th, .summary-table td { padding: 12px; text-align: left; border-bottom: 1px solid #eee; }
    .summary-table th { background: #f8f9fa; font-weight: 600; }
    .chart-container { display: flex; justify-content: center; gap: 40px; padding: 20px; }
    .donut-chart { width: 200px; height: 200px; position: relative; }
    .disclaimer { background: #fff3e0; border-left: 4px solid #ff9800; padding: 15px 20px; margin-top: 30px; border-radius: 0 8px 8px 0; }
    .footer { text-align: center; padding: 30px; color: #666; font-size: 0.9em; }
    @media print {
      .container { max-width: 100%; }
      .card { break-inside: avoid; }
      .finding { break-inside: avoid; }
    }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <h1>🔒 Penetration Test Report</h1>
      <div class="meta">
        <strong>Target:</strong> ${exec.target}<br>
        <strong>Assessment Date:</strong> ${exec.scanDate}<br>
        <strong>Scan Type:</strong> ${exec.scanType.toUpperCase()}
      </div>
    </div>

    <div class="card">
      <div class="card-header">📊 Executive Summary</div>
      <div class="card-body">
        <div class="executive-summary">
          <div class="stat-box">
            <div class="value" style="color: ${exec.riskColor}">${exec.riskScore}</div>
            <div class="label">Security Score</div>
          </div>
          <div class="stat-box">
            <div class="value">${exec.totalFindings}</div>
            <div class="label">Total Findings</div>
          </div>
          <div class="stat-box">
            <div class="value" style="color: var(--critical)">${exec.summary.CRITICAL || 0}</div>
            <div class="label">Critical</div>
          </div>
          <div class="stat-box">
            <div class="value" style="color: var(--high)">${exec.summary.HIGH || 0}</div>
            <div class="label">High</div>
          </div>
          <div class="stat-box">
            <div class="value" style="color: var(--medium)">${exec.summary.MEDIUM || 0}</div>
            <div class="label">Medium</div>
          </div>
          <div class="stat-box">
            <div class="value" style="color: var(--low)">${exec.summary.LOW || 0}</div>
            <div class="label">Low</div>
          </div>
        </div>
        <div class="risk-meter">
          <div class="fill" style="width: ${exec.riskScore}%; background: ${exec.riskColor}"></div>
        </div>
        <p style="text-align: center; font-size: 1.2em;">
          Overall Risk Assessment: <strong style="color: ${exec.riskColor}">${exec.riskRating}</strong>
        </p>
      </div>
    </div>

    <div class="card">
      <div class="card-header">📋 Findings Summary</div>
      <div class="card-body">
        <table class="summary-table">
          <thead>
            <tr>
              <th>Severity</th>
              <th>Finding</th>
              <th>OWASP Category</th>
              <th>CWE</th>
            </tr>
          </thead>
          <tbody>
            ${findings.map(f => `
              <tr>
                <td><span class="severity-badge severity-${f.severity}">${f.severity}</span></td>
                <td>${f.title}</td>
                <td><span class="owasp-tag">${f.category}</span> ${f.owasp.name}</td>
                <td>${f.cwe}</td>
              </tr>
            `).join("")}
          </tbody>
        </table>
      </div>
    </div>

    <div class="card">
      <div class="card-header">🔍 Detailed Findings</div>
      <div class="card-body">
        ${findings.map((f, i) => `
          <div class="finding">
            <div class="finding-header">
              <div>
                <span class="finding-title">${i + 1}. ${f.title}</span>
                <span class="owasp-tag">${f.category}: ${f.owasp.name}</span>
              </div>
              <span class="severity-badge severity-${f.severity}">${f.severity}</span>
            </div>
            <div class="finding-body">
              <div class="finding-section">
                <h4>Description</h4>
                <p>${f.description}</p>
              </div>
              <div class="finding-section">
                <h4>Business Impact</h4>
                <p>${f.impact}</p>
              </div>
              <div class="finding-section">
                <h4>Remediation</h4>
                <div class="code-block">${f.remediation}</div>
              </div>
              ${f.evidence ? `
              <div class="finding-section">
                <h4>Evidence</h4>
                <div class="code-block">${f.evidence}</div>
              </div>
              ` : ""}
              <div class="finding-section">
                <h4>References</h4>
                <ul class="reference-list">
                  ${f.references.map(r => `<li><a href="${r}" target="_blank">${r}</a></li>`).join("")}
                </ul>
              </div>
              <div class="finding-section">
                <h4>Classification</h4>
                <p><strong>CWE:</strong> ${f.cwe}</p>
              </div>
            </div>
          </div>
        `).join("")}
      </div>
    </div>

    <div class="disclaimer">
      <strong>⚠️ Disclaimer:</strong> This report is provided for authorized security assessment purposes only.
      The findings represent the state of the target at the time of testing and may not reflect current conditions.
      Remediation should be verified through follow-up testing.
    </div>

    <div class="footer">
      <p>Generated by Kali Pentester Skill v1.0</p>
      <p>Report Generated: ${new Date().toISOString()}</p>
    </div>
  </div>
</body>
</html>`;
}

/**
 * Generate Markdown report
 */
function generateMarkdownReport(data) {
  const exec = generateExecutiveSummary(data);
  const findings = data.findings.map(f => ({
    ...f,
    ...getRemediation(f.title),
    owasp: OWASP_CATEGORIES[f.category] || { name: f.category, description: "" }
  }));

  return `# 🔒 Penetration Test Report

**Target:** ${exec.target}
**Assessment Date:** ${exec.scanDate}
**Scan Type:** ${exec.scanType.toUpperCase()}

---

## 📊 Executive Summary

| Metric | Value |
|--------|-------|
| Security Score | **${exec.riskScore}/100** |
| Risk Rating | **${exec.riskRating}** |
| Total Findings | ${exec.totalFindings} |
| Critical | ${exec.summary.CRITICAL || 0} |
| High | ${exec.summary.HIGH || 0} |
| Medium | ${exec.summary.MEDIUM || 0} |
| Low | ${exec.summary.LOW || 0} |
| Info | ${exec.summary.INFO || 0} |

---

## 📋 Findings Summary

| # | Severity | Finding | OWASP | CWE |
|---|----------|---------|-------|-----|
${findings.map((f, i) => `| ${i + 1} | ${f.severity} | ${f.title} | ${f.category}: ${f.owasp.name} | ${f.cwe} |`).join("\n")}

---

## 🔍 Detailed Findings

${findings.map((f, i) => `
### ${i + 1}. ${f.title}

**Severity:** ${f.severity}
**OWASP Category:** ${f.category} - ${f.owasp.name}
**CWE:** ${f.cwe}

#### Description
${f.description}

#### Business Impact
${f.impact}

#### Remediation
\`\`\`
${f.remediation}
\`\`\`

${f.evidence ? `#### Evidence
\`\`\`
${f.evidence}
\`\`\`
` : ""}

#### References
${f.references.map(r => `- ${r}`).join("\n")}

---
`).join("\n")}

## ⚠️ Disclaimer

This report is provided for authorized security assessment purposes only. The findings represent the state of the target at the time of testing and may not reflect current conditions. Remediation should be verified through follow-up testing.

---

*Generated by Kali Pentester Skill v1.0*
*Report Generated: ${new Date().toISOString()}*
`;
}

/**
 * List available scan results
 */
async function listResults() {
  if (!existsSync(RESULTS_DIR)) {
    console.log("📂 No scan results found");
    return [];
  }

  const files = await readdir(RESULTS_DIR);
  const jsonFiles = files.filter(f => f.endsWith(".json")).sort().reverse();

  if (jsonFiles.length === 0) {
    console.log("📂 No scan results found");
    return [];
  }

  console.log("📂 Available Scan Results:\n");
  console.log("─".repeat(80));

  for (const file of jsonFiles.slice(0, 10)) {
    const data = JSON.parse(await readFile(join(RESULTS_DIR, file), "utf-8"));
    const date = new Date(data.timestamp).toLocaleString();
    console.log(`  📄 ${file}`);
    console.log(`     Target: ${data.target}`);
    console.log(`     Date: ${date}`);
    console.log(`     Findings: ${data.findings.length}`);
    console.log("─".repeat(80));
  }

  return jsonFiles;
}

/**
 * Generate report from scan result
 */
async function generateReport(inputFile, format = "html", outputFile = null) {
  let filePath = inputFile;

  // Handle relative paths
  if (!existsSync(filePath)) {
    filePath = join(RESULTS_DIR, inputFile);
  }

  if (!existsSync(filePath)) {
    console.error(`❌ File not found: ${inputFile}`);
    process.exit(1);
  }

  const data = JSON.parse(await readFile(filePath, "utf-8"));

  let report;
  let ext;

  if (format === "html") {
    report = generateHTMLReport(data);
    ext = "html";
  } else {
    report = generateMarkdownReport(data);
    ext = "md";
  }

  const output = outputFile || join(
    RESULTS_DIR,
    `report_${basename(inputFile, ".json")}.${ext}`
  );

  await writeFile(output, report);

  console.log(`\n✅ Professional report generated!`);
  console.log(`📄 Format: ${format.toUpperCase()}`);
  console.log(`📁 Output: ${output}`);
  console.log(`\n📊 Summary:`);
  console.log(`   Target: ${data.target}`);
  console.log(`   Findings: ${data.findings.length}`);
  console.log(`   Security Score: ${calculateRiskScore(data.findings)}/100`);

  return output;
}

/**
 * Generate report for latest scan
 */
async function generateLatestReport(format = "html") {
  if (!existsSync(RESULTS_DIR)) {
    console.error("❌ No scan results found");
    process.exit(1);
  }

  const files = await readdir(RESULTS_DIR);
  const jsonFiles = files.filter(f => f.startsWith("scan_") && f.endsWith(".json")).sort().reverse();

  if (jsonFiles.length === 0) {
    console.error("❌ No scan results found");
    process.exit(1);
  }

  return generateReport(jsonFiles[0], format);
}

// CLI
async function main() {
  const args = process.argv.slice(2);
  const command = args[0];

  switch (command) {
    case "generate":
      const inputFile = args[1];
      if (!inputFile) {
        console.error("❌ Input file required");
        console.log("Usage: node report_generator.js generate <scan-result.json> [--format html|md]");
        process.exit(1);
      }
      const formatIdx = args.indexOf("--format");
      const format = formatIdx > -1 ? args[formatIdx + 1] : "html";
      const outputIdx = args.indexOf("--output");
      const output = outputIdx > -1 ? args[outputIdx + 1] : null;
      await generateReport(inputFile, format, output);
      break;

    case "latest":
      const latestFormatIdx = args.indexOf("--format");
      const latestFormat = latestFormatIdx > -1 ? args[latestFormatIdx + 1] : "html";
      await generateLatestReport(latestFormat);
      break;

    case "list":
      await listResults();
      break;

    default:
      console.log(`
Professional Penetration Test Report Generator

Commands:
  generate <file> [--format html|md] [--output <file>]
                          Generate report from scan result JSON
  latest [--format html|md]
                          Generate report for most recent scan
  list                    List available scan results

Examples:
  node report_generator.js list
  node report_generator.js generate scan_123456.json --format html
  node report_generator.js latest --format md
      `);
  }
}

export { generateHTMLReport, generateMarkdownReport, generateReport, listResults };

main().catch(console.error);
