#!/usr/bin/env node

/**
 * Automated Vulnerability Scanner
 *
 * OWASP Top 10:2025 compliant scanning with severity-based reporting.
 *
 * Usage:
 *   node vulnerability_scanner.js full <url> [--output <file>]
 *   node vulnerability_scanner.js owasp <url> --category <A01-A10>
 *   node vulnerability_scanner.js quick <url>
 */

import { exec } from "node:child_process";
import { promisify } from "node:util";
import { writeFile, mkdir } from "node:fs/promises";
import { existsSync } from "node:fs";
import { homedir } from "node:os";
import { join } from "node:path";

const execPromise = promisify(exec);

const CONFIG_DIR = join(homedir(), ".kali-pentester");
const RESULTS_DIR = join(CONFIG_DIR, "results");

// OWASP Top 10:2025 Categories
const OWASP_CATEGORIES = {
  A01: {
    name: "Broken Access Control",
    tests: ["idor", "path-traversal", "privilege-escalation", "force-browse"]
  },
  A02: {
    name: "Cryptographic Failures",
    tests: ["ssl-scan", "cipher-check", "certificate-validation"]
  },
  A03: {
    name: "Supply Chain Failures",
    tests: ["dependency-check", "sca-scan"]
  },
  A04: {
    name: "Injection",
    tests: ["sql-injection", "xss", "command-injection", "template-injection"]
  },
  A05: {
    name: "Security Misconfiguration",
    tests: ["headers-check", "default-creds", "directory-listing", "error-disclosure"]
  },
  A06: {
    name: "Insecure Design",
    tests: ["rate-limit", "business-logic"]
  },
  A07: {
    name: "Identification Failures",
    tests: ["brute-force", "session-analysis", "password-policy"]
  },
  A08: {
    name: "Data Integrity Failures",
    tests: ["deserialization", "sri-check"]
  },
  A09: {
    name: "Logging Failures",
    tests: ["log-injection", "audit-check"]
  },
  A10: {
    name: "SSRF",
    tests: ["ssrf-internal", "ssrf-cloud-metadata", "ssrf-protocol"]
  }
};

// Findings storage
const findings = [];

/**
 * Execute command in Kali (via docker or ssh based on config)
 */
async function runKaliCommand(command, timeout = 300000) {
  // Read config to determine connection type
  let configPath = join(CONFIG_DIR, "config.json");
  let config = { connectionType: "docker", dockerContainer: "kali" };

  if (existsSync(configPath)) {
    try {
      const data = await import("fs/promises").then(fs => fs.readFile(configPath, "utf-8"));
      config = JSON.parse(data);
    } catch (e) {
      // Use defaults
    }
  }

  let fullCommand;
  if (config.connectionType === "docker") {
    const escaped = command.replace(/"/g, '\\"');
    fullCommand = `docker exec ${config.dockerContainer} bash -c "${escaped}"`;
  } else {
    const escaped = command.replace(/"/g, '\\"');
    fullCommand = `ssh -p ${config.sshPort} ${config.sshUser}@${config.sshHost} "${escaped}"`;
  }

  try {
    const { stdout, stderr } = await execPromise(fullCommand, { timeout });
    return { success: true, output: stdout.trim(), errors: stderr.trim() };
  } catch (error) {
    return { success: false, output: "", errors: error.message };
  }
}

/**
 * Add finding to results
 */
function addFinding(category, severity, title, description, evidence = "") {
  findings.push({
    category,
    severity,
    title,
    description,
    evidence: evidence.substring(0, 500),
    timestamp: new Date().toISOString()
  });
}

/**
 * Run SSL/TLS scan
 */
async function runSSLScan(url) {
  console.log("  🔐 Running SSL/TLS scan...");
  const host = new URL(url).hostname;

  const result = await runKaliCommand(`sslscan --no-colour ${host} 2>/dev/null | head -100`);

  if (result.success && result.output) {
    // Check for weak configurations
    if (/SSLv[23]|TLSv1\.0/.test(result.output)) {
      addFinding("A02", "HIGH", "Weak SSL/TLS Protocol", "Server supports outdated protocols", result.output);
    }
    if (/RC4|DES|NULL|EXPORT/.test(result.output)) {
      addFinding("A02", "HIGH", "Weak Cipher Suites", "Server supports weak ciphers", result.output);
    }
    if (/Heartbleed/.test(result.output)) {
      addFinding("A02", "CRITICAL", "Heartbleed Vulnerability", "Server vulnerable to Heartbleed", result.output);
    }
  }

  return result;
}

/**
 * Run Nikto scan
 */
async function runNiktoScan(url) {
  console.log("  🔍 Running Nikto scan...");

  const result = await runKaliCommand(`nikto -h ${url} -maxtime 120s 2>/dev/null | head -200`, 180000);

  if (result.success && result.output) {
    // Parse Nikto findings
    const lines = result.output.split("\n");
    for (const line of lines) {
      if (line.includes("OSVDB-") || line.includes("CVE-")) {
        addFinding("A05", "MEDIUM", "Nikto Finding", line.trim(), line);
      }
      if (/directory listing/i.test(line)) {
        addFinding("A05", "MEDIUM", "Directory Listing Enabled", line.trim(), line);
      }
      if (/default|admin|backup/i.test(line)) {
        addFinding("A05", "LOW", "Potentially Sensitive Path", line.trim(), line);
      }
    }
  }

  return result;
}

/**
 * Run SQLmap scan
 */
async function runSQLmapScan(url) {
  console.log("  💉 Running SQL injection scan...");

  // Only test if URL has parameters
  if (!url.includes("?")) {
    console.log("    ⏭️ No parameters to test");
    return { success: true, output: "No parameters" };
  }

  const result = await runKaliCommand(
    `sqlmap -u "${url}" --batch --risk=2 --level=3 --timeout=10 --retries=1 2>/dev/null | tail -50`,
    120000
  );

  if (result.success && result.output) {
    if (/injectable|vulnerable/i.test(result.output)) {
      addFinding("A04", "CRITICAL", "SQL Injection Vulnerability", "Parameter is vulnerable to SQL injection", result.output);
    }
  }

  return result;
}

/**
 * Check security headers
 */
async function checkSecurityHeaders(url) {
  console.log("  📋 Checking security headers...");

  const result = await runKaliCommand(`curl -sI "${url}" | head -30`);

  if (result.success && result.output) {
    const headers = result.output.toLowerCase();

    if (!headers.includes("x-frame-options")) {
      addFinding("A05", "MEDIUM", "Missing X-Frame-Options", "Clickjacking protection header missing", "");
    }
    if (!headers.includes("x-content-type-options")) {
      addFinding("A05", "LOW", "Missing X-Content-Type-Options", "MIME type sniffing protection missing", "");
    }
    if (!headers.includes("content-security-policy")) {
      addFinding("A05", "MEDIUM", "Missing Content-Security-Policy", "CSP header not configured", "");
    }
    if (!headers.includes("strict-transport-security")) {
      addFinding("A02", "MEDIUM", "Missing HSTS", "HSTS header not configured", "");
    }
    if (headers.includes("server:")) {
      const serverMatch = result.output.match(/Server:\s*(.+)/i);
      if (serverMatch) {
        addFinding("A05", "INFO", "Server Version Disclosed", `Server header reveals: ${serverMatch[1]}`, serverMatch[1]);
      }
    }
  }

  return result;
}

/**
 * Run directory bruteforce
 */
async function runDirBrute(url) {
  console.log("  📁 Running directory discovery...");

  const result = await runKaliCommand(
    `gobuster dir -u "${url}" -w /usr/share/wordlists/dirb/common.txt -q -t 20 --timeout 5s 2>/dev/null | head -30`,
    120000
  );

  if (result.success && result.output) {
    const lines = result.output.split("\n").filter(l => l.includes("Status:"));
    for (const line of lines) {
      if (/admin|backup|config|\.git|\.env|debug/i.test(line)) {
        addFinding("A01", "MEDIUM", "Sensitive Directory Found", line.trim(), line);
      }
    }
  }

  return result;
}

/**
 * Check for SSRF vulnerabilities
 */
async function checkSSRF(url) {
  console.log("  🔗 Checking for SSRF...");

  // Test common SSRF payloads if URL has parameters
  if (!url.includes("?")) {
    return { success: true, output: "No parameters to test" };
  }

  const ssrfPayloads = [
    "http://127.0.0.1",
    "http://localhost",
    "http://169.254.169.254/latest/meta-data/"
  ];

  for (const payload of ssrfPayloads) {
    const testUrl = url.replace(/=([^&]+)/, `=${encodeURIComponent(payload)}`);
    const result = await runKaliCommand(`curl -s -o /dev/null -w "%{http_code}" "${testUrl}" --max-time 5`);

    if (result.success && result.output === "200") {
      addFinding("A10", "HIGH", "Potential SSRF", `URL accepts internal address: ${payload}`, payload);
    }
  }

  return { success: true, output: "SSRF check completed" };
}

/**
 * Run quick scan
 */
async function quickScan(url) {
  console.log(`\n⚡ Quick Scan: ${url}`);
  console.log("═".repeat(60));

  await checkSecurityHeaders(url);
  await runSSLScan(url);

  return generateReport(url, "quick");
}

/**
 * Run full OWASP scan
 */
async function fullScan(url) {
  console.log(`\n🔍 Full OWASP Scan: ${url}`);
  console.log("═".repeat(60));

  console.log("\n📍 Phase 1: Reconnaissance");
  await checkSecurityHeaders(url);
  await runDirBrute(url);

  console.log("\n🔐 Phase 2: Cryptography (A02)");
  await runSSLScan(url);

  console.log("\n🔍 Phase 3: Configuration (A05)");
  await runNiktoScan(url);

  console.log("\n💉 Phase 4: Injection (A04)");
  await runSQLmapScan(url);

  console.log("\n🔗 Phase 5: SSRF (A10)");
  await checkSSRF(url);

  return generateReport(url, "full");
}

/**
 * Scan specific OWASP category
 */
async function owaspScan(url, category) {
  const cat = OWASP_CATEGORIES[category];
  if (!cat) {
    console.error(`❌ Unknown category: ${category}`);
    console.log("Valid categories: A01-A10");
    process.exit(1);
  }

  console.log(`\n🎯 OWASP ${category}: ${cat.name}`);
  console.log("═".repeat(60));

  for (const test of cat.tests) {
    console.log(`  Running: ${test}`);
    // Map tests to functions
    switch (test) {
      case "ssl-scan":
      case "cipher-check":
        await runSSLScan(url);
        break;
      case "headers-check":
        await checkSecurityHeaders(url);
        break;
      case "sql-injection":
        await runSQLmapScan(url);
        break;
      case "force-browse":
        await runDirBrute(url);
        break;
      case "ssrf-internal":
      case "ssrf-cloud-metadata":
        await checkSSRF(url);
        break;
      default:
        console.log(`    ⚠️ Test not automated: ${test} (manual testing required)`);
    }
  }

  return generateReport(url, `owasp-${category}`);
}

/**
 * Generate report
 */
async function generateReport(url, scanType) {
  console.log("\n📊 Generating Report");
  console.log("═".repeat(60));

  // Count by severity
  const counts = { CRITICAL: 0, HIGH: 0, MEDIUM: 0, LOW: 0, INFO: 0 };
  for (const f of findings) {
    counts[f.severity] = (counts[f.severity] || 0) + 1;
  }

  console.log("\n📋 Summary:");
  console.log(`   🔴 Critical: ${counts.CRITICAL}`);
  console.log(`   🟠 High: ${counts.HIGH}`);
  console.log(`   🟡 Medium: ${counts.MEDIUM}`);
  console.log(`   🔵 Low: ${counts.LOW}`);
  console.log(`   ⚪ Info: ${counts.INFO}`);
  console.log(`   Total: ${findings.length}`);

  if (findings.length > 0) {
    console.log("\n📝 Findings:");
    for (const f of findings) {
      const icon = f.severity === "CRITICAL" ? "🔴" :
                   f.severity === "HIGH" ? "🟠" :
                   f.severity === "MEDIUM" ? "🟡" :
                   f.severity === "LOW" ? "🔵" : "⚪";
      console.log(`   ${icon} [${f.category}] ${f.title}`);
    }
  }

  // Save report
  if (!existsSync(RESULTS_DIR)) {
    await mkdir(RESULTS_DIR, { recursive: true });
  }

  const report = {
    target: url,
    scanType,
    timestamp: new Date().toISOString(),
    summary: counts,
    findings
  };

  const reportPath = join(RESULTS_DIR, `scan_${Date.now()}.json`);
  await writeFile(reportPath, JSON.stringify(report, null, 2));
  console.log(`\n📄 Report saved: ${reportPath}`);

  return report;
}

// CLI
async function main() {
  const args = process.argv.slice(2);
  const command = args[0];
  const url = args[1];

  if (!command || !url) {
    console.log(`
Automated Vulnerability Scanner

Usage:
  quick <url>                    Quick security headers + SSL check
  full <url>                     Full OWASP Top 10 scan
  owasp <url> --category <A01-A10>  Scan specific OWASP category

Examples:
  node vulnerability_scanner.js quick https://example.com
  node vulnerability_scanner.js full https://example.com
  node vulnerability_scanner.js owasp https://example.com --category A04
    `);
    process.exit(1);
  }

  switch (command) {
    case "quick":
      await quickScan(url);
      break;

    case "full":
      await fullScan(url);
      break;

    case "owasp":
      const catIdx = args.indexOf("--category");
      if (catIdx === -1 || !args[catIdx + 1]) {
        console.error("❌ Category required: --category A01-A10");
        process.exit(1);
      }
      await owaspScan(url, args[catIdx + 1].toUpperCase());
      break;

    default:
      console.error(`❌ Unknown command: ${command}`);
      process.exit(1);
  }
}

export { quickScan, fullScan, owaspScan };

main().catch(console.error);
