#!/usr/bin/env python3
"""
Kali Pentester v2.0 Orchestrator

Multi-agent penetration testing with ultrathink support.

Usage:
    python orchestrator.py scan --target URL --level 5
    python orchestrator.py workflow --target URL
    python orchestrator.py agents --list
"""

import argparse
import json
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from typing import List, Dict


@dataclass
class AgentConfig:
    """Agent configuration."""
    name: str
    role: str
    focus: str
    subagent_type: str = "general-purpose"
    model: str = "sonnet"
    ultrathink: bool = False


@dataclass
class ScanConfig:
    """Scan configuration."""
    target: str
    level: int = 5
    domains: List[str] = field(default_factory=lambda: ["web", "api", "llm", "supply_chain"])
    ultrathink: bool = True
    output_formats: List[str] = field(default_factory=lambda: ["html", "md", "json"])

    @property
    def agent_count(self) -> int:
        return {1: 2, 2: 3, 3: 4, 4: 5, 5: 6}.get(self.level, 5)

    @property
    def intel_agents(self) -> int:
        return min(3, self.level)

    @property
    def scan_agents(self) -> int:
        return min(5, self.level + 1)


# Agent definitions
AGENTS = {
    "planning": AgentConfig(
        name="planning",
        role="Planning Agent",
        focus="Scope analysis and threat modeling",
        subagent_type="Plan",
        model="opus",
        ultrathink=True
    ),
    "intel_cve": AgentConfig(
        name="intel_cve",
        role="CVE Intel Agent",
        focus="CVE and vulnerability intelligence",
        subagent_type="Explore",
        model="haiku"
    ),
    "intel_tech": AgentConfig(
        name="intel_tech",
        role="Tech Intel Agent",
        focus="Technology-specific threats",
        subagent_type="Explore",
        model="haiku"
    ),
    "intel_compliance": AgentConfig(
        name="intel_compliance",
        role="Compliance Intel Agent",
        focus="OWASP 2025, compliance updates",
        subagent_type="Explore",
        model="haiku"
    ),
    "scan_web": AgentConfig(
        name="scan_web",
        role="Web/API Scanner",
        focus="OWASP Top 10 2025 web vulnerabilities",
        model="sonnet"
    ),
    "scan_llm": AgentConfig(
        name="scan_llm",
        role="LLM Security Tester",
        focus="Prompt injection, jailbreak, data leakage",
        model="opus",
        ultrathink=True
    ),
    "scan_supply": AgentConfig(
        name="scan_supply",
        role="Supply Chain Auditor",
        focus="Dependencies, SBOM, container security",
        model="sonnet"
    ),
    "scan_injection": AgentConfig(
        name="scan_injection",
        role="Injection Tester",
        focus="SQLi, XSS, command injection",
        model="sonnet"
    ),
    "scan_config": AgentConfig(
        name="scan_config",
        role="Config Auditor",
        focus="Misconfigurations, headers, SSL/TLS",
        model="sonnet"
    ),
    "exploit_analyst": AgentConfig(
        name="exploit_analyst",
        role="Exploit Analyst",
        focus="Deep vulnerability analysis, CVSS scoring",
        model="opus",
        ultrathink=True
    ),
    "synthesis": AgentConfig(
        name="synthesis",
        role="Synthesis Agent",
        focus="Report generation (HTML/MD/JSON)",
        model="sonnet"
    )
}


def generate_workflow(config: ScanConfig) -> str:
    """Generate Claude Code workflow instructions."""
    return f'''
## Kali Pentester v2.0 - Level {config.level} Workflow

**Target:** {config.target}
**Agents:** {config.agent_count}
**Ultrathink:** {"Enabled" if config.ultrathink else "Disabled"}
**Output:** {", ".join(config.output_formats)}

### Phase 0: Self-Upgrade

```
Task(
  subagent_type='claude-code-guide',
  prompt='Get latest on: OWASP 2025, LLM security, supply chain threats',
  description='Upgrade security knowledge'
)

WebSearch(query='CVE 2025 critical web vulnerabilities')
WebSearch(query='OWASP Top 10 2025 changes')
WebSearch(query='LLM prompt injection 2025')
```

### Phase 1: Environment Setup

```bash
# Verify Kali connection
python3 scripts/kali_connector.py test
```

### Phase 2: Threat Intelligence (PARALLEL)

Spawn {config.intel_agents} Intel Agents in ONE message:

```python
Task(subagent_type='Explore', prompt='CVE intel for target tech...', description='Intel: CVE', model='haiku')
Task(subagent_type='Explore', prompt='Technology threat patterns...', description='Intel: Tech', model='haiku')
Task(subagent_type='Explore', prompt='OWASP 2025, compliance...', description='Intel: Compliance', model='haiku')
```

### Phase 3: Security Scanning (PARALLEL)

Spawn {config.scan_agents} Scan Agents in ONE message:

```python
Task(subagent_type='general-purpose', prompt='OWASP scan of {config.target}...', description='Scan: Web/API', model='sonnet')
Task(subagent_type='general-purpose', prompt='LLM security with ultrathink...', description='Scan: LLM', model='opus')
Task(subagent_type='general-purpose', prompt='Supply chain audit...', description='Scan: Supply Chain', model='sonnet')
Task(subagent_type='general-purpose', prompt='Injection testing...', description='Scan: Injection', model='sonnet')
Task(subagent_type='general-purpose', prompt='Config audit...', description='Scan: Config', model='sonnet')
```

### Phase 4: Deep Analysis (Ultrathink)

```python
Task(
  subagent_type='general-purpose',
  prompt='''Deep vulnerability analysis with extended thinking:
    1. Analyze exploitation paths
    2. Calculate CVSS scores
    3. Assess business impact
    4. Identify false positives
    5. Prioritize remediation

    Use thorough step-by-step reasoning.''',
  description='Exploit analysis (ultrathink)',
  model='opus'
)
```

### Phase 5: Synthesis & Reporting

```python
Task(
  subagent_type='general-purpose',
  prompt='Generate Level {config.level} security report: executive summary, findings, OWASP compliance, remediation timeline',
  description='Generate report',
  model='sonnet'
)
```

```bash
python3 scripts/report_generator.py --input ./results --format {",".join(config.output_formats)}
```

### Phase 6: Self-Upgrade

```bash
python3 scripts/self_upgrade.py --findings ./results
```
'''


def list_agents() -> str:
    """List all available agents."""
    lines = ["## Available Agents\n"]
    for name, agent in AGENTS.items():
        ut = " (Ultrathink)" if agent.ultrathink else ""
        lines.append(f"- **{agent.role}** [{agent.model}{ut}]: {agent.focus}")
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Kali Pentester v2.0 Orchestrator")
    subparsers = parser.add_subparsers(dest="command", required=True)

    # Scan command
    scan = subparsers.add_parser("scan", help="Generate scan workflow")
    scan.add_argument("--target", "-t", required=True, help="Target URL")
    scan.add_argument("--level", "-l", type=int, default=5, choices=[1, 2, 3, 4, 5])
    scan.add_argument("--output", "-o", help="Output file")

    # Workflow command
    wf = subparsers.add_parser("workflow", help="Show full workflow")
    wf.add_argument("--target", "-t", required=True, help="Target URL")
    wf.add_argument("--level", "-l", type=int, default=5)

    # Agents command
    agents = subparsers.add_parser("agents", help="List agents")
    agents.add_argument("--list", action="store_true")

    args = parser.parse_args()

    if args.command == "scan":
        config = ScanConfig(target=args.target, level=args.level)
        workflow = generate_workflow(config)
        print(workflow)
        if args.output:
            Path(args.output).write_text(workflow)

    elif args.command == "workflow":
        config = ScanConfig(target=args.target, level=args.level)
        print(generate_workflow(config))

    elif args.command == "agents":
        print(list_agents())


if __name__ == "__main__":
    main()
