#!/usr/bin/env python3
"""
Report Generator for Kali Pentester v2.0

Generates executive-level HTML, Markdown, and JSON reports.

Usage:
    python report_generator.py --input ./results --format all
    python report_generator.py --input ./results --format html --output ./reports
"""

import argparse
import json
from datetime import datetime
from pathlib import Path
from typing import Dict, List


def generate_html_report(data: Dict, output_path: Path) -> None:
    """Generate styled HTML report."""
    risk_score = data.get("risk_score", 0)
    risk_color = "#dc3545" if risk_score >= 70 else "#ffc107" if risk_score >= 40 else "#28a745"

    html = f'''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Security Assessment Report - {data.get("target", "Unknown")}</title>
    <style>
        :root {{
            --primary: #1a73e8;
            --danger: #dc3545;
            --warning: #ffc107;
            --success: #28a745;
            --dark: #212529;
            --light: #f8f9fa;
        }}
        * {{ box-sizing: border-box; margin: 0; padding: 0; }}
        body {{
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: var(--dark);
            background: var(--light);
        }}
        .container {{ max-width: 1200px; margin: 0 auto; padding: 20px; }}
        header {{
            background: linear-gradient(135deg, var(--primary), #0d47a1);
            color: white;
            padding: 40px;
            border-radius: 12px;
            margin-bottom: 30px;
        }}
        h1 {{ font-size: 2.5rem; margin-bottom: 10px; }}
        .meta {{ opacity: 0.9; }}
        .risk-score {{
            display: inline-block;
            width: 120px;
            height: 120px;
            border-radius: 50%;
            background: {risk_color};
            color: white;
            text-align: center;
            line-height: 120px;
            font-size: 2.5rem;
            font-weight: bold;
            float: right;
        }}
        .card {{
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }}
        .card h2 {{
            color: var(--primary);
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid var(--light);
        }}
        .finding {{
            border-left: 4px solid;
            padding: 15px;
            margin: 10px 0;
            background: var(--light);
            border-radius: 0 8px 8px 0;
        }}
        .finding.critical {{ border-color: var(--danger); }}
        .finding.high {{ border-color: #fd7e14; }}
        .finding.medium {{ border-color: var(--warning); }}
        .finding.low {{ border-color: var(--success); }}
        .badge {{
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            color: white;
        }}
        .badge.critical {{ background: var(--danger); }}
        .badge.high {{ background: #fd7e14; }}
        .badge.medium {{ background: var(--warning); color: var(--dark); }}
        .badge.low {{ background: var(--success); }}
        table {{
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }}
        th, td {{
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #dee2e6;
        }}
        th {{ background: var(--light); font-weight: 600; }}
        tr:hover {{ background: #f1f3f4; }}
        .progress {{
            height: 10px;
            background: #e9ecef;
            border-radius: 5px;
            overflow: hidden;
        }}
        .progress-bar {{
            height: 100%;
            background: var(--primary);
        }}
        footer {{
            text-align: center;
            padding: 20px;
            color: #6c757d;
            font-size: 0.9rem;
        }}
    </style>
</head>
<body>
    <div class="container">
        <header>
            <div class="risk-score">{risk_score}</div>
            <h1>Security Assessment Report</h1>
            <p class="meta">
                <strong>Target:</strong> {data.get("target", "N/A")}<br>
                <strong>Date:</strong> {datetime.now().strftime("%Y-%m-%d %H:%M")}<br>
                <strong>Level:</strong> {data.get("level", 5)} | <strong>Risk:</strong> {data.get("risk_level", "Unknown").upper()}
            </p>
        </header>

        <div class="card">
            <h2>Executive Summary</h2>
            <p>{data.get("executive_summary", "No summary available.")}</p>
        </div>

        <div class="card">
            <h2>Findings Overview</h2>
            <table>
                <tr>
                    <th>Severity</th>
                    <th>Count</th>
                    <th>Distribution</th>
                </tr>
                <tr>
                    <td><span class="badge critical">Critical</span></td>
                    <td>{data.get("summary", {}).get("critical", 0)}</td>
                    <td><div class="progress"><div class="progress-bar" style="width: 100%; background: var(--danger);"></div></div></td>
                </tr>
                <tr>
                    <td><span class="badge high">High</span></td>
                    <td>{data.get("summary", {}).get("high", 0)}</td>
                    <td><div class="progress"><div class="progress-bar" style="width: 75%; background: #fd7e14;"></div></div></td>
                </tr>
                <tr>
                    <td><span class="badge medium">Medium</span></td>
                    <td>{data.get("summary", {}).get("medium", 0)}</td>
                    <td><div class="progress"><div class="progress-bar" style="width: 50%; background: var(--warning);"></div></div></td>
                </tr>
                <tr>
                    <td><span class="badge low">Low</span></td>
                    <td>{data.get("summary", {}).get("low", 0)}</td>
                    <td><div class="progress"><div class="progress-bar" style="width: 25%; background: var(--success);"></div></div></td>
                </tr>
            </table>
        </div>

        <div class="card">
            <h2>Critical & High Findings</h2>
            {"".join([f'''
            <div class="finding {f.get("severity", "medium")}">
                <strong>{f.get("title", "Untitled")}</strong>
                <span class="badge {f.get("severity", "medium")}">{f.get("severity", "medium").upper()}</span>
                <p>{f.get("description", "")}</p>
                <p><strong>OWASP:</strong> {f.get("owasp", "N/A")} | <strong>CVSS:</strong> {f.get("cvss", "N/A")}</p>
                <p><strong>Remediation:</strong> {f.get("remediation", "N/A")}</p>
            </div>
            ''' for f in data.get("findings", []) if f.get("severity") in ["critical", "high"]])}
        </div>

        <div class="card">
            <h2>OWASP 2025 Compliance</h2>
            <table>
                <tr><th>Category</th><th>Status</th><th>Findings</th></tr>
                {"".join([f'<tr><td>{k}</td><td>{v.get("status", "N/A")}</td><td>{v.get("count", 0)}</td></tr>' for k, v in data.get("owasp_compliance", {}).items()])}
            </table>
        </div>

        <footer>
            Generated by Kali Pentester v2.0 - Ultrathink Expert Edition<br>
            {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}
        </footer>
    </div>
</body>
</html>'''

    output_path.write_text(html)
    print(f"HTML report saved: {output_path}")


def generate_markdown_report(data: Dict, output_path: Path) -> None:
    """Generate Markdown report."""
    md = f'''# Security Assessment Report

**Target:** {data.get("target", "N/A")}
**Date:** {datetime.now().strftime("%Y-%m-%d %H:%M")}
**Level:** {data.get("level", 5)}
**Risk Score:** {data.get("risk_score", 0)}/100
**Risk Level:** {data.get("risk_level", "Unknown").upper()}

---

## Executive Summary

{data.get("executive_summary", "No summary available.")}

---

## Findings Summary

| Severity | Count |
|----------|-------|
| Critical | {data.get("summary", {}).get("critical", 0)} |
| High | {data.get("summary", {}).get("high", 0)} |
| Medium | {data.get("summary", {}).get("medium", 0)} |
| Low | {data.get("summary", {}).get("low", 0)} |

---

## Critical & High Findings

'''

    for f in data.get("findings", []):
        if f.get("severity") in ["critical", "high"]:
            md += f'''### {f.get("title", "Untitled")} [{f.get("severity", "").upper()}]

**OWASP:** {f.get("owasp", "N/A")} | **CVSS:** {f.get("cvss", "N/A")}

{f.get("description", "")}

**Remediation:** {f.get("remediation", "N/A")}

---

'''

    md += f'''
## OWASP 2025 Compliance

| Category | Status | Findings |
|----------|--------|----------|
'''
    for k, v in data.get("owasp_compliance", {}).items():
        md += f'| {k} | {v.get("status", "N/A")} | {v.get("count", 0)} |\n'

    md += f'''
---

*Generated by Kali Pentester v2.0 - Ultrathink Expert Edition*
*{datetime.now().strftime("%Y-%m-%d %H:%M:%S")}*
'''

    output_path.write_text(md)
    print(f"Markdown report saved: {output_path}")


def main():
    parser = argparse.ArgumentParser(description="Report Generator")
    parser.add_argument("--input", "-i", required=True, help="Input JSON file or directory")
    parser.add_argument("--format", "-f", default="all", choices=["html", "md", "json", "all"])
    parser.add_argument("--output", "-o", default="./reports", help="Output directory")

    args = parser.parse_args()

    input_path = Path(args.input)
    output_dir = Path(args.output)
    output_dir.mkdir(parents=True, exist_ok=True)

    # Load data
    if input_path.is_file():
        data = json.loads(input_path.read_text())
    else:
        # Merge all JSON files in directory
        data = {"findings": [], "summary": {"critical": 0, "high": 0, "medium": 0, "low": 0}}
        for f in input_path.glob("*.json"):
            file_data = json.loads(f.read_text())
            if "findings" in file_data:
                data["findings"].extend(file_data["findings"])

    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")

    if args.format in ["html", "all"]:
        generate_html_report(data, output_dir / f"report_{timestamp}.html")

    if args.format in ["md", "all"]:
        generate_markdown_report(data, output_dir / f"report_{timestamp}.md")

    if args.format in ["json", "all"]:
        json_path = output_dir / f"report_{timestamp}.json"
        json_path.write_text(json.dumps(data, indent=2))
        print(f"JSON report saved: {json_path}")


if __name__ == "__main__":
    main()
