---
name: latex-writing
description: |
  Guide LaTeX document authoring following best practices and proper semantic 
  markup. Use proactively when: (1) writing or editing .tex files, (2) writing 
  or editing .nw literate programming files, (3) literate-programming skill is 
  active and working with .nw files, (4) user mentions LaTeX, BibTeX, or 
  document formatting, (5) reviewing LaTeX code quality. Ensures proper use of 
  semantic environments (description vs itemize), csquotes (\enquote{} not 
  ``...''), and cleveref (\cref{} not \S\ref{}).
---

# LaTeX Writing Best Practices

This skill guides the creation of well-structured, semantically correct LaTeX documents following established best practices.

## Core Principle: Semantic Markup

Use LaTeX environments that match the semantic meaning of the content, not just the visual appearance.

## List Environments: When to Use What

### Use `description` for Term-Definition Pairs

When you have labels followed by explanations, definitions, or descriptions, use the `description` environment:

```latex
\begin{description}
\item[Term] Definition or explanation of the term
\item[Label] Content associated with the label
\item[Property] Description of the property
\end{description}
```

**NEVER do this:**
```latex
\begin{itemize}
\item \textbf{Term:} Definition or explanation
\item \textbf{Label:} Content associated with label
\end{itemize}
```

### Common Use Cases for `description`

- **API parameters**: `\item[username] The user's login name`
- **Configuration options**: `\item[timeout] Maximum wait time in seconds`
- **Glossary entries**: `\item[LaTeX] A document preparation system`
- **Passes/Fails examples**: `\item[Passes] Correct implementation...`
- **Feature descriptions**: `\item[Auto-save] Automatically saves every 5 minutes`

### Use `itemize` for Simple Lists

Use `itemize` when items are uniform list elements without labels:

```latex
\begin{itemize}
\item First uniform item
\item Second uniform item
\item Third uniform item
\end{itemize}
```

### Use `enumerate` for Numbered Steps or Rankings

Use `enumerate` when order matters:

```latex
\begin{enumerate}
\item First step in the process
\item Second step in the process
\item Third step in the process
\end{enumerate}
```

## Recognition Patterns

When reviewing or writing LaTeX, look for these patterns that indicate `description` should be used:

- `\item \textbf{SomeLabel:}` → Should use `\item[SomeLabel]`
- `\item \emph{SomeLabel:}` → Should use `\item[SomeLabel]`
- `\item SomeLabel ---` → Should use `\item[SomeLabel]`
- Lists where every item starts with bold/emphasized text

## Fixing Common Anti-Patterns

### Anti-Pattern: Bold Labels in Itemize
```latex
% INCORRECT
\begin{itemize}
\item \textbf{Passes:} \verb|\documentclass{article}|
\item \textbf{Fails:} No documentclass declaration
\end{itemize}
```

### Correct: Description Environment
```latex
% CORRECT
\begin{description}
\item[Passes] \verb|\documentclass{article}|
\item[Fails] No documentclass declaration
\end{description}
```

### Anti-Pattern: Manual Formatting Instead of Semantic Structure
```latex
% INCORRECT
\noindent\textbf{Configuration:} Set timeout to 30 seconds.\\
\textbf{Performance:} Optimized for large datasets.
```

### Correct: Semantic Description
```latex
% CORRECT
\begin{description}
\item[Configuration] Set timeout to 30 seconds
\item[Performance] Optimized for large datasets
\end{description}
```

## Literate Programming (.nw files)

**CRITICAL**: When writing LaTeX in literate programming files (.nw), use noweb's `[[code]]` notation for quoting code, not `\texttt` with manual escaping.

### Use `[[code]]` Notation, Not `\texttt{...\_...}`

The noweb literate programming system provides special notation for code references that automatically handles special characters like underscores.

**Anti-pattern**: Manual underscore escaping with `\texttt`
```latex
% INCORRECT - in .nw files
The \texttt{get\_submission()} method calls \texttt{\_\_getattribute\_\_}.
We store \texttt{\_original\_get\_submission} in the closure.
The \texttt{NOREFRESH\_GRADES} constant defines final grades.
```

**Correct**: Use `[[code]]` notation
```latex
% CORRECT - in .nw files
The [[get_submission()]] method calls [[__getattribute__]].
We store [[_original_get_submission]] in the closure.
The [[NOREFRESH_GRADES]] constant defines final grades.
```

**Why this matters**:
- Automatically escapes special characters (underscores, backslashes, etc.)
- Makes source code more readable (no manual escaping)
- Follows literate programming conventions
- Prevents LaTeX errors from forgotten escapes
- Clearly distinguishes code from prose

### When to Use `[[code]]` vs `\texttt`

**Use `[[code]]`** in .nw files for:
- Function and method names: `[[get_submissions()]]`, `[[__init__]]`
- Variable names: `[[_includes]]`, `[[user_id]]`
- Class names: `[[LazySubmission]]`, `[[Assignment]]`
- Constants: `[[NOREFRESH_GRADES]]`, `[[MAX_RETRIES]]`
- Module names: `[[pickle]]`, `[[Canvas]]`
- Any identifier with underscores or special characters

**Use `\texttt`** in .nw files for:
- Short non-code technical terms without special characters
- File extensions: `\texttt{.py}`, `\texttt{.nw}`
- Simple commands without underscores

**In regular .tex files** (not literate programs):
- Use `\texttt` with proper escaping as `[[...]]` is not available
- Or use packages like `minted` or `listings` for code

### Recognition Pattern for Review

When reviewing .nw files, look for these anti-patterns:
- `\texttt{..._...}` → Should use `[[...]]`
- `\texttt{...__...}` → Should use `[[...]]`
- `\item[SOME\_CONSTANT behavior]` → Should use better label + `[[SOME_CONSTANT]]` in text

### Examples from Real Code

**Documenting methods:**
```latex
% INCORRECT
The \texttt{\_\_getstate\_\_} method excludes \texttt{\_original\_get\_submission}.

% CORRECT
The [[__getstate__]] method excludes [[_original_get_submission]].
```

**Documenting constants:**
```latex
% INCORRECT
\item[NOREFRESH\_GRADES behavior] Submissions with final grades...

% CORRECT
\item[Final grade policy] Submissions with final grades (A, P, P+, complete)
  are never refreshed, maintaining the [[NOREFRESH_GRADES]] policy.
```

**Documenting attributes:**
```latex
% INCORRECT
The decorator adds a \texttt{\_\_cache} dictionary and \texttt{\_\_all\_fetched} flag.

% CORRECT
The decorator adds a [[__cache]] dictionary and [[__all_fetched]] flag.
```

## Additional Best Practices

### Cross-References
- **Always** use `\cref{...}` (cleveref package) for all cross-references
- **Never** use `\S\ref{...}` or manually type section/figure prefixes
- Use descriptive labels: `\label{sec:introduction}` not `\label{s1}`
- Examples:
  - Sections: `\cref{sec:background}` → "Section 2.1"
  - Figures: `\cref{fig:diagram}` → "Figure 3"
  - Tables: `\cref{tab:results}` → "Table 1"
  - Multiple: `\cref{sec:intro,sec:conclusion}` → "Sections 1 and 4"

**Anti-pattern**: Manual prefixes
```latex
% INCORRECT
Section~\ref{sec:intro} shows...
\S\ref{sec:background} discusses...
Figure~\ref{fig:plot} demonstrates...

% CORRECT
\cref{sec:intro} shows...
\cref{sec:background} discusses...
\cref{fig:plot} demonstrates...
```

**Why**: The cleveref package automatically adds the correct prefix (Section, Figure, etc.) and handles pluralization, ranges, and language-specific formatting.

### Citations
- Use proper citation commands (`\cite`, `\citep`, `\citet`) not manual references
- Never write `[1]` or `(Smith 2020)` manually

### Quotations (csquotes package)
- **Always** use `\enquote{...}` for quotes, never manual quote marks
- Handles nested quotes automatically: `\enquote{outer \enquote{inner} quote}`
- Language-aware: Swedish uses »...« or "...", English uses "..." or '...'
- For block quotes, use `\begin{displayquote}...\end{displayquote}`

**Anti-pattern**: Manual quotes
```latex
% INCORRECT
"This is a quote"
``This is a quote''
'single quotes'
```

**Correct**: Use csquotes
```latex
% CORRECT
\enquote{This is a quote}
\enquote{outer \enquote{inner} quote}
```

**Why**: Manual quote marks don't adapt to language settings and can cause typographical inconsistencies. The csquotes package handles all quote styling correctly based on document language.

### Emphasis
- **Never** use ALL CAPITALS for emphasis in running text
- Use `\emph{...}` to emphasize words or phrases
- For strong emphasis, use `\textbf{...}` or nested `\emph{\emph{...}}`
- Let LaTeX handle the typographic styling

**Anti-pattern**: ALL CAPITALS for emphasis
```latex
% INCORRECT
This is VERY important to understand.
We must do this NOW before moving forward.
The BENEFITS of classes are clear.
```

**Correct**: Semantic emphasis
```latex
% CORRECT
This is \emph{very} important to understand.
We must do this \emph{now} before moving forward.
The \emph{benefits} of classes are clear.
```

**Why**: ALL CAPITALS in running text is considered shouting and poor typography. It's harder to read and looks unprofessional. Use `\emph{...}` to provide semantic emphasis, and LaTeX will render it appropriately (typically as italics, but this can be configured based on context and document style).

**Exception**: Acronyms and proper names that are conventionally written in capitals (e.g., NASA, USA, PDF) are fine and should not be emphasized.

### Floats: Figures and Tables

**Core principle**: An image is not a figure, but a figure can contain an image. Use proper figure and table environments with captions and labels.

#### Using sidecaption (memoir class)

When using the memoir document class, prefer `sidecaption` over traditional `\caption` for better layout and accessibility:

**For figures:**
```latex
\begin{figure}
  \begin{sidecaption}{Clear description of image content}[fig:label]
    \includegraphics[width=0.7\textwidth]{path/to/image}
  \end{sidecaption}
\end{figure}
```

**For tables:**
```latex
\begin{table}
  \begin{sidecaption}{Description of table content}[tab:label]
    \begin{tabular}{lcc}
      \toprule
      Header1 & Header2 & Header3 \\
      \midrule
      Data1 & Data2 & Data3 \\
      \bottomrule
    \end{tabular}
  \end{sidecaption}
\end{table}
```

**Key features of sidecaption:**
- **Caption placement**: Caption appears alongside the content (figure/table), not above or below
- **Space efficiency**: Better use of page width, especially for narrow images/tables
- **Improved readability**: Caption and content are visually connected
- **Accessibility**: Clear semantic separation between caption text and content
- **Consistent labeling**: Label goes in optional second argument `[fig:label]`

**When to use sidecaption:**
- Medium-sized figures/tables that don't need full text width
- Educational materials where caption-content proximity aids understanding
- Documents using memoir class (Beamer presentations with memoir features)

**Traditional caption alternative:**
```latex
\begin{figure}
  \centering
  \includegraphics[width=0.7\textwidth]{path/to/image}
  \caption{Description of image content}
  \label{fig:label}
\end{figure}
```

Use traditional `\caption` + `\label` when:
- Not using memoir class
- Figure/table spans full text width
- Caption naturally belongs below content (e.g., wide tables)

#### Referencing figures and tables

Always use `\cref{fig:label}` (cleveref package) to reference figures and tables:

```latex
As shown in \cref{fig:memory-hierarchy}, secondary memory is slower but non-volatile.

The results in \cref{tab:benchmark} demonstrate...
```

**Never** hard-code references like "Figure 1", "Table 3.2", or use manual prefixes like `Figure~\ref{fig:label}`—let cleveref handle both numbering and prefixes automatically.

### Verbatim and Code
- Use `listings` package for code with syntax highlighting
- Use `\verb` for inline code snippets
- Never paste code as normal text

### Paths
- Always use forward slashes in paths: `figures/diagram.pdf` not `figures\diagram.pdf`
- Use platform-independent path specifications

## Workflow for Writing LaTeX

When writing or editing LaTeX content:

1. **Check file type**: Are you in a .nw literate programming file? If yes, use `[[code]]` notation, not `\texttt{...\_...}`
2. **Identify content structure**: Is this a list of uniform items or term-definition pairs?
3. **Choose semantic environment**: Match the environment to the content meaning
4. **Use proper commands**: Leverage LaTeX's semantic commands rather than manual formatting
5. **Verify cross-references**: Ensure labels and references are descriptive and correct
6. **Check for anti-patterns**: Review for `\textbf{Label:}` in itemize, `\texttt{..._...}` in .nw files

## When Reviewing LaTeX Code

Check for these common issues:
- [ ] Lists using `\textbf{Label:}` instead of `description` environment
- [ ] Hard-coded numbers instead of `\ref`
- [ ] Manual cross-reference prefixes (`\S\ref`, `Section~\ref`, `Figure~\ref`) instead of `\cref`
- [ ] Manual citation formatting instead of `\cite` commands
- [ ] Manual quotes (`"..."`, `'...'`, `` `...` ``) instead of `\enquote{...}`
- [ ] Images without `figure` environment
- [ ] Code without proper formatting (listings/verbatim)
- [ ] Windows-style backslashes in paths

**Additional checks for .nw literate programming files:**
- [ ] `\texttt{..._...}` or `\texttt{...__...}` → Should use `[[...]]` notation
- [ ] Description labels with underscores `\item[FOO\_BAR behavior]` → Use better label + `[[FOO_BAR]]` in text
- [ ] Any manually escaped underscores in code references → Use `[[...]]` instead

## Beamer: Presentation vs Article Mode

When creating Beamer presentations that also generate article versions, use `\mode<presentation>` and `\mode<article>` to provide appropriate content for each format.

### When to Split Content

**Verbose text environments**: Semantic environments (definition, remark, example, block) with more than 2-3 lines of prose are too verbose for slides.

**Solution**: Provide concise versions for presentations and full explanations for articles:

```latex
\begin{frame}
  \mode<presentation>{%
    \begin{remark}[Key Point]
      \begin{itemize}
        \item Concise point 1
        \item Concise point 2
        \item Concise point 3
      \end{itemize}
    \end{remark}
  }
  \mode<article>{%
    \begin{remark}[Key Point]
      Full explanatory paragraph with detailed reasoning and context
      that would overwhelm a slide but provides value in written form.

      Additional paragraphs can elaborate on nuances and implications.
    \end{remark}
  }
\end{frame}
```

### Visual Elements

**Side-by-side layouts** using `\textbytext`:
- Presentation: `\textbytext` (non-starred, column width, works in beamer frames)
- Article: `\textbytext*` (starred, fullwidth, better for printed documents)

```latex
\begin{frame}
  \mode<presentation>{%
    \textbytext{Left content}{Right content}
  }
  \mode<article>{%
    \textbytext*{Left content}{Right content}
  }
\end{frame}
```

### Principle

Slides need **visual clarity** and **conciseness** (bullets, short phrases). Articles can provide **depth** and **explanation** (full sentences, paragraphs). Design content appropriate for each medium.

## Standard Preamble

For new LaTeX documents, use the standard preamble from `references/preamble.tex`. Copy it verbatim to your project's `doc/preamble.tex` and include it with `\input{preamble}` after `\documentclass`.

**Document structure:**
```latex
\documentclass[a4paper,oneside]{memoir}
\input{preamble}

\title{Document Title}
\author{Author Name}
\date{\today}

\begin{document}
\frontmatter
\maketitle
\tableofcontents

\mainmatter
% Content here

\backmatter
\end{document}
```

**The standard preamble provides:**
- Language support (babel with swedish, british)
- Bibliography (biblatex with alphabetic style)
- Code highlighting (minted, noweb)
- Mathematics (amsmath, amssymb, mathtools, amsthm)
- Cross-references (cleveref with custom labels)
- Quotations (csquotes)
- Tables (booktabs)
- Various utilities (enumitem, acro, siunitx, etc.)

This ensures consistent formatting across all documents and projects.

## Remember

LaTeX is a **document preparation system** based on **semantic markup**, not a word processor. The goal is to describe what content *is*, not how it should *look*. Let LaTeX handle the formatting based on the semantic structure you provide.
