# DSP Architecture: [PluginName]

**CRITICAL CONTRACT:** This specification is immutable during Stages 1-4 implementation. Stage 1 Planning cannot proceed without this file. Stage 3 (DSP) implements this exact architecture.

**Generated by:** Stage 0 Research
**Referenced by:** Stage 1 (Planning), Stage 3 (DSP Implementation)
**Purpose:** DSP specification defining processing components, signal flow, and JUCE module usage

---

## Core Components

[For each DSP component, create a subsection with this structure:]

### [Component Name]
- **JUCE Class:** `juce::dsp::ClassName` or "Custom implementation (description)"
- **Purpose:** [What this component does]
- **Parameters Affected:** [List parameter IDs that control this component]
- **Configuration:**
  - [Specific settings, ranges, formulas]
  - [Initialization parameters]
  - [Any special handling]

**Example:**
```markdown
### Reverb Engine
- **JUCE Class:** `juce::dsp::Reverb`
- **Purpose:** Generate plate-style reverb with controllable size and decay
- **Parameters Affected:** SIZE, DECAY
- **Configuration:**
  - SIZE (0-100%) maps to roomSize parameter (0.0-1.0)
  - DECAY (0.1-10s) maps to combined roomSize + damping
  - Width fixed at 1.0 (full stereo)
  - freezeMode disabled
```

---

## Processing Chain

[ASCII diagram showing signal flow and parameter connections]

**Example:**
```
Input
  ↓
Dry/Wet Mixer (capture dry) ← MIX parameter
  ↓
Plate Reverb ← SIZE, DECAY parameters
  ↓
Modulation Delay ← AGE parameter
  ↓
Tape Saturation ← DRIVE parameter
  ↓
Filter ← TONE parameter
  ↓
Dry/Wet Mixer (blend) ← MIX parameter
  ↓
Output
```

**Routing notes:**
- [Describe any conditional routing]
- [Parallel processing paths]
- [Feedback loops]
- [Sidechain inputs]

---

## System Architecture

[Document non-DSP systems that the plugin requires. This section captures features beyond audio processing like file I/O, multi-output routing, MIDI handling, and state persistence. Skip subsections that don't apply to your plugin.]

### File I/O System (if applicable)

**File types supported:** [List supported formats: .wav, .aiff, .mp3, .flac, etc.]

**Loading strategy:**
- [Background thread loading - non-blocking]
- [Lazy loading - load on demand]
- [Preload all - load at initialization]
- [Streaming - load chunks as needed]

**JUCE classes:**
- `juce::File` - File path management
- `juce::AudioFormatManager` - Format detection and reading
- `juce::AudioFormatReader` - Audio file reading
- `juce::AudioFormatWriter` - Audio file writing (if applicable)
- [Other relevant classes]

**Thread safety:**
- File I/O operations run on [background thread / message thread]
- Communication with audio thread via [lock-free queue / atomic flags / mutex]
- Audio thread never blocks on file operations

**Error handling:**
- Missing files: [Behavior when saved file paths don't exist]
- Invalid formats: [How unsupported formats are handled]
- Read failures: [Fallback behavior on I/O errors]

**Example (DrumRoulette):**
```markdown
**File types supported:** .wav, .aiff, .mp3 (via system codecs)

**Loading strategy:** Background thread loading with lock-free queue
- Folder scanning happens on button click (background thread)
- Individual sample loading happens on slot assignment (background thread)
- Audio thread only accesses pre-loaded sample buffers

**JUCE classes:**
- `juce::File::findChildFiles()` - Recursive folder scanning
- `juce::AudioFormatManager` - Format detection
- `juce::AudioFormatReader` - Sample loading into memory

**Thread safety:**
- File scanning on background thread via `Thread::launch()`
- Loaded samples communicated via `std::atomic<bool>` ready flags
- Audio thread reads from pre-loaded `AudioBuffer<float>` (no locks)

**Error handling:**
- Missing folder on restore: Show warning, request new folder
- Invalid audio file: Skip file, continue scanning
- Large folders (>1000 files): Show progress indicator
```

---

### Multi-Output Routing (if applicable)

**Bus configuration:** [Describe input and output bus structure]
- Main stereo input + [additional inputs]
- Main stereo output + [N individual outputs]
- Total channel count: [X inputs, Y outputs]

**JUCE setup:**
- Configure via `BusesProperties` in AudioProcessor constructor
- Pattern: `BusesProperties().withInput("Input", AudioChannelSet::stereo()).withOutput("Main", AudioChannelSet::stereo()).withOutput("Slot1", AudioChannelSet::stereo())...`
- **CRITICAL:** Must be in constructor, NOT prepareToPlay (see juce8-critical-patterns.md)

**Output mapping:**
- [Describe which plugin elements route to which output buses]
- Main output: [What goes to main stereo bus]
- Individual outputs: [What goes to each additional bus]

**DAW compatibility notes:**
- Not all DAWs support >2 outputs
- Test required with: Logic Pro, Ableton Live, FL Studio, Reaper
- Fallback: [Behavior if DAW only supports stereo output]

**Example (DrumRoulette - 18 outputs):**
```markdown
**Bus configuration:**
- Input: 1 stereo input (MIDI input, no audio sidechain)
- Outputs:
  - Main stereo output (channels 0-1)
  - 8 individual stereo outputs (channels 2-3, 4-5, 6-7, 8-9, 10-11, 12-13, 14-15, 16-17)
- Total: 2 input channels, 18 output channels

**JUCE setup:**
```cpp
BusesProperties()
    .withInput("Input", juce::AudioChannelSet::stereo())
    .withOutput("Main", juce::AudioChannelSet::stereo())
    .withOutput("Slot1", juce::AudioChannelSet::stereo())
    .withOutput("Slot2", juce::AudioChannelSet::stereo())
    // ... repeat for 8 slots
```

**Output mapping:**
- Main output (0-1): Mixed output of all 8 slots
- Slot1 output (2-3): Isolated output for slot 1 (MIDI note 36)
- Slot2 output (4-5): Isolated output for slot 2 (MIDI note 38)
- ... (pattern continues for all 8 slots)

**DAW compatibility notes:**
- Logic Pro: Full support for 18 outputs
- Ableton Live: Supports multi-output, requires "Configure" in plugin device
- FL Studio: Supports via "Split mixer tracks"
- Fallback: If DAW doesn't support, all audio routes to main output
```

---

### MIDI Routing (if applicable)

**Input handling:** [How MIDI events are received and processed]

**Note mapping:**
- [MIDI note number → Plugin action/slot/oscillator]
- [Velocity handling: affects volume, filter, other parameters]
- [Channel handling: single channel or omni mode]

**JUCE classes:**
- `juce::MidiBuffer` - MIDI event storage
- `juce::MidiMessage` - Individual MIDI message parsing
- `juce::MPEValue` (if MPE support) - Polyphonic expression

**Processing:**
- MIDI messages processed in `processBlock()` on audio thread
- Note-on triggers: [Sample playback, oscillator start, etc.]
- Note-off handling: [Stop sample, apply release envelope, etc.]
- CC handling: [Which CCs map to parameters]

**Example (DrumRoulette - 8-slot drum sampler):**
```markdown
**Input handling:** Omni mode (responds to all MIDI channels)

**Note mapping:**
- MIDI note 36 (C1) → Slot 1 (Kick)
- MIDI note 38 (D1) → Slot 2 (Snare)
- MIDI note 42 (F#1) → Slot 3 (Closed Hi-Hat)
- ... (8 slots total, standard GM drum mapping)
- Velocity → Affects sample playback volume (0-127 mapped to 0.0-1.0)
- Notes outside slot range → Ignored

**JUCE classes:**
- `juce::MidiBuffer` - Iterate MIDI events in processBlock
- `juce::MidiMessage::getNoteNumber()` - Extract note number
- `juce::MidiMessage::getVelocity()` - Extract velocity

**Processing:**
```cpp
for (const auto metadata : midiMessages)
{
    auto message = metadata.getMessage();
    if (message.isNoteOn())
    {
        int note = message.getNoteNumber();
        int slotIndex = mapNoteToSlot(note); // 36→0, 38→1, etc.
        float velocity = message.getVelocity() / 127.0f;
        triggerSlot(slotIndex, velocity);
    }
}
```
```

---

### State Persistence (if applicable)

**What state is saved:**
- [Parameter values - automatically via APVTS]
- [Non-parameter state - folder paths, UI state, lock buttons, etc.]
- [User preferences - last selected preset, UI size, etc.]

**Serialization format:**
- APVTS parameters: Automatic via `AudioProcessorValueTreeState`
- Custom state: [XML via ValueTree, custom binary, JSON]

**JUCE classes:**
- `juce::AudioProcessorValueTreeState` - Parameter persistence (automatic)
- `juce::ValueTree` - Custom state tree
- `juce::XmlElement` - XML serialization (if used)
- `juce::MemoryBlock` - Binary serialization (if used)

**Save/restore methods:**
- `getStateInformation(MemoryBlock& destData)` - Serialize state
- `setStateInformation(const void* data, int sizeInBytes)` - Deserialize state

**Restore behavior:**
- Folder paths don't exist: [Show error, request new path, use default, clear slot]
- Invalid data: [Use defaults, show warning, skip corrupted sections]
- Version migration: [How old plugin versions are handled]

**Example (DrumRoulette):**
```markdown
**What state is saved:**
- APVTS parameters: Attack, Decay, Sustain, Release (automatic)
- Custom state:
  - Folder path for each slot (8 paths)
  - Currently loaded sample file for each slot (8 files)
  - Lock state for each slot (8 booleans)
  - Randomization mode (all/individual)

**Serialization format:**
- APVTS: XML via ValueTree (automatic)
- Custom state: ValueTree with structure:
  ```
  <PluginState>
    <Slot index="0" folderPath="/path/to/kicks" currentFile="kick1.wav" locked="1"/>
    <Slot index="1" folderPath="/path/to/snares" currentFile="snare3.wav" locked="0"/>
    ...
  </PluginState>
  ```

**JUCE classes:**
- `AudioProcessorValueTreeState` - APVTS automatic persistence
- `juce::ValueTree` - Custom state tree for slots
- `ValueTree::toXmlString()` - Serialize to XML string

**Restore behavior:**
- Folder path doesn't exist: Show warning in UI, slot shows "Missing Folder"
- Sample file doesn't exist: Re-randomize from folder (if folder exists)
- Lock states: Restored even if folder missing (prevents unwanted randomization)
- Invalid XML: Use defaults (empty slots), show error message
```

---

## Parameter Mapping

[Table mapping every parameter ID to DSP component and usage]

| Parameter ID | Type | Range | DSP Component | Usage |
|-------------|------|-------|---------------|-------|
| SIZE | Float | 0-100% | Reverb Engine | Maps to roomSize (0.0-1.0) |
| DECAY | Float | 0.1-10s | Reverb Engine | Maps to damping (inverse) |
| AGE | Float | 0-100% | Modulation Delay | Modulation depth (±20%) |
| DRIVE | Float | 0-100% | Tape Saturation | Gain factor (1.0-10.0) |
| TONE | Float | -100 to +100 | Filter | LP (negative) / HP (positive) |
| MIX | Float | 0-100% | Dry/Wet Mixer | Wet proportion (0.0-1.0) |
| MOD_MODE | Bool | 0/1 | Routing | Wet-only (0) or Wet+Dry (1) |

---

## Algorithm Details

[For each algorithm/component, describe the implementation approach]

### [Component Name]

**Algorithm:** [Mathematical description or processing approach]

**Implementation notes:**
- [Key formulas]
- [Coefficient calculation]
- [Interpolation method]
- [Edge case handling]

**Example:**
```markdown
### Tape Saturation

**Algorithm:** Hyperbolic tangent waveshaping

**Implementation notes:**
- Transfer function: `output = tanh(gain * input)`
- Gain curve: Linear 1.0-10.0 based on DRIVE (0-100%)
- Formula: `gain = 1.0 + (driveValue * 9.0)` where driveValue is 0.0-1.0
- No oversampling (reverb already has high-frequency rolloff)
- Apply per-sample in processBlock loop
```

---

## Integration Points

[Document how features interact and depend on each other. This section prevents "parts work but whole fails" scenarios by analyzing feature dependencies, parameter interactions, processing order requirements, and thread boundaries.]

### Feature Dependencies

[List dependencies between features identified during research]

**Pattern:**
- [Feature A] depends on [Feature B] because [reason]
- [Feature C] must complete before [Feature D] because [reason]

**Example (Shimmer Reverb):**
```markdown
- Pitch shifting → Feedback mixer: Pitch-shifted signal must feed back into reverb input
- Feedback mixer → Reverb engine: Feedback signal is input to reverb processing
- Reverb engine → Dry/wet mixer: Reverb output must be available for mixing with dry signal
- All features → Processing order: Sequential dependencies require specific order (see Processing Order below)
```

**Example (DrumRoulette):**
```markdown
- Folder scanning → File loading: Must scan folder before loading samples
- File loading → Sample playback: Must load sample into memory before playback
- Randomization → Folder scanning: Needs file list from scan to select random samples
- Lock buttons → Randomization: Locked slots are excluded from randomization
- MIDI routing → Sample playback: MIDI note-on triggers sample playback
- Multi-output routing → Sample playback: Each slot routes to specific output bus
```

---

### Parameter Interactions

[Document how parameters affect each other or influence multiple features]

**Pattern:**
- [Parameter X] affects [Parameter Y]: [description of interaction]
- [Parameter Z] influences [Feature A] and [Feature B]: [description of dual influence]
- [Parameter combination] creates [special behavior]: [conditional routing or processing]

**Example (Shimmer Reverb):**
```markdown
- Shimmer amount (feedback gain) affects reverb decay time:
  - Higher feedback → Longer perceived decay (feedback loop extends reverb tail)
  - Feedback + Decay interaction: High values for both can cause runaway feedback
  - Mitigation: Limit feedback gain based on decay time setting

- Reverb decay affects feedback loop stability:
  - Short decay + high feedback: Stable but abrupt
  - Long decay + high feedback: Risk of runaway (requires limiting)

- Dry/wet mix affects perceived shimmer intensity:
  - High wet mix → Shimmer dominates
  - Low wet mix → Shimmer is subtle effect
  - Does NOT affect feedback loop (feedback tapped pre-dry/wet mixer)
```

**Example (DrumRoulette):**
```markdown
- Attack/Decay/Sustain/Release affect all slots equally:
  - Global envelope applied to all triggered samples
  - No per-slot envelope control (design decision for simplicity)
  - Trade-off: Simpler UI vs. reduced flexibility

- Volume is per-slot (not APVTS parameter):
  - Each loaded sample has inherent volume (from file)
  - MIDI velocity scales playback volume (0-127 → 0.0-1.0 gain)
  - No global volume parameter (use DAW mixer instead)

- Lock state doesn't have APVTS parameter:
  - UI-only state (not automatable)
  - Persisted in custom state (ValueTree)
  - Prevents randomization from changing locked slots
```

---

### Processing Order Requirements

[Document the required order of operations in the processing chain. Critical for signal flow correctness.]

**Pattern:**
1. [Step 1]: [Operation] - [Reason why first]
2. [Step 2]: [Operation] - [Reason why second]
3. [Step 3]: [Operation] - [Reason for order]
...

**Parallel processing notes:**
- [Component A] and [Component B] can run in parallel because [reason]

**Example (Shimmer Reverb):**
```markdown
**Sequential processing order (REQUIRED):**

1. **Capture dry signal:** Store unprocessed input for later mixing
   - Must happen first to preserve clean input before any processing

2. **Pitch shift input:** Apply +1 octave pitch shift to input signal
   - Happens before feedback mixing to create shimmer harmonics

3. **Mix with feedback:** Combine pitch-shifted input with feedback buffer
   - Feedback from previous cycle mixes with current pitch-shifted signal

4. **Process through reverb:** Apply reverb to combined signal
   - Reverb processes the pitch-shifted + feedback mix

5. **Apply feedback gain:** Scale reverb output for feedback loop
   - Controls feedback intensity and prevents runaway

6. **Route to feedback buffer:** Store scaled output for next cycle
   - Feedback buffer used in step 3 of next processing cycle

7. **Blend with dry signal:** Mix reverb output with captured dry signal
   - Final output is dry/wet blend controlled by MIX parameter

**Why order matters:**
- Pitch shift before reverb: Creates shimmer effect (pitch-shifted reverb tail)
- Pitch shift after reverb: Would NOT create shimmer (just transposed reverb)
- Dry capture first: Prevents latency-corrupted dry signal in dry/wet mix
- Feedback loop closure: Step 6 output feeds step 3 input of next cycle
```

**Example (DrumRoulette):**
```markdown
**Per-sample processing order:**

1. **MIDI input received:** Host delivers MIDI buffer to processBlock()
   - Happens first (host-controlled timing)

2. **Note mapped to slot:** MIDI note number → Slot index (0-7)
   - Note 36 → Slot 0, Note 38 → Slot 1, etc.

3. **Check lock state:** If slot locked, skip triggering
   - Lock prevents re-triggering during performance

4. **Check sample loaded:** Verify slot has valid sample in memory
   - Empty slots are silent (no error)

5. **Trigger sample playback:** Start playing from sample buffer
   - Apply MIDI velocity to playback gain (0-127 → 0.0-1.0)

6. **Apply ADSR envelope:** Shape amplitude over time
   - Attack/Decay/Sustain/Release from APVTS parameters

7. **Route to output bus:** Write to appropriate output buffer
   - Main output: Mixed output (all slots)
   - Individual outputs: Slot-specific outputs (if enabled by DAW)

**Parallel processing:**
- All 8 slots process in parallel (independent voices)
- Each slot has own playback position, envelope state, velocity
- No inter-slot dependencies
```

---

### Thread Boundaries

[Document which operations run on which threads. Critical for thread safety and avoiding audio glitches.]

**Threads:**
- **Audio thread:** Real-time processing in processBlock() - NO allocations, NO locks, NO file I/O
- **Message thread:** UI interactions, parameter changes, button clicks
- **Background thread:** File I/O, folder scanning, long operations

**Communication mechanisms:**
- Atomic variables: Thread-safe flags and simple values
- Lock-free queues: Thread-safe message passing
- Mutexes: Used sparingly, NEVER in audio thread

**Example (Shimmer Reverb):**
```markdown
**Audio thread:**
- All DSP processing (pitch shift, reverb, feedback, dry/wet mix)
- Parameter reads via `APVTS::getRawParameterValue()->load()` (atomic)
- Sample-by-sample processing in processBlock()

**Message thread:**
- Parameter updates from UI via APVTS (atomic writes)
- Preset loading/saving
- UI repaints

**No background thread needed:**
- No file I/O
- No long-running operations
- All processing is real-time compatible

**Communication:**
- APVTS parameters: Atomic reads (audio thread) / atomic writes (message thread)
- No custom thread communication needed (APVTS handles it)
```

**Example (DrumRoulette):**
```markdown
**Audio thread:**
- Sample playback from pre-loaded buffers
- MIDI message processing
- ADSR envelope calculation
- Output routing to multiple buses
- Lock state reads via `std::atomic<bool>` (per-slot)

**Message thread:**
- UI button clicks (randomize, lock, folder browse)
- Folder path changes from UI
- Lock button state changes → `std::atomic<bool>` writes
- Trigger background scan via `Thread::launch()`

**Background thread:**
- Folder scanning via `File::findChildFiles()` (can take seconds for large folders)
- Sample loading via `AudioFormatReader` (can take milliseconds per file)
- File list population
- Progress updates via atomic counter

**Communication:**
- Lock states: `std::array<std::atomic<bool>, 8>` (lock flags per slot)
- Sample buffers: Pointer swap via `std::atomic<AudioBuffer<float>*>` (lock-free)
- File list: Lock-free queue (`juce::AbstractFifo`) for passing file paths
- Scan complete: `std::atomic<bool>` flag checked by message thread

**Safety guarantees:**
- Audio thread NEVER waits on file I/O
- Audio thread only reads from pre-loaded buffers (no locks)
- Background thread NEVER touches audio buffers in use
- Double-buffering for sample loading (load to temp, atomic swap)
```

---

## Implementation Risks

[Per-feature risk assessment with complexity ratings, risk factors, alternatives, and fallback architectures. Document Plan B before implementation starts.]

### [Feature Name]

**Complexity:** [LOW | MEDIUM | HIGH]

**Risk Level:** [LOW | MEDIUM | HIGH]

**Risk factors:**
- [Specific risk 1: e.g., "Phase vocoder is algorithmically complex"]
- [Specific risk 2: e.g., "Requires deep FFT understanding"]
- [Specific risk 3: e.g., "High CPU cost may cause performance issues"]

**Alternative approaches:**
1. [Alternative 1]: [Description, tradeoffs, when to use]
2. [Alternative 2]: [Description, tradeoffs, when to use]

**Fallback architecture:**
- [Reduced-scope version if primary approach fails]
- [Simpler algorithm with lower quality/features]
- [Conditions under which to fall back]

**Mitigation strategy:**
- [How to reduce risk: reference implementations, testing approach, incremental development]

**Example (Shimmer Reverb - Phase Vocoder):**
```markdown
### Phase Vocoder Pitch Shifting

**Complexity:** HIGH
- Requires FFT → phase adjustment → IFFT pipeline
- Overlapping FFT windows (75% overlap typical)
- Phase correction for bin resampling
- Buffer management for overlap-add synthesis

**Risk Level:** HIGH

**Risk factors:**
1. Phase vocoder is non-trivial to implement correctly
   - Phase unwrapping and adjustment requires expertise
   - Artifacts (phasiness, metallic sound) if done incorrectly
2. No existing JUCE phase vocoder class (custom implementation required)
3. High CPU cost (~40-60% single core for good quality)
4. Latency (FFT size = 2048 samples = ~42ms @ 48kHz)

**Alternative approaches:**
1. **Granular synthesis:**
   - Complexity: MEDIUM
   - Quality: Lower than phase vocoder (graininess artifacts)
   - CPU: ~20-30% single core
   - Latency: Lower (~10-20ms)
   - Best for: Less critical pitch shift quality

2. **Delay-based pitch shift:**
   - Complexity: LOW
   - Quality: Lowest (only works for small shifts, audible artifacts)
   - CPU: ~5% single core
   - Latency: Minimal (~1-2ms)
   - Best for: Prototyping, reduced-scope version

**Fallback architecture:**
- **Primary:** Phase vocoder (highest quality)
- **Fallback 1:** If phase vocoder artifacts unacceptable → Granular synthesis
- **Fallback 2:** If granular fails → Simple delay-based pitch shift + reverb
- **Fallback 3:** If all pitch shifting fails → Standard reverb with modulation (no shimmer, reduced scope)

**Mitigation strategy:**
1. Start with delay-based prototype to validate concept
2. Research open-source phase vocoder implementations (Rubber Band Library, DIRAC)
3. Implement phase vocoder in isolation, test with unit tests before integration
4. Benchmark CPU usage early (may need quality vs. performance modes)
5. Consider using third-party DSP library if custom implementation too complex
```

**Example (DrumRoulette - Multi-Output Routing):**
```markdown
### Multi-Output Routing (18 outputs)

**Complexity:** MEDIUM
- BusesProperties configuration is straightforward
- Output buffer management requires careful indexing
- DAW compatibility testing is time-consuming

**Risk Level:** MEDIUM

**Risk factors:**
1. Not all DAWs support >2 outputs (compatibility issue)
2. BusesProperties must be in constructor (common mistake → crash)
3. Buffer indexing errors can cause crashes or silent outputs
4. Some DAWs limit output count (e.g., 16 channels max)

**Alternative approaches:**
1. **Main stereo output only:**
   - Complexity: LOW
   - Functionality: Internal mixing of all slots
   - Limitation: No per-slot routing in DAW
   - Best for: Maximum compatibility

2. **User-selectable output count:**
   - Complexity: MEDIUM
   - Options: 2/8/18 outputs (user setting in plugin)
   - Benefit: User chooses compatibility vs. routing flexibility
   - Implementation: BusesProperties based on config

**Fallback architecture:**
- **Primary:** 18 outputs (main + 8 stereos)
- **Fallback 1:** If 18 outputs fail DAW compatibility → 10 outputs (main + 4 stereos)
- **Fallback 2:** If multi-output proves problematic → Main stereo only with internal mixing
- **Condition:** Fall back if user reports crashes or silent outputs in specific DAW

**Mitigation strategy:**
1. Check `juce8-critical-patterns.md` for BusesProperties gotchas (constructor timing)
2. Test with multiple DAWs early (Logic Pro, Ableton, FL Studio, Reaper)
3. Implement buffer indexing with bounds checking (debug builds)
4. Document DAW compatibility in user manual (which DAWs support 18 outputs)
5. Add UI indicator showing "Main output only" if DAW doesn't support multi-output
```

---

### Overall Project Risk

**Overall complexity:** [LOW | MEDIUM | HIGH]
- Based on highest complexity feature + number of high-risk features

**Highest risk component:** [Component name]
- [Why this is the highest risk]
- [What percentage of project risk does this represent]

**Recommended approach:**
1. [Start with low-risk features to validate architecture]
2. [Tackle medium-risk features with fallbacks ready]
3. [Implement high-risk feature last with prototype/testing phase]
4. [Keep fallback architectures documented and tested]

**Example (Shimmer Reverb):**
```markdown
**Overall complexity:** HIGH
- Phase vocoder (HIGH) + reverb (MEDIUM) + feedback loop (MEDIUM)
- 3 complex features with interdependencies

**Highest risk component:** Phase Vocoder
- Represents ~70% of project risk
- Most algorithmically complex
- No JUCE class (custom implementation)
- Highest CPU cost and artifact risk

**Recommended approach:**
1. **Phase 1 - Validate concept:** Implement delay-based pitch shift + reverb (LOW risk)
2. **Phase 2 - Build foundation:** Implement feedback loop and dry/wet mixing (MEDIUM risk)
3. **Phase 3 - Core implementation:** Replace delay-based with phase vocoder (HIGH risk)
4. **Phase 4 - Fallback testing:** If phase vocoder fails, implement granular synthesis (MEDIUM risk)
5. **Phase 5 - Polish:** Optimize CPU, tune parameters, reduce artifacts
```

---

## Architecture Decisions

[Document WHY this approach was chosen vs. alternatives. Critical for understanding reasoning when problems arise or improvements are needed.]

### [Decision Name]

**Decision:** [What was decided]

**Rationale:** [Why this approach was chosen]

**Alternatives considered:**
1. [Alternative 1]: [Why rejected - e.g., "Higher CPU cost", "No JUCE support"]
2. [Alternative 2]: [Why rejected]

**Tradeoffs accepted:**
- [What we're giving up - e.g., "Higher latency for better quality"]
- [Why this tradeoff is acceptable]

**Example (Shimmer Reverb - Pitch Shifting Algorithm Choice):**
```markdown
### Pitch Shifting Algorithm Choice

**Decision:** Use phase vocoder (FFT-based) for shimmer pitch shifting

**Rationale:**
- Industry standard approach (Strymon BigSky, Valhalla Shimmer, Eventide)
- Highest quality pitch shifting with minimal artifacts
- JUCE has FFT support (`juce::dsp::FFT`) reducing implementation complexity
- Phase vocoder is well-documented in DSP literature

**Alternatives considered:**
1. **Granular synthesis:**
   - Why rejected: Lower quality (graininess artifacts), professional plugins don't use this approach
   - When to reconsider: If phase vocoder CPU cost proves too high OR implementation too complex

2. **Delay-based pitch shift:**
   - Why rejected: Lowest quality (only works for small shifts, obvious artifacts), not professional-grade
   - When to reconsider: For initial prototype only (not production)

3. **Third-party library (Rubber Band Library):**
   - Why rejected: Adds external dependency, licensing complexity, doesn't use JUCE ecosystem
   - When to reconsider: If custom phase vocoder implementation fails or takes too long

**Tradeoffs accepted:**
- **Higher CPU usage:** Phase vocoder uses ~40-60% single core vs ~20-30% for granular or ~5% for delay-based
  - Acceptable because: Shimmer reverb is inherently CPU-intensive, target is studio use (not real-time performance)
- **Higher implementation complexity:** Phase vocoder is algorithmically complex vs simple granular or delay
  - Acceptable because: Research phase validated feasibility, fallback options documented, quality is priority
- **Latency:** FFT size of 2048 samples = ~42ms latency @ 48kHz
  - Acceptable because: Reverb already adds latency, shimmer is typically post-processing effect not real-time tracking

**When to revisit:**
- If CPU usage exceeds 80% single core (implement quality vs. performance modes)
- If implementation proves too complex (consider granular synthesis fallback)
- If artifacts are unacceptable despite tuning (consider Rubber Band Library)
```

**Example (DrumRoulette - 18-Output Design):**
```markdown
### Multi-Output Routing Design

**Decision:** Implement 18 outputs (main stereo + 8 individual stereos) from start

**Rationale:**
- Professional drum samplers offer per-slot routing (Battery, Kontakt, MPC)
- Mix engineers expect individual outputs for processing flexibility
- JUCE supports multi-output via BusesProperties (standard pattern)
- Target DAWs (Logic Pro, Ableton) support 18+ outputs

**Alternatives considered:**
1. **Main stereo output only:**
   - Why rejected: Reduces plugin value (no routing flexibility), not competitive with professional samplers
   - When to reconsider: If multi-output causes significant compatibility or implementation issues

2. **User-selectable output count (2/8/18):**
   - Why rejected: Adds UI complexity, most users want all outputs available, no significant benefit
   - When to reconsider: If DAW compatibility proves problematic (e.g., crashes in specific DAWs)

3. **Main + 4 stereo pairs (10 outputs):**
   - Why rejected: Arbitrary limitation (why 4 not 8?), doesn't align with 8-slot design
   - When to reconsider: If 18 outputs exceed DAW limits in testing

**Tradeoffs accepted:**
- **DAW compatibility:** Not all DAWs support >2 outputs
  - Mitigation: Fallback to main output if multi-output unavailable, document compatibility in manual
- **Increased complexity:** Managing 9 buses vs 1 bus
  - Acceptable because: BusesProperties pattern is well-documented, complexity is one-time setup cost
- **Buffer management:** Routing to 18 channels requires careful indexing
  - Mitigation: Bounds checking in debug builds, unit tests for output routing logic

**When to revisit:**
- If >20% of users report crashes with multi-output (fall back to main output only)
- If specific DAW has hard limit <18 outputs (implement user-selectable output count)
- If buffer management causes performance issues (profile and optimize)
```

---

## Special Considerations

### Thread Safety
- [Parameter access patterns - atomic reads?]
- [Lock-free update mechanisms]
- [Buffer ownership]

**Example:**
```markdown
- All parameter reads use atomic getRawParameterValue()->load()
- Filter coefficient updates happen in audio thread (no allocations)
- LFO phase state is per-channel (no shared state between channels)
```

### Performance
- [Estimated CPU usage per component]
- [Hot paths and optimization opportunities]
- [Buffer size sensitivity]

**Example:**
```markdown
- Reverb: ~30% CPU (most expensive component)
- Delay line interpolation: ~10% CPU
- Total estimated: ~50% single core at 48kHz
```

### Denormal Protection
- [How denormals are handled]
- [Which components need protection]

**Example:**
```markdown
- Use juce::ScopedNoDenormals in processBlock()
- All JUCE DSP components handle denormals internally
- Custom LFO uses phase wrapping to avoid denormals
```

### Sample Rate Handling
- [Sample rate dependent calculations]
- [Coefficient recalculation on rate change]
- [prepareToPlay requirements]

**Example:**
```markdown
- Reverb reinitialized in prepareToPlay() with new sample rate
- Delay line buffer sized for 200ms at maximum expected rate (192kHz)
- LFO phase increment: `(frequency * 2π) / sampleRate`
```

### Latency
- [Processing latency sources]
- [Host compensation requirements]

**Example:**
```markdown
- Base delay: 50ms (2400 samples at 48kHz)
- Reverb: 20-50ms internal latency
- Report total via getLatencySamples() for host compensation
```

---

## Research References

### Professional Plugins
[List 3-5 industry plugins researched, with observations]

**Example:**
```markdown
1. **Valhalla VintageVerb**
   - Classic plate reverb algorithms
   - Parameter range: Decay 0.1-20s typical
   - Observed: Heavy damping at short decay times

2. **FabFilter Pro-R**
   - Modern algorithmic reverb
   - Real-time spectrum display for feedback
   - Noted: Decay time affects both roomSize and damping parameters

3. **UAD EMT 140 Plate**
   - Hardware plate reverb emulation
   - Typical settings: Decay 1-5s for musical material
   - Warmth control similar to saturation drive
```

### JUCE Documentation
[Document JUCE classes researched and key findings]

**Example:**
```markdown
- **juce::dsp::Reverb**: Built-in reverb with roomSize, damping, width, freeze
- **juce::dsp::DelayLine**: Variable delay with multiple interpolation types
  - Lagrange3rd recommended for pitch modulation (smooth, low artifacts)
- **juce::dsp::IIR::Filter**: Biquad filter with coefficient helpers
- **juce::dsp::DryWetMixer**: Handles latency-compensated dry/wet mixing
```

### Technical Resources
[Any academic papers, DSP books, tutorials referenced]

**Example:**
```markdown
- DAFX (Digital Audio Effects) - Reverb chapter for algorithm understanding
- Designing Audio Effect Plugins in C++ (Will Pirkle) - Filter design patterns
- freeverb algorithm - Open-source reference for understanding Reverb parameters
```

---

## Notes

[Any additional context, decisions made during research, open questions]

**Example:**
```markdown
- Chose Lagrange3rd interpolation over Linear for smoother pitch modulation
- Decided against oversampling for saturation (reverb already band-limits)
- MOD_MODE routing decision: wet-only vs wet+dry based on creative brief
- Filter bypass zone (±0.5%) prevents clicking when sweeping through center
```
