"""
Python Workflow Example

Demonstrates durable, multi-step workflows with:
- Step definitions using @step.do decorator
- DAG dependencies between steps
- Sleep/delay functionality
- Error handling

Requirements:
- Add "python_workflows" to compatibility_flags in wrangler.jsonc
- Add workflow binding configuration
"""

from workers import WorkflowEntrypoint, WorkerEntrypoint, Response
from js import fetch
import json


class DataProcessingWorkflow(WorkflowEntrypoint):
    """
    Example workflow that fetches data, processes it, and stores results.

    Steps are automatically persisted and retried on failure.
    """

    async def run(self, event, step):
        """
        Main workflow execution.

        Args:
            event: Workflow event with payload
            step: Step helper for defining durable steps

        Returns:
            Final workflow result (must be JSON-serializable)
        """
        # Access payload passed when workflow was created
        payload = event.get("payload", {})

        # ============================================
        # STEP 1: Fetch external data
        # ============================================
        @step.do("fetch_data")
        async def fetch_data():
            """
            Fetch data from external API.

            IMPORTANT: All I/O (fetch, database, etc.) must be inside steps!
            This ensures durability - if the step fails, it will be retried.
            """
            url = payload.get("url", "https://api.example.com/data")
            response = await fetch(url)
            data = await response.json()
            return data

        data = await fetch_data()

        # ============================================
        # STEP 2: Wait/delay
        # ============================================
        await step.sleep("wait_before_processing", "5 seconds")

        # ============================================
        # STEP 3: Process data
        # ============================================
        @step.do("process_data")
        async def process_data():
            """Process the fetched data."""
            # Example processing
            processed = {
                "item_count": len(data) if isinstance(data, list) else 1,
                "processed_at": "2025-12-08T00:00:00Z",  # Use string for JSON
                "status": "completed"
            }
            return processed

        result = await process_data()

        # ============================================
        # STEP 4: Store results (if using KV/D1)
        # ============================================
        # @step.do("store_results")
        # async def store_results():
        #     await self.env.MY_KV.put(
        #         f"result:{payload.get('id')}",
        #         json.dumps(result)
        #     )
        #     return {"stored": True}
        #
        # await store_results()

        return result


class DAGWorkflowExample(WorkflowEntrypoint):
    """
    Example of DAG (Directed Acyclic Graph) workflow with parallel steps.

    Steps can declare dependencies, and independent steps run concurrently.
    """

    async def run(self, event, step):
        # ============================================
        # Independent steps (can run in parallel)
        # ============================================
        @step.do("fetch_users")
        async def fetch_users():
            # Simulated fetch
            return [{"id": 1, "name": "Alice"}, {"id": 2, "name": "Bob"}]

        @step.do("fetch_products")
        async def fetch_products():
            # Simulated fetch
            return [{"id": 101, "name": "Widget"}]

        # ============================================
        # Dependent step (waits for both above)
        # ============================================
        @step.do("combine_data", depends=[fetch_users, fetch_products], concurrent=True)
        async def combine_data(users, products):
            """
            This step receives results from its dependencies.

            Args:
                users: Result from fetch_users
                products: Result from fetch_products
            """
            return {
                "user_count": len(users),
                "product_count": len(products),
                "combined": True
            }

        # Execute the DAG (dependencies resolve automatically)
        final_result = await combine_data()

        return final_result


class Default(WorkerEntrypoint):
    """
    Worker entry point that triggers workflows.
    """

    async def fetch(self, request):
        """Handle HTTP requests and trigger workflows."""
        url = request.url
        path = url.split("?")[0].rstrip("/")

        # Start data processing workflow
        if path.endswith("/workflow/start"):
            try:
                body = await request.json()
            except:
                body = {}

            # Create workflow instance
            instance = await self.env.DATA_WORKFLOW.create(
                params={"payload": body}
            )

            return Response(
                json.dumps({
                    "message": "Workflow started",
                    "instance_id": instance.id
                }),
                headers={"Content-Type": "application/json"}
            )

        # Check workflow status
        if path.endswith("/workflow/status"):
            # Get instance ID from query params
            # instance = await self.env.DATA_WORKFLOW.get(instance_id)
            # status = await instance.status()
            return Response(
                json.dumps({"message": "Status endpoint"}),
                headers={"Content-Type": "application/json"}
            )

        return Response(
            json.dumps({
                "endpoints": [
                    "POST /workflow/start - Start workflow",
                    "GET /workflow/status?id=xxx - Check status"
                ]
            }),
            headers={"Content-Type": "application/json"}
        )


# ============================================
# WRANGLER.JSONC CONFIGURATION
# ============================================
#
# Add to your wrangler.jsonc:
#
# {
#   "compatibility_flags": ["python_workers", "python_workflows"],
#   "compatibility_date": "2025-08-01",
#   "workflows": [
#     {
#       "name": "data-processing-workflow",
#       "binding": "DATA_WORKFLOW",
#       "class_name": "DataProcessingWorkflow"
#     },
#     {
#       "name": "dag-workflow-example",
#       "binding": "DAG_WORKFLOW",
#       "class_name": "DAGWorkflowExample"
#     }
#   ]
# }
