/**
 * Google Chat Interactive Bot (Cloudflare Worker)
 *
 * Full-featured bot that handles:
 * - User messages
 * - Button clicks
 * - Form submissions
 * - Bearer token verification
 */

interface Env {
  // Secrets
  VERIFY_TOKENS: 'true' | 'false' // Set to 'false' for development
}

interface ChatEvent {
  type: 'MESSAGE' | 'CARD_CLICKED' | 'ADDED_TO_SPACE' | 'REMOVED_FROM_SPACE'
  message?: {
    text?: string
    sender: { displayName: string }
    thread: { name: string }
  }
  action?: {
    actionMethodName: string
    parameters: Array<{ key: string; value: string }>
  }
  space: { name: string }
}

export default {
  async fetch(request: Request, env: Env): Promise<Response> {
    // Verify bearer token in production
    if (env.VERIFY_TOKENS === 'true') {
      const token = request.headers.get('Authorization')?.split('Bearer ')[1]
      if (!token || !(await verifyBearerToken(token))) {
        return new Response('Unauthorized', { status: 401 })
      }
    }

    const event: ChatEvent = await request.json()

    // Handle different event types
    switch (event.type) {
      case 'ADDED_TO_SPACE':
        return handleAddedToSpace(event)

      case 'MESSAGE':
        return handleMessage(event)

      case 'CARD_CLICKED':
        return handleCardClicked(event)

      case 'REMOVED_FROM_SPACE':
        return new Response(null, { status: 200 })

      default:
        return Response.json({ text: 'Unknown event type' })
    }
  }
}

function handleAddedToSpace(event: ChatEvent) {
  return Response.json({
    text: `Hello! I'm your bot.`,
    cardsV2: [{
      cardId: 'welcome-card',
      card: {
        header: { title: '👋 Welcome!' },
        sections: [{
          widgets: [
            {
              textParagraph: {
                text: 'I can help you with various tasks. Click a button below to get started.'
              }
            },
            {
              buttonList: {
                buttons: [
                  {
                    text: 'Show Form',
                    onClick: {
                      action: {
                        function: 'showForm'
                      }
                    }
                  },
                  {
                    text: 'Get Info',
                    onClick: {
                      action: {
                        function: 'getInfo'
                      }
                    }
                  }
                ]
              }
            }
          ]
        }]
      }
    }]
  })
}

function handleMessage(event: ChatEvent) {
  const userMessage = event.message?.text || ''
  const userName = event.message?.sender.displayName || 'User'

  return Response.json({
    text: `Hi ${userName}! You said: "${userMessage}"`,
    thread: {
      name: event.message?.thread.name
    }
  })
}

function handleCardClicked(event: ChatEvent) {
  const action = event.action?.actionMethodName

  switch (action) {
    case 'showForm':
      return showDataCollectionForm()

    case 'getInfo':
      return Response.json({
        text: 'Here is some information',
        cardsV2: [{
          cardId: 'info-card',
          card: {
            sections: [{
              widgets: [{
                textParagraph: {
                  text: 'This is a sample information card.'
                }
              }]
            }]
          }
        }]
      })

    case 'submitForm':
      return handleFormSubmission(event)

    default:
      return Response.json({ text: 'Unknown action' })
  }
}

function showDataCollectionForm() {
  return Response.json({
    actionResponse: {
      type: 'DIALOG',
      dialogAction: {
        dialog: {
          body: {
            sections: [{
              header: 'Enter Your Information',
              widgets: [
                {
                  textInput: {
                    name: 'name',
                    label: 'Name',
                    type: 'SINGLE_LINE',
                    hintText: 'John Doe'
                  }
                },
                {
                  textInput: {
                    name: 'email',
                    label: 'Email',
                    type: 'SINGLE_LINE',
                    hintText: 'john@example.com'
                  }
                },
                {
                  selectionInput: {
                    name: 'department',
                    label: 'Department',
                    type: 'DROPDOWN',
                    items: [
                      { text: 'Engineering', value: 'eng' },
                      { text: 'Sales', value: 'sales' },
                      { text: 'Marketing', value: 'marketing' }
                    ]
                  }
                },
                {
                  buttonList: {
                    buttons: [{
                      text: 'Submit',
                      onClick: {
                        action: {
                          function: 'submitForm'
                        }
                      }
                    }]
                  }
                }
              ]
            }]
          }
        }
      }
    }
  })
}

function handleFormSubmission(event: ChatEvent) {
  // Get form values from event
  const params = event.action?.parameters || []
  const formData = Object.fromEntries(
    params.map(p => [p.key, p.value])
  )

  // Validate
  if (!formData.name || !formData.email) {
    return Response.json({
      actionResponse: {
        type: 'DIALOG',
        dialogAction: {
          actionStatus: {
            statusCode: 'INVALID_ARGUMENT',
            userFacingMessage: 'Name and email are required'
          }
        }
      }
    })
  }

  // Success
  return Response.json({
    actionResponse: {
      type: 'NEW_MESSAGE'
    },
    text: `✅ Form submitted!\n\nName: ${formData.name}\nEmail: ${formData.email}\nDepartment: ${formData.department}`
  })
}

async function verifyBearerToken(token: string): Promise<boolean> {
  // See bearer-token-verify.ts for full implementation
  // For now, basic validation
  return token.length > 0
}
