# Project Planning Skill

**Version**: 2.0
**Last Updated**: 2025-11-17
**Status**: Production Ready ✅

---

## Purpose

This skill generates comprehensive, context-optimized planning documentation for web application projects. It structures work into manageable phases with built-in verification, ensuring projects can be built iteratively while maintaining clarity between sessions.

---

## When to Use This Skill

### Primary Use Cases

**1. Starting a New Project** (Most Common)
```
User: "I want to build a task management app with authentication"
Claude: "Let me use the project-planning skill to structure this properly"
→ Skill generates planning docs
→ Review and refine
→ Start building
```

**2. Adding a Major Feature**
```
User: "I need to add real-time collaboration to the existing app"
Claude: "Let me use project-planning to create phases for this feature"
→ Skill generates new phases
→ Append to existing IMPLEMENTATION_PHASES.md
```

**3. Re-planning a Messy Project**
```
User: "This project has gotten out of hand, let's restructure it"
Claude: "Let me use project-planning to reorganize the work"
→ Skill analyzes current state
→ Generates new phase structure
```

**4. Validating Existing Phases**
```
User: "Does Phase 3 look reasonable, or is it too big?"
Claude: "Let me use project-planning to validate this phase"
→ Skill checks sizing rules
→ Suggests improvements
```

---

## What This Skill Generates

### Core Output (Always)
- **IMPLEMENTATION_PHASES.md** - Structured phase breakdown with verification criteria
  - **NEW in v1.1**: Enhanced with file-level detail including:
    - **File Maps** - Shows which files to create/modify, their purpose, and dependencies
    - **Data Flow Diagrams** - Mermaid diagrams showing request/response flows
    - **Critical Dependencies** - Internal files, external packages, config, bindings
    - **Gotchas & Known Issues** - Security concerns, performance patterns, common pitfalls

### Optional Outputs (Generated When Needed)
- **DATABASE_SCHEMA.md** - Tables, relationships, indexes, migrations
- **API_ENDPOINTS.md** - Routes, methods, auth, request/response schemas
- **ARCHITECTURE.md** - System design, data flow, service boundaries
- **UI_COMPONENTS.md** - Component hierarchy, state management, forms, phase-aligned installation strategy for shadcn/ui
- **CRITICAL_WORKFLOWS.md** - Complex setup steps, order-sensitive workflows, gotchas documentation
- **INSTALLATION_COMMANDS.md** - Copy-paste commands per phase (quick command reference)
- **ENV_VARIABLES.md** - All secrets, API keys, configuration guide (dev/prod setup)
- **SESSION.md** - Compact tracking template (<200 lines, references phases instead of duplicating)
- **TESTING.md** - Test strategy, E2E flows, integration tests
- **AGENTS_CONFIG.md** - AI agents, tools, workflows (for AI-enabled apps)
- **INTEGRATION.md** - Third-party services, webhooks, API integrations

**Decision Logic**: Skill asks if additional docs are needed, or auto-generates based on project complexity
- Database with >3 tables → Suggest DATABASE_SCHEMA.md
- API with >5 endpoints → Suggest API_ENDPOINTS.md
- Multiple services/workers → Suggest ARCHITECTURE.md
- AI-powered features → Suggest AGENTS_CONFIG.md

---

## How It Works

### 1. Smart Defaults for Web Apps

The skill knows your preferred stack from CLAUDE.md:
- **Frontend**: Vite + React + Tailwind v4 + shadcn/ui
- **Backend**: Cloudflare Workers (with Static Assets for frontend)
- **Database**: D1 (SQL migrations)
- **Storage**: R2 (files), KV (config/cache)
- **Auth**: Clerk (JWT verification)
- **Deployment**: Wrangler CLI

It only asks clarifying questions when:
- Non-standard tech is mentioned
- Project complexity suggests alternatives
- User specifies different preferences

### 2. Hybrid Interaction

**Template-driven with smart questions**:
1. Analyzes project description
2. Identifies phase types needed (Infrastructure, Database, API, UI, Integration, Testing)
3. Asks targeted questions:
   - "Do you need social auth (Google, GitHub)?"
   - "Should users have roles/permissions?"
   - "Any real-time features (Durable Objects)?"
4. Generates docs with completed defaults and your specific choices

### 3. Phase Validation Rules

Every generated phase follows context-safe sizing:
- **Max file scope**: 5-8 files per phase
- **Max dependencies**: Shouldn't require deep knowledge of >2 other phases
- **Time estimate**: Includes implementation + verification + expected fixes
- **Verification required**: Every phase has checkbox criteria
- **Exit criteria required**: Clear "done" definition

If a phase violates these rules, skill auto-suggests splitting it.

---

## Phase Types

The skill uses standardized templates for common web app patterns:

### Infrastructure Phase
- Project scaffolding (Vite, Wrangler, dependencies)
- Build/deploy configuration
- **Verification**: Can deploy to Cloudflare, dev server runs

### Database Phase
- Schema design (tables, relationships, indexes)
- Migration files
- Seed data for testing
- **Verification**: CRUD operations work, constraints enforced

### API Phase
- Route definitions
- Middleware (auth, CORS, error handling)
- Request/response validation (Zod schemas)
- **Verification**: Endpoint tests pass (200, 401, 400, 500 cases)

### UI Phase
- Component structure (shadcn/ui composition)
- Form validation (Zod + React Hook Form)
- State management (TanStack Query for server state, Zustand for client state)
- **Verification**: User flows work, forms validate, states update correctly

### Integration Phase
- Third-party API setup (Clerk, Stripe, OpenAI, etc)
- Webhook handlers
- Configuration (environment variables, bindings)
- **Verification**: External service integration works, webhooks fire correctly

### Testing Phase (Optional)
- E2E test setup
- Integration tests for critical flows
- **Verification**: Test suite passes, coverage meets threshold

---

## File-Level Navigation (NEW in v1.1)

### What Problem Does This Solve?

**Before file maps**: Claude needs to grep/glob through your codebase to understand where files are and what they do. This burns tokens and sometimes results in code being placed in the wrong files.

**With file maps**: Each phase includes a detailed map showing:
- Which files to create or modify
- What each file's purpose is
- Dependencies between files
- Security and performance considerations

### Token Efficiency Gains

**Example: "Implement task CRUD endpoints"**

| Approach | Token Usage | Corrections Needed | Time |
|----------|-------------|-------------------|------|
| Without file maps | ~15k tokens | 2-3 corrections | ~10 min |
| With file maps | ~3.5k tokens | 0 corrections | ~3 min |
| **Savings** | **~77% reduction** | **No corrections** | **~70% faster** |

### Enhanced Phase Structure

Each phase now includes (when applicable):

**1. File Map**
```markdown
- `src/routes/tasks.ts` (estimated ~150 lines)
  - Purpose: CRUD endpoints for tasks
  - Key exports: GET, POST, PATCH, DELETE handlers
  - Dependencies: schemas.ts, auth middleware, D1 binding
  - Used by: Frontend task components
```

**2. Data Flow Diagrams** (Mermaid)
- Sequence diagrams for API calls
- Flowcharts for component logic
- Architecture diagrams for system components

**3. Critical Dependencies**
- Internal files (what imports what)
- External packages (with version hints)
- Configuration (env vars, Cloudflare bindings)

**4. Gotchas & Known Issues**
- Security patterns (ownership checks, auth)
- Performance considerations (pagination, caching)
- Framework quirks (Cloudflare Workers limits)

### When File Maps Are Included

**Always include** for:
- API phases (prevents wrong endpoint placement)
- UI phases (shows component hierarchy)
- Integration phases (shows external service touchpoints)

**Optional** for:
- Infrastructure phases (scaffolding is self-evident)
- Database phases (schema files are self-documenting)

### Example: Enhanced Phase

See `references/example-enhanced-phase.md` for complete before/after comparison with multiple Mermaid diagram examples.

---

## New in v2.0: Enhanced Planning Outputs

### 1. Phase-Aligned UI Component Installation (shadcn/ui)

**Problem Solved**: Users were installing all shadcn components upfront, then forgetting which components each phase needed.

**Solution**: UI_COMPONENTS.md now includes phase-aligned installation strategy:
- **Phase 1: Auth UI** → Install button, input, label, card, sonner
- **Phase 2: Forms** → Install textarea, checkbox, select, switch
- **Phase 3: Data Display** → Install table, data-table, dialog, alert-dialog
- Quick reference commands (MVP install, full install, update all)
- Component usage by route mapping
- Design decisions documented (why sonner vs toast, etc.)

**Time Saved**: ~15-20 minutes per project (no more "which components do I need?")

### 2. Critical Workflows Documentation

**Problem Solved**: Complex, order-sensitive setup steps (like D1 binding order, better-auth factory patterns) were causing users to get stuck.

**Solution**: CRITICAL_WORKFLOWS.md documents:
- Step-by-step workflows with specific commands
- Why order matters (prevents "what breaks if done wrong")
- Common mistakes and how to recover
- Links to official docs and related skills

**Example**: D1 Database Binding workflow prevents the "cannot bind to non-existent Worker" error

**Time Saved**: ~30-60 minutes per complex workflow (no more trial-and-error)

### 3. Installation Commands Reference

**Problem Solved**: Users repeatedly asking "what was that command again?" between sessions.

**Solution**: INSTALLATION_COMMANDS.md provides:
- Copy-paste commands organized by phase
- Database commands (create, migrate, query)
- Deployment commands (deploy, set secrets, tail logs)
- Troubleshooting commands (clear cache, verify bindings)

**Time Saved**: ~5-10 minutes per session (instant command lookup)

### 4. Environment Variables Guide

**Problem Solved**: Users unsure which secrets/API keys needed, where to get them, how to configure.

**Solution**: ENV_VARIABLES.md includes:
- Development (.dev.vars) vs Production (wrangler secrets) setup
- Where to get each API key (with links)
- Security notes (what to commit, what to never commit)
- Setup checklists for local dev and production
- If secrets leaked recovery steps

**Time Saved**: ~20-30 minutes initial setup + prevents security mistakes

### 5. Compact SESSION.md Template

**Problem Solved**: SESSION.md was growing to 400+ lines by duplicating phase information from IMPLEMENTATION_PHASES.md.

**Solution**: New compact template (<200 lines):
- References IMPLEMENTATION_PHASES.md instead of duplicating
- Phase 0 + Phase 1 expanded, rest collapsed to 2-3 lines each
- Links to CRITICAL_WORKFLOWS.md prominently
- Critical Reminders section (what to check before starting)
- Known Risks section (high-risk phases identified upfront)

**Token Saved**: ~8-12k tokens per session (no duplicate phase info)

### 6. Pre-Planning Validation

**Problem Solved**: Users planning before prototyping new frameworks, missing critical workflows.

**Solution**: Skill now asks before generating docs:
1. "Have you built a prototype or POC?" → Suggests spike if no
2. "Any complex setup workflows discovered?" → Generates CRITICAL_WORKFLOWS.md if yes
3. "Tech stack familiarity?" → Adds learning curve buffer to estimates if learning

**Time Saved**: Prevents hours of backtracking from wrong assumptions

### 7. Post-Generation Validation Checklist

**Problem Solved**: Users starting Phase 1 without reviewing docs, environment setup, or critical workflows.

**Solution**: Skill outputs checklist after generating docs:
- **Files**: All docs reviewed, SESSION.md references correct files, critical workflows read
- **Understanding**: Phase 1 tasks understood, dependencies clear, risks identified
- **Environment**: GitHub repo, dev tools installed, accounts set up, CLI authenticated

**Errors Prevented**: "Missing IMPLEMENTATION_PHASES.md", "Forgot to set secrets", "Skipped critical workflow"

---

## v2.0 Impact Summary

| Metric | v1.1 | v2.0 | Improvement |
|--------|------|------|-------------|
| Docs Generated | 7 types | 11 types | +57% |
| Time Saved per Project | ~2 hours | ~4-5 hours | +2-3 hours |
| Common Errors Prevented | 5-7 | 12-15 | +100% |
| Token Efficiency (SESSION.md) | Baseline | -60% tokens | ~10k tokens saved |
| Setup Mistakes Prevented | No validation | Pre/Post checks | Fewer stuck users |

**Production Tested**: TanStack Cloudflare Starter project (revealed all these improvements)

---

## Auto-Trigger Keywords

Claude Code will automatically discover this skill when you mention:

**Project Initiation**:
- "new project", "start a project", "create app", "build app", "scaffold project"
- "planning documentation", "project planning", "implementation phases"
- "structure project", "organize project", "plan features"

**Planning Documents**:
- "IMPLEMENTATION_PHASES.md", "DATABASE_SCHEMA.md", "API_ENDPOINTS.md"
- "ARCHITECTURE.md", "session handoff", "phase breakdown"
- "planning docs", "project spec", "requirements doc"

**Phase Management**:
- "phases", "phased approach", "implementation phases", "context-safe phases"
- "verification criteria", "exit criteria", "phase validation"
- "split phase", "merge phases", "adjust phases"

**Stack Keywords**:
- "Cloudflare Workers", "Vite + React", "D1 database", "full-stack app"
- "Clerk auth", "Tailwind v4", "shadcn/ui", "TanStack Query"

**Use Cases**:
- "major feature", "restructure project", "plan sprint", "roadmap"
- "session tracking", "context management", "git checkpoint"

**AI/Integration Projects**:
- "AI agents", "LLM features", "third-party integration", "webhooks"
- "real-time features", "Durable Objects", "websockets"

**File-Level Navigation** (NEW):
- "file map", "code navigation", "data flow", "mermaid diagram"
- "file dependencies", "gotchas", "known issues", "security patterns"
- "token efficiency", "which files", "where to start"

---

## Invocation

### Automatic (Recommended)
When you describe a new project, Claude Code will automatically suggest using this skill:
```
User: "I want to build a blog with comments and user profiles"
Claude: "Let me use the project-planning skill to structure this"
```

### Manual
Explicitly request the skill:
```
"Use the project-planning skill to plan this project"
"Generate planning docs for this feature"
"Validate my existing phases with project-planning"
```

---

## Example Workflow

**User**: "I want to build a URL shortener with analytics"

**Skill**:
```
Analyzing project requirements...

Project Type: Web App (CRUD + Analytics)
Suggested Stack: Cloudflare Workers + D1 + Vite + React

Questions:
1. Auth required? (public tool vs user accounts)
2. Custom short URLs or auto-generated?
3. Analytics detail level? (simple count vs detailed visitor data)
4. Admin dashboard needed?
```

**User**: "User accounts, auto-generated, detailed analytics, yes to dashboard"

**Skill Generates**:
- IMPLEMENTATION_PHASES.md (6 phases)
- DATABASE_SCHEMA.md (users, urls, analytics_events tables)
- API_ENDPOINTS.md (auth, URL CRUD, redirect, analytics)

**Output Preview**:
```markdown
# Implementation Phases: URL Shortener

## Phase 1: Project Setup (2 hours)
**Type**: Infrastructure
**Files**: package.json, wrangler.jsonc, vite.config.ts, src/index.ts

**Tasks**:
- [x] Scaffold Cloudflare Worker with Vite
- [x] Configure Tailwind v4 + shadcn/ui
- [x] Setup D1 database binding
- [x] Test deployment

**Verification**:
- [ ] `npm run dev` starts without errors
- [ ] Can deploy to Cloudflare
- [ ] Worker serves React app

**Exit Criteria**: Working dev environment and successful deployment

---

## Phase 2: Database Schema (2-3 hours)
**Type**: Database
**Files**: migrations/0001_initial.sql, src/db/schema.ts

[... and so on for each phase ...]
```

---

## What This Skill Does NOT Do

**Not a code generator**: Generates planning docs, not implementation code
**Not a session tracker**: Doesn't update SESSION.md (that's Claude's job)
**Not an architect**: Doesn't make technical decisions (Claude does that with you)
**Not mandatory**: Optional tool - use when beneficial

---

## Expanding the Skill

Easy to add new document templates:
1. Create template in `templates/` directory
2. Add generation logic to SKILL.md
3. Update this README

Future possibilities:
- `phase-split`: Break oversized phase into sub-phases
- `session-init`: Generate SESSION.md from IMPLEMENTATION_PHASES.md
- More phase types (Migration, Optimization, Security)

---

## Files in This Skill

```
project-planning/
├── README.md                          # This file
├── SKILL.md                           # Main skill logic (planning assistant)
├── templates/
│   ├── IMPLEMENTATION_PHASES.md       # Phase breakdown template (enhanced with file maps)
│   ├── DATABASE_SCHEMA.md             # Database design template
│   ├── API_ENDPOINTS.md               # API routes template
│   ├── ARCHITECTURE.md                # System design template
│   ├── UI_COMPONENTS.md               # Component structure template
│   ├── TESTING.md                     # Test strategy template
│   ├── AGENTS_CONFIG.md               # AI agents template
│   └── INTEGRATION.md                 # Third-party integrations template
└── references/
    ├── example-enhanced-phase.md      # NEW: File maps, Mermaid diagrams, before/after
    └── example-outputs/
        ├── simple-web-app.md          # Basic CRUD example
        ├── auth-web-app.md            # Authentication example
        └── ai-web-app.md              # AI-powered app example
```

---

## Tips for Best Results

**Be specific in project description**: "Task manager with tags and due dates" is better than "to-do app"

**Mention non-standard requirements early**: "Need to support 10k concurrent users" or "Must integrate with legacy SOAP API"

**Review generated phases before starting**: Easier to refine planning docs than mid-implementation

**Use examples for reference**: Check `references/example-outputs/` directory for similar projects

**Iterate if needed**: Regenerate or refine phases if initial output doesn't feel right

---

## Integration with Session Workflow

After this skill generates IMPLEMENTATION_PHASES.md:

1. Review and refine phases
2. Create SESSION.md (manually or with future `session-init` tool)
3. Start Phase 1
4. Update SESSION.md as you progress
5. Create git checkpoints at phase boundaries

See main CLAUDE.md for full session handoff protocol.

---

## Questions?

This skill is designed to evolve based on usage. If something doesn't work as expected or you'd like additional capabilities, refine SKILL.md and templates directly.
