# Django Schema Migration Checklist

Quick reference checklist for common Django migration scenarios. Use this to ensure you follow best practices and avoid common pitfalls.

## Pre-Migration Checklist

Before creating any migration:

- [ ] **Understand the change** - What schema modification is needed?
- [ ] **Check for data loss** - Will any data be lost or need transformation?
- [ ] **Consider multi-tenancy** - Does this affect tenant isolation or indexes?
- [ ] **Check production impact** - How long will migration take? Need downtime?
- [ ] **Plan rollback strategy** - How to reverse if something goes wrong?
- [ ] **Review dependencies** - What other apps/models does this affect?

## Creating Migration

- [ ] **Make model changes** in `models.py`
- [ ] **Generate migration**: `python manage.py makemigrations [app_name]`
- [ ] **Review generated migration file** - Don't blindly trust autogenerated code
- [ ] **Verify migration name** is descriptive
- [ ] **Check dependencies** are correct
- [ ] **Validate operations** match your intent

## Common Scenario Checklists

### Adding a New Field

- [ ] **Choose appropriate field type**
- [ ] **Provide default value** or make nullable
  - [ ] If required field: provide default or use two-step migration
  - [ ] If nullable: consider if null vs blank distinction matters
- [ ] **Add help_text** for documentation
- [ ] **Consider indexes** - does this field need indexing?
- [ ] **Multi-tenant check**: Does this need tenant consideration?
- [ ] **Test migration forward and backward**

### Removing a Field

Two-step process recommended:

**Step 1: Make nullable and stop using**
- [ ] Make field `null=True, blank=True`
- [ ] Remove all code references to the field
- [ ] Deploy and wait for all servers to update
- [ ] Generate migration: `python manage.py makemigrations`
- [ ] Test migration
- [ ] Deploy

**Step 2: Remove field**
- [ ] Remove field from model
- [ ] Generate migration: `python manage.py makemigrations`
- [ ] Test migration (verify no errors)
- [ ] Deploy

### Renaming a Field

- [ ] **Update model** with new name
- [ ] **Generate migration**: Answer 'yes' to rename prompt
- [ ] **Verify RenameField** operation (not AddField + RemoveField)
- [ ] **Update all code references** to use new name
- [ ] **Search codebase** for old name: `grep -r "old_name" .`
- [ ] **Check queries, forms, serializers** for field references
- [ ] **Test thoroughly** - renames can break unexpected code

### Changing Field Type

Safe two-step process:

**Step 1: Add new field**
- [ ] Add new field with new type: `new_field_name`
- [ ] Make it nullable: `null=True`
- [ ] Generate migration
- [ ] Test migration

**Step 2: Data migration**
- [ ] Create empty migration: `python manage.py makemigrations --empty app_name`
- [ ] Add RunPython to transform data: `old_field` → `new_field_name`
- [ ] Test data migration
- [ ] Verify data integrity

**Step 3: Remove old field**
- [ ] Remove old field from model
- [ ] Generate migration
- [ ] Test migration

**Step 4: Rename new field**
- [ ] Rename `new_field_name` to original name
- [ ] Generate migration (RenameField)
- [ ] Test migration

### Adding an Index

Multi-tenant specific:

- [ ] **Put tenant first** in composite indexes: `['tenant', 'field1', 'field2']`
- [ ] **Add to Meta.indexes** in model
- [ ] **Consider DESC ordering**: Use `-field` for descending
- [ ] **Large tables?** Use `AddIndexConcurrently` (PostgreSQL)
  - [ ] Set `atomic = False` in migration
- [ ] **Generate migration**
- [ ] **Review index name** - Django auto-generates, may be long
- [ ] **Test migration** - indexes can take time on large tables

### Adding Unique Constraint

- [ ] **Include tenant field** in constraint: `['tenant', 'field1', 'field2']`
- [ ] **Verify existing data** doesn't violate constraint
  - [ ] Run query to find duplicates
  - [ ] Clean up duplicates if needed
- [ ] **Use UniqueConstraint** (not unique_together - it's deprecated)
- [ ] **Generate migration**
- [ ] **Test migration** - will fail if duplicates exist

### Adding Foreign Key

- [ ] **Choose on_delete behavior**:
  - [ ] `CASCADE` - Delete related objects (most common)
  - [ ] `PROTECT` - Prevent deletion if related objects exist
  - [ ] `SET_NULL` - Set to null (requires `null=True`)
  - [ ] `SET_DEFAULT` - Set to default value
- [ ] **Specify related_name** - Required for multiple FKs to same model
- [ ] **Multi-tenant check**: Does related model have same tenant?
- [ ] **Consider indexes** - Django auto-creates, but consider composite with tenant
- [ ] **Make nullable temporarily** for existing data
- [ ] **Generate migration**
- [ ] **Test with existing data**

## Multi-Tenant Specific Checklist

For any schema change in multi-tenant projects:

- [ ] **Tenant field present** - Does model inherit from TenantBaseModel?
- [ ] **Indexes tenant-first** - All composite indexes start with `['tenant', ...]`
- [ ] **Unique constraints include tenant** - `['tenant', 'field1', ...]`
- [ ] **Foreign keys validated** - Related objects share same tenant
- [ ] **Test tenant isolation** - Verify migration doesn't break isolation
- [ ] **Query optimization** - Will queries filter by tenant efficiently?

## Data Migration Checklist

When you need to transform data:

- [ ] **Create empty migration**: `python manage.py makemigrations --empty app_name --name descriptive_name`
- [ ] **Use RunPython** (not RunSQL unless necessary)
- [ ] **Use apps.get_model()** - Never import models directly
- [ ] **Provide reverse function** - Make migration reversible
- [ ] **Handle large datasets** - Use batching/iterator
- [ ] **Handle missing data** - Check for None/null values
- [ ] **Add error handling** - Catch and log exceptions
- [ ] **Add progress logging** - For long-running migrations
- [ ] **Test on production-like data** - Sample size and complexity
- [ ] **Verify data integrity** after migration

## Testing Checklist

Before deploying any migration:

- [ ] **Test forward migration**: `python manage.py migrate`
- [ ] **Verify database state** matches expectations
- [ ] **Test backward migration**: `python manage.py migrate app_name previous_migration`
- [ ] **Test forward again** - Ensure reversibility works
- [ ] **Test with existing data** - Not just empty database
- [ ] **Test multi-tenant isolation** - If applicable
- [ ] **Run application tests** - Ensure code works with changes
- [ ] **Test on staging** with production-like data
- [ ] **Check migration performance** - How long does it take?
- [ ] **Verify no N+1 queries** in data migrations

## Pre-Deployment Checklist

Before deploying to production:

- [ ] **Review migration SQL**: `python manage.py sqlmigrate app_name migration_number`
- [ ] **Estimate downtime** - Will migration lock tables?
- [ ] **Plan deployment window** - Low-traffic time if possible
- [ ] **Prepare rollback plan** - Know how to reverse changes
- [ ] **Backup database** - Always have recent backup
- [ ] **Notify team** - Communicate deployment plan
- [ ] **Monitor resources** - Check disk space, memory, CPU
- [ ] **Test migration speed** on production-sized staging data
- [ ] **Review blocking operations**:
  - [ ] Adding non-nullable field without default
  - [ ] Adding unique constraint with duplicates
  - [ ] Large data transformations
  - [ ] Adding indexes without CONCURRENTLY (PostgreSQL)

## Deployment Checklist

During deployment:

- [ ] **Apply migration**: `python manage.py migrate`
- [ ] **Monitor progress** - Watch logs for errors
- [ ] **Check for locks** - Are queries being blocked?
- [ ] **Verify completion** - Check migration status
- [ ] **Spot-check data** - Verify transformations worked
- [ ] **Run smoke tests** - Verify application works
- [ ] **Monitor errors** - Watch error logs and monitoring
- [ ] **Check performance** - Verify no performance degradation

## Post-Deployment Checklist

After successful deployment:

- [ ] **Verify migration applied**: `python manage.py showmigrations`
- [ ] **Check data integrity**:
  - [ ] Row counts match expectations
  - [ ] Spot-check transformed data
  - [ ] Verify constraints working
  - [ ] Check indexes created
- [ ] **Monitor application performance**
- [ ] **Monitor error rates**
- [ ] **Check for unexpected issues**
- [ ] **Document any issues** encountered
- [ ] **Update team** - Migration complete

## Rollback Checklist

If migration causes issues:

- [ ] **Identify the problem** - What went wrong?
- [ ] **Can it be fixed forward?** - Sometimes easier than rollback
- [ ] **Stop deployment** - If still in progress
- [ ] **Revert code changes** first
- [ ] **Then rollback migration**: `python manage.py migrate app_name previous_migration`
- [ ] **Verify database state** after rollback
- [ ] **Check data integrity** - Did rollback work correctly?
- [ ] **Document what happened** - For post-mortem
- [ ] **Fix the migration** before retrying
- [ ] **Test fix thoroughly** before redeploying

## Common Pitfalls to Avoid

- [ ] ❌ **Don't edit applied migrations** - Create new one instead
- [ ] ❌ **Don't delete migration files** - Creates history gaps
- [ ] ❌ **Don't import models in migrations** - Use apps.get_model()
- [ ] ❌ **Don't skip testing reversibility**
- [ ] ❌ **Don't forget tenant in indexes** - Multi-tenant projects
- [ ] ❌ **Don't assume atomicity** - MySQL/SQLite don't support atomic DDL
- [ ] ❌ **Don't run --fake without understanding** - Can corrupt migration state
- [ ] ❌ **Don't deploy without testing on staging**
- [ ] ❌ **Don't forget to review auto-generated migrations**
- [ ] ❌ **Don't create migrations for temporary changes**

## Quick Command Reference

```bash
# Create migration
python manage.py makemigrations [app_name]

# Dry run (see what would be created)
python manage.py makemigrations --dry-run --verbosity 3

# Create empty migration for data migration
python manage.py makemigrations --empty app_name --name migration_name

# Show migration status
python manage.py showmigrations

# Show migration plan
python manage.py showmigrations --plan

# Show SQL for migration
python manage.py sqlmigrate app_name migration_number

# Apply migrations
python manage.py migrate

# Apply specific migration
python manage.py migrate app_name migration_number

# Rollback to previous migration
python manage.py migrate app_name previous_migration_number

# Check for conflicts
python manage.py makemigrations --merge

# Squash migrations
python manage.py squashmigrations app_name start_migration end_migration

# Fake migration (mark as applied without running)
python manage.py migrate --fake app_name migration_number
```

## Notes

- **Always review** auto-generated migrations before applying
- **Test thoroughly** in development and staging
- **Plan for failure** - have rollback strategy ready
- **Monitor production** deployment closely
- **Document decisions** for future reference
- **Communicate with team** about migration plans

## For More Information

See `references/migration-patterns.md` for detailed examples and patterns.
