# Observability & Monitoring Examples

This document provides comprehensive, production-ready examples for implementing monitoring and observability using Prometheus, Grafana, and related tools.

## Table of Contents

1. [Prometheus Configuration Examples](#prometheus-configuration-examples)
2. [PromQL Query Examples](#promql-query-examples)
3. [Alert Rule Examples](#alert-rule-examples)
4. [Grafana Dashboard Examples](#grafana-dashboard-examples)
5. [Custom Exporter Examples](#custom-exporter-examples)
6. [Recording Rule Examples](#recording-rule-examples)
7. [Service Discovery Examples](#service-discovery-examples)
8. [SLO Monitoring Examples](#slo-monitoring-examples)
9. [Multi-Cluster Federation Examples](#multi-cluster-federation-examples)
10. [Advanced Patterns](#advanced-patterns)

---

## Prometheus Configuration Examples

### Example 1: Production Prometheus Configuration

Complete production-ready Prometheus configuration with multiple scrape targets and best practices.

```yaml
# prometheus.yml
global:
  scrape_interval: 15s
  scrape_timeout: 10s
  evaluation_interval: 15s
  external_labels:
    cluster: 'production-us-west-2'
    environment: 'production'
    region: 'us-west-2'

# Alertmanager configuration
alerting:
  alertmanagers:
    - static_configs:
        - targets:
            - 'alertmanager-1:9093'
            - 'alertmanager-2:9093'
      timeout: 10s
      api_version: v2

# Rule files
rule_files:
  - '/etc/prometheus/rules/*.yml'
  - '/etc/prometheus/alerts/*.yml'

# Storage configuration
storage:
  tsdb:
    path: /prometheus/data
    retention.time: 30d
    retention.size: 100GB

# Remote write for long-term storage
remote_write:
  - url: "https://thanos-receive.example.com/api/v1/receive"
    queue_config:
      capacity: 10000
      max_shards: 50
      min_shards: 1
      max_samples_per_send: 5000
      batch_send_deadline: 5s
      min_backoff: 30ms
      max_backoff: 100ms
    write_relabel_configs:
      # Keep only production metrics
      - source_labels: [environment]
        regex: 'production'
        action: keep
      # Drop verbose debug metrics
      - source_labels: [__name__]
        regex: 'debug_.*'
        action: drop

# Scrape configurations
scrape_configs:
  # Prometheus self-monitoring
  - job_name: 'prometheus'
    honor_labels: true
    static_configs:
      - targets:
          - 'localhost:9090'
        labels:
          service: 'prometheus'

  # Node exporter for system metrics
  - job_name: 'node'
    static_configs:
      - targets:
          - 'node-1.prod.example.com:9100'
          - 'node-2.prod.example.com:9100'
          - 'node-3.prod.example.com:9100'
        labels:
          datacenter: 'dc1'
          tier: 'infrastructure'
    relabel_configs:
      # Extract hostname from FQDN
      - source_labels: [__address__]
        regex: '([^.]+)\..*'
        target_label: instance
        replacement: '${1}'
      # Add custom labels
      - target_label: job_type
        replacement: 'system_metrics'

  # API service
  - job_name: 'api'
    static_configs:
      - targets:
          - 'api-1.prod.example.com:8080'
          - 'api-2.prod.example.com:8080'
          - 'api-3.prod.example.com:8080'
        labels:
          environment: 'production'
          tier: 'backend'
          service: 'api'
          version: 'v2.1.0'
    metric_relabel_configs:
      # Drop high-cardinality metrics
      - source_labels: [__name__]
        regex: 'http_request_duration_microseconds_.*'
        action: drop
      # Normalize status codes to classes
      - source_labels: [status]
        regex: '([0-9])[0-9]{2}'
        target_label: status_class
        replacement: '${1}xx'

  # PostgreSQL exporter
  - job_name: 'postgres'
    static_configs:
      - targets:
          - 'postgres-exporter-1:9187'
          - 'postgres-exporter-2:9187'
        labels:
          database_type: 'postgresql'
          tier: 'database'

  # Redis exporter
  - job_name: 'redis'
    static_configs:
      - targets:
          - 'redis-exporter:9121'
        labels:
          cache_type: 'redis'
          tier: 'cache'

  # Blackbox exporter for endpoint probing
  - job_name: 'blackbox-http'
    metrics_path: /probe
    params:
      module: [http_2xx]
    static_configs:
      - targets:
          - https://api.example.com/health
          - https://www.example.com
          - https://admin.example.com
        labels:
          probe_type: 'http'
    relabel_configs:
      - source_labels: [__address__]
        target_label: __param_target
      - source_labels: [__param_target]
        target_label: instance
      - target_label: __address__
        replacement: blackbox-exporter:9115

  # Custom application metrics
  - job_name: 'webapp'
    static_configs:
      - targets:
          - 'webapp-1:9090'
          - 'webapp-2:9090'
          - 'webapp-3:9090'
        labels:
          app: 'webapp'
          tier: 'frontend'
    scrape_interval: 10s  # Override global for critical service
```

### Example 2: Kubernetes Service Discovery

Complete Kubernetes service discovery configuration for pod, service, and node monitoring.

```yaml
scrape_configs:
  # Kubernetes API server
  - job_name: 'kubernetes-apiservers'
    kubernetes_sd_configs:
      - role: endpoints
    scheme: https
    tls_config:
      ca_file: /var/run/secrets/kubernetes.io/serviceaccount/ca.crt
    bearer_token_file: /var/run/secrets/kubernetes.io/serviceaccount/token
    relabel_configs:
      - source_labels: [__meta_kubernetes_namespace, __meta_kubernetes_service_name, __meta_kubernetes_endpoint_port_name]
        action: keep
        regex: default;kubernetes;https

  # Kubernetes nodes (kubelet)
  - job_name: 'kubernetes-nodes'
    kubernetes_sd_configs:
      - role: node
    scheme: https
    tls_config:
      ca_file: /var/run/secrets/kubernetes.io/serviceaccount/ca.crt
    bearer_token_file: /var/run/secrets/kubernetes.io/serviceaccount/token
    relabel_configs:
      - action: labelmap
        regex: __meta_kubernetes_node_label_(.+)

  # Kubernetes pods
  - job_name: 'kubernetes-pods'
    kubernetes_sd_configs:
      - role: pod
    relabel_configs:
      # Only scrape pods with prometheus.io/scrape: "true"
      - source_labels: [__meta_kubernetes_pod_annotation_prometheus_io_scrape]
        action: keep
        regex: true
      # Use custom scrape port if defined
      - source_labels: [__meta_kubernetes_pod_annotation_prometheus_io_port, __meta_kubernetes_pod_ip]
        action: replace
        regex: (\d+);([^:]+)(?::\d+)?
        replacement: $2:$1
        target_label: __address__
      # Use custom metrics path if defined
      - source_labels: [__meta_kubernetes_pod_annotation_prometheus_io_path]
        action: replace
        target_label: __metrics_path__
        regex: (.+)
      # Add namespace as label
      - source_labels: [__meta_kubernetes_namespace]
        action: replace
        target_label: kubernetes_namespace
      # Add pod name as label
      - source_labels: [__meta_kubernetes_pod_name]
        action: replace
        target_label: kubernetes_pod_name
      # Add pod labels as metric labels
      - action: labelmap
        regex: __meta_kubernetes_pod_label_(.+)
      # Add pod phase as label
      - source_labels: [__meta_kubernetes_pod_phase]
        action: replace
        target_label: kubernetes_pod_phase

  # Kubernetes services
  - job_name: 'kubernetes-services'
    kubernetes_sd_configs:
      - role: service
    metrics_path: /probe
    params:
      module: [http_2xx]
    relabel_configs:
      # Only probe services with prometheus.io/probe: "true"
      - source_labels: [__meta_kubernetes_service_annotation_prometheus_io_probe]
        action: keep
        regex: true
      - source_labels: [__address__]
        target_label: __param_target
      - target_label: __address__
        replacement: blackbox-exporter:9115
      - source_labels: [__param_target]
        target_label: instance
      - action: labelmap
        regex: __meta_kubernetes_service_label_(.+)
      - source_labels: [__meta_kubernetes_namespace]
        target_label: kubernetes_namespace
      - source_labels: [__meta_kubernetes_service_name]
        target_label: kubernetes_service_name

  # Kubernetes ingresses
  - job_name: 'kubernetes-ingresses'
    kubernetes_sd_configs:
      - role: ingress
    metrics_path: /probe
    params:
      module: [http_2xx]
    relabel_configs:
      - source_labels: [__meta_kubernetes_ingress_scheme, __address__, __meta_kubernetes_ingress_path]
        regex: (.+);(.+);(.+)
        replacement: ${1}://${2}${3}
        target_label: __param_target
      - target_label: __address__
        replacement: blackbox-exporter:9115
      - source_labels: [__param_target]
        target_label: instance
      - action: labelmap
        regex: __meta_kubernetes_ingress_label_(.+)
      - source_labels: [__meta_kubernetes_namespace]
        target_label: kubernetes_namespace
      - source_labels: [__meta_kubernetes_ingress_name]
        target_label: kubernetes_ingress_name
```

### Example 3: AWS EC2 Service Discovery

Automatically discover and monitor EC2 instances using AWS service discovery.

```yaml
scrape_configs:
  - job_name: 'ec2-nodes'
    ec2_sd_configs:
      - region: us-west-2
        port: 9100
        filters:
          - name: tag:Environment
            values:
              - production
          - name: tag:Monitoring
            values:
              - enabled
          - name: instance-state-name
            values:
              - running
    relabel_configs:
      # Use private IP
      - source_labels: [__meta_ec2_private_ip]
        target_label: __address__
        replacement: '${1}:9100'
      # Add instance ID
      - source_labels: [__meta_ec2_instance_id]
        target_label: instance_id
      # Add instance type
      - source_labels: [__meta_ec2_instance_type]
        target_label: instance_type
      # Add availability zone
      - source_labels: [__meta_ec2_availability_zone]
        target_label: availability_zone
      # Add EC2 tags as labels
      - source_labels: [__meta_ec2_tag_Name]
        target_label: instance_name
      - source_labels: [__meta_ec2_tag_Environment]
        target_label: environment
      - source_labels: [__meta_ec2_tag_Team]
        target_label: team
      - source_labels: [__meta_ec2_tag_Service]
        target_label: service
```

---

## PromQL Query Examples

### Example 4: Request Rate and Throughput

```promql
# Requests per second (RPS) - instant rate
rate(http_requests_total[5m])

# Total RPS across all instances
sum(rate(http_requests_total[5m]))

# RPS grouped by service
sum(rate(http_requests_total[5m])) by (service)

# RPS grouped by method and endpoint
sum(rate(http_requests_total[5m])) by (method, endpoint)

# Top 5 endpoints by request volume
topk(5, sum(rate(http_requests_total[5m])) by (endpoint))

# Total requests in the last hour
sum(increase(http_requests_total[1h]))

# Average RPS over the last 24 hours
avg_over_time(sum(rate(http_requests_total[5m]))[24h:5m])

# Predict RPS in 4 hours based on 1-hour trend
predict_linear(sum(rate(http_requests_total[5m]))[1h:], 4 * 3600)
```

### Example 5: Error Rate Analysis

```promql
# Error rate (5xx errors) as percentage
sum(rate(http_requests_total{status=~"5.."}[5m]))
/
sum(rate(http_requests_total[5m])) * 100

# Error rate per service
sum(rate(http_requests_total{status=~"5.."}[5m])) by (service)
/
sum(rate(http_requests_total[5m])) by (service) * 100

# Success rate (2xx responses)
sum(rate(http_requests_total{status=~"2.."}[5m]))
/
sum(rate(http_requests_total[5m])) * 100

# 4xx client error rate
sum(rate(http_requests_total{status=~"4.."}[5m]))
/
sum(rate(http_requests_total[5m])) * 100

# Errors grouped by status code
sum(rate(http_requests_total{status=~"5.."}[5m])) by (status)

# Services with error rate > 1%
(
  sum(rate(http_requests_total{status=~"5.."}[5m])) by (service)
  /
  sum(rate(http_requests_total[5m])) by (service)
) > 0.01

# Error spike detection (current vs 1 hour ago)
sum(rate(http_requests_total{status=~"5.."}[5m]))
/
sum(rate(http_requests_total{status=~"5.."}[5m] offset 1h))
> 2
```

### Example 6: Latency Percentiles and Histograms

```promql
# P50 (median) latency
histogram_quantile(0.50,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (le)
)

# P95 latency
histogram_quantile(0.95,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (le)
)

# P99 latency
histogram_quantile(0.99,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (le)
)

# P99 latency per service
histogram_quantile(0.99,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (service, le)
)

# P99 latency per endpoint
histogram_quantile(0.99,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (endpoint, le)
)

# Average latency (from histogram)
sum(rate(http_request_duration_seconds_sum[5m]))
/
sum(rate(http_request_duration_seconds_count[5m]))

# Latency standard deviation
stddev_over_time(
  (
    sum(rate(http_request_duration_seconds_sum[5m]))
    /
    sum(rate(http_request_duration_seconds_count[5m]))
  )[10m:1m]
)

# Requests exceeding 1 second latency
sum(rate(http_request_duration_seconds_bucket{le="1.0"}[5m]))
/
sum(rate(http_request_duration_seconds_bucket{le="+Inf"}[5m]))
```

### Example 7: Resource Utilization

```promql
# CPU usage percentage
100 - (avg by (instance) (irate(node_cpu_seconds_total{mode="idle"}[5m])) * 100)

# CPU usage by core
sum by (cpu) (irate(node_cpu_seconds_total{mode!="idle"}[5m])) * 100

# Memory usage percentage
(1 - (node_memory_MemAvailable_bytes / node_memory_MemTotal_bytes)) * 100

# Memory usage in GB
(node_memory_MemTotal_bytes - node_memory_MemAvailable_bytes) / 1024 / 1024 / 1024

# Disk usage percentage
(
  node_filesystem_size_bytes{mountpoint="/"} -
  node_filesystem_avail_bytes{mountpoint="/"}
)
/
node_filesystem_size_bytes{mountpoint="/"} * 100

# Disk I/O utilization
rate(node_disk_io_time_seconds_total[5m]) * 100

# Network receive throughput (MB/s)
rate(node_network_receive_bytes_total[5m]) / 1024 / 1024

# Network transmit throughput (MB/s)
rate(node_network_transmit_bytes_total[5m]) / 1024 / 1024

# Load average per CPU
node_load1 / count(node_cpu_seconds_total{mode="idle"}) by (instance)

# Container memory usage
sum(container_memory_usage_bytes{container!=""}) by (pod, namespace)

# Container CPU usage
sum(rate(container_cpu_usage_seconds_total{container!=""}[5m])) by (pod, namespace)
```

---

## Alert Rule Examples

### Example 8: Comprehensive Application Alerts

```yaml
# alerts/application_alerts.yml
groups:
  - name: application_health
    interval: 30s
    rules:
      # Service down
      - alert: ServiceDown
        expr: up{job="api"} == 0
        for: 2m
        labels:
          severity: critical
          team: backend
          category: availability
        annotations:
          summary: "Service {{ $labels.instance }} is down"
          description: "{{ $labels.job }} on {{ $labels.instance }} has been unreachable for more than 2 minutes."
          runbook_url: "https://runbooks.example.com/ServiceDown"
          dashboard_url: "https://grafana.example.com/d/service-health"

      # High error rate
      - alert: HighErrorRate
        expr: |
          (
            sum(rate(http_requests_total{status=~"5.."}[5m])) by (service)
            /
            sum(rate(http_requests_total[5m])) by (service)
          ) > 0.05
        for: 5m
        labels:
          severity: critical
          team: backend
          category: errors
        annotations:
          summary: "High error rate on {{ $labels.service }}"
          description: "Error rate is {{ $value | humanizePercentage }} on {{ $labels.service }} (threshold: 5%)"
          runbook_url: "https://runbooks.example.com/HighErrorRate"

      # Elevated error rate (warning)
      - alert: ElevatedErrorRate
        expr: |
          (
            sum(rate(http_requests_total{status=~"5.."}[5m])) by (service)
            /
            sum(rate(http_requests_total[5m])) by (service)
          ) > 0.01
        for: 10m
        labels:
          severity: warning
          team: backend
          category: errors
        annotations:
          summary: "Elevated error rate on {{ $labels.service }}"
          description: "Error rate is {{ $value | humanizePercentage }} on {{ $labels.service }} (threshold: 1%)"

      # High latency (P99)
      - alert: HighLatencyP99
        expr: |
          histogram_quantile(0.99,
            sum(rate(http_request_duration_seconds_bucket[5m])) by (service, le)
          ) > 1.0
        for: 10m
        labels:
          severity: warning
          team: backend
          category: performance
        annotations:
          summary: "High P99 latency on {{ $labels.service }}"
          description: "P99 latency is {{ $value }}s on {{ $labels.service }} (threshold: 1s)"
          runbook_url: "https://runbooks.example.com/HighLatency"

      # Critical latency (P99)
      - alert: CriticalLatencyP99
        expr: |
          histogram_quantile(0.99,
            sum(rate(http_request_duration_seconds_bucket[5m])) by (service, le)
          ) > 5.0
        for: 5m
        labels:
          severity: critical
          team: backend
          category: performance
        annotations:
          summary: "Critical P99 latency on {{ $labels.service }}"
          description: "P99 latency is {{ $value }}s on {{ $labels.service }} (threshold: 5s)"

      # Low request volume
      - alert: LowRequestVolume
        expr: |
          sum(rate(http_requests_total[5m])) by (service) < 10
        for: 15m
        labels:
          severity: warning
          team: backend
          category: traffic
        annotations:
          summary: "Low request volume on {{ $labels.service }}"
          description: "Request rate is {{ $value | humanize }} req/s on {{ $labels.service }} (expected > 10 req/s)"

      # Traffic spike
      - alert: TrafficSpike
        expr: |
          sum(rate(http_requests_total[5m])) by (service)
          /
          avg_over_time(sum(rate(http_requests_total[5m])) by (service)[1h:5m])
          > 3
        for: 5m
        labels:
          severity: warning
          team: backend
          category: traffic
        annotations:
          summary: "Traffic spike detected on {{ $labels.service }}"
          description: "Current traffic is {{ $value }}x normal levels on {{ $labels.service }}"
```

### Example 9: Infrastructure Alerts

```yaml
# alerts/infrastructure_alerts.yml
groups:
  - name: infrastructure_health
    interval: 30s
    rules:
      # High CPU usage
      - alert: HighCPUUsage
        expr: |
          100 - (avg by (instance) (irate(node_cpu_seconds_total{mode="idle"}[5m])) * 100) > 80
        for: 15m
        labels:
          severity: warning
          team: sre
          category: resources
        annotations:
          summary: "High CPU usage on {{ $labels.instance }}"
          description: "CPU usage is {{ $value | humanize }}% on {{ $labels.instance }} (threshold: 80%)"
          runbook_url: "https://runbooks.example.com/HighCPU"

      # Critical CPU usage
      - alert: CriticalCPUUsage
        expr: |
          100 - (avg by (instance) (irate(node_cpu_seconds_total{mode="idle"}[5m])) * 100) > 95
        for: 5m
        labels:
          severity: critical
          team: sre
          category: resources
        annotations:
          summary: "Critical CPU usage on {{ $labels.instance }}"
          description: "CPU usage is {{ $value | humanize }}% on {{ $labels.instance }} (threshold: 95%)"

      # High memory usage
      - alert: HighMemoryUsage
        expr: |
          (1 - (node_memory_MemAvailable_bytes / node_memory_MemTotal_bytes)) * 100 > 85
        for: 10m
        labels:
          severity: warning
          team: sre
          category: resources
        annotations:
          summary: "High memory usage on {{ $labels.instance }}"
          description: "Memory usage is {{ $value | humanize }}% on {{ $labels.instance }} (threshold: 85%)"

      # Critical memory usage
      - alert: CriticalMemoryUsage
        expr: |
          (1 - (node_memory_MemAvailable_bytes / node_memory_MemTotal_bytes)) * 100 > 95
        for: 5m
        labels:
          severity: critical
          team: sre
          category: resources
        annotations:
          summary: "Critical memory usage on {{ $labels.instance }}"
          description: "Memory usage is {{ $value | humanize }}% on {{ $labels.instance }} (threshold: 95%)"

      # Low disk space
      - alert: LowDiskSpace
        expr: |
          (
            node_filesystem_avail_bytes{mountpoint="/", fstype!="rootfs"}
            /
            node_filesystem_size_bytes{mountpoint="/", fstype!="rootfs"}
          ) * 100 < 15
        for: 5m
        labels:
          severity: warning
          team: sre
          category: storage
        annotations:
          summary: "Low disk space on {{ $labels.instance }}"
          description: "Disk space available is {{ $value | humanize }}% on {{ $labels.instance }}:{{ $labels.mountpoint }}"

      # Critical disk space
      - alert: CriticalDiskSpace
        expr: |
          (
            node_filesystem_avail_bytes{mountpoint="/", fstype!="rootfs"}
            /
            node_filesystem_size_bytes{mountpoint="/", fstype!="rootfs"}
          ) * 100 < 5
        for: 2m
        labels:
          severity: critical
          team: sre
          category: storage
        annotations:
          summary: "Critical disk space on {{ $labels.instance }}"
          description: "Disk space available is {{ $value | humanize }}% on {{ $labels.instance }}:{{ $labels.mountpoint }}"

      # Disk will fill in 4 hours
      - alert: DiskWillFillSoon
        expr: |
          predict_linear(node_filesystem_avail_bytes{mountpoint="/"}[1h], 4 * 3600) < 0
        for: 5m
        labels:
          severity: warning
          team: sre
          category: capacity
        annotations:
          summary: "Disk predicted to fill in 4 hours on {{ $labels.instance }}"
          description: "Based on current trend, disk will be full in approximately 4 hours on {{ $labels.instance }}"

      # High disk I/O
      - alert: HighDiskIO
        expr: |
          rate(node_disk_io_time_seconds_total[5m]) * 100 > 80
        for: 10m
        labels:
          severity: warning
          team: sre
          category: performance
        annotations:
          summary: "High disk I/O on {{ $labels.instance }}"
          description: "Disk I/O utilization is {{ $value | humanize }}% on {{ $labels.instance }}:{{ $labels.device }}"

      # Network errors
      - alert: NetworkErrors
        expr: |
          rate(node_network_receive_errs_total[5m]) + rate(node_network_transmit_errs_total[5m]) > 10
        for: 5m
        labels:
          severity: warning
          team: sre
          category: network
        annotations:
          summary: "Network errors on {{ $labels.instance }}"
          description: "Network error rate is {{ $value | humanize }} errors/s on {{ $labels.instance }}:{{ $labels.device }}"

      # High load average
      - alert: HighLoadAverage
        expr: |
          node_load15 / count(node_cpu_seconds_total{mode="idle"}) by (instance) > 2
        for: 15m
        labels:
          severity: warning
          team: sre
          category: performance
        annotations:
          summary: "High load average on {{ $labels.instance }}"
          description: "Load average per CPU is {{ $value | humanize }} on {{ $labels.instance }}"
```

### Example 10: Database Alerts

```yaml
# alerts/database_alerts.yml
groups:
  - name: database_health
    interval: 30s
    rules:
      # PostgreSQL down
      - alert: PostgreSQLDown
        expr: pg_up == 0
        for: 1m
        labels:
          severity: critical
          team: dba
          category: availability
        annotations:
          summary: "PostgreSQL database is down on {{ $labels.instance }}"
          description: "PostgreSQL exporter cannot connect to database on {{ $labels.instance }}"
          runbook_url: "https://runbooks.example.com/PostgreSQLDown"

      # Too many connections
      - alert: PostgreSQLTooManyConnections
        expr: |
          sum(pg_stat_activity_count) by (instance)
          /
          pg_settings_max_connections * 100 > 80
        for: 5m
        labels:
          severity: warning
          team: dba
          category: resources
        annotations:
          summary: "PostgreSQL connection pool near capacity on {{ $labels.instance }}"
          description: "Connection usage is {{ $value | humanize }}% on {{ $labels.instance }}"

      # Replication lag
      - alert: PostgreSQLReplicationLag
        expr: pg_replication_lag > 30
        for: 5m
        labels:
          severity: warning
          team: dba
          category: replication
        annotations:
          summary: "PostgreSQL replication lag on {{ $labels.instance }}"
          description: "Replication lag is {{ $value | humanize }} seconds on {{ $labels.instance }}"

      # High transaction rate
      - alert: PostgreSQLHighTransactionRate
        expr: |
          rate(pg_stat_database_xact_commit[5m]) + rate(pg_stat_database_xact_rollback[5m]) > 10000
        for: 10m
        labels:
          severity: warning
          team: dba
          category: performance
        annotations:
          summary: "High transaction rate on {{ $labels.instance }}"
          description: "Transaction rate is {{ $value | humanize }} tx/s on {{ $labels.instance }}"

      # Deadlocks detected
      - alert: PostgreSQLDeadlocks
        expr: rate(pg_stat_database_deadlocks[5m]) > 0
        for: 5m
        labels:
          severity: warning
          team: dba
          category: locks
        annotations:
          summary: "Deadlocks detected on {{ $labels.instance }}"
          description: "Deadlock rate is {{ $value | humanize }}/s on {{ $labels.instance }}"

      # Slow queries
      - alert: PostgreSQLSlowQueries
        expr: |
          pg_stat_activity_max_tx_duration > 300
        for: 5m
        labels:
          severity: warning
          team: dba
          category: performance
        annotations:
          summary: "Slow queries detected on {{ $labels.instance }}"
          description: "Long-running query detected ({{ $value | humanize }}s) on {{ $labels.instance }}"

      # Low cache hit ratio
      - alert: PostgreSQLLowCacheHitRatio
        expr: |
          (
            sum(pg_stat_database_blks_hit) by (instance)
            /
            (sum(pg_stat_database_blks_hit) by (instance) + sum(pg_stat_database_blks_read) by (instance))
          ) < 0.90
        for: 10m
        labels:
          severity: warning
          team: dba
          category: performance
        annotations:
          summary: "Low cache hit ratio on {{ $labels.instance }}"
          description: "Cache hit ratio is {{ $value | humanizePercentage }} on {{ $labels.instance }} (expected > 90%)"
```

---

## Grafana Dashboard Examples

### Example 11: Complete RED Method Dashboard

```json
{
  "dashboard": {
    "title": "RED Method - Service Performance",
    "tags": ["red", "performance", "slo"],
    "timezone": "browser",
    "refresh": "30s",
    "time": {
      "from": "now-1h",
      "to": "now"
    },
    "templating": {
      "list": [
        {
          "name": "cluster",
          "type": "query",
          "datasource": "Prometheus",
          "query": "label_values(up, cluster)",
          "refresh": 2,
          "multi": false,
          "includeAll": false
        },
        {
          "name": "service",
          "type": "query",
          "datasource": "Prometheus",
          "query": "label_values(up{cluster=\"$cluster\"}, service)",
          "refresh": 2,
          "multi": true,
          "includeAll": true,
          "allValue": ".*"
        },
        {
          "name": "percentile",
          "type": "custom",
          "query": "0.50,0.95,0.99",
          "multi": false,
          "current": {
            "value": "0.99",
            "text": "p99"
          }
        }
      ]
    },
    "panels": [
      {
        "id": 1,
        "title": "Request Rate (req/s)",
        "type": "graph",
        "gridPos": {"h": 8, "w": 12, "x": 0, "y": 0},
        "targets": [
          {
            "expr": "sum(rate(http_requests_total{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (service)",
            "legendFormat": "{{ service }}",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "reqps",
            "decimals": 2
          }
        },
        "options": {
          "legend": {
            "displayMode": "table",
            "placement": "right",
            "calcs": ["lastNotNull", "mean", "max"]
          },
          "tooltip": {
            "mode": "multi"
          }
        }
      },
      {
        "id": 2,
        "title": "Error Rate (%)",
        "type": "graph",
        "gridPos": {"h": 8, "w": 12, "x": 12, "y": 0},
        "targets": [
          {
            "expr": "sum(rate(http_requests_total{cluster=\"$cluster\",service=~\"$service\",status=~\"5..\"}[$__rate_interval])) by (service) / sum(rate(http_requests_total{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (service) * 100",
            "legendFormat": "{{ service }}",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "percent",
            "decimals": 2,
            "thresholds": {
              "mode": "absolute",
              "steps": [
                {"value": null, "color": "green"},
                {"value": 1, "color": "yellow"},
                {"value": 5, "color": "red"}
              ]
            }
          }
        },
        "alert": {
          "conditions": [
            {
              "evaluator": {"params": [5], "type": "gt"},
              "operator": {"type": "and"},
              "query": {"params": ["A", "5m", "now"]},
              "reducer": {"params": [], "type": "avg"},
              "type": "query"
            }
          ],
          "name": "High Error Rate Alert"
        }
      },
      {
        "id": 3,
        "title": "Latency (Duration)",
        "type": "graph",
        "gridPos": {"h": 8, "w": 24, "x": 0, "y": 8},
        "targets": [
          {
            "expr": "histogram_quantile(0.99, sum(rate(http_request_duration_seconds_bucket{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (service, le))",
            "legendFormat": "{{ service }} p99",
            "refId": "A"
          },
          {
            "expr": "histogram_quantile(0.95, sum(rate(http_request_duration_seconds_bucket{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (service, le))",
            "legendFormat": "{{ service }} p95",
            "refId": "B"
          },
          {
            "expr": "histogram_quantile(0.50, sum(rate(http_request_duration_seconds_bucket{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (service, le))",
            "legendFormat": "{{ service }} p50",
            "refId": "C"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "s",
            "decimals": 3
          }
        }
      },
      {
        "id": 4,
        "title": "Request Volume by Status",
        "type": "timeseries",
        "gridPos": {"h": 8, "w": 12, "x": 0, "y": 16},
        "targets": [
          {
            "expr": "sum(rate(http_requests_total{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (status)",
            "legendFormat": "{{ status }}",
            "refId": "A"
          }
        ],
        "options": {
          "stacking": {
            "mode": "normal"
          }
        }
      },
      {
        "id": 5,
        "title": "Top Endpoints by Request Rate",
        "type": "bargauge",
        "gridPos": {"h": 8, "w": 12, "x": 12, "y": 16},
        "targets": [
          {
            "expr": "topk(10, sum(rate(http_requests_total{cluster=\"$cluster\",service=~\"$service\"}[$__rate_interval])) by (endpoint))",
            "legendFormat": "{{ endpoint }}",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "reqps"
          }
        },
        "options": {
          "orientation": "horizontal",
          "displayMode": "gradient"
        }
      }
    ]
  }
}
```

### Example 12: SLO Dashboard

```json
{
  "dashboard": {
    "title": "SLO Tracking Dashboard",
    "tags": ["slo", "sli", "reliability"],
    "panels": [
      {
        "id": 1,
        "title": "Availability SLI (Target: 99.9%)",
        "type": "stat",
        "gridPos": {"h": 6, "w": 6, "x": 0, "y": 0},
        "targets": [
          {
            "expr": "sum(rate(http_requests_total{status=~\"2..\"}[30d])) / sum(rate(http_requests_total[30d])) * 100",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "percent",
            "decimals": 3,
            "thresholds": {
              "mode": "absolute",
              "steps": [
                {"value": null, "color": "red"},
                {"value": 99.9, "color": "green"}
              ]
            }
          }
        },
        "options": {
          "graphMode": "area",
          "colorMode": "background"
        }
      },
      {
        "id": 2,
        "title": "Error Budget Remaining (30 days)",
        "type": "gauge",
        "gridPos": {"h": 6, "w": 6, "x": 6, "y": 0},
        "targets": [
          {
            "expr": "(1 - ((1 - (sum(rate(http_requests_total{status=~\"2..\"}[30d])) / sum(rate(http_requests_total[30d])))) / (1 - 0.999))) * 100",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "percent",
            "min": 0,
            "max": 100,
            "thresholds": {
              "mode": "absolute",
              "steps": [
                {"value": null, "color": "red"},
                {"value": 20, "color": "orange"},
                {"value": 50, "color": "yellow"},
                {"value": 80, "color": "green"}
              ]
            }
          }
        }
      },
      {
        "id": 3,
        "title": "Latency SLI - % Requests < 500ms",
        "type": "stat",
        "gridPos": {"h": 6, "w": 6, "x": 12, "y": 0},
        "targets": [
          {
            "expr": "sum(rate(http_request_duration_seconds_bucket{le=\"0.5\"}[30d])) / sum(rate(http_request_duration_seconds_bucket{le=\"+Inf\"}[30d])) * 100",
            "refId": "A"
          }
        ],
        "fieldConfig": {
          "defaults": {
            "unit": "percent",
            "decimals": 2,
            "thresholds": {
              "mode": "absolute",
              "steps": [
                {"value": null, "color": "red"},
                {"value": 95, "color": "green"}
              ]
            }
          }
        }
      },
      {
        "id": 4,
        "title": "SLO Burn Rate (1h window)",
        "type": "graph",
        "gridPos": {"h": 8, "w": 24, "x": 0, "y": 6},
        "targets": [
          {
            "expr": "(sum(rate(http_requests_total{status=~\"5..\"}[1h])) / sum(rate(http_requests_total[1h]))) / (1 - 0.999)",
            "legendFormat": "1h burn rate",
            "refId": "A"
          },
          {
            "expr": "(sum(rate(http_requests_total{status=~\"5..\"}[6h])) / sum(rate(http_requests_total[6h]))) / (1 - 0.999)",
            "legendFormat": "6h burn rate",
            "refId": "B"
          }
        ],
        "options": {
          "legend": {
            "displayMode": "table",
            "placement": "bottom"
          }
        }
      }
    ]
  }
}
```

---

## Custom Exporter Examples

### Example 13: Advanced Python Exporter with Multiple Metrics

```python
#!/usr/bin/env python3
"""
Advanced application exporter for Prometheus
Exposes various application metrics including business metrics
"""

from prometheus_client import start_http_server, Counter, Gauge, Histogram, Summary, Enum, Info
from prometheus_client import CollectorRegistry, generate_latest, CONTENT_TYPE_LATEST
import time
import random
import psutil
import requests
from flask import Flask, Response
from threading import Thread

# Create custom registry
registry = CollectorRegistry()

# HTTP Request metrics
http_requests_total = Counter(
    'app_http_requests_total',
    'Total HTTP requests',
    ['method', 'endpoint', 'status', 'service'],
    registry=registry
)

http_request_duration_seconds = Histogram(
    'app_http_request_duration_seconds',
    'HTTP request latency in seconds',
    ['method', 'endpoint', 'service'],
    buckets=[0.005, 0.01, 0.025, 0.05, 0.075, 0.1, 0.25, 0.5, 0.75, 1.0, 2.5, 5.0, 7.5, 10.0],
    registry=registry
)

http_request_size_bytes = Summary(
    'app_http_request_size_bytes',
    'HTTP request size in bytes',
    ['method', 'endpoint'],
    registry=registry
)

http_response_size_bytes = Summary(
    'app_http_response_size_bytes',
    'HTTP response size in bytes',
    ['method', 'endpoint'],
    registry=registry
)

# Application state metrics
active_users = Gauge(
    'app_active_users',
    'Number of active users',
    registry=registry
)

active_sessions = Gauge(
    'app_active_sessions',
    'Number of active sessions',
    registry=registry
)

queue_size = Gauge(
    'app_queue_size',
    'Current queue size',
    ['queue_name', 'priority'],
    registry=registry
)

# Database metrics
database_connections = Gauge(
    'app_database_connections',
    'Number of database connections',
    ['pool', 'state'],
    registry=registry
)

database_query_duration_seconds = Histogram(
    'app_database_query_duration_seconds',
    'Database query duration',
    ['query_type', 'table'],
    buckets=[0.001, 0.005, 0.01, 0.05, 0.1, 0.5, 1.0, 5.0],
    registry=registry
)

# Cache metrics
cache_operations_total = Counter(
    'app_cache_operations_total',
    'Total cache operations',
    ['cache_name', 'operation'],
    registry=registry
)

cache_hit_ratio = Gauge(
    'app_cache_hit_ratio',
    'Cache hit ratio',
    ['cache_name'],
    registry=registry
)

# Business metrics
orders_total = Counter(
    'app_orders_total',
    'Total orders processed',
    ['status', 'payment_method'],
    registry=registry
)

revenue_total = Counter(
    'app_revenue_total',
    'Total revenue in USD',
    ['currency', 'payment_method'],
    registry=registry
)

# Feature flags
feature_flag = Enum(
    'app_feature_flag_state',
    'Feature flag states',
    ['feature_name'],
    states=['enabled', 'disabled', 'canary'],
    registry=registry
)

# Application info
app_info = Info(
    'app_version',
    'Application version information',
    registry=registry
)

# Set static info
app_info.info({
    'version': '2.1.0',
    'build': '20250118',
    'environment': 'production',
    'region': 'us-west-2'
})

# Worker threads metrics
worker_threads_active = Gauge(
    'app_worker_threads_active',
    'Number of active worker threads',
    ['worker_type'],
    registry=registry
)

worker_tasks_processed_total = Counter(
    'app_worker_tasks_processed_total',
    'Total tasks processed by workers',
    ['worker_type', 'status'],
    registry=registry
)

# Resource usage metrics
process_cpu_usage_percent = Gauge(
    'app_process_cpu_usage_percent',
    'Process CPU usage percentage',
    registry=registry
)

process_memory_bytes = Gauge(
    'app_process_memory_bytes',
    'Process memory usage in bytes',
    ['type'],
    registry=registry
)

def collect_system_metrics():
    """Collect system-level metrics"""
    process = psutil.Process()

    # CPU usage
    process_cpu_usage_percent.set(process.cpu_percent(interval=1))

    # Memory usage
    memory_info = process.memory_info()
    process_memory_bytes.labels(type='rss').set(memory_info.rss)
    process_memory_bytes.labels(type='vms').set(memory_info.vms)

def simulate_application_metrics():
    """Simulate application metrics"""
    methods = ['GET', 'POST', 'PUT', 'DELETE', 'PATCH']
    endpoints = ['/api/users', '/api/products', '/api/orders', '/api/auth', '/api/payments']
    statuses = ['200', '201', '400', '404', '500', '502']
    status_weights = [70, 10, 5, 3, 1, 1]  # Weighted distribution

    while True:
        # Simulate HTTP requests
        method = random.choice(methods)
        endpoint = random.choice(endpoints)
        status = random.choices(statuses, weights=status_weights)[0]

        http_requests_total.labels(
            method=method,
            endpoint=endpoint,
            status=status,
            service='api'
        ).inc()

        # Simulate latency
        latency = random.gauss(0.1, 0.05)  # Normal distribution
        if status.startswith('5'):
            latency *= 3  # Slower for errors

        http_request_duration_seconds.labels(
            method=method,
            endpoint=endpoint,
            service='api'
        ).observe(max(0.001, latency))

        # Request/response sizes
        http_request_size_bytes.labels(method=method, endpoint=endpoint).observe(
            random.randint(100, 5000)
        )
        http_response_size_bytes.labels(method=method, endpoint=endpoint).observe(
            random.randint(500, 50000)
        )

        # Active users and sessions
        active_users.set(random.randint(100, 1000))
        active_sessions.set(random.randint(150, 1500))

        # Queue metrics
        queue_size.labels(queue_name='jobs', priority='high').set(random.randint(0, 20))
        queue_size.labels(queue_name='jobs', priority='normal').set(random.randint(0, 100))
        queue_size.labels(queue_name='emails', priority='normal').set(random.randint(0, 50))

        # Database connections
        database_connections.labels(pool='main', state='active').set(random.randint(5, 30))
        database_connections.labels(pool='main', state='idle').set(random.randint(10, 50))
        database_connections.labels(pool='readonly', state='active').set(random.randint(2, 15))

        # Database query duration
        database_query_duration_seconds.labels(
            query_type='SELECT',
            table='users'
        ).observe(random.uniform(0.001, 0.1))

        # Cache operations
        if random.random() > 0.3:
            cache_operations_total.labels(cache_name='redis', operation='hit').inc()
        else:
            cache_operations_total.labels(cache_name='redis', operation='miss').inc()

        # Update cache hit ratio
        hits = cache_operations_total.labels(cache_name='redis', operation='hit')._value.get()
        total = (
            cache_operations_total.labels(cache_name='redis', operation='hit')._value.get() +
            cache_operations_total.labels(cache_name='redis', operation='miss')._value.get()
        )
        if total > 0:
            cache_hit_ratio.labels(cache_name='redis').set(hits / total)

        # Business metrics (occasionally)
        if random.random() > 0.9:
            payment_method = random.choice(['credit_card', 'paypal', 'stripe'])
            order_status = random.choice(['completed', 'pending', 'cancelled'])

            orders_total.labels(
                status=order_status,
                payment_method=payment_method
            ).inc()

            if order_status == 'completed':
                revenue = random.uniform(10, 500)
                revenue_total.labels(
                    currency='USD',
                    payment_method=payment_method
                ).inc(revenue)

        # Worker metrics
        worker_threads_active.labels(worker_type='background').set(random.randint(3, 10))
        worker_tasks_processed_total.labels(
            worker_type='background',
            status='success'
        ).inc()

        # Collect system metrics
        collect_system_metrics()

        time.sleep(0.5)

# Flask app for serving metrics
app = Flask(__name__)

@app.route('/metrics')
def metrics():
    """Metrics endpoint"""
    return Response(generate_latest(registry), mimetype=CONTENT_TYPE_LATEST)

@app.route('/health')
def health():
    """Health check endpoint"""
    return {'status': 'healthy', 'timestamp': time.time()}

if __name__ == '__main__':
    # Start metric simulation in background thread
    simulator_thread = Thread(target=simulate_application_metrics, daemon=True)
    simulator_thread.start()

    # Start Flask server
    print("Starting metrics server on port 8000")
    print("Metrics available at http://localhost:8000/metrics")
    app.run(host='0.0.0.0', port=8000)
```

### Example 14: Go Exporter with Custom Collector

```go
package main

import (
    "log"
    "math/rand"
    "net/http"
    "time"

    "github.com/prometheus/client_golang/prometheus"
    "github.com/prometheus/client_golang/prometheus/promhttp"
)

// Custom collector for dynamic metrics
type AppCollector struct {
    requestsTotal      *prometheus.CounterVec
    requestDuration    *prometheus.HistogramVec
    activeConnections  *prometheus.GaugeVec
    queueLength        *prometheus.GaugeVec
    errorRate          *prometheus.GaugeVec
}

func NewAppCollector() *AppCollector {
    return &AppCollector{
        requestsTotal: prometheus.NewCounterVec(
            prometheus.CounterOpts{
                Name: "app_http_requests_total",
                Help: "Total number of HTTP requests",
            },
            []string{"method", "endpoint", "status", "instance"},
        ),
        requestDuration: prometheus.NewHistogramVec(
            prometheus.HistogramOpts{
                Name:    "app_http_request_duration_seconds",
                Help:    "HTTP request latency in seconds",
                Buckets: prometheus.ExponentialBuckets(0.001, 2, 15),
            },
            []string{"method", "endpoint", "instance"},
        ),
        activeConnections: prometheus.NewGaugeVec(
            prometheus.GaugeOpts{
                Name: "app_active_connections",
                Help: "Number of active connections",
            },
            []string{"instance", "state"},
        ),
        queueLength: prometheus.NewGaugeVec(
            prometheus.GaugeOpts{
                Name: "app_queue_length",
                Help: "Current queue length",
            },
            []string{"queue_name", "instance"},
        ),
        errorRate: prometheus.NewGaugeVec(
            prometheus.GaugeOpts{
                Name: "app_error_rate",
                Help: "Current error rate (errors/second)",
            },
            []string{"service", "instance"},
        ),
    }
}

func (c *AppCollector) Describe(ch chan<- *prometheus.Desc) {
    c.requestsTotal.Describe(ch)
    c.requestDuration.Describe(ch)
    c.activeConnections.Describe(ch)
    c.queueLength.Describe(ch)
    c.errorRate.Describe(ch)
}

func (c *AppCollector) Collect(ch chan<- prometheus.Metric) {
    c.requestsTotal.Collect(ch)
    c.requestDuration.Collect(ch)
    c.activeConnections.Collect(ch)
    c.queueLength.Collect(ch)
    c.errorRate.Collect(ch)
}

func (c *AppCollector) simulateMetrics() {
    instance := "api-server-1"
    methods := []string{"GET", "POST", "PUT", "DELETE"}
    endpoints := []string{"/api/users", "/api/products", "/api/orders", "/api/health"}
    statuses := []string{"200", "201", "400", "404", "500"}

    ticker := time.NewTicker(100 * time.Millisecond)
    defer ticker.Stop()

    for range ticker.C {
        // Simulate requests
        method := methods[rand.Intn(len(methods))]
        endpoint := endpoints[rand.Intn(len(endpoints))]
        status := statuses[rand.Intn(len(statuses))]

        c.requestsTotal.WithLabelValues(method, endpoint, status, instance).Inc()
        c.requestDuration.WithLabelValues(method, endpoint, instance).Observe(rand.Float64() * 2)

        // Update gauges
        c.activeConnections.WithLabelValues(instance, "active").Set(float64(rand.Intn(100) + 50))
        c.activeConnections.WithLabelValues(instance, "idle").Set(float64(rand.Intn(50) + 10))

        c.queueLength.WithLabelValues("jobs", instance).Set(float64(rand.Intn(100)))
        c.queueLength.WithLabelValues("emails", instance).Set(float64(rand.Intn(50)))

        c.errorRate.WithLabelValues("api", instance).Set(rand.Float64() * 10)
    }
}

func main() {
    // Create and register collector
    collector := NewAppCollector()
    prometheus.MustRegister(collector)

    // Start simulating metrics
    go collector.simulateMetrics()

    // Expose metrics endpoint
    http.Handle("/metrics", promhttp.Handler())

    // Health endpoint
    http.HandleFunc("/health", func(w http.ResponseWriter, r *http.Request) {
        w.WriteHeader(http.StatusOK)
        w.Write([]byte("OK"))
    })

    log.Println("Starting metrics server on :8000")
    log.Println("Metrics endpoint: http://localhost:8000/metrics")
    log.Fatal(http.ListenAndServe(":8000", nil))
}
```

---

## Recording Rule Examples

### Example 15: Comprehensive Recording Rules

```yaml
# recording_rules.yml
groups:
  - name: http_performance_rules
    interval: 30s
    rules:
      # Request rate aggregations
      - record: job:http_requests:rate5m
        expr: sum(rate(http_requests_total[5m])) by (job)

      - record: job:http_requests:rate1m
        expr: sum(rate(http_requests_total[1m])) by (job)

      - record: service:http_requests:rate5m
        expr: sum(rate(http_requests_total[5m])) by (service)

      - record: instance:http_requests:rate5m
        expr: sum(rate(http_requests_total[5m])) by (instance)

      # Error rate calculations
      - record: job:http_requests_errors:rate5m
        expr: sum(rate(http_requests_total{status=~"5.."}[5m])) by (job)

      - record: job:http_requests_error_ratio:rate5m
        expr: |
          sum(rate(http_requests_total{status=~"5.."}[5m])) by (job)
          /
          sum(rate(http_requests_total[5m])) by (job)

      - record: service:http_requests_error_ratio:rate5m
        expr: |
          sum(rate(http_requests_total{status=~"5.."}[5m])) by (service)
          /
          sum(rate(http_requests_total[5m])) by (service)

      # Success rate
      - record: job:http_requests_success_ratio:rate5m
        expr: |
          sum(rate(http_requests_total{status=~"2.."}[5m])) by (job)
          /
          sum(rate(http_requests_total[5m])) by (job)

      # Latency percentiles
      - record: job:http_request_duration:p50
        expr: histogram_quantile(0.50, sum(rate(http_request_duration_seconds_bucket[5m])) by (job, le))

      - record: job:http_request_duration:p95
        expr: histogram_quantile(0.95, sum(rate(http_request_duration_seconds_bucket[5m])) by (job, le))

      - record: job:http_request_duration:p99
        expr: histogram_quantile(0.99, sum(rate(http_request_duration_seconds_bucket[5m])) by (job, le))

      - record: service:http_request_duration:p99
        expr: histogram_quantile(0.99, sum(rate(http_request_duration_seconds_bucket[5m])) by (service, le))

      # Average latency
      - record: job:http_request_duration:avg
        expr: |
          sum(rate(http_request_duration_seconds_sum[5m])) by (job)
          /
          sum(rate(http_request_duration_seconds_count[5m])) by (job)

  - name: resource_aggregations
    interval: 30s
    rules:
      # CPU usage by instance
      - record: instance:node_cpu_usage:percent
        expr: 100 - (avg by (instance) (irate(node_cpu_seconds_total{mode="idle"}[5m])) * 100)

      # Memory usage by instance
      - record: instance:node_memory_usage:percent
        expr: (1 - (node_memory_MemAvailable_bytes / node_memory_MemTotal_bytes)) * 100

      # Disk usage by instance and mountpoint
      - record: instance:node_disk_usage:percent
        expr: |
          (
            node_filesystem_size_bytes{fstype!="rootfs"} -
            node_filesystem_avail_bytes{fstype!="rootfs"}
          )
          /
          node_filesystem_size_bytes{fstype!="rootfs"} * 100

      # Network throughput
      - record: instance:node_network_receive:rate5m
        expr: rate(node_network_receive_bytes_total[5m])

      - record: instance:node_network_transmit:rate5m
        expr: rate(node_network_transmit_bytes_total[5m])

  - name: slo_tracking
    interval: 30s
    rules:
      # Availability SLI (30 day window)
      - record: slo:availability:30d
        expr: |
          sum(rate(http_requests_total{status=~"2.."}[30d]))
          /
          sum(rate(http_requests_total[30d]))

      # Latency SLI (% of requests under 500ms)
      - record: slo:latency:30d
        expr: |
          sum(rate(http_request_duration_seconds_bucket{le="0.5"}[30d]))
          /
          sum(rate(http_request_duration_seconds_bucket{le="+Inf"}[30d]))

      # Error budget remaining (30 days, 99.9% target)
      - record: slo:error_budget:remaining_percent
        expr: |
          (1 - ((1 - slo:availability:30d) / (1 - 0.999))) * 100

      # Burn rate (1 hour window)
      - record: slo:burn_rate:1h
        expr: |
          (
            sum(rate(http_requests_total{status=~"5.."}[1h]))
            /
            sum(rate(http_requests_total[1h]))
          ) / (1 - 0.999)

      # Burn rate (6 hour window)
      - record: slo:burn_rate:6h
        expr: |
          (
            sum(rate(http_requests_total{status=~"5.."}[6h]))
            /
            sum(rate(http_requests_total[6h]))
          ) / (1 - 0.999)
```

---

## SLO Monitoring Examples

### Example 16: Complete SLO Implementation

```yaml
# slo_rules.yml
groups:
  - name: api_slo_availability
    interval: 30s
    rules:
      # Define SLI: Availability (success rate)
      - record: api:sli:availability:5m
        expr: |
          sum(rate(http_requests_total{job="api",status=~"2.."}[5m]))
          /
          sum(rate(http_requests_total{job="api"}[5m]))

      - record: api:sli:availability:1h
        expr: |
          sum(rate(http_requests_total{job="api",status=~"2.."}[1h]))
          /
          sum(rate(http_requests_total{job="api"}[1h]))

      - record: api:sli:availability:24h
        expr: |
          sum(rate(http_requests_total{job="api",status=~"2.."}[24h]))
          /
          sum(rate(http_requests_total{job="api"}[24h]))

      - record: api:sli:availability:30d
        expr: |
          sum(rate(http_requests_total{job="api",status=~"2.."}[30d]))
          /
          sum(rate(http_requests_total{job="api"}[30d]))

      # Error budget calculations (99.9% SLO = 0.1% error budget)
      - record: api:error_budget:consumed:1h
        expr: (1 - api:sli:availability:1h) / (1 - 0.999)

      - record: api:error_budget:consumed:24h
        expr: (1 - api:sli:availability:24h) / (1 - 0.999)

      - record: api:error_budget:consumed:30d
        expr: (1 - api:sli:availability:30d) / (1 - 0.999)

      - record: api:error_budget:remaining:30d
        expr: 1 - api:error_budget:consumed:30d

      # Burn rate alerts (multi-window, multi-burn-rate)
      # Page-worthy: 2% budget burn in 1 hour
      - alert: APIErrorBudgetBurnCritical
        expr: |
          api:error_budget:consumed:1h > 0.02
          and
          api:error_budget:consumed:5m > 0.02
        for: 2m
        labels:
          severity: critical
          slo: "availability"
          window: "1h"
        annotations:
          summary: "Critical error budget burn on API"
          description: "API is burning through error budget at {{ $value | humanizePercentage }} of monthly budget per hour"

      # Ticket-worthy: 5% budget burn in 6 hours
      - alert: APIErrorBudgetBurnHigh
        expr: |
          api:error_budget:consumed:6h > 0.05
          and
          api:error_budget:consumed:30m > 0.05
        for: 15m
        labels:
          severity: warning
          slo: "availability"
          window: "6h"
        annotations:
          summary: "High error budget burn on API"
          description: "API is burning through error budget at {{ $value | humanizePercentage }} of monthly budget per 6 hours"

      # Exhausted error budget
      - alert: APIErrorBudgetExhausted
        expr: api:error_budget:remaining:30d < 0
        for: 5m
        labels:
          severity: critical
          slo: "availability"
        annotations:
          summary: "API error budget exhausted"
          description: "API has consumed entire 30-day error budget. Error budget remaining: {{ $value | humanizePercentage }}"

  - name: api_slo_latency
    interval: 30s
    rules:
      # Define SLI: Latency (% requests under threshold)
      - record: api:sli:latency_under_500ms:5m
        expr: |
          sum(rate(http_request_duration_seconds_bucket{job="api",le="0.5"}[5m]))
          /
          sum(rate(http_request_duration_seconds_bucket{job="api",le="+Inf"}[5m]))

      - record: api:sli:latency_under_500ms:1h
        expr: |
          sum(rate(http_request_duration_seconds_bucket{job="api",le="0.5"}[1h]))
          /
          sum(rate(http_request_duration_seconds_bucket{job="api",le="+Inf"}[1h]))

      - record: api:sli:latency_under_500ms:30d
        expr: |
          sum(rate(http_request_duration_seconds_bucket{job="api",le="0.5"}[30d]))
          /
          sum(rate(http_request_duration_seconds_bucket{job="api",le="+Inf"}[30d]))

      # Latency SLO: 95% of requests under 500ms
      - record: api:latency_error_budget:consumed:30d
        expr: (1 - api:sli:latency_under_500ms:30d) / (1 - 0.95)

      - alert: APILatencySLOViolation
        expr: api:sli:latency_under_500ms:1h < 0.95
        for: 10m
        labels:
          severity: warning
          slo: "latency"
        annotations:
          summary: "API latency SLO violation"
          description: "Only {{ $value | humanizePercentage }} of requests are under 500ms (target: 95%)"
```

---

## Multi-Cluster Federation Examples

### Example 17: Federation Configuration

```yaml
# Global Prometheus federating from regional Prometheus instances
global:
  scrape_interval: 30s
  evaluation_interval: 30s
  external_labels:
    cluster: 'global'
    environment: 'production'

scrape_configs:
  # Federate from US West region
  - job_name: 'federate-us-west'
    scrape_interval: 30s
    honor_labels: true
    metrics_path: '/federate'
    params:
      'match[]':
        # Federate job-level aggregations
        - '{job="prometheus"}'
        - '{__name__=~"job:.*"}'
        - '{__name__=~"service:.*"}'
        - '{__name__=~"slo:.*"}'
        # Federate alerts
        - '{__name__=~"ALERTS.*"}'
    static_configs:
      - targets:
          - 'prometheus-us-west-1.example.com:9090'
          - 'prometheus-us-west-2.example.com:9090'
        labels:
          region: 'us-west'

  # Federate from US East region
  - job_name: 'federate-us-east'
    scrape_interval: 30s
    honor_labels: true
    metrics_path: '/federate'
    params:
      'match[]':
        - '{job="prometheus"}'
        - '{__name__=~"job:.*"}'
        - '{__name__=~"service:.*"}'
        - '{__name__=~"slo:.*"}'
        - '{__name__=~"ALERTS.*"}'
    static_configs:
      - targets:
          - 'prometheus-us-east-1.example.com:9090'
          - 'prometheus-us-east-2.example.com:9090'
        labels:
          region: 'us-east'

  # Federate from EU region
  - job_name: 'federate-eu-central'
    scrape_interval: 30s
    honor_labels: true
    metrics_path: '/federate'
    params:
      'match[]':
        - '{job="prometheus"}'
        - '{__name__=~"job:.*"}'
        - '{__name__=~"service:.*"}'
        - '{__name__=~"slo:.*"}'
        - '{__name__=~"ALERTS.*"}'
    static_configs:
      - targets:
          - 'prometheus-eu-central-1.example.com:9090'
        labels:
          region: 'eu-central'

# Global recording rules
rule_files:
  - 'global_rules.yml'
```

```yaml
# global_rules.yml
groups:
  - name: global_aggregations
    interval: 60s
    rules:
      # Global request rate across all regions
      - record: global:http_requests:rate5m
        expr: sum(job:http_requests:rate5m) by (job)

      # Request rate by region
      - record: region:http_requests:rate5m
        expr: sum(job:http_requests:rate5m) by (region, job)

      # Global error rate
      - record: global:http_requests_error_ratio:rate5m
        expr: |
          sum(job:http_requests_errors:rate5m)
          /
          sum(job:http_requests:rate5m)

      # Global availability SLI
      - record: global:slo:availability:30d
        expr: |
          sum(slo:availability:30d) by (region)
          /
          count(slo:availability:30d)
```

---

## Advanced Patterns

### Example 18: Anomaly Detection with PromQL

```promql
# Detect CPU usage anomalies using standard deviation
abs(
  instance:node_cpu_usage:percent
  -
  avg_over_time(instance:node_cpu_usage:percent[1h])
)
>
3 * stddev_over_time(instance:node_cpu_usage:percent[1h])

# Detect request rate anomalies
abs(
  sum(rate(http_requests_total[5m]))
  -
  avg_over_time(sum(rate(http_requests_total[5m]))[1h:5m])
)
>
2 * stddev_over_time(sum(rate(http_requests_total[5m]))[1h:5m])

# Detect latency spikes
histogram_quantile(0.99,
  sum(rate(http_request_duration_seconds_bucket[5m])) by (le)
)
>
1.5 * histogram_quantile(0.99,
  sum(rate(http_request_duration_seconds_bucket[5m] offset 1h)) by (le)
)
```

### Example 19: Cost Monitoring

```yaml
# cost_monitoring.yml
groups:
  - name: cost_tracking
    interval: 5m
    rules:
      # EC2 instance cost (example rates)
      - record: cloud:ec2:cost_per_hour
        expr: |
          sum(
            ec2_instance_running
            *
            on(instance_type) group_left
            ec2_instance_type_cost_per_hour
          ) by (region, environment)

      # Kubernetes CPU cost
      - record: cloud:k8s:cpu_cost_per_hour
        expr: |
          sum(
            kube_pod_container_resource_requests{resource="cpu"}
          ) * 0.03  # $0.03 per vCPU hour

      # Kubernetes memory cost
      - record: cloud:k8s:memory_cost_per_hour
        expr: |
          sum(
            kube_pod_container_resource_requests{resource="memory"}
            / 1024 / 1024 / 1024
          ) * 0.005  # $0.005 per GB hour

      # Total cloud cost per hour
      - record: cloud:total_cost_per_hour
        expr: |
          sum(cloud:ec2:cost_per_hour)
          +
          sum(cloud:k8s:cpu_cost_per_hour)
          +
          sum(cloud:k8s:memory_cost_per_hour)

      # Monthly cost projection
      - record: cloud:monthly_cost_projection
        expr: cloud:total_cost_per_hour * 730

      # Cost per service
      - record: service:cost_per_hour
        expr: |
          sum(
            kube_pod_container_resource_requests{resource="cpu"}
            * 0.03
            +
            kube_pod_container_resource_requests{resource="memory"}
            / 1024 / 1024 / 1024
            * 0.005
          ) by (namespace, service)
```

### Example 20: Capacity Planning Queries

```promql
# Predict when disk will be full (4 hour prediction)
predict_linear(node_filesystem_avail_bytes{mountpoint="/"}[1h], 4 * 3600) < 0

# Predict when memory will be exhausted
predict_linear(node_memory_MemAvailable_bytes[2h], 6 * 3600) < (1024 * 1024 * 1024)  # 1GB

# Database growth rate (bytes/day)
deriv(
  pg_database_size_bytes[7d]
) * 86400

# Request volume growth (requests/day)
(
  avg_over_time(sum(rate(http_requests_total[5m]))[24h:5m])
  -
  avg_over_time(sum(rate(http_requests_total[5m] offset 7d))[24h:5m])
)
/
avg_over_time(sum(rate(http_requests_total[5m] offset 7d))[24h:5m])
* 100

# Time until connection pool exhausted (at current rate)
(
  pg_settings_max_connections
  -
  sum(pg_stat_activity_count)
)
/
deriv(sum(pg_stat_activity_count)[1h])
```

---

**Total Examples**: 20 comprehensive examples covering all aspects of observability and monitoring with Prometheus, Grafana, and related tools.

**Categories Covered**:
- Prometheus configuration (3 examples)
- PromQL queries (4 examples)
- Alert rules (3 examples)
- Grafana dashboards (2 examples)
- Custom exporters (2 examples)
- Recording rules (1 example)
- SLO monitoring (1 example)
- Multi-cluster federation (1 example)
- Advanced patterns (3 examples)

**File Version**: 1.0.0
**Last Updated**: October 2025
