---
name: forge-lang-python
description: Python development standards including pytest, ruff, black, and mypy. Use when working with Python files, tests, or dependencies.
---

# Python Development

## Testing

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=src --cov-report=term-missing

# Run specific test
pytest tests/test_module.py::test_function

# Run with verbose output
pytest -v
```

## Linting

```bash
# Check with ruff
ruff check .

# Fix auto-fixable issues
ruff check --fix .

# Type checking
mypy src/
```

## Formatting

```bash
# Format with ruff (or black)
ruff format .

# Check without changing
ruff format --check .

# Sort imports
ruff check --select I --fix .
```

## Project Structure

```
project/
├── src/
│   └── package/
│       ├── __init__.py
│       └── module.py
├── tests/
│   ├── __init__.py
│   ├── conftest.py
│   └── test_module.py
├── pyproject.toml
└── README.md
```

## pyproject.toml Template

```toml
[project]
name = "package-name"
version = "0.1.0"
requires-python = ">=3.10"

[tool.pytest.ini_options]
testpaths = ["tests"]
addopts = "-ra -q"

[tool.ruff]
line-length = 88
select = ["E", "F", "I", "N", "W"]

[tool.ruff.format]
quote-style = "double"

[tool.mypy]
python_version = "3.10"
strict = true
```

## TDD Cycle Commands

```bash
# RED: Write test, run to see it fail
pytest tests/test_new_feature.py -v

# GREEN: Implement, run to see it pass
pytest tests/test_new_feature.py -v

# REFACTOR: Clean up, ensure tests still pass
pytest && ruff check . && ruff format --check .
```
