#!/bin/bash
# check-tests.sh - Run project tests and report status
# Returns: 0 if tests pass, 1 if tests fail

set -e

# Detect test framework
if [ -f "pytest.ini" ] || [ -f "pyproject.toml" ] || [ -d "tests" ]; then
    echo "Detected: Python (pytest)"
    if command -v pytest &> /dev/null; then
        pytest --tb=short -q
        exit $?
    else
        echo "pytest not installed"
        exit 1
    fi
elif [ -f "Cargo.toml" ]; then
    echo "Detected: Rust (cargo)"
    cargo test --quiet
    exit $?
elif [ -f "package.json" ]; then
    echo "Detected: Node.js"
    if grep -q '"test"' package.json; then
        npm test --silent
        exit $?
    else
        echo "No test script in package.json"
        exit 1
    fi
elif [ -f "go.mod" ]; then
    echo "Detected: Go"
    go test ./... -v
    exit $?
elif [ -f "Makefile" ] && grep -q "^test:" Makefile; then
    echo "Detected: Makefile with test target"
    make test
    exit $?
elif [ -f "CMakeLists.txt" ]; then
    echo "Detected: CMake project"
    if [ -d "build" ]; then
        cd build && ctest
        exit $?
    else
        echo "No build directory found. Run cmake first."
        exit 1
    fi
elif [ -d "test" ] && ls test/*.bats &> /dev/null 2>&1; then
    echo "Detected: Bats tests"
    bats test/
    exit $?
else
    echo "No recognized test framework found"
    echo "Supported: pytest, cargo test, npm test, go test, make test, ctest, bats"
    exit 1
fi
