_mt_stow() {
  # Check for required stow command
  command -v stow &>/dev/null || {
    echo "Error: 'stow' command is required but not found." >&2
    echo "Please install GNU Stow (e.g. 'apt install stow' or 'brew install stow')" >&2
    return 1
  }

  # Use MT_PKG_DIR from environment or set default
  : "${MT_PKG_DIR:=${HOME}/.metool}"

  if [[ $# -lt 1 ]]; then
    echo "Usage: mt stow [STOW_OPTIONS] DIRECTORY..." >&2
    return 1
  fi

  # Get stow options and package paths
  declare -a stow_opts=()
  declare -a pkg_paths=()
  declare -a skipped_files=()
  local mt_verbose=false
  local found_valid_path=false

  # Parse arguments into options and paths
  for arg in "$@"; do
    if [[ "$arg" == "--mt-verbose" ]]; then
      mt_verbose=true
    elif [[ "$arg" == -* ]]; then
      stow_opts+=("$arg")
    elif [[ -d "$arg" ]]; then
      pkg_paths+=("$(realpath "$arg")")
      found_valid_path=true
    elif [[ -e "$arg" ]]; then
      # File exists but is not a directory - track it
      skipped_files+=("$arg")
    elif [[ -L "$arg" ]]; then
      # It's a symlink but target doesn't exist (broken symlink)
      _mt_error "Broken symlink: $arg -> $(readlink "$arg")"
    else
      _mt_error "Path not found: $arg"
    fi
  done
  
  # If we have skipped files, check if they look like explicit arguments
  if [[ ${#skipped_files[@]} -gt 0 ]]; then
    # If we have many arguments (likely from wildcard) and some are directories, 
    # then the files were probably included by wildcard expansion
    if [[ $# -gt 3 ]] && [[ ${#pkg_paths[@]} -gt 0 ]]; then
      # Silently skip - this looks like wildcard expansion
      :
    else
      # Few arguments or no directories found - likely explicit file arguments
      for file in "${skipped_files[@]}"; do
        _mt_error "Not a directory: $file"
      done
    fi
  fi
  
  # Check if we found any valid directories to process
  if [[ "${#pkg_paths[@]}" -eq 0 ]]; then
    if [[ "${#stow_opts[@]}" -eq 0 ]] || ! $found_valid_path; then
      _mt_error "No valid directories to install. Please provide at least one existing directory."
      return 1
    fi
  fi

  # Track if any errors occurred
  local had_errors=false
  local pkg_results=()

  # Process each directory
  for pkg_path in "${pkg_paths[@]}"; do
    # Determine package name - prefer working set name over directory name
    # This ensures bin symlinks use the working set symlink path
    local pkg_name=""

    # Check if there's a symlink in packages/ that points to this path
    if [[ -d "${MT_PKG_DIR}/packages" ]]; then
      local real_pkg_path
      real_pkg_path=$(realpath "$pkg_path" 2>/dev/null)
      for ws_link in "${MT_PKG_DIR}/packages/"*; do
        [[ -L "$ws_link" ]] || continue
        local ws_target
        ws_target=$(realpath "$ws_link" 2>/dev/null)
        if [[ "$ws_target" == "$real_pkg_path" ]]; then
          pkg_name=$(basename "$ws_link")
          break
        fi
      done
    fi

    # Fall back to directory name if not found in working set
    if [[ -z "$pkg_name" ]]; then
      pkg_name="$(basename "$pkg_path")"
    fi

    local pkg_status=""
    local pkg_had_error=false

    # Handle bin/
    # Create symlinks through packages/ directory for cleaner package management
    # This allows package location changes without updating individual bin symlinks
    if [[ -d "${pkg_path}/bin" ]]; then
      command mkdir -p "${MT_PKG_DIR}/bin"
      local bin_had_error=false
      local bin_created=0
      local bin_conflict=0
      local bin_conflicts=()

      for bin_file in "${pkg_path}/bin/"*; do
        [[ -e "$bin_file" ]] || continue  # Skip if glob didn't match
        [[ -f "$bin_file" ]] || [[ -L "$bin_file" ]] || continue  # Only files and symlinks

        local bin_name
        bin_name=$(basename "$bin_file")
        local target_link="${MT_PKG_DIR}/bin/${bin_name}"
        local new_target="../packages/${pkg_name}/bin/${bin_name}"

        # Check if symlink already exists
        if [[ -L "$target_link" ]]; then
          local existing_target
          existing_target=$(readlink "$target_link")

          if [[ "$existing_target" == "$new_target" ]]; then
            # Already correct, skip
            ((bin_created++))
            continue
          fi

          # Check if existing symlink points to same ultimate destination
          local existing_real new_real
          existing_real=$(realpath "$target_link" 2>/dev/null)
          new_real=$(realpath "${MT_PKG_DIR}/packages/${pkg_name}/bin/${bin_name}" 2>/dev/null)

          if [[ "$existing_real" == "$new_real" ]] && [[ -n "$existing_real" ]]; then
            # Same destination, just different path - safe to update
            rm "$target_link"
            if ln -s "$new_target" "$target_link"; then
              ((bin_created++))
            else
              bin_had_error=true
              bin_conflicts+=("$bin_name (failed to create)")
            fi
          else
            # Different destination - conflict
            ((bin_conflict++))
            bin_conflicts+=("$bin_name -> $existing_target")
            bin_had_error=true
          fi
        elif [[ -e "$target_link" ]]; then
          # Regular file exists - conflict
          ((bin_conflict++))
          bin_conflicts+=("$bin_name (regular file)")
          bin_had_error=true
        else
          # No conflict, create symlink
          if ln -s "$new_target" "$target_link"; then
            ((bin_created++))
          else
            bin_had_error=true
            bin_conflicts+=("$bin_name (failed to create)")
          fi
        fi
      done

      if $bin_had_error; then
        pkg_status+="${MT_COLOR_ERROR}bin${MT_COLOR_RESET} "
        pkg_had_error=true
        had_errors=true
        if [[ ${#bin_conflicts[@]} -gt 0 ]]; then
          echo "[${pkg_name}:bin] Conflicts detected:" >&2
          for conflict in "${bin_conflicts[@]}"; do
            echo "[${pkg_name}:bin]   * $conflict" >&2
          done
          echo "[${pkg_name}:bin] Use 'mt package install --force' to override" >&2
        fi
      else
        pkg_status+="${MT_COLOR_INFO}bin${MT_COLOR_RESET} "
      fi
    fi

    # Handle config/
    if [[ -d "${pkg_path}/config" ]]; then
      # First, create an intermediate directory for configs
      command mkdir -p "${MT_PKG_DIR}/config/${pkg_name}"

      # Pre-check: update symlinks that resolve to same content but have different paths
      # This handles package relocations gracefully
      local config_target_dir="${MT_PKG_DIR}/config/${pkg_name}"
      for config_file in "${pkg_path}/config/"*; do
        [[ -e "$config_file" ]] || continue
        local config_name
        config_name=$(basename "$config_file")
        local existing_link="${config_target_dir}/${config_name}"

        if [[ -L "$existing_link" ]]; then
          local existing_real new_real
          existing_real=$(realpath "$existing_link" 2>/dev/null)
          new_real=$(realpath "$config_file" 2>/dev/null)

          if [[ "$existing_real" == "$new_real" ]] && [[ -n "$existing_real" ]]; then
            # Same destination, different path - safe to remove and let stow recreate
            rm "$existing_link"
          fi
        fi
      done

      # Stow from package to metool config dir
      if command stow ${stow_opts[@]+"${stow_opts[@]}"} --dir="${pkg_path}" --target="${MT_PKG_DIR}/config/${pkg_name}" config &>/tmp/mt_stow_output; then
        # Now stow from metool config dir to HOME
        if command stow ${stow_opts[@]+"${stow_opts[@]}"} --dir="${MT_PKG_DIR}/config" --target="${HOME}" --dotfiles "${pkg_name}" &>/tmp/mt_stow_output_2; then
          pkg_status+="${MT_COLOR_INFO}config${MT_COLOR_RESET} "
        else
          pkg_status+="${MT_COLOR_ERROR}config${MT_COLOR_RESET} "
          pkg_had_error=true
          had_errors=true
          
          # Enhanced conflict resolution
          command cat /tmp/mt_stow_output_2 | sed "s/^/[${pkg_name}:config->home] /" >&2
          
          # Extract conflict files from stow output
          local conflict_files=()
          while read -r line; do
            if [[ "$line" =~ \*\ existing\ target\ is\ not\ owned\ by\ stow:\ (.+) ]]; then
              conflict_files+=("${BASH_REMATCH[1]}")
            fi
          done < /tmp/mt_stow_output_2
          
          # Handle each conflict file
          if [[ ${#conflict_files[@]} -gt 0 ]]; then
            echo -e "\n${MT_COLOR_WARNING}Conflicts detected during installation of ${pkg_name}:${MT_COLOR_RESET}"
            
            for conflict in "${conflict_files[@]}"; do
              local conflict_path="${HOME}/${conflict}"
              
              # Check if the file is a symlink
              if [[ -L "${conflict_path}" ]]; then
                local link_target=$(readlink -f "${conflict_path}")
                local link_status="valid"
                
                # Check if it's a broken symlink
                if [[ ! -e "${conflict_path}" ]]; then
                  link_status="broken"
                fi
                
                echo -e "  ${MT_COLOR_WARNING}→ ${conflict} is a ${link_status} symlink pointing to:${MT_COLOR_RESET}"
                echo -e "    ${link_target}"
                
                # Offer to fix broken symlinks automatically
                if [[ "$link_status" == "broken" ]]; then
                  echo -e "  ${MT_COLOR_WARNING}This is a broken symlink and could be safely removed.${MT_COLOR_RESET}"
                fi
              else
                echo -e "  ${MT_COLOR_WARNING}→ ${conflict} is a regular file or directory${MT_COLOR_RESET}"
              fi
              
              # Offer to remove the conflicting item
              # Source prompt functions if needed
              if ! type -t _mt_confirm >/dev/null; then
                source "$(dirname "${BASH_SOURCE[0]}")/prompt.sh"
              fi
              
              _mt_confirm "  Remove this conflicting item and try again?"
              if [[ $? -eq 0 ]]; then
                echo -e "  ${MT_COLOR_INFO}→ Removing ${conflict_path}...${MT_COLOR_RESET}"
                command rm -f "${conflict_path}"
                
                # Try stowing again after resolving conflict
                if command stow ${stow_opts[@]+"${stow_opts[@]}"} --dir="${MT_PKG_DIR}/config" --target="${HOME}" --dotfiles "${pkg_name}" &>/tmp/mt_stow_retry; then
                  echo -e "  ${MT_COLOR_INFO}→ Conflict resolved successfully!${MT_COLOR_RESET}"
                  # Update status to success
                  pkg_status=${pkg_status/${MT_COLOR_ERROR}config${MT_COLOR_RESET}/${MT_COLOR_INFO}config${MT_COLOR_RESET}}
                  pkg_had_error=false
                  had_errors=false  # This might need to be set at a higher level based on other errors
                else
                  echo -e "  ${MT_COLOR_ERROR}→ Still having issues after conflict resolution:${MT_COLOR_RESET}"
                  command cat /tmp/mt_stow_retry | sed "s/^/    /" >&2
                fi
              else
                echo -e "  ${MT_COLOR_WARNING}→ Skipping conflict resolution for this item${MT_COLOR_RESET}"
              fi
            done
          fi
        fi
      else
        pkg_status+="${MT_COLOR_ERROR}config${MT_COLOR_RESET} "
        pkg_had_error=true
        had_errors=true
        command cat /tmp/mt_stow_output | sed "s/^/[${pkg_name}:config->metool] /" >&2
      fi
    fi

    # Handle shell/
    if [[ -d "${pkg_path}/shell" ]]; then
      command mkdir -p "${MT_PKG_DIR}/shell/${pkg_name}"

      # Pre-check: update symlinks that resolve to same content but have different paths
      local shell_target_dir="${MT_PKG_DIR}/shell/${pkg_name}"
      for shell_file in "${pkg_path}/shell/"*; do
        [[ -e "$shell_file" ]] || continue
        local shell_name
        shell_name=$(basename "$shell_file")
        local existing_link="${shell_target_dir}/${shell_name}"

        if [[ -L "$existing_link" ]]; then
          local existing_real new_real
          existing_real=$(realpath "$existing_link" 2>/dev/null)
          new_real=$(realpath "$shell_file" 2>/dev/null)

          if [[ "$existing_real" == "$new_real" ]] && [[ -n "$existing_real" ]]; then
            # Same destination, different path - safe to remove and let stow recreate
            rm "$existing_link"
          fi
        fi
      done

      if command stow ${stow_opts[@]+"${stow_opts[@]}"} --dir="${pkg_path}" --target="${MT_PKG_DIR}/shell/${pkg_name}" shell &>/tmp/mt_stow_output; then
        pkg_status+="${MT_COLOR_INFO}shell${MT_COLOR_RESET} "
      else
        pkg_status+="${MT_COLOR_ERROR}shell${MT_COLOR_RESET} "
        pkg_had_error=true
        had_errors=true
        command cat /tmp/mt_stow_output | sed "s/^/[${pkg_name}:shell] /" >&2
      fi
    fi

    # Handle SKILL.md - stow entire package for Claude Code
    if [[ -f "${pkg_path}/SKILL.md" ]]; then
      command mkdir -p "${MT_PKG_DIR}/skills"

      # Stage 1: Create symlink from ~/.metool/skills/{package} to package directory
      local skills_target="${MT_PKG_DIR}/skills/${pkg_name}"
      if [[ -L "$skills_target" ]]; then
        rm "$skills_target"
      fi

      if ln -s "${pkg_path}" "$skills_target"; then
        # Stage 2: Create symlink in ~/.claude/skills/
        mkdir -p "${HOME}/.claude/skills"
        local claude_skill_link="${HOME}/.claude/skills/${pkg_name}"

        if [[ -L "$claude_skill_link" ]]; then
          rm "$claude_skill_link"
        fi

        if ln -s "$skills_target" "$claude_skill_link"; then
          pkg_status+="${MT_COLOR_INFO}skill${MT_COLOR_RESET} "
        else
          _mt_warning "Failed to create Claude Code skill symlink for ${pkg_name}"
          pkg_status+="${MT_COLOR_WARNING}skill${MT_COLOR_RESET} "
        fi
      else
        pkg_status+="${MT_COLOR_ERROR}skill${MT_COLOR_RESET} "
        pkg_had_error=true
        had_errors=true
        echo "[${pkg_name}:skill] Failed to create metool skill symlink" >&2
      fi
    fi

    # Only show detailed output for packages with errors or if verbose mode is enabled
    if $pkg_had_error || $mt_verbose; then
      printf "${pkg_name}: ${pkg_status}\n"
    fi

    # Store result for summary
    if $pkg_had_error; then
      pkg_results+=("\033[0;31m${pkg_name}\033[0m")
    else
      pkg_results+=("\033[0;34m${pkg_name}\033[0m")
    fi
  done

  # Print summary
  if $had_errors; then
    printf "\nStow completed (with errors) in: ${pkg_results[*]}\n"
    printf "Use 'mt stow --mt-verbose' to see all package details\n"
  elif $mt_verbose; then
    printf "\nStow completed successfully for: ${pkg_results[*]}\n"
  else
    printf "Stow completed successfully for: ${pkg_results[*]}\n"
  fi

  # Clean up temp file
  command rm -f /tmp/mt_stow_output

  # If metool itself was installed and no errors occurred, offer to update .bashrc
  for pkg_path in "${pkg_paths[@]}"; do
    pkg_name="$(basename "$pkg_path")"
    if [[ "$pkg_name" == "metool" ]] && ! $had_errors; then
      _mt_update_bashrc
      break
    fi
  done

  # Invalidate cache after installation
  _mt_invalidate_cache
}
