#!/usr/bin/env bats
# Tests for module commands (MT-11)

load test_helper

setup() {
  # Create temporary test directory
  export TEST_DIR="${BATS_TMPDIR}/metool-test-$$"
  export MT_PKG_DIR="${TEST_DIR}/.metool"
  export MT_MODULES_DIR="${MT_PKG_DIR}/modules"
  export MT_PACKAGES_DIR="${MT_PKG_DIR}/packages"
  export MT_ROOT="${BATS_TEST_DIRNAME}/.."
  export HOME="${TEST_DIR}/home"

  mkdir -p "${TEST_DIR}"
  mkdir -p "${MT_MODULES_DIR}"
  mkdir -p "${MT_PACKAGES_DIR}"
  mkdir -p "${HOME}"

  # Source required libraries
  source "${MT_ROOT}/lib/functions.sh"
  source "${MT_ROOT}/lib/colors.sh"
  source "${MT_ROOT}/lib/git.sh"
  source "${MT_ROOT}/lib/working-set.sh"
  source "${MT_ROOT}/lib/module.sh"
}

teardown() {
  # Clean up test directory
  if [ -n "${TEST_DIR}" ] && [ -d "${TEST_DIR}" ]; then
    rm -rf "${TEST_DIR}"
  fi
}

# mt module list tests

@test "module: list shows empty working set" {
  run _mt_module_list
  [ $status -eq 0 ]
  [[ "$output" =~ "No modules in working set" ]]
}

@test "module: list shows valid module" {
  # Create test module
  local module_dir="${TEST_DIR}/test-module"
  mkdir -p "${module_dir}"
  ln -s "${module_dir}" "${MT_MODULES_DIR}/test-module"

  run _mt_module_list
  [ $status -eq 0 ]
  [[ "$output" =~ "test-module" ]]
  [[ "$output" =~ "MODULE" ]]
  [[ "$output" =~ "PATH" ]]
}

@test "module: list shows broken module symlink" {
  # Create broken symlink
  ln -s "/nonexistent/path" "${MT_MODULES_DIR}/broken-module"

  run _mt_module_list
  [ $status -eq 0 ]
  [[ "$output" =~ "broken-module" ]]
  [[ "$output" =~ "broken" ]]
}

@test "module: list shows all modules" {
  # Create multiple modules
  for i in {1..3}; do
    local module_dir="${TEST_DIR}/module-${i}"
    mkdir -p "${module_dir}"
    ln -s "${module_dir}" "${MT_MODULES_DIR}/module-${i}"
  done

  run _mt_module_list
  [ $status -eq 0 ]
  [[ "$output" =~ "module-1" ]]
  [[ "$output" =~ "module-2" ]]
  [[ "$output" =~ "module-3" ]]
}

# mt module add tests

@test "module: add creates module directory structure" {
  skip "Requires git clone functionality"
  # This test would need a real or mock git repository
}

@test "module: add validates input" {
  run _mt_module_add
  [ $status -ne 0 ]
  [[ "$output" =~ "Usage" ]]
}

@test "module: add detects existing module" {
  skip "Requires mocking git functions which is complex in bats"
  # This would test that adding the same module twice gives a message
  # Test covered by integration testing instead
}

@test "module: add creates symlink for existing repository" {
  skip "Requires mocking git functions which is complex in bats"
  # This would test that _mt_module_add creates the symlink
  # Test covered by integration testing instead
}

# mt module remove tests

@test "module: remove requires module name" {
  run _mt_module_remove
  [ $status -ne 0 ]
  [[ "$output" =~ "Usage" ]]
}

@test "module: remove validates module exists" {
  run _mt_module_remove "nonexistent"
  [ $status -ne 0 ]
  [[ "$output" =~ "not found" ]]
}

@test "module: remove deletes symlink" {
  # Create test module
  local module_dir="${TEST_DIR}/test-module"
  mkdir -p "${module_dir}"
  ln -s "${module_dir}" "${MT_MODULES_DIR}/test-module"

  # Remove with automatic yes using input redirection
  _mt_module_remove "test-module" < <(echo "y") >/dev/null 2>&1

  # Symlink should be removed
  [ ! -L "${MT_MODULES_DIR}/test-module" ]
}

@test "module: remove warns about dependent packages" {
  skip "Interactive confirmation test - tested manually"
  # This test requires interactive input which is complex to test in bats
  # Functionality verified through integration testing
}

@test "module: remove preserves repository" {
  # Create test module
  local module_dir="${TEST_DIR}/test-module"
  mkdir -p "${module_dir}"
  ln -s "${module_dir}" "${MT_MODULES_DIR}/test-module"

  # Remove with automatic yes using input redirection
  _mt_module_remove "test-module" < <(echo "y") >/dev/null 2>&1

  # Repository should still exist
  [ -d "${module_dir}" ]
}

# mt module edit tests

@test "module: edit requires module name" {
  run _mt_module_edit
  [ $status -ne 0 ]
  [[ "$output" =~ "Usage" ]]
}

@test "module: edit validates module exists" {
  run _mt_module_edit "nonexistent"
  [ $status -ne 0 ]
  [[ "$output" =~ "not found" ]]
}

@test "module: edit validates symlink is not broken" {
  # Create broken symlink
  ln -s "/nonexistent/path" "${MT_MODULES_DIR}/broken-module"

  run _mt_module_edit "broken-module"
  [ $status -ne 0 ]
  [[ "$output" =~ "broken" ]]
}

@test "module: edit uses EDITOR environment variable" {
  # Create test module
  local module_dir="${TEST_DIR}/test-module"
  mkdir -p "${module_dir}"
  ln -s "${module_dir}" "${MT_MODULES_DIR}/test-module"

  # Set custom editor that just echoes the path
  export EDITOR="echo"

  run _mt_module_edit "test-module"
  [ $status -eq 0 ]
  [[ "$output" =~ "${module_dir}" ]]
}
