# SLSA Framework Implementation Reference

Detailed guidance for implementing SLSA (Supply-chain Levels for Software Artifacts) compliance.

## SLSA Levels Explained

### SLSA Level 1: Documentation

**Requirements:**

- Build process is scripted/automated
- Provenance is generated (even if unsigned)
- Documentation exists

**What You Get:**

- Basic transparency into build process
- Audit trail of what was built
- Foundation for higher levels

**Implementation:**

```yaml
# GitHub Actions - SLSA Level 1
name: Build with Provenance
on: [push]

jobs:
  build:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v5

      - name: Build
        run: |
          npm ci
          npm run build

      - name: Generate provenance
        run: |
          cat << EOF > provenance.json
          {
            "builder": {
              "id": "https://github.com/${{ github.repository }}/actions"
            },
            "buildType": "https://github.com/actions/runner",
            "invocation": {
              "configSource": {
                "uri": "git+https://github.com/${{ github.repository }}@${{ github.ref }}",
                "digest": {
                  "sha1": "${{ github.sha }}"
                },
                "entryPoint": ".github/workflows/build.yml"
              }
            },
            "materials": [
              {
                "uri": "git+https://github.com/${{ github.repository }}",
                "digest": {
                  "sha1": "${{ github.sha }}"
                }
              }
            ]
          }
          EOF

      - name: Upload provenance
        uses: actions/upload-artifact@v4
        with:
          name: provenance
          path: provenance.json
```

### SLSA Level 2: Authenticated Provenance

**Requirements (in addition to Level 1):**

- Build runs on hosted platform
- Provenance is authenticated/signed
- Provenance is non-forgeable by the build service

**What You Get:**

- Confidence that provenance wasn't tampered after build
- Verification that build ran on claimed platform
- Protection against post-build modifications

**Implementation with GitHub Actions:**

```yaml
# GitHub Actions - SLSA Level 2 with SLSA Generator
name: SLSA Level 2 Build
on:
  release:
    types: [published]

permissions:
  contents: write
  id-token: write
  attestations: write

jobs:
  build:
    runs-on: ubuntu-latest
    outputs:
      hashes: ${{ steps.hash.outputs.hashes }}

    steps:
      - uses: actions/checkout@v5

      - name: Build
        run: |
          npm ci
          npm run build
          tar -czf dist.tar.gz dist/

      - name: Generate hash
        id: hash
        run: |
          echo "hashes=$(sha256sum dist.tar.gz | base64 -w0)" >> "$GITHUB_OUTPUT"

      - name: Upload artifact
        uses: actions/upload-artifact@v4
        with:
          name: dist
          path: dist.tar.gz

  provenance:
    needs: [build]
    permissions:
      actions: read
      id-token: write
      contents: write

    uses: slsa-framework/slsa-github-generator/.github/workflows/generator_generic_slsa3.yml@v2.0.0
    with:
      base64-subjects: "${{ needs.build.outputs.hashes }}"
      upload-assets: true
```

### SLSA Level 3: Hardened Build

**Requirements (in addition to Level 2):**

- Build platform is hardened
- Non-falsifiable provenance (cryptographic guarantee)
- Isolated build environment
- Reproducible builds (recommended)

**What You Get:**

- Protection against tampering during build
- Strong cryptographic verification
- Confidence in build isolation

**Implementation:**

```yaml
# GitHub Actions - SLSA Level 3
name: SLSA Level 3 Container Build

on:
  release:
    types: [published]

permissions:
  contents: read
  packages: write
  id-token: write

jobs:
  build:
    runs-on: ubuntu-latest
    outputs:
      image: ${{ steps.build.outputs.image }}
      digest: ${{ steps.build.outputs.digest }}

    steps:
      - uses: actions/checkout@v5

      - name: Set up Docker Buildx
        uses: docker/setup-buildx-action@v3

      - name: Login to GHCR
        uses: docker/login-action@v3
        with:
          registry: ghcr.io
          username: ${{ github.actor }}
          password: ${{ secrets.GITHUB_TOKEN }}

      - name: Build and push
        id: build
        uses: docker/build-push-action@v5
        with:
          context: .
          push: true
          tags: ghcr.io/${{ github.repository }}:${{ github.ref_name }}
          # Reproducible build settings
          provenance: true
          sbom: true

      - name: Output image info
        run: |
          echo "image=ghcr.io/${{ github.repository }}" >> "$GITHUB_OUTPUT"
          echo "digest=${{ steps.build.outputs.digest }}" >> "$GITHUB_OUTPUT"

  provenance:
    needs: [build]
    permissions:
      actions: read
      id-token: write
      packages: write

    uses: slsa-framework/slsa-github-generator/.github/workflows/generator_container_slsa3.yml@v2.0.0
    with:
      image: ${{ needs.build.outputs.image }}
      digest: ${{ needs.build.outputs.digest }}
      registry-username: ${{ github.actor }}
    secrets:
      registry-password: ${{ secrets.GITHUB_TOKEN }}
```

### SLSA Level 4: Maximum Assurance

**Requirements (in addition to Level 3):**

- Two-person review for all changes
- Hermetic builds (no network access during build)
- Reproducible builds
- Build parameters from source

**What You Get:**

- Protection against insider threats
- Maximum assurance of software integrity
- Full auditability

**Hermetic Build Example:**

```dockerfile
# Dockerfile for hermetic builds
# All dependencies must be vendored or pinned

FROM golang:1.22-alpine AS builder

# Copy vendored dependencies (no network fetch)
COPY vendor/ /app/vendor/
COPY go.mod go.sum /app/
COPY . /app/

WORKDIR /app

# Build with no network (hermetic)
RUN CGO_ENABLED=0 GOOS=linux GOFLAGS="-mod=vendor" \
    go build -ldflags="-s -w" -o /app/bin/myapp ./cmd/myapp

# Minimal runtime image
FROM scratch
COPY --from=builder /app/bin/myapp /myapp
ENTRYPOINT ["/myapp"]
```

```yaml
# GitHub Actions - Hermetic build enforcement
name: Hermetic Build
on: [push]

jobs:
  build:
    runs-on: ubuntu-latest
    container:
      image: golang:1.22-alpine
      options: --network none  # No network access

    steps:
      - uses: actions/checkout@v5

      - name: Verify vendored dependencies
        run: |
          if [ ! -d vendor ]; then
            echo "Error: Dependencies must be vendored for hermetic builds"
            exit 1
          fi

      - name: Build
        run: |
          CGO_ENABLED=0 GOFLAGS="-mod=vendor" go build -o myapp ./cmd/myapp

      - name: Upload artifact
        uses: actions/upload-artifact@v4
        with:
          name: myapp
          path: myapp
```

## Provenance Verification

### Verify SLSA Provenance with slsa-verifier

```bash
# Install slsa-verifier
go install github.com/slsa-framework/slsa-verifier/v2/cli/slsa-verifier@latest

# Verify artifact provenance
slsa-verifier verify-artifact \
  --provenance-path provenance.intoto.jsonl \
  --source-uri github.com/myorg/myproject \
  --source-tag v1.0.0 \
  myartifact.tar.gz

# Verify container image
slsa-verifier verify-image \
  --source-uri github.com/myorg/myproject \
  --source-tag v1.0.0 \
  ghcr.io/myorg/myproject:v1.0.0
```

### Programmatic Verification

```csharp
using System.Diagnostics;
using System.Text.Json;
using System.Text.Json.Nodes;

/// <summary>
/// Verify SLSA provenance programmatically.
/// </summary>
public sealed record ProvenanceVerificationResult(
    bool Verified,
    string? BuilderId = null,
    string? SourceUri = null,
    string? SourceRef = null,
    string? BuildTrigger = null,
    string? Error = null);

public sealed class SlsaVerifier
{
    /// <summary>
    /// Verify artifact SLSA provenance.
    /// </summary>
    public async Task<ProvenanceVerificationResult> VerifyArtifactProvenanceAsync(
        string artifactPath,
        string provenancePath,
        string expectedSourceUri,
        string? expectedSourceTag = null,
        CancellationToken ct = default)
    {
        using var process = new Process
        {
            StartInfo = new ProcessStartInfo
            {
                FileName = "slsa-verifier",
                RedirectStandardOutput = true,
                RedirectStandardError = true,
                UseShellExecute = false,
                CreateNoWindow = true
            }
        };

        process.StartInfo.ArgumentList.Add("verify-artifact");
        process.StartInfo.ArgumentList.Add("--provenance-path");
        process.StartInfo.ArgumentList.Add(provenancePath);
        process.StartInfo.ArgumentList.Add("--source-uri");
        process.StartInfo.ArgumentList.Add(expectedSourceUri);
        process.StartInfo.ArgumentList.Add(artifactPath);

        if (!string.IsNullOrEmpty(expectedSourceTag))
        {
            process.StartInfo.ArgumentList.Add("--source-tag");
            process.StartInfo.ArgumentList.Add(expectedSourceTag);
        }

        process.Start();
        var stderr = await process.StandardError.ReadToEndAsync(ct);
        await process.WaitForExitAsync(ct);

        if (process.ExitCode == 0)
        {
            // Parse provenance for details
            var firstLine = (await File.ReadAllLinesAsync(provenancePath, ct)).FirstOrDefault();
            if (firstLine is null)
                return new ProvenanceVerificationResult(Verified: true, SourceUri: expectedSourceUri);

            var prov = JsonNode.Parse(firstLine);
            var predicate = prov?["predicate"]?.AsObject();
            var invocation = predicate?["invocation"]?.AsObject();
            var configSource = invocation?["configSource"]?.AsObject();

            return new ProvenanceVerificationResult(
                Verified: true,
                BuilderId: predicate?["builder"]?["id"]?.GetValue<string>(),
                SourceUri: configSource?["uri"]?.GetValue<string>(),
                SourceRef: configSource?["digest"]?["sha1"]?.GetValue<string>(),
                BuildTrigger: invocation?["parameters"]?["github.event_name"]?.GetValue<string>());
        }

        return new ProvenanceVerificationResult(Verified: false, Error: stderr);
    }

    /// <summary>
    /// Verify container image SLSA provenance.
    /// </summary>
    public async Task<ProvenanceVerificationResult> VerifyContainerProvenanceAsync(
        string imageRef,
        string expectedSourceUri,
        string? expectedSourceTag = null,
        CancellationToken ct = default)
    {
        using var process = new Process
        {
            StartInfo = new ProcessStartInfo
            {
                FileName = "slsa-verifier",
                RedirectStandardOutput = true,
                RedirectStandardError = true,
                UseShellExecute = false,
                CreateNoWindow = true
            }
        };

        process.StartInfo.ArgumentList.Add("verify-image");
        process.StartInfo.ArgumentList.Add("--source-uri");
        process.StartInfo.ArgumentList.Add(expectedSourceUri);
        process.StartInfo.ArgumentList.Add(imageRef);

        if (!string.IsNullOrEmpty(expectedSourceTag))
        {
            process.StartInfo.ArgumentList.Add("--source-tag");
            process.StartInfo.ArgumentList.Add(expectedSourceTag);
        }

        process.Start();
        var stderr = await process.StandardError.ReadToEndAsync(ct);
        await process.WaitForExitAsync(ct);

        if (process.ExitCode == 0)
            return new ProvenanceVerificationResult(Verified: true, SourceUri: expectedSourceUri);

        return new ProvenanceVerificationResult(Verified: false, Error: stderr);
    }
}

// Example usage
var verifier = new SlsaVerifier();
var result = await verifier.VerifyArtifactProvenanceAsync(
    "dist.tar.gz",
    "provenance.intoto.jsonl",
    "github.com/myorg/myproject",
    "v1.0.0");

if (result.Verified)
{
    Console.WriteLine("✅ Artifact verified!");
    Console.WriteLine($"   Builder: {result.BuilderId}");
    Console.WriteLine($"   Source: {result.SourceUri}");
}
else
{
    Console.WriteLine($"❌ Verification failed: {result.Error}");
}
```

## In-Toto Attestations

### Understanding In-Toto Layout

SLSA provenance uses the in-toto attestation format:

```json
{
  "_type": "https://in-toto.io/Statement/v1",
  "subject": [
    {
      "name": "myartifact.tar.gz",
      "digest": {
        "sha256": "abc123..."
      }
    }
  ],
  "predicateType": "https://slsa.dev/provenance/v1",
  "predicate": {
    "buildDefinition": {
      "buildType": "https://slsa-framework.github.io/github-actions-buildtypes/workflow/v1",
      "externalParameters": {
        "workflow": {
          "ref": "refs/tags/v1.0.0",
          "repository": "https://github.com/myorg/myproject",
          "path": ".github/workflows/release.yml"
        }
      },
      "internalParameters": {
        "github": {
          "event_name": "release",
          "repository_id": "123456789",
          "repository_owner_id": "987654321"
        }
      },
      "resolvedDependencies": [
        {
          "uri": "git+https://github.com/myorg/myproject@refs/tags/v1.0.0",
          "digest": {
            "gitCommit": "abc123def456..."
          }
        }
      ]
    },
    "runDetails": {
      "builder": {
        "id": "https://github.com/slsa-framework/slsa-github-generator/.github/workflows/generator_generic_slsa3.yml@refs/tags/v2.0.0"
      },
      "metadata": {
        "invocationId": "https://github.com/myorg/myproject/actions/runs/123456789/attempts/1"
      }
    }
  }
}
```

### Creating Custom In-Toto Attestations

```csharp
using System.Security.Cryptography;
using System.Text.Json;
using System.Text.Json.Nodes;

/// <summary>
/// Create custom in-toto attestations.
/// </summary>
public sealed record AttestationSubject(string Name, Dictionary<string, string> Digest);

public static class InTotoAttestationBuilder
{
    /// <summary>
    /// Create SLSA v1 provenance attestation.
    /// </summary>
    public static JsonObject CreateProvenanceAttestation(
        IEnumerable<AttestationSubject> subjects,
        string builderId,
        string buildType,
        string sourceRepo,
        string sourceRef,
        string sourceDigest,
        string workflowPath,
        string runId)
    {
        var subjectArray = new JsonArray();
        foreach (var subject in subjects)
        {
            var digestObj = new JsonObject();
            foreach (var (key, value) in subject.Digest)
                digestObj[key] = value;

            subjectArray.Add(new JsonObject
            {
                ["name"] = subject.Name,
                ["digest"] = digestObj
            });
        }

        return new JsonObject
        {
            ["_type"] = "https://in-toto.io/Statement/v1",
            ["subject"] = subjectArray,
            ["predicateType"] = "https://slsa.dev/provenance/v1",
            ["predicate"] = new JsonObject
            {
                ["buildDefinition"] = new JsonObject
                {
                    ["buildType"] = buildType,
                    ["externalParameters"] = new JsonObject
                    {
                        ["workflow"] = new JsonObject
                        {
                            ["ref"] = sourceRef,
                            ["repository"] = sourceRepo,
                            ["path"] = workflowPath
                        }
                    },
                    ["resolvedDependencies"] = new JsonArray
                    {
                        new JsonObject
                        {
                            ["uri"] = $"git+{sourceRepo}@{sourceRef}",
                            ["digest"] = new JsonObject
                            {
                                ["gitCommit"] = sourceDigest
                            }
                        }
                    }
                },
                ["runDetails"] = new JsonObject
                {
                    ["builder"] = new JsonObject
                    {
                        ["id"] = builderId
                    },
                    ["metadata"] = new JsonObject
                    {
                        ["invocationId"] = runId,
                        ["startedOn"] = DateTime.UtcNow.ToString("O")
                    }
                }
            }
        };
    }

    /// <summary>
    /// Calculate file hash.
    /// </summary>
    public static async Task<string> HashFileAsync(
        string path,
        CancellationToken ct = default)
    {
        await using var stream = File.OpenRead(path);
        var hashBytes = await SHA256.HashDataAsync(stream, ct);
        return Convert.ToHexStringLower(hashBytes);
    }

    /// <summary>
    /// Create subject entry for attestation.
    /// </summary>
    public static async Task<AttestationSubject> CreateSubjectAsync(
        string path,
        CancellationToken ct = default)
    {
        var hash = await HashFileAsync(path, ct);
        return new AttestationSubject(
            Name: Path.GetFileName(path),
            Digest: new Dictionary<string, string> { ["sha256"] = hash });
    }
}

// Example: Create attestation for multiple artifacts
var artifacts = new[] { "dist/app.tar.gz", "dist/app.sig" };
var subjects = new List<AttestationSubject>();

foreach (var artifact in artifacts.Where(File.Exists))
    subjects.Add(await InTotoAttestationBuilder.CreateSubjectAsync(artifact));

var attestation = InTotoAttestationBuilder.CreateProvenanceAttestation(
    subjects: subjects,
    builderId: "https://github.com/myorg/myproject/actions",
    buildType: "https://github.com/actions/runner",
    sourceRepo: "https://github.com/myorg/myproject",
    sourceRef: Environment.GetEnvironmentVariable("GITHUB_REF") ?? "refs/heads/main",
    sourceDigest: Environment.GetEnvironmentVariable("GITHUB_SHA") ?? "unknown",
    workflowPath: ".github/workflows/build.yml",
    runId: Environment.GetEnvironmentVariable("GITHUB_RUN_ID") ?? "local");

var options = new JsonSerializerOptions { WriteIndented = true };
await File.WriteAllTextAsync(
    "provenance.intoto.json",
    attestation.ToJsonString(options));
```

## Reproducible Builds

### Go Reproducible Build

```go
// Build with reproducible flags
// go build -trimpath -ldflags="-s -w -buildid=" -o myapp

package main

import (
    "fmt"
    "runtime/debug"
)

func main() {
    info, ok := debug.ReadBuildInfo()
    if ok {
        fmt.Printf("Module: %s\n", info.Main.Path)
        fmt.Printf("Version: %s\n", info.Main.Version)

        // Show build settings for reproducibility verification
        for _, setting := range info.Settings {
            fmt.Printf("%s: %s\n", setting.Key, setting.Value)
        }
    }
}
```

```makefile
# Makefile for reproducible Go builds
.PHONY: build

VERSION ?= $(shell git describe --tags --always --dirty)
COMMIT  ?= $(shell git rev-parse HEAD)
DATE    ?= $(shell date -u +%Y-%m-%dT%H:%M:%SZ)

# Reproducible build flags
LDFLAGS := -s -w \
    -X main.version=$(VERSION) \
    -X main.commit=$(COMMIT) \
    -X main.date=$(DATE) \
    -buildid=

build:
    CGO_ENABLED=0 \
    GOOS=linux \
    GOARCH=amd64 \
    go build \
        -trimpath \
        -ldflags="$(LDFLAGS)" \
        -o bin/myapp \
        ./cmd/myapp
```

### NPM Reproducible Build

```json
{
  "scripts": {
    "build:reproducible": "npm ci --ignore-scripts && npm run build"
  },
  "engines": {
    "node": ">=20.0.0",
    "npm": ">=10.0.0"
  }
}
```

```yaml
# .github/workflows/reproducible.yml
name: Reproducible Build Verification

on: [push]

jobs:
  build-verify:
    runs-on: ubuntu-latest
    strategy:
      matrix:
        run: [1, 2, 3]

    steps:
      - uses: actions/checkout@v5

      - name: Setup Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'

      - name: Build
        run: |
          npm ci --ignore-scripts
          npm run build

      - name: Calculate hash
        id: hash
        run: |
          find dist -type f -exec sha256sum {} \; | sort > checksums.txt
          echo "hash=$(sha256sum checksums.txt | cut -d' ' -f1)" >> "$GITHUB_OUTPUT"

      - name: Upload checksum
        uses: actions/upload-artifact@v4
        with:
          name: checksums-${{ matrix.run }}
          path: checksums.txt

  compare:
    needs: build-verify
    runs-on: ubuntu-latest
    steps:
      - name: Download all checksums
        uses: actions/download-artifact@v4

      - name: Compare builds
        run: |
          echo "Comparing build outputs..."
          diff checksums-1/checksums.txt checksums-2/checksums.txt
          diff checksums-2/checksums.txt checksums-3/checksums.txt
          echo "✅ All builds are reproducible!"
```

## SLSA Compliance Checklist

### Level 1 Checklist

- [ ] Build process is fully scripted (no manual steps)
- [ ] Build runs in CI/CD system
- [ ] Build produces provenance document
- [ ] Provenance includes:
  - [ ] Builder identity
  - [ ] Source repository
  - [ ] Source revision (commit SHA)
  - [ ] Build entry point (workflow file)

### Level 2 Checklist

- [ ] All Level 1 requirements
- [ ] Build runs on hosted platform (GitHub Actions, GitLab CI, etc.)
- [ ] Provenance is cryptographically signed
- [ ] Provenance signature is verifiable
- [ ] Provenance cannot be modified by the build job itself

### Level 3 Checklist

- [ ] All Level 2 requirements
- [ ] Build platform is hardened and isolated
- [ ] Build environment is ephemeral
- [ ] Provenance is non-falsifiable (signed by trusted builder)
- [ ] Using official SLSA generator (slsa-github-generator)
- [ ] Build parameters come from verified sources

### Level 4 Checklist

- [ ] All Level 3 requirements
- [ ] Two-person review required for all code changes
- [ ] Build is hermetic (no network access during build)
- [ ] All dependencies are vendored or pinned with hashes
- [ ] Build is reproducible (same inputs = same outputs)
- [ ] Build configuration is in source control
