# Database Migrations with SQLModel

SQLModel uses SQLAlchemy under the hood, so migrations are handled via Alembic.

## Setup

```bash
uv add alembic
alembic init migrations
```

## Configuration

Edit `alembic.ini`:
```ini
sqlalchemy.url = postgresql://user:pass@host/db
```

Edit `migrations/env.py`:
```python
from sqlmodel import SQLModel
from app.models import *  # Import all models

target_metadata = SQLModel.metadata
```

## Commands

```bash
# Create migration
alembic revision --autogenerate -m "Add tasks table"

# Apply migrations
alembic upgrade head

# Rollback one step
alembic downgrade -1

# Show current version
alembic current

# Show history
alembic history
```

## Migration Script Example

```python
"""Add tasks table

Revision ID: abc123
Revises:
Create Date: 2024-01-15

"""
from alembic import op
import sqlalchemy as sa
import sqlmodel

revision = 'abc123'
down_revision = None
branch_labels = None
depends_on = None

def upgrade():
    op.create_table(
        'task',
        sa.Column('id', sa.Integer(), nullable=False),
        sa.Column('title', sa.String(length=200), nullable=False),
        sa.Column('status', sa.String(), nullable=False),
        sa.Column('created_at', sa.DateTime(), nullable=False),
        sa.PrimaryKeyConstraint('id')
    )
    op.create_index('ix_task_title', 'task', ['title'])

def downgrade():
    op.drop_index('ix_task_title', 'task')
    op.drop_table('task')
```

## Async Migrations

For async engines, modify `migrations/env.py`:

```python
import asyncio
from sqlalchemy.ext.asyncio import create_async_engine

def run_migrations_online():
    connectable = create_async_engine(get_url())

    async def do_run_migrations():
        async with connectable.connect() as connection:
            await connection.run_sync(do_run_migrations_sync)

    asyncio.run(do_run_migrations())

def do_run_migrations_sync(connection):
    context.configure(
        connection=connection,
        target_metadata=target_metadata
    )
    with context.begin_transaction():
        context.run_migrations()
```

## Best Practices

1. **Always review autogenerated migrations** - Alembic may miss some changes
2. **Test migrations on a copy of production data** before deploying
3. **Keep migrations small and focused** - one logical change per migration
4. **Never edit applied migrations** - create new ones instead
5. **Include both upgrade and downgrade** - enable rollbacks

## Common Operations

### Add column
```python
op.add_column('task', sa.Column('priority', sa.String(), default='medium'))
```

### Remove column
```python
op.drop_column('task', 'priority')
```

### Add index
```python
op.create_index('ix_task_status', 'task', ['status'])
```

### Add foreign key
```python
op.add_column('task', sa.Column('project_id', sa.Integer()))
op.create_foreign_key('fk_task_project', 'task', 'project', ['project_id'], ['id'])
```

### Rename column
```python
op.alter_column('task', 'name', new_column_name='title')
```
