# Documentation Generation Examples

Practical examples demonstrating documentation generation patterns using real tools and workflows.

---

## Example 1: Sphinx Documentation for Python Package

Scenario: Generate API documentation for a Python package using Sphinx with autodoc.

Setup Steps:

1. Install Sphinx and extensions:
   - pip install sphinx sphinx-autodoc-typehints sphinx-rtd-theme

2. Initialize Sphinx in docs directory:
   - Run sphinx-quickstart docs
   - Answer prompts for project name, author, version

3. Configure docs/conf.py:
   - Add parent directory to sys.path for module discovery
   - Set extensions list to include autodoc, napoleon, viewcode
   - Configure html_theme to sphinx_rtd_theme
   - Set autodoc_member_order to bysource

4. Generate API documentation:
   - Run sphinx-apidoc -o docs/api src/mypackage
   - This creates .rst files for each module

5. Build HTML documentation:
   - Run make html in docs directory
   - Output appears in docs/_build/html

Result: Professional HTML documentation with automatically generated API reference from docstrings, syntax highlighting, and navigation sidebar.

---

## Example 2: MkDocs with mkdocstrings

Scenario: Create documentation site with automatic Python API docs using MkDocs Material.

Setup Steps:

1. Install MkDocs and plugins:
   - pip install mkdocs mkdocs-material mkdocstrings mkdocstrings-python

2. Create mkdocs.yml configuration:
   - Set site_name to project name
   - Configure theme with name material
   - Add mkdocstrings to plugins list
   - Define nav structure with docs and API sections

3. Create docs/api.md with mkdocstrings syntax:
   - Use ::: mypackage.module.ClassName to document classes
   - Use ::: mypackage.module.function_name for functions
   - Configure options for rendering style

4. Serve locally for preview:
   - Run mkdocs serve
   - Access at http://localhost:8000

5. Deploy to GitHub Pages:
   - Run mkdocs gh-deploy
   - Documentation available at username.github.io/repo

Result: Modern documentation site with search, dark mode, and API documentation generated directly from Python docstrings.

---

## Example 3: TypeDoc for TypeScript Library

Scenario: Generate API documentation for a TypeScript library.

Setup Steps:

1. Install TypeDoc:
   - npm install typedoc --save-dev

2. Create typedoc.json configuration:
   - Set entryPoints to src/index.ts
   - Configure out to docs/api
   - Enable includeVersion
   - Set excludePrivate to true

3. Add documentation script to package.json:
   - Add script docs:generate with command typedoc

4. Write TSDoc comments in source:
   - Add summary as first paragraph
   - Use @param for each parameter
   - Use @returns for return value
   - Use @example for usage examples

5. Generate documentation:
   - Run npm run docs:generate
   - HTML output in docs/api directory

Result: Comprehensive TypeScript API documentation with type information, inheritance hierarchies, and source links.

---

## Example 4: OpenAPI from FastAPI

Scenario: Generate and customize OpenAPI documentation for a FastAPI application.

Enhancement Steps:

1. Add detailed route docstrings:
   - Include description of endpoint purpose
   - Document expected behavior
   - Note authentication requirements

2. Configure Pydantic models:
   - Add Field with description parameter
   - Include example values
   - Document validation rules

3. Organize with tags:
   - Add tags parameter to route decorators
   - Define tag metadata in app setup
   - Group related endpoints

4. Access documentation:
   - Swagger UI at /docs
   - ReDoc at /redoc
   - OpenAPI JSON at /openapi.json

5. Export for external tools:
   - Save openapi.json to file
   - Use with Postman, Stoplight, or SDK generators

Result: Interactive API documentation with try-it-out feature, automatically updated as code changes.

---

## Example 5: Docusaurus Documentation Site

Scenario: Create a comprehensive documentation site with tutorials and API reference.

Setup Steps:

1. Initialize Docusaurus:
   - Run npx create-docusaurus@latest my-docs classic

2. Configure docusaurus.config.js:
   - Set title, tagline, and url
   - Configure navbar with documentation links
   - Add footer with resources
   - Enable search with Algolia

3. Organize documentation:
   - Create tutorial section in docs/tutorials
   - Add getting-started guide
   - Include API reference section
   - Add blog for updates

4. Add sidebar configuration:
   - Create sidebars.js with category structure
   - Use autogenerated sidebars for large sections
   - Add custom ordering where needed

5. Deploy to hosting:
   - Build with npm run build
   - Deploy to Vercel, Netlify, or GitHub Pages

Result: Full-featured documentation site with versioning, search, blog, and responsive design.

---

## Example 6: CI/CD Documentation Pipeline

Scenario: Automate documentation generation and deployment in GitHub Actions.

Workflow Steps:

1. Create .github/workflows/docs.yml:
   - Trigger on push to main branch
   - Trigger on changes to src or docs directories

2. Setup job with checkout and runtime:
   - Use actions/checkout
   - Setup Python or Node.js as needed

3. Install documentation dependencies:
   - For Python: pip install sphinx or mkdocs
   - For Node: npm ci

4. Generate documentation:
   - Run appropriate build command
   - Validate output exists

5. Deploy to GitHub Pages:
   - Use peaceiris/actions-gh-pages action
   - Configure publish directory
   - Set custom domain if needed

Result: Documentation automatically rebuilt and deployed on every code change, keeping docs in sync with code.

---

## Common Patterns

### Pattern 1: Documentation as Code

Treat documentation like source code:
- Store in version control alongside code
- Review documentation changes in pull requests
- Use CI/CD to build and deploy
- Track issues for documentation improvements

### Pattern 2: Single Source of Truth

Generate documentation from authoritative sources:
- API docs from code annotations and types
- Configuration docs from schema files
- Architecture docs from diagrams-as-code

### Pattern 3: Progressive Documentation

Layer documentation for different audiences:
- Quick start for new users
- Tutorials for learning
- How-to guides for tasks
- Reference for lookup
- Explanation for understanding

---

## Anti-Patterns to Avoid

### Anti-Pattern 1: Orphaned Documentation

Problem: Documentation stored separately from code becomes outdated.

Solution: Keep documentation in same repository, update in same PRs as code changes.

### Anti-Pattern 2: Over-Documentation

Problem: Documenting obvious code clutters documentation and increases maintenance.

Solution: Document why, not what. Focus on non-obvious decisions and usage patterns.

### Anti-Pattern 3: No Documentation Ownership

Problem: No clear ownership leads to documentation decay.

Solution: Assign documentation responsibilities, include in code review process.

---

Version: 2.0.0
Last Updated: 2025-12-30
