# C++ Production-Ready Code Examples

## Complete CMake Project

### Project Structure

```
cpp_project/
├── CMakeLists.txt
├── vcpkg.json
├── include/
│   ├── mylib/
│   │   ├── core.hpp
│   │   ├── config.hpp
│   │   ├── utils.hpp
│   │   └── models/
│   │       ├── user.hpp
│   │       └── product.hpp
├── src/
│   ├── core.cpp
│   ├── config.cpp
│   ├── utils.cpp
│   └── models/
│       ├── user.cpp
│       └── product.cpp
├── tests/
│   ├── core_test.cpp
│   ├── utils_test.cpp
│   └── models/
│       └── user_test.cpp
├── examples/
│   └── basic_usage.cpp
└── docs/
    └── README.md
```

### Root CMakeLists.txt

```cmake
# CMakeLists.txt
cmake_minimum_required(VERSION 3.28)
project(MyProject VERSION 1.0.0 LANGUAGES CXX)

# C++23 standard
set(CMAKE_CXX_STANDARD 23)
set(CMAKE_CXX_STANDARD_REQUIRED ON)
set(CMAKE_CXX_EXTENSIONS OFF)

# Export compile commands for IDE integration
set(CMAKE_EXPORT_COMPILE_COMMANDS ON)

# Compiler warnings
add_compile_options(
    $<$<CXX_COMPILER_ID:GNU,Clang>:-Wall -Wextra -Wpedantic -Werror>
    $<$<CXX_COMPILER_ID:MSVC>:/W4 /WX>
)

# Optimization flags
if(CMAKE_BUILD_TYPE STREQUAL "Release")
    add_compile_options(
        $<$<CXX_COMPILER_ID:GNU,Clang>:-O3 -march=native>
        $<$<CXX_COMPILER_ID:MSVC>:/O2>
    )
endif()

# Dependencies with FetchContent
include(FetchContent)

FetchContent_Declare(fmt
    GIT_REPOSITORY https://github.com/fmtlib/fmt
    GIT_TAG 10.2.1
    GIT_SHALLOW TRUE)

FetchContent_Declare(spdlog
    GIT_REPOSITORY https://github.com/gabime/spdlog
    GIT_TAG v1.13.0
    GIT_SHALLOW TRUE)

FetchContent_Declare(googletest
    GIT_REPOSITORY https://github.com/google/googletest
    GIT_TAG v1.14.0
    GIT_SHALLOW TRUE)

FetchContent_Declare(nlohmann_json
    GIT_REPOSITORY https://github.com/nlohmann/json
    GIT_TAG v3.11.3
    GIT_SHALLOW TRUE)

FetchContent_MakeAvailable(fmt spdlog googletest nlohmann_json)

# Main library
add_library(mylib STATIC
    src/core.cpp
    src/config.cpp
    src/utils.cpp
    src/models/user.cpp
    src/models/product.cpp
)

target_include_directories(mylib PUBLIC
    $<BUILD_INTERFACE:${CMAKE_CURRENT_SOURCE_DIR}/include>
    $<INSTALL_INTERFACE:include>
)

target_link_libraries(mylib PUBLIC
    fmt::fmt
    spdlog::spdlog
    nlohmann_json::nlohmann_json
)

# Main executable
add_executable(myapp src/main.cpp)
target_link_libraries(myapp PRIVATE mylib)

# Testing
enable_testing()

add_executable(mylib_tests
    tests/core_test.cpp
    tests/utils_test.cpp
    tests/models/user_test.cpp
)

target_link_libraries(mylib_tests PRIVATE
    mylib
    GTest::gtest_main
)

include(GoogleTest)
gtest_discover_tests(mylib_tests)

# Installation
install(TARGETS mylib myapp
    LIBRARY DESTINATION lib
    ARCHIVE DESTINATION lib
    RUNTIME DESTINATION bin
    INCLUDES DESTINATION include
)

install(DIRECTORY include/mylib DESTINATION include)
```

### vcpkg.json (Package Management)

```json
{
  "$schema": "https://raw.githubusercontent.com/microsoft/vcpkg-tool/main/docs/vcpkg.schema.json",
  "name": "myproject",
  "version": "1.0.0",
  "dependencies": [
    "fmt",
    "spdlog",
    "nlohmann-json",
    {
      "name": "gtest",
      "features": ["gmock"]
    }
  ],
  "builtin-baseline": "a42af01b72c28a8e1d7b48107b33e4f286a55ef6"
}
```

---

## C++23 Features Examples

### std::expected Error Handling

```cpp
// include/mylib/utils.hpp
#pragma once
#include <expected>
#include <string>
#include <string_view>
#include <system_error>

namespace mylib {

enum class ParseError {
    InvalidFormat,
    OutOfRange,
    EmptyInput,
    UnknownError
};

class parse_error_category : public std::error_category {
public:
    auto name() const noexcept -> const char* override {
        return "parse_error";
    }

    auto message(int ev) const -> std::string override {
        switch (static_cast<ParseError>(ev)) {
            case ParseError::InvalidFormat:
                return "Invalid format";
            case ParseError::OutOfRange:
                return "Value out of range";
            case ParseError::EmptyInput:
                return "Empty input";
            default:
                return "Unknown error";
        }
    }
};

inline const parse_error_category& parse_error_category_instance() {
    static parse_error_category instance;
    return instance;
}

auto make_error_code(ParseError e) -> std::error_code {
    return {static_cast<int>(e), parse_error_category_instance()};
}

auto parse_int(std::string_view str) -> std::expected<int, std::error_code>;
auto parse_double(std::string_view str) -> std::expected<double, std::error_code>;

template<typename T>
auto safe_divide(T numerator, T denominator) -> std::expected<T, std::error_code> {
    if (denominator == T{0}) {
        return std::unexpected(make_error_code(ParseError::OutOfRange));
    }
    return numerator / denominator;
}

} // namespace mylib

namespace std {
template<>
struct is_error_code_enum<mylib::ParseError> : true_type {};
}
```

```cpp
// src/utils.cpp
#include "mylib/utils.hpp"
#include <charconv>
#include <stdexcept>

namespace mylib {

auto parse_int(std::string_view str) -> std::expected<int, std::error_code> {
    if (str.empty()) {
        return std::unexpected(make_error_code(ParseError::EmptyInput));
    }

    int value{};
    auto [ptr, ec] = std::from_chars(str.data(), str.data() + str.size(), value);

    if (ec == std::errc::invalid_argument) {
        return std::unexpected(make_error_code(ParseError::InvalidFormat));
    }
    if (ec == std::errc::result_out_of_range) {
        return std::unexpected(make_error_code(ParseError::OutOfRange));
    }

    return value;
}

auto parse_double(std::string_view str) -> std::expected<double, std::error_code> {
    if (str.empty()) {
        return std::unexpected(make_error_code(ParseError::EmptyInput));
    }

    try {
        size_t pos = 0;
        double value = std::stod(std::string(str), &pos);
        if (pos != str.size()) {
            return std::unexpected(make_error_code(ParseError::InvalidFormat));
        }
        return value;
    } catch (const std::invalid_argument&) {
        return std::unexpected(make_error_code(ParseError::InvalidFormat));
    } catch (const std::out_of_range&) {
        return std::unexpected(make_error_code(ParseError::OutOfRange));
    }
}

} // namespace mylib
```

### std::print and std::format

```cpp
// examples/print_format.cpp
#include <print>
#include <format>
#include <vector>
#include <string>

struct User {
    int id;
    std::string name;
    double balance;
};

template<>
struct std::formatter<User> {
    constexpr auto parse(format_parse_context& ctx) {
        return ctx.begin();
    }

    auto format(const User& user, format_context& ctx) const {
        return std::format_to(
            ctx.out(),
            "User(id={}, name='{}', balance=${:.2f})",
            user.id, user.name, user.balance
        );
    }
};

int main() {
    // Basic printing
    std::println("Hello, {}!", "World");
    std::print("Value: {}, Hex: {:#x}\n", 255, 255);
    std::println("Formatted: {:.2f}", 3.14159);

    // Custom formatter
    User user{1, "Alice", 1234.56};
    std::println("{}", user);
    // Output: User(id=1, name='Alice', balance=$1234.56)

    // Vector formatting
    std::vector<int> numbers{1, 2, 3, 4, 5};
    std::println("Numbers: [{}]",
        std::format("{}", fmt::join(numbers, ", ")));

    // Alignment and width
    std::println("{:>10} | {:>10}", "Name", "Age");
    std::println("{:->21}", "");
    std::println("{:>10} | {:>10}", "Alice", 30);
    std::println("{:>10} | {:>10}", "Bob", 25);

    return 0;
}
```

### Deducing This (Explicit Object Parameter)

```cpp
// include/mylib/builder.hpp
#pragma once
#include <string>
#include <utility>

namespace mylib {

class QueryBuilder {
    std::string query_;

public:
    QueryBuilder() = default;

    template<typename Self>
    auto select(this Self&& self, std::string_view columns) -> Self&& {
        self.query_ += "SELECT ";
        self.query_ += columns;
        return std::forward<Self>(self);
    }

    template<typename Self>
    auto from(this Self&& self, std::string_view table) -> Self&& {
        self.query_ += " FROM ";
        self.query_ += table;
        return std::forward<Self>(self);
    }

    template<typename Self>
    auto where(this Self&& self, std::string_view condition) -> Self&& {
        self.query_ += " WHERE ";
        self.query_ += condition;
        return std::forward<Self>(self);
    }

    template<typename Self>
    auto order_by(this Self&& self, std::string_view column) -> Self&& {
        self.query_ += " ORDER BY ";
        self.query_ += column;
        return std::forward<Self>(self);
    }

    auto build() const -> std::string {
        return query_;
    }
};

} // namespace mylib
```

```cpp
// Usage example
auto query = mylib::QueryBuilder{}
    .select("id, name, email")
    .from("users")
    .where("age > 18")
    .order_by("name ASC")
    .build();
// Result: "SELECT id, name, email FROM users WHERE age > 18 ORDER BY name ASC"
```

---

## C++20 Features Examples

### Concepts and Constraints

```cpp
// include/mylib/concepts.hpp
#pragma once
#include <concepts>
#include <ranges>
#include <string>

namespace mylib::concepts {

// Basic concepts
template<typename T>
concept Numeric = std::integral<T> || std::floating_point<T>;

template<typename T>
concept Arithmetic = requires(T a, T b) {
    { a + b } -> std::same_as<T>;
    { a - b } -> std::same_as<T>;
    { a * b } -> std::same_as<T>;
    { a / b } -> std::same_as<T>;
};

template<typename T>
concept Comparable = requires(const T& a, const T& b) {
    { a < b } -> std::convertible_to<bool>;
    { a > b } -> std::convertible_to<bool>;
    { a <= b } -> std::convertible_to<bool>;
    { a >= b } -> std::convertible_to<bool>;
    { a == b } -> std::convertible_to<bool>;
    { a != b } -> std::convertible_to<bool>;
};

template<typename T>
concept Hashable = requires(T a) {
    { std::hash<T>{}(a) } -> std::convertible_to<std::size_t>;
};

template<typename T>
concept Serializable = requires(const T& t) {
    { t.to_json() } -> std::convertible_to<std::string>;
    { T::from_json(std::declval<std::string>()) } -> std::same_as<T>;
};

template<typename Container, typename Value>
concept ContainerOf = std::ranges::range<Container> &&
    std::same_as<std::ranges::range_value_t<Container>, Value>;

// Advanced concepts
template<typename T>
concept Resource = requires(T t) {
    typename T::handle_type;
    { t.acquire() } -> std::same_as<typename T::handle_type>;
    { t.release(std::declval<typename T::handle_type>()) } -> std::same_as<void>;
};

} // namespace mylib::concepts
```

```cpp
// src/algorithms.cpp
#include "mylib/concepts.hpp"
#include <algorithm>
#include <vector>
#include <ranges>

namespace mylib {

template<concepts::Numeric T>
auto sum(const std::vector<T>& values) -> T {
    T result{};
    for (const auto& v : values) {
        result += v;
    }
    return result;
}

template<concepts::Comparable T>
auto find_max(const std::vector<T>& values) -> T {
    return *std::ranges::max_element(values);
}

template<typename Container>
    requires concepts::ContainerOf<Container, int>
auto filter_positive(const Container& container) -> std::vector<int> {
    auto positive_only = container
        | std::views::filter([](int n) { return n > 0; });
    return {positive_only.begin(), positive_only.end()};
}

} // namespace mylib
```

### Ranges and Views

```cpp
// include/mylib/range_algorithms.hpp
#pragma once
#include <ranges>
#include <vector>
#include <algorithm>
#include <functional>

namespace mylib::ranges {

template<std::ranges::input_range R, typename Pred>
auto filter_and_transform(R&& range, Pred&& predicate, auto&& transform) {
    return std::forward<R>(range)
        | std::views::filter(std::forward<Pred>(predicate))
        | std::views::transform(std::forward<decltype(transform)>(transform));
}

template<std::ranges::input_range R>
auto take_while_sum_less_than(R&& range, int threshold) {
    auto result = std::vector<std::ranges::range_value_t<R>>{};
    int current_sum = 0;

    for (const auto& value : range) {
        if (current_sum + value >= threshold) {
            break;
        }
        current_sum += value;
        result.push_back(value);
    }

    return result;
}

template<std::ranges::input_range R>
auto sliding_window(R&& range, std::size_t window_size) {
    return std::forward<R>(range)
        | std::views::slide(window_size);
}

} // namespace mylib::ranges
```

```cpp
// Usage examples
#include "mylib/range_algorithms.hpp"
#include <print>

int main() {
    std::vector<int> numbers{1, 2, 3, 4, 5, 6, 7, 8, 9, 10};

    // Filter even numbers and square them
    auto result = mylib::ranges::filter_and_transform(
        numbers,
        [](int n) { return n % 2 == 0; },
        [](int n) { return n * n; }
    );

    for (const auto& n : result) {
        std::println("{}", n);  // Prints: 4, 16, 36, 64, 100
    }

    // Take elements while sum is less than threshold
    auto partial = mylib::ranges::take_while_sum_less_than(numbers, 20);
    // Result: {1, 2, 3, 4, 5, 6} (sum = 21 would exceed 20)

    // Sliding window
    for (const auto window : mylib::ranges::sliding_window(numbers, 3)) {
        std::print("[");
        for (const auto& n : window) {
            std::print("{}, ", n);
        }
        std::println("]");
    }

    return 0;
}
```

### Coroutines

```cpp
// include/mylib/generator.hpp
#pragma once
#include <coroutine>
#include <exception>
#include <utility>

namespace mylib {

template<typename T>
class Generator {
public:
    struct promise_type {
        T current_value;
        std::exception_ptr exception;

        auto get_return_object() -> Generator {
            return Generator{
                std::coroutine_handle<promise_type>::from_promise(*this)
            };
        }

        auto initial_suspend() -> std::suspend_always { return {}; }
        auto final_suspend() noexcept -> std::suspend_always { return {}; }

        auto yield_value(T value) -> std::suspend_always {
            current_value = std::move(value);
            return {};
        }

        auto return_void() -> void {}

        auto unhandled_exception() -> void {
            exception = std::current_exception();
        }
    };

    explicit Generator(std::coroutine_handle<promise_type> h)
        : handle_(h) {}

    ~Generator() {
        if (handle_) {
            handle_.destroy();
        }
    }

    Generator(const Generator&) = delete;
    auto operator=(const Generator&) -> Generator& = delete;

    Generator(Generator&& other) noexcept
        : handle_(std::exchange(other.handle_, {})) {}

    auto operator=(Generator&& other) noexcept -> Generator& {
        if (this != &other) {
            if (handle_) {
                handle_.destroy();
            }
            handle_ = std::exchange(other.handle_, {});
        }
        return *this;
    }

    auto next() -> bool {
        handle_.resume();
        return !handle_.done();
    }

    auto value() const -> const T& {
        return handle_.promise().current_value;
    }

private:
    std::coroutine_handle<promise_type> handle_;
};

} // namespace mylib
```

```cpp
// examples/coroutine_usage.cpp
#include "mylib/generator.hpp"
#include <print>

auto fibonacci() -> mylib::Generator<int> {
    int a = 0, b = 1;
    while (true) {
        co_yield a;
        auto temp = a;
        a = b;
        b = temp + b;
    }
}

auto range(int start, int end, int step = 1) -> mylib::Generator<int> {
    for (int i = start; i < end; i += step) {
        co_yield i;
    }
}

int main() {
    // Fibonacci sequence
    auto fib = fibonacci();
    for (int i = 0; i < 10 && fib.next(); ++i) {
        std::println("{}", fib.value());
    }

    // Custom range
    auto nums = range(0, 20, 3);
    while (nums.next()) {
        std::println("{}", nums.value());
    }

    return 0;
}
```

---

## Modern Memory Management

### RAII Patterns

```cpp
// include/mylib/file_handle.hpp
#pragma once
#include <cstdio>
#include <string_view>
#include <expected>
#include <system_error>

namespace mylib {

class FileHandle {
    FILE* file_ = nullptr;

public:
    FileHandle() = default;

    explicit FileHandle(std::string_view filename, std::string_view mode) {
        file_ = std::fopen(filename.data(), mode.data());
    }

    ~FileHandle() {
        close();
    }

    // Delete copy operations
    FileHandle(const FileHandle&) = delete;
    auto operator=(const FileHandle&) -> FileHandle& = delete;

    // Move operations
    FileHandle(FileHandle&& other) noexcept
        : file_(std::exchange(other.file_, nullptr)) {}

    auto operator=(FileHandle&& other) noexcept -> FileHandle& {
        if (this != &other) {
            close();
            file_ = std::exchange(other.file_, nullptr);
        }
        return *this;
    }

    auto is_open() const -> bool {
        return file_ != nullptr;
    }

    auto get() const -> FILE* {
        return file_;
    }

    auto close() -> void {
        if (file_) {
            std::fclose(file_);
            file_ = nullptr;
        }
    }

    auto write(std::string_view data) -> std::expected<void, std::error_code> {
        if (!file_) {
            return std::unexpected(
                std::make_error_code(std::errc::bad_file_descriptor)
            );
        }

        auto written = std::fwrite(data.data(), 1, data.size(), file_);
        if (written != data.size()) {
            return std::unexpected(
                std::make_error_code(std::errc::io_error)
            );
        }

        return {};
    }

    auto read(char* buffer, size_t size)
        -> std::expected<size_t, std::error_code> {
        if (!file_) {
            return std::unexpected(
                std::make_error_code(std::errc::bad_file_descriptor)
            );
        }

        auto bytes_read = std::fread(buffer, 1, size, file_);
        if (std::ferror(file_)) {
            return std::unexpected(
                std::make_error_code(std::errc::io_error)
            );
        }

        return bytes_read;
    }
};

} // namespace mylib
```

### Smart Pointer Patterns

```cpp
// include/mylib/connection_pool.hpp
#pragma once
#include <memory>
#include <vector>
#include <mutex>
#include <condition_variable>
#include <chrono>

namespace mylib {

class Connection {
public:
    explicit Connection(int id) : id_(id) {}
    auto id() const -> int { return id_; }
    auto execute_query(std::string_view query) -> bool {
        // Simulate query execution
        return true;
    }

private:
    int id_;
};

class ConnectionPool {
    std::vector<std::unique_ptr<Connection>> available_;
    std::vector<std::unique_ptr<Connection>> in_use_;
    mutable std::mutex mutex_;
    std::condition_variable cv_;
    size_t max_connections_;

public:
    explicit ConnectionPool(size_t max_connections)
        : max_connections_(max_connections) {
        available_.reserve(max_connections);
        for (size_t i = 0; i < max_connections; ++i) {
            available_.push_back(std::make_unique<Connection>(i));
        }
    }

    auto acquire(std::chrono::milliseconds timeout = std::chrono::seconds(5))
        -> std::unique_ptr<Connection> {
        std::unique_lock lock(mutex_);

        if (!cv_.wait_for(lock, timeout, [this] {
            return !available_.empty();
        })) {
            return nullptr;  // Timeout
        }

        auto conn = std::move(available_.back());
        available_.pop_back();
        in_use_.push_back(std::move(conn));
        return std::move(in_use_.back());
    }

    auto release(std::unique_ptr<Connection> conn) -> void {
        std::lock_guard lock(mutex_);

        auto it = std::ranges::find_if(in_use_, [&](const auto& c) {
            return c.get() == conn.get();
        });

        if (it != in_use_.end()) {
            available_.push_back(std::move(*it));
            in_use_.erase(it);
            cv_.notify_one();
        }
    }

    auto size() const -> size_t {
        std::lock_guard lock(mutex_);
        return available_.size() + in_use_.size();
    }
};

} // namespace mylib
```

### Custom Deleters

```cpp
// include/mylib/resource_handle.hpp
#pragma once
#include <memory>
#include <cstdlib>

namespace mylib {

// Custom deleter for C-style arrays
struct CArrayDeleter {
    auto operator()(void* ptr) const -> void {
        std::free(ptr);
    }
};

using CArrayPtr = std::unique_ptr<void, CArrayDeleter>;

// Factory function for C-style arrays
template<typename T>
auto make_c_array(size_t size) -> std::unique_ptr<T[], CArrayDeleter> {
    auto ptr = static_cast<T*>(std::malloc(size * sizeof(T)));
    return {ptr, CArrayDeleter{}};
}

// Socket handle with custom deleter
struct SocketDeleter {
    auto operator()(int* socket) const -> void {
        if (socket && *socket >= 0) {
            // close(*socket);  // Platform-specific close
            delete socket;
        }
    }
};

using SocketHandle = std::unique_ptr<int, SocketDeleter>;

auto create_socket() -> SocketHandle {
    // Simulate socket creation
    int fd = 42;  // Simulated file descriptor
    return SocketHandle(new int(fd));
}

} // namespace mylib
```

---

## Testing with Google Test

### Basic Test Structure

```cpp
// tests/core_test.cpp
#include <gtest/gtest.h>
#include "mylib/utils.hpp"
#include "mylib/concepts.hpp"

class UtilsTest : public ::testing::Test {
protected:
    void SetUp() override {
        // Setup code before each test
    }

    void TearDown() override {
        // Cleanup code after each test
    }
};

TEST_F(UtilsTest, ParseIntSuccess) {
    auto result = mylib::parse_int("42");
    ASSERT_TRUE(result.has_value());
    EXPECT_EQ(*result, 42);
}

TEST_F(UtilsTest, ParseIntNegative) {
    auto result = mylib::parse_int("-123");
    ASSERT_TRUE(result.has_value());
    EXPECT_EQ(*result, -123);
}

TEST_F(UtilsTest, ParseIntInvalidFormat) {
    auto result = mylib::parse_int("abc");
    ASSERT_FALSE(result.has_value());
    EXPECT_EQ(result.error(), mylib::make_error_code(mylib::ParseError::InvalidFormat));
}

TEST_F(UtilsTest, ParseIntEmptyInput) {
    auto result = mylib::parse_int("");
    ASSERT_FALSE(result.has_value());
    EXPECT_EQ(result.error(), mylib::make_error_code(mylib::ParseError::EmptyInput));
}

TEST(SafeDivideTest, ValidDivision) {
    auto result = mylib::safe_divide(10, 2);
    ASSERT_TRUE(result.has_value());
    EXPECT_EQ(*result, 5);
}

TEST(SafeDivideTest, DivisionByZero) {
    auto result = mylib::safe_divide(10, 0);
    ASSERT_FALSE(result.has_value());
}
```

### Parameterized Tests

```cpp
// tests/parametrized_test.cpp
#include <gtest/gtest.h>
#include "mylib/utils.hpp"

struct ParseTestCase {
    std::string input;
    bool should_succeed;
    int expected_value;
};

class ParseIntParametrizedTest
    : public ::testing::TestWithParam<ParseTestCase> {
};

TEST_P(ParseIntParametrizedTest, ParseVariousInputs) {
    const auto& test_case = GetParam();
    auto result = mylib::parse_int(test_case.input);

    if (test_case.should_succeed) {
        ASSERT_TRUE(result.has_value());
        EXPECT_EQ(*result, test_case.expected_value);
    } else {
        ASSERT_FALSE(result.has_value());
    }
}

INSTANTIATE_TEST_SUITE_P(
    ParseIntTests,
    ParseIntParametrizedTest,
    ::testing::Values(
        ParseTestCase{"0", true, 0},
        ParseTestCase{"42", true, 42},
        ParseTestCase{"-123", true, -123},
        ParseTestCase{"2147483647", true, 2147483647},
        ParseTestCase{"-2147483648", true, -2147483648},
        ParseTestCase{"abc", false, 0},
        ParseTestCase{"12.34", false, 0},
        ParseTestCase{"", false, 0}
    )
);
```

### Mock Objects

```cpp
// tests/mock_test.cpp
#include <gtest/gtest.h>
#include <gmock/gmock.h>

class DatabaseInterface {
public:
    virtual ~DatabaseInterface() = default;
    virtual auto execute_query(std::string_view query) -> bool = 0;
    virtual auto fetch_result() -> std::vector<std::string> = 0;
};

class MockDatabase : public DatabaseInterface {
public:
    MOCK_METHOD(bool, execute_query, (std::string_view), (override));
    MOCK_METHOD(std::vector<std::string>, fetch_result, (), (override));
};

class UserService {
    DatabaseInterface& db_;

public:
    explicit UserService(DatabaseInterface& db) : db_(db) {}

    auto get_user_names() -> std::vector<std::string> {
        if (db_.execute_query("SELECT name FROM users")) {
            return db_.fetch_result();
        }
        return {};
    }
};

TEST(UserServiceTest, GetUserNamesSuccess) {
    MockDatabase mock_db;

    EXPECT_CALL(mock_db, execute_query(::testing::_))
        .WillOnce(::testing::Return(true));

    EXPECT_CALL(mock_db, fetch_result())
        .WillOnce(::testing::Return(
            std::vector<std::string>{"Alice", "Bob", "Charlie"}
        ));

    UserService service(mock_db);
    auto names = service.get_user_names();

    ASSERT_EQ(names.size(), 3);
    EXPECT_EQ(names[0], "Alice");
    EXPECT_EQ(names[1], "Bob");
    EXPECT_EQ(names[2], "Charlie");
}
```

---

## Concurrency Examples

### std::jthread and Stop Tokens

```cpp
// examples/jthread_example.cpp
#include <thread>
#include <stop_token>
#include <chrono>
#include <print>
#include <queue>
#include <mutex>

class TaskQueue {
    std::queue<std::string> tasks_;
    mutable std::mutex mutex_;

public:
    auto push(std::string task) -> void {
        std::lock_guard lock(mutex_);
        tasks_.push(std::move(task));
    }

    auto pop() -> std::optional<std::string> {
        std::lock_guard lock(mutex_);
        if (tasks_.empty()) {
            return std::nullopt;
        }
        auto task = std::move(tasks_.front());
        tasks_.pop();
        return task;
    }

    auto size() const -> size_t {
        std::lock_guard lock(mutex_);
        return tasks_.size();
    }
};

auto worker(std::stop_token stoken, TaskQueue& queue, int worker_id) -> void {
    std::println("Worker {} started", worker_id);

    while (!stoken.stop_requested()) {
        if (auto task = queue.pop()) {
            std::println("Worker {} processing: {}", worker_id, *task);
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        } else {
            std::this_thread::sleep_for(std::chrono::milliseconds(10));
        }
    }

    std::println("Worker {} stopped", worker_id);
}

int main() {
    TaskQueue queue;

    // Create worker threads
    std::jthread worker1(worker, std::ref(queue), 1);
    std::jthread worker2(worker, std::ref(queue), 2);

    // Add tasks
    for (int i = 0; i < 10; ++i) {
        queue.push(std::format("Task {}", i));
    }

    // Wait for tasks to complete
    while (queue.size() > 0) {
        std::this_thread::sleep_for(std::chrono::milliseconds(50));
    }

    // Workers automatically stopped when jthread is destroyed
    std::println("All tasks completed");

    return 0;
}
```

### std::latch and std::barrier

```cpp
// examples/sync_primitives.cpp
#include <latch>
#include <barrier>
#include <thread>
#include <vector>
#include <print>
#include <chrono>

auto latch_example() -> void {
    const int num_workers = 5;
    std::latch ready(num_workers);
    std::latch done(num_workers);

    auto worker = [&](int id) {
        std::println("Worker {} initializing...", id);
        std::this_thread::sleep_for(std::chrono::milliseconds(100 * id));

        ready.count_down();
        ready.wait();  // Wait for all workers to be ready

        std::println("Worker {} executing", id);
        std::this_thread::sleep_for(std::chrono::milliseconds(50));

        done.count_down();
    };

    std::vector<std::jthread> workers;
    for (int i = 0; i < num_workers; ++i) {
        workers.emplace_back(worker, i);
    }

    done.wait();
    std::println("All workers completed");
}

auto barrier_example() -> void {
    const int num_iterations = 3;
    const int num_workers = 4;

    auto on_completion = []() noexcept {
        std::println("--- Phase completed ---");
    };

    std::barrier sync_point(num_workers, on_completion);

    auto worker = [&](int id) {
        for (int i = 0; i < num_iterations; ++i) {
            std::println("Worker {} phase {}", id, i);
            std::this_thread::sleep_for(std::chrono::milliseconds(50 * id));
            sync_point.arrive_and_wait();
        }
    };

    std::vector<std::jthread> workers;
    for (int i = 0; i < num_workers; ++i) {
        workers.emplace_back(worker, i);
    }
}

int main() {
    std::println("=== Latch Example ===");
    latch_example();

    std::println("\n=== Barrier Example ===");
    barrier_example();

    return 0;
}
```

---

Last Updated: 2026-01-10
Version: 1.0.0
