#!/bin/bash
# TYPO3 DDEV Prerequisites Validation Script
# Auto-generated by enhanced typo3-ddev skill
#
# This script validates all prerequisites for DDEV TYPO3 development:
# - Docker daemon running
# - Docker CLI version >= 20.10
# - Docker Compose version >= 2.0
# - DDEV installed
# - TYPO3 extension project structure

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo "🔍 Validating TYPO3 DDEV Prerequisites..."
echo ""

FAILED=0

# 1. Docker daemon
echo -n "Checking Docker daemon... "
if docker info >/dev/null 2>&1; then
    echo -e "${GREEN}✅ Running${NC}"
else
    echo -e "${RED}❌ Not running${NC}"
    echo ""
    echo "Docker daemon is not running. Start it with:"
    echo "  Linux/WSL2: sudo service docker start"
    echo "  macOS: Open Docker Desktop"
    echo "  Windows: Open Docker Desktop"
    FAILED=1
fi

# 2. Docker version
echo -n "Checking Docker CLI version... "
DOCKER_VERSION=$(docker version --format '{{.Client.Version}}' 2>/dev/null || echo "")
if [ -n "$DOCKER_VERSION" ]; then
    MAJOR=$(echo "$DOCKER_VERSION" | cut -d. -f1)
    MINOR=$(echo "$DOCKER_VERSION" | cut -d. -f2)
    if [ "$MAJOR" -gt 20 ] || ([ "$MAJOR" -eq 20 ] && [ "$MINOR" -ge 10 ]); then
        echo -e "${GREEN}✅ $DOCKER_VERSION (>= 20.10)${NC}"
    else
        echo -e "${YELLOW}⚠️  $DOCKER_VERSION (need >= 20.10)${NC}"
        echo "  Update Docker to version 20.10 or newer"
        FAILED=1
    fi
else
    echo -e "${RED}❌ Not installed${NC}"
    FAILED=1
fi

# 3. Docker Compose version
echo -n "Checking Docker Compose version... "
COMPOSE_VERSION=$(docker compose version --short 2>/dev/null || echo "")
if [ -n "$COMPOSE_VERSION" ]; then
    COMPOSE_MAJOR=$(echo "$COMPOSE_VERSION" | cut -d. -f1)
    if [ "$COMPOSE_MAJOR" -ge 2 ]; then
        echo -e "${GREEN}✅ $COMPOSE_VERSION (>= 2.0)${NC}"
    else
        echo -e "${YELLOW}⚠️  $COMPOSE_VERSION (need >= 2.0)${NC}"
        echo "  Update to Docker Compose v2"
        FAILED=1
    fi
else
    echo -e "${RED}❌ Not installed${NC}"
    echo "  Install Docker Compose v2 (included with Docker 20.10+)"
    FAILED=1
fi

# 4. DDEV
echo -n "Checking DDEV installation... "
if command -v ddev >/dev/null 2>&1; then
    DDEV_VERSION=$(ddev version 2>&1 | head -n1 | grep -oP 'v\K[0-9.]+' || echo "unknown")
    echo -e "${GREEN}✅ v$DDEV_VERSION${NC}"
else
    echo -e "${RED}❌ Not installed${NC}"
    echo ""
    echo "Install DDEV:"
    echo "  macOS: brew install ddev/ddev/ddev"
    echo "  Linux: curl -fsSL https://raw.githubusercontent.com/ddev/ddev/master/scripts/install_ddev.sh | bash"
    echo "  Windows: choco install ddev"
    FAILED=1
fi

# 5. TYPO3 Extension (only check if in a directory, not fatal)
echo -n "Checking TYPO3 extension project... "
if [ -f "ext_emconf.php" ] || grep -q '"type".*"typo3-cms-extension"' composer.json 2>/dev/null; then
    echo -e "${GREEN}✅ Detected${NC}"
else
    echo -e "${YELLOW}⚠️  Not detected (optional check)${NC}"
    echo "  This check only matters if you're in an extension directory"
fi

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

if [ $FAILED -eq 0 ]; then
    echo -e "${GREEN}✅ All prerequisites validated successfully!${NC}"
    echo ""
    echo "You can now run DDEV commands:"
    echo "  ddev start"
    echo "  ddev install-all"
    exit 0
else
    echo -e "${RED}❌ Prerequisites validation failed${NC}"
    echo ""
    echo "Please resolve the issues above before proceeding."
    echo "For detailed instructions, see:"
    echo "  ~/.claude/plugins/marketplaces/netresearch-claude-code-marketplace/skills/typo3-ddev/SKILL.md"
    exit 1
fi
