# Quality Tools for TYPO3 Development

Automated code quality and static analysis tools for TYPO3 extensions.

## Overview

- **PHPStan**: Static analysis for type safety and bugs
- **Rector**: Automated code refactoring and modernization
- **php-cs-fixer**: Code style enforcement (PSR-12, TYPO3 CGL)
- **phplint**: PHP syntax validation

## PHPStan

### Installation

```bash
composer require --dev phpstan/phpstan phpstan/phpstan-strict-rules saschaegerer/phpstan-typo3
```

### Configuration

Create `Build/phpstan.neon`:

```neon
includes:
    - vendor/phpstan/phpstan-strict-rules/rules.neon
    - vendor/saschaegerer/phpstan-typo3/extension.neon

parameters:
    level: max  # Level 10 - maximum strictness
    paths:
        - Classes
        - Tests
    excludePaths:
        - Tests/Acceptance/_output/*
    reportUnmatchedIgnoredErrors: true
    checkGenericClassInNonGenericObjectType: false
    checkMissingIterableValueType: false
```

### Running PHPStan

```bash
# Via runTests.sh
Build/Scripts/runTests.sh -s phpstan

# Directly
vendor/bin/phpstan analyze --configuration Build/phpstan.neon

# With baseline (ignore existing errors)
vendor/bin/phpstan analyze --generate-baseline Build/phpstan-baseline.neon

# Clear cache
vendor/bin/phpstan clear-result-cache
```

### PHPStan Rule Levels

**Level 0-10** (use `max` for level 10): Increasing strictness
- **Level 0**: Basic checks (undefined variables, unknown functions)
- **Level 5**: Type checks, unknown properties, unknown methods
- **Level 9**: Strict mixed types, unused parameters
- **Level 10 (max)**: Maximum strictness - explicit mixed types, pure functions

**Recommendation**:
- **New projects**: Start with level 5, aim for level 10 (max)
- **Existing extensions**: Level 8 is practical - levels 9/10 require extensive type annotations for `$GLOBALS`, TCA, and dynamic TYPO3 patterns

**Why Level 8 for existing extensions?**
- Strict boolean conditions and nullability checks
- Avoids excessive ignoreErrors for TYPO3's inherently untyped patterns
- Good balance between strictness and maintainability

**Why Level 10 for new projects?**
- Enforces explicit type declarations (`mixed` must be declared, not implicit)
- Catches more potential bugs at development time
- Aligns with TYPO3 13 strict typing standards (`declare(strict_types=1)`)
- Required for PHPStan Level 10 compliant extensions

### Ignoring Errors

```php
/** @phpstan-ignore-next-line */
$value = $this->legacyMethod();

// Or in neon file
parameters:
    ignoreErrors:
        - '#Call to an undefined method.*::getRepository\(\)#'
```

### TYPO3-Specific ignoreErrors (Level 8)

For existing TYPO3 extensions, these ignoreErrors handle common TYPO3 patterns:

```neon
parameters:
    level: 8
    ignoreErrors:
        # TYPO3 TCA/GLOBALS access patterns - inherently untyped
        - '#Cannot access offset .* on mixed#'
        - '#Parameter .* of function array_key_exists expects array, mixed given#'
        - '#Parameter .* of function array_merge expects array, mixed given#'
        - '#Parameter .* of function in_array expects array, mixed given#'
        - '#Argument of an invalid type mixed supplied for foreach#'
        - '#Cannot cast mixed to int#'
        - '#Cannot cast mixed to string#'
        - '#Possibly invalid array key type#'

        # Legacy code array type specifications
        - '#no value type specified in iterable type array#'
        - '#return type has no value type specified in iterable type#'
        - '#type has no value type specified in iterable type#'

        # TYPO3 v12/v13 API changes - during migration
        - '#deprecated class TYPO3\\CMS\\Frontend\\Controller\\TypoScriptFrontendController#'
        - '#Call to an undefined method TYPO3\\CMS\\Core\\Database\\Query\\QueryBuilder::execute#'

        # Doctrine DBAL 4.x type parameter changes (int → ParameterType enum)
        - '~Parameter \\#2 \\$type of method .* expects .*, int given~'

        # PHPStan strict rules violations in legacy code
        - '#Construct empty\\(\\) is not allowed#'
        - '#Strict comparison using .* will always evaluate to#'
```

### CI Workflow Paths with Build/ Configuration

When configs are in Build/, update CI workflows:

```yaml
# .github/workflows/ci.yml
- name: Run PHPStan
  run: vendor/bin/phpstan analyse -c Build/phpstan.neon --no-progress

- name: Run PHP-CS-Fixer
  run: vendor/bin/php-cs-fixer fix --config=Build/php-cs-fixer.php --dry-run --diff

- name: Run PHPCS
  run: vendor/bin/phpcs --standard=Build/phpcs.xml
```

**Path resolution note**: PHPStan's `paths:` and `includes:` are resolved relative to the config file location. When config is in Build/:

```neon
# Build/phpstan.neon
includes:
    - ../vendor/phpstan/phpstan-strict-rules/rules.neon  # ← Note ../
parameters:
    paths:
        - ../Classes/     # ← Note ../
        - ../Tests/
    excludePaths:
        - ../vendor/*
        - ../.Build/*
```

### PHPStan in Tests - Common Patterns

When writing tests that validate runtime behavior guaranteed by PHPDoc types, PHPStan Level 9+ may report "alreadyNarrowedType" errors. These tests are still valuable as they verify implementation matches type declarations.

**Common Test-Specific Ignore Identifiers:**

| Identifier | When to Use |
|------------|-------------|
| `staticMethod.alreadyNarrowedType` | `assertTrue()`, `assertFalse()`, `assertIsArray()` when PHPStan knows the result |
| `function.alreadyNarrowedType` | `is_subclass_of()`, `is_array()`, `is_string()` when type is known from PHPDoc |

**Example - Testing Contract Guarantees:**

```php
#[Test]
public function allDiscoveredClassesExtendBaseClass(): void
{
    $registry = new MatcherRegistry();
    $result = $registry->getMatcherClasses(); // Returns array<class-string<AbstractCoreMatcher>>

    foreach ($result as $matcherClass) {
        // PHPStan knows this is always true from PHPDoc, but test validates runtime behavior
        // @phpstan-ignore staticMethod.alreadyNarrowedType
        self::assertTrue(
            is_subclass_of($matcherClass, AbstractCoreMatcher::class), // @phpstan-ignore function.alreadyNarrowedType
            sprintf('%s should extend AbstractCoreMatcher', $matcherClass)
        );
    }
}

#[Test]
public function allConfigurationsAreArrays(): void
{
    $registry = new MatcherRegistry();
    $configurations = $registry->getMatcherConfigurations(); // Returns array<class-string, array<string, mixed>>

    foreach ($configurations as $matcherClass => $configuration) {
        // @phpstan-ignore staticMethod.alreadyNarrowedType
        self::assertIsArray(
            $configuration,
            sprintf('Configuration for %s should be an array', $matcherClass)
        );
    }
}
```

**When to Use These Ignores:**
- ✅ Tests validating that implementation matches PHPDoc contracts
- ✅ Tests checking class hierarchies or type relationships
- ✅ Tests ensuring configuration structures are correct
- ❌ Production code (fix the types instead)
- ❌ Tests where the assertion actually could fail

**Placement Rules:**
- **Next-line comment** (`// @phpstan-ignore ...`): Applies to the **next** line
- **Inline comment**: Applies to the **same** line where it appears
- Multiple identifiers: Separate with comma (`// @phpstan-ignore id1, id2`)

### TYPO3-Specific Rules

```php
// PHPStan understands TYPO3 classes
$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
    ->getQueryBuilderForTable('pages');
// ✅ PHPStan knows this returns QueryBuilder

// Detects TYPO3 API misuse
TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(MyService::class);
// ✅ Checks if MyService is a valid class
```

## Rector

### Installation

```bash
composer require --dev rector/rector ssch/typo3-rector
```

### Configuration

Create `rector.php`:

```php
<?php

declare(strict_types=1);

use Rector\Config\RectorConfig;
use Rector\Set\ValueObject\LevelSetList;
use Rector\Set\ValueObject\SetList;
use Ssch\TYPO3Rector\Set\Typo3SetList;

return RectorConfig::configure()
    ->withPaths([
        __DIR__ . '/Classes',
        __DIR__ . '/Tests',
    ])
    ->withSkip([
        __DIR__ . '/Tests/Acceptance/_output',
    ])
    ->withPhpSets(php82: true)
    ->withSets([
        LevelSetList::UP_TO_PHP_82,
        SetList::CODE_QUALITY,
        SetList::DEAD_CODE,
        SetList::TYPE_DECLARATION,
        Typo3SetList::TYPO3_13,
    ]);
```

### Running Rector

```bash
# Dry run (show changes)
vendor/bin/rector process --dry-run

# Apply changes
vendor/bin/rector process

# Via runTests.sh
Build/Scripts/runTests.sh -s rector
```

### Common Refactorings

**TYPO3 API Modernization**:
```php
// Before
$GLOBALS['TYPO3_DB']->exec_SELECTgetRows('*', 'pages', 'uid=1');

// After (Rector auto-refactors)
GeneralUtility::makeInstance(ConnectionPool::class)
    ->getConnectionForTable('pages')
    ->select(['*'], 'pages', ['uid' => 1])
    ->fetchAllAssociative();
```

**Type Declarations**:
```php
// Before
public function process($data)
{
    return $data;
}

// After
public function process(array $data): array
{
    return $data;
}
```

## php-cs-fixer

### Installation

```bash
composer require --dev friendsofphp/php-cs-fixer
```

### Configuration

Create `Build/php-cs-fixer.php`:

```php
<?php

declare(strict_types=1);

$finder = (new PhpCsFixer\Finder())
    ->in(__DIR__ . '/../Classes')
    ->in(__DIR__ . '/../Tests')
    ->exclude('_output');

return (new PhpCsFixer\Config())
    ->setRules([
        '@PSR12' => true,
        '@PhpCsFixer' => true,
        'array_syntax' => ['syntax' => 'short'],
        'concat_space' => ['spacing' => 'one'],
        'declare_strict_types' => true,
        'ordered_imports' => ['sort_algorithm' => 'alpha'],
        'no_unused_imports' => true,
        'single_line_throw' => false,
        'phpdoc_align' => false,
        'phpdoc_no_empty_return' => false,
        'phpdoc_summary' => false,
    ])
    ->setRiskyAllowed(true)
    ->setFinder($finder);
```

### Running php-cs-fixer

```bash
# Check only (dry run)
vendor/bin/php-cs-fixer fix --config Build/php-cs-fixer.php --dry-run --diff

# Fix files
vendor/bin/php-cs-fixer fix --config Build/php-cs-fixer.php

# Via runTests.sh
Build/Scripts/runTests.sh -s cgl
```

### Common Rules

```php
// array_syntax: short
$array = [1, 2, 3]; // ✅
$array = array(1, 2, 3); // ❌

// concat_space: one
$message = 'Hello ' . $name; // ✅
$message = 'Hello '.$name; // ❌

// declare_strict_types
<?php

declare(strict_types=1); // ✅ Required at top of file

// ordered_imports
use Vendor\Extension\Domain\Model\Product; // ✅ Alphabetical
use Vendor\Extension\Domain\Repository\ProductRepository;
```

## phplint

### Installation

```bash
composer require --dev overtrue/phplint
```

### Configuration

Create `.phplint.yml`:

```yaml
path: ./
jobs: 10
cache: var/cache/phplint.cache
exclude:
    - vendor
    - var
    - .Build
extensions:
    - php
```

### Running phplint

```bash
# Lint all PHP files
vendor/bin/phplint

# Via runTests.sh
Build/Scripts/runTests.sh -s lint

# Specific directory
vendor/bin/phplint Classes/
```

## Composer Script Integration

```json
{
    "scripts": {
        "ci:test:php:lint": "phplint",
        "ci:test:php:phpstan": "phpstan analyze --configuration Build/phpstan.neon --no-progress",
        "ci:test:php:rector": "rector process --dry-run",
        "ci:test:php:cgl": "php-cs-fixer fix --config Build/php-cs-fixer.php --dry-run --diff",
        "ci:test:php:security": "composer audit",

        "fix:cgl": "php-cs-fixer fix --config Build/php-cs-fixer.php",
        "fix:rector": "rector process",

        "ci:test": [
            "@ci:test:php:lint",
            "@ci:test:php:phpstan",
            "@ci:test:php:rector",
            "@ci:test:php:cgl",
            "@ci:test:php:security"
        ]
    }
}
```

> **Security Note**: `composer audit` checks for known security vulnerabilities in dependencies. Run this regularly and especially before releases.

## Pre-commit Hook

Create `.git/hooks/pre-commit`:

```bash
#!/bin/sh

echo "Running quality checks..."

# Lint
vendor/bin/phplint || exit 1

# PHPStan
vendor/bin/phpstan analyze --configuration Build/phpstan.neon --error-format=table --no-progress || exit 1

# Code style
vendor/bin/php-cs-fixer fix --config Build/php-cs-fixer.php --dry-run --diff || exit 1

echo "✓ All checks passed"
```

## CI/CD Integration

### GitHub Actions

```yaml
quality:
  runs-on: ubuntu-latest
  steps:
    - uses: actions/checkout@v4
    - uses: shivammathur/setup-php@v2
      with:
        php-version: '8.4'  # Use latest PHP for quality tools
    - run: composer install
    - run: composer ci:test:php:lint
    - run: composer ci:test:php:phpstan
    - run: composer ci:test:php:cgl
    - run: composer ci:test:php:rector
    - run: composer ci:test:php:security
```

## IDE Integration

### PHPStorm

1. **PHPStan**: Settings → PHP → Quality Tools → PHPStan
2. **php-cs-fixer**: Settings → PHP → Quality Tools → PHP CS Fixer
3. **File Watchers**: Auto-run on file save

### VS Code

```json
{
    "php.validate.executablePath": "/usr/bin/php",
    "phpstan.enabled": true,
    "phpstan.configFile": "Build/phpstan.neon",
    "php-cs-fixer.onsave": true,
    "php-cs-fixer.config": "Build/php-cs-fixer.php"
}
```

## Best Practices

1. **PHPStan Level 10**: Aim for `level: max` in modern TYPO3 13 projects
2. **Baseline for Legacy**: Use baselines to track existing issues during migration
3. **Security Audits**: Run `composer audit` regularly and in CI
4. **Auto-fix in CI**: Run fixes automatically, fail on violations
5. **Consistent Rules**: Share config across team
6. **Pre-commit Checks**: Catch issues before commit (lint, PHPStan, CGL, security)
7. **Latest PHP**: Run quality tools with latest PHP version (8.4+)
8. **Regular Updates**: Keep tools and rules updated

## Resources

- [PHPStan Documentation](https://phpstan.org/user-guide/getting-started)
- [Rector Documentation](https://getrector.com/documentation)
- [PHP CS Fixer Documentation](https://github.com/PHP-CS-Fixer/PHP-CS-Fixer)
- [TYPO3 Coding Guidelines](https://docs.typo3.org/m/typo3/reference-coreapi/main/en-us/CodingGuidelines/)
