# API Reference

## Core Modules

### skill_loader

Load Claude skills from any location.

```python
from src.core.skill_loader import SkillLoader, Skill

# Load a skill
loader = SkillLoader()
skill = loader.load("~/.claude/skills/pptx")

print(skill.name)          # "pptx"
print(skill.description)   # Skill description
print(skill.instructions)  # Full SKILL.md content
print(skill.references)    # List of reference files
print(skill.scripts)       # List of script files
```

### claude_executor

Execute tasks using Claude Code CLI.

```python
from src.core.claude_executor import ClaudeCodeExecutor, ExecutorConfig

# Create executor
config = ExecutorConfig(timeout=300, save_traces=True)
executor = ClaudeCodeExecutor(config)

# Execute a task
trace = executor.execute(
    prompt="Create a presentation",
    task_id="task_001",
    skill_path="~/.claude/skills/pptx"
)

print(trace.success)        # True/False
print(trace.execution_time) # Seconds
print(trace.files_created)  # List of created files
print(trace.errors)         # List of errors
```

### skill_optimizer

Main optimization loop.

```python
from src.core.skill_optimizer import (
    SkillOptimizer,
    OptimizationConfig,
    load_test_cases
)

# Load config and tests
config = OptimizationConfig.from_yaml("config.yaml")
tests = load_test_cases("test_cases.yaml")

# Run optimization
optimizer = SkillOptimizer(config, tests)
result = optimizer.optimize()

print(result.improvement)
print(result.best_candidate.id)
```

## Evaluators

### Base Classes

```python
from src.evaluators.base import BaseEvaluator, EvaluationResult

class MyEvaluator(BaseEvaluator):
    def evaluate(self, trace, test_case=None):
        return EvaluationResult(
            metric_name=self.name,
            score=0.85,
            details={},
            issues=[]
        )
```

### Built-in Evaluators

| Evaluator | Type | Description |
|-----------|------|-------------|
| `BinaryEvaluator` | binary | Success/failure |
| `FileExistsEvaluator` | file_exists | Check file patterns |
| `FileValidityEvaluator` | file_validity | Validate file formats |
| `ContentMatchEvaluator` | content_match | Check expected content |
| `ErrorRateEvaluator` | error_rate | Score based on errors |
| `EfficiencyEvaluator` | efficiency | Execution time |
| `CompositeEvaluator` | composite | Combine multiple |

## Generators

### test_generator

Generate test cases automatically.

```python
from src.generators.test_generator import TestCaseGenerator
from src.core.skill_loader import load_skill

skill = load_skill("~/.claude/skills/pptx")
generator = TestCaseGenerator(skill)

tests = generator.generate(count=20, include_edge_cases=True)
generator.save_to_yaml(tests, "test_cases.yaml")
```

## CLI Commands

### skill-optimizer

```bash
# Initialize
skill-optimizer init --skill-path PATH --output-dir DIR

# Optimize
skill-optimizer optimize --config CONFIG [--max-iterations N] [--mock]

# Evaluate
skill-optimizer evaluate --skill-path PATH --test-cases FILE
```

### skill-test-gen

```bash
skill-test-gen --skill-path PATH --output FILE --count N
```
