# PPTX Brand Compliance Optimization

This example optimizes Claude's ability to create **brand-compliant presentations** that follow corporate identity guidelines.

## 🎯 What This Optimizes

| Aspect | Description |
|--------|-------------|
| **Color Accuracy** | Using exact brand colors (hex/RGB values) |
| **Typography** | Correct fonts, weights, and sizes per brand guide |
| **Logo Usage** | Proper placement, sizing, clear space, and versions |
| **Visual Elements** | Icons, charts, backgrounds following brand style |
| **Tone & Voice** | Content style matching brand personality |

## 🎨 Brand Guidelines Example

```yaml
# brand_config.yaml
brand:
  name: "Acme Corporation"
  
  colors:
    primary:
      name: "Acme Navy"
      hex: "#1E3A5F"
      usage: "Headlines, primary backgrounds"
    secondary:
      name: "Acme Gold"
      hex: "#D4AF37"
      usage: "Accents, highlights, CTAs"
    tertiary:
      name: "Slate Gray"
      hex: "#64748B"
      usage: "Body text, secondary elements"
    background:
      light: "#FFFFFF"
      dark: "#1E3A5F"
      
  typography:
    heading:
      font: "Montserrat"
      fallback: "Arial"
      weights: [600, 700]
      sizes:
        title: 44
        h1: 36
        h2: 28
        h3: 24
    body:
      font: "Open Sans"
      fallback: "Calibri"
      weights: [400, 600]
      sizes:
        normal: 18
        small: 14
        caption: 12
        
  logo:
    primary: "acme_logo_full.svg"
    icon: "acme_icon.svg"
    white: "acme_logo_white.svg"
    min_size: 100  # pixels
    clear_space: 20  # pixels around logo
    placement:
      title_slide: "center"
      content_slides: "bottom-right"
      
  visual_style:
    corners: "rounded"  # or "square"
    shadows: "subtle"
    gradients: false
    photography_style: "professional, diverse, authentic"
    icon_style: "outlined"
```

## 📁 Files

```
brand-compliance/
├── README.md                 # This file
├── config.yaml              # Optimization configuration
├── test_cases.yaml          # 25 brand compliance tests
├── run_optimization.py      # Runner script
├── brand_config.yaml        # Brand guidelines specification
├── evaluators/
│   └── brand_evaluator.py   # Custom brand compliance evaluator
└── assets/
    └── logos/               # Sample brand assets
```

## 🚀 Quick Start

```bash
# 1. Define your brand (edit brand_config.yaml)
vim brand_config.yaml

# 2. Run optimization
python run_optimization.py

# 3. Review compliance report
cat results/brand_compliance_report.md
```

## 📊 Test Categories

### Color Compliance (6 cases)
- Exact hex color usage
- No off-brand colors
- Proper color for context (dark bg vs light bg)
- Chart/graph color schemes
- Gradient restrictions
- Accessibility contrast ratios

### Typography Compliance (5 cases)
- Correct font families
- Proper weight usage
- Size hierarchy adherence
- Line spacing standards
- Text alignment rules

### Logo Usage (5 cases)
- Correct logo version selection
- Minimum size requirements
- Clear space around logo
- Placement consistency
- Co-branding scenarios

### Visual Elements (5 cases)
- Icon style consistency
- Image treatment
- Shape/corner styling
- Chart styling
- Background patterns

### Content Style (4 cases)
- Headline capitalization
- Call-to-action phrasing
- Disclaimers and legal text
- Contact information format

## ⚙️ Custom Evaluator

```python
# evaluators/brand_evaluator.py
class BrandComplianceEvaluator(BaseEvaluator):
    """Evaluates brand guideline compliance."""
    
    def __init__(self, brand_config: dict, **kwargs):
        super().__init__(**kwargs)
        self.brand = brand_config
        
    def evaluate(self, trace, test_case=None):
        pptx_path = self._find_output_pptx(trace)
        
        scores = {
            "color_compliance": self._check_colors(pptx_path),
            "typography_compliance": self._check_fonts(pptx_path),
            "logo_usage": self._check_logo(pptx_path),
            "visual_style": self._check_visuals(pptx_path),
        }
        
        issues = []
        
        # Check for off-brand colors
        off_brand_colors = self._find_off_brand_colors(pptx_path)
        if off_brand_colors:
            issues.append(f"Off-brand colors found: {off_brand_colors[:3]}")
            scores["color_compliance"] *= 0.7
            
        # Check for wrong fonts
        wrong_fonts = self._find_wrong_fonts(pptx_path)
        if wrong_fonts:
            issues.append(f"Non-brand fonts: {wrong_fonts[:3]}")
            scores["typography_compliance"] *= 0.7
        
        overall = sum(scores.values()) / len(scores)
        
        return EvaluationResult(
            metric_name="brand_compliance",
            score=overall,
            details=scores,
            issues=issues
        )
    
    def _check_colors(self, pptx_path: str) -> float:
        """Check all colors against brand palette."""
        from pptx import Presentation
        from pptx.util import Inches
        from pptx.dml.color import RGBColor
        
        prs = Presentation(pptx_path)
        brand_colors = self._get_brand_colors()
        
        total_colors = 0
        matching_colors = 0
        
        for slide in prs.slides:
            for shape in slide.shapes:
                colors = self._extract_colors(shape)
                for color in colors:
                    total_colors += 1
                    if self._is_brand_color(color, brand_colors):
                        matching_colors += 1
        
        return matching_colors / max(total_colors, 1)
    
    def _is_brand_color(self, color: str, brand_colors: list) -> bool:
        """Check if color matches brand palette (with tolerance)."""
        if not color:
            return True  # No color = theme color = OK
            
        # Allow 5% tolerance for color matching
        tolerance = 13  # ~5% of 255
        
        for brand_color in brand_colors:
            if self._colors_similar(color, brand_color, tolerance):
                return True
        return False
```

## 📈 Expected Improvements

| Metric | Before | After | Improvement |
|--------|--------|-------|-------------|
| Color Compliance | 60% | 92% | +32% |
| Typography | 70% | 95% | +25% |
| Logo Usage | 55% | 85% | +30% |
| Visual Consistency | 65% | 88% | +23% |
| **Overall Brand Score** | **63%** | **90%** | **+27%** |

## 🔍 Common Issues Fixed

### Before Optimization
```
❌ Hardcoded colors (#0000FF instead of brand blue)
❌ Default fonts (Calibri instead of Montserrat)
❌ Logo stretched or too small
❌ Inconsistent spacing and alignment
❌ Charts using Excel default colors
```

### After Optimization
```
✅ Brand colors used throughout
✅ Correct typography hierarchy
✅ Logo properly placed with clear space
✅ Consistent visual grid
✅ Charts match brand palette
```

## 📝 Sample Test Case

```yaml
- id: brand_color_001
  description: "Use exact brand colors for chart"
  prompt: |
    Create a sales performance slide with a bar chart showing
    quarterly results. The chart should use only brand colors:
    - Primary (Acme Navy #1E3A5F) for main bars
    - Secondary (Acme Gold #D4AF37) for highlights
    - Tertiary (Slate Gray #64748B) for secondary data
    
    Do NOT use default chart colors.
    
  brand_requirements:
    colors:
      must_use: ["#1E3A5F", "#D4AF37"]
      forbidden: ["#4472C4", "#ED7D31", "#A5A5A5"]  # Excel defaults
      
  quality_criteria:
    - "Chart uses brand colors only"
    - "No default blue/orange Excel colors"
    - "Color usage follows hierarchy (primary > secondary)"
    
  tags: [brand, colors, charts]
  complexity: medium
```

## 🛠️ Integration with Template Workflow

Combine brand compliance with template workflow:

```yaml
# Combined config
skill:
  path: ~/.claude/skills/pptx

optimization:
  optimize:
    - template_workflow      # From template example
    - brand_compliance       # From this example
    
evaluation:
  metrics:
    - name: template_compliance
      weight: 0.25
    - name: brand_compliance
      weight: 0.35
    - name: visual_quality
      weight: 0.25
    - name: task_completion
      weight: 0.15
```

## 📚 Related Examples

- [Template Workflow](../template-workflow/) - Use with brand templates
- [Image Embedding](../image-embedding/) - Brand-compliant images
- [Charts & Tables](../charts-tables/) - Data viz with brand colors
