# PPTX Charts & Tables Optimization

This example optimizes Claude's ability to create **professional data visualizations** including charts, graphs, and tables in PowerPoint.

## 🎯 What This Optimizes

| Aspect | Description |
|--------|-------------|
| **Chart Type Selection** | Choosing appropriate visualization for data |
| **Data Accuracy** | Correctly representing values and labels |
| **Styling** | Colors, fonts, legends matching brand/template |
| **Readability** | Proper sizing, labeling, contrast |
| **Table Formatting** | Headers, alignment, borders, alternating rows |

## 📊 Supported Visualizations

### Charts
| Type | Best For |
|------|----------|
| Bar Chart | Comparing categories |
| Column Chart | Time series, categories |
| Line Chart | Trends over time |
| Pie Chart | Parts of a whole (≤6 segments) |
| Donut Chart | Parts of whole with center metric |
| Area Chart | Cumulative trends |
| Scatter Plot | Correlation analysis |
| Combo Chart | Multiple data types |

### Tables
| Style | Use Case |
|-------|----------|
| Simple | Basic data display |
| Striped | Long tables, readability |
| Bordered | Financial data, precision |
| Minimal | Clean, modern look |
| Comparison | Side-by-side analysis |

## 📁 Files

```
charts-tables/
├── README.md                 # This file
├── config.yaml              # Optimization configuration
├── test_cases.yaml          # 25 chart/table tests
├── run_optimization.py      # Runner script
├── evaluators/
│   ├── chart_evaluator.py   # Chart quality evaluator
│   └── table_evaluator.py   # Table quality evaluator
└── assets/
    └── sample_data/         # CSV/JSON test data
```

## 📊 Test Categories

### Chart Type Selection (5 cases)
- Bar vs Column decision
- When to use Pie vs Bar
- Line vs Area for trends
- Combo chart scenarios
- Small multiples

### Data Accuracy (5 cases)
- Correct values plotted
- Axis scaling appropriate
- Labels match data
- Legend accuracy
- Data point precision

### Chart Styling (5 cases)
- Brand colors applied
- Font consistency
- Legend placement
- Gridline styling
- Title formatting

### Table Quality (5 cases)
- Header formatting
- Cell alignment
- Number formatting
- Alternating rows
- Border styling

### Complex Scenarios (5 cases)
- Dashboard with multiple charts
- Chart + table combination
- Real-time data placeholders
- Interactive chart notes
- Responsive sizing

## ⚙️ Configuration

```yaml
# config.yaml
skill:
  path: ~/.claude/skills/pptx

optimization:
  max_iterations: 15
  optimize:
    - chart_type_selection
    - data_binding_logic
    - chart_styling_rules
    - table_formatting_rules
    - legend_placement

evaluation:
  metrics:
    - name: chart_type_appropriateness
      weight: 0.25
      type: custom
      evaluator: chart_evaluator.ChartTypeEvaluator
      
    - name: data_accuracy
      weight: 0.30
      type: custom
      evaluator: chart_evaluator.DataAccuracyEvaluator
      
    - name: visual_quality
      weight: 0.25
      type: visual
      checks:
        - readability
        - color_contrast
        - label_clarity
        
    - name: styling_compliance
      weight: 0.20
      type: custom

chart_defaults:
  colors:
    - "#1E3A5F"  # Primary
    - "#D4AF37"  # Accent
    - "#64748B"  # Secondary
    - "#059669"  # Success
    - "#DC2626"  # Error
    
  fonts:
    title: 14
    labels: 10
    legend: 9
    
  legend:
    position: "bottom"  # or "right", "top", "none"
    
table_defaults:
  header:
    background: "#1E3A5F"
    text_color: "#FFFFFF"
    font_weight: "bold"
    
  rows:
    alternating: true
    even_color: "#F8FAFC"
    odd_color: "#FFFFFF"
    
  alignment:
    text: "left"
    numbers: "right"
    dates: "center"
```

## 📝 Sample Test Cases

```yaml
test_cases:
  # ============================================================
  # CHART TYPE SELECTION
  # ============================================================
  
  - id: chart_type_001
    description: "Select appropriate chart for category comparison"
    prompt: |
      Create a chart showing Q4 sales by region:
      - North America: $2.5M
      - Europe: $1.8M
      - Asia Pacific: $2.1M
      - Latin America: $0.9M
      
      Choose the most appropriate chart type for comparing
      these discrete categories.
    
    expected_chart_type: "bar"  # or column
    data:
      categories: ["North America", "Europe", "Asia Pacific", "Latin America"]
      values: [2.5, 1.8, 2.1, 0.9]
      
    quality_criteria:
      - "Uses bar or column chart (not pie for 4+ categories)"
      - "Bars clearly differentiated"
      - "Values labeled or axis scaled correctly"
      
    tags: [chart_type, comparison]
    complexity: simple

  - id: chart_type_002
    description: "Time series trend visualization"
    prompt: |
      Create a chart showing monthly revenue trend for 2024:
      Jan: $1.2M, Feb: $1.1M, Mar: $1.4M, Apr: $1.3M,
      May: $1.5M, Jun: $1.7M, Jul: $1.6M, Aug: $1.8M,
      Sep: $2.0M, Oct: $1.9M, Nov: $2.2M, Dec: $2.5M
      
      Show the trend clearly over time.
    
    expected_chart_type: "line"
    data:
      labels: ["Jan", "Feb", "Mar", "Apr", "May", "Jun", 
               "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]
      values: [1.2, 1.1, 1.4, 1.3, 1.5, 1.7, 1.6, 1.8, 2.0, 1.9, 2.2, 2.5]
      
    quality_criteria:
      - "Uses line chart for time series"
      - "X-axis shows time progression"
      - "Trend clearly visible"
      
    tags: [chart_type, trend, time_series]
    complexity: simple

  # ============================================================
  # DATA ACCURACY
  # ============================================================
  
  - id: data_accuracy_001
    description: "Verify chart data matches source"
    prompt: |
      Create a pie chart showing market share:
      - Our Product: 35%
      - Competitor A: 28%
      - Competitor B: 22%
      - Others: 15%
      
      Values must total exactly 100%.
    
    data:
      labels: ["Our Product", "Competitor A", "Competitor B", "Others"]
      values: [35, 28, 22, 15]
      expected_total: 100
      
    quality_criteria:
      - "All percentages accurately represented"
      - "Segments proportional to values"
      - "Labels match data"
      
    tags: [data_accuracy, pie_chart]
    complexity: medium

  # ============================================================
  # TABLE FORMATTING
  # ============================================================
  
  - id: table_format_001
    description: "Create formatted financial table"
    prompt: |
      Create a quarterly financial summary table:
      
      | Metric      | Q1      | Q2      | Q3      | Q4      |
      |-------------|---------|---------|---------|---------|
      | Revenue     | $2.1M   | $2.3M   | $2.5M   | $2.8M   |
      | Expenses    | $1.8M   | $1.9M   | $2.0M   | $2.1M   |
      | Profit      | $0.3M   | $0.4M   | $0.5M   | $0.7M   |
      | Margin      | 14.3%   | 17.4%   | 20.0%   | 25.0%   |
      
      Format:
      - Header row with dark background
      - Numbers right-aligned
      - Currency properly formatted
      - Alternating row colors
    
    quality_criteria:
      - "Header styled distinctly"
      - "Numbers right-aligned"
      - "Currency formatting consistent"
      - "Alternating rows for readability"
      
    tags: [table, financial, formatting]
    complexity: medium

  # ============================================================
  # COMPLEX SCENARIOS
  # ============================================================
  
  - id: complex_dashboard_001
    description: "Create KPI dashboard with multiple visualizations"
    prompt: |
      Create a dashboard slide with:
      
      1. Top row: 4 KPI boxes
         - Revenue: $10.2M (↑ 15%)
         - Customers: 15,420 (↑ 8%)
         - NPS Score: 72 (↑ 5)
         - Churn: 2.3% (↓ 0.5%)
      
      2. Bottom left: Line chart showing 12-month revenue trend
      
      3. Bottom right: Pie chart showing revenue by product:
         - Product A: 45%
         - Product B: 30%
         - Product C: 25%
      
      All visualizations should use brand colors and be
      properly sized to fit together.
    
    quality_criteria:
      - "All 4 KPIs visible and formatted"
      - "Trend indicators (arrows) correct direction"
      - "Charts properly sized"
      - "Consistent styling throughout"
      - "No overlapping elements"
      
    tags: [complex, dashboard, multiple_charts]
    complexity: complex
```

## 🔧 Custom Evaluators

```python
# evaluators/chart_evaluator.py
class ChartTypeEvaluator(BaseEvaluator):
    """Evaluates if appropriate chart type was selected."""
    
    CHART_RECOMMENDATIONS = {
        "comparison": ["bar", "column"],
        "trend": ["line", "area"],
        "composition": ["pie", "donut", "stacked_bar"],
        "distribution": ["histogram", "scatter"],
        "relationship": ["scatter", "bubble"],
    }
    
    def evaluate(self, trace, test_case=None):
        pptx_path = self._find_output_pptx(trace)
        expected_type = test_case.get("expected_chart_type")
        data_pattern = self._analyze_data_pattern(test_case.get("data", {}))
        
        # Extract chart type from PPTX
        actual_type = self._extract_chart_type(pptx_path)
        
        score = 1.0
        issues = []
        
        # Check if chart type is appropriate
        if expected_type and actual_type != expected_type:
            if actual_type in self.CHART_RECOMMENDATIONS.get(data_pattern, []):
                score = 0.8  # Acceptable alternative
            else:
                score = 0.5
                issues.append(f"Expected {expected_type}, got {actual_type}")
        
        return EvaluationResult(
            metric_name="chart_type",
            score=score,
            details={
                "expected": expected_type,
                "actual": actual_type,
                "data_pattern": data_pattern
            },
            issues=issues
        )


class DataAccuracyEvaluator(BaseEvaluator):
    """Evaluates if chart data matches source."""
    
    def evaluate(self, trace, test_case=None):
        pptx_path = self._find_output_pptx(trace)
        expected_data = test_case.get("data", {})
        
        # Extract data from chart
        actual_data = self._extract_chart_data(pptx_path)
        
        score = self._compare_data(expected_data, actual_data)
        
        issues = []
        if score < 1.0:
            mismatches = self._find_mismatches(expected_data, actual_data)
            issues.extend(mismatches)
        
        return EvaluationResult(
            metric_name="data_accuracy",
            score=score,
            details={
                "expected_values": expected_data.get("values"),
                "actual_values": actual_data.get("values"),
                "match_percentage": score * 100
            },
            issues=issues
        )
```

## 📈 Expected Improvements

| Metric | Before | After | Improvement |
|--------|--------|-------|-------------|
| Chart Type Selection | 70% | 90% | +20% |
| Data Accuracy | 75% | 98% | +23% |
| Visual Quality | 65% | 88% | +23% |
| Table Formatting | 60% | 92% | +32% |

## 🔗 Related Examples

- [Template Workflow](../template-workflow/) - Charts in templates
- [Brand Compliance](../brand-compliance/) - Brand colors in charts
- [Image Embedding](../image-embedding/) - Charts with images
