# PPTX Image Embedding Optimization

This example optimizes Claude's ability to **embed, position, and style images** in PowerPoint presentations.

## 🎯 What This Optimizes

| Aspect | Description |
|--------|-------------|
| **Image Placement** | Correct positioning within slides and placeholders |
| **Sizing & Scaling** | Maintaining aspect ratios, fitting containers |
| **Quality Preservation** | Avoiding compression artifacts, resolution issues |
| **Layout Integration** | Images working with text and other elements |
| **Accessibility** | Alt text, proper descriptions |

## 🖼️ Supported Image Scenarios

### 1. Placeholder Images
Using template image placeholders correctly:
```
┌─────────────────────────────┐
│ Title: Product Launch       │
├─────────────┬───────────────┤
│             │               │
│   [IMAGE]   │  Description  │
│             │  • Feature 1  │
│             │  • Feature 2  │
└─────────────┴───────────────┘
```

### 2. Full-Bleed Backgrounds
Images covering entire slide:
```
┌─────────────────────────────┐
│▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓│
│▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓│
│▓▓▓▓▓  Text Overlay  ▓▓▓▓▓▓▓│
│▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓│
│▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓│
└─────────────────────────────┘
```

### 3. Image Grids
Multiple images in gallery layout:
```
┌─────────────────────────────┐
│ Our Team                    │
├─────────┬─────────┬─────────┤
│ [Photo] │ [Photo] │ [Photo] │
│  Name   │  Name   │  Name   │
├─────────┼─────────┼─────────┤
│ [Photo] │ [Photo] │ [Photo] │
│  Name   │  Name   │  Name   │
└─────────┴─────────┴─────────┘
```

### 4. Inline with Text
Images flowing with content:
```
┌─────────────────────────────┐
│ Product Features            │
│                             │
│ [Icon] Feature description  │
│        continues here...    │
│                             │
│ [Icon] Another feature      │
│        description text     │
└─────────────────────────────┘
```

## 📁 Files

```
image-embedding/
├── README.md                 # This file
├── config.yaml              # Optimization configuration
├── test_cases.yaml          # 20 image embedding tests
├── run_optimization.py      # Runner script
├── evaluators/
│   └── image_evaluator.py   # Custom image quality evaluator
└── assets/
    └── test_images/         # Sample images for testing
```

## 📊 Test Categories

### Basic Embedding (4 cases)
- Single image in placeholder
- Image with caption
- Image from URL
- Image from base64

### Sizing & Positioning (5 cases)
- Maintain aspect ratio
- Fit to container (contain)
- Fill container (cover)
- Specific dimensions
- Center alignment

### Advanced Layouts (5 cases)
- Image grids (2x2, 3x3)
- Side-by-side with text
- Overlapping images
- Image with shape mask
- Background with overlay

### Quality & Optimization (3 cases)
- High-res images (no downgrade)
- SVG embedding
- Compression settings

### Accessibility (3 cases)
- Alt text inclusion
- Decorative vs informative
- Screen reader considerations

## ⚙️ Configuration

```yaml
# config.yaml
skill:
  path: ~/.claude/skills/pptx

optimization:
  optimize:
    - image_placement_logic
    - sizing_calculations
    - aspect_ratio_handling
    - layout_integration

evaluation:
  metrics:
    - name: image_placement
      weight: 0.30
      type: custom
      checks:
        - within_bounds
        - proper_alignment
        - no_overlap_issues
        
    - name: aspect_ratio
      weight: 0.25
      type: computed
      tolerance: 0.02  # 2% deviation allowed
      
    - name: quality_preservation
      weight: 0.25
      type: custom
      checks:
        - resolution_maintained
        - no_compression_artifacts
        
    - name: layout_integration
      weight: 0.20
      type: visual

image_handling:
  default_dpi: 150
  max_file_size_mb: 10
  supported_formats:
    - png
    - jpg
    - jpeg
    - svg
    - gif
    - webp
  
  sizing_modes:
    contain: "Fit within bounds, maintain aspect ratio"
    cover: "Fill bounds, crop if needed"
    stretch: "Fill bounds, may distort"
    none: "Original size"
```

## 📝 Sample Test Cases

```yaml
test_cases:
  - id: image_basic_001
    description: "Insert image in placeholder maintaining aspect ratio"
    prompt: |
      Using the template's "Picture with Caption" layout, add:
      - Image: team_photo.jpg (landscape, 1920x1080)
      - Caption: "Our Leadership Team"
      
      The image should fit the placeholder without stretching
      or cropping important content.
    
    input_files:
      - path: assets/test_images/team_photo.jpg
        dimensions: [1920, 1080]
        
    quality_criteria:
      - "Image not stretched (aspect ratio preserved)"
      - "Image fills placeholder appropriately"
      - "Caption visible and properly positioned"
      - "No pixelation or quality loss"
      
    tags: [basic, placeholder, aspect_ratio]
    complexity: simple

  - id: image_grid_001
    description: "Create 3x2 image grid with consistent sizing"
    prompt: |
      Create a "Team Members" slide with a 3x2 grid of photos:
      - Row 1: Alice, Bob, Carol
      - Row 2: David, Eve, Frank
      
      Each photo should be:
      - Same size (square crop is OK)
      - Evenly spaced
      - With name below
      
      Use provided headshot images.
    
    input_files:
      - path: assets/test_images/headshots/*.jpg
        count: 6
        
    quality_criteria:
      - "All images same size"
      - "Grid evenly spaced"
      - "Names aligned with photos"
      - "No image stretching"
      
    tags: [grid, multiple_images, layout]
    complexity: medium

  - id: image_fullbleed_001
    description: "Full-bleed background with text overlay"
    prompt: |
      Create a section divider slide with:
      - Full-bleed background image (city skyline)
      - Semi-transparent dark overlay (50% opacity)
      - Large white text: "Our Global Presence"
      - Subtitle: "Offices in 25 countries"
      
      Text should be readable against the image.
    
    input_files:
      - path: assets/test_images/cityscape.jpg
        dimensions: [3840, 2160]
        
    quality_criteria:
      - "Image covers entire slide"
      - "Overlay improves text readability"
      - "Text properly contrasted"
      - "Professional appearance"
      
    tags: [fullbleed, background, overlay]
    complexity: medium
```

## 🔧 Custom Image Evaluator

```python
# evaluators/image_evaluator.py
from PIL import Image
import io
from pptx import Presentation
from pptx.util import Emu

class ImageQualityEvaluator(BaseEvaluator):
    """Evaluates image embedding quality."""
    
    def evaluate(self, trace, test_case=None):
        pptx_path = self._find_output_pptx(trace)
        input_images = test_case.get("input_files", [])
        
        scores = {
            "placement": self._check_placement(pptx_path),
            "aspect_ratio": self._check_aspect_ratios(pptx_path, input_images),
            "quality": self._check_image_quality(pptx_path),
            "alignment": self._check_alignment(pptx_path),
        }
        
        issues = []
        
        # Check for stretched images
        stretched = self._find_stretched_images(pptx_path, input_images)
        if stretched:
            issues.append(f"Stretched images detected: {len(stretched)}")
            scores["aspect_ratio"] *= 0.5
            
        # Check for out-of-bounds
        oob = self._find_out_of_bounds(pptx_path)
        if oob:
            issues.append(f"Images outside slide bounds: {len(oob)}")
            scores["placement"] *= 0.7
        
        overall = sum(scores.values()) / len(scores)
        
        return EvaluationResult(
            metric_name="image_quality",
            score=overall,
            details=scores,
            issues=issues
        )
    
    def _check_aspect_ratios(self, pptx_path, input_images):
        """Compare embedded image ratios to originals."""
        prs = Presentation(pptx_path)
        
        for slide in prs.slides:
            for shape in slide.shapes:
                if shape.shape_type == 13:  # Picture
                    embedded_ratio = shape.width / shape.height
                    # Compare to original...
                    
        return 1.0  # Simplified
```

## 📈 Expected Improvements

| Metric | Before | After | Improvement |
|--------|--------|-------|-------------|
| Image Placement | 70% | 92% | +22% |
| Aspect Ratio | 65% | 95% | +30% |
| Quality Preservation | 75% | 90% | +15% |
| Layout Integration | 60% | 85% | +25% |

## 🔗 Related Examples

- [Template Workflow](../template-workflow/) - Image placeholders in templates
- [Brand Compliance](../brand-compliance/) - Brand-approved images
- [Charts & Tables](../charts-tables/) - Images with data
