# PPTX Template Workflow Optimization

This example demonstrates how to optimize Claude's skill for working with **corporate PowerPoint templates**.

## 🎯 What This Optimizes

| Aspect | Description |
|--------|-------------|
| **Template Loading** | Correctly loading and using .potx/.pptx templates |
| **Layout Selection** | Choosing appropriate slide layouts from template |
| **Style Preservation** | Maintaining template fonts, colors, and formatting |
| **Placeholder Usage** | Properly populating title, content, footer placeholders |
| **Master Slides** | Respecting master slide configurations |

## 📁 Files

```
template-workflow/
├── README.md                 # This file
├── config.yaml              # Optimization configuration
├── test_cases.yaml          # 25 comprehensive test cases
├── run_optimization.py      # Runner script
├── brand_config.yaml        # Brand/template specifications
└── expected_outputs/        # Reference outputs
```

## 🚀 Quick Start

```bash
# 1. Review and customize config
cat config.yaml

# 2. Run optimization
python run_optimization.py

# 3. Review results
cat results/report.md
```

## 📊 Test Case Categories

### Basic Template Usage (5 cases)
- Load template and create title slide
- Use different slide layouts
- Populate standard placeholders
- Maintain template fonts
- Preserve color scheme

### Complex Layouts (5 cases)
- Multi-column content slides
- Comparison layouts (side-by-side)
- Section headers with subtitles
- Agenda slides with navigation
- Mixed content (text + image placeholders)

### Template Variations (5 cases)
- Corporate template (formal)
- Creative template (colorful)
- Minimal template (clean)
- Data-heavy template (charts/tables)
- Executive summary template

### Edge Cases (5 cases)
- Template with custom fonts (not installed)
- Template with embedded macros
- Very old .pot format
- Corrupted template recovery
- Template with locked elements

### Integration Scenarios (5 cases)
- Template + brand guidelines together
- Template + external images
- Template + data from CSV
- Template + multi-language content
- Template update (new version)

## ⚙️ Configuration

```yaml
# config.yaml
skill:
  path: ~/.claude/skills/pptx
  components:
    - SKILL.md
    - html2pptx.md
    - ooxml.md        # Template editing uses OOXML

optimization:
  max_iterations: 15
  optimize:
    - template_loading_instructions
    - layout_selection_guidance
    - placeholder_population_rules
    - style_preservation_checks
    
evaluation:
  metrics:
    - name: template_compliance
      weight: 0.35
      type: custom
      evaluator: template_evaluator.TemplateComplianceEvaluator
      
    - name: layout_correctness
      weight: 0.25
      type: custom
      
    - name: style_preservation
      weight: 0.25
      type: visual
      
    - name: task_completion
      weight: 0.15
      type: binary
```

## 📈 Expected Improvements

| Metric | Before | After | Improvement |
|--------|--------|-------|-------------|
| Template Compliance | 65% | 88% | +23% |
| Layout Correctness | 70% | 90% | +20% |
| Style Preservation | 60% | 85% | +25% |
| Task Completion | 75% | 92% | +17% |

## 🔍 Common Issues Addressed

### 1. Wrong Layout Selection
**Before**: Claude picks "Title and Content" for all slides
**After**: Claude analyzes content and selects appropriate layout

### 2. Font Substitution
**Before**: Template fonts replaced with defaults
**After**: Fonts preserved or graceful fallback applied

### 3. Placeholder Overflow
**Before**: Text overflows placeholder boundaries
**After**: Auto-sizing or content splitting

### 4. Color Scheme Drift
**Before**: Hardcoded colors override theme
**After**: Theme colors used consistently

## 📝 Sample Test Case

```yaml
- id: template_complex_001
  description: "Create executive summary from corporate template"
  prompt: |
    Using the Acme Corp template (acme_corporate.potx), create a 
    5-slide executive summary presentation:
    
    1. Title slide with "Q4 2024 Results" and subtitle "Executive Summary"
    2. Agenda slide with 4 topics
    3. Key metrics slide with 4 KPI boxes
    4. Comparison slide: Q3 vs Q4 performance
    5. Next steps slide with action items
    
    Ensure all slides use appropriate layouts from the template.
    Maintain the corporate color scheme (Navy #1E3A5F, Gold #D4AF37).
  
  expected_outputs:
    - type: file
      pattern: "*.pptx"
      
  quality_criteria:
    - "Uses layouts from template (not blank slides)"
    - "Title placeholder properly populated"
    - "Corporate colors preserved"
    - "Footer shows company name"
    - "Slide numbers present"
    
  template_requirements:
    template_path: "assets/acme_corporate.potx"
    required_layouts:
      - "Title Slide"
      - "Section Header"
      - "Two Content"
      - "Comparison"
    color_scheme:
      primary: "#1E3A5F"
      accent: "#D4AF37"
      
  tags: [template, executive, complex]
  complexity: complex
```

## 🛠️ Custom Evaluator

```python
# template_evaluator.py
class TemplateComplianceEvaluator(BaseEvaluator):
    """Evaluates how well output follows template specifications."""
    
    def evaluate(self, trace, test_case=None):
        pptx_path = self._find_output_pptx(trace)
        template_reqs = test_case.get("template_requirements", {})
        
        scores = {
            "layout_usage": self._check_layouts(pptx_path, template_reqs),
            "color_compliance": self._check_colors(pptx_path, template_reqs),
            "font_compliance": self._check_fonts(pptx_path, template_reqs),
            "placeholder_usage": self._check_placeholders(pptx_path),
        }
        
        overall = sum(scores.values()) / len(scores)
        
        return EvaluationResult(
            metric_name="template_compliance",
            score=overall,
            details=scores,
            issues=self._collect_issues(scores)
        )
```

## 📚 Related Examples

- [Brand Compliance](../brand-compliance/) - Ensure brand guidelines
- [Image Embedding](../image-embedding/) - Add images to templates
- [Charts & Tables](../charts-tables/) - Data visualization in templates
