#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.9"
# dependencies = [
#     "pyyaml>=6.0",
#     "pillow>=10.0.0",
# ]
# ///
"""
PPTX Template Workflow Optimization Runner

This script runs the GEPA optimization for PowerPoint template usage.
Uses uv inline script dependencies - no virtual environment needed.

Usage:
    # With uv (preferred - no residual files):
    uv run run_optimization.py [--max-iterations N] [--mock]

    # Or directly (uv auto-detected via shebang):
    ./run_optimization.py [--max-iterations N] [--mock]

    # Legacy pip (creates venv):
    pip install -e ../../../ && python run_optimization.py
"""

import argparse
import os
import sys
from pathlib import Path

# Add src to path
project_root = Path(__file__).parent.parent.parent.parent
sys.path.insert(0, str(project_root / "src"))

from core.skill_optimizer import SkillOptimizer, OptimizationConfig, load_test_cases
from core.skill_loader import SkillLoader


def main():
    parser = argparse.ArgumentParser(description="Run PPTX template workflow optimization")
    parser.add_argument("--max-iterations", type=int, default=15, help="Maximum iterations")
    parser.add_argument("--mock", action="store_true", help="Use mock executor for testing")
    parser.add_argument("--skill-path", default=None, help="Override skill path")
    parser.add_argument("--config", default="config.yaml", help="Config file")
    parser.add_argument("--test-cases", default="test_cases.yaml", help="Test cases file")
    args = parser.parse_args()

    print("=" * 60)
    print("PPTX Template Workflow Optimization")
    print("=" * 60)

    # Get paths
    example_dir = Path(__file__).parent
    config_path = example_dir / args.config
    test_cases_path = example_dir / args.test_cases

    # Load configuration
    print(f"\nLoading config from: {config_path}")
    config = OptimizationConfig.from_yaml(str(config_path))

    # Override settings from args
    if args.max_iterations:
        config.max_iterations = args.max_iterations
    if args.mock:
        config.executor_mode = "mock"
    if args.skill_path:
        config.skill_path = args.skill_path

    # Set output directory
    config.output_dir = str(example_dir / "results")
    os.makedirs(config.output_dir, exist_ok=True)

    # Load test cases
    print(f"Loading test cases from: {test_cases_path}")
    test_cases = load_test_cases(str(test_cases_path))
    print(f"   Loaded {len(test_cases)} test cases")

    # Show test case distribution
    complexity_counts = {}
    for tc in test_cases:
        c = tc.complexity
        complexity_counts[c] = complexity_counts.get(c, 0) + 1

    print("\nTest case distribution:")
    for complexity, count in sorted(complexity_counts.items()):
        print(f"   - {complexity}: {count}")

    # Verify skill exists
    print(f"\nChecking skill: {config.skill_path}")
    loader = SkillLoader()
    try:
        skill = loader.load(config.skill_path)
        print(f"   Loaded skill: {skill.name}")
        print(f"   Components: {len(skill.references)} references, {len(skill.scripts)} scripts")
    except FileNotFoundError as e:
        print(f"   Skill not found: {e}")
        print("\nTip: Ensure the PPTX skill is installed at the specified path")
        print("   For Claude Code: ~/.claude/skills/pptx/")
        print("   For Claude.ai: /mnt/skills/public/pptx/")

        if args.mock:
            print("\n   Continuing with mock executor...")
        else:
            return 1

    # Run optimization
    print("\nStarting optimization...")
    print(f"   Max iterations: {config.max_iterations}")
    print(f"   Executor mode: {config.executor_mode}")
    print()

    try:
        optimizer = SkillOptimizer(config, test_cases)
        result = optimizer.optimize()

        # Print results
        print("\n" + "=" * 60)
        print("Optimization Results")
        print("=" * 60)

        print(f"\nBest candidate: {result.best_candidate.id}")
        print(f"   Generation: {result.best_candidate.generation}")

        print("\nScore Improvements:")
        for metric, improvement in result.improvement.items():
            initial = result.initial_scores.get(metric, 0)
            final = result.final_scores.get(metric, 0)
            direction = "up" if improvement > 0 else "down" if improvement < 0 else "same"
            print(f"   [{direction}] {metric}: {initial:.2%} -> {final:.2%} ({improvement:+.2%})")

        print(f"\nRuntime: {result.runtime_seconds:.1f} seconds")
        print(f"Results saved to: {config.output_dir}")

        # Save summary
        summary_path = Path(config.output_dir) / "summary.txt"
        with open(summary_path, "w") as f:
            f.write(result.to_report())
        print(f"Summary report: {summary_path}")

        return 0

    except Exception as e:
        print(f"\nError during optimization: {e}")
        import traceback
        traceback.print_exc()
        return 1


if __name__ == "__main__":
    sys.exit(main())
