#!/bin/bash
# Install the skill-optimizer meta-skill to Claude Code
# Prefers uv if available, falls back to copying files directly

set -e

SKILL_NAME="skill-optimizer"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_SOURCE="$SCRIPT_DIR/skill"

# Determine installation target
if [ -n "$1" ]; then
    SKILL_TARGET="$1"
elif [ -d "$HOME/.claude/skills" ]; then
    SKILL_TARGET="$HOME/.claude/skills/$SKILL_NAME"
else
    SKILL_TARGET="$HOME/.claude/skills/$SKILL_NAME"
    mkdir -p "$HOME/.claude/skills"
fi

echo "Installing Claude Skill Optimizer meta-skill..."
echo ""
echo "   Source: $SKILL_SOURCE"
echo "   Target: $SKILL_TARGET"
echo ""

# Create target directory
mkdir -p "$SKILL_TARGET"

# Copy skill files
cp "$SKILL_SOURCE/SKILL.md" "$SKILL_TARGET/"
if [ -f "$SKILL_SOURCE/README.md" ]; then
    cp "$SKILL_SOURCE/README.md" "$SKILL_TARGET/"
fi

# Copy source code for the skill to reference
mkdir -p "$SKILL_TARGET/src"
cp -r "$SCRIPT_DIR/src/"* "$SKILL_TARGET/src/"

# Check if uv is available and provide appropriate instructions
if command -v uv &> /dev/null; then
    echo "Detected uv - scripts will run inline without creating environments"
    echo ""

    # Create a runner script that uses uv
    cat > "$SKILL_TARGET/run.sh" << 'RUNNER_EOF'
#!/bin/bash
# Run skill optimizer commands using uv (inline, no residual files)
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"
uv run --script "$@"
RUNNER_EOF
    chmod +x "$SKILL_TARGET/run.sh"

    echo "Meta-skill installed successfully!"
    echo ""
    echo "Location: $SKILL_TARGET"
    echo ""
    echo "Usage (uv - preferred, no residual files):"
    echo "   uv run python -m src.core.skill_optimizer --help"
    echo "   uv run python -m src.generators.test_generator --help"
else
    echo "Note: uv not found. Install uv for cleaner execution:"
    echo "   curl -LsSf https://astral.sh/uv/install.sh | sh"
    echo ""
    echo "Meta-skill installed successfully!"
    echo ""
    echo "Location: $SKILL_TARGET"
    echo ""
    echo "Usage (legacy pip):"
    echo "   cd $SCRIPT_DIR && pip install -e ."
    echo "   skill-optimizer --help"
fi

echo ""
echo "Ask Claude Code:"
echo "   'Help me optimize my PPTX skill'"
echo "   'Analyze why my docx skill fails on long documents'"
echo ""
echo "The skill includes:"
echo "   - SKILL.md: Instructions for skill optimization"
echo "   - src/: Python modules for GEPA optimization"
echo ""
