"""
Generic Skill Loader

Loads any Claude skill from various locations:
- ~/.claude/skills/{skill-name}/
- ./.claude/skills/{skill-name}/
- /mnt/skills/public/{skill-name}/
- Any custom path

Extracts all components: SKILL.md, references, scripts, assets
"""

import os
import re
import yaml
from dataclasses import dataclass, field
from pathlib import Path
from typing import Optional


@dataclass
class SkillComponent:
    """A single component of a skill (file or section)."""
    name: str
    path: str
    content: str
    component_type: str  # 'instructions', 'reference', 'script', 'asset'
    sections: dict = field(default_factory=dict)  # Extracted sections


@dataclass
class Skill:
    """Complete representation of a Claude skill."""
    
    name: str
    path: str
    description: str
    
    # Core content
    instructions: str  # Full SKILL.md content
    
    # Extracted sections from SKILL.md
    sections: dict = field(default_factory=dict)
    
    # Additional components
    references: list[SkillComponent] = field(default_factory=list)
    scripts: list[SkillComponent] = field(default_factory=list)
    assets: list[str] = field(default_factory=list)  # Just paths for binary assets
    
    # Metadata from frontmatter
    metadata: dict = field(default_factory=dict)
    
    def get_section(self, section_name: str) -> Optional[str]:
        """Get a specific section from the skill."""
        return self.sections.get(section_name)
    
    def get_optimizable_components(self) -> dict[str, str]:
        """Get all text components that can be optimized."""
        components = {
            "instructions": self.instructions
        }
        
        for ref in self.references:
            components[f"reference:{ref.name}"] = ref.content
            
        for script in self.scripts:
            components[f"script:{script.name}"] = script.content
            
        return components
    
    def to_prompt_context(self) -> str:
        """Convert skill to context string for LLM."""
        parts = [
            f"# Skill: {self.name}",
            f"\n## Description\n{self.description}",
            f"\n## Instructions\n{self.instructions[:5000]}..."  # Truncate
        ]
        
        if self.references:
            parts.append("\n## References")
            for ref in self.references[:3]:
                parts.append(f"\n### {ref.name}\n{ref.content[:1000]}...")
                
        return "\n".join(parts)


class SkillLoader:
    """
    Load Claude skills from any location.
    
    Supports:
    - Standard skill structure (SKILL.md + optional subdirs)
    - Frontmatter metadata extraction
    - Section parsing from markdown
    - Reference and script loading
    """
    
    # Common skill locations
    SKILL_PATHS = [
        "~/.claude/skills",           # User skills (Claude Code)
        "./.claude/skills",           # Project skills
        "/mnt/skills/public",         # Claude.ai public skills
        "/mnt/skills/user",           # Claude.ai user skills
        "/mnt/skills/examples",       # Claude.ai example skills
    ]
    
    def __init__(self):
        pass
    
    def load(self, skill_path: str) -> Skill:
        """
        Load a skill from the given path.
        
        Args:
            skill_path: Path to skill directory or SKILL.md file
            
        Returns:
            Skill object with all components loaded
        """
        # Normalize path
        skill_path = os.path.expanduser(skill_path)
        
        # Handle both directory and file paths
        if os.path.isfile(skill_path):
            skill_dir = os.path.dirname(skill_path)
            skill_file = skill_path
        else:
            skill_dir = skill_path
            skill_file = os.path.join(skill_dir, "SKILL.md")
        
        if not os.path.exists(skill_file):
            raise FileNotFoundError(f"SKILL.md not found at {skill_file}")
        
        # Read main skill file
        with open(skill_file, "r", encoding="utf-8") as f:
            content = f.read()
        
        # Parse frontmatter and content
        metadata, instructions = self._parse_frontmatter(content)
        
        # Extract name and description
        name = metadata.get("name", os.path.basename(skill_dir))
        description = metadata.get("description", "")
        
        # Parse sections
        sections = self._parse_sections(instructions)
        
        # Load references
        references = self._load_references(skill_dir)
        
        # Load scripts
        scripts = self._load_scripts(skill_dir)
        
        # Find assets
        assets = self._find_assets(skill_dir)
        
        return Skill(
            name=name,
            path=skill_dir,
            description=description,
            instructions=instructions,
            sections=sections,
            references=references,
            scripts=scripts,
            assets=assets,
            metadata=metadata
        )
    
    def find_skill(self, skill_name: str) -> Optional[str]:
        """
        Find a skill by name in standard locations.
        
        Args:
            skill_name: Name of the skill to find
            
        Returns:
            Path to skill directory, or None if not found
        """
        for base_path in self.SKILL_PATHS:
            expanded = os.path.expanduser(base_path)
            skill_path = os.path.join(expanded, skill_name)
            
            if os.path.exists(os.path.join(skill_path, "SKILL.md")):
                return skill_path
        
        return None
    
    def list_skills(self) -> list[dict]:
        """List all available skills."""
        skills = []
        
        for base_path in self.SKILL_PATHS:
            expanded = os.path.expanduser(base_path)
            
            if not os.path.exists(expanded):
                continue
                
            for item in os.listdir(expanded):
                skill_dir = os.path.join(expanded, item)
                skill_file = os.path.join(skill_dir, "SKILL.md")
                
                if os.path.isdir(skill_dir) and os.path.exists(skill_file):
                    try:
                        with open(skill_file, "r") as f:
                            content = f.read()
                        metadata, _ = self._parse_frontmatter(content)
                        
                        skills.append({
                            "name": metadata.get("name", item),
                            "path": skill_dir,
                            "description": metadata.get("description", "")[:100]
                        })
                    except Exception:
                        skills.append({
                            "name": item,
                            "path": skill_dir,
                            "description": "(Could not read)"
                        })
        
        return skills
    
    def _parse_frontmatter(self, content: str) -> tuple[dict, str]:
        """
        Parse YAML frontmatter from skill content.
        
        Returns:
            (metadata_dict, remaining_content)
        """
        # Check for frontmatter
        if not content.startswith("---"):
            return {}, content
        
        # Find end of frontmatter
        end_match = re.search(r"\n---\s*\n", content[3:])
        if not end_match:
            return {}, content
        
        end_pos = end_match.end() + 3
        frontmatter = content[3:end_match.start() + 3]
        remaining = content[end_pos:]
        
        try:
            metadata = yaml.safe_load(frontmatter)
            if not isinstance(metadata, dict):
                metadata = {}
        except yaml.YAMLError:
            metadata = {}
        
        return metadata, remaining.strip()
    
    def _parse_sections(self, content: str) -> dict[str, str]:
        """
        Parse markdown sections from skill content.
        
        Returns:
            Dict mapping section headers to content
        """
        sections = {}
        current_section = "intro"
        current_content = []
        
        for line in content.split("\n"):
            # Check for header
            header_match = re.match(r"^(#{1,3})\s+(.+)$", line)
            
            if header_match:
                # Save previous section
                if current_content:
                    sections[current_section] = "\n".join(current_content).strip()
                
                # Start new section
                current_section = header_match.group(2).strip()
                current_content = []
            else:
                current_content.append(line)
        
        # Save last section
        if current_content:
            sections[current_section] = "\n".join(current_content).strip()
        
        return sections
    
    def _load_references(self, skill_dir: str) -> list[SkillComponent]:
        """Load reference files from skill directory."""
        references = []
        
        # Check common reference locations
        ref_dirs = [
            os.path.join(skill_dir, "references"),
            os.path.join(skill_dir, "docs"),
            skill_dir  # Some skills have .md files alongside SKILL.md
        ]
        
        for ref_dir in ref_dirs:
            if not os.path.exists(ref_dir):
                continue
                
            for item in os.listdir(ref_dir):
                if item == "SKILL.md":
                    continue
                    
                if item.endswith(".md"):
                    ref_path = os.path.join(ref_dir, item)
                    try:
                        with open(ref_path, "r", encoding="utf-8") as f:
                            content = f.read()
                        
                        references.append(SkillComponent(
                            name=item,
                            path=ref_path,
                            content=content,
                            component_type="reference",
                            sections=self._parse_sections(content)
                        ))
                    except Exception:
                        pass
        
        return references
    
    def _load_scripts(self, skill_dir: str) -> list[SkillComponent]:
        """Load script files from skill directory."""
        scripts = []
        
        scripts_dir = os.path.join(skill_dir, "scripts")
        
        if not os.path.exists(scripts_dir):
            return scripts
        
        for item in os.listdir(scripts_dir):
            if item.endswith((".py", ".js", ".sh", ".ts")):
                script_path = os.path.join(scripts_dir, item)
                try:
                    with open(script_path, "r", encoding="utf-8") as f:
                        content = f.read()
                    
                    scripts.append(SkillComponent(
                        name=item,
                        path=script_path,
                        content=content,
                        component_type="script"
                    ))
                except Exception:
                    pass
        
        return scripts
    
    def _find_assets(self, skill_dir: str) -> list[str]:
        """Find asset files in skill directory."""
        assets = []
        
        assets_dir = os.path.join(skill_dir, "assets")
        
        if not os.path.exists(assets_dir):
            return assets
        
        for root, _, files in os.walk(assets_dir):
            for file in files:
                assets.append(os.path.join(root, file))
        
        return assets


class SkillWriter:
    """
    Write modified skills back to disk.
    """
    
    def __init__(self):
        pass
    
    def write(self, skill: Skill, output_path: str):
        """
        Write a skill to the given path.
        
        Args:
            skill: Skill object to write
            output_path: Directory to write to
        """
        os.makedirs(output_path, exist_ok=True)
        
        # Write SKILL.md with frontmatter
        skill_content = self._format_skill_md(skill)
        
        with open(os.path.join(output_path, "SKILL.md"), "w") as f:
            f.write(skill_content)
        
        # Write references
        if skill.references:
            refs_dir = os.path.join(output_path, "references")
            os.makedirs(refs_dir, exist_ok=True)
            
            for ref in skill.references:
                with open(os.path.join(refs_dir, ref.name), "w") as f:
                    f.write(ref.content)
        
        # Write scripts
        if skill.scripts:
            scripts_dir = os.path.join(output_path, "scripts")
            os.makedirs(scripts_dir, exist_ok=True)
            
            for script in skill.scripts:
                with open(os.path.join(scripts_dir, script.name), "w") as f:
                    f.write(script.content)
    
    def _format_skill_md(self, skill: Skill) -> str:
        """Format skill as SKILL.md content."""
        parts = []
        
        # Frontmatter
        parts.append("---")
        parts.append(f"name: {skill.name}")
        parts.append(f'description: "{skill.description}"')
        
        for key, value in skill.metadata.items():
            if key not in ("name", "description"):
                parts.append(f"{key}: {value}")
        
        parts.append("---")
        parts.append("")
        
        # Instructions
        parts.append(skill.instructions)
        
        return "\n".join(parts)


def load_skill(path: str) -> Skill:
    """Convenience function to load a skill."""
    loader = SkillLoader()
    return loader.load(path)


def find_skill(name: str) -> Optional[Skill]:
    """Convenience function to find and load a skill by name."""
    loader = SkillLoader()
    path = loader.find_skill(name)
    if path:
        return loader.load(path)
    return None
