"""
Pytest configuration and fixtures.
"""

import pytest
from dataclasses import dataclass
from typing import Optional


@pytest.fixture
def sample_trace():
    """Create a sample execution trace for testing."""
    from src.core.claude_executor import ExecutionTrace
    
    return ExecutionTrace(
        task_id="test_001",
        prompt="Create a presentation",
        success=True,
        exit_code=0,
        execution_time=15.0,
        reasoning=["Analyzing request", "Creating slides"],
        code_blocks=[
            {"language": "javascript", "content": "const pptx = new pptxgen();"}
        ],
        commands=[
            {"cmd": "node convert.js", "exit_code": 0, "stdout": "Done"}
        ],
        files_created=[
            {"path": "output.pptx", "size": 45000, "type": "presentation"}
        ],
        errors=[],
        warnings=[]
    )


@pytest.fixture
def sample_test_case():
    """Create a sample test case for testing."""
    return {
        "id": "test_001",
        "description": "Create simple presentation",
        "prompt": "Create a 3-slide presentation about AI",
        "expected_outputs": [
            {"type": "file", "pattern": "*.pptx"}
        ],
        "quality_criteria": [
            "Contains 3 slides",
            "Has title slide"
        ],
        "tags": ["basic"],
        "complexity": "simple"
    }


@pytest.fixture
def mock_skill_path(tmp_path):
    """Create a temporary skill directory for testing."""
    skill_dir = tmp_path / "test-skill"
    skill_dir.mkdir()
    
    # Create SKILL.md
    skill_md = skill_dir / "SKILL.md"
    skill_md.write_text("""---
name: test-skill
description: A test skill for unit testing
---

# Test Skill

This is a test skill.

## Instructions

Do the thing.
""")
    
    return str(skill_dir)


@pytest.fixture
def sample_config():
    """Create a sample optimization config."""
    from src.core.skill_optimizer import OptimizationConfig
    
    return OptimizationConfig(
        skill_path="/tmp/test-skill",
        skill_name="test-skill",
        max_iterations=3,
        max_evaluations=10,
        executor_mode="mock"
    )
