"""
Unit tests for evaluators.
"""

import pytest
from src.evaluators.base import (
    BinaryEvaluator,
    FileExistsEvaluator,
    ErrorRateEvaluator,
    EfficiencyEvaluator,
    EvaluationResult,
)
from src.core.claude_executor import ExecutionTrace


class TestBinaryEvaluator:
    """Tests for BinaryEvaluator."""
    
    def test_success_returns_1(self):
        """Successful execution should return score of 1.0."""
        evaluator = BinaryEvaluator(name="test")
        trace = ExecutionTrace(task_id="test", success=True)
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 1.0
        assert result.metric_name == "test"
        assert len(result.issues) == 0
    
    def test_failure_returns_0(self):
        """Failed execution should return score of 0.0."""
        evaluator = BinaryEvaluator(name="test")
        trace = ExecutionTrace(task_id="test", success=False)
        trace.errors = ["Some error"]
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.0
        assert "Some error" in result.issues


class TestFileExistsEvaluator:
    """Tests for FileExistsEvaluator."""
    
    def test_file_exists_returns_1(self):
        """Should return 1.0 when expected file exists."""
        evaluator = FileExistsEvaluator(name="files", patterns=["*.pptx"])
        trace = ExecutionTrace(task_id="test", success=True)
        trace.files_created = [{"path": "output.pptx", "size": 1000}]
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 1.0
    
    def test_missing_file_returns_0(self):
        """Should return 0.0 when expected file missing."""
        evaluator = FileExistsEvaluator(name="files", patterns=["*.pptx"])
        trace = ExecutionTrace(task_id="test", success=True)
        trace.files_created = []  # No files
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.0
        assert len(result.issues) > 0
    
    def test_partial_files_returns_fraction(self):
        """Should return fraction when some files exist."""
        evaluator = FileExistsEvaluator(
            name="files", 
            patterns=["*.pptx", "*.pdf"]
        )
        trace = ExecutionTrace(task_id="test", success=True)
        trace.files_created = [{"path": "output.pptx", "size": 1000}]
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.5  # 1 of 2 patterns matched


class TestErrorRateEvaluator:
    """Tests for ErrorRateEvaluator."""
    
    def test_no_errors_returns_1(self):
        """No errors should return score of 1.0."""
        evaluator = ErrorRateEvaluator(name="errors")
        trace = ExecutionTrace(task_id="test", success=True)
        trace.errors = []
        trace.warnings = []
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 1.0
    
    def test_one_error_reduces_score(self):
        """One error should reduce score."""
        evaluator = ErrorRateEvaluator(name="errors")
        trace = ExecutionTrace(task_id="test", success=False)
        trace.errors = ["Error 1"]
        trace.warnings = []
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.7
    
    def test_multiple_errors_further_reduce(self):
        """Multiple errors should further reduce score."""
        evaluator = ErrorRateEvaluator(name="errors")
        trace = ExecutionTrace(task_id="test", success=False)
        trace.errors = ["Error 1", "Error 2", "Error 3"]
        trace.warnings = []
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.3


class TestEfficiencyEvaluator:
    """Tests for EfficiencyEvaluator."""
    
    def test_fast_execution_returns_1(self):
        """Fast execution should return 1.0."""
        evaluator = EfficiencyEvaluator(
            name="efficiency",
            target_time=30.0,
            max_time=120.0
        )
        trace = ExecutionTrace(task_id="test", success=True)
        trace.execution_time = 10.0  # Very fast
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 1.0
    
    def test_slow_execution_reduces_score(self):
        """Slow execution should reduce score."""
        evaluator = EfficiencyEvaluator(
            name="efficiency",
            target_time=30.0,
            max_time=120.0
        )
        trace = ExecutionTrace(task_id="test", success=True)
        trace.execution_time = 75.0  # Halfway between target and max
        
        result = evaluator.evaluate(trace)
        
        assert 0.4 < result.score < 0.6
    
    def test_timeout_returns_0(self):
        """Execution exceeding max time should return 0."""
        evaluator = EfficiencyEvaluator(
            name="efficiency",
            target_time=30.0,
            max_time=120.0
        )
        trace = ExecutionTrace(task_id="test", success=False)
        trace.execution_time = 150.0  # Exceeds max
        
        result = evaluator.evaluate(trace)
        
        assert result.score == 0.0
        assert len(result.issues) > 0


class TestEvaluationResult:
    """Tests for EvaluationResult dataclass."""
    
    def test_creation(self):
        """Should create result with all fields."""
        result = EvaluationResult(
            metric_name="test",
            score=0.85,
            details={"key": "value"},
            issues=["issue 1"]
        )
        
        assert result.metric_name == "test"
        assert result.score == 0.85
        assert result.details == {"key": "value"}
        assert result.issues == ["issue 1"]
